from DisplayContainer import DisplayContainer
from TargetGroup import TargetGroup
from DisplayTarget import DisplayTarget
from utils.Observable import Observable
from DisplayConfigurator import DisplayConfigurator
from main import DEFAULT_SENSOR, UNSET_COORD
from main import admin
from utils import vfs

import gtk
import os


#
# Class for display windows.
#
class Display(Observable, gtk.EventBox):

    # observer commands
    OBS_CLOSE = 0
    OBS_RESTART = 1


    def __init__(self, id):

        # the container of this display; we use a proxy until we get
        # the real thing; the proxy remembers set values and passes them on
        # to the real container
        self.__container = DisplayContainer(self)

        # the path of the .display file
        self.__path = vfs.getcwd()

        # the unique ID of this display
        self.__id = id

        # the sensors of this window
        self.__sensors = {}

        # the last selected targets (used for detecting events)
        self.__last_targets = []

        # the last position of the mouse pointer (used for filtering out
        # unwanted events)
        self.__last_pointer_pos = (-1, -1)

        # mapping between sensors and targets; which target watches
        # which sensor?
        # (sensor, port) -> (target, property)
        self.__mapping = {}

        # temporary data for remembering the position of the last mouse click
        self.__pointer_pos = (0, 0)

        # whether the display reacts on events
        self.__is_sensitive = gtk.TRUE

        gtk.EventBox.__init__(self)
        self.show()

        # set up event handlers
        self.connect("button-press-event", self.__on_button, 0)
        self.connect("button-release-event", self.__on_button, 1)
        self.connect("scroll-event", self.__on_scroll)
        self.connect("motion-notify-event", self.__on_motion, 0)
        self.connect("leave-notify-event", self.__on_motion, 1)
        self.connect("delete-event", self.__on_close)
        self.add_events(gtk.gdk.BUTTON_PRESS_MASK |
                        gtk.gdk.BUTTON_RELEASE_MASK |
                        gtk.gdk.LEAVE_NOTIFY_MASK |
                        gtk.gdk.POINTER_MOTION_MASK)



    #
    # Sets the container of the display.
    #
    def set_container(self, container):

        # pass the values stored by the proxy on to the real container
        self.__container.pass_values(container)
        self.__container = container


    #
    # Sets the sensitive flag of the display. Insensitive displays don't react
    # on user events.
    #
    def set_sensitive(self, value):

        self.__is_sensitive = value


    #
    # Returns the path of the .display file.
    #
    def get_path(self):

        return self.__path



    def add_children(self, childrendata):

        # create the root TargetGroup
        self.__group = TargetGroup(None, self)
        self.__group.set_position(UNSET_COORD, UNSET_COORD)
        self.__group.add_children(childrendata)
        self.add(self.__group)
        self.__group.add_observer(self.__on_observe_group)




    #
    # Opens the configuration dialog for this display.
    #
    def __open_configurator(self):

        configurators = []
        sensors = self.__sensors.values()
        for s in sensors:
            configurators.append(s.get_configurator())

        dconf = DisplayConfigurator(configurators)
        dconf.set_transient_for(self.get_toplevel())


    #
    # Removes this display.
    #
    def remove_display(self):

        self.drop_observers()
        for s in self.__sensors.values():
            s.stop()

        del self.__sensors
        del self.__mapping
        del self.__last_targets

        self.remove(self.__group)
        self.__group.delete()
        self.__group.destroy()
        del self.__group
        self.__container.close()


    #
    # Reacts on closing the window.
    #
    def __on_close(self, src, event):

        #self.__remove_display()
        self.update_observer(self.OBS_CLOSE, self.__id)




    def file_drop(self, files, x, y):
        ''' Sends the call with path and files to the sensor '''

        targets = self.__get_target_at(x, y)
        # find target and invoke action handler
        rtargets = targets[:]
        rtargets.reverse()
        for t, path in rtargets:
            if (t.has_action(DisplayTarget.ACTION_FILE_DROP)):
                call = t.get_action_call(DisplayTarget.ACTION_FILE_DROP)
                print files, t, path
                self._call_sensor(call, path, files, x, y)
                break
        #end for



    #
    # Reacts on button events.
    #
    def __on_button(self, src, event, is_release = 0):

        if (not self.__is_sensitive): return
        
        px, py = self.get_pointer()
        lx, ly = self.__pointer_pos
        button = event.button
        targets = self.__get_target_at(px, py)

        # determine action
        if (not is_release):
            self.__pointer_pos = (px, py)
            if (button == 1 and event.type == gtk.gdk._2BUTTON_PRESS):
                action = DisplayTarget.ACTION_DOUBLECLICK
            elif (button == 1):
                action = DisplayTarget.ACTION_PRESS
            elif (button == 2):
                #action = DisplayTarget.ACTION_PRESS
                return
            elif (button == 3):
                return
            else:
                return

        else:
            if (button == 1):
                if (abs(lx - px) < 10 and abs(ly - py) < 10):
                    action = DisplayTarget.ACTION_CLICK
                else:
                    action = DisplayTarget.ACTION_RELEASE
            elif (button == 2):
                #action = DisplayTarget.ACTION_RELEASE
                return
            elif (button == 3):
                action = DisplayTarget.ACTION_MENU
            else:
                return

        #end if

        # find target and invoke action handler
        called = 0
        for t, path in targets:
            if (t.has_action(action)):
                call = t.get_action_call(action)
                self._call_sensor(call, path, button)
                called = 1
                break
        #end for

        # make sure that there is always a popup menu
        if (action == DisplayTarget.ACTION_MENU and not called):
            self._call_sensor([(DEFAULT_SENSOR, "menu", [])], [], 3)



    #
    # Reacts on moving the mouse.
    #
    def __on_motion(self, src, event, is_leave):

        if (not self.__is_sensitive): return

        px, py = self.get_pointer()
        if ((px, py) == self.__last_pointer_pos): return
        else: self.__last_pointer_pos = (px, py)

        targets = self.__get_target_at(px, py)

        # how to detect enter and leave:
        # get covered targets and compare with previously covered targets;
        # what's new in covered targets is entered;
        # what's only in previously covered targets is left

        # some braindead window managers treat mouse clicks as leave-notify
        # events; work around this by checking if the mouse really has left
        # the window
        nil, nil, width, height = self.__group.get_geometry()
        if (is_leave and 0 <= px < width and 0 <= py < height): is_leave = 0
        if (is_leave): targets = []
        
        # TODO: make this more efficient; don't check for existence in lists
        for t, path in targets + self.__last_targets:
            # enter
            if (not (t, path) in self.__last_targets and
                t.has_action(t.ACTION_ENTER)):
                self._call_sensor(t.get_action_call(t.ACTION_ENTER), path)

            # leave
            elif (not (t, path) in targets and t.has_action(t.ACTION_LEAVE)):
                self._call_sensor(t.get_action_call(t.ACTION_LEAVE), path)


            # motion
            elif ((t, path) in targets and t.has_action(t.ACTION_MOTION)):
                tx, ty = t.get_pointer()
                self._call_sensor(t.get_action_call(t.ACTION_MOTION),
                                   path, tx, ty)

        #end for

        self.__last_targets = targets



    #
    # Reacts on rolling the mouse wheel.
    #
    def __on_scroll(self, src, event):

        if (not self.__is_sensitive): return

        px, py = self.get_pointer()
        targets = self.__get_target_at(px, py)


        if (event.direction == gtk.gdk.SCROLL_UP):
            direction = 0
        elif (event.direction == gtk.gdk.SCROLL_DOWN):
            direction = 1
        else:
            direction = -1

        for t, path in targets:
            if (t.has_action(DisplayTarget.ACTION_SCROLL)):
                call = t.get_action_call(DisplayTarget.ACTION_SCROLL)
                self._call_sensor(call, path, direction)
                break
        #end for
    


    #
    # Observer for sensors.
    #
    def __on_observe_sensor(self, src, cmd, data):

        # propagate the incoming sensor output
        if (cmd == src.OBS_OUTPUT):
            self.__set_settings(src, data)

        elif (cmd == src.OBS_CMD_CONFIGURE):
            self.__open_configurator()

        elif (cmd == src.OBS_CMD_REMOVE):
            self.update_observer(self.OBS_CLOSE, self.__id)

        elif (cmd == src.OBS_CMD_RESTART):
            self.update_observer(self.OBS_RESTART, self.__id)

        elif (cmd == src.OBS_CMD_DUPLICATE):
            dsp_uri = admin.get_displays().get(self.__id)
            if (dsp_uri): admin.add_display(dsp_uri)



    #
    # Observer for the root group.
    #
    def __on_observe_group(self, src, cmd, *args):

        if (cmd == src.OBS_MOVE):
            x, y, w, h = args
            ux, uy, nil, nil = self.__group.get_user_geometry()
            ax, ay = self.__group.get_anchored_coords(x, y, w, h)
            if (ux == uy == UNSET_COORD):
                self.__container.set_position(ux, uy)
            else:
                self.__container.set_position(ax, ay)
            self._call_sensor([(DEFAULT_SENSOR, "positioned", [])], [""])
                
            if (w != 0 and h != 0):
                # TODO: set the display size instead of the window size
                self.__container.resize(w, h)
                # storing the size is useless, but it's added by request;
                # it makes life easy for desklets pagers
                self._call_sensor([(DEFAULT_SENSOR, "size", [])], [""], w, h)



    #
    # Calls a function of a Sensor.
    #
    def _call_sensor(self, cmd, path, *args):
        assert(cmd)

        args = list(args)
        for id, callname, userargs in cmd:
            if (not id): id = DEFAULT_SENSOR
            sensor = self.__get_sensor(id) or self.__get_sensor(DEFAULT_SENSOR)
            allargs = args + userargs

            # the sensor is an external module, so we make sure it cannot crash
            # the application

            try:
                sensor.send_action(callname, path, allargs)

            except StandardError, e:
                print "The sensor produced an error:", e
                import traceback; traceback.print_exc()
        #end for



    #
    # Returns the target and its path at the given position.
    #
    def __get_target_at(self, px, py):

        targets = self.__group.get_target_at(px, py, [], 1)
        return targets


    #
    # Saves the the given position of the display. The position is nw-anchored.
    #
    def save_position(self, x, y):

        nil, nil, w, h = self.__group.get_geometry()
        ax, ay = self.__group.get_anchored_coords(x, y, w, h)
        dx, dy= x - ax, y - ay
        self._call_sensor([(DEFAULT_SENSOR, "move", [])], [""],
                           x + dx, y + dy, x, y)
        self.__group.set_position(x, y, update = 0)
    

    #
    # Sets the configuration settings.
    #
    def __set_settings(self, sensor, settings):

        for key, value in settings.get_entries():
            # extract array indexes
            if ("[" in key):
                indexpart = key[key.find("[") + 1:-1]
                indexes = indexpart.split("][")
                key = key[:key.find("[")]
            else:
                indexes = []

            # get all (target, property) tuples that are watching the given
            # sensor key and notify the targets
            entries = self.__mapping.get((sensor, key), [])
            for target, property in entries:
                if (indexes):
                    # special handling for arrays
                    sensor_id = self.__get_sensor_id(sensor)
                    target.distribute_sensor_output(sensor_id, indexes[:],
                                                    key, value)
                else:
                    target.set_config(property, value)
            #end for
        #end for



    #
    # Sets the configuration.
    #
    def set_config(self, key, value):

        if (key == "window-flags"):
            self.__container.set_window_flags(value)

        elif (key == "shape"):
            file = vfs.join(self.get_path(), value)
            try:
                loader = gtk.gdk.PixbufLoader()
                fd = vfs.open(file, "r")
                data = vfs.read_all(fd)
                fd.close()
                loader.write(data, len(data))
                loader.close()
                pixbuf = loader.get_pixbuf()
                pix, mask = pixbuf.render_pixmap_and_mask(1)
                self.__container.set_shape(mask)
            except:
                print "could not set shape", file
        
        else:
            self.__group.set_config(key, value)



    #
    # Adds a sensor to this display.
    #
    def add_sensor(self, id, sensor):

        self.__sensors[id] = sensor
        sensor.add_observer(self.__on_observe_sensor)



    #
    # Returns the sensor with the given ID. Returns None if the sensor does not
    # exist.
    #
    def __get_sensor(self, id): return self.__sensors.get(id)



    #
    # Returns the ID of the given sensor.
    #
    def __get_sensor_id(self, sensor):

        for k, v in self.__sensors.items():
            if (v == sensor): return k

        return ""



    #
    # Maps a sensor output to a target.
    #
    def add_mapping(self, sensorplug, target, property):

        id, port = sensorplug.split(":")
        sensor = self.__get_sensor(id)
        if (not self.__mapping.has_key((sensor, port))):
            self.__mapping[(sensor, port)] = []

        if (not (target, property) in self.__mapping[(sensor, port)]):
            self.__mapping[(sensor, port)].append((target, property))
