/* Copyright (C) 2003 Benoît Dejean <bnet@ifrance.com> */

/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or */
/* (at your option) any later version. */

/* This program is distributed in the hope that it will be useful, */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the */
/* GNU General Public License for more details. */

/* You should have received a copy of the GNU General Public License */
/* along with this program; if not, write to the Free Software */
/* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/* http:www.gnu.org/licenses/gpl.html */


/* Benoît Dejean bnet at ifrance dot com */

#include <stddef.h>

#include <Python.h>

#include <glibtop/cpu.h>
#include <glibtop/loadavg.h>
#include <glibtop/mem.h>
#include <glibtop/swap.h>
#include <glibtop/ppp.h>
#include <glibtop/netload.h>
#include <glibtop/uptime.h>
#include <glibtop/proclist.h>
#include <glibtop/xmalloc.h>
#include <glibtop/procstate.h>
#include <glibtop/procuid.h>
#include <glibtop/procmem.h>
#include <glibtop/proctime.h>
#include <glibtop/procsignal.h>
#include <glibtop/prockernel.h>
#include <glibtop/procsegment.h>
#include <glibtop/procargs.h>
#include <glibtop/procmap.h>
#include <glibtop/mountlist.h>
#include <glibtop/fsusage.h>


#define FILL_TUPLE(tuple, converter, tab, len) do { \
size_t i=0; \
while(i < (len)) \
{ \
PyTuple_SetItem(tuple, i, (converter)((tab)[i])); \
++i; \
} \
} \
while(0)



#define FILL_TUPLE2(tuple, converter, ptab, len) do { \
size_t i=0; \
while(i < (len)) \
{ \
PyTuple_SetItem(tuple, i, (converter)((ptab)+i)); \
++i; \
} \
} \
while(0)



#define ADD_TUPLE(tuple, index, converter, tab, len) do { \
PyObject * const inner = PyTuple_New(len); \
FILL_TUPLE(inner, converter, tab, len); \
PyTuple_SetItem(tuple, index, inner); \
} \
while(0)


#define PyL_ULL PyLong_FromUnsignedLongLong
#define PyL_UL  PyLong_FromUnsignedLong
#define PyF_D   PyFloat_FromDouble
#define PyS_S   PyString_FromString
#define PyI_L   PyInt_FromLong




static PyObject * UnicodeToASCII(PyObject *unicode)
{
  const Py_UNICODE * const raw = PyUnicode_AS_UNICODE(unicode);
  const int raw_size = PyUnicode_GET_SIZE(unicode);
  return PyUnicode_EncodeASCII(raw, raw_size, "replace");
}
  




PyObject* get_cpu()
{
  PyObject * const t = PyTuple_New(GLIBTOP_MAX_CPU);

  glibtop_cpu buf;
  glibtop_get_cpu(&buf);

  PyTuple_SetItem(t, GLIBTOP_CPU_TOTAL,	    PyL_ULL (buf.total));
  PyTuple_SetItem(t, GLIBTOP_CPU_USER,      PyL_ULL (buf.user));
  PyTuple_SetItem(t, GLIBTOP_CPU_NICE,      PyL_ULL (buf.nice));
  PyTuple_SetItem(t, GLIBTOP_CPU_SYS,       PyL_ULL (buf.sys));
  PyTuple_SetItem(t, GLIBTOP_CPU_IDLE,      PyL_ULL (buf.idle));
  PyTuple_SetItem(t, GLIBTOP_CPU_FREQUENCY, PyL_ULL (buf.frequency));
  ADD_TUPLE(      t, GLIBTOP_XCPU_TOTAL,    PyL_ULL, buf.xcpu_total, GLIBTOP_NCPU);
  ADD_TUPLE(      t, GLIBTOP_XCPU_USER,     PyL_ULL, buf.xcpu_user,  GLIBTOP_NCPU);
  ADD_TUPLE(      t, GLIBTOP_XCPU_NICE,     PyL_ULL, buf.xcpu_nice,  GLIBTOP_NCPU);
  ADD_TUPLE(      t, GLIBTOP_XCPU_SYS,      PyL_ULL, buf.xcpu_sys,   GLIBTOP_NCPU); 
  ADD_TUPLE(      t, GLIBTOP_XCPU_IDLE,     PyL_ULL, buf.xcpu_idle,  GLIBTOP_NCPU);

  return t;  
}



PyObject* get_mem()
{
  PyObject * const t = PyTuple_New(GLIBTOP_MAX_MEM);

  glibtop_mem buf;
  glibtop_get_mem(&buf);

  PyTuple_SetItem(t, GLIBTOP_MEM_TOTAL,  PyL_ULL(buf.total));
  PyTuple_SetItem(t, GLIBTOP_MEM_USED,   PyL_ULL(buf.used));
  PyTuple_SetItem(t, GLIBTOP_MEM_FREE,   PyL_ULL(buf.free));
  PyTuple_SetItem(t, GLIBTOP_MEM_SHARED, PyL_ULL(buf.shared));
  PyTuple_SetItem(t, GLIBTOP_MEM_BUFFER, PyL_ULL(buf.buffer));
  PyTuple_SetItem(t, GLIBTOP_MEM_CACHED, PyL_ULL(buf.cached));
  PyTuple_SetItem(t, GLIBTOP_MEM_USER,   PyL_ULL(buf.user));
  PyTuple_SetItem(t, GLIBTOP_MEM_LOCKED, PyL_ULL(buf.locked));

  return t;
}



PyObject* get_swap()
{
  PyObject * const t = PyTuple_New(GLIBTOP_MAX_SWAP);

  glibtop_swap buf;
  glibtop_get_swap(&buf);

  PyTuple_SetItem(t, GLIBTOP_SWAP_TOTAL,   PyL_ULL(buf.total));
  PyTuple_SetItem(t, GLIBTOP_SWAP_USED,    PyL_ULL(buf.used));
  PyTuple_SetItem(t, GLIBTOP_SWAP_FREE,    PyL_ULL(buf.free));
  PyTuple_SetItem(t, GLIBTOP_SWAP_PAGEIN,  PyL_ULL(buf.pagein));
  PyTuple_SetItem(t, GLIBTOP_SWAP_PAGEOUT, PyL_ULL(buf.pageout));

  return t;
}



PyObject* get_uptime()
{
  PyObject * const t = PyTuple_New(GLIBTOP_MAX_UPTIME);

  glibtop_uptime buf;
  glibtop_get_uptime(&buf);

  PyTuple_SetItem(t, GLIBTOP_UPTIME_UPTIME,   PyF_D(buf.uptime));
  PyTuple_SetItem(t, GLIBTOP_UPTIME_IDLETIME, PyF_D(buf.idletime));

  return t;
}



PyObject* get_loadavg()
{
  PyObject * const t = PyTuple_New(GLIBTOP_MAX_LOADAVG);

  glibtop_loadavg buf;
  glibtop_get_loadavg(&buf);

  ADD_TUPLE(      t, GLIBTOP_LOADAVG_LOADAVG,    PyF_D,  buf.loadavg,  3);
  PyTuple_SetItem(t, GLIBTOP_LOADAVG_NR_RUNNING, PyL_ULL(buf.nr_running));
  PyTuple_SetItem(t, GLIBTOP_LOADAVG_NR_TASKS,   PyL_ULL(buf.nr_tasks));
  PyTuple_SetItem(t, GLIBTOP_LOADAVG_LAST_PID,   PyL_ULL(buf.last_pid));

  return t;  
}



PyObject* get_netload(PyObject *iinterface)
{
  PyObject * t = NULL;
  const char *interface= NULL;


  /* used if iinterface is an unicode object */
  PyObject *ascii = NULL;  

  if(PyString_Check(iinterface))
    {
      interface = PyString_AsString(iinterface);
    }
  else if(PyUnicode_Check(iinterface))
    {
      ascii = UnicodeToASCII(iinterface);
      interface = PyString_AsString(ascii);
    }
  else
    {
      t = Py_None;
    }
 
  if(t == NULL) /* && interface==NULL */
    {      
      glibtop_netload buf;
      glibtop_get_netload(&buf, interface);

      t = PyTuple_New(GLIBTOP_MAX_NETLOAD);

      PyTuple_SetItem(t, GLIBTOP_NETLOAD_IF_FLAGS,      PyL_ULL(buf.if_flags));
      PyTuple_SetItem(t, GLIBTOP_NETLOAD_MTU,           PyL_ULL(buf.mtu));
      PyTuple_SetItem(t, GLIBTOP_NETLOAD_SUBNET,        PyL_ULL(buf.subnet));
      PyTuple_SetItem(t, GLIBTOP_NETLOAD_ADDRESS,       PyL_ULL(buf.address));
      PyTuple_SetItem(t, GLIBTOP_NETLOAD_PACKETS_IN,    PyL_ULL(buf.packets_in));
      PyTuple_SetItem(t, GLIBTOP_NETLOAD_PACKETS_OUT,   PyL_ULL(buf.packets_out));
      PyTuple_SetItem(t, GLIBTOP_NETLOAD_PACKETS_TOTAL, PyL_ULL(buf.packets_total));
      PyTuple_SetItem(t, GLIBTOP_NETLOAD_BYTES_IN,      PyL_ULL(buf.bytes_in));
      PyTuple_SetItem(t, GLIBTOP_NETLOAD_BYTES_OUT,     PyL_ULL(buf.bytes_out));
      PyTuple_SetItem(t, GLIBTOP_NETLOAD_BYTES_TOTAL,   PyL_ULL(buf.bytes_total));
      PyTuple_SetItem(t, GLIBTOP_NETLOAD_ERRORS_IN,     PyL_ULL(buf.errors_in));
      PyTuple_SetItem(t, GLIBTOP_NETLOAD_ERRORS_OUT,    PyL_ULL(buf.errors_out));
      PyTuple_SetItem(t, GLIBTOP_NETLOAD_ERRORS_TOTAL,  PyL_ULL(buf.errors_total));
      PyTuple_SetItem(t, GLIBTOP_NETLOAD_COLLISIONS,    PyL_ULL(buf.collisions));
    }
    
  if(ascii!=NULL)
    {
      Py_DECREF(ascii);
    }

  return t;    
}



PyObject* get_ppp(PyObject* ddevice)
{
  PyObject * const t = PyTuple_New(GLIBTOP_MAX_PPP);

  const unsigned short device = PyInt_AsLong(ddevice);

  glibtop_ppp buf;
  glibtop_get_ppp(&buf, device);

  PyTuple_SetItem(t, GLIBTOP_PPP_STATE,     PyL_ULL(buf.state));
  PyTuple_SetItem(t, GLIBTOP_PPP_BYTES_IN,  PyL_ULL(buf.bytes_in));
  PyTuple_SetItem(t, GLIBTOP_PPP_BYTES_OUT, PyL_ULL(buf.bytes_out));

  return t;
}



PyObject* get_proclist(PyObject* wwhich, PyObject *aarg)
{
  PyObject* t = NULL;

  const long long which = PyLong_AsLongLong(wwhich);
  const long long   arg = PyLong_AsLongLong(aarg);

  glibtop_proclist buf;
  const unsigned * const list = glibtop_get_proclist(&buf, which, arg);

  t = PyTuple_New(buf.number);

  FILL_TUPLE(t, PyL_UL, list, buf.number);

  glibtop_free(list);

  return t;
}



PyObject* get_proc_state(PyObject *ppid)
{
  PyObject * const t = PyTuple_New(GLIBTOP_MAX_PROC_STATE);

  const pid_t pid = PyLong_AsLong(ppid);

  glibtop_proc_state buf;
  glibtop_get_proc_state(&buf, pid);
  
  {
    const char state[2]={buf.state, '\0'};

    PyTuple_SetItem(t, GLIBTOP_PROC_STATE_CMD,   PyS_S(buf.cmd));
    PyTuple_SetItem(t, GLIBTOP_PROC_STATE_STATE, PyS_S(state));
    PyTuple_SetItem(t, GLIBTOP_PROC_STATE_UID,   PyI_L(buf.uid));
    PyTuple_SetItem(t, GLIBTOP_PROC_STATE_GID,   PyI_L(buf.gid));
  }

  return t;
}



PyObject* get_proc_uid(PyObject *ppid)
{
  PyObject * const t = PyTuple_New(GLIBTOP_MAX_PROC_UID);

  const pid_t pid = PyLong_AsLong(ppid); 
 
  glibtop_proc_uid buf;
  glibtop_get_proc_uid(&buf, pid); 

  PyTuple_SetItem(t, GLIBTOP_PROC_UID_UID,      PyI_L(buf.uid));
  PyTuple_SetItem(t, GLIBTOP_PROC_UID_EUID,     PyI_L(buf.euid));
  PyTuple_SetItem(t, GLIBTOP_PROC_UID_GID,      PyI_L(buf.gid));
  PyTuple_SetItem(t, GLIBTOP_PROC_UID_EGID,     PyI_L(buf.egid));
  PyTuple_SetItem(t, GLIBTOP_PROC_UID_PID,      PyI_L(buf.pid));
  PyTuple_SetItem(t, GLIBTOP_PROC_UID_PPID,     PyI_L(buf.ppid));
  PyTuple_SetItem(t, GLIBTOP_PROC_UID_PGRP,     PyI_L(buf.pgrp));
  PyTuple_SetItem(t, GLIBTOP_PROC_UID_SESSION,  PyI_L(buf.session));
  PyTuple_SetItem(t, GLIBTOP_PROC_UID_TTY,      PyI_L(buf.tty));
  PyTuple_SetItem(t, GLIBTOP_PROC_UID_TPGID,    PyI_L(buf.tpgid));
  PyTuple_SetItem(t, GLIBTOP_PROC_UID_PRIORITY, PyI_L(buf.priority));
  PyTuple_SetItem(t, GLIBTOP_PROC_UID_NICE,     PyI_L(buf.nice));

  return t;
}



PyObject* get_proc_mem(PyObject *ppid)
{
  PyObject * const t = PyTuple_New(GLIBTOP_MAX_PROC_MEM);
  
  const pid_t pid = PyLong_AsLong(ppid);

  glibtop_proc_mem buf;  
  glibtop_get_proc_mem(&buf, pid);
  
  PyTuple_SetItem(t, GLIBTOP_PROC_MEM_SIZE,     PyL_ULL(buf.size));
  PyTuple_SetItem(t, GLIBTOP_PROC_MEM_VSIZE,    PyL_ULL(buf.vsize));
  PyTuple_SetItem(t, GLIBTOP_PROC_MEM_RESIDENT, PyL_ULL(buf.resident));
  PyTuple_SetItem(t, GLIBTOP_PROC_MEM_SHARE,    PyL_ULL(buf.share));
  PyTuple_SetItem(t, GLIBTOP_PROC_MEM_RSS,      PyL_ULL(buf.rss));
  PyTuple_SetItem(t, GLIBTOP_PROC_MEM_RSS_RLIM, PyL_ULL(buf.rss_rlim));
  
  return t;
}



PyObject* get_proc_time(PyObject *ppid)
{
  PyObject * const t = PyTuple_New(GLIBTOP_MAX_PROC_TIME);

  const pid_t pid = PyLong_AsLong(ppid);

  glibtop_proc_time buf;  
  glibtop_get_proc_time(&buf, pid);
 
  PyTuple_SetItem(t,  GLIBTOP_PROC_TIME_START_TIME,    PyL_ULL (buf.start_time));
  PyTuple_SetItem(t,  GLIBTOP_PROC_TIME_RTIME,         PyL_ULL (buf.rtime));
  PyTuple_SetItem(t,  GLIBTOP_PROC_TIME_UTIME,         PyL_ULL (buf.utime));
  PyTuple_SetItem(t,  GLIBTOP_PROC_TIME_STIME,         PyL_ULL (buf.stime));
  PyTuple_SetItem(t,  GLIBTOP_PROC_TIME_CUTIME,        PyL_ULL (buf.cutime));
  PyTuple_SetItem(t,  GLIBTOP_PROC_TIME_CSTIME,        PyL_ULL (buf.cstime));
  PyTuple_SetItem(t,  GLIBTOP_PROC_TIME_TIMEOUT,       PyL_ULL (buf.timeout));
  PyTuple_SetItem(t,  GLIBTOP_PROC_TIME_IT_REAL_VALUE, PyL_ULL (buf.it_real_value));
  PyTuple_SetItem(t,  GLIBTOP_PROC_TIME_FREQUENCY,     PyL_ULL (buf.frequency));
  ADD_TUPLE(      t,  GLIBTOP_PROC_TIME_XCPU_UTIME,    PyL_ULL, buf.xcpu_utime, GLIBTOP_NCPU);
  ADD_TUPLE(      t,  GLIBTOP_PROC_TIME_STIME,         PyL_ULL, buf.xcpu_stime, GLIBTOP_NCPU);

  return t;
}



PyObject* get_proc_signal(PyObject* ppid)
{
  PyObject * const t = PyTuple_New(GLIBTOP_MAX_PROC_SIGNAL);

  const pid_t pid = PyLong_AsLong(ppid);

  glibtop_proc_signal buf;  
  glibtop_get_proc_signal(&buf, pid);

  ADD_TUPLE(t, GLIBTOP_PROC_SIGNAL_SIGNAL,    PyL_ULL, buf.signal,    2);
  ADD_TUPLE(t, GLIBTOP_PROC_SIGNAL_BLOCKED,   PyL_ULL, buf.blocked,   2);
  ADD_TUPLE(t, GLIBTOP_PROC_SIGNAL_SIGIGNORE, PyL_ULL, buf.sigignore, 2);
  ADD_TUPLE(t, GLIBTOP_PROC_SIGNAL_SIGCATCH,  PyL_ULL, buf.sigcatch,  2);

  return t;
}



PyObject* get_proc_kernel(PyObject* ppid)
{
  PyObject * t = PyTuple_New(GLIBTOP_MAX_PROC_KERNEL);

  const pid_t pid = PyLong_AsLong(ppid);

  glibtop_proc_kernel buf;  
  glibtop_get_proc_kernel(&buf, pid);
 
  PyTuple_SetItem(t, GLIBTOP_PROC_KERNEL_K_FLAGS,  PyL_ULL(buf.k_flags));
  PyTuple_SetItem(t, GLIBTOP_PROC_KERNEL_MIN_FLT,  PyL_ULL(buf.min_flt));
  PyTuple_SetItem(t, GLIBTOP_PROC_KERNEL_MAJ_FLT,  PyL_ULL(buf.maj_flt));
  PyTuple_SetItem(t, GLIBTOP_PROC_KERNEL_CMIN_FLT, PyL_ULL(buf.cmin_flt));
  PyTuple_SetItem(t, GLIBTOP_PROC_KERNEL_CMAJ_FLT, PyL_ULL(buf.cmaj_flt));
  PyTuple_SetItem(t, GLIBTOP_PROC_KERNEL_KSTK_ESP, PyL_ULL(buf.kstk_esp));
  PyTuple_SetItem(t, GLIBTOP_PROC_KERNEL_KSTK_EIP, PyL_ULL(buf.kstk_eip));
  PyTuple_SetItem(t, GLIBTOP_PROC_KERNEL_NWCHAN,   PyL_ULL(buf.nwchan));
  PyTuple_SetItem(t, GLIBTOP_PROC_KERNEL_WCHAN,    PyS_S  (buf.wchan));

  return t;
}



PyObject* get_proc_segment(PyObject *ppid)
{
  PyObject * const t = PyTuple_New(GLIBTOP_MAX_PROC_SEGMENT);

  const pid_t pid = PyLong_AsLong(ppid);

  glibtop_proc_segment buf;  
  glibtop_get_proc_segment(&buf, pid);

  PyTuple_SetItem(t, GLIBTOP_PROC_SEGMENT_TEXT_RSS,    PyL_ULL(buf.text_rss));
  PyTuple_SetItem(t, GLIBTOP_PROC_SEGMENT_SHLIB_RSS,   PyL_ULL(buf.shlib_rss));
  PyTuple_SetItem(t, GLIBTOP_PROC_SEGMENT_DATA_RSS,    PyL_ULL(buf.data_rss));
  PyTuple_SetItem(t, GLIBTOP_PROC_SEGMENT_STACK_RSS,   PyL_ULL(buf.stack_rss));
  PyTuple_SetItem(t, GLIBTOP_PROC_SEGMENT_DIRTY_SIZE,  PyL_ULL(buf.dirty_size));
  PyTuple_SetItem(t, GLIBTOP_PROC_SEGMENT_START_CODE,  PyL_ULL(buf.start_code));
  PyTuple_SetItem(t, GLIBTOP_PROC_SEGMENT_END_CODE,    PyL_ULL(buf.end_code)); 
  PyTuple_SetItem(t, GLIBTOP_PROC_SEGMENT_START_STACK, PyL_ULL(buf.start_stack));

  return t;
}



PyObject* get_proc_args(PyObject *ppid)
{
  PyObject * t = NULL;

  const pid_t pid = PyLong_AsLong(ppid);

  glibtop_proc_args buf;
  const char * const args0= glibtop_get_proc_args(&buf, pid, 0u);

  /* splitting args0 */
  /* args0 [......0..   ....0.........0] */

  size_t argc=0;
  const char **argv= PyMem_Malloc(4 * sizeof *argv);
  size_t allocated=4;

  {
    const char *arg = NULL;

    for(arg = args0; arg != (args0 + buf.size); )
      {
	/*const size_t len = strlen(arg);*/

	if(argc==allocated)
	  {
	    argv = PyMem_Realloc(argv, 2 * allocated * sizeof *argv);
	    allocated*=2;
	  }

	argv[argc] = arg;
	++argc;
	arg+= strlen(arg)+1;
      }
  }
	
  t = PyTuple_New(argc);

  FILL_TUPLE(t, PyS_S, argv, argc);	

  PyMem_Free(argv);
  glibtop_free(args0);
  
  return t;
}



static PyObject* map_entry_to_PyTuple(const glibtop_map_entry *e)
{
  PyObject * const t = PyTuple_New(GLIBTOP_MAX_MAP_ENTRY);
 
  PyTuple_SetItem(t, GLIBTOP_MAP_ENTRY_START,    PyL_ULL(e->start));
  PyTuple_SetItem(t, GLIBTOP_MAP_ENTRY_END,      PyL_ULL(e->end));
  PyTuple_SetItem(t, GLIBTOP_MAP_ENTRY_OFFSET,   PyL_ULL(e->offset));
  PyTuple_SetItem(t, GLIBTOP_MAP_ENTRY_PERM,     PyL_ULL(e->perm));
  PyTuple_SetItem(t, GLIBTOP_MAP_ENTRY_INODE,    PyL_ULL(e->inode));
  PyTuple_SetItem(t, GLIBTOP_MAP_ENTRY_DEVICE,   PyL_ULL(e->device));
  PyTuple_SetItem(t, GLIBTOP_MAP_ENTRY_FILENAME, PyS_S  (e->filename));

  return t;
}


PyObject * get_proc_map(PyObject *ppid)
{
  PyObject * t = NULL;

  const pid_t pid = PyLong_AsLong(ppid);

  glibtop_proc_map buf;
  const glibtop_map_entry * const entries = glibtop_get_proc_map(&buf, pid);

  t = PyTuple_New(buf.number);

  FILL_TUPLE2(t, map_entry_to_PyTuple, entries, buf.number);

  glibtop_free(entries);

  return t;
}



static PyObject* mountentry_to_PyTuple(const glibtop_mountentry *e)
{
  PyObject * const t = PyTuple_New(4);

  PyTuple_SetItem(t, 0, PyL_ULL(e->dev));  
  PyTuple_SetItem(t, 1, PyS_S        (e->devname));
  PyTuple_SetItem(t, 2, PyS_S        (e->mountdir));
  PyTuple_SetItem(t, 3, PyS_S        (e->type));

  return t;
}


PyObject* get_mountlist(PyObject *aallfs)
{
  PyObject * t = NULL;

  const int allfs = PyInt_AsLong(aallfs);

  glibtop_mountlist buf;
  const glibtop_mountentry * const entries= glibtop_get_mountlist(&buf, allfs);

  t = PyTuple_New(buf.number);

  FILL_TUPLE2(t, mountentry_to_PyTuple, entries, buf.number);

  glibtop_free(entries);

  return t;
}



PyObject* get_fsusage(PyObject *mmount_dir)
{
  PyObject * t = NULL;
  const char * mount_dir = NULL;

  /* used if mmount_dir is an unicode object */
  PyObject *ascii = NULL;  

  if(PyString_Check(mmount_dir))
    {
      mount_dir = PyString_AsString(mmount_dir);
    }
  else if(PyUnicode_Check(mmount_dir))
    {
      ascii = UnicodeToASCII(mmount_dir);
      mount_dir = PyString_AsString(ascii);
    }
  else
    {
      t = Py_None;
    }

  if(t == NULL) /* && mount_dir==NULL */
    {      
      glibtop_fsusage buf;  
      glibtop_get_fsusage(&buf, mount_dir);

      t = PyTuple_New(GLIBTOP_MAX_FSUSAGE);
  
      PyTuple_SetItem(t, GLIBTOP_FSUSAGE_BLOCKS, PyL_ULL(buf.blocks));
      PyTuple_SetItem(t, GLIBTOP_FSUSAGE_BFREE,  PyL_ULL(buf.bfree));
      PyTuple_SetItem(t, GLIBTOP_FSUSAGE_BAVAIL, PyL_ULL(buf.bavail));
      PyTuple_SetItem(t, GLIBTOP_FSUSAGE_FILES,  PyL_ULL(buf.files));
      PyTuple_SetItem(t, GLIBTOP_FSUSAGE_FFREE,  PyL_ULL(buf.ffree));
    }

  if(ascii!=NULL)
    {
      Py_DECREF(ascii);
    }

  return t;
}
