/* sercomm.c
 *
 * Copyright 2001, 2002 Sun Microsystems, Inc.,
 * Copyright 2001, 2002 BAUM Retec, A.G.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <termios.h>
#include <sys/time.h>
#include <signal.h>

#include <glib.h>

#include "sercomm.h"

#ifndef TTYNAME0
#define TTYNAME0 "/dev/ttys0"
#endif /*TTYNAME0*/

#ifndef TTYNAME1
#define TTYNAME1 "/dev/ttys1"
#endif /*TTYNAME1*/


/* Globals */

static int 		fdSerial; 								/* file descriptor for the serial port */
static BRL_SER_CALLBACK	ClientCallback = NULL;		/*	client callback */

GIOChannel 		*gioch = NULL;						/* GLIB io channel */
gboolean		glib_poll = TRUE;

/*
extern int 		errno;										
*/
/* thread error no */


/* Functions */

/* ... */
int brl_ser_open_port (int Port)
{
	
	char *pname; /* port name */

	switch (Port)
	{
		case 1:
			pname = TTYNAME0; 	
		break;

		case 2:
			pname = TTYNAME1;
		break;

		default:
			fprintf (stderr, "brl_open_port: Invalid serial port number %d\n", Port);
/*			errno = -1;			*/
			return 0;
		break;
	}

	fdSerial = open(  pname,
										O_RDWR |		
                   	O_NOCTTY | 		/* this program doesn't want be the "controlling terminal" for the port */
                   	O_NDELAY |		/* doesn't care what hte state of the DCD signal line is in */
                   	O_NONBLOCK	/* !!! NEW !!!                   	 */
                   	);		
                   	
	if (fdSerial == -1)
	{
		fprintf (stderr, "brl_open_port: Unable to open the serial port (%s).\n", pname);
		return 0;
	}

	fcntl (fdSerial, F_SETFL, 0);
	
	/* fcntl (fdSerial, F_SETFL, O_NOBLOCK);                   	 */
	/* fcntl (fdSerial, F_SETFD, FD_CLOEXEC);                   	 */
	
	return 1;
}

/* ... */
int brl_ser_set_comm_param (long BaudRate, char Parity, short StopBits, char FlowCtrl)
{
	struct termios options;

	/* get the current options from the port */
	tcgetattr (fdSerial, &options);

	/* set the baud rate */
	switch (BaudRate)
	{
		case 9600:
			cfsetispeed (&options, B9600);		
			cfsetospeed (&options, B9600);		
		break;

		case 19200:
			cfsetispeed (&options, B19200);
			cfsetospeed (&options, B19200);
		break;

		case 38400:
			cfsetispeed (&options, B38400);		
			cfsetospeed (&options, B38400);		
		break;
#ifdef B57600
		case 57600:
			cfsetispeed (&options, B57600);		
			cfsetospeed (&options, B57600);		
		break;
#endif
#ifdef B115200
		case 115200:
			cfsetispeed (&options, B115200);		
			cfsetospeed (&options, B115200);		
		break;
#endif
		default:
			return 0;		
		break;	    		
	}

	/* set parity */
	switch (Parity)
	{
   		case 'N': 	case 'n': default:		/* none */
			options.c_cflag &= ~PARENB;	/* disable parity */
			options.c_iflag &=  ~INPCK;
		break;

		case 'E': case 'e':	/* even parity */
			options.c_cflag |= PARENB;		/* enable parity */
			options.c_cflag &= ~PARODD;	/* even */
			options.c_iflag |=  (INPCK | ISTRIP);	/* enable checking and stripping of the parityy bit */
		break;

		case 'O': case 'o':	/* odd parity */
			options.c_cflag &= ~PARENB;		/* mask parity bits						 */
			options.c_cflag |= PARODD;		/* odd */
			options.c_iflag |=  (INPCK | ISTRIP);	/* enable checking and stripping of the parityy bit */
		break;
	}
	
	/* set character size */
	options.c_cflag &= ~CSIZE;	/* mask character size bits */
	options.c_cflag |= CS8;			/* set the character size to 8 bit */


	/* set stop bits */
	if (StopBits < 2)
	{
		options.c_cflag &= ~CSTOPB;	/* 1 stop bit */
	}
	else
	{
		options.c_cflag |= CSTOPB;	/* 2 stop bit	!!! TBR !!! */
	}

	
	/* set flow control */

	switch (FlowCtrl)
	{
		case 'N': case 'n': default:
#ifdef CRTSCTS
			options.c_cflag &= ~CRTSCTS;	/* disable CTS/RTS flow control	 */
#endif
			options.c_iflag &= ~(IXON | IXOFF | IXANY);	/* disable XON/XOFF flow control */
		break;

		case 'H': case 'h':	/* hardware flow control */
#ifdef CRTSCTS
      options.c_cflag |= CRTSCTS;	/* enable CTS/RTS flow control */
#endif
			options.c_iflag &= ~(IXON | IXOFF | IXANY);	/* disable XON/XOFF flow control */
		break;

		case 'S': case 's':	/* software flow control */
#ifdef CRTSCTS
			options.c_cflag &= ~CRTSCTS;	/* disable CTS/RTS flow control	 */
#endif
			options.c_iflag |= IXON | IXOFF | IXANY;	/* enable XON/XOFF flow control */
		break;
	}

	/* choose raw input */
	options.c_lflag &= ~(	ICANON |	/* disable canonical input */
										ECHO |				/* disable echoing of input chars */
										ECHOE |   		/* disable echoing of erase characters as BS-SP-BS */
										ISIG);				/* disable the SIGINTR, SIGSUSP, SIGDSUSP and SIGQUIT signals */

	/* choose raw output */
	options.c_oflag &= ~OPOST;

      
	/* cfmakeraw (&options); 	 just for sure ;-) */
/*
  options.c_iflag &= ~(IGNBRK|BRKINT|PARMRK|ISTRIP|INLCR|IGNCR|ICRNL|IXON);
	options.c_iflag &= ~OPOST;
	options.c_iflag &= ~(ECHO|ECHONL|ICANON|ISIG|IEXTEN);
	options.c_iflag &= ~(CSIZE|PARENB);
	options.c_iflag |= CS8;
*/

	/* enable the receiver and set local mode*/
	options.c_cflag |= (CLOCAL | CREAD);

	/* timeouts */
	/* options.c_cc[VMIN] = 0; */
	/* options.c_cc[VTIME] = 10; */

	/* set the options for the port */
	tcsetattr (fdSerial, TCSANOW, &options);

	return 1;

}

/* ... */
int brl_ser_close_port ()
{
	close (fdSerial);
	return 1;
}

/* ... */
int brl_ser_send_data (char *Data, int DataSize, short Blocking)
{
	int n;

	n = write (fdSerial, Data, DataSize);
	if (Blocking) tcdrain(fdSerial);	
	
	if (n < 0)
  {
   	fprintf (stderr, "bra_ser_send_data: failed sending data\n");		
		return 0;
  }

	return 1;
}

/* ... */
int brl_ser_read_data (char *DataBuff, int MaxLen)
{
  /*	fcntl (fdSerial, F_SETFL, FNDELAY);  */
  fcntl (fdSerial, F_SETFL, O_NONBLOCK);
/* read function returns 0 if no characters are available on the port */
	return read (fdSerial, DataBuff, MaxLen);
}

/* ... */
void brl_ser_sig_alarm ( int sig )
{
    /* extern volatile int cnt; */
    /* cnt--; */
    /* if (!(cnt % 10)) fprintf(stderr,"%d...", cnt); */
	int n, i;
	unsigned char Data[256];
	n = brl_ser_read_data (&Data[0], 256);
	
	/* printf ("brl_ser_sig_alarm\n"); */
	
	for (i = 0; i < n; ++i)
	{
		/* fprintf (stderr, "%c", Data[i]);		 !!! TEST !!! */
		/* dispatch bytes to client callback if callback available */
		if (ClientCallback)
		{
			while (ClientCallback(Data[i]));
		}
	}
}

int brl_ser_start_timer (long Interval)
{
    /* struct sigaction sa = {brl_ser_sig_alarm, 0, 0, 0}; */
    struct sigaction sa = {{brl_ser_sig_alarm}};
    struct itimerval iv = {{0, 10000}, {0, 10000}}; /* default 10 ms periodic */

	if (Interval)
	{
		iv.it_value.tv_sec = 0;
		iv.it_value.tv_usec = Interval * 1000;
		iv.it_interval.tv_sec = 0;
		iv.it_interval.tv_usec = Interval * 1000;
	}

    sigaction (SIGALRM, &sa, NULL);
    setitimer (ITIMER_REAL, &iv, NULL);

	return 1;	/* !!! TBR !!! should depend on sigaction and setitimer returning success */
}

/*... */
int brl_ser_stop_timer (long Interval)
{
	
	struct itimerval iv = {{0, 0}, {0, 0}}; /* default 10 ms periodic	 */
	
	sigaction (SIGALRM, NULL, NULL);	/* disconnect callback !!! TBR !!! OK? */
	setitimer (ITIMER_REAL, &iv, NULL);  /* disable the timer after its next expiration */
	
	return 1;
}

/*... */
void brl_ser_set_callback (BRL_SER_CALLBACK Callback)
{
	ClientCallback = Callback;	
}

/* GLIB input polling */

/* ... */

static gboolean brl_ser_glib_cb (GIOChannel *source, GIOCondition condition, gpointer data)
{
	/* if (conditon & (G_IO_IN | G_IO_PRI)) */
			
	int n, i;
	unsigned char Data[256];
	
	if (!glib_poll) return FALSE;
	
	/* !!! TBR !!! - read until no more data available */
	n = brl_ser_read_data (&Data[0], 256);	/* !!! TBR !!! g_io_channel_read ??? */
	
	for (i = 0; i < n; ++i)
	{
		/* fprintf (stderr, "%c", Data[i]);		 !!! TEST !!! */
		/* dispatch bytes to client callback if callback available */
		if (ClientCallback)
		{
			while (ClientCallback(Data[i]));
		}
	}	
	return TRUE;	
}

/* ... */

int brl_ser_init_glib_poll ()
{
		
	/* printf ("brl_ser_init_glib_poll\n"); */
	glib_poll = TRUE;
	gioch = g_io_channel_unix_new (fdSerial);
	g_io_add_watch (gioch, G_IO_IN | G_IO_PRI, brl_ser_glib_cb, NULL);
	
	return TRUE; /* !!! TBR !!! */
	
}

int brl_ser_exit_glib_poll ()
{
	glib_poll = FALSE;
	if (gioch) g_io_channel_unref (gioch);
	return 0; /* FIXME */
}



