/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/**
 * gpdf bonobo control
 * nee eog-control.c
 *
 * Authors:
 *   Martin Baulig <baulig@suse.de> (eog-control.c)
 *   Martin Kretzschmar <Martin.Kretzschmar@inf.tu-dresden.de>
 *
 * Copyright 2000 SuSE GmbH.
 * Copyright 2002 Martin Kretzschmar
 */

#include <math.h>
#include <time.h>
#include "gpdf-control.h"
#include "gpdf-stock-icons.h"
#include "gpdf-view.h"
#include "gpdf-util.h"
#include "page-control.h"
#include "pdf-info-dict-util.h"
#include "pdf-properties-display.h"
#include <bonobo.h>

#define noPDF_DEBUG
#ifdef PDF_DEBUG
#  define DBG(x) x
#else
#  define DBG(x)
#endif

BEGIN_EXTERN_C

struct _GPdfControlPrivate {
	GPdfPersistStream *persist_stream;

	GtkWidget *scrolled_window;
	GtkWidget *gpdf_view;

	BonoboZoomable *zoomable;
	gboolean has_zoomable_frame;

	BonoboControl *bonobo_page_control;

	guint idle_id;
};

#define PARENT_TYPE BONOBO_TYPE_CONTROL
static BonoboControlClass *parent_class;

#define GPDF_IS_NON_NULL_CONTROL(obj) \
(((obj) != NULL) && (GPDF_IS_CONTROL ((obj))))


static void
gpdf_control_properties_dialog_response_cb (GtkDialog *dialog, gint response_id,
					    gpointer data)
{
	if (response_id == GTK_RESPONSE_CLOSE)
		gtk_widget_destroy (GTK_WIDGET (dialog));
}

static void
verb_FileProperties_cb (BonoboUIComponent *uic, gpointer user_data,
			const char *cname)
{
	GPdfPersistStream *persist_stream;
	GtkWidget *dialog;
	GtkWidget *props;

	g_return_if_fail (GPDF_IS_NON_NULL_CONTROL (user_data));

	dialog = gtk_dialog_new_with_buttons (_("PDF Properties"),
					      NULL,
					      (GtkDialogFlags)0,
					      GTK_STOCK_CLOSE,
					      GTK_RESPONSE_CLOSE,
					      NULL);
	g_signal_connect (
		GTK_WIDGET (dialog),
		"response",
		G_CALLBACK (gpdf_control_properties_dialog_response_cb),
		NULL);
	bonobo_control_set_transient_for (BONOBO_CONTROL (user_data),
					  GTK_WINDOW (dialog), NULL);

	persist_stream = GPDF_CONTROL (user_data)->priv->persist_stream;
	props = GTK_WIDGET (g_object_new (GPDF_TYPE_PROPERTIES_DISPLAY, NULL));
	pdf_doc_process_properties (
		gpdf_persist_stream_get_pdf_doc (persist_stream),
		gpdf_persist_stream_get_length (persist_stream),
		G_OBJECT (props));

	gtk_widget_show (props);
	gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), props);
	gtk_widget_show (dialog);
}


static void
verb_GoPagePrev_cb (BonoboUIComponent *uic, gpointer user_data, const char*cname)
{
	g_return_if_fail (GPDF_IS_NON_NULL_CONTROL (user_data));

	gpdf_view_page_prev (
		GPDF_VIEW (GPDF_CONTROL (user_data)->priv->gpdf_view));
}

static void
verb_GoPageNext_cb (BonoboUIComponent *uic, gpointer user_data, const char*cname)
{
	g_return_if_fail (GPDF_IS_NON_NULL_CONTROL (user_data));

	gpdf_view_page_next (
		GPDF_VIEW (GPDF_CONTROL (user_data)->priv->gpdf_view));
}

static void
verb_GoPageFirst_cb (BonoboUIComponent *uic, gpointer user_data, const char*cname)
{
	g_return_if_fail (GPDF_IS_NON_NULL_CONTROL (user_data));

	gpdf_view_page_first (
		GPDF_VIEW (GPDF_CONTROL (user_data)->priv->gpdf_view));
}

static void
verb_GoPageLast_cb (BonoboUIComponent *uic, gpointer user_data, const char*cname)
{
	g_return_if_fail (GPDF_IS_NON_NULL_CONTROL (user_data));

	gpdf_view_page_last (
		GPDF_VIEW (GPDF_CONTROL (user_data)->priv->gpdf_view));
}

static void
ui_component_set_sensitive (BonoboUIComponent *ui,
			    const char *path,
			    gboolean sensitive)
{
	bonobo_ui_component_set_prop (ui, path, "sensitive",
				      sensitive ? "1" : "0", NULL);
}

static void
gc_enable_go_back_verbs (GPdfControl *control, gboolean sensitive)
{
	BonoboUIComponent *ui;

	g_return_if_fail (GPDF_IS_NON_NULL_CONTROL (control));

	ui = bonobo_control_get_ui_component (BONOBO_CONTROL (control));
	if (ui == NULL)
		return;

	ui_component_set_sensitive (ui, "/commands/GoPageFirst", sensitive);
	ui_component_set_sensitive (ui, "/commands/GoPagePrev", sensitive);
}

static void
gc_enable_go_forward_verbs (GPdfControl *control, gboolean sensitive)
{
	BonoboUIComponent *ui;

	g_return_if_fail (GPDF_IS_NON_NULL_CONTROL (control));

	ui = bonobo_control_get_ui_component (BONOBO_CONTROL (control));
	if (ui == NULL)
		return;

	ui_component_set_sensitive (ui, "/commands/GoPageNext", sensitive);
	ui_component_set_sensitive (ui, "/commands/GoPageLast", sensitive);
}

static void
gc_page_changed_enable_page_buttons (GPdfView *gpdf_view, gint page,
				     GPdfControl *control)
{
	g_return_if_fail (GPDF_IS_NON_NULL_CONTROL (control));

	gpdf_view = GPDF_VIEW (control->priv->gpdf_view);

	gc_enable_go_back_verbs (control, !gpdf_view_is_first_page (gpdf_view));
	gc_enable_go_forward_verbs (control,
				    !gpdf_view_is_last_page (gpdf_view));
}



#define MAGSTEP   1.2
#define MAGSTEP2  MAGSTEP * MAGSTEP
#define MAGSTEP4  MAGSTEP2 * MAGSTEP2
#define IMAGSTEP  0.8333333333
#define IMAGSTEP2 IMAGSTEP * IMAGSTEP
#define IMAGSTEP4 IMAGSTEP2 * IMAGSTEP2

static float preferred_zoom_levels [] = {
	IMAGSTEP4 * IMAGSTEP4, IMAGSTEP4 * IMAGSTEP2 * IMAGSTEP,
	IMAGSTEP4 * IMAGSTEP2, IMAGSTEP4 * IMAGSTEP, IMAGSTEP4,
	IMAGSTEP2 * IMAGSTEP, IMAGSTEP2, IMAGSTEP,
	1.0,
	MAGSTEP, MAGSTEP2, MAGSTEP2 * MAGSTEP, MAGSTEP4,
	MAGSTEP4 * MAGSTEP, MAGSTEP4 * MAGSTEP2, MAGSTEP4 * MAGSTEP2 * MAGSTEP,
	MAGSTEP4 * MAGSTEP4
};

static const gchar *preferred_zoom_level_names [] = {
	"-8", "-7", "-6", "-5", "-4", "-3", "-2", "-1",
	"0",
	"+1", "+2", "+3", "+4", "+5", "+6", "+7", "+8"
};

static const gint n_zoom_levels = (sizeof (preferred_zoom_levels) /
				   sizeof (float));

void
gpdf_control_zoom_in (GPdfControl *gpdf_control)
{
	g_return_if_fail (GPDF_IS_NON_NULL_CONTROL (gpdf_control));

	gpdf_view_zoom_in (GPDF_VIEW (gpdf_control->priv->gpdf_view));
}

void
gpdf_control_zoom_out (GPdfControl *gpdf_control)
{
	g_return_if_fail (GPDF_IS_NON_NULL_CONTROL (gpdf_control));

	gpdf_view_zoom_out (GPDF_VIEW (gpdf_control->priv->gpdf_view));
}

/*
 * Idle handler to reset the scrollbar policy
 */
static gboolean
gc_set_policy_idle_cb (gpointer data)
{
	GPdfControlPrivate *priv;

	g_return_val_if_fail (GPDF_IS_NON_NULL_CONTROL (data), FALSE);

	priv = GPDF_CONTROL (data)->priv;

	priv->idle_id = 0;

	gtk_scrolled_window_set_policy (
		GTK_SCROLLED_WINDOW (priv->scrolled_window),
		GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

	return FALSE;
}

void
gpdf_control_zoom_fit_width (GPdfControl *gpdf_control)
{
	GPdfControlPrivate *priv;
	GtkScrolledWindow *scrolled_window;
	int w;

	g_return_if_fail (GPDF_IS_NON_NULL_CONTROL (gpdf_control));

	priv = gpdf_control->priv;

	scrolled_window = GTK_SCROLLED_WINDOW (priv->scrolled_window);

	w = GTK_WIDGET (scrolled_window)->allocation.width;
	/* FIXME this doesn't look right to me */
	w -= GTK_WIDGET (scrolled_window->vscrollbar)->allocation.width;
	w -= 2 * GTK_WIDGET (scrolled_window)->style->xthickness;

	gtk_scrolled_window_set_policy (scrolled_window,
					GTK_POLICY_NEVER,
					GTK_POLICY_AUTOMATIC);

	gpdf_view_zoom_fit_width (GPDF_VIEW (priv->gpdf_view), w);

	if (!priv->idle_id)
		priv->idle_id = g_idle_add (gc_set_policy_idle_cb,
					    gpdf_control);
}

void
gpdf_control_zoom_fit (GPdfControl *gpdf_control)
{
	GPdfControlPrivate *priv;
	GtkScrolledWindow *scrolled_window;
	int h, w;

	g_return_if_fail (GPDF_IS_NON_NULL_CONTROL (gpdf_control));

	priv = gpdf_control->priv;

	scrolled_window = GTK_SCROLLED_WINDOW (priv->scrolled_window);

	h = GTK_WIDGET (scrolled_window)->allocation.height;
	h -= 2 * GTK_WIDGET (scrolled_window)->style->ythickness;

	w = GTK_WIDGET (scrolled_window)->allocation.width;
	w -= 2 * GTK_WIDGET (scrolled_window)->style->xthickness;

	gtk_scrolled_window_set_policy (scrolled_window,
					GTK_POLICY_NEVER,
					GTK_POLICY_NEVER);

	gpdf_view_zoom_fit (GPDF_VIEW (priv->gpdf_view), w, h);

	if (!priv->idle_id)
		priv->idle_id = g_idle_add (gc_set_policy_idle_cb,
					    gpdf_control);
}

void
gpdf_control_zoom_default (GPdfControl *gpdf_control)
{
	g_return_if_fail (GPDF_IS_NON_NULL_CONTROL (gpdf_control));

	gpdf_view_zoom_default (GPDF_VIEW (gpdf_control->priv->gpdf_view));
}

static void
verb_ZoomIn_cb (BonoboUIComponent *uic, gpointer user_data, const char *cname)
{
	gpdf_control_zoom_in (GPDF_CONTROL (user_data));
}

static void
verb_ZoomOut_cb (BonoboUIComponent *uic, gpointer user_data, const char *cname)
{
	gpdf_control_zoom_out (GPDF_CONTROL (user_data));
}

static void
verb_ZoomFit_cb (BonoboUIComponent *uic, gpointer user_data, const char *cname)
{
	gpdf_control_zoom_fit (GPDF_CONTROL (user_data));
}

static void
verb_ZoomFitWidth_cb (BonoboUIComponent *uic, gpointer user_data,
		      const char *cname)
{
	gpdf_control_zoom_fit_width (GPDF_CONTROL (user_data));
}

static void
gc_zoom_in_cb (GtkObject *source, gpointer data)
{
	gpdf_control_zoom_in (GPDF_CONTROL (data));
}

static void
gc_zoom_out_cb (GtkObject *source, gpointer data)
{
	gpdf_control_zoom_out (GPDF_CONTROL (data));
}

static void
gc_zoom_to_fit_cb (GtkObject *source, gpointer data)
{
	gpdf_control_zoom_fit (GPDF_CONTROL (data));
}

static void
gc_zoom_to_fit_width_cb (GtkObject *source, gpointer data)
{
	gpdf_control_zoom_fit_width (GPDF_CONTROL (data));
}

static void
gc_zoom_to_default_cb (GtkObject *source, gpointer data)
{
	gpdf_control_zoom_default (GPDF_CONTROL (data));
}

#define GDOUBLE_FROM_CORBA_FLOAT(val) (gdouble)val
#define CORBA_FLOAT_FROM_GDOUBLE(val) (CORBA_float)val
static void
gc_zoomable_set_zoom_level_cb (BonoboZoomable *zoomable,
			       CORBA_float new_zoom_level,
			       GPdfControl *control)
{
	g_return_if_fail (GPDF_IS_NON_NULL_CONTROL (control));

	gpdf_view_zoom (GPDF_VIEW (control->priv->gpdf_view),
			GDOUBLE_FROM_CORBA_FLOAT (new_zoom_level),
			FALSE);
}

static void
gc_zoomable_set_frame_cb (BonoboZoomable *zoomable, GPdfControl *control)
{
	g_return_if_fail (GPDF_IS_NON_NULL_CONTROL (control));

	control->priv->has_zoomable_frame = TRUE;
}


static void
gc_zoom_changed_report_to_zoomable (GPdfView *gpdf_view, gdouble zoom,
				    GPdfControl *control)
{
	g_return_if_fail (GPDF_IS_NON_NULL_CONTROL (control));

	bonobo_zoomable_report_zoom_level_changed (
		control->priv->zoomable,
		CORBA_FLOAT_FROM_GDOUBLE (zoom),
		NULL);
}

static void
gc_set_page_cb (GPdfPageControl *page_control, gint page, GPdfControl *control)
{
	g_return_if_fail (GPDF_IS_NON_NULL_CONTROL (control));

	gpdf_view_goto_page (GPDF_VIEW (control->priv->gpdf_view), page);
}

static void
gc_page_changed_update_page_control (GPdfView *gpdf_view, gint page,
				     GPdfControl *control)
{
	GPdfPageControl *pc;

	g_return_if_fail (GPDF_IS_NON_NULL_CONTROL (control));

	pc = GPDF_PAGE_CONTROL (
		bonobo_control_get_widget (control->priv->bonobo_page_control));
	gpdf_page_control_set_page (pc, page);
}

static void
ui_component_set_hidden (BonoboUIComponent *ui,
			 const char *path,
			 gboolean hidden)
{
	bonobo_ui_component_set_prop (ui, path, "hidden",
				      hidden ? "1" : "0", NULL);
}

static void
gc_set_zoom_items_visibility (GPdfControl *control, gboolean visible)
{
	BonoboUIComponent *ui;

	g_return_if_fail (GPDF_IS_NON_NULL_CONTROL (control));

	ui = bonobo_control_get_ui_component (BONOBO_CONTROL (control));
	if (ui == NULL)
		return;

	ui_component_set_hidden (ui, "/commands/ZoomIn", !visible);
	ui_component_set_hidden (ui, "/commands/ZoomOut", !visible);
	ui_component_set_hidden (ui, "/commands/ZoomFit", !visible);
}


static BonoboUIVerb gc_verbs [] = {
	BONOBO_UI_VERB ("FileProperties", verb_FileProperties_cb),

	BONOBO_UI_VERB ("GoPagePrev",     verb_GoPagePrev_cb),
	BONOBO_UI_VERB ("GoPageNext",     verb_GoPageNext_cb),
	BONOBO_UI_VERB ("GoPageFirst",    verb_GoPageFirst_cb),
	BONOBO_UI_VERB ("GoPageLast",     verb_GoPageLast_cb),

	BONOBO_UI_VERB ("ZoomIn",         verb_ZoomIn_cb),
	BONOBO_UI_VERB ("ZoomOut",        verb_ZoomOut_cb),
	BONOBO_UI_VERB ("ZoomFit",        verb_ZoomFit_cb),
	BONOBO_UI_VERB ("ZoomFitWidth",   verb_ZoomFitWidth_cb),

	BONOBO_UI_VERB_END
};

static void
gc_ui_set_pixbufs (GPdfControl *control)
{
	GtkWidget *gpdf_view;
	BonoboUIComponent *ui_component;
	GtkIconSet *icon_set;
	GdkPixbuf *pixbuf, *menu_pixbuf;
	gchar *pixbuf_xml, *menu_pixbuf_xml;
	gint menu_width, menu_height;

	gpdf_view = control->priv->gpdf_view;

	ui_component =
		bonobo_control_get_ui_component (BONOBO_CONTROL (control));
	if (ui_component == NULL)
		return;

	icon_set = gtk_style_lookup_icon_set (gtk_widget_get_style (gpdf_view),
					      GPDF_STOCK_ZOOM_FIT_WIDTH);
	if (icon_set == NULL)
		return;

	pixbuf = gtk_icon_set_render_icon (icon_set,
					   gtk_widget_get_style (gpdf_view),
					   gtk_widget_get_direction (gpdf_view),
					   GTK_STATE_NORMAL,
					   GTK_ICON_SIZE_LARGE_TOOLBAR,
					   NULL, NULL);

	bonobo_ui_util_set_pixbuf (ui_component,
				   "/Toolbar/ZoomFitWidth",
				   pixbuf,
				   NULL);

	menu_pixbuf = gtk_icon_set_render_icon (
		icon_set,
		gtk_widget_get_style (gpdf_view),
		gtk_widget_get_direction (gpdf_view),
		GTK_STATE_NORMAL,
		GTK_ICON_SIZE_MENU,
		NULL, NULL);

	bonobo_ui_util_set_pixbuf (
		ui_component,
		"/menu/View/Zoom Items Placeholder/ZoomFitWidth",
		menu_pixbuf,
		NULL);

	gdk_pixbuf_unref (pixbuf);
}

static void
gc_ui_add_page_control (GPdfControl *control)
{
	BonoboControl *bonobo_page_control;
	BonoboUIComponent *ui_component;

	bonobo_page_control = control->priv->bonobo_page_control;

	ui_component =
		bonobo_control_get_ui_component (BONOBO_CONTROL (control));
	bonobo_ui_component_object_set (
		ui_component,
		"/Toolbar/Page Control Placeholder/PageControl",
		BONOBO_OBJREF (bonobo_page_control),
		NULL);
}

static void
gc_set_ui_container (GPdfControl *control, Bonobo_UIContainer ui_container)
{
	BonoboUIComponent *ui_component;

	g_return_if_fail (GPDF_IS_NON_NULL_CONTROL (control));
	g_return_if_fail (ui_container != CORBA_OBJECT_NIL);

	ui_component = bonobo_control_get_ui_component (
		BONOBO_CONTROL (control));
	g_assert (ui_component != NULL);

	bonobo_ui_component_set_container (ui_component, ui_container, NULL);
	bonobo_ui_component_add_verb_list_with_data (ui_component, gc_verbs,
						     control);

	bonobo_ui_util_set_ui (ui_component, DATADIR,
			       "gpdf-control-ui.xml", "GPDF", NULL);
	gc_ui_set_pixbufs (control);
	gc_ui_add_page_control (control);
	gc_page_changed_update_page_control (
		GPDF_VIEW (control->priv->gpdf_view), 1, control);
	gc_set_zoom_items_visibility (control,
				      !control->priv->has_zoomable_frame);
}

static void
gc_unset_ui_container (GPdfControl *control)
{
	BonoboUIComponent *ui_component;
	GPdfControlPrivate *priv;

	g_return_if_fail (GPDF_IS_NON_NULL_CONTROL (control));

	ui_component = bonobo_control_get_ui_component (
	     BONOBO_CONTROL (control));
	g_assert (ui_component != NULL);

	bonobo_ui_component_unset_container (ui_component, NULL);

	priv = control->priv;
}

static void
gpdf_control_activate (BonoboControl *control, gboolean activate)
{
	GPdfControl *gpdf_control = GPDF_CONTROL (control);

	g_return_if_fail (GPDF_IS_NON_NULL_CONTROL (control));

	if (activate) {
		Bonobo_UIContainer ui_container =
			bonobo_control_get_remote_ui_container (control, NULL);

		if (ui_container != CORBA_OBJECT_NIL) {
			gc_set_ui_container (gpdf_control, ui_container);
			bonobo_object_release_unref (ui_container, NULL);
		}
	} else {
		gc_unset_ui_container (gpdf_control);
	}

	BONOBO_CALL_PARENT (
		BONOBO_CONTROL_CLASS, activate, (control, activate));
}

static void
gpdf_control_destroy (BonoboObject *object)
{
	GPdfControlPrivate *priv;

	g_return_if_fail (GPDF_IS_NON_NULL_CONTROL (object));

	DBG (g_message ("Destroying GPdfControl"));

	priv = GPDF_CONTROL (object)->priv;

	if (priv->bonobo_page_control) {
		bonobo_object_unref (BONOBO_OBJECT (priv->bonobo_page_control));
		priv->bonobo_page_control = NULL;
	}

	BONOBO_CALL_PARENT (BONOBO_OBJECT_CLASS, destroy, (object));
}

static void
gpdf_control_finalize (GObject *object)
{
	g_return_if_fail (GPDF_IS_NON_NULL_CONTROL (object));

	g_free ((GPDF_CONTROL (object))->priv);

	BONOBO_CALL_PARENT (G_OBJECT_CLASS, finalize, (object));
}

static void
gpdf_control_class_init (GPdfControlClass *klass)
{
	parent_class = BONOBO_CONTROL_CLASS (gtk_type_class (PARENT_TYPE));

	G_OBJECT_CLASS (klass)->finalize = gpdf_control_finalize;
	BONOBO_OBJECT_CLASS (klass)->destroy = gpdf_control_destroy;
	BONOBO_CONTROL_CLASS (klass)->activate = gpdf_control_activate;
}

static void
gpdf_control_init (GPdfControl *control)
{
	control->priv = g_new0 (GPdfControlPrivate, 1);
}

static void
gc_set_pdf_cb (GPdfPersistStream *persist_stream, gpointer user_data)
{
	PDFDoc *pdf_doc = NULL;
	GPdfControlPrivate *priv;
	GPdfView *gpdf_view;
	GPdfPageControl *page_control;

	g_return_if_fail (persist_stream != NULL);
	g_return_if_fail (GPDF_IS_PERSIST_STREAM (persist_stream));
	g_return_if_fail (GPDF_IS_NON_NULL_CONTROL (user_data));

	pdf_doc = gpdf_persist_stream_get_pdf_doc (persist_stream);

	priv = GPDF_CONTROL (user_data)->priv;
	gpdf_view = GPDF_VIEW (priv->gpdf_view);
	page_control = GPDF_PAGE_CONTROL (
		bonobo_control_get_widget (priv->bonobo_page_control));
	if (pdf_doc) {
		gpdf_view_set_pdf_doc (gpdf_view, pdf_doc);
		gpdf_page_control_set_total_pages (page_control,
						   pdf_doc->getNumPages ());
	} else {
		gpdf_page_control_set_total_pages (page_control, -1);
	}
}

static void
gc_setup_persist (GPdfControl *control, GPdfPersistStream *persist_stream)
{
	control->priv->persist_stream = persist_stream;

	g_signal_connect (G_OBJECT (persist_stream),
			  "set_pdf",
			  G_CALLBACK (gc_set_pdf_cb),
			  control);

	bonobo_object_ref (BONOBO_OBJECT (persist_stream));
	bonobo_object_add_interface (BONOBO_OBJECT (control),
				     BONOBO_OBJECT (persist_stream));
}

static void
gc_setup_view_widgets (GPdfControl *control, GPdfPersistStream *persist_stream)
{
	GtkWidget *sw;
	GtkWidget *gpdf_view;

	sw = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);

	gpdf_view = gpdf_view_new ();
	control->priv->gpdf_view = gpdf_view;
	gtk_container_add (GTK_CONTAINER (sw), control->priv->gpdf_view);
	control->priv->scrolled_window = sw;

	g_signal_connect (G_OBJECT (gpdf_view), "page_changed",
			  G_CALLBACK (gc_page_changed_enable_page_buttons),
			  control);

	gtk_widget_show_all (sw);
	gtk_widget_grab_focus (GTK_WIDGET (control->priv->gpdf_view));
}

static void
gc_setup_zoomable (GPdfControl *control)
{
	BonoboZoomable *zoomable = bonobo_zoomable_new ();

	control->priv->zoomable = zoomable;

	g_signal_connect (G_OBJECT (zoomable), "set_frame",
			  G_CALLBACK (gc_zoomable_set_frame_cb),
			  control);
	g_signal_connect (G_OBJECT (zoomable), "set_zoom_level",
			  G_CALLBACK (gc_zoomable_set_zoom_level_cb),
			  control);
	g_signal_connect (G_OBJECT (zoomable), "zoom_in",
			  G_CALLBACK (gc_zoom_in_cb),
			  control);
	g_signal_connect (G_OBJECT (zoomable), "zoom_out",
			  G_CALLBACK (gc_zoom_out_cb),
			  control);
	g_signal_connect (G_OBJECT (zoomable), "zoom_to_fit",
			  G_CALLBACK (gc_zoom_to_fit_cb),
			  control);
	g_signal_connect (G_OBJECT (zoomable), "zoom_to_default",
			  G_CALLBACK (gc_zoom_to_default_cb),
			  control);

	g_signal_connect (G_OBJECT (control->priv->gpdf_view), "zoom_changed",
			  G_CALLBACK (gc_zoom_changed_report_to_zoomable),
			  control);

	bonobo_zoomable_set_parameters_full (
		control->priv->zoomable,
		1.0,
		preferred_zoom_levels [0],
		preferred_zoom_levels [n_zoom_levels - 1],
		TRUE, TRUE, TRUE,
		preferred_zoom_levels,
		preferred_zoom_level_names,
		n_zoom_levels);

	bonobo_object_add_interface (BONOBO_OBJECT (control),
				     BONOBO_OBJECT (zoomable));
}

static void
gc_setup_page_control (GPdfControl *control)
{
	GtkWidget *page_control;
	BonoboControl *bonobo_page_control;

	page_control = GTK_WIDGET (g_object_new (GPDF_TYPE_PAGE_CONTROL, NULL));

	g_signal_connect (G_OBJECT (page_control), "set_page",
			  G_CALLBACK (gc_set_page_cb),
			  control);

	g_signal_connect (G_OBJECT (control->priv->gpdf_view), "page_changed",
			  G_CALLBACK (gc_page_changed_update_page_control),
			  control);

	gtk_widget_show (page_control);
	bonobo_page_control = bonobo_control_new (page_control);

	control->priv->bonobo_page_control = bonobo_page_control;
}

GPdfControl *
gpdf_control_construct (GPdfControl *control,
			GPdfPersistStream *persist_stream)
{
	g_return_val_if_fail (GPDF_IS_NON_NULL_CONTROL (control), NULL);
	g_return_val_if_fail (persist_stream != NULL, NULL);
	g_return_val_if_fail (GPDF_IS_PERSIST_STREAM (persist_stream), NULL);

	gc_setup_persist (control, persist_stream);
	gc_setup_view_widgets (control, persist_stream);
	gc_setup_zoomable (control);
	gc_setup_page_control (control);
	bonobo_control_construct (BONOBO_CONTROL (control),
				  control->priv->scrolled_window);

	return control;
}

GPdfControl *
gpdf_control_new (GPdfPersistStream *gpdf_persist_stream)
{
	GPdfControl *control;

	g_return_val_if_fail (gpdf_persist_stream != NULL, NULL);
	g_return_val_if_fail (GPDF_IS_PERSIST_STREAM (gpdf_persist_stream),
			      NULL);

	DBG (g_message ("Creating GPdfControl"));

	control = GPDF_CONTROL (g_object_new (GPDF_TYPE_CONTROL, NULL));

	return gpdf_control_construct (control, gpdf_persist_stream);
}

BONOBO_TYPE_FUNC (GPdfControl, PARENT_TYPE, gpdf_control)

END_EXTERN_C
