// Alphabet.cpp
//
// Copyright (c) 2008 The Dasher Team
//
// This file is part of Dasher.
//
// Dasher is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Dasher is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Dasher; if not, write to the Free Software 
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

#include "../../Common/Common.h"
#include "../TrainingHelper.h"
#include "Alphabet.h"
#include "../AlphabetManagerFactory.h"
#include "AlphabetMap.h"


using namespace Dasher;
using namespace std;

// Track memory leaks on Windows to the line that new'd the memory
#ifdef _WIN32
#ifdef _DEBUG
#define DEBUG_NEW new( _NORMAL_BLOCK, THIS_FILE, __LINE__ )
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif
#endif

/////////////////////////////////////////////////////////////////////////////

CAlphabet::CAlphabet()
:m_DefaultEncoding(Opts::Western), m_Orientation(Opts::LeftToRight), m_ControlSymbol(-1) {
  m_Characters.push_back("");
  m_Display.push_back("");
  m_Colours.push_back(-1);
  m_Foreground.push_back("");

  m_pTrainingHelper = new CTrainingHelper;
}

/////////////////////////////////////////////////////////////////////////////

CAlphabet::CAlphabet(const CAlphIO::AlphInfo &AlphInfo)
:m_DefaultEncoding(Opts::Western), m_Orientation(Opts::LeftToRight), m_ControlSymbol(-1) {
  m_Characters.push_back("");
  m_Display.push_back("");
  m_Colours.push_back(-1);
  m_Foreground.push_back("");

  m_StartConversionSymbol = -1;
  m_EndConversionSymbol = -1;

  m_pTrainingHelper = NULL;

  m_strDefaultContext = AlphInfo.m_strDefaultContext;

  // Set miscellaneous options

  SetOrientation(AlphInfo.Orientation);
  SetLanguage(AlphInfo.Type);
  SetTrainingFile(AlphInfo.TrainingFile);
  SetGameModeFile(AlphInfo.GameModeFile);
  SetPalette(AlphInfo.PreferredColours);

  for(std::vector<CAlphIO::AlphInfo::character>::const_iterator it(AlphInfo.m_vCharacters.begin()); it != AlphInfo.m_vCharacters.end(); ++it)
    AddChar(it->Text, it->Display, it->Colour, it->Foreground);


  // TODO: Special characters are a mess - really need to think these through

  // Set Space character if requested

  // This line makes it a bit easier for our WindowsCE compiler
  std::string empty = "";

  if(AlphInfo.ParagraphCharacter.Text != empty)
    AddParagraphSymbol(AlphInfo.ParagraphCharacter.Text, AlphInfo.ParagraphCharacter.Display, AlphInfo.ParagraphCharacter.Colour, AlphInfo.ParagraphCharacter.Foreground);

  if(AlphInfo.SpaceCharacter.Text != empty)
    AddSpaceSymbol(AlphInfo.SpaceCharacter.Text, AlphInfo.SpaceCharacter.Display, AlphInfo.SpaceCharacter.Colour, AlphInfo.SpaceCharacter.Foreground);

  //-- Added for Kanji Conversion 13 July 2005 by T.Kaburagi START
  if(AlphInfo.StartConvertCharacter.Text != empty)
    AddStartConversionSymbol(AlphInfo.StartConvertCharacter.Text, AlphInfo.StartConvertCharacter.Display, AlphInfo.StartConvertCharacter.Colour, AlphInfo.StartConvertCharacter.Foreground);

  if(AlphInfo.EndConvertCharacter.Text != empty)
    AddEndConversionSymbol(AlphInfo.EndConvertCharacter.Text, AlphInfo.EndConvertCharacter.Display, AlphInfo.EndConvertCharacter.Colour, AlphInfo.EndConvertCharacter.Foreground);
  //-- Added for Kanji Conversion 13 July 2005 by T.Kaburagi END

  // DJW - now the control symbol is always a part of the alphabet
  // DasherModel knows whether or not to use it

  // FIXME - We really need to ensure that the control symbol is last in the alphabet with the current logic.

  if(AlphInfo.ControlCharacter.Display != std::string("") && GetControlSymbol() == -1)
    AddControlSymbol(AlphInfo.ControlCharacter.Text, AlphInfo.ControlCharacter.Display, AlphInfo.ControlCharacter.Colour, AlphInfo.ControlCharacter.Foreground);



  // New group stuff

  m_pBaseGroup = AlphInfo.m_pBaseGroup;

#ifdef DASHER_TRACE
  Trace();
#endif
}

CAlphabet::~CAlphabet() {
  delete m_pTrainingHelper;
}

/////////////////////////////////////////////////////////////////////////////

void CAlphabet::GetSymbols(std::vector<symbol > *Symbols, std::string *Input, bool IsMore) const {
  string Tmp;
  symbol CurSymbol = 0, TmpSymbol = 0;
  bool KeyIsPrefix = false;
  int extras;
  unsigned int bit;

  for(unsigned int i = 0; i < Input->size(); i++) {

    Tmp = (*Input)[i];

    /* The string we've been given is in UTF-8. The symbols are
       also in UTF-8, so we need to pass the entire UTF-8 character
       which may be several bytes long. RFC 2279 describes this
       encoding */

    if((*Input)[i] & 0x80) {    // Character is more than 1 byte long
      extras = 1;
      for(bit = 0x20; ((*Input)[i] & bit) != 0; bit >>= 1)
        extras++;
      if(extras > 5) {
      }                         // Malformed character
      while(extras-- > 0) {
        Tmp += (*Input)[++i];
      }
    }

    CurSymbol = TextMap.Get(Tmp, &KeyIsPrefix);

    if(KeyIsPrefix) {
      CurSymbol = 0;
      for(; i < Input->size(); i++) {

        Tmp += (*Input)[i];

        TmpSymbol = TextMap.Get(Tmp, &KeyIsPrefix);
        if(TmpSymbol > 0) {
          CurSymbol = TmpSymbol;
        }
        if(!KeyIsPrefix) {
          if(CurSymbol != 0) {
            Symbols->push_back(CurSymbol);
          }
          else {
            i -= Tmp.size() - 1;
            //Tmp.erase(Tmp.begin(), Tmp.end());
            Tmp = "";
          }
          break;
        }
      }
    }
    else {
      if(CurSymbol != 0)
        Symbols->push_back(CurSymbol);
    }
  }

  if(IsMore)
    if(KeyIsPrefix)
      *Input = Tmp;
    else
      *Input = "";
  else if(KeyIsPrefix)
    Symbols->push_back(CurSymbol);
}

void CAlphabet::GetSymbolsFull(std::vector<symbol > *Symbols, std::string *Input) const {
 
  std::string::iterator it = Input->begin();

  while(it != Input->end()) {
    unsigned char c = static_cast<unsigned char>(*it);

    int iNBytes;

    if(c <= 0x7F)
      iNBytes = 1;
    else if((c >= 0xC2) && (c <= 0xDF))
      iNBytes = 2;
    else if((c >= 0xE0) && (c <= 0xEF))
      iNBytes = 3;
    else if((c >= 0xF0) && (c <= 0xF4))
      iNBytes = 4;
    else {
      // TODO: Error condition - handle this.
      iNBytes = 1;
    }

    std::string strCurrentSymbol(1, *it);

    for(int i = 0; i < iNBytes - 1; ++i) {
      ++it;
      strCurrentSymbol += *it;
    }

    // TODO: Error condition on reaching end of string prematurely.

    Symbols->push_back(TextMap.Get(strCurrentSymbol, NULL));

    ++it;
  }
}

// add single char to the character set
void CAlphabet::AddChar(const std::string NewCharacter, const std::string Display, int Colour, const std::string Foreground) {
  m_Characters.push_back(NewCharacter);
  m_Display.push_back(Display);
  m_Colours.push_back(Colour);
  m_Foreground.push_back(Foreground);

  symbol ThisSymbol = m_Characters.size() - 1;
  TextMap.Add(NewCharacter, ThisSymbol);
}

/////////////////////////////////////////////////////////////////////////////

void CAlphabet::AddParagraphSymbol(const std::string NewCharacter, const std::string Display, int Colour, const std::string Foreground) {
  AddChar(NewCharacter, Display, Colour, Foreground);
  m_ParagraphSymbol = GetNumberSymbols() - 1;
}

/////////////////////////////////////////////////////////////////////////////

void CAlphabet::AddSpaceSymbol(const std::string NewCharacter, const std::string Display, int Colour, const std::string Foreground) {
  AddChar(NewCharacter, Display, Colour, Foreground);
  m_SpaceSymbol = GetNumberSymbols() - 1;
}

/////////////////////////////////////////////////////////////////////////////

void CAlphabet::AddControlSymbol(const std::string NewCharacter, const std::string Display, int Colour, const std::string Foreground) {
  AddChar(NewCharacter, Display, Colour, Foreground);
  m_ControlSymbol = GetNumberSymbols() - 1;
}

/////////////////////////////////////////////////////////////////////////////

void CAlphabet::AddStartConversionSymbol(const std::string NewCharacter, const std::string Display, int Colour, const std::string Foreground) {
  AddChar(NewCharacter, Display, Colour, Foreground);
  m_StartConversionSymbol = GetNumberSymbols() - 1;
}

/////////////////////////////////////////////////////////////////////////////

void CAlphabet::AddEndConversionSymbol(const std::string NewCharacter, const std::string Display, int Colour, const std::string Foreground) {
  AddChar(NewCharacter, Display, Colour, Foreground);
  m_EndConversionSymbol = GetNumberSymbols() - 1;
}


/////////////////////////////////////////////////////////////////////////////
// diagnostic dump of character set

void CAlphabet::Trace() const {
//   int i;
//   DASHER_TRACEOUTPUT("GetNumberSymbols() %d\n", GetNumberSymbols());
//   DASHER_TRACEOUTPUT("GetNumberTextSymbols() %d\n", GetNumberTextSymbols());

//   int iGroup = 0;
//   for(i = 0; i < (int) m_Characters.size(); i++) {
//     if(iGroup < m_iGroups && i == m_GroupStart[iGroup]) {
//       DASHER_TRACEOUTPUT("Group %d '%s'\n", iGroup, m_GroupLabel[iGroup].c_str());
//     }
//     if(iGroup < m_iGroups && i == m_GroupEnd[iGroup]) {
//       DASHER_TRACEOUTPUT("--------\n");
//       iGroup++;
//     }

//     DASHER_TRACEOUTPUT("Symbol %d Character:'%s' Display:'%s'\n", i, m_Characters[i].c_str(), m_Display[i].c_str());

//   }

}

/////////////////////////////////////////////////////////////////////////////

int CAlphabet::GetTextColour(symbol Symbol) {
  std::string TextColour = m_Foreground[Symbol];
  if(TextColour != std::string("")) {
    return atoi(TextColour.c_str());
  }
  else {
    return 4;
  }
}


void 
CAlphabet::Train(const std::string &strUserLoc, 
		 const std::string &strSystemLoc, 
		 CTrainer *pTrainer) {

  std::string strTrainingFile = GetTrainingFile();

  std::string strUserPathFull = strUserLoc + strTrainingFile;

  m_pTrainingHelper->LoadFile(strUserPathFull, pTrainer, this);
  m_pTrainingHelper->LoadFile(strSystemLoc + strTrainingFile, pTrainer, this);
}

void 
CAlphabet::Train(const std::string &strPath, 
      CTrainer *pTrainer) {
  
  m_pTrainingHelper->LoadFile(strPath, pTrainer, this);
}

int 
CAlphabet::GetColour(symbol i, int iPhase) const {
  int iColour = m_Colours[i];
  
  // This is for backwards compatibility with old alphabet files -
  // ideally make this log a warning (unrelated TODO: automate
  // validation of alphabet files, plus maintenance of repository
  // etc.)
  if(iColour == -1) {
    if(i == m_SpaceSymbol) {
      iColour = 9;
    }
    else {
      iColour = (i % 3) + 10;
    }
  }
  
  // Loop on low colours for nodes (TODO: go back to colour namespaces?)
  if(iPhase == 1 && iColour < 130)
    iColour += 130;
  
  return iColour;
} 

