/* -*- mode:C++; tab-width:8; c-basic-offset:8; indent-tabs-mode:true -*- */

/*
 * written by J. Marcin Gorycki <marcin.gorycki@intel.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * For more details see the file COPYING.
 */

#include "tetris.h"
#include "field.h"
#include "blockops.h"
#include "blocks.h"
#include "preview.h"
#include "scoreframe.h"
#include "sound.h"
#include "highscores.h"
#include "renderer.h"

#include <games-gconf.h>
#include <games-frame.h>
#include <games-controls.h>
#include <games-stock.h>

#include <libgnomevfs/gnome-vfs.h>

#include <gdk/gdkkeysyms.h>
#include <config.h>
#include <dirent.h>
#include <string.h>
#include <math.h>

int LINES = 20;
int COLUMNS = 11;

int BLOCK_SIZE = 40;

int blocknr_next = -1;
int rot_next = -1;
int color_next = -1;

bool random_block_colors = false;
bool do_preview = true;
bool rotateCounterClockWise = true;

#define TETRIS_OBJECT "gnometris-tetris-object"

#define KEY_BASE "/apps/gnometris"

#define KEY_OPTIONS_DIR KEY_BASE "/options"

#define KEY_SOUND KEY_OPTIONS_DIR "/sound"
#define KEY_BLOCK_PIXMAP KEY_OPTIONS_DIR "/block_pixmap"
#define KEY_THEME KEY_OPTIONS_DIR "/theme"
#define KEY_STARTING_LEVEL KEY_OPTIONS_DIR "/starting_level"
#define KEY_DO_PREVIEW KEY_OPTIONS_DIR "/do_preview"
#define KEY_USE_TARGET KEY_OPTIONS_DIR "/use_target"
#define KEY_RANDOM_BLOCK_COLORS KEY_OPTIONS_DIR "/random_block_colors"
#define KEY_ROTATE_COUNTER_CLOCKWISE KEY_OPTIONS_DIR "/rotate_counter_clock_wise"
#define KEY_LINE_FILL_HEIGHT KEY_OPTIONS_DIR "/line_fill_height"
#define KEY_LINE_FILL_PROBABILITY KEY_OPTIONS_DIR "/line_fill_probability"

#define KEY_CONTROLS_DIR KEY_BASE "/controls"
#define KEY_MOVE_LEFT KEY_CONTROLS_DIR "/key_left"
#define KEY_MOVE_RIGHT KEY_CONTROLS_DIR "/key_right"
#define KEY_MOVE_DOWN KEY_CONTROLS_DIR "/key_down"
#define KEY_MOVE_DROP KEY_CONTROLS_DIR "/key_drop"
#define KEY_MOVE_ROTATE KEY_CONTROLS_DIR "/key_rotate"
#define KEY_MOVE_PAUSE KEY_CONTROLS_DIR "/key_pause"

#define KEY_BG_COLOUR KEY_OPTIONS_DIR "/bgcolor"
#define KEY_USE_BG_IMAGE KEY_OPTIONS_DIR "/usebgimage"

#define KEY_HEIGHT KEY_OPTIONS_DIR "/height"
#define KEY_WIDTH KEY_OPTIONS_DIR "/width"

#define TILE_THRESHOLD 65

enum {
	URI_LIST,
	TEXT_PLAIN,
	COLOUR,
	RESET
};


Tetris::Tetris(int cmdlLevel): 
	themeno (0),
	field(0),
	paused(false), 
	timeoutId(0), 
	onePause(false), 
	inPlay(false),
	useTarget(false),
	bgimage(0),
	setupdialog(0), 
	cmdlineLevel(cmdlLevel), 
	fastFall(false),
        dropBlock(false)
{
	GtkUIManager *ui_manager;
	GtkAccelGroup *accel_group;
	GtkActionGroup *action_group;
	GtkWidget *vbox;
	GtkWidget *aspect_frame;
	GtkWidget *menubar;
	gint width, height;

	gchar *outdir;
	const GtkTargetEntry targets[] = {{"text/uri-list", 0, URI_LIST}, 
					  {"property/bgimage", 0, URI_LIST},
					  {"text/plain", 0, TEXT_PLAIN},
					  {"STRING", 0, TEXT_PLAIN},
					  {"application/x-color", 0, COLOUR},
					  {"x-special/gnome-reset-background", 0, RESET}};

	const GtkActionEntry actions[] = {
	{ "GameMenu", NULL, N_("_Game") },
	{ "SettingsMenu", NULL, N_("_Settings") },
	{ "HelpMenu", NULL, N_("_Help") },
	{ "NewGame", GAMES_STOCK_NEW_GAME, NULL, NULL, NULL, G_CALLBACK (gameNew) },
	{ "Pause", GAMES_STOCK_PAUSE_GAME, NULL, NULL, NULL, G_CALLBACK (gamePause) },
	{ "Resume", GAMES_STOCK_RESUME_GAME, NULL, NULL, NULL, G_CALLBACK (gamePause) },
	{ "Scores", GAMES_STOCK_SCORES, NULL, NULL, NULL, G_CALLBACK (gameTopTen) },
	{ "EndGame", GAMES_STOCK_END_GAME, NULL, NULL, NULL, G_CALLBACK (gameEnd) },
	{ "Quit", GTK_STOCK_QUIT, NULL, NULL, NULL, G_CALLBACK (gameQuit) },
	{ "Preferences", GTK_STOCK_PREFERENCES, NULL, NULL, NULL, G_CALLBACK (gameProperties) },
	{ "Contents", GAMES_STOCK_CONTENTS, NULL, NULL, NULL, G_CALLBACK (gameHelp) },
	{ "About", GTK_STOCK_ABOUT, NULL, NULL, NULL, G_CALLBACK (gameAbout) }
	};

	const char ui_description[] =
	"<ui>"
	"  <menubar name='MainMenu'>"
	"    <menu action='GameMenu'>"
	"      <menuitem action='NewGame'/>"
	"      <menuitem action='Pause'/>"
	"      <menuitem action='Resume'/>"
	"      <separator/>"
	"      <menuitem action='Scores'/>"
	"      <menuitem action='EndGame'/>"
	"      <separator/>"
	"      <menuitem action='Quit'/>"
	"    </menu>"
	"    <menu action='SettingsMenu'>"
	"      <menuitem action='Preferences'/>"
	"    </menu>"
	"    <menu action='HelpMenu'>"
	"      <menuitem action='Contents'/>"
	"      <menuitem action='About'/>"
	"    </menu>"
	"  </menubar>"
	"</ui>";
	
	if (!sound)
		sound = new Sound();

	/* Locate our background image. */
	outdir = g_build_filename (gnome_user_dir_get (), "gnometris.d", 
				   NULL);
	if (!g_file_test (outdir, G_FILE_TEST_EXISTS))
	    mkdir (outdir, 0700);
	bgPixmap = g_build_filename (outdir, "background.bin", NULL);
	g_free (outdir);
	
	w = gnome_app_new("gnometris", _("Gnometris"));
	g_signal_connect (w, "delete_event", G_CALLBACK (gameQuit), this);
	gtk_drag_dest_set (w, GTK_DEST_DEFAULT_ALL, targets, 
			   G_N_ELEMENTS(targets), 
			   GDK_ACTION_MOVE);
	g_signal_connect (G_OBJECT (w), "drag_data_received", 
			  G_CALLBACK (dragDrop), this);
	g_signal_connect (G_OBJECT (w), "focus_out_event",
			  G_CALLBACK (focusOut), this);
	g_signal_connect (G_OBJECT (w), "configure_event",
			  G_CALLBACK (configure), this);

	line_fill_height = 0;
	line_fill_prob = 5;

	/* init gconf */
	gconf_client = gconf_client_get_default ();

	gconf_client_add_dir (gconf_client, KEY_OPTIONS_DIR, GCONF_CLIENT_PRELOAD_NONE, NULL);
	gconf_client_notify_add (gconf_client, KEY_OPTIONS_DIR, gconfNotify, this, NULL, NULL);

	gconf_client_add_dir (gconf_client, KEY_CONTROLS_DIR, GCONF_CLIENT_PRELOAD_NONE, NULL);
	gconf_client_notify_add (gconf_client, KEY_CONTROLS_DIR, gconfNotify, this, NULL, NULL);


	width = gconf_client_get_int (gconf_client, KEY_WIDTH, NULL);
	if (width <= 0)
		width = 440;
	height = gconf_client_get_int (gconf_client, KEY_HEIGHT, NULL);
	if (height <= 0)
		height = 550;

	gtk_window_set_default_size (GTK_WINDOW (w), width, height);

	preview = new Preview ();
	field = new Field();
	field->setUseTarget (false);

	initOptions ();

	/* prepare menus */
	games_stock_init ();
	action_group = gtk_action_group_new ("MenuActions");
	gtk_action_group_set_translation_domain (action_group, GETTEXT_PACKAGE);
	gtk_action_group_add_actions (action_group, actions, G_N_ELEMENTS (actions), this);
	ui_manager = gtk_ui_manager_new ();
	gtk_ui_manager_insert_action_group (ui_manager, action_group, 0);
	gtk_ui_manager_add_ui_from_string (ui_manager, ui_description, -1, NULL);
	accel_group = gtk_ui_manager_get_accel_group (ui_manager);
	gtk_window_add_accel_group (GTK_WINDOW (w), accel_group);

	new_game_action = gtk_action_group_get_action (action_group, "NewGame");
	pause_action = gtk_action_group_get_action (action_group, "Pause");
	resume_action = gtk_action_group_get_action (action_group, "Resume");
	scores_action = gtk_action_group_get_action (action_group, "Scores");
	end_game_action = gtk_action_group_get_action (action_group, "EndGame");
	preferences_action = gtk_action_group_get_action (action_group, "Preferences");

	games_stock_set_pause_actions (pause_action, resume_action);

	menubar = gtk_ui_manager_get_widget (ui_manager, "/MainMenu");

	GtkWidget * hb = gtk_hbox_new(FALSE, 0);

	vbox = gtk_vbox_new (FALSE, 0);
	gnome_app_set_contents (GNOME_APP (w), vbox);
	gtk_box_pack_start (GTK_BOX (vbox), menubar, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), hb, TRUE, TRUE, 0);

	aspect_frame = gtk_aspect_frame_new (NULL, 0.5, 0.5, (float) COLUMNS / (float) LINES, FALSE);
	gtk_frame_set_shadow_type (GTK_FRAME (aspect_frame), GTK_SHADOW_NONE);
	gtk_container_add (GTK_CONTAINER (aspect_frame), field->getWidget());

	gtk_widget_set_events(w, gtk_widget_get_events(w) | 
						  GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK);

	GtkWidget *vb1 = gtk_vbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(vb1), 10);
	gtk_box_pack_start_defaults(GTK_BOX(vb1), aspect_frame);
	gtk_box_pack_start_defaults(GTK_BOX(hb), vb1);

	setupPixmap();

	g_signal_connect (w, "key_press_event", G_CALLBACK (keyPressHandler), this);
	g_signal_connect (w, "key_release_event", G_CALLBACK (keyReleaseHandler), this);
  
	GtkWidget *vb2 = gtk_vbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(vb2), 10);
	gtk_box_pack_end(GTK_BOX(hb), vb2, 0, 0, 0);
	
	gtk_box_pack_start(GTK_BOX(vb2), preview->getWidget(), FALSE, FALSE, 0);
	
	scoreFrame = new ScoreFrame(cmdlineLevel);
	
	gtk_box_pack_end(GTK_BOX(vb2), scoreFrame->getWidget(), TRUE, FALSE, 0);
	high_scores = new HighScores ();

	setOptions ();
	setupScoreState ();

	themeList = NULL;
	
	gtk_widget_show(hb);
	gtk_widget_show(vb1);
	gtk_widget_show(vb2);
	gtk_widget_show(aspect_frame);
	gtk_widget_show(field->getWidget());
	scoreFrame->show();
	gtk_widget_show(w);

	gtk_action_set_sensitive(pause_action, FALSE);
	gtk_action_set_sensitive(end_game_action, FALSE);
	gtk_action_set_sensitive(preferences_action, TRUE);
}

Tetris::~Tetris()
{
	delete field;
	delete preview;
	delete scoreFrame;

	if (bgimage)
		g_object_unref (G_OBJECT (bgimage));		

	if (bgPixmap)
		g_free(bgPixmap);
}

void
Tetris::setupScoreState ()
{
	if (high_scores->empty ()) {
		gtk_action_set_sensitive (scores_action, FALSE);
	} else {
		gtk_action_set_sensitive (scores_action, TRUE);
	}
}

void 
Tetris::setupdialogDestroy(GtkWidget *widget, void *d)
{
	Tetris *t = (Tetris*) d;
	if (t->setupdialog) {
		delete t->theme_preview;
		gtk_widget_destroy(t->setupdialog);
	}
	t->setupdialog = 0;
	gtk_action_set_sensitive(t->new_game_action, TRUE);
}

void
Tetris::setupdialogResponse (GtkWidget *dialog, gint response_id, void *d)
{
	setupdialogDestroy (NULL, d);
}

void
Tetris::setupPixmap()
{
	if (bgimage)
		g_object_unref (G_OBJECT (bgimage));

	if (!usebg)
		bgimage = NULL;
	else {
		if (g_file_test (bgPixmap, G_FILE_TEST_EXISTS)) 
			bgimage = gdk_pixbuf_new_from_file (bgPixmap, NULL);
		else 
			bgimage = NULL;
	}

	/* A nasty hack to tile the image if it looks tileable (i.e. it
	 * is small enough. */
	if (bgimage) {
		int width, height;
		int bgwidth, bgheight;
			
		bgwidth = COLUMNS*BLOCK_SIZE;
		bgheight = LINES*BLOCK_SIZE;

		width = gdk_pixbuf_get_width (bgimage);
		height = gdk_pixbuf_get_height (bgimage);

		/* The heuristic is, anything less than 65 pixels on a side,
		 * or is square and smaller than the playing field is tiled. */
		/* Note that this heuristic fails for the standard nautilus
		 * background burlap.jpg because it is 97x91 */
		if ((width < TILE_THRESHOLD) || (height < TILE_THRESHOLD) ||
		    ((width == height) && (width < bgwidth))) {
			GdkPixbuf * temp;
			int i, j;
			
			temp = gdk_pixbuf_new (GDK_COLORSPACE_RGB, TRUE, 8, 
					       bgwidth, bgheight); 
		
			for (i=0; i<=bgwidth/width; i++) {
				for (j=0; j<=bgheight/height; j++) {
					int x, y, ww, hh;

					x = i*width;
					y = j*height;
					ww = MIN (width, bgwidth - x);
					hh = MIN (height, bgheight - y);

					gdk_pixbuf_copy_area (bgimage, 0, 0,
							      ww, hh, temp,
							      x, y);
				}
			}
			g_object_unref (bgimage);
			bgimage = temp;
		}
	}

	if (field)
	{
		if (bgimage)
			field->setBackground (bgimage);
		else
			field->setBackground (&bgcolour);
	}
	
	if (preview)
	{
		/* FIXME: We should do an update once the preview actually
		 * uses the background pixbuf. */
	}
}

void
Tetris::gconfNotify (GConfClient *tmp_client, guint cnx_id, GConfEntry *tmp_entry, gpointer tmp_data)
{
	Tetris *t = (Tetris *)tmp_data;

	t->initOptions ();
	t->setOptions ();
}

char *
Tetris::gconfGetString (GConfClient *client, const char *key, const char *default_val)
{
	char *val;
	GError *error = NULL;

	val = gconf_client_get_string (client, key, &error);
	if (error) {
		g_warning ("gconf error: %s\n", error->message);
		g_error_free (error);
		error = NULL;
		val = g_strdup (default_val);
	}

	return val;
}

int
Tetris::gconfGetInt (GConfClient *client, const char *key, int default_val)
{
	int val;
	GError *error = NULL;

	val = gconf_client_get_int (client, key, &error);
	if (error) {
		g_warning ("gconf error: %s\n", error->message);
		g_error_free (error);
		error = NULL;
		val = default_val;
	}

	return val;
}

gboolean
Tetris::gconfGetBoolean (GConfClient *client, const char *key, gboolean default_val)
{
	gboolean val;
	GError *error = NULL;

	val = gconf_client_get_bool (client, key, &error);
	if (error) {
		g_warning ("gconf error: %s\n", error->message);
		g_error_free (error);
		error = NULL;
		val = default_val;
	}

	return val;
}

void
Tetris::initOptions ()
{
	gchar *bgcolourstr;

	themeno = themeNameToNumber (gconfGetString (gconf_client, KEY_THEME, "plain"));
	field->setTheme (themeno);
	preview->setTheme (themeno);

	startingLevel = gconfGetInt (gconf_client, KEY_STARTING_LEVEL, 1);
	if (startingLevel < 1) 
		startingLevel = 1;
	if (startingLevel > 20) 
		startingLevel = 20;

	if (gconfGetBoolean (gconf_client, KEY_SOUND, TRUE)) 
		sound->turnOn ();
	else
		sound->turnOff ();

	useTarget = gconfGetBoolean (gconf_client, KEY_USE_TARGET, FALSE);

	do_preview = gconfGetBoolean (gconf_client, KEY_DO_PREVIEW, TRUE);
	
	if (preview) {
		preview->enable(do_preview);
	}

	random_block_colors = gconfGetBoolean (gconf_client, KEY_RANDOM_BLOCK_COLORS, TRUE);

	rotateCounterClockWise = gconfGetBoolean (gconf_client, KEY_ROTATE_COUNTER_CLOCKWISE, TRUE);

	line_fill_height = gconfGetInt (gconf_client, KEY_LINE_FILL_HEIGHT, 0);
	if (line_fill_height < 0)
		line_fill_height = 0;
	if (line_fill_height > 19)
		line_fill_height = 19;

	line_fill_prob = gconfGetInt (gconf_client, KEY_LINE_FILL_PROBABILITY, 0);
	if (line_fill_prob < 0)
		line_fill_prob = 0;
	if (line_fill_prob > 10)
		line_fill_prob = 10;

	moveLeft = gconfGetInt (gconf_client, KEY_MOVE_LEFT, GDK_Left);
	moveRight = gconfGetInt (gconf_client, KEY_MOVE_RIGHT, GDK_Right);
	moveDown = gconfGetInt (gconf_client, KEY_MOVE_DOWN, GDK_Down);
	moveDrop = gconfGetInt (gconf_client, KEY_MOVE_DROP, GDK_Pause);
	moveRotate = gconfGetInt (gconf_client, KEY_MOVE_ROTATE, GDK_Up);
	movePause = gconfGetInt (gconf_client, KEY_MOVE_PAUSE, GDK_space);

	bgcolourstr = gconfGetString (gconf_client, KEY_BG_COLOUR, "Black");
	gdk_color_parse (bgcolourstr, &bgcolour);
	g_free (bgcolourstr);

	usebg = gconfGetBoolean (gconf_client, KEY_USE_BG_IMAGE, FALSE);
}

void
Tetris::setOptions ()
{
	if (setupdialog) {
		gtk_spin_button_set_value (GTK_SPIN_BUTTON (sentry), startingLevel);
		gtk_spin_button_set_value (GTK_SPIN_BUTTON (fill_prob_spinner), line_fill_prob);
		gtk_spin_button_set_value (GTK_SPIN_BUTTON (fill_height_spinner), line_fill_height);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (sound_toggle), sound->isOn ());
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (do_preview_toggle), do_preview);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (random_block_colors_toggle), random_block_colors);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (rotate_counter_clock_wise_toggle), rotateCounterClockWise);

		if (theme_preview) {
			theme_preview->setTheme (themeno);
			gtk_widget_queue_draw(theme_preview->getWidget());
		}
	}

	scoreFrame->setLevel (startingLevel);
	scoreFrame->setStartingLevel (startingLevel);
	setupPixmap ();
}

void
Tetris::setSound (GtkWidget *widget, gpointer data)
{
	Tetris *t = (Tetris*)data;
	gconf_client_set_bool (t->gconf_client, KEY_SOUND,
			       GTK_TOGGLE_BUTTON (widget)->active, NULL);
}

void 
Tetris::setSelectionPreview(GtkWidget *widget, void *d)
{
	Tetris *t = (Tetris*) d;
	gconf_client_set_bool (t->gconf_client, KEY_DO_PREVIEW,
			       GTK_TOGGLE_BUTTON (widget)->active, NULL);
}

void 
Tetris::setSelectionBlocks(GtkWidget *widget, void *d)
{
	Tetris *t = (Tetris*) d;
	gconf_client_set_bool (t->gconf_client, KEY_RANDOM_BLOCK_COLORS,
			       GTK_TOGGLE_BUTTON (widget)->active, NULL);
}

void 
Tetris::setRotateCounterClockWise(GtkWidget *widget, void *d)
{
	Tetris *t = (Tetris*) d;
	gconf_client_set_bool (t->gconf_client, KEY_ROTATE_COUNTER_CLOCKWISE,
			       GTK_TOGGLE_BUTTON (widget)->active, NULL);
}

void
Tetris::setSelection(GtkWidget *widget, void *data)
{
	Tetris *t;
	GList * item;

      	t = (Tetris *)data;

	t->themeno = gtk_combo_box_get_active (GTK_COMBO_BOX (widget));
	t->field->setTheme (t->themeno);
	gconf_client_set_string (t->gconf_client, KEY_THEME,
				 ThemeTable[t->themeno].id, NULL);
}

void 
Tetris::setTarget (GtkWidget *widget, void *data)
{
	Tetris *t;
	
	t = (Tetris *)data;

	t->useTarget = GTK_TOGGLE_BUTTON (widget)->active;

	gconf_client_set_bool (t->gconf_client, KEY_USE_TARGET, 
			       t->useTarget, NULL);
}

void
Tetris::lineFillHeightChanged (GtkWidget *spin, gpointer data)
{
	Tetris *t = (Tetris *)data;
	gint value = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin));
	gconf_client_set_int (t->gconf_client, KEY_LINE_FILL_HEIGHT,
			      value, NULL);
}

void
Tetris::lineFillProbChanged (GtkWidget *spin, gpointer data)
{
	Tetris *t = (Tetris *)data;
	gint value = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin));
	gconf_client_set_int (t->gconf_client, KEY_LINE_FILL_PROBABILITY,
			      value, NULL);
}

void
Tetris::startingLevelChanged (GtkWidget *spin, gpointer data)
{
	Tetris *t = (Tetris *)data;
	gint value = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin));
	gconf_client_set_int (t->gconf_client, KEY_STARTING_LEVEL,
			      value, NULL);
}

void
Tetris::fillMenu(GtkWidget *menu, char *pixname, char *dirname, 
		 GList ** listp, bool addnone /*= false*/)
{
	struct dirent *e;
	char *dname = gnome_program_locate_file (NULL, GNOME_FILE_DOMAIN_APP_PIXMAP, dirname, FALSE, NULL);
	DIR *dir;
	int itemno = 0;
	GList * list;

	list = *listp;

	if (list) {
		g_list_foreach (list, (GFunc) g_free, NULL);
		g_list_free (list);
		*listp = NULL;
	}
	
	dir = opendir (dname);

	if (!dir)
		return;
	
	char *s;
	
	while ((e = readdir (dir)) != 0)
	{
		s = g_strdup(e->d_name);

		if (!(strstr (e->d_name, ".png") || 
		      strstr (e->d_name, ".jpg") ||
		      strstr (e->d_name, ".svg"))) 
		{
			free(s);
			continue;
		}

		gtk_combo_box_append_text (GTK_COMBO_BOX (menu), s);
		*listp = g_list_append (*listp, s);
			  
		if (!strcmp(pixname, s))
		{
		  gtk_combo_box_set_active(GTK_COMBO_BOX (menu), itemno);
		}
		itemno++;
	}
	
	if (addnone)
	{
		s = g_strdup(_("<none>"));
		gtk_combo_box_append_text (GTK_COMBO_BOX (menu), _("<none>"));
		*listp = g_list_append (*listp, s);
	}
	
	closedir(dir);
}

int 
Tetris::gameProperties(GtkAction *action, void *d)
{
	GtkWidget *notebook;
	GtkWidget *vbox;
	GtkWidget *label;
	GtkWidget *frame;
	GtkWidget *table;
	GtkWidget *fvbox;
	GtkObject *adj;
	GtkWidget *controls_list;

	Tetris *t = (Tetris*) d;
	
	if (t->setupdialog) {
		gtk_window_present (GTK_WINDOW(t->setupdialog));
		return FALSE;
	}

	/* create the dialog */
	t->setupdialog =
		gtk_dialog_new_with_buttons(_("Gnometris Preferences"), 
					    GTK_WINDOW (t->w),
					    (GtkDialogFlags)0,
					    GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE,
					    NULL);
	gtk_dialog_set_has_separator (GTK_DIALOG (t->setupdialog), FALSE);
	gtk_container_set_border_width (GTK_CONTAINER (t->setupdialog), 5);
	gtk_box_set_spacing (GTK_BOX (GTK_DIALOG (t->setupdialog)->vbox), 2);
	g_signal_connect (t->setupdialog, "close",
			  G_CALLBACK (setupdialogDestroy), d);
	g_signal_connect (t->setupdialog, "response",
			  G_CALLBACK (setupdialogResponse), d);

	notebook = gtk_notebook_new ();
	gtk_container_set_border_width (GTK_CONTAINER (notebook), 5);
	gtk_box_pack_start (GTK_BOX(GTK_DIALOG(t->setupdialog)->vbox), notebook,
			    TRUE, TRUE, 0);

	/* game page */
	vbox = gtk_vbox_new (FALSE, 18);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), 12);
	label = gtk_label_new (_("Game"));
	gtk_notebook_append_page (GTK_NOTEBOOK(notebook), vbox, label);

	frame = games_frame_new (_("Setup"));
	table = gtk_table_new (3, 2, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE (table), 6);
	gtk_table_set_col_spacings (GTK_TABLE (table), 12);

	/* pre-filled rows */
	label = gtk_label_new_with_mnemonic (_("_Number of pre-filled rows:"));
	gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1,
	                  (GtkAttachOptions) GTK_FILL, 
			  (GtkAttachOptions) 0,
			  0, 0);

	adj = gtk_adjustment_new (t->line_fill_height, 0, LINES-1, 1, 5, 0);
	t->fill_height_spinner = gtk_spin_button_new (GTK_ADJUSTMENT (adj), 10, 0);
	gtk_spin_button_set_update_policy
		(GTK_SPIN_BUTTON (t->fill_height_spinner), GTK_UPDATE_ALWAYS);
	gtk_spin_button_set_snap_to_ticks 
		(GTK_SPIN_BUTTON (t->fill_height_spinner), TRUE);
	g_signal_connect (t->fill_height_spinner, "value_changed",
			  G_CALLBACK (lineFillHeightChanged), t);
	gtk_table_attach_defaults (GTK_TABLE (table), t->fill_height_spinner, 1, 2, 0, 1);
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), t->fill_height_spinner);

	/* pre-filled rows density */
	label = gtk_label_new_with_mnemonic (_("_Density of blocks in a pre-filled row:"));
	gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2,
	                  (GtkAttachOptions) GTK_FILL, 
			  (GtkAttachOptions) 0,
			  0, 0);

	adj = gtk_adjustment_new (t->line_fill_prob, 0, 10, 1, 5, 0);
	t->fill_prob_spinner = gtk_spin_button_new (GTK_ADJUSTMENT (adj), 10, 0);
	gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (t->fill_prob_spinner),
					  GTK_UPDATE_ALWAYS);			  
	gtk_spin_button_set_snap_to_ticks
		(GTK_SPIN_BUTTON (t->fill_prob_spinner), TRUE);
	g_signal_connect (t->fill_prob_spinner, "value_changed",
		          G_CALLBACK (lineFillProbChanged), t);
	gtk_table_attach_defaults (GTK_TABLE (table), t->fill_prob_spinner, 1, 2, 1, 2);
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), t->fill_prob_spinner);

	/* starting level */
	label = gtk_label_new_with_mnemonic (_("_Starting level:"));
	gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 2, 3,
	                  (GtkAttachOptions) GTK_FILL, 
			  (GtkAttachOptions) 0,
			  0, 0);

	adj = gtk_adjustment_new (t->startingLevel, 1, 20, 1, 5, 20);
	t->sentry = gtk_spin_button_new (GTK_ADJUSTMENT (adj), 10.0, 0);
	gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (t->sentry),
					   GTK_UPDATE_ALWAYS);
	gtk_spin_button_set_snap_to_ticks (GTK_SPIN_BUTTON (t->sentry), TRUE);
	g_signal_connect (t->sentry, "value_changed",
			  G_CALLBACK (startingLevelChanged), t);
	gtk_table_attach_defaults (GTK_TABLE (table), t->sentry, 1, 2, 2, 3);
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), t->sentry);

	gtk_container_add (GTK_CONTAINER (frame), table);
	gtk_box_pack_start (GTK_BOX (vbox), frame, 
			    FALSE, FALSE, 0);

	frame = games_frame_new (_("Operation"));
	fvbox = gtk_vbox_new (FALSE, 6);

	/* sound */
	t->sound_toggle =
		gtk_check_button_new_with_mnemonic (_("_Enable sounds"));
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (t->sound_toggle),
				     sound->isOn ());
	g_signal_connect (t->sound_toggle, "clicked",
			  G_CALLBACK (setSound), d);
	gtk_box_pack_start (GTK_BOX (fvbox), t->sound_toggle, 0, 0, 0);
	
	/* preview next block */
	t->do_preview_toggle =
		gtk_check_button_new_with_mnemonic (_("_Preview next block"));
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (t->do_preview_toggle),
				     do_preview);
	g_signal_connect (t->do_preview_toggle, "clicked",
			  G_CALLBACK (setSelectionPreview), d);
	gtk_box_pack_start (GTK_BOX (fvbox), t->do_preview_toggle, 0, 0, 0);

	/* random blocks */
	t->random_block_colors_toggle =
		gtk_check_button_new_with_mnemonic (_("_Use random block colors"));
	gtk_toggle_button_set_active
		(GTK_TOGGLE_BUTTON (t->random_block_colors_toggle),
		 random_block_colors);
	g_signal_connect (t->random_block_colors_toggle, "clicked",
			  G_CALLBACK (setSelectionBlocks), d);
	gtk_box_pack_start (GTK_BOX (fvbox), t->random_block_colors_toggle,
			    0, 0, 0);

	/* rotate counter clock wise */
 	t->rotate_counter_clock_wise_toggle =
		gtk_check_button_new_with_mnemonic (_("_Rotate blocks counterclockwise"));
 	gtk_toggle_button_set_active
		(GTK_TOGGLE_BUTTON (t->rotate_counter_clock_wise_toggle),
		 rotateCounterClockWise);
	g_signal_connect (t->rotate_counter_clock_wise_toggle, "clicked",
			  G_CALLBACK (setRotateCounterClockWise), d);
 	gtk_box_pack_start (GTK_BOX (fvbox), t->rotate_counter_clock_wise_toggle,
			    0, 0, 0);

	t->useTargetToggle = gtk_check_button_new_with_mnemonic (_("Show _where the block will land"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (t->useTargetToggle), 
				      t->useTarget);
	g_signal_connect (t->useTargetToggle, "clicked",
			  G_CALLBACK (setTarget), d);
 	gtk_box_pack_start (GTK_BOX (fvbox), t->useTargetToggle,
			    0, 0, 0);

	gtk_container_add (GTK_CONTAINER (frame), fvbox);
	gtk_box_pack_start (GTK_BOX (vbox), frame, 
			    FALSE, FALSE, 0);

	frame = games_frame_new (_("Theme"));
	table = gtk_table_new (2, 2, FALSE);
	gtk_container_set_border_width (GTK_CONTAINER (table), 0);
	gtk_table_set_row_spacings (GTK_TABLE (table), 6);
	gtk_table_set_col_spacings (GTK_TABLE (table), 12);

	/* controls page */
	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), 12);
	label = gtk_label_new (_("Controls"));
	gtk_notebook_append_page (GTK_NOTEBOOK(notebook), vbox, label);

	frame = games_frame_new (_("Keyboard Controls"));
	gtk_container_add (GTK_CONTAINER (vbox), frame);

	fvbox = gtk_vbox_new (FALSE, 6);
	gtk_container_add (GTK_CONTAINER (frame), fvbox);

	controls_list = games_controls_list_new ();
	games_controls_list_add_controls (GAMES_CONTROLS_LIST (controls_list),
					  KEY_MOVE_LEFT,
					  KEY_MOVE_RIGHT,
					  KEY_MOVE_DOWN,
					  KEY_MOVE_DROP,
					  KEY_MOVE_ROTATE,
					  KEY_MOVE_PAUSE,
					  NULL);

	gtk_box_pack_start (GTK_BOX (fvbox), controls_list, TRUE, TRUE, 0);

	/* theme page */
	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), 12);
	label = gtk_label_new (_("Theme"));
	gtk_notebook_append_page (GTK_NOTEBOOK(notebook), vbox, label);

	frame = games_frame_new (_("Block Style"));
	gtk_container_add (GTK_CONTAINER (vbox), frame);

	fvbox = gtk_vbox_new (FALSE, 6);
	gtk_container_add (GTK_CONTAINER (frame), fvbox);

	GtkWidget *omenu = gtk_combo_box_new_text ();
	const ThemeTableEntry *entry = ThemeTable;
	while (entry->id) {
		gtk_combo_box_append_text (GTK_COMBO_BOX (omenu), entry->name);
		entry++;
	}
	gtk_combo_box_set_active (GTK_COMBO_BOX (omenu), t->themeno);
	g_signal_connect (omenu, "changed", G_CALLBACK (setSelection), t);
	gtk_box_pack_start (GTK_BOX (fvbox), omenu, FALSE, FALSE, 0);

	t->theme_preview = new Preview();
	t->theme_preview->setTheme (t->themeno);
	gtk_box_pack_start(GTK_BOX(fvbox), t->theme_preview->getWidget(), TRUE, TRUE, 0);

	t->theme_preview->previewBlock(4, 0, 0);

	gtk_widget_show_all (t->setupdialog);
	gtk_action_set_sensitive(t->new_game_action, FALSE);

	return TRUE;
}

int 
Tetris::focusOut(GtkWidget *widget, GdkEvent *e, Tetris *t)
{
	if (t->inPlay && !t->paused)
		t->togglePause();
	return TRUE;
}

int
Tetris::gamePause(GtkAction *action, void *d)
{
	Tetris *t = (Tetris*) d;
	t->togglePause();
	return TRUE;
}

int
Tetris::gameEnd(GtkAction *action, void *d)
{
	Tetris *t = (Tetris*) d;

	g_source_remove(t->timeoutId);
	t->timeoutId = 0;
	blocknr_next = -1;
	t->endOfGame();
	return TRUE;
}

int
Tetris::gameQuit(GtkAction *action, void *d)
{
	Tetris *t = (Tetris*) d;

	/* Record the score if the game isn't over. */
	if (t->inPlay && (t->scoreFrame->getScore() > 0))
		t->high_scores->add (t->scoreFrame->getScore());

	if (t->w)
		gtk_widget_destroy(t->w);
	gtk_main_quit();

	return TRUE;
}

void
Tetris::generateTimer(int level)
{
	g_return_if_fail (level > 0);

	if (timeoutId > 0)
		g_source_remove(timeoutId);

	// With 0.8, the old level 10 should be hit at about level 20.
	int intv = (int) round (1000.0 * pow (0.8, level - 1));
	if (intv <= 10)
		intv = 10;
		
	timeoutId = g_timeout_add (intv, timeoutHandler, this);
}

void
Tetris::manageFallen()
{
	field->fallingToLaying();
	sound->playSound (SOUND_LAND);

	int levelBefore = scoreFrame->getLevel();

	int levelAfter = scoreFrame->scoreLines (field->checkFullLines());
	if (levelAfter != levelBefore) 
		sound->playSound (SOUND_GNOMETRIS);
	if ((levelBefore != levelAfter) || fastFall)
		generateTimer(levelAfter);
	
	if (field->isFieldEmpty ())
		scoreFrame->scoreLastLineBonus ();

	generate();
}

int
Tetris::timeoutHandler(void *d)
{
	Tetris *t = (Tetris*) d;
	
	if (t->paused)
		return TRUE;

 	if (t->onePause)
 	{
		t->onePause = false;
		t->field->redraw();
	}
 	else
	{
		bool res = t->field->moveBlockDown();

		t->field->redraw();

		if (res)
		{
			t->manageFallen();
			if (t->fastFall && t->inPlay) {
				t->fastFall = false;
			}
		}
	}

	return TRUE;	
}

gboolean
Tetris::keyPressHandler(GtkWidget *widget, GdkEvent *event, Tetris *t)
{
	int keyval;
	bool res = false;

	if (t->timeoutId == 0)
		return FALSE;

	keyval = ((GdkEventKey*)event)->keyval;

	if (keyval == t->movePause)
	{
		t->togglePause();
		return TRUE;
	}

	if (t->paused)
		return FALSE;

	if (keyval == t->moveLeft) {
		res = t->field->moveBlockLeft();
		if (res)
			sound->playSound (SOUND_SLIDE);
		t->onePause = false;
	} else if (keyval == t->moveRight) {
		res = t->field->moveBlockRight();
		if (res)
			sound->playSound (SOUND_SLIDE);
		t->onePause = false;
	} else if (keyval == t->moveRotate) {
		res = t->field->rotateBlock(rotateCounterClockWise);
		if (res)
			sound->playSound (SOUND_TURN);
		t->onePause = false;
	} else if (keyval == t->moveDown) {
		if (!t->fastFall && !t->onePause) {
			t->fastFall = true;
			g_source_remove (t->timeoutId);
			t->timeoutId = g_timeout_add (10, timeoutHandler, t);
			res = true;
		}
	} else if (keyval == t->moveDrop) {
		if (!t->dropBlock) {
			t->dropBlock = true;
			t->field->dropBlock();
			t->manageFallen();
			res = TRUE;
		}        
	}

	t->field->redraw();

	return res;
}

gint
Tetris::keyReleaseHandler(GtkWidget *widget, GdkEvent *event, Tetris *t)
{
	bool res = false;

	if (t->timeoutId == 0)
		return FALSE;
	
	if (t->paused)
		return FALSE;

	int keyval = ((GdkEventKey*)event)->keyval;

	if (keyval == t->moveDown) {
		if (t->fastFall) {
			t->fastFall = false;
 			t->generateTimer(t->scoreFrame->getLevel());
		}
		res = TRUE;
	} else if (keyval == t->moveDrop) {
		t->dropBlock = false;
		res = TRUE;
	}

	return res;
}

void Tetris::saveBgOptions ()
{
	gchar * cbuffer;

	gconf_client_set_bool (gconf_client, KEY_USE_BG_IMAGE, 
				  usebg, NULL);

	cbuffer = g_strdup_printf ("#%04x%04x%04x", bgcolour.red,
				   bgcolour.green, bgcolour.blue);
	gconf_client_set_string (gconf_client, KEY_BG_COLOUR, cbuffer,
				 NULL);
	g_free (cbuffer);
}

void
Tetris::decodeColour (guint16 *data, Tetris *t)
{
	t->bgcolour.red = data[0];
	t->bgcolour.green = data[1];
	t->bgcolour.blue = data[2];
	/* Ignore the alpha channel. */

	t->usebg = FALSE;
	t->saveBgOptions ();
}

void
Tetris::resetColour (Tetris *t)
{
	t->bgcolour.red = 0;
	t->bgcolour.green = 0;
	t->bgcolour.blue = 0;
	/* Ignore the alpha channel. */

	t->usebg = FALSE;
	t->saveBgOptions ();
}

gchar * 
Tetris::decodeDropData(gchar * data, gint type)
{
	gchar *start, *end;

	if (data == NULL)
		return NULL;

	if (type == TEXT_PLAIN)
		return g_strdup (data);

	if (type == URI_LIST) {
		start = data;
		/* Skip any comments. */
		if (*start == '#') {
			while (*start != '\n') {
				start++;
				if (*start == '\0')
					return NULL;
			}
			start++;
			if (*start == '\0')
				return NULL;
		}

		/* Now extract the first URI. */
		end = start;
		while ((*end != '\0') && (*end != '\r') && (*end != '\n'))
			end++;
		*end = '\0';

		return g_strdup (start);
	}

	return NULL;
}

void
Tetris::dragDrop(GtkWidget *widget, GdkDragContext *context,
		 gint x, gint y, GtkSelectionData *data, guint info, 
		 guint time, Tetris * t)
{
	gchar * fileuri;
	GnomeVFSHandle *inhandle;
	GnomeVFSHandle *outhandle;
	GnomeVFSResult result;
	GnomeVFSFileInfo fileinfo;
	GnomeVFSFileSize bytesread;
	GnomeVFSFileSize filesize;
	GdkPixbufLoader * loader;
	GdkPixbuf * pixbuf;
	guchar * buffer;


	/* Accept a dropped filename and try and load it as the
	   background image. In the event of any kind of failure we
	   silently ignore it. */
	
	/* FIXME: We don't handle colour gradients (e.g. from the gimp) */

	/* FIXME: Dropped URLs from mozilla don't work (see below). */

	if (data->length < 0) {
		gtk_drag_finish (context, FALSE, FALSE, time);
		return;
	}

	gtk_drag_finish (context, TRUE, FALSE, time);

	if (info == COLOUR) {
		if (data->length == 8) 
			decodeColour ((guint16 *)data->data, t);
		return;
	}

	if (info == RESET) {
		resetColour (t);
		return;
	}

	fileuri = decodeDropData ((gchar *)data->data, info);
	/* Silently ignore bad data. */
	if (fileuri == NULL)
		goto error_exit;

	/* Now that we have a URI we load it and test it to see if it is 
	 * an image file. */

	/* FIXME: All this is slow and synchronous. It also
	 * doesn't give any feedback about errors. */
	
	result = gnome_vfs_open (&inhandle, fileuri, GNOME_VFS_OPEN_READ);
	if (result != GNOME_VFS_OK)
		goto error_exit;

	result = gnome_vfs_get_file_info_from_handle (inhandle, &fileinfo,
						      GNOME_VFS_FILE_INFO_FOLLOW_LINKS);
	if (result != GNOME_VFS_OK)
		goto error_exit_handle;

	/* This is where Drag and Drop of URLs from mozilla fails. */

	if (!(fileinfo.valid_fields & GNOME_VFS_FILE_INFO_FIELDS_SIZE))
		goto error_exit_handle;

	filesize = fileinfo.size;

	buffer = (guchar *)g_malloc (filesize);
	if (buffer == NULL)
		goto error_exit_handle;
	
	result = gnome_vfs_read (inhandle, buffer, filesize, &bytesread);
	/* FIXME: We should reread if not enough was read. */
	if ((result != GNOME_VFS_OK) || (bytesread != filesize))
		goto error_exit_buffer;

	loader = gdk_pixbuf_loader_new ();

	if (!gdk_pixbuf_loader_write (loader, buffer, filesize, NULL))
		goto error_exit_loader;

	gdk_pixbuf_loader_close (loader, NULL);

	pixbuf = gdk_pixbuf_loader_get_pixbuf (loader);
	if (pixbuf == NULL)
		goto error_exit_loader;

	g_object_ref (pixbuf);

	/* We now have an image file, in memory, that we know gdk-pixbuf
	 * can handle. Now we save it to disk. This is necessary so that
	 * "slow" URIs (e.g. http) behave well in the long run. */

	result = gnome_vfs_create (&outhandle, t->bgPixmap, GNOME_VFS_OPEN_WRITE,
				   FALSE, 0600);
	if (result != GNOME_VFS_OK)
		goto error_exit_loader;

	result = gnome_vfs_write (outhandle, buffer, filesize, &bytesread);
	if ((result != GNOME_VFS_OK) || (bytesread != filesize))
	    goto error_exit_saver;

	t->usebg = TRUE;
	t->saveBgOptions ();

 error_exit_saver:
	gnome_vfs_close (outhandle);
 error_exit_loader:
	g_object_unref (loader);
 error_exit_buffer:
	g_free (buffer);
 error_exit_handle:
	gnome_vfs_close (inhandle);
 error_exit:
	return;
}

void
Tetris::togglePause()
{
	paused = !paused;

	if (paused)
		field->showPauseMessage();
	else
		field->hidePauseMessage();

	gtk_action_set_sensitive (pause_action, !paused);
	gtk_action_set_sensitive (resume_action, paused);
	gtk_action_set_visible (pause_action, !paused);
	gtk_action_set_visible (resume_action, paused);
}

void
Tetris::generate()
{
	if (field->generateFallingBlock())
	{
		field->putBlockInField(false);
		preview->previewBlock(blocknr_next, rot_next, color_next);
		gtk_widget_queue_draw(preview->getWidget());
		onePause = true;
	}
	else
	{
		g_source_remove(timeoutId);
		timeoutId = 0;
		blocknr_next = -1;
		
		endOfGame();
	}
}

void
Tetris::endOfGame()
{
	if (paused) togglePause();
	gtk_action_set_sensitive (pause_action, FALSE);
	gtk_action_set_sensitive (end_game_action, FALSE);
	gtk_action_set_sensitive (preferences_action, TRUE);

	color_next = -1;
	blocknr_next = -1;
	rot_next = -1;
	preview->previewBlock(-1, -1, -1);
	gtk_widget_queue_draw(preview->getWidget());
	field->hidePauseMessage();
	field->showGameOverMessage();
	sound->playSound (SOUND_GAMEOVER);
	inPlay = false;

	if (scoreFrame->getScore() > 0) 
	{
		int pos = high_scores->add (scoreFrame->getScore());
		high_scores->show (GTK_WINDOW (w), pos);
	}

	field->setUseTarget (false);
}

int
Tetris::gameNew(GtkAction *action, void *d)
{
	Tetris *t = (Tetris*) d;

	if (t->timeoutId) 
	{
		g_source_remove(t->timeoutId);
		t->timeoutId = 0;

		/* Catch the case where we started a new game without
		 * finishing the old one. */
		if ((t->scoreFrame->getScore() > 0) && t->inPlay)
			t->high_scores->add (t->scoreFrame->getScore());
	}

	t->inPlay = true;

	int level = t->cmdlineLevel ? t->cmdlineLevel : t->startingLevel;

	t->fastFall = false;
	
	t->scoreFrame->setLevel(level);
	t->scoreFrame->setStartingLevel(level);

	t->field->setUseTarget (t->useTarget);

	t->generateTimer (level);
	t->field->emptyField(t->line_fill_height,t->line_fill_prob);

	t->scoreFrame->resetScore();
	t->paused = false;
	
	t->field->generateFallingBlock();
	t->field->redraw();
	t->preview->previewBlock(blocknr_next, rot_next, color_next);
	gtk_widget_queue_draw(t->preview->getWidget());

	gtk_action_set_visible(t->pause_action, TRUE);
	gtk_action_set_visible(t->resume_action, FALSE);
	gtk_action_set_sensitive(t->pause_action, TRUE);
	gtk_action_set_sensitive(t->end_game_action, TRUE);
	gtk_action_set_sensitive(t->preferences_action, FALSE);

	t->field->hidePauseMessage();
	t->field->hideGameOverMessage();

	sound->playSound (SOUND_GNOMETRIS);

	return TRUE;
}

int
Tetris::gameHelp(GtkAction *action, void *d)
{
	Tetris *t = (Tetris*) d;

	gnome_help_display ("gnometris.xml", NULL, NULL);

	return TRUE;
}

int
Tetris::gameAbout(GtkAction *action, void *d)
{
	Tetris *t = (Tetris*) d;

	const gchar * const authors[] = { "J. Marcin Gorycki", NULL };

	const gchar * const documenters[] = { "Angela Boyle", NULL };

	gchar *license = games_get_license (_("Gnometris"));

	gtk_show_about_dialog (GTK_WINDOW (t->getWidget()),
			       "name", _("Gnometris"),
			       "version", VERSION,
			       "comments", _("Written for my wife, Matylda\n"
					     "Send comments and bug reports to: \n"
					     "janusz.gorycki@intel.com"),
			       "copyright", "Copyright \xc2\xa9 1999-2007 J. Marcin Gorycki",
			       "license", license,
			       "authors", authors,
			       "documenters", documenters,				
			       "translator_credits", _("translator-credits"),
			       "logo-icon-name", "gnome-gnometris",
			       "website", "http://www.gnome.org/projects/gnome-games/",
			       "wrap-license", TRUE,
			       NULL);
	g_free (license);

	return TRUE;
}

int
Tetris::gameTopTen(GtkAction *action, void *d)
{
	Tetris *t = (Tetris*) d;
	t->high_scores->show(0);

	return TRUE;
}

gboolean Tetris::configure (GtkWidget *widget, GdkEventConfigure *event, Tetris *t)
{

	gconf_client_set_int (t->gconf_client, KEY_WIDTH, event->width, NULL);
	gconf_client_set_int (t->gconf_client, KEY_HEIGHT, event->height, 
			      NULL);

	return FALSE;
}
