/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 *  gpa-settings.c: 
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Authors :
 *    Jose M. Celorio <chema@ximian.com>
 *    Lauris Kaplinski <lauris@ximian.com>
 *
 *  Copyright (C) 2000-2001 Ximian, Inc. and Jose M. Celorio
 *
 */

#define __GPA_SETTINGS_C__

#include "config.h"
#include <string.h>
#include <libxml/globals.h>
#include "gpa-utils.h"
#include "gpa-value.h"
#include "gpa-reference.h"
#include "gpa-model.h"
#include "gpa-option.h"
#include "gpa-key.h"
#include "gpa-settings.h"
#include "gpa-printer.h"

static void gpa_settings_class_init (GPASettingsClass *klass);
static void gpa_settings_init (GPASettings *settings);
static void gpa_settings_finalize (GObject *object);

static GPANode * gpa_settings_duplicate (GPANode *node);
static gboolean  gpa_settings_verify    (GPANode *node);
static guchar *  gpa_settings_get_value (GPANode *node);
static GPANode * gpa_settings_get_child (GPANode *node, GPANode *previous_child);
static GPANode * gpa_settings_lookup    (GPANode *node, const guchar *path);
static void      gpa_settings_modified  (GPANode *node, guint flags);

static GPANodeClass *parent_class = NULL;

GType
gpa_settings_get_type (void)
{
	static GType type = 0;
	if (!type) {
		static const GTypeInfo info = {
			sizeof (GPASettingsClass),
			NULL, NULL,
			(GClassInitFunc) gpa_settings_class_init,
			NULL, NULL,
			sizeof (GPASettings),
			0,
			(GInstanceInitFunc) gpa_settings_init
		};
		type = g_type_register_static (GPA_TYPE_NODE, "GPASettings", &info, 0);
	}
	return type;
}

static void
gpa_settings_class_init (GPASettingsClass *klass)
{
	GObjectClass *object_class;
	GPANodeClass *node_class;

	object_class = (GObjectClass *) klass;
	node_class = (GPANodeClass *) klass;

	parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = gpa_settings_finalize;

	node_class->duplicate = gpa_settings_duplicate;
	node_class->verify    = gpa_settings_verify;
	node_class->get_value = gpa_settings_get_value;
	node_class->get_child = gpa_settings_get_child;
	node_class->lookup    = gpa_settings_lookup;
	node_class->modified  = gpa_settings_modified;
}

static void
gpa_settings_init (GPASettings *settings)
{
	settings->name  = NULL;
	settings->keys  = NULL;
	settings->model = NULL;
}

static void
gpa_settings_finalize (GObject *object)
{
	GPASettings *settings;

	settings = GPA_SETTINGS (object);

	settings->name  = gpa_node_detach_unref (GPA_NODE (settings), GPA_NODE (settings->name));
	settings->model = gpa_node_detach_unref (GPA_NODE (settings), GPA_NODE (settings->model));

	while (settings->keys) {
		if (G_OBJECT (settings->keys)->ref_count > 1) {
			g_warning ("GPASettings: Child %s has refcount %d\n",
				   gpa_node_id (settings->keys),
				   G_OBJECT (settings->keys)->ref_count);
		}
		settings->keys = gpa_node_detach_unref_next (GPA_NODE (settings),
							     GPA_NODE (settings->keys));
	}

	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static GPANode *
gpa_settings_duplicate (GPANode *node)
{
	GPASettings *settings, *new;
	GPANode *child;
	GSList *list;

	settings = GPA_SETTINGS (node);

	new = (GPASettings *) gpa_node_new (GPA_TYPE_SETTINGS, gpa_node_id (node));

	g_assert (settings->name);
	g_assert (settings->model);
		
	new->name = gpa_node_attach (GPA_NODE (new), gpa_node_duplicate (settings->name));
	new->model = gpa_node_attach (GPA_NODE (new), gpa_node_duplicate (settings->model));

	list = NULL;
	for (child = settings->keys; child != NULL; child = child->next) {
		GPANode *newchild;
		newchild = gpa_node_duplicate (child);
		if (newchild)
			list = g_slist_prepend (list, newchild);
		else
			g_warning ("Could not duplicate %s inside gpa_settings_duplicate\n",
				   gpa_node_id (child));
	}

	while (list) {
		GPANode *newchild;

		newchild = GPA_NODE (list->data);
		newchild->parent = GPA_NODE (new);
		newchild->next = new->keys;
		new->keys = newchild;
		
		list = g_slist_remove (list, newchild); /* Equiv to list = list->next */
	}

	return GPA_NODE (new);
}

static gboolean
gpa_settings_verify (GPANode *node)
{
	if (gpa_node_id (node) == NULL) {
		g_print ("Settings needs to have an ID\n");
		return FALSE;
	}

	return TRUE;
}

static guchar *
gpa_settings_get_value (GPANode *node)
{
	return NULL;
}

static GPANode *
gpa_settings_get_child (GPANode *node, GPANode *previous_child)
{
	GPASettings *settings;
	GPANode *child = NULL;

	settings = GPA_SETTINGS (node);

	if (!previous_child) {
		child = settings->name;
	} else if (previous_child == settings->name) {
		child = settings->model;
	} else if (previous_child == settings->model) {
		child = settings->keys;
	} else {
		if (previous_child->next)
			child = previous_child->next;
	}

	if (child)
		gpa_node_ref (child);

	return child;
}

static GPANode *
gpa_settings_lookup (GPANode *node, const guchar *path)
{
	GPASettings *settings;
	GPANode *child;
	const guchar *dot, *next;
	gint len;

	settings = GPA_SETTINGS (node);

	child = NULL;

	if (gpa_node_lookup_helper (GPA_NODE (settings->name),  path, "Name",  &child))
		return child;
	if (gpa_node_lookup_helper (GPA_NODE (settings->model), path, "Model", &child))
		return child;

	dot = strchr (path, '.');
	if (dot != NULL) {
		len = dot - path;
		next = dot + 1;
	} else {
		len = strlen (path);
		next = path + len;
	}

	for (child = settings->keys; child != NULL; child = child->next) {
		const guchar *id;
		g_assert (GPA_IS_KEY (child));
		id = gpa_node_id (child);
		if (id && strlen (id) == len && !strncmp (id, path, len)) {
			if (!next) {
				gpa_node_ref (child);
				return child;
			} else {
				return gpa_node_lookup (child, next);
			}
		}
	}

	return NULL;
}

static void
gpa_settings_modified (GPANode *node, guint flags)
{
	GPASettings *settings;
	GPANode *child;

	settings = GPA_SETTINGS (node);

	if (GPA_NODE_FLAGS (settings->name) & GPA_NODE_MODIFIED_FLAG)
		gpa_node_emit_modified (settings->name, 0);
	
	if (GPA_NODE_FLAGS (settings->model) & GPA_NODE_MODIFIED_FLAG)
		gpa_node_emit_modified (settings->model, 0);

	child = settings->keys;
	while (child) {
		GPANode *next;
		next = child->next;
		if (GPA_NODE_FLAGS (child) & GPA_NODE_MODIFIED_FLAG) {
			gpa_node_ref (child);
			gpa_node_emit_modified (child, 0);
			gpa_node_unref (child);
		}
		child = next;
	}
}

/* FIXME: I don't understand what would we use this for */
GPANode *
gpa_settings_new_empty (const guchar *name, const guchar *id)
{
	GPASettings *settings;

	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (*name != '\0', NULL);

	settings = (GPASettings *) gpa_node_new (GPA_TYPE_SETTINGS, id);

	settings->name = gpa_value_new ("Name", name);
	settings->name->parent = GPA_NODE (settings);
	settings->model = gpa_reference_new_empty ();
	settings->model->parent = GPA_NODE (settings);

	return GPA_NODE (settings);
}

GPANode *
gpa_settings_new_from_model_full (GPANode *model, const guchar *name, const guchar *id)
{
	GPASettings *settings;
	GPANode *child;
	GSList *list;

	g_return_val_if_fail (model != NULL, NULL);
	g_return_val_if_fail (GPA_IS_MODEL (model), NULL);
	g_return_val_if_fail (id != NULL, NULL);
	g_return_val_if_fail (*id != '\0', NULL);
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (*name != '\0', NULL);
	g_return_val_if_fail (gpa_node_verify (GPA_NODE (model)), NULL);
	g_return_val_if_fail (GPA_MODEL_ENSURE_LOADED (model), NULL);

	settings = (GPASettings *) gpa_node_new (GPA_TYPE_SETTINGS, id);

	settings->name  = gpa_node_attach (GPA_NODE (settings), gpa_value_new ("Name", name));
	settings->model = gpa_node_attach (GPA_NODE (settings), gpa_reference_new (model));

	list = NULL;
	for (child = GPA_LIST (GPA_MODEL (model)->options)->children; child != NULL; child = child->next) {
		GPANode *key;
		key = gpa_key_new_from_option (child);
		if (key)
			list = g_slist_prepend (list, key);
	}

	while (list) {
		GPANode *key;
		
		key = GPA_NODE (list->data);
		key->parent = GPA_NODE (settings);
		key->next = settings->keys;
		settings->keys = key;
		
		list = g_slist_remove (list, key);
	}

	return (GPANode *) settings;
}

GPANode *
gpa_settings_new_from_model (GPANode *model, const guchar *name)
{
	GPASettings *settings;
	guchar *id;

	g_return_val_if_fail (model != NULL, NULL);
	g_return_val_if_fail (GPA_IS_MODEL (model), NULL);
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (*name != '\0', NULL);
	g_return_val_if_fail (gpa_node_verify (model), NULL);
	g_return_val_if_fail (GPA_MODEL_ENSURE_LOADED (model), NULL);

	id = gpa_id_new ("SETTINGS");
	settings = (GPASettings *) gpa_settings_new_from_model_full (model, name, id);
	g_free (id);

	return (GPANode *) settings;
}

GPANode *
gpa_settings_new_from_model_and_tree (GPANode *model, xmlNodePtr tree)
{
	GPASettings *settings;
	xmlChar *settings_id;
	xmlNodePtr xml_node;

	g_return_val_if_fail (model != NULL, NULL);
	g_return_val_if_fail (GPA_IS_MODEL (model), NULL);
	g_return_val_if_fail (tree != NULL, NULL);
	g_return_val_if_fail (gpa_node_verify (GPA_NODE (model)), NULL);
	g_return_val_if_fail (GPA_MODEL_ENSURE_LOADED (model), NULL);
	g_return_val_if_fail (!strcmp (tree->name, "Settings"), NULL);

	settings_id = xmlGetProp (tree, "Id");
	g_return_val_if_fail (settings_id != NULL, NULL);

	settings = NULL;
	for (xml_node = tree->xmlChildrenNode; xml_node != NULL; xml_node = xml_node->next) {
		
		/* <Name> */
		if (strcmp (xml_node->name, "Name") == 0) {
			xmlChar *settings_name;
			settings_name = xmlNodeGetContent (xml_node);
			if (!settings_name || !*settings_name) {
				g_warning ("Settings do not contain a valid <Name>\n");
				continue;
			}
			settings = (GPASettings *) gpa_settings_new_from_model_full (model, settings_name, settings_id);
			xmlFree (settings_name);
			continue;
		}

		/* <Key> */
		if (strcmp (xml_node->name, "Key") == 0) {
			xmlChar *key_id;
			GPANode *key;
			
			if (!settings) {
				g_print ("Can't have <Key> before <Name> in settings\n");
				continue;
			}
			
			key_id = xmlGetProp (xml_node, "Id");

			if (!key_id || !*key_id) {
				g_warning ("Invalid Key id while parsing settings %s\n", settings_id);
				xmlFree (key_id);
				continue;
			}
			
			for (key = settings->keys; key != NULL; key = key->next) {
				if (GPA_NODE_ID_COMPARE (key, key_id)) {
					gpa_key_merge_from_tree (key, xml_node);
					break;
				}
			}
			xmlFree (key_id);
		}
	}

	if (!settings) {
		g_warning ("Could not create the \"%s\" settings.\n", settings_id);
	}
	xmlFree (settings_id);

	return (GPANode *) settings;
}

xmlNodePtr
gpa_settings_write (xmlDocPtr doc, GPANode *node)
{
	GPASettings *settings;
	xmlNodePtr xmln, xmlc;
	GPANode *child;

	settings = GPA_SETTINGS (node);

	xmln = xmlNewDocNode (doc, NULL, "Settings", NULL);
	xmlSetProp (xmln, "Id", gpa_node_id (node));

	xmlc = xmlNewChild (xmln, NULL, "Name", GPA_VALUE (settings->name)->value);

	for (child = settings->keys; child != NULL; child = child->next) {
		xmlc = gpa_key_write (doc, child);
		if (xmlc)
			xmlAddChild (xmln, xmlc);
	}

	return xmln;
}

gboolean
gpa_settings_copy (GPASettings *dst, GPASettings *src)
{
	GPANode *child;
	GSList *sl, *dl;

	g_return_val_if_fail (dst != NULL, FALSE);
	g_return_val_if_fail (GPA_IS_SETTINGS (dst), FALSE);
	g_return_val_if_fail (src != NULL, FALSE);
	g_return_val_if_fail (GPA_IS_SETTINGS (src), FALSE);

	g_return_val_if_fail (GPA_VALUE_VALUE (src->name), FALSE);
	g_return_val_if_fail (GPA_VALUE_VALUE (dst->name), FALSE);

	g_return_val_if_fail (src->model != NULL, FALSE);
	g_return_val_if_fail (dst->model != NULL, FALSE);

	gpa_value_set_value_forced (GPA_VALUE (dst->name), GPA_VALUE_VALUE (src->name));

	gpa_reference_set_reference (GPA_REFERENCE (dst->model), GPA_REFERENCE_REFERENCE (src->model));

	dl = NULL;
	while (dst->keys) {
		dl = g_slist_prepend (dl, dst->keys);
		dst->keys = gpa_node_detach_next (GPA_NODE (dst), dst->keys);
	}

	sl = NULL;
	for (child = src->keys; child != NULL; child = child->next) {
		sl = g_slist_prepend (sl, child);
	}

	while (sl) {
		GSList *l;
		for (l = dl; l != NULL; l = l->next) {
			if (GPA_NODE (l->data)->qid && (GPA_NODE (l->data)->qid == GPA_NODE (sl->data)->qid)) {
				/* We are in original too */
				child = GPA_NODE (l->data);
				dl = g_slist_remove (dl, l->data);
				child->parent = GPA_NODE (dst);
				child->next = dst->keys;
				dst->keys = child;
				gpa_key_merge_from_key (GPA_KEY (child), GPA_KEY (sl->data));
				break;
			}
		}
		if (!l) {
			/* Create new child */
			child = gpa_node_duplicate (GPA_NODE (sl->data));
			child->parent = GPA_NODE (dst);
			child->next = dst->keys;
			dst->keys = child;
		}
		sl = g_slist_remove (sl, sl->data);
	}

	while (dl) {
		gpa_node_unref (GPA_NODE (dl->data));
		dl = g_slist_remove (dl, dl->data);
	}

	gpa_node_request_modified (GPA_NODE (dst), 0);

	return TRUE;
}
