/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */

/* 
 * Copyright (C) 2002 Thomas Vander Stichele
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more view.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Thomas Vander Stichele <thomas at apestaart dot org>
 */

/* audio-view.c - audio view code to be shared among implementations
 */

#include <config.h>

//#include <gdk-pixbuf/gdk-pixbuf.h>
#include <gtk/gtk.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkliststore.h>
#include <gtk/gtkcellrenderertext.h>
#include <gtk/gtkscrolledwindow.h>
#include <gtk/gtktreeview.h>


#include <libgnomevfs/gnome-vfs-directory.h>
#include <libgnome/gnome-macros.h>

#include <gst/play/play.h>
#include <gst/gconf/gconf.h>

#include <string.h>

#include "i18n-support.h"
#include "audio-info.h"
#include "audio-play.h"
#include "audio-view.h"

/* list columns */
enum {
	FILE_COLUMN,
	TYPE_COLUMN,
	LENGTH_COLUMN,
	//TITLE_COLUMN,
	//ARTIST_COLUMN,
	//BITRATE_COLUMN,
	//TIME_COLUMN,
	//SAMPLE_RATE_COLUMN,
	COMMENT_COLUMN,
	PATH_URI_COLUMN,
	AUDIO_INFO_COLUMN,
	NUM_COLUMNS
};

struct AudioView {
	GtkWidget *widget;	/* top-level view widget */

	GtkWidget *scroll_window;
	GtkListStore *list_store;
	GtkWidget *tree_view;

	GtkWidget *control;
	GtkWidget *prev_button;
	GtkWidget *stop_button;
	GtkWidget *play_button;
	GtkWidget *next_button;
	GtkWidget *seek_scale;
	/* GtkAdjustment *seek_adj; */
	GtkWidget *time;

	GtkWidget *status;
	
	GtkWidget *event_box;
	char *location;
	char *selection;	/* which track do you want to play ? */

	gulong seek_changed_id;
	gdouble last_seek_value; /* so we can see if we changed the slider
				    or the user */

	GList *audio_list;
	AudioPlay *audio_play;

};

/* forward declarations */
static void	audio_view_set_time 	(AudioView *view, 
		                         gint seconds, gint length);

/* helper functions */
static GtkWidget *
control_box_add_labeled_button (GtkWidget *control, 
		                const gchar *label, const gchar *file)
{
	GtkWidget *button;
	GdkPixbuf *pixbuf;
	GtkWidget *image;
	gchar *location;

	button = gtk_button_new ();
	
	location = g_strdup_printf ("%s/%s", PIXMAPSDIR, file);
	pixbuf = gdk_pixbuf_new_from_file (location, NULL);
	g_free (location);
	if (pixbuf) 
	{
		image = gtk_image_new_from_pixbuf (pixbuf);
		gtk_container_add (GTK_CONTAINER (button), image);
	}
	else
		gtk_container_add (GTK_CONTAINER (button),
				   gtk_label_new (label));
	gtk_box_pack_start (GTK_BOX (control), button, FALSE, FALSE, 0);

	return button;
}

/* set next track to play
 * return FALSE if this is the last track */
static gboolean
audio_view_play_next (AudioView *view)
{
	AudioPlay *play = view->audio_play;
	GtkTreeSelection *sel;
	GtkTreeModel *model;
	GtkTreeIter iter;
	GtkTreePath *path;
	AudioInfo *info;

	g_assert (IS_AUDIO_PLAY (play));

	audio_view_set_time (view, 0, 0);
	sel = gtk_tree_view_get_selection (GTK_TREE_VIEW (view->tree_view));
	model = gtk_tree_view_get_model (GTK_TREE_VIEW (view->tree_view));
	g_assert (GTK_IS_TREE_MODEL (model));
	if (gtk_tree_selection_get_selected (sel, &model, &iter))
	{
		GtkTreePath *old_path;
		path = gtk_tree_model_get_path (model, &iter);
		old_path = gtk_tree_path_copy (path);
		gtk_tree_path_next (path);
		if (!gtk_tree_model_get_iter (model, &iter, path))
		{
			g_print ("DEBUG: play_next: already last\n");
			gtk_tree_selection_unselect_path (sel, old_path);
			gtk_tree_path_free (path);
			return FALSE;
		}
		gtk_tree_model_get (model, &iter,
				    AUDIO_INFO_COLUMN, &info,
				    -1);
	}
	audio_view_set_playing (view, info, path);
	gtk_tree_path_free (path);
}

/* ui functions */
static void
audio_view_set_time (AudioView *view, gint seconds, gint length)
{
	gchar *text;
	if (length == 0)
		text = g_strdup_printf ("<span size=\"larger\">Unknown</span>");
	else
		text = g_strdup_printf ("<span size=\"larger\">"
				        "[%d:%02d/%d.%02d]</span>", 
			                seconds / 60, seconds % 60,
				        length / 60, length % 60);
	gtk_label_set_markup (GTK_LABEL (view->time), text);
	g_free (text);
}

/* load the given location in the audio view */
void
audio_view_load_location (AudioView *view, const char *location)
{
  g_assert (location != NULL);

  g_free (view->location);
  view->location = g_strdup (location);
  view->selection = NULL;

  audio_view_update (view);
}
	
/* set this track to play and update the treeview to the new path 
 * if path is NULL, don't update it */
static void
audio_view_set_playing (AudioView *view, AudioInfo *info,
		        GtkTreePath *path)
{
	gchar *message;
	AudioPlay *play;
	GError **error = NULL;

	/* play it */
	play = view->audio_play;
	g_assert (IS_AUDIO_PLAY (play));
	if (audio_play_get_state (play) == GST_STATE_PLAYING)
	{
		g_print ("DEBUG: set_playing: was playing, set to READY\n");
		audio_play_set_state (play, GST_STATE_READY, NULL);
		g_print ("DEBUG: set_playing: changed state to READY\n");
	}
	g_print ("DEBUG: set_playing: setting %s to play\n", info->path);
	audio_play_set_location (play, info->path, NULL);
	audio_play_set_state (play, GST_STATE_PLAYING, error);
	if (error)
	{
		/* FIXME: handleme */
		message = g_strdup_printf ("ERROR: %s", (*error)->message);
		gtk_label_set_text (GTK_LABEL (view->status), message);
		g_free (message);
		g_free (error);
		return;
	}

	/* status update */
	message = g_strdup_printf ("Playing %s", info->file);
	gtk_label_set_text (GTK_LABEL (view->status), message);
	g_free (message);

	/* selection update */
	if (path)
		gtk_tree_view_set_cursor (GTK_TREE_VIEW (view->tree_view),
			                  path, NULL, FALSE);
}
	
			    
/* ui callbacks */
/* FIXME */
static void
row_activated_callback (GtkTreeView *tree_view, GtkTreePath *path, 
                        GtkTreeViewColumn *column, 
                        AudioView *view)
{
	GtkTreeIter iter;
	GtkTreeModel *model = gtk_tree_view_get_model (tree_view);
	AudioInfo *info;

	g_print ("You activated a row\n");
	
	if (! (gtk_tree_model_get_iter (model, &iter, path)))
			return;
	
	gtk_tree_model_get (model, &iter,
			    AUDIO_INFO_COLUMN, &info,
			    -1);
	audio_view_set_playing (view, info, NULL);
	view->selection = g_strdup ("ENOCLUE");
}

static void
prev_activate (GtkButton *prev_button, AudioView *view)
{
	AudioPlay *play;
	GtkTreeSelection *sel;
	GtkTreeModel *model;
	GtkTreeIter iter;
	GtkTreePath *path;
	AudioInfo *info;

	play = view->audio_play;
	g_assert (IS_AUDIO_PLAY (play));

	sel = gtk_tree_view_get_selection (GTK_TREE_VIEW (view->tree_view));
	model = gtk_tree_view_get_model (GTK_TREE_VIEW (view->tree_view));
	g_assert (GTK_IS_TREE_MODEL (model));
	if (gtk_tree_selection_get_selected (sel, &model, &iter))
	{
		path = gtk_tree_model_get_path (model, &iter);
		if (!gtk_tree_path_prev (path))
			return;
		if (!gtk_tree_model_get_iter (model, &iter, path))
			return;
		gtk_tree_model_get (model, &iter,
				    AUDIO_INFO_COLUMN, &info,
				    -1);
	}
	audio_view_set_playing (view, info, path);
}

static void
play_activate (GtkButton *play_button, AudioView *view)
{
	gchar *message;
	gchar *selection = view->selection;
	GtkTreeSelection *sel;
	GtkTreeModel *model;
	GtkTreeIter iter;
	AudioPlay *play;
	gchar *path = NULL;
	gchar *file;
	AudioInfo *info;

	/* try to get the selected track */
	sel = gtk_tree_view_get_selection (GTK_TREE_VIEW (view->tree_view));
	model = gtk_tree_view_get_model (GTK_TREE_VIEW (view->tree_view));
	g_assert (GTK_IS_TREE_MODEL (model));
	if (gtk_tree_selection_get_selected (sel, &model, &iter))
	{
		gtk_tree_model_get (model, &iter,
				    AUDIO_INFO_COLUMN, &info,
				    -1);
		audio_view_set_playing (view, info, NULL);
	}
	else
		message = g_strdup_printf ("Make a selection first !");
	gtk_label_set_text (GTK_LABEL (view->status),
			    message);
	g_free (message);
}
	
static void
stop_activate (GtkButton *stop_button, AudioView *view)
{
	AudioPlay *play;

	play = view->audio_play;
	g_assert (IS_AUDIO_PLAY (play));
	if (audio_play_get_state (play) == GST_STATE_PLAYING)
	{
		g_print ("DEBUG: pressed stop, setting to ready\n");
		audio_play_set_state (play, GST_STATE_READY, NULL);
	}
	audio_view_set_time (view, 0, 0);
	gtk_label_set_text (GTK_LABEL (view->status),
			    "Stopped.");
}

static void
next_activate (GtkButton *next_button, AudioView *view)
{
	audio_view_play_next (view);
}

/* GStreamer callbacks */
static void
have_tick_callback (AudioPlay *play, gint64 time_nanos, AudioView *view)
{
	gchar *text;
	gint length_seconds;
	gint seconds = (gint) (time_nanos / GST_SECOND);

	length_seconds = (gint) (audio_play_get_length (play) / GST_SECOND);
	g_print ("DEBUG: tick: length %d\n", length_seconds);
	/* update time display */
	audio_view_set_time (view, seconds, length_seconds);

	/* update time slider */
	g_signal_handler_block (GTK_RANGE (view->seek_scale),
			        view->seek_changed_id);
	gtk_range_set_increments (GTK_RANGE (view->seek_scale),
			          (1.0 / length_seconds),
				  (10.0 / length_seconds));
	gtk_range_set_value (GTK_RANGE (view->seek_scale),
			     (double) seconds / length_seconds);
	g_signal_handler_unblock (GTK_RANGE (view->seek_scale),
			        view->seek_changed_id);
}

static void
have_eos_callback (AudioPlay *play, AudioView *view)
{
	g_print ("DEBUG: have_eos callback activated\n");
	//audio_play_set_state (play, GST_STATE_READY, NULL);
	if (! audio_view_play_next (view))
	{
		gtk_label_set_text (GTK_LABEL (view->status), "Stopped.");
		gtk_widget_show (view->status);
	}
}

static void
seek_changed_callback (GtkWidget *widget, AudioView *view)
{
	AudioPlay *play = view->audio_play;
	gdouble value = gtk_range_get_value (GTK_RANGE (widget));
	gdouble last_seek_value = view->last_seek_value;
	g_assert (IS_AUDIO_PLAY (play));

	//if (abs (last_seek_value - value) >= 1E-4)
		/* user probably made the seek instead of us */
		audio_play_seek_to_pos (play, value);
	view->last_seek_value = value;
}

/* look up corresponding iter 
 * FIXME: this might get slow, browsing a list like this */
static GtkTreeIter *
audio_view_get_iter_from_info (AudioView *view, AudioInfo *info)
{
	GtkTreeIter iter, *ip;
	AudioInfo *tree_info;
	GtkTreeModel *model;
	model = gtk_tree_view_get_model (GTK_TREE_VIEW (view->tree_view));

	if (! gtk_tree_model_get_iter_first (model, &iter))
		return NULL;

	do
	{
		gtk_tree_model_get (model, &iter, 
				    AUDIO_INFO_COLUMN, &tree_info, -1);
		if (info == tree_info) return gtk_tree_iter_copy (&iter);
	} while (gtk_tree_model_iter_next (model, &iter));
	return NULL;
}
static gboolean
audio_view_get_metadata_idler (AudioView *view)
{
	GList *p;
	AudioInfo *info;
	GtkTreeIter iter;
	
	g_print ("DEBUG: running idler for view %p\n", view);
   	for (p = view->audio_list; p != NULL; p = p->next) {
		int min, sec, msec, remainder;
		gchar *time;
		GtkTreeIter *iterp;
		info = (AudioInfo *) p->data;
	
		g_print ("getting metadata for info %p\n", info);
		audio_info_update (info);
		if (info->length > 0)
		{
			msec = info->length % 1000;
			remainder = info->length / 1000;
			sec = remainder % 60;
			min = remainder / 60;
			time = g_strdup_printf ("%d:%02d",
					        min, sec);
		}
		else
			time = g_strdup ("Unkown");

		/* fixme: get right position based on info instead of appd */
		iterp = audio_view_get_iter_from_info (view, info);
		g_print ("DEBUG: iter ptr %p\n", iterp);
		gtk_list_store_set (view->list_store,
				    iterp,
				    LENGTH_COLUMN, time,
				    //TITLE_COLUMN, g_strdup (info->title),
				    //ARTIST_COLUMN, g_strdup (info->artist),
				    //BITRATE_COLUMN, info->bitrate,
				    //TIME_COLUMN, info->track_time,
				    //SAMPLE_RATE_COLUMN, info->samprate,
				    COMMENT_COLUMN, g_strdup (info->comment),
				    //COMMENT_COLUMN, g_strdup_printf ("%p", info),
				    AUDIO_INFO_COLUMN, info,
				    -1);
		g_free (time);
	}
	/* release song list */
	/* FIXME: rather not free the audio info's here, we need them no ? */
	//eel_g_list_free_deep_custom (song_list, (GFunc) audio_info_free, NULL);
	return FALSE;
}



/* updates the view with info from the uri 
 * shows file names, then starts handler to get rest of info
 * to get a more responsive ui
 */
static void
audio_view_update (AudioView *view)
{
	GnomeVFSResult result;
	GnomeVFSFileInfo *current_file_info;
	GList *list, *node;

	GList *p;
	GList *audio_list = NULL;
	GList *attributes;
	AudioInfo *info;
	char *path_uri, *escaped_name;
	char *path;
        gchar *message;
	GtkTreeIter iter;

	int file_index;
	int image_count;

	/* try reading the dir */
	result = gnome_vfs_directory_list_load (&list, view->location,
				GNOME_VFS_FILE_INFO_GET_MIME_TYPE
				| GNOME_VFS_FILE_INFO_FOLLOW_LINKS);
	if (result != GNOME_VFS_OK) {
		path = (char *) gnome_vfs_get_local_path_from_uri (view->location);
		message = g_strdup_printf (_("Sorry, but there was an error reading %s."), path);
		eel_show_error_dialog (message, _("Can't Read Folder"),
				       GTK_WINDOW (gtk_widget_get_toplevel (GTK_WIDGET (view->event_box))));
		g_free (path);
		g_free (message);

		return;
	}

	/* iterate through the directory, collecting playable audio files 
	 * and FIXME: extracting metadata data if present */
	for (node = list; node != NULL; node = node->next) {
        	current_file_info = node->data;

	        /* skip invisible files, for now */
	        if (current_file_info->name[0] == '.') continue;

                escaped_name = (char *) gnome_vfs_escape_string (current_file_info->name);
                path_uri = g_build_filename (view->location, 
				             current_file_info->name, NULL);
                g_free (escaped_name);

		/* we will get NULL if it's not playable */
                info = audio_info_new (path_uri);
		if (info)
		{
			/* add to list */
	        	info->path = path_uri;
			info->file = g_strdup (current_file_info->name);
		        file_index += 1;
			audio_list = g_list_prepend (audio_list, info);
		}
        }
        gnome_vfs_file_info_list_free (list);

        audio_list = g_list_reverse (audio_list);

        /* populate the list with filenames */
        gtk_list_store_clear (view->list_store);
        for (p = audio_list; p != NULL; p = p->next) {
		info = (AudioInfo *) p->data;

		gtk_list_store_append (view->list_store,
				       &iter);
		gtk_list_store_set (view->list_store,
				    &iter,
				    FILE_COLUMN, g_strdup (info->file),
				    TYPE_COLUMN, g_strdup (info->type),
				    AUDIO_INFO_COLUMN, info,
				    -1);
	}
	view->audio_list = audio_list;
	/* release song list */
	/* FIXME: rather not free the audio info's here, we need them no ? */
	/* so where do we move it to ? */
	//eel_g_list_free_deep_custom (song_list, (GFunc) audio_info_free, NULL);

	//g_free (uri);
	//FIXME: add back metadata
	//g_idle_add ((GSourceFunc) audio_view_get_metadata_idler, view);
}

/* set up the tree view based on the AudioView */
static void
set_up_tree_view (AudioView *view)
{
	GtkCellRenderer *cell;
        GtkTreeViewColumn *column;
        GtkTreeView *tree_view;

        tree_view = GTK_TREE_VIEW (view->tree_view);
	gtk_tree_view_set_rules_hint (tree_view, TRUE);

        /* The file column */
        cell = gtk_cell_renderer_text_new ();
	g_object_set (cell, "yalign", 0.0, NULL);

        column = gtk_tree_view_column_new_with_attributes (_("File"),
                                                           cell,
                                                           "text", 
                                                           FILE_COLUMN,
							   NULL);
        gtk_tree_view_column_set_sort_column_id (column, FILE_COLUMN);
        gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_append_column (tree_view, column);

        /* The type column */
        cell = gtk_cell_renderer_text_new ();
	g_object_set (cell, "yalign", 0.0, NULL);
        column = gtk_tree_view_column_new_with_attributes (_("Type"),
                                                           cell,
                                                           "text", 
                                                           TYPE_COLUMN,
							   NULL);
        gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_append_column (tree_view, column);

        /* The length column */
        cell = gtk_cell_renderer_text_new ();
	g_object_set (cell, "yalign", 0.0, NULL);
        column = gtk_tree_view_column_new_with_attributes (_("Length"),
                                                           cell,
                                                           "text", 
                                                           LENGTH_COLUMN,
							   NULL);
        gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_append_column (tree_view, column);

        /* The comment column */
        column = gtk_tree_view_column_new_with_attributes (_("Comment"),
                                                           cell,
                                                           "text", 
                                                           COMMENT_COLUMN,
							   NULL);
        gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_append_column (tree_view, column);
}

GtkWidget *
audio_view_get_widget (AudioView *view)
{
	return view->widget;
}

AudioView *
audio_view_new ()
{
	GError *error = NULL;
	AudioView *view;
	GstElement *audio_sink;
	GtkWidget *image;

	view = g_new0 (AudioView, 1);
	/* get a toplevel vbox */
	view->widget = gtk_vbox_new (FALSE, 5);
	
	/* create list model */
	view->list_store = gtk_list_store_new (
		NUM_COLUMNS,
		G_TYPE_STRING,	/* FILE_COLUMN */
		G_TYPE_STRING,  /* TYPE_COLUMN */
		G_TYPE_STRING,  /* LENGTH_COLUMN */
		G_TYPE_STRING,  /* comment */
		G_TYPE_STRING,	/* path */
		G_TYPE_POINTER	/* AUDIO_INFO_COLUMN */
	);
	view->tree_view = gtk_tree_view_new_with_model (
		GTK_TREE_MODEL (view->list_store)
	);
        g_signal_connect (view->tree_view,
                          "row_activated",
                          G_CALLBACK (row_activated_callback),
                          view);

        g_object_unref (view->list_store);
        set_up_tree_view (view);

	/* get the scroll window */
	view->scroll_window = gtk_scrolled_window_new (NULL, NULL);
        gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (view->scroll_window),
                                        GTK_POLICY_AUTOMATIC, 
					GTK_POLICY_AUTOMATIC);
        gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (view->scroll_window),
                                             GTK_SHADOW_ETCHED_IN);
        gtk_container_add (GTK_CONTAINER (view->scroll_window), 
                           view->tree_view);
	gtk_box_pack_start (GTK_BOX (view->widget),
			    view->scroll_window, TRUE, TRUE, 0);

	/* get controls */
	view->control = gtk_hbox_new (FALSE, 0);

	/* prev */
	view->prev_button = control_box_add_labeled_button (
			view->control, "Previous", "prev.png");
	g_signal_connect (G_OBJECT (view->prev_button),
			  "clicked", G_CALLBACK (prev_activate), view);
	/* stop */
	view->stop_button = control_box_add_labeled_button (
			view->control, "Stop", "stop.png");
	g_signal_connect (G_OBJECT (view->stop_button),
			  "clicked", G_CALLBACK (stop_activate), view);

	/* play */
	view->play_button = control_box_add_labeled_button (
			view->control, "Play", "play.png");
	g_signal_connect (G_OBJECT (view->play_button),
			  "clicked", G_CALLBACK (play_activate), view);
	
	/* nex */
	view->next_button = control_box_add_labeled_button (
			view->control, "Next", "next.png");
	g_signal_connect (G_OBJECT (view->next_button),
			  "clicked", G_CALLBACK (next_activate), view);

	/* seek bar */
	/*
	view->seek_adj = gtk_adjustment_new (0.0, 0.0, 1.0, 
			                              0.01, 0.1, 1);
	view->seek_scale = gtk_hscale_new (view->seek_adj);
						      */
	view->seek_scale = gtk_hscale_new_with_range (0.0, 1.0, 0.001);
	gtk_range_set_update_policy (GTK_RANGE (view->seek_scale),
				     GTK_UPDATE_DELAYED);
	view->seek_changed_id = g_signal_connect (G_OBJECT (view->seek_scale), 
			                          "value_changed",
			  G_CALLBACK (seek_changed_callback), view);

	gtk_scale_set_draw_value (GTK_SCALE (view->seek_scale), FALSE);
	gtk_box_pack_start (GTK_BOX (view->control),
			   view->seek_scale, TRUE, TRUE, 0);

	/* time */
	view->time = gtk_label_new (NULL);
	audio_view_set_time (view, 0, 0);
	gtk_box_pack_start (GTK_BOX (view->control),
			   view->time, FALSE, FALSE, 0);

	gtk_box_pack_start (GTK_BOX (view->widget),
			    view->control, FALSE, FALSE, 0);

	/* get a status area */
	view->status = gtk_label_new ("Not playing");
	gtk_misc_set_alignment (GTK_MISC (view->status), 0.0, 0.5);
	gtk_box_pack_start (GTK_BOX (view->widget),
			    view->status, FALSE, FALSE, 0);

	/* initialize GStreamer */
	gst_init (NULL, NULL);

	/* create a play object */
	view->audio_play = audio_play_new (&error);
	g_assert (IS_AUDIO_PLAY (view->audio_play));
	if (error) g_error ("got an error\n");
	audio_sink = gst_gconf_get_default_audio_sink ();
	g_assert (GST_IS_ELEMENT (audio_sink));
	audio_play_set_audio_sink (view->audio_play, audio_sink);
	g_signal_connect (G_OBJECT (view->audio_play), "eos",
			  G_CALLBACK (have_eos_callback), view);
	g_signal_connect (G_OBJECT (view->audio_play), "tick",
			  G_CALLBACK (have_tick_callback), view);

	return view;
}

void
audio_view_dispose (AudioView *view)
{
	g_print ("DEBUG: audio_view_finalize\n");
	/* no need to destroy widgets, gets done automatically on toplevel */
	g_print ("DEBUG: audio_play: %p\n", view->audio_play);
	audio_play_dispose (view->audio_play);
}
