/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */

/* 
 * Copyright (C) 2002 Thomas Vander Stichele
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Thomas Vander Stichele <thomas at apestaart dot org>
 */

/* audio-info.c - get information on files using AudioInfo object
 */

#include <config.h>
#include <libgnomevfs/gnome-vfs-mime-utils.h>
#include <gst/play/play.h>
#include "audio-info.h"

static void
have_information (GstPlay *play, GstElement *element, GParamSpec *param,
		  AudioInfo *info)
{
	const gchar *pname;
	
	g_assert (GST_IS_ELEMENT (element));
	g_return_if_fail (G_IS_PARAM_SPEC (param));
	pname = g_param_spec_get_name (param);

	g_print ("have information for param %s\n", pname);
	if (strcmp (pname, "channels") == 0)
	{
		g_print ("getting channels\n");
		gst_element_get (element, "channels", 
			         &(info->channels));
	}
	else if (strcmp (pname, "samplerate") == 0)
	{
		g_print ("getting samplerate\n");
		gst_element_get (element, "samplerate", 
			         &(info->samplerate));
	}
	else if (strcmp (pname, "bitrate") == 0)
	{
		g_print ("getting bitrate\n");
		gst_element_get (element, "bitrate", 
			         &(info->bitrate));
	}
	else if (strcmp (pname, "bitrate-nominal") == 0) /* vorbis */
	{
		g_print ("getting bitrate-nominal\n");
		gst_element_get (element, "bitrate-nominal", 
			         &(info->bitrate));
	}
	else
	{
		g_print ("unhandled information\n");
	}
	g_print ("handled info.\n");
	if (info->channels && info->samplerate && info->bitrate)
	{
		g_print ("got the goods, bailing out\n");
		gst_main_quit ();
	}
	
}

/* metadata callback */
/* this should move into a gst lib */
static void
have_metadata_callback (GstElement *element, GstCaps **metadatap)
{
        GstCaps *metadata;
        const gchar *string;

	g_print ("we have metadata\n");
        g_object_get (G_OBJECT (element), "metadata", &metadata, NULL);
        if (metadata == NULL)
	{ 
		*metadatap = NULL;
		return;
	}
	*metadatap = metadata;
	return;
}

/* return allocated string representation of metadata */
static gchar *
metadata_to_string (GstCaps *metadata)
{
	gchar *string;
	gint size;
	gchar **string_array, **sap;
	GList *props;

	g_print ("DEBUG: have metadata %p\n", metadata);
        if (!metadata) return NULL;
        gst_caps_debug (metadata, "metadata caps");
	if (!metadata->properties) return NULL;
	props = metadata->properties->properties;
	size = g_list_length (props);
	string_array = g_malloc (sizeof (gchar *) * (size + 1));
	for (sap = string_array; sap < string_array + size; ++sap)
		*sap = NULL;
	sap = string_array;
        while (props) 
	{
        	const GstPropsEntry *entry = (GstPropsEntry *) props->data;
                const gchar *name = gst_props_entry_get_name (entry);
		/* only do artists and titles */
		if (g_ascii_strncasecmp (name, "artist", 6) == 0 ||
		    g_ascii_strncasecmp (name, "title", 5) == 0)
		{
                	switch (gst_props_entry_get_type (entry)) 
			{
                		case GST_PROPS_STRING_TYPE:
                  			gst_props_entry_get_string (entry, 
						(const gchar **) &string);
					*sap = g_strdup_printf  ("%s: %s", 
                                                         	 name, 
							 	 string);
					++sap;
                  			break;
                		default:
                  			break;
                	}
		}
                props = g_list_next (props);
	}
	*sap = NULL;
	string = g_strjoinv ("\n", string_array);
	for (sap = string_array; sap < string_array + size; ++sap)
		if (*sap) g_free (*sap);
	g_free (string_array);
	return string;
}

static gchar *
audio_info_get_vorbis_metadata (const gchar *location)
{
	GstElement *bin;
        GstElement *source, *vorbisfile;
	GstCaps *metadata = NULL;
        gint streams;
        gchar *streaminfo = NULL;
        int i;

        bin = gst_pipeline_new (NULL);
        source = gst_element_factory_make ("gnomevfssrc", "source");
        vorbisfile = gst_element_factory_make ("vorbisfile", "vorbisfile");
        g_assert (GST_IS_ELEMENT (source));
        g_assert (GST_IS_ELEMENT (vorbisfile));
        gst_element_set (source, "location", location, NULL);

        gst_bin_add_many (GST_BIN (bin), source, vorbisfile, NULL);
        gst_element_connect (source, vorbisfile);

        g_assert (GST_IS_BIN (bin));

	g_print ("metadata ptr: %p\n", metadata);
        g_signal_connect (G_OBJECT (vorbisfile), "have_metadata",
                          G_CALLBACK (have_metadata_callback), &metadata);
        gst_element_set_state (GST_ELEMENT (bin), GST_STATE_PLAYING);
        gst_bin_iterate (GST_BIN (bin));
        /* ask for number of streams */
        g_object_get (G_OBJECT (vorbisfile), "streams", &streams, NULL);
        g_print ("Number of logical bitstreams: %d\n", streams);
	g_print ("metadata ptr: %p\n", metadata);
	if (metadata != NULL)
		streaminfo = metadata_to_string (metadata);
	else
		g_print ("warning: didn't get metadata !\n");
	return streaminfo;
}

static void
have_stream_end (GstPlay *play, gint64 length_nanosec, AudioInfo *info)
{
	g_print ("eos, quitting\n");
	gst_main_quit ();
}

static void
have_stream_length (GstPlay *play, gint64 length_nanosec, AudioInfo *info)
{
	g_print ("DEBUG: have_stream_length, %lld\n", length_nanosec);
	g_print ("DEBUG: comment: %s\n", info->comment);
	info->have_length = TRUE;
	info->length = (gint) (length_nanosec / GST_MSECOND);
	/* FIXME: for now  we're happy when we have the length */
	gst_main_quit ();
}

/* check if the uri is playable and return the mime type if it is, NULL
 * if not */
char *
audio_info_get_mime_type_playable (const char *uri)
{
	char *type = NULL;

	type = gnome_vfs_get_mime_type (uri);
	/* FIXME: assuming we get an allocated copy and have ownership */
	if (!type) return NULL;
	if (
	    (strcmp (type, "application/x-ogg") != 0) &&
	    (strcmp (type, "audio/x-flac") != 0) &&
	    (strcmp (type, "audio/x-mp3") != 0) &&
	    (strcmp (type, "audio/x-wav") != 0)
	    )
	{
		g_free (type);
		return NULL;
	}
	return type;
}
	
/* updates an allocated AudioInfo with metadata */
void audio_info_update (AudioInfo *info)
{
	gulong stream_length_id, stream_end_id, information_id; /* signals */
	static GstPlay *play = NULL;
	static GstElement *fakesink = NULL;
	GError *error = NULL;
	
	if (!fakesink)
	{
		g_print ("creating fakesink\n");
		fakesink = gst_element_factory_make ("fakesink", "fakesink");
		g_assert (GST_IS_ELEMENT (fakesink));
		g_object_set (G_OBJECT (fakesink), "silent", TRUE, NULL);
	}
	if (!play) 
	{ 
		g_print ("creating play object\n");
		play = gst_play_new (GST_PLAY_PIPE_VIDEO, &error); 
		if (error)
		{
			g_warning ("Error: %s\n", error->message);
			g_error_free (error);
		}
		g_assert (GST_IS_PLAY (play));
		g_assert (GST_IS_ELEMENT (fakesink));
		gst_play_set_audio_sink (play, fakesink);
	}

	gst_play_set_location (play, info->path);
	stream_length_id = g_signal_connect (G_OBJECT (play), "stream_length", 
			  	  G_CALLBACK (have_stream_length), 
				  (gpointer) info);
	/* FIXME: we should rethink how to get information */
	/*
	information_id = g_signal_connect (G_OBJECT (play), "information", 
			  	  G_CALLBACK (have_information), 
				  (gpointer) info);
				  */
	stream_end_id = g_signal_connect (G_OBJECT (play), "stream_end", 
			  	  G_CALLBACK (have_stream_end), 
				  (gpointer) info);
	gst_play_set_state (play, GST_STATE_PLAYING);
	gst_main ();
	if (info->have_length)
	{
		g_print ("Length in seconds: %d.%03d\n", 
		 	 info->length / 1000, info->length % 1000);
	}
	else
	{
		g_print ("Could not get length !\n");
	}
	g_signal_handler_disconnect (play, stream_length_id);
	g_signal_handler_disconnect (play, stream_end_id);
	/* g_signal_handler_disconnect (play, information_id); */
	gst_play_set_state (play, GST_STATE_READY);

	/* get metadata */
	if (strcmp (info->type, "application/x-ogg") == 0)
		info->comment = audio_info_get_vorbis_metadata (info->path);
	else
		info->comment = g_strdup ("No comment.");
}

/* creates and initializes AudioInfo based on uri
 * Returns new AudioInfo if the file is playable, NULL if not */
AudioInfo *
audio_info_new (const char *uri)
{
	AudioInfo *info;
	char *type = audio_info_get_mime_type_playable (uri);
       
	if (!type) return NULL;
 	info = g_new0 (AudioInfo, 1);
	info->type = type;
	
	/* initialize */
	info->have_length = FALSE;
	info->length = 0;
	info->channels = 0;
	info->bitrate = 0;
	info->samplerate = 0;

	info->path = g_strdup_printf (uri);
	info->file = NULL;
	info->comment = NULL;

	return info;
}

/* frees AudioInfo */
void
audio_info_free (AudioInfo *info)
{
	if (info->file) g_free (info->file);
	if (info->type) g_free (info->type);
	if (info->comment) g_free (info->comment);
	if (info->path) g_free (info->path);

	g_free (info);
}


