#include <locale.h>
#include <config.h>
#include <glade/glade.h>
#include "gststatusarea.h"

static void gst_status_area_class_init (GstStatusAreaClass *klass);
static void gst_status_area_init       (GstStatusArea *area);
static gint gst_status_area_text_width (GtkWidget *widget, const gchar *text);
static void gst_status_area_toggle_media_info(GtkToggleButton *togglebutton, GstStatusArea *area);

/* signals and args */
enum {
	LAST_SIGNAL
};

enum {
	ARG_0,
};

#define STR_READY   "Ready"
#define STR_PLAYING "Playing"
#define STR_PAUSED  "Paused"
#define STR_STOPPED "Stopped"

static GtkDrawingArea *parent_class = NULL;

GtkType 
gst_status_area_get_type (void) 
{
	static GtkType status_area_type = 0;

	if (!status_area_type) {
		static const GtkTypeInfo status_area_info = {
			"GstStatusArea",
			sizeof(GstStatusArea),
			sizeof(GstStatusAreaClass),
			(GtkClassInitFunc)gst_status_area_class_init,
			(GtkObjectInitFunc)gst_status_area_init,
			NULL,
			NULL,
			(GtkClassInitFunc)NULL,
		};
		status_area_type = gtk_type_unique (gtk_hbox_get_type (),&status_area_info);
	}
	return status_area_type;
}

static void 
gst_status_area_class_init (GstStatusAreaClass *klass) 
{
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;

	parent_class = gtk_type_class (gtk_widget_get_type ());

	object_class = (GtkObjectClass*)klass;
	widget_class = (GtkWidgetClass*)klass;
}

static void 
gst_status_area_init (GstStatusArea *area) 
{

}

GtkWidget *
gst_status_area_new (GladeXML *xml) 
{
	GstStatusArea *area;
	GtkWidget *widget;
	gint width;

	area = gtk_type_new (GST_TYPE_STATUS_AREA);
	area->xml = xml;
	gtk_box_pack_start (GTK_BOX (area), glade_xml_get_widget(area->xml, "vbox_media_info"), TRUE, TRUE, 0);

	/* find the maximum space that the state label will take up */
	widget = glade_xml_get_widget(area->xml, "label_state");
	width = gst_status_area_text_width(widget, STR_READY);
	width = MAX(width, gst_status_area_text_width(widget, STR_PAUSED));
	width = MAX(width, gst_status_area_text_width(widget, STR_PLAYING));
	width = MAX(width, gst_status_area_text_width(widget, STR_STOPPED));
	gtk_widget_set_size_request (widget, width, 1);
	
	widget = glade_xml_get_widget(area->xml, "label_time");
	width = gst_status_area_text_width(widget, "00:00 / 00:00");
	gtk_widget_set_size_request (widget, width, 1);

	glade_xml_signal_connect_data(area->xml, "button_media_info_toggle", G_CALLBACK (gst_status_area_toggle_media_info), area);
	gst_status_area_reset_media_info(area);

	return GTK_WIDGET (area);
}

void 
gst_status_area_set_state (GstStatusArea *area, GstElementState old_state, GstElementState new_state) 
{
	GtkWidget *widget;
	g_return_if_fail(GST_IS_STATUS_AREA(area));

	widget = glade_xml_get_widget(area->xml, "label_state");
	g_return_if_fail(widget != NULL);
	switch (new_state){
	case GST_STATE_PLAYING:
		gtk_label_set_text(GTK_LABEL(widget), STR_PLAYING);
		break;
	case GST_STATE_PAUSED:
		gtk_label_set_text(GTK_LABEL(widget), STR_PAUSED);
		break;
	case GST_STATE_READY:
		if (old_state = GST_STATE_PAUSED){
			gtk_label_set_text(GTK_LABEL(widget), STR_STOPPED);
		}
		else {
			gtk_label_set_text(GTK_LABEL(widget), STR_READY);
		}
		break;
	default:
		gtk_label_set_text(GTK_LABEL(widget), STR_READY);
		break;
	}
}

static void
gst_status_area_toggle_media_info(GtkToggleButton *togglebutton, GstStatusArea *area)
{
	g_return_if_fail(GST_IS_STATUS_AREA(area));
	if (gtk_toggle_button_get_active(togglebutton)){
		gtk_widget_show(glade_xml_get_widget(area->xml, "frame_media_info"));
	}
	else {
		gtk_widget_hide(glade_xml_get_widget(area->xml, "frame_media_info"));
	}

}

void
gst_status_area_reset_media_info(GstStatusArea *area)
{
	GtkWidget *button_media_info;
	g_return_if_fail(GST_IS_STATUS_AREA(area));

	button_media_info = glade_xml_get_widget(area->xml, "button_media_info");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button_media_info), FALSE);
	gtk_widget_set_sensitive(button_media_info, FALSE);

	gtk_widget_hide(glade_xml_get_widget(area->xml, "label_label_channels"));
	gtk_widget_hide(glade_xml_get_widget(area->xml, "label_channels"));

	gtk_widget_hide(glade_xml_get_widget(area->xml, "label_label_sample_rate"));
	gtk_widget_hide(glade_xml_get_widget(area->xml, "label_sample_rate"));

	gtk_widget_hide(glade_xml_get_widget(area->xml, "label_label_bitrate"));
	gtk_widget_hide(glade_xml_get_widget(area->xml, "label_bitrate"));

	gtk_widget_hide(glade_xml_get_widget(area->xml, "label_label_framerate"));
	gtk_widget_hide(glade_xml_get_widget(area->xml, "label_framerate"));
}

void
gst_status_area_set_location (GstStatusArea *area, const gchar *location)
{
	g_return_if_fail(GST_IS_STATUS_AREA(area));
	g_return_if_fail(location != NULL);
	gtk_label_set_text(GTK_LABEL(glade_xml_get_widget(area->xml, "label_media_name")), location);
}

static void
gst_status_area_update_time_label(GstStatusArea *area)
{
	gchar *time_str;
	gint time_seconds = (gint) (area->time_nanos / GST_SECOND);
	gint length_seconds = (gint) (area->length_nanos / GST_SECOND);

	if (area->length_nanos <= 0LL){
		time_str = g_strdup_printf("%02d:%02d", time_seconds / 60, time_seconds % 60);
	}
	else {
		time_str = g_strdup_printf("%02d:%02d / %02d:%02d", 
		                           time_seconds / 60, time_seconds % 60,
		                           length_seconds / 60, length_seconds % 60);
	}

	gtk_label_set_text(GTK_LABEL(glade_xml_get_widget(area->xml, "label_time")), time_str);

	if (area->time_str){
		g_free(area->time_str);
	}
	area->time_str = time_str;
}

void
gst_status_area_set_time(GstStatusArea *area, gint64 time_nanos)
{
	g_return_if_fail(GST_IS_STATUS_AREA(area));
	area->time_nanos = time_nanos;
	gst_status_area_update_time_label(area);

}

void
gst_status_area_set_length(GstStatusArea *area, gint64 length_nanos)
{
	g_return_if_fail(GST_IS_STATUS_AREA(area));
	area->length_nanos = length_nanos;
	gst_status_area_update_time_label(area);
}

void
gst_status_area_set_information(GstStatusArea *area, GstElement *element, GParamSpec *param)
{
	const gchar *pname;
	GtkWidget *widget;
	g_return_if_fail(GST_IS_STATUS_AREA(area));
	g_return_if_fail(GST_IS_ELEMENT(element));
	g_return_if_fail(G_IS_PARAM_SPEC(param));
	pname = g_param_spec_get_name(param);

	if (strstr(pname, "channels")){
		gint val;
		g_object_get(G_OBJECT(element), "channels", &val, NULL);

		widget = glade_xml_get_widget(area->xml, "label_channels");
		gtk_label_set_markup(GTK_LABEL(widget), g_strdup_printf("<small>%d</small>", val));
		gtk_widget_show(widget);

		gtk_widget_show(glade_xml_get_widget(area->xml, "label_label_channels"));
		gtk_widget_set_sensitive(glade_xml_get_widget(area->xml, "button_media_info"), TRUE);
		return;
	}
	if (strstr(pname, "samplerate")){
		gint val;
		gfloat fval;
		g_object_get(G_OBJECT(element), "samplerate", &val, NULL);
		fval = (gfloat)val / 1000;

		widget = glade_xml_get_widget(area->xml, "label_sample_rate");
		gtk_label_set_markup(GTK_LABEL(widget), g_strdup_printf("<small>%.1f kHz</small>", fval));
		gtk_widget_show(widget);

		gtk_widget_show(glade_xml_get_widget(area->xml, "label_label_sample_rate"));
		gtk_widget_set_sensitive(glade_xml_get_widget(area->xml, "button_media_info"), TRUE);
		return;
	}
	if (strstr(pname, "bitrate")){
		gint val;
		gfloat fval;
		g_object_get(G_OBJECT(element), "bitrate", &val, NULL);
		fval = (gfloat)val / 1000;

		widget = glade_xml_get_widget(area->xml, "label_bitrate");
		gtk_label_set_markup(GTK_LABEL(widget), g_strdup_printf("<small>%.0f kb/s</small>", fval));
		gtk_widget_show(widget);

		gtk_widget_show(glade_xml_get_widget(area->xml, "label_label_bitrate"));
		gtk_widget_set_sensitive(glade_xml_get_widget(area->xml, "button_media_info"), TRUE);
		return;
	}
	if (strstr(pname, "frame-rate")){
		gfloat val;
		g_object_get(G_OBJECT(element), "frame-rate", &val, NULL);

		widget = glade_xml_get_widget(area->xml, "label_framerate");
		gtk_label_set_markup(GTK_LABEL(widget), g_strdup_printf("<small>%.0f fps</small>", val));
		gtk_widget_show(widget);

		gtk_widget_show(glade_xml_get_widget(area->xml, "label_label_framerate"));
		gtk_widget_set_sensitive(glade_xml_get_widget(area->xml, "button_media_info"), TRUE);
		return;
	}
}

static gint
gst_status_area_text_width (GtkWidget *widget, const gchar *text)
{
	PangoLayout *layout;
	gint width, height;

	layout = gtk_widget_create_pango_layout(widget, text);
	pango_layout_get_pixel_size(layout, &width, &height);

	g_object_unref(G_OBJECT(layout));

	return width;
}

