/*
 * $Id: g_session.c,v 1.5 1998/05/03 23:09:11 jochen Exp $
 * GXSNMP -- An snmp management application
 * Copyright (C) 1998 Gregory McLean & Jochen Friedrich
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc.,  59 Temple Place - Suite 330, Cambridge, MA 02139, USA.
 *
 * Snmp asynchronous session support
 */
#ifndef lint
static char const copyright[] =
"@(#) Copyright (c) 1998 Gregory McLean & Jochen Friedrich";
#endif
static char const rcsid[] =
"$Id: g_session.c,v 1.5 1998/05/03 23:09:11 jochen Exp $";

#include <g_snmp.h>
#include <sys/socket.h>

extern int     errno;

static GSList  *rq_list     = NULL;   /* track the active requests */
static gint     id          = 1;      /* SNMP request id */
       gint     snmpsocket  = 0;      /* file handle for SNMP traffic */

/*
 * lookup address of node and set port number or default.
 * Maybe this one should be nuked completely. gnome has a nice async
 * dns helper and the collector might use cached IP addresses for speedup
 * anyways.
 *
 */

gboolean
g_setup_address (host_snmp *host)
{
  struct servent *port;
  struct hostent *hp;
 
  host->address.sin_family = AF_INET;
  hp = gethostbyname(host->name);
  if (!hp)
    return FALSE;

  g_memmove((char *)&host->address.sin_addr, (char *)hp->h_addr, hp->h_length);
  if (host->port) 
    {
      host->address.sin_port = htons(host->port);
    }
  else 
    {
      port = getservbyname("snmp","udp");
      if (!port)
        return FALSE;
      host->address.sin_port = port->s_port;
    }
  return TRUE;
}

/* 
 * query/set one mib from a snmp host
 *
 * host    -- Host info in question
 * callback-- Pointer to function that will handle the reply 
 *
 */      
gpointer
g_async_send (host_snmp *host, guint req, GSList *objs, guint arg1, guint arg2)
{
  snmp_request * request;
  time_t         now;
  guchar         buffer[MAX_DGRAM_SIZE], *ptr;
  guint          len;

  now = time(NULL);

  request = g_malloc(sizeof(snmp_request));

  if (host->done_callback)
    request->callback = host->done_callback;
  else
    request->callback = NULL;

  if (host->time_callback)
    request->timeout  = host->time_callback;
  else
    request->timeout  = NULL;

  if (!g_setup_address(host))
    {
      g_slist_free(objs);
      g_free(request);
      return NULL;
    }

  request->pdu.request.id           = id++;
  request->pdu.request.type         = req;
  request->pdu.request.error_status = arg1;
  request->pdu.request.error_index  = arg2;

/* This should go to g_auth.c */

  if (req == SNMP_PDU_SET)
    {
      strcpy(request->auth.name, host->wcomm);
      request->auth.nlen            = strlen(host->wcomm);
    }
  else
    {
      strcpy(request->auth.name, host->rcomm);
      request->auth.nlen            = strlen(host->rcomm);
    }

  request->auth.type                = AUTH_COMMUNITY;

  request->retries                  = host->retries;
  request->timeoutval               = host->timeoutval;
  request->magic                    = host->magic;
  request->version                  = host->version;
  request->host                     = host;
  request->time                     = now + request->timeoutval;

  g_memmove(&request->address, &host->address, sizeof(struct sockaddr_in));

  ptr = buffer;
  len = sizeof(buffer);

  g_snmp_encode(&ptr, &len, &request->pdu, &request->auth, request->version, 
                objs);

  g_slist_free(objs);

  request->buffer = g_malloc(len);
  g_memmove((char *)request->buffer, ptr, len);
  request->length = len;
 
  rq_list = g_slist_append(rq_list, request);

  /* 
   * What happens if sendto returns -1 (ie failed?)
   */
  sendto(snmpsocket, request->buffer, request->length, 0, 
    (struct sockaddr *)&request->address, sizeof(request->address));
 
  return request;
}

gpointer
g_async_get (host_snmp *host, GSList *pdu)
{
  return g_async_send(host, SNMP_PDU_GET, pdu, 0, 0);
}

gpointer
g_async_getnext (host_snmp *host, GSList *pdu)
{
  return g_async_send(host, SNMP_PDU_NEXT, pdu, 0, 0);
}

gpointer
g_async_bulk (host_snmp *host, GSList *pdu, guint nonrep, guint maxiter)
{
  return g_async_send(host, SNMP_PDU_BULK, pdu, nonrep, maxiter);
}

gpointer
g_async_set (host_snmp *host, GSList *pdu)
{
  return g_async_send(host, SNMP_PDU_SET, pdu, 0, 0);
}

gboolean
g_pdu_add_name(GSList **pdu, guchar *name, guchar type, gpointer value)
{
  guint  id_len;
  gulong id[SNMP_SIZE_OBJECTID];
  
  id_len  = SNMP_SIZE_OBJECTID;

  if (!read_objid (name, id, &id_len))
    return FALSE;
  return g_pdu_add_oid(pdu, id, id_len, type, value);
}

gboolean
g_pdu_add_oid(GSList **pdu, gulong *myoid, guint mylength, guchar type, 
              gpointer value)
{
  struct _SNMP_OBJECT *obj;

  obj = g_malloc(sizeof(struct _SNMP_OBJECT));

  g_memmove(obj->id, myoid, mylength * sizeof(gulong));
  obj->request = 0;
  obj->id_len  = mylength;
  obj->type    = type;

  switch(type) 
    {
      case SNMP_INTEGER:
        obj->syntax.lng     = *((glong *) value);
        obj->syntax_len     = sizeof(glong);
        break;
      case SNMP_COUNTER:
      case SNMP_GAUGE:
      case SNMP_TIMETICKS:
        obj->syntax.ulng    = *((gulong *) value);
        obj->syntax_len     = sizeof(gulong);
        break;
      case SNMP_OCTETSTR:
      case SNMP_OPAQUE:
        strncpy(obj->syntax.bufchr, (guchar *) value, SNMP_SIZE_BUFCHR);
        obj->syntax_len     = strlen((guchar *) value);
        break;
      case SNMP_NULL:
        break;
      default:
        g_free(obj);
        return FALSE;
    }

  *pdu = g_slist_append(*pdu, obj);
  return TRUE;
}

/* This should be nuked once the new parser and mib module are available.
   For now, either use this or the print function in struct tree          */

void 
g_snmp_printf(char *buf, int buflen, struct _SNMP_OBJECT *obj)
{
  int timeticks, seconds, minutes, hours, days;

  /*
   * Changed all the sprintf's to snprintf, paranoid I know but
   * I'd rather not get caught with any buffer overflows..
   */
  switch(obj->type)
    {
      case SNMP_INTEGER:
        snprintf(buf, buflen, "%d", obj->syntax.lng);
        break;
      case SNMP_COUNTER:
      case SNMP_GAUGE:
        snprintf(buf, buflen, "%u", obj->syntax.ulng);
        break;
      case SNMP_TIMETICKS:
	/* replaced this duplicated code with a call to existing code */
	timetick_string (obj->syntax.ulng, buf);
        break;
      case SNMP_OCTETSTR:
      case SNMP_OPAQUE:
	/*
         * Changed to strncpy to prevent potential buffer overruns
         * as that appears to be all the rage to attempt to crash/hack/
         * crack systems. Am I being paranoid?
         */
        strncpy(buf, obj->syntax.bufchr,
                obj->syntax_len > buflen ? buflen: obj->syntax_len);
	/*
         * BUG FIX: the returned object is not _always_ terminated
         *          with a null. This ensures that it in fact is.
         */
	buf[obj->syntax_len > buflen ? buflen: obj->syntax_len] = '\0';
        break;
      case SNMP_IPADDR:
        snprintf(buf, buflen, "%d.%d.%d.%d", obj->syntax.bufchr[0],
                                             obj->syntax.bufchr[1],
                                             obj->syntax.bufchr[2],
                                             obj->syntax.bufchr[3]);
    }
}

/*
 * The request queue functions
 */

snmp_request *
g_find_request (guint reqid)
{
  GSList       * list;
  snmp_request * retval;

  list = rq_list;
  while(list)
    {
      retval = (snmp_request *) list->data;
      if(retval->pdu.request.id == reqid)
	  return retval;
      list = list->next;
    }
  return NULL;
}

gboolean
g_remove_request (snmp_request *request)
{
  rq_list = g_slist_remove(rq_list, request);
  g_free(request->buffer);
  g_free(request);
  return TRUE;
}

/*
 * The low level callbacks
 */

int
g_snmp_timeout_cb (gpointer data)
{
  GSList       *mylist;
  time_t        now;
  snmp_request *request;
  char          buf[256];

  now = time(NULL);
  mylist = rq_list;

  while(mylist)
    {
      request = (snmp_request *) mylist->data;
      mylist = mylist->next;
      if (request->time <= now)
        {
          if (request->retries)
            {
              request->retries--;
              request->time = now + request->timeoutval;
	      /* 
               * Again what happens on a -1 return to sendto
               * (ie a failure)
               *
               * We simply ignore the failure and will retry later. A failure
               * may happen i.e. if an ICMP arrives on certain architectures
               * or if the kernel is tight on buffer space. We really can't do
               * any better here. -- Jochen
               *
	       */
              sendto(snmpsocket, request->buffer, request->length, 0, 
                     (struct sockaddr *)&request->address, 
                     sizeof(request->address));
            }
          else
            {
              if (request->timeout)
                {
                   request->timeout(request->host, request->magic);
                }
              g_remove_request(request);
            }
        }
    }
  return TRUE;
}

void
g_snmp_input_cb (gpointer data)
{
  guchar              buffer[MAX_DGRAM_SIZE];
  char                buf[256];
  int                 adrsize, len, objlen, comsize;
  GSList             *objs;
  SNMP_PDU            spdu;
  SNMP_AUTH           auth;
  snmp_request       *request;
  struct sockaddr_in  address;
  guint               version;

  /*
   * need to init adrsize properly
   */
  adrsize = sizeof (address);
  len = recvfrom(snmpsocket, buffer, sizeof(buffer), 0, 
                 (struct sockaddr *)&address, &adrsize);

  /*
   * Need to check the return val of recvfrom
   */
  if (len <= 0) return;
  
  /* All we can do for now */

  g_snmp_decode(buffer, len, &spdu, &auth, &version, &objs);

  if (request = g_find_request(spdu.request.id))
    {
      if (request->pdu.request.type == SNMP_PDU_SET)
        {
          if (memcmp(auth.name, request->auth.name, auth.nlen))
            {
              g_slist_free(objs);
              return;
            }
        }
      else
        {
          if (memcmp(auth.name, request->auth.name, auth.nlen))
            {
              g_slist_free(objs);
              return;
            }
        }
      if (memcmp(&address.sin_addr, &request->address.sin_addr,
          sizeof (struct in_addr)))
        {
          g_slist_free(objs);
          return;
        }
      request->host->status = spdu.request.error_status;
      if (request->callback)
        {
          request->callback(request->host, request->magic, &spdu, objs);
        }
      g_remove_request(request);
    }
  g_slist_free(objs); 
}

/* EOF */
