/* GnomENIUS Calculator
 * Copyright (C) 1997 George Lebl.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <string.h>
#include <glib.h>
#include "eval.h"
#include "dict.h"
#include "util.h"
#include "funclib.h"

static context_t context={NULL,-1,0};

/*return current context number (0 is global, -1 is uninitialized)*/
int
d_curcontext(void)
{
	return context.top;
}

/*make builtin function and return it*/
func_t *
d_makebifunc(char *id, dictfunc f, int nargs, int dynamic)
{
	func_t *n;

	n=(func_t *)g_malloc(sizeof(func_t));
	n->id=(char *)g_malloc(strlen(id)+1);
	strcpy(n->id,id);
	n->data.func=f;
	n->nargs=nargs;
	n->context=context.top;
	n->dynamic=dynamic;
	n->type=BUILTIN_FUNC;

	return n;
}

/*make a user function and return it*/
func_t *
d_makeufunc(char *id, tree_t *value, dict_t *dict, int nargs, int dynamic)
{
	func_t *n;

	n=(func_t *)g_malloc(sizeof(func_t));
	n->id=(char *)g_malloc(strlen(id)+1);
	strcpy(n->id,id);
	n->data.user.value=value;
	n->data.user.dict=dict;
	n->nargs=nargs;
	n->context=context.top;
	n->dynamic=dynamic;
	n->type=USER_FUNC;

	return n;
}

/*make a new blank dictionary and return it*/
dict_t *
d_newdict(void)
{
	dict_t *n;
	n=(dict_t *)g_malloc(sizeof(dict_t));
	n->size=256;
	n->top=-1; /*-1 means that dictionary is empty!*/
	n->dict=(func_t * *)g_malloc(sizeof(func_t *)*256);
	return n;
}

void
d_initcontext(void)
{
	context.size=256;
	context.top=0; /*0 means that element 0 exists!*/
	context.stack=(dict_t * *)g_malloc(sizeof(dict_t *)*256);
	context.stack[0]=d_newdict();

	/*this is where the built in functions register into the global
	  dictionary*/
	funclib_addall();
}

/*add a functuion struct to the dict (in current context)*/
func_t *
d_addfunc(func_t *func)
{
	func_t *n;

	/*we already found it (in current context)*/
	n=d_lookup(func->id,FALSE);
	if(n) {
		replacefunc(n,func);
		return n;
	}

	if((CURDICT->top+1)>=CURDICT->size) {
		/*make the dict larger*/
		CURDICT->dict=(func_t * *)my_realloc(
			CURDICT->dict,sizeof(func_t *)*CURDICT->size,
			sizeof(func_t *)*(CURDICT->size+256));
		if(CURDICT->dict==NULL) /*out of mem right here!*/
			return NULL;
		CURDICT->size+=256;
	}
	CURDICT->dict[++CURDICT->top]=func;
	return func;
}

/*set value of an existing function (in local context), used for arguments
  WARNING, does not free the memory allocated by previous value!*/
int
d_setvalue(char *id,tree_t *value)
{
	func_t *f;
	f=d_lookup(id,FALSE);
	if(!f || f->type!=USER_FUNC)
		return FALSE;
	f->data.user.value=value;
	return TRUE;
}

/*dictionary functions*/

/*lookup a function in the dictionary, either the whole thing if global
  is TRUE, or just the current context otherwise
  a terribly inefficent linear search, it was just easy to code nothing
  else, a hash would be more appropriate*/
func_t *
d_lookup(char *id,int global)
{
	int i,n;

	for(i=0;i<=CURDICT->top;i++)
		if(strcmp(CURDICT->dict[i]->id,id)==0)
			return CURDICT->dict[i];

	if(global)
		for(n=context.top;n>=0;n--)
			for(i=0;i<=DICTAT(n)->top;i++)
				if(strcmp(DICTAT(n)->dict[i]->id,id)==0)
					return DICTAT(n)->dict[i];

	return NULL;
}

int
d_delete(char *id)
{
	/*FIXME: Delete function!*/
	return FALSE;
}

/*clear all context dictionaries and pop out all the contexts except
  the global one
  also init the context stack if it hasn't been done*/
void
d_singlecontext(void)
{
	if(context.stack==NULL)
		d_initcontext();
	else
		while(context.top>0)
			freedict(d_popcontext(),TRUE);
}

/*free all memory allocated by a dictionary (except for non-dynamic (static)
  functions if dynamiconly is TRUE)*/
void
freedict(dict_t *n,int dynamiconly)
{
	int i;

	if(!n)
		return;

	for(i=0;i<=n->top;i++)
		if(!dynamiconly || n->dict[i]->dynamic)
			freefunc(n->dict[i]);
	g_free(n);
}

void
freefunc(func_t *n)
{
	if(!n)
		return;
	if(n->id)
		g_free(n->id);
	if(n->type==USER_FUNC) {
		if(n->data.user.value) 
			freetree(n->data.user.value);
		if(n->data.user.dict) 
			freedict(n->data.user.dict,FALSE);
	}
	g_free(n);
}

/*replace old with stuff from new and free new*/
void
replacefunc(func_t *old,func_t *new)
{
	if(!old || !new)
		return;
	if(old->id)
		g_free(old->id);
	if(old->type==USER_FUNC) {
		if(old->data.user.value) 
			freetree(old->data.user.value);
		if(old->data.user.dict) 
			freedict(old->data.user.dict,FALSE);
	}
	old->id=new->id;
	old->context=new->context;
	old->dynamic=new->dynamic;
	old->type=new->type;
	old->data=new->data;
	g_free(new);
}

/*copy a dictionary, but not functions, they stay the same pointers, this
  should only be done with dictionaries with no dymanic entries as those
  would be easily cleared*/
dict_t *
copydict(dict_t *n)
{
	dict_t *r;
	int i;

	if(!n)
		return NULL;

	r=(dict_t *)g_malloc(sizeof(dict_t));
	if(!r)
		return NULL;
	r->dict=(func_t * *)g_malloc(sizeof(func_t *)*n->size);
	r->size=n->size;
	r->top=n->top;
	for(i=0;i<=n->top;i++)
		r->dict[i]=n->dict[i];

	return r;
}

/*push a new dictionary onto the context stack*/
int
d_addcontext(dict_t *n)
{
	if((context.top+1)>=context.size) {
		/*make the context stack larger*/
		context.stack=(dict_t * *)my_realloc(
			context.stack,sizeof(dict_t *)*context.size,
			sizeof(dict_t *)*(context.size+256));
		if(context.stack==NULL) /*out of mem right here!*/
			return FALSE;
		context.size+=256;
	}
	if(!n)
		context.stack[++context.top]=d_newdict();
	else
		context.stack[++context.top]=n;
	return TRUE;
}

/*gimme the last dictinary*/
dict_t *
d_popcontext(void)
{
	if(context.top==-1)
		return NULL;
	else
		return context.stack[context.top--];
}
