;;;; 	Copyright (C) 1996 Christopher Lee
;;;; 
;;;; This program is free software; you can redistribute it and/or modify
;;;; it under the terms of the GNU General Public License as published by
;;;; the Free Software Foundation; either version 2, or (at your option)
;;;; any later version.
;;;; 
;;;; This program is distributed in the hope that it will be useful,
;;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;;; GNU General Public License for more details.
;;;; 
;;;; You should have received a copy of the GNU General Public License
;;;; along with this software; see the file COPYING.  If not, write to
;;;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
;;;; 

;; This is a utility for generating glue code for interfacing a C-library
;;  to Guile.
;;
;; (open-c-file fname)
;;    Open a text file with name <fname>.c, and a guile initialization
;;    function 
;;       void init_<fname> (void).
;;
;; (output-c where . lst)
;;    Write the strings in <lst> at the end of section <where>.
;;    
;; (c-file-include-local-header x)
;;    Insert an '#include "x"' line in the includes section of the file.
;;
;; (c-file-include-global-header x)
;;    Insert an '#include <x>' line in the includes section of the file.
;;
;; (define-constant sym-name varb type)
;;    Define a constant with C-value <varb> and C-type type, and export
;;    into the interpreter with scheme-name <sym-name>.
;;
;; (new-type c-name c-print-name c-die-name c-eq-name)
;;  - Defines a new scheme type corresponding to C-type <c-name>.  Actually,
;;    an SCM wrapper around a pointer to C-type <c-name> is generated.
;;    Objects of this type are printed with C-function <c-print-name>,
;;    deallocated by C-function <c-die-name>, and checked for equality with
;;    C-function <c-eq-name>.
;;  - In the current implementation, a smob wrapper is used for this.
;;  - A function called <c-name>? (e.g. "VEC?") is generated to test if
;;    any scheme object is an object of this type.
;;  - Returns a pointer to the type so that it may be used to specify types
;;    for return values and parameter types of C-functions to be exported to
;;    the interpreter.
;;
;; (new-function c-name scheme-name ret-type types description)
;;    Exports a C-function to the scheme interpreter.  The name of the
;;    function to be exported is <c-name>, the name of the function within
;;    the interpreter is <scheme-name>.  The return type of the function is
;;    <ret-type>, and the C-parameters of the function have the types listed
;;    in the <types> parameter.  A description of the function is given in
;;    <description>.  This parameter will be used when descriptions of
;;    functions can be exported to the interpreter to document built-in
;;    procedures.
;;
;; (close-c-file)
;;    Generates the C-file described with previous calls.

(define-module (g-wrap)
  :use-module (ice-9 q))

;;; output-file

;; This is a utility for writing a structured text file.
;; 
;; Open a new output file with
;;   (define outfile (make-outfile "file-name" '(list of file sections)))
;; Add text to a section of the file with
;;   (outfile:add-to-section outfile 'section1 "some text here\n")
;; Write the outputfile with
;;   (outfile:close outfile)
;;
;; Function outfile:add-to-section inserts a string or a tree of strings
;;  at the end of a list corresponding to a given section of the file.
;;  When the file is closed with outfile:close, the tree of each section
;;  is traversed (left-to-right), and each string encountered is
;;  display'ed into the file.

(define outfile-desc (make-record-type "outfile"
				       '(port sections)))
(define outfile:port-of (record-accessor outfile-desc 'port))
(define outfile:sections (record-accessor outfile-desc 'sections))

(define make-outfile 
  (let ((constr (record-constructor outfile-desc)))
    (lambda (name section-headers)
      (let ((port (open-output-file name))
	    (sections
	     (let loop ((sec section-headers))
	       (cond ((null? sec) ())
		     (else
		      (cons
		       (cons (car sec) (make-q))
		       (loop (cdr sec))))))))
	(constr port sections)))))

(define (outfile:add-to-section outfile section data)
  (let ((sect-q (assq section (outfile:sections outfile))))
    (if (not sect-q)
	(error "outfile:add-to-section -- section not found " section)
	(enq! (cdr sect-q) data)))
  #t)

(define (outfile:close outfile)
  (let ((port (outfile:port-of outfile)))
    (if (not (output-port? port))
	(error "outfile already closed")
	(let loop ((sections (outfile:sections outfile)))
	  (cond ((null? sections) ())
		(else
		 (let ((section (cdar sections)))
		   (do () ((q-empty? section))
		     (flatten-display (deq! section) port)))
		 (loop (cdr sections))))
	  (close-output-port port)))))

(define (flatten-display lst port)
  (cond ((null? lst) ())
	((pair? lst) (flatten-display (car lst) port)
		     (flatten-display (cdr lst) port))
	(else
	 (display lst port))))

;	(else
;	 (error "flatten-display: bad element found in the tree" lst))))

;;; g-wrap

(define c-file '())

(define-public (open-c-file fname)
  (set! c-file
	(make-outfile (string-append fname ".c")
		      '(file-begin
			includes
			fn-declarations
			type-headers 
			fn-wrappers
			fn-inits
			type-inits
			file-end)))
  (output-c 
   'file-begin
   "/* Generated by G-Wrap: an experimental Guile function-wrapper engine */"
   "\n\n")
  (c-file-include-global-header "libguile.h")
  (c-file-include-global-header "libguile/__scm.h")
  (output-c 'type-headers "\n")
  (output-c 'fn-inits
	    "void init_" fname " (void) {\n")
  (output-c 'fn-declarations
	    "\n\n/* Internal function declarations */\n\n")
  (output-c 'file-end "}\n"))

(define-public (c-file-include-local-header x)
  (output-c 'includes
	    "#include \"" x "\"\n"))

(define-public (c-file-include-global-header x)
  (output-c 'includes
	    "#include <" x ">\n"))

(define-public (close-c-file)
  (outfile:close c-file))

(define-public (output-c where . lst)
  (outfile:add-to-section c-file where lst))

(define-public (define-constant sym-name varb type)
  (output-c
   'type-inits
   "  scm_sysintern (\"" sym-name "\", "
   (make-conversion-to-scm type varb)
   ");\n"))

(define-public (new-type c-name c-print-name c-die-name c-eq-name)
  (let ((smob-name (string-append c-name "_smob"))
	(ptr-name  (string-append c-name "*"))
	(t16-name  (string-append "t16_" c-name)))
    (output-c
     'fn-declarations
     "scm_sizet gwrap_" c-die-name "(SCM x);\n"
     "int gwrap_" c-print-name "(SCM x, SCM port, scm_print_state* pstate);\n"
     "SCM gwrap_" c-eq-name "(SCM x, SCM y);\n")
    (output-c
     'type-headers
     "long " t16-name ";\n"
     "static scm_smobfuns " smob-name " = {\n"
     "  scm_mark0, gwrap_" c-die-name ", gwrap_" c-print-name 
     ", gwrap_" c-eq-name "\n"
     "};\n\n")
    (output-c
     'type-inits
     "  " t16-name " = scm_newsmob(&" smob-name ");\n")
    (output-c 
     'fn-wrappers
     "SCM " c-name "_to_SCM(" ptr-name " x ) {\n"
     "  SCM scm;\n"
     "  SCM_NEWCELL(scm);\n"
     "  SCM_CAR(scm) = " t16-name ";\n"
     "  SCM_CDR(scm) = (SCM)x;\n"
     "  return scm;\n"
     "}\n\n")
    (let ((the-type
	   (make-c-type 
	    ptr-name
	    ;fn-convert-to-scm 
	    (lambda (x) (list c-name "_to_SCM(" x ")"))
	    ;fn-convert-from-scm 
	    (lambda (x) (list "((" ptr-name ")SCM_CDR(" x "))"))
	    ;fn-scm-is-a
	    (lambda (x) 
	      (list "(SCM_NIMP(" x ") && (SCM_TYP16(" x ") == " t16-name
		    "))")))))
      (output-c
       'fn-wrappers
       "SCM is_a_" c-name "(SCM x) {\n"
       "  return (" (make-isa-check the-type "x") 
       " ? SCM_BOOL_T : SCM_BOOL_F);\n"
       "}\n\n")
      (output-c
       'fn-inits
       "  scm_make_gsubr( \"" c-name "?\", 1, 0, 0, "
       "is_a_" c-name " );\n")
      (output-c
       'fn-wrappers
       "int gwrap_" c-print-name 
       "(SCM x, SCM port, scm_print_state* pstate) {\n"
       "  " c-print-name "(" (make-conversion-from-scm the-type "x") 
       ", port, SCM_WRITINGP(pstate));\n"
       "  return 1;\n"
       "}\n\n")
      (output-c
       'fn-wrappers
       "scm_sizet gwrap_" c-die-name "(SCM x) {\n"
       "  " c-die-name "(" (make-conversion-from-scm the-type "x") ");\n" 
       "  return sizeof(" c-name ");\n"
       "}\n\n")
      (output-c
       'fn-wrappers
       "SCM gwrap_" c-eq-name "(SCM x, SCM y) {\n"
       "  return (" c-eq-name "(" (make-conversion-from-scm the-type "x")
       ", " (make-conversion-from-scm the-type "y") 
       ") ? SCM_BOOL_T : SCM_BOOL_F);\n"
       "}\n\n")
      the-type)))

(define-public (new-function c-name scheme-name ret-type types description)
  (let* ((params (make-params types))
	 (nargs (length params))
	 (fn-c-wrapper (string-append "gwrap_" c-name))
	 (fn-c-string  (string-append "gwrap_" c-name "_s")))
    (output-c
     'fn-wrappers
     "char * " fn-c-string " = \"" scheme-name "\";\n"
     "SCM " fn-c-wrapper " ( " (make-param-declarations params) " )\n"
     "{\n"
     (if (eq? ret-type void)
	 "  /* no return variable */\n"
	 (list "  " (c-name-of ret-type)  " ret;\n"))
     (make-c-param-protos params)
     "\n  /* Type checks */\n"
     (make-param-assertions params fn-c-string)
     "  /* Type conversions */\n"
     (make-param-assignments params)
     "  /* Call function */\n"
     "  SCM_DEFER_INTS;\n"
     (if (eq? ret-type void)
	 (list "   " c-name "( " (make-param-list params) " );\n")
	 (list "   ret = " c-name "( " (make-param-list params) " );\n"))
     "  SCM_ALLOW_INTS;\n"
     "  return " (c-to-scm ret-type "ret") ";\n"
     "}\n\n")

    (output-c
     'fn-inits
     "  scm_make_gsubr( "
     fn-c-string ", "
     (number->string nargs) ", 0, 0, "
     fn-c-wrapper ");\n")))

(define-public (new-c-code c-name scheme-name ret-type c-code types desc)
  (let* ((params (make-params types))
	 (nargs (length params))
	 (fn-c-wrapper (string-append "gwrap_" c-name))
	 (fn-c-string  (string-append "gwrap_" c-name "_s")))
    (output-c
     'fn-wrappers
     "char * " fn-c-string " = \"" scheme-name "\";\n"
     "SCM " fn-c-wrapper " ( " (make-param-declarations params) " )\n"
     "{\n"
     (if (eq? ret-type void)
	 "  /* no return variable */\n"
	 (list "  " (c-name-of ret-type)  " ret;\n"))
     (make-c-param-protos params)
     "\n  /* Type checks */\n"
     (make-param-assertions params fn-c-string)
     "  /* Type conversions */\n"
     (make-param-assignments params)
     "  /* Call function */\n"
     "  SCM_DEFER_INTS;\n"
     c-code
     "  SCM_ALLOW_INTS;\n"
     "  return " (c-to-scm ret-type "ret") ";\n"
     "}\n\n")

    (output-c
     'fn-inits
     "  scm_make_gsubr( "
     fn-c-string ", "
     (number->string nargs) ", 0, 0, "
     fn-c-wrapper ");\n")))

(define-public (new-predicate c-name scheme-name type)
  (let ((fn-c-wrapper (string-append "gwrap_" c-name))
	(fn-c-string  (string-append "gwrap_" c-name "_s")))
    (output-c
     'fn-wrappers
     "char * " fn-c-string " = \"" scheme-name "\";\n"
     "SCM " fn-c-wrapper " (SCM obj)\n"
     "{\n"
     "  return " (make-isa-check type "obj") "? SCM_BOOL_T : SCM_BOOL_F;\n"
     "}\n\n")

    (output-c
     'fn-inits
     "  scm_make_gsubr( "
     fn-c-string ", 1, 0, 0, "
     fn-c-wrapper ");\n")))

;;; Utility functions
(define (make-params types)
  (cond 
   ((and (= (length types) 1)
	 (eq? (car types) void))
    '())
   (else
    (let loop ((t types) (n 0))
      (cond ((null? t) ())
	    (else
	     (cons
	      (make-param (string-append "param" (number->string n))
			  (car t) 
			  n)
	      (loop (cdr t) (+ n 1)))))))))

(define (make-param-declarations params)
  (cond ((null? params) ())
	(else
	 (cons
	  (list
	   "SCM " (s-name-of (car params)) 
	   (if (null? (cdr params))
	       " "
	       ", "))
	  (make-param-declarations (cdr params))))))

(define (c-type-string param)
  (c-name-of (type-of param)))

(define (make-c-param-protos params)  
  (cond ((null? params) ())
	(else
	 (cons
	  (list "  " (c-name-of (type-of (car params))) 
		" " (name-of (car params)) ";\n")
	  (make-c-param-protos (cdr params))))))

(define (make-param-assertions params fn-c-string)
  (cond ((null? params) ())
	(else
	 (cons
	  (let ((param (car params)))
	    (list 
	     "  SCM_ASSERT("
	     (make-isa-check (type-of param) (s-name-of param))  ","
	     (s-name-of param) ","
	     "SCM_ARG" 
	     (if (< (number-of param) 5)
		 (number->string (+ 1 (number-of param)))
		 "n")
	     ","
	     fn-c-string ");\n"))
	  (make-param-assertions (cdr params) fn-c-string)))))

(define (make-param-list params)  
  (cond ((null? params) ())
	(else
	 (cons
	  (list 
	   (name-of (car params))
	   (if (null? (cdr params))
	       " "
	       ", "))
	  (make-param-list (cdr params))))))
	   
(define (c-to-scm ret-type var)
  (make-conversion-to-scm ret-type var))

(define (make-param-assignments params)
  (cond ((null? params) ())
	(else
	 (cons
	  (list
	   "  " (name-of (car params)) " = " 
	   (make-conversion-from-scm (type-of (car params))
				     (s-name-of (car params)))
	   ";\n")
	  (make-param-assignments (cdr params))))))

;;; Function parameters
(define param-desc (make-record-type "param"
				'(name s-name type number)))
(define name-of (record-accessor param-desc 'name))
(define s-name-of (record-accessor param-desc 's-name))
(define type-of (record-accessor param-desc 'type))
(define number-of (record-accessor param-desc 'number))
(define make-param 
  (let ((constr (record-constructor param-desc)))
    (lambda (name type number)
      (constr name (string-append "scm_" name) type number))))

;;; C types
(define c-type-desc 
  (make-record-type "c-type" 
		    '(c-name fn-convert-to-scm fn-convert-from-scm
			     fn-scm-is-a)))

(define-public make-c-type (record-constructor c-type-desc))

(define c-name-of       (record-accessor c-type-desc 'c-name))
(define fn-to-scm-of    (record-accessor c-type-desc 'fn-convert-to-scm))
(define fn-from-scm-of  (record-accessor c-type-desc 'fn-convert-from-scm))
(define fn-isa-check-of (record-accessor c-type-desc 'fn-scm-is-a))

(define-public c-type-name c-name-of)

(define make-conversion-to-scm
  (let ((fn-to-scm-of (record-accessor c-type-desc 'fn-convert-to-scm)))
    (lambda (type var)
      ((fn-to-scm-of type) var))))
(define make-conversion-from-scm
  (let ((fn-from-scm-of (record-accessor c-type-desc 'fn-convert-from-scm)))
    (lambda (type var)
      ((fn-from-scm-of type) var))))
(define make-isa-check
  (let ((fn-isa-of (record-accessor c-type-desc 'fn-scm-is-a)))
    (lambda (type var)
      ((fn-isa-of type) var))))

;;; basic types

(define-public void
  (make-c-type "void" 
	       ;fn-convert-to-scm 
	       (lambda (x) "SCM_UNSPECIFIED")
	       ;fn-convert-from-scm 
	       (lambda (x) "(void)")
	       ;fn-scm-is-a
	       (lambda (x) 1)))

(define-public int 
  (make-c-type "int" 
	       ;fn-convert-to-scm 
	       (lambda (x) (list "SCM_MAKINUM(" x ")"))
	       ;fn-convert-from-scm 
	       (lambda (x) (list "SCM_INUM(" x ")"))
	       ;fn-scm-is-a
	       (lambda (x) (list "SCM_INUMP(" x ")"))))

(define-public double
  (make-c-type "double" 
	       ;fn-convert-to-scm 
	       (lambda (x) (list "scm_makdbl(" x ",0.0)"))
	       ;fn-convert-from-scm 
	       (lambda (x) (list "scm_num2dbl(" x ", \"scm_2_double\")"))
	       ;fn-scm-is-a
	       (lambda (x) (list "(SCM_NIMP(" x ") && SCM_REALP(" x "))"))))

(define-public float
  (make-c-type "float" 
	       ;fn-convert-to-scm 
	       (lambda (x) (list "scm_makdbl((double)" x ",0.0)"))
	       ;fn-convert-from-scm 
	       (lambda (x) 
		 (list "(float)scm_num2dbl(" x ", \"scm_2_double\")"))
	       ;fn-scm-is-a
	       (lambda (x) (list "(SCM_NIMP(" x ") && SCM_REALP(" x "))"))))

(define-public tSCM
  (make-c-type "SCM" 
	       (lambda (x) x)    ;fn-convert-to-scm 
	       (lambda (x) x)    ;fn-convert-from-scm 
	       (lambda (x) 1)))  ;fn-scm-is-a
