// Copyright 2013-2015, The Gtk-rs Project Developers.
// See the COPYRIGHT file at the top-level directory of this distribution.
// Licensed under the MIT license, see the LICENSE file or <http://opensource.org/licenses/MIT>

use gdk_sys;
use glib::translate::*;
use glib::GString;
use libc::c_uint;

pub fn keyval_name(keyval: u32) -> Option<GString> {
    skip_assert_initialized!();
    unsafe { from_glib_none(gdk_sys::gdk_keyval_name(keyval as c_uint)) }
}

pub fn keyval_to_unicode(keyval: u32) -> Option<char> {
    skip_assert_initialized!();
    unsafe { ::std::char::from_u32(gdk_sys::gdk_keyval_to_unicode(keyval)).filter(|x| *x != '\0') }
}

#[derive(Clone, Debug, PartialEq, Eq, PartialOrd, Ord, Hash)]
pub struct Key(u32);

impl ::std::ops::Deref for Key {
    type Target = u32;

    fn deref(&self) -> &u32 {
        &self.0
    }
}

impl ::std::ops::DerefMut for Key {
    fn deref_mut(&mut self) -> &mut u32 {
        &mut self.0
    }
}

impl FromGlib<u32> for Key {
    fn from_glib(value: u32) -> Self {
        skip_assert_initialized!();
        Key(value)
    }
}

impl ToGlib for Key {
    type GlibType = u32;

    fn to_glib(&self) -> u32 {
        **self
    }
}

impl Key {
    pub fn to_unicode(&self) -> Option<char> {
        keyval_to_unicode(**self)
    }

    pub fn name(&self) -> Option<GString> {
        keyval_name(**self)
    }
}

impl ::std::fmt::Display for Key {
    fn fmt(&self, f: &mut ::std::fmt::Formatter<'_>) -> ::std::fmt::Result {
        write!(f, "Key({})", self.0)
    }
}

#[allow(non_upper_case_globals)]
pub mod constants {
    use super::Key;

    pub const VoidSymbol: Key = Key(gdk_sys::GDK_KEY_VoidSymbol as u32);
    pub const BackSpace: Key = Key(gdk_sys::GDK_KEY_BackSpace as u32);
    pub const Tab: Key = Key(gdk_sys::GDK_KEY_Tab as u32);
    pub const Linefeed: Key = Key(gdk_sys::GDK_KEY_Linefeed as u32);
    pub const Clear: Key = Key(gdk_sys::GDK_KEY_Clear as u32);
    pub const Return: Key = Key(gdk_sys::GDK_KEY_Return as u32);
    pub const Pause: Key = Key(gdk_sys::GDK_KEY_Pause as u32);
    pub const Scroll_Lock: Key = Key(gdk_sys::GDK_KEY_Scroll_Lock as u32);
    pub const Sys_Req: Key = Key(gdk_sys::GDK_KEY_Sys_Req as u32);
    pub const Escape: Key = Key(gdk_sys::GDK_KEY_Escape as u32);
    pub const Delete: Key = Key(gdk_sys::GDK_KEY_Delete as u32);
    pub const Multi_key: Key = Key(gdk_sys::GDK_KEY_Multi_key as u32);
    pub const Codeinput: Key = Key(gdk_sys::GDK_KEY_Codeinput as u32);
    pub const SingleCandidate: Key = Key(gdk_sys::GDK_KEY_SingleCandidate as u32);
    pub const MultipleCandidate: Key = Key(gdk_sys::GDK_KEY_MultipleCandidate as u32);
    pub const PreviousCandidate: Key = Key(gdk_sys::GDK_KEY_PreviousCandidate as u32);
    pub const Kanji: Key = Key(gdk_sys::GDK_KEY_Kanji as u32);
    pub const Muhenkan: Key = Key(gdk_sys::GDK_KEY_Muhenkan as u32);
    pub const Henkan_Mode: Key = Key(gdk_sys::GDK_KEY_Henkan_Mode as u32);
    pub const Henkan: Key = Key(gdk_sys::GDK_KEY_Henkan as u32);
    pub const Romaji: Key = Key(gdk_sys::GDK_KEY_Romaji as u32);
    pub const Hiragana: Key = Key(gdk_sys::GDK_KEY_Hiragana as u32);
    pub const Katakana: Key = Key(gdk_sys::GDK_KEY_Katakana as u32);
    pub const Hiragana_Katakana: Key = Key(gdk_sys::GDK_KEY_Hiragana_Katakana as u32);
    pub const Zenkaku: Key = Key(gdk_sys::GDK_KEY_Zenkaku as u32);
    pub const Hankaku: Key = Key(gdk_sys::GDK_KEY_Hankaku as u32);
    pub const Zenkaku_Hankaku: Key = Key(gdk_sys::GDK_KEY_Zenkaku_Hankaku as u32);
    pub const Touroku: Key = Key(gdk_sys::GDK_KEY_Touroku as u32);
    pub const Massyo: Key = Key(gdk_sys::GDK_KEY_Massyo as u32);
    pub const Kana_Lock: Key = Key(gdk_sys::GDK_KEY_Kana_Lock as u32);
    pub const Kana_Shift: Key = Key(gdk_sys::GDK_KEY_Kana_Shift as u32);
    pub const Eisu_Shift: Key = Key(gdk_sys::GDK_KEY_Eisu_Shift as u32);
    pub const Eisu_toggle: Key = Key(gdk_sys::GDK_KEY_Eisu_toggle as u32);
    pub const Kanji_Bangou: Key = Key(gdk_sys::GDK_KEY_Kanji_Bangou as u32);
    pub const Zen_Koho: Key = Key(gdk_sys::GDK_KEY_Zen_Koho as u32);
    pub const Mae_Koho: Key = Key(gdk_sys::GDK_KEY_Mae_Koho as u32);
    pub const Home: Key = Key(gdk_sys::GDK_KEY_Home as u32);
    pub const Left: Key = Key(gdk_sys::GDK_KEY_Left as u32);
    pub const Up: Key = Key(gdk_sys::GDK_KEY_Up as u32);
    pub const Right: Key = Key(gdk_sys::GDK_KEY_Right as u32);
    pub const Down: Key = Key(gdk_sys::GDK_KEY_Down as u32);
    pub const Prior: Key = Key(gdk_sys::GDK_KEY_Prior as u32);
    pub const Page_Up: Key = Key(gdk_sys::GDK_KEY_Page_Up as u32);
    pub const Next: Key = Key(gdk_sys::GDK_KEY_Next as u32);
    pub const Page_Down: Key = Key(gdk_sys::GDK_KEY_Page_Down as u32);
    pub const End: Key = Key(gdk_sys::GDK_KEY_End as u32);
    pub const Begin: Key = Key(gdk_sys::GDK_KEY_Begin as u32);
    pub const Select: Key = Key(gdk_sys::GDK_KEY_Select as u32);
    pub const Print: Key = Key(gdk_sys::GDK_KEY_Print as u32);
    pub const Execute: Key = Key(gdk_sys::GDK_KEY_Execute as u32);
    pub const Insert: Key = Key(gdk_sys::GDK_KEY_Insert as u32);
    pub const Undo: Key = Key(gdk_sys::GDK_KEY_Undo as u32);
    pub const Redo: Key = Key(gdk_sys::GDK_KEY_Redo as u32);
    pub const Menu: Key = Key(gdk_sys::GDK_KEY_Menu as u32);
    pub const Find: Key = Key(gdk_sys::GDK_KEY_Find as u32);
    pub const Cancel: Key = Key(gdk_sys::GDK_KEY_Cancel as u32);
    pub const Help: Key = Key(gdk_sys::GDK_KEY_Help as u32);
    pub const Break: Key = Key(gdk_sys::GDK_KEY_Break as u32);
    pub const Mode_switch: Key = Key(gdk_sys::GDK_KEY_Mode_switch as u32);
    pub const script_switch: Key = Key(gdk_sys::GDK_KEY_script_switch as u32);
    pub const Num_Lock: Key = Key(gdk_sys::GDK_KEY_Num_Lock as u32);
    pub const KP_Space: Key = Key(gdk_sys::GDK_KEY_KP_Space as u32);
    pub const KP_Tab: Key = Key(gdk_sys::GDK_KEY_KP_Tab as u32);
    pub const KP_Enter: Key = Key(gdk_sys::GDK_KEY_KP_Enter as u32);
    pub const KP_F1: Key = Key(gdk_sys::GDK_KEY_KP_F1 as u32);
    pub const KP_F2: Key = Key(gdk_sys::GDK_KEY_KP_F2 as u32);
    pub const KP_F3: Key = Key(gdk_sys::GDK_KEY_KP_F3 as u32);
    pub const KP_F4: Key = Key(gdk_sys::GDK_KEY_KP_F4 as u32);
    pub const KP_Home: Key = Key(gdk_sys::GDK_KEY_KP_Home as u32);
    pub const KP_Left: Key = Key(gdk_sys::GDK_KEY_KP_Left as u32);
    pub const KP_Up: Key = Key(gdk_sys::GDK_KEY_KP_Up as u32);
    pub const KP_Right: Key = Key(gdk_sys::GDK_KEY_KP_Right as u32);
    pub const KP_Down: Key = Key(gdk_sys::GDK_KEY_KP_Down as u32);
    pub const KP_Prior: Key = Key(gdk_sys::GDK_KEY_KP_Prior as u32);
    pub const KP_Page_Up: Key = Key(gdk_sys::GDK_KEY_KP_Page_Up as u32);
    pub const KP_Next: Key = Key(gdk_sys::GDK_KEY_KP_Next as u32);
    pub const KP_Page_Down: Key = Key(gdk_sys::GDK_KEY_KP_Page_Down as u32);
    pub const KP_End: Key = Key(gdk_sys::GDK_KEY_KP_End as u32);
    pub const KP_Begin: Key = Key(gdk_sys::GDK_KEY_KP_Begin as u32);
    pub const KP_Insert: Key = Key(gdk_sys::GDK_KEY_KP_Insert as u32);
    pub const KP_Delete: Key = Key(gdk_sys::GDK_KEY_KP_Delete as u32);
    pub const KP_Equal: Key = Key(gdk_sys::GDK_KEY_KP_Equal as u32);
    pub const KP_Multiply: Key = Key(gdk_sys::GDK_KEY_KP_Multiply as u32);
    pub const KP_Add: Key = Key(gdk_sys::GDK_KEY_KP_Add as u32);
    pub const KP_Separator: Key = Key(gdk_sys::GDK_KEY_KP_Separator as u32);
    pub const KP_Subtract: Key = Key(gdk_sys::GDK_KEY_KP_Subtract as u32);
    pub const KP_Decimal: Key = Key(gdk_sys::GDK_KEY_KP_Decimal as u32);
    pub const KP_Divide: Key = Key(gdk_sys::GDK_KEY_KP_Divide as u32);
    pub const KP_0: Key = Key(gdk_sys::GDK_KEY_KP_0 as u32);
    pub const KP_1: Key = Key(gdk_sys::GDK_KEY_KP_1 as u32);
    pub const KP_2: Key = Key(gdk_sys::GDK_KEY_KP_2 as u32);
    pub const KP_3: Key = Key(gdk_sys::GDK_KEY_KP_3 as u32);
    pub const KP_4: Key = Key(gdk_sys::GDK_KEY_KP_4 as u32);
    pub const KP_5: Key = Key(gdk_sys::GDK_KEY_KP_5 as u32);
    pub const KP_6: Key = Key(gdk_sys::GDK_KEY_KP_6 as u32);
    pub const KP_7: Key = Key(gdk_sys::GDK_KEY_KP_7 as u32);
    pub const KP_8: Key = Key(gdk_sys::GDK_KEY_KP_8 as u32);
    pub const KP_9: Key = Key(gdk_sys::GDK_KEY_KP_9 as u32);
    pub const F1: Key = Key(gdk_sys::GDK_KEY_F1 as u32);
    pub const F2: Key = Key(gdk_sys::GDK_KEY_F2 as u32);
    pub const F3: Key = Key(gdk_sys::GDK_KEY_F3 as u32);
    pub const F4: Key = Key(gdk_sys::GDK_KEY_F4 as u32);
    pub const F5: Key = Key(gdk_sys::GDK_KEY_F5 as u32);
    pub const F6: Key = Key(gdk_sys::GDK_KEY_F6 as u32);
    pub const F7: Key = Key(gdk_sys::GDK_KEY_F7 as u32);
    pub const F8: Key = Key(gdk_sys::GDK_KEY_F8 as u32);
    pub const F9: Key = Key(gdk_sys::GDK_KEY_F9 as u32);
    pub const F10: Key = Key(gdk_sys::GDK_KEY_F10 as u32);
    pub const F11: Key = Key(gdk_sys::GDK_KEY_F11 as u32);
    pub const L1: Key = Key(gdk_sys::GDK_KEY_L1 as u32);
    pub const F12: Key = Key(gdk_sys::GDK_KEY_F12 as u32);
    pub const L2: Key = Key(gdk_sys::GDK_KEY_L2 as u32);
    pub const F13: Key = Key(gdk_sys::GDK_KEY_F13 as u32);
    pub const L3: Key = Key(gdk_sys::GDK_KEY_L3 as u32);
    pub const F14: Key = Key(gdk_sys::GDK_KEY_F14 as u32);
    pub const L4: Key = Key(gdk_sys::GDK_KEY_L4 as u32);
    pub const F15: Key = Key(gdk_sys::GDK_KEY_F15 as u32);
    pub const L5: Key = Key(gdk_sys::GDK_KEY_L5 as u32);
    pub const F16: Key = Key(gdk_sys::GDK_KEY_F16 as u32);
    pub const L6: Key = Key(gdk_sys::GDK_KEY_L6 as u32);
    pub const F17: Key = Key(gdk_sys::GDK_KEY_F17 as u32);
    pub const L7: Key = Key(gdk_sys::GDK_KEY_L7 as u32);
    pub const F18: Key = Key(gdk_sys::GDK_KEY_F18 as u32);
    pub const L8: Key = Key(gdk_sys::GDK_KEY_L8 as u32);
    pub const F19: Key = Key(gdk_sys::GDK_KEY_F19 as u32);
    pub const L9: Key = Key(gdk_sys::GDK_KEY_L9 as u32);
    pub const F20: Key = Key(gdk_sys::GDK_KEY_F20 as u32);
    pub const L10: Key = Key(gdk_sys::GDK_KEY_L10 as u32);
    pub const F21: Key = Key(gdk_sys::GDK_KEY_F21 as u32);
    pub const R1: Key = Key(gdk_sys::GDK_KEY_R1 as u32);
    pub const F22: Key = Key(gdk_sys::GDK_KEY_F22 as u32);
    pub const R2: Key = Key(gdk_sys::GDK_KEY_R2 as u32);
    pub const F23: Key = Key(gdk_sys::GDK_KEY_F23 as u32);
    pub const R3: Key = Key(gdk_sys::GDK_KEY_R3 as u32);
    pub const F24: Key = Key(gdk_sys::GDK_KEY_F24 as u32);
    pub const R4: Key = Key(gdk_sys::GDK_KEY_R4 as u32);
    pub const F25: Key = Key(gdk_sys::GDK_KEY_F25 as u32);
    pub const R5: Key = Key(gdk_sys::GDK_KEY_R5 as u32);
    pub const F26: Key = Key(gdk_sys::GDK_KEY_F26 as u32);
    pub const R6: Key = Key(gdk_sys::GDK_KEY_R6 as u32);
    pub const F27: Key = Key(gdk_sys::GDK_KEY_F27 as u32);
    pub const R7: Key = Key(gdk_sys::GDK_KEY_R7 as u32);
    pub const F28: Key = Key(gdk_sys::GDK_KEY_F28 as u32);
    pub const R8: Key = Key(gdk_sys::GDK_KEY_R8 as u32);
    pub const F29: Key = Key(gdk_sys::GDK_KEY_F29 as u32);
    pub const R9: Key = Key(gdk_sys::GDK_KEY_R9 as u32);
    pub const F30: Key = Key(gdk_sys::GDK_KEY_F30 as u32);
    pub const R10: Key = Key(gdk_sys::GDK_KEY_R10 as u32);
    pub const F31: Key = Key(gdk_sys::GDK_KEY_F31 as u32);
    pub const R11: Key = Key(gdk_sys::GDK_KEY_R11 as u32);
    pub const F32: Key = Key(gdk_sys::GDK_KEY_F32 as u32);
    pub const R12: Key = Key(gdk_sys::GDK_KEY_R12 as u32);
    pub const F33: Key = Key(gdk_sys::GDK_KEY_F33 as u32);
    pub const R13: Key = Key(gdk_sys::GDK_KEY_R13 as u32);
    pub const F34: Key = Key(gdk_sys::GDK_KEY_F34 as u32);
    pub const R14: Key = Key(gdk_sys::GDK_KEY_R14 as u32);
    pub const F35: Key = Key(gdk_sys::GDK_KEY_F35 as u32);
    pub const R15: Key = Key(gdk_sys::GDK_KEY_R15 as u32);
    pub const Shift_L: Key = Key(gdk_sys::GDK_KEY_Shift_L as u32);
    pub const Shift_R: Key = Key(gdk_sys::GDK_KEY_Shift_R as u32);
    pub const Control_L: Key = Key(gdk_sys::GDK_KEY_Control_L as u32);
    pub const Control_R: Key = Key(gdk_sys::GDK_KEY_Control_R as u32);
    pub const Caps_Lock: Key = Key(gdk_sys::GDK_KEY_Caps_Lock as u32);
    pub const Shift_Lock: Key = Key(gdk_sys::GDK_KEY_Shift_Lock as u32);
    pub const Meta_L: Key = Key(gdk_sys::GDK_KEY_Meta_L as u32);
    pub const Meta_R: Key = Key(gdk_sys::GDK_KEY_Meta_R as u32);
    pub const Alt_L: Key = Key(gdk_sys::GDK_KEY_Alt_L as u32);
    pub const Alt_R: Key = Key(gdk_sys::GDK_KEY_Alt_R as u32);
    pub const Super_L: Key = Key(gdk_sys::GDK_KEY_Super_L as u32);
    pub const Super_R: Key = Key(gdk_sys::GDK_KEY_Super_R as u32);
    pub const Hyper_L: Key = Key(gdk_sys::GDK_KEY_Hyper_L as u32);
    pub const Hyper_R: Key = Key(gdk_sys::GDK_KEY_Hyper_R as u32);
    pub const ISO_Lock: Key = Key(gdk_sys::GDK_KEY_ISO_Lock as u32);
    pub const ISO_Level2_Latch: Key = Key(gdk_sys::GDK_KEY_ISO_Level2_Latch as u32);
    pub const ISO_Level3_Shift: Key = Key(gdk_sys::GDK_KEY_ISO_Level3_Shift as u32);
    pub const ISO_Level3_Latch: Key = Key(gdk_sys::GDK_KEY_ISO_Level3_Latch as u32);
    pub const ISO_Level3_Lock: Key = Key(gdk_sys::GDK_KEY_ISO_Level3_Lock as u32);
    pub const ISO_Level5_Shift: Key = Key(gdk_sys::GDK_KEY_ISO_Level5_Shift as u32);
    pub const ISO_Level5_Latch: Key = Key(gdk_sys::GDK_KEY_ISO_Level5_Latch as u32);
    pub const ISO_Level5_Lock: Key = Key(gdk_sys::GDK_KEY_ISO_Level5_Lock as u32);
    pub const ISO_Group_Shift: Key = Key(gdk_sys::GDK_KEY_ISO_Group_Shift as u32);
    pub const ISO_Group_Latch: Key = Key(gdk_sys::GDK_KEY_ISO_Group_Latch as u32);
    pub const ISO_Group_Lock: Key = Key(gdk_sys::GDK_KEY_ISO_Group_Lock as u32);
    pub const ISO_Next_Group: Key = Key(gdk_sys::GDK_KEY_ISO_Next_Group as u32);
    pub const ISO_Next_Group_Lock: Key = Key(gdk_sys::GDK_KEY_ISO_Next_Group_Lock as u32);
    pub const ISO_Prev_Group: Key = Key(gdk_sys::GDK_KEY_ISO_Prev_Group as u32);
    pub const ISO_Prev_Group_Lock: Key = Key(gdk_sys::GDK_KEY_ISO_Prev_Group_Lock as u32);
    pub const ISO_First_Group: Key = Key(gdk_sys::GDK_KEY_ISO_First_Group as u32);
    pub const ISO_First_Group_Lock: Key = Key(gdk_sys::GDK_KEY_ISO_First_Group_Lock as u32);
    pub const ISO_Last_Group: Key = Key(gdk_sys::GDK_KEY_ISO_Last_Group as u32);
    pub const ISO_Last_Group_Lock: Key = Key(gdk_sys::GDK_KEY_ISO_Last_Group_Lock as u32);
    pub const ISO_Left_Tab: Key = Key(gdk_sys::GDK_KEY_ISO_Left_Tab as u32);
    pub const ISO_Move_Line_Up: Key = Key(gdk_sys::GDK_KEY_ISO_Move_Line_Up as u32);
    pub const ISO_Move_Line_Down: Key = Key(gdk_sys::GDK_KEY_ISO_Move_Line_Down as u32);
    pub const ISO_Partial_Line_Up: Key = Key(gdk_sys::GDK_KEY_ISO_Partial_Line_Up as u32);
    pub const ISO_Partial_Line_Down: Key = Key(gdk_sys::GDK_KEY_ISO_Partial_Line_Down as u32);
    pub const ISO_Partial_Space_Left: Key = Key(gdk_sys::GDK_KEY_ISO_Partial_Space_Left as u32);
    pub const ISO_Partial_Space_Right: Key = Key(gdk_sys::GDK_KEY_ISO_Partial_Space_Right as u32);
    pub const ISO_Set_Margin_Left: Key = Key(gdk_sys::GDK_KEY_ISO_Set_Margin_Left as u32);
    pub const ISO_Set_Margin_Right: Key = Key(gdk_sys::GDK_KEY_ISO_Set_Margin_Right as u32);
    pub const ISO_Release_Margin_Left: Key = Key(gdk_sys::GDK_KEY_ISO_Release_Margin_Left as u32);
    pub const ISO_Release_Margin_Right: Key = Key(gdk_sys::GDK_KEY_ISO_Release_Margin_Right as u32);
    pub const ISO_Release_Both_Margins: Key = Key(gdk_sys::GDK_KEY_ISO_Release_Both_Margins as u32);
    pub const ISO_Fast_Cursor_Left: Key = Key(gdk_sys::GDK_KEY_ISO_Fast_Cursor_Left as u32);
    pub const ISO_Fast_Cursor_Right: Key = Key(gdk_sys::GDK_KEY_ISO_Fast_Cursor_Right as u32);
    pub const ISO_Fast_Cursor_Up: Key = Key(gdk_sys::GDK_KEY_ISO_Fast_Cursor_Up as u32);
    pub const ISO_Fast_Cursor_Down: Key = Key(gdk_sys::GDK_KEY_ISO_Fast_Cursor_Down as u32);
    pub const ISO_Continuous_Underline: Key = Key(gdk_sys::GDK_KEY_ISO_Continuous_Underline as u32);
    pub const ISO_Discontinuous_Underline: Key =
        Key(gdk_sys::GDK_KEY_ISO_Discontinuous_Underline as u32);
    pub const ISO_Emphasize: Key = Key(gdk_sys::GDK_KEY_ISO_Emphasize as u32);
    pub const ISO_Center_Object: Key = Key(gdk_sys::GDK_KEY_ISO_Center_Object as u32);
    pub const ISO_Enter: Key = Key(gdk_sys::GDK_KEY_ISO_Enter as u32);
    pub const dead_grave: Key = Key(gdk_sys::GDK_KEY_dead_grave as u32);
    pub const dead_acute: Key = Key(gdk_sys::GDK_KEY_dead_acute as u32);
    pub const dead_circumflex: Key = Key(gdk_sys::GDK_KEY_dead_circumflex as u32);
    pub const dead_tilde: Key = Key(gdk_sys::GDK_KEY_dead_tilde as u32);
    pub const dead_perispomeni: Key = Key(gdk_sys::GDK_KEY_dead_perispomeni as u32);
    pub const dead_macron: Key = Key(gdk_sys::GDK_KEY_dead_macron as u32);
    pub const dead_breve: Key = Key(gdk_sys::GDK_KEY_dead_breve as u32);
    pub const dead_abovedot: Key = Key(gdk_sys::GDK_KEY_dead_abovedot as u32);
    pub const dead_diaeresis: Key = Key(gdk_sys::GDK_KEY_dead_diaeresis as u32);
    pub const dead_abovering: Key = Key(gdk_sys::GDK_KEY_dead_abovering as u32);
    pub const dead_doubleacute: Key = Key(gdk_sys::GDK_KEY_dead_doubleacute as u32);
    pub const dead_caron: Key = Key(gdk_sys::GDK_KEY_dead_caron as u32);
    pub const dead_cedilla: Key = Key(gdk_sys::GDK_KEY_dead_cedilla as u32);
    pub const dead_ogonek: Key = Key(gdk_sys::GDK_KEY_dead_ogonek as u32);
    pub const dead_iota: Key = Key(gdk_sys::GDK_KEY_dead_iota as u32);
    pub const dead_voiced_sound: Key = Key(gdk_sys::GDK_KEY_dead_voiced_sound as u32);
    pub const dead_semivoiced_sound: Key = Key(gdk_sys::GDK_KEY_dead_semivoiced_sound as u32);
    pub const dead_belowdot: Key = Key(gdk_sys::GDK_KEY_dead_belowdot as u32);
    pub const dead_hook: Key = Key(gdk_sys::GDK_KEY_dead_hook as u32);
    pub const dead_horn: Key = Key(gdk_sys::GDK_KEY_dead_horn as u32);
    pub const dead_stroke: Key = Key(gdk_sys::GDK_KEY_dead_stroke as u32);
    pub const dead_abovecomma: Key = Key(gdk_sys::GDK_KEY_dead_abovecomma as u32);
    pub const dead_psili: Key = Key(gdk_sys::GDK_KEY_dead_psili as u32);
    pub const dead_abovereversedcomma: Key = Key(gdk_sys::GDK_KEY_dead_abovereversedcomma as u32);
    pub const dead_dasia: Key = Key(gdk_sys::GDK_KEY_dead_dasia as u32);
    pub const dead_doublegrave: Key = Key(gdk_sys::GDK_KEY_dead_doublegrave as u32);
    pub const dead_belowring: Key = Key(gdk_sys::GDK_KEY_dead_belowring as u32);
    pub const dead_belowmacron: Key = Key(gdk_sys::GDK_KEY_dead_belowmacron as u32);
    pub const dead_belowcircumflex: Key = Key(gdk_sys::GDK_KEY_dead_belowcircumflex as u32);
    pub const dead_belowtilde: Key = Key(gdk_sys::GDK_KEY_dead_belowtilde as u32);
    pub const dead_belowbreve: Key = Key(gdk_sys::GDK_KEY_dead_belowbreve as u32);
    pub const dead_belowdiaeresis: Key = Key(gdk_sys::GDK_KEY_dead_belowdiaeresis as u32);
    pub const dead_invertedbreve: Key = Key(gdk_sys::GDK_KEY_dead_invertedbreve as u32);
    pub const dead_belowcomma: Key = Key(gdk_sys::GDK_KEY_dead_belowcomma as u32);
    pub const dead_currency: Key = Key(gdk_sys::GDK_KEY_dead_currency as u32);
    pub const dead_a: Key = Key(gdk_sys::GDK_KEY_dead_a as u32);
    pub const dead_A: Key = Key(gdk_sys::GDK_KEY_dead_A as u32);
    pub const dead_e: Key = Key(gdk_sys::GDK_KEY_dead_e as u32);
    pub const dead_E: Key = Key(gdk_sys::GDK_KEY_dead_E as u32);
    pub const dead_i: Key = Key(gdk_sys::GDK_KEY_dead_i as u32);
    pub const dead_I: Key = Key(gdk_sys::GDK_KEY_dead_I as u32);
    pub const dead_o: Key = Key(gdk_sys::GDK_KEY_dead_o as u32);
    pub const dead_O: Key = Key(gdk_sys::GDK_KEY_dead_O as u32);
    pub const dead_u: Key = Key(gdk_sys::GDK_KEY_dead_u as u32);
    pub const dead_U: Key = Key(gdk_sys::GDK_KEY_dead_U as u32);
    pub const dead_small_schwa: Key = Key(gdk_sys::GDK_KEY_dead_small_schwa as u32);
    pub const dead_capital_schwa: Key = Key(gdk_sys::GDK_KEY_dead_capital_schwa as u32);
    pub const dead_greek: Key = Key(gdk_sys::GDK_KEY_dead_greek as u32);
    pub const First_Virtual_Screen: Key = Key(gdk_sys::GDK_KEY_First_Virtual_Screen as u32);
    pub const Prev_Virtual_Screen: Key = Key(gdk_sys::GDK_KEY_Prev_Virtual_Screen as u32);
    pub const Next_Virtual_Screen: Key = Key(gdk_sys::GDK_KEY_Next_Virtual_Screen as u32);
    pub const Last_Virtual_Screen: Key = Key(gdk_sys::GDK_KEY_Last_Virtual_Screen as u32);
    pub const Terminate_Server: Key = Key(gdk_sys::GDK_KEY_Terminate_Server as u32);
    pub const AccessX_Enable: Key = Key(gdk_sys::GDK_KEY_AccessX_Enable as u32);
    pub const AccessX_Feedback_Enable: Key = Key(gdk_sys::GDK_KEY_AccessX_Feedback_Enable as u32);
    pub const RepeatKeys_Enable: Key = Key(gdk_sys::GDK_KEY_RepeatKeys_Enable as u32);
    pub const SlowKeys_Enable: Key = Key(gdk_sys::GDK_KEY_SlowKeys_Enable as u32);
    pub const BounceKeys_Enable: Key = Key(gdk_sys::GDK_KEY_BounceKeys_Enable as u32);
    pub const StickyKeys_Enable: Key = Key(gdk_sys::GDK_KEY_StickyKeys_Enable as u32);
    pub const MouseKeys_Enable: Key = Key(gdk_sys::GDK_KEY_MouseKeys_Enable as u32);
    pub const MouseKeys_Accel_Enable: Key = Key(gdk_sys::GDK_KEY_MouseKeys_Accel_Enable as u32);
    pub const Overlay1_Enable: Key = Key(gdk_sys::GDK_KEY_Overlay1_Enable as u32);
    pub const Overlay2_Enable: Key = Key(gdk_sys::GDK_KEY_Overlay2_Enable as u32);
    pub const AudibleBell_Enable: Key = Key(gdk_sys::GDK_KEY_AudibleBell_Enable as u32);
    pub const Pointer_Left: Key = Key(gdk_sys::GDK_KEY_Pointer_Left as u32);
    pub const Pointer_Right: Key = Key(gdk_sys::GDK_KEY_Pointer_Right as u32);
    pub const Pointer_Up: Key = Key(gdk_sys::GDK_KEY_Pointer_Up as u32);
    pub const Pointer_Down: Key = Key(gdk_sys::GDK_KEY_Pointer_Down as u32);
    pub const Pointer_UpLeft: Key = Key(gdk_sys::GDK_KEY_Pointer_UpLeft as u32);
    pub const Pointer_UpRight: Key = Key(gdk_sys::GDK_KEY_Pointer_UpRight as u32);
    pub const Pointer_DownLeft: Key = Key(gdk_sys::GDK_KEY_Pointer_DownLeft as u32);
    pub const Pointer_DownRight: Key = Key(gdk_sys::GDK_KEY_Pointer_DownRight as u32);
    pub const Pointer_Button_Dflt: Key = Key(gdk_sys::GDK_KEY_Pointer_Button_Dflt as u32);
    pub const Pointer_Button1: Key = Key(gdk_sys::GDK_KEY_Pointer_Button1 as u32);
    pub const Pointer_Button2: Key = Key(gdk_sys::GDK_KEY_Pointer_Button2 as u32);
    pub const Pointer_Button3: Key = Key(gdk_sys::GDK_KEY_Pointer_Button3 as u32);
    pub const Pointer_Button4: Key = Key(gdk_sys::GDK_KEY_Pointer_Button4 as u32);
    pub const Pointer_Button5: Key = Key(gdk_sys::GDK_KEY_Pointer_Button5 as u32);
    pub const Pointer_DblClick_Dflt: Key = Key(gdk_sys::GDK_KEY_Pointer_DblClick_Dflt as u32);
    pub const Pointer_DblClick1: Key = Key(gdk_sys::GDK_KEY_Pointer_DblClick1 as u32);
    pub const Pointer_DblClick2: Key = Key(gdk_sys::GDK_KEY_Pointer_DblClick2 as u32);
    pub const Pointer_DblClick3: Key = Key(gdk_sys::GDK_KEY_Pointer_DblClick3 as u32);
    pub const Pointer_DblClick4: Key = Key(gdk_sys::GDK_KEY_Pointer_DblClick4 as u32);
    pub const Pointer_DblClick5: Key = Key(gdk_sys::GDK_KEY_Pointer_DblClick5 as u32);
    pub const Pointer_Drag_Dflt: Key = Key(gdk_sys::GDK_KEY_Pointer_Drag_Dflt as u32);
    pub const Pointer_Drag1: Key = Key(gdk_sys::GDK_KEY_Pointer_Drag1 as u32);
    pub const Pointer_Drag2: Key = Key(gdk_sys::GDK_KEY_Pointer_Drag2 as u32);
    pub const Pointer_Drag3: Key = Key(gdk_sys::GDK_KEY_Pointer_Drag3 as u32);
    pub const Pointer_Drag4: Key = Key(gdk_sys::GDK_KEY_Pointer_Drag4 as u32);
    pub const Pointer_Drag5: Key = Key(gdk_sys::GDK_KEY_Pointer_Drag5 as u32);
    pub const Pointer_EnableKeys: Key = Key(gdk_sys::GDK_KEY_Pointer_EnableKeys as u32);
    pub const Pointer_Accelerate: Key = Key(gdk_sys::GDK_KEY_Pointer_Accelerate as u32);
    pub const Pointer_DfltBtnNext: Key = Key(gdk_sys::GDK_KEY_Pointer_DfltBtnNext as u32);
    pub const Pointer_DfltBtnPrev: Key = Key(gdk_sys::GDK_KEY_Pointer_DfltBtnPrev as u32);
    pub const ch: Key = Key(gdk_sys::GDK_KEY_ch as u32);
    pub const Ch: Key = Key(gdk_sys::GDK_KEY_Ch as u32);
    pub const CH: Key = Key(gdk_sys::GDK_KEY_CH as u32);
    pub const c_h: Key = Key(gdk_sys::GDK_KEY_c_h as u32);
    pub const C_h: Key = Key(gdk_sys::GDK_KEY_C_h as u32);
    pub const C_H: Key = Key(gdk_sys::GDK_KEY_C_H as u32);
    pub const _3270_Duplicate: Key = Key(gdk_sys::GDK_KEY_3270_Duplicate as u32);
    pub const _3270_FieldMark: Key = Key(gdk_sys::GDK_KEY_3270_FieldMark as u32);
    pub const _3270_Right2: Key = Key(gdk_sys::GDK_KEY_3270_Right2 as u32);
    pub const _3270_Left2: Key = Key(gdk_sys::GDK_KEY_3270_Left2 as u32);
    pub const _3270_BackTab: Key = Key(gdk_sys::GDK_KEY_3270_BackTab as u32);
    pub const _3270_EraseEOF: Key = Key(gdk_sys::GDK_KEY_3270_EraseEOF as u32);
    pub const _3270_EraseInput: Key = Key(gdk_sys::GDK_KEY_3270_EraseInput as u32);
    pub const _3270_Reset: Key = Key(gdk_sys::GDK_KEY_3270_Reset as u32);
    pub const _3270_Quit: Key = Key(gdk_sys::GDK_KEY_3270_Quit as u32);
    pub const _3270_PA1: Key = Key(gdk_sys::GDK_KEY_3270_PA1 as u32);
    pub const _3270_PA2: Key = Key(gdk_sys::GDK_KEY_3270_PA2 as u32);
    pub const _3270_PA3: Key = Key(gdk_sys::GDK_KEY_3270_PA3 as u32);
    pub const _3270_Test: Key = Key(gdk_sys::GDK_KEY_3270_Test as u32);
    pub const _3270_Attn: Key = Key(gdk_sys::GDK_KEY_3270_Attn as u32);
    pub const _3270_CursorBlink: Key = Key(gdk_sys::GDK_KEY_3270_CursorBlink as u32);
    pub const _3270_AltCursor: Key = Key(gdk_sys::GDK_KEY_3270_AltCursor as u32);
    pub const _3270_KeyClick: Key = Key(gdk_sys::GDK_KEY_3270_KeyClick as u32);
    pub const _3270_Jump: Key = Key(gdk_sys::GDK_KEY_3270_Jump as u32);
    pub const _3270_Ident: Key = Key(gdk_sys::GDK_KEY_3270_Ident as u32);
    pub const _3270_Rule: Key = Key(gdk_sys::GDK_KEY_3270_Rule as u32);
    pub const _3270_Copy: Key = Key(gdk_sys::GDK_KEY_3270_Copy as u32);
    pub const _3270_Play: Key = Key(gdk_sys::GDK_KEY_3270_Play as u32);
    pub const _3270_Setup: Key = Key(gdk_sys::GDK_KEY_3270_Setup as u32);
    pub const _3270_Record: Key = Key(gdk_sys::GDK_KEY_3270_Record as u32);
    pub const _3270_ChangeScreen: Key = Key(gdk_sys::GDK_KEY_3270_ChangeScreen as u32);
    pub const _3270_DeleteWord: Key = Key(gdk_sys::GDK_KEY_3270_DeleteWord as u32);
    pub const _3270_ExSelect: Key = Key(gdk_sys::GDK_KEY_3270_ExSelect as u32);
    pub const _3270_CursorSelect: Key = Key(gdk_sys::GDK_KEY_3270_CursorSelect as u32);
    pub const _3270_PrintScreen: Key = Key(gdk_sys::GDK_KEY_3270_PrintScreen as u32);
    pub const _3270_Enter: Key = Key(gdk_sys::GDK_KEY_3270_Enter as u32);
    pub const space: Key = Key(gdk_sys::GDK_KEY_space as u32);
    pub const exclam: Key = Key(gdk_sys::GDK_KEY_exclam as u32);
    pub const quotedbl: Key = Key(gdk_sys::GDK_KEY_quotedbl as u32);
    pub const numbersign: Key = Key(gdk_sys::GDK_KEY_numbersign as u32);
    pub const dollar: Key = Key(gdk_sys::GDK_KEY_dollar as u32);
    pub const percent: Key = Key(gdk_sys::GDK_KEY_percent as u32);
    pub const ampersand: Key = Key(gdk_sys::GDK_KEY_ampersand as u32);
    pub const apostrophe: Key = Key(gdk_sys::GDK_KEY_apostrophe as u32);
    pub const quoteright: Key = Key(gdk_sys::GDK_KEY_quoteright as u32);
    pub const parenleft: Key = Key(gdk_sys::GDK_KEY_parenleft as u32);
    pub const parenright: Key = Key(gdk_sys::GDK_KEY_parenright as u32);
    pub const asterisk: Key = Key(gdk_sys::GDK_KEY_asterisk as u32);
    pub const plus: Key = Key(gdk_sys::GDK_KEY_plus as u32);
    pub const comma: Key = Key(gdk_sys::GDK_KEY_comma as u32);
    pub const minus: Key = Key(gdk_sys::GDK_KEY_minus as u32);
    pub const period: Key = Key(gdk_sys::GDK_KEY_period as u32);
    pub const slash: Key = Key(gdk_sys::GDK_KEY_slash as u32);
    pub const _0: Key = Key(gdk_sys::GDK_KEY_0 as u32); // equivalent to keypad '0'
    pub const _1: Key = Key(gdk_sys::GDK_KEY_1 as u32); // equivalent to keypad '1'
    pub const _2: Key = Key(gdk_sys::GDK_KEY_2 as u32); // equivalent to keypad '2'
    pub const _3: Key = Key(gdk_sys::GDK_KEY_3 as u32); // equivalent to keypad '3'
    pub const _4: Key = Key(gdk_sys::GDK_KEY_4 as u32); // equivalent to keypad '4'
    pub const _5: Key = Key(gdk_sys::GDK_KEY_5 as u32); // equivalent to keypad '5'
    pub const _6: Key = Key(gdk_sys::GDK_KEY_6 as u32); // equivalent to keypad '6'
    pub const _7: Key = Key(gdk_sys::GDK_KEY_7 as u32); // equivalent to keypad '7'
    pub const _8: Key = Key(gdk_sys::GDK_KEY_8 as u32); // equivalent to keypad '8'
    pub const _9: Key = Key(gdk_sys::GDK_KEY_9 as u32); // equivalent to keypad '9'
    pub const colon: Key = Key(gdk_sys::GDK_KEY_colon as u32);
    pub const semicolon: Key = Key(gdk_sys::GDK_KEY_semicolon as u32);
    pub const less: Key = Key(gdk_sys::GDK_KEY_less as u32);
    pub const equal: Key = Key(gdk_sys::GDK_KEY_equal as u32);
    pub const greater: Key = Key(gdk_sys::GDK_KEY_greater as u32);
    pub const question: Key = Key(gdk_sys::GDK_KEY_question as u32);
    pub const at: Key = Key(gdk_sys::GDK_KEY_at as u32);
    pub const A: Key = Key(gdk_sys::GDK_KEY_A as u32);
    pub const B: Key = Key(gdk_sys::GDK_KEY_B as u32);
    pub const C: Key = Key(gdk_sys::GDK_KEY_C as u32);
    pub const D: Key = Key(gdk_sys::GDK_KEY_D as u32);
    pub const E: Key = Key(gdk_sys::GDK_KEY_E as u32);
    pub const F: Key = Key(gdk_sys::GDK_KEY_F as u32);
    pub const G: Key = Key(gdk_sys::GDK_KEY_G as u32);
    pub const H: Key = Key(gdk_sys::GDK_KEY_H as u32);
    pub const I: Key = Key(gdk_sys::GDK_KEY_I as u32);
    pub const J: Key = Key(gdk_sys::GDK_KEY_J as u32);
    pub const K: Key = Key(gdk_sys::GDK_KEY_K as u32);
    pub const L: Key = Key(gdk_sys::GDK_KEY_L as u32);
    pub const M: Key = Key(gdk_sys::GDK_KEY_M as u32);
    pub const N: Key = Key(gdk_sys::GDK_KEY_N as u32);
    pub const O: Key = Key(gdk_sys::GDK_KEY_O as u32);
    pub const P: Key = Key(gdk_sys::GDK_KEY_P as u32);
    pub const Q: Key = Key(gdk_sys::GDK_KEY_Q as u32);
    pub const R: Key = Key(gdk_sys::GDK_KEY_R as u32);
    pub const S: Key = Key(gdk_sys::GDK_KEY_S as u32);
    pub const T: Key = Key(gdk_sys::GDK_KEY_T as u32);
    pub const U: Key = Key(gdk_sys::GDK_KEY_U as u32);
    pub const V: Key = Key(gdk_sys::GDK_KEY_V as u32);
    pub const W: Key = Key(gdk_sys::GDK_KEY_W as u32);
    pub const X: Key = Key(gdk_sys::GDK_KEY_X as u32);
    pub const Y: Key = Key(gdk_sys::GDK_KEY_Y as u32);
    pub const Z: Key = Key(gdk_sys::GDK_KEY_Z as u32);
    pub const bracketleft: Key = Key(gdk_sys::GDK_KEY_bracketleft as u32);
    pub const backslash: Key = Key(gdk_sys::GDK_KEY_backslash as u32);
    pub const bracketright: Key = Key(gdk_sys::GDK_KEY_bracketright as u32);
    pub const asciicircum: Key = Key(gdk_sys::GDK_KEY_asciicircum as u32);
    pub const underscore: Key = Key(gdk_sys::GDK_KEY_underscore as u32);
    pub const grave: Key = Key(gdk_sys::GDK_KEY_grave as u32);
    pub const quoteleft: Key = Key(gdk_sys::GDK_KEY_quoteleft as u32);
    pub const a: Key = Key(gdk_sys::GDK_KEY_a as u32);
    pub const b: Key = Key(gdk_sys::GDK_KEY_b as u32);
    pub const c: Key = Key(gdk_sys::GDK_KEY_c as u32);
    pub const d: Key = Key(gdk_sys::GDK_KEY_d as u32);
    pub const e: Key = Key(gdk_sys::GDK_KEY_e as u32);
    pub const f: Key = Key(gdk_sys::GDK_KEY_f as u32);
    pub const g: Key = Key(gdk_sys::GDK_KEY_g as u32);
    pub const h: Key = Key(gdk_sys::GDK_KEY_h as u32);
    pub const i: Key = Key(gdk_sys::GDK_KEY_i as u32);
    pub const j: Key = Key(gdk_sys::GDK_KEY_j as u32);
    pub const k: Key = Key(gdk_sys::GDK_KEY_k as u32);
    pub const l: Key = Key(gdk_sys::GDK_KEY_l as u32);
    pub const m: Key = Key(gdk_sys::GDK_KEY_m as u32);
    pub const n: Key = Key(gdk_sys::GDK_KEY_n as u32);
    pub const o: Key = Key(gdk_sys::GDK_KEY_o as u32);
    pub const p: Key = Key(gdk_sys::GDK_KEY_p as u32);
    pub const q: Key = Key(gdk_sys::GDK_KEY_q as u32);
    pub const r: Key = Key(gdk_sys::GDK_KEY_r as u32);
    pub const s: Key = Key(gdk_sys::GDK_KEY_s as u32);
    pub const t: Key = Key(gdk_sys::GDK_KEY_t as u32);
    pub const u: Key = Key(gdk_sys::GDK_KEY_u as u32);
    pub const v: Key = Key(gdk_sys::GDK_KEY_v as u32);
    pub const w: Key = Key(gdk_sys::GDK_KEY_w as u32);
    pub const x: Key = Key(gdk_sys::GDK_KEY_x as u32);
    pub const y: Key = Key(gdk_sys::GDK_KEY_y as u32);
    pub const z: Key = Key(gdk_sys::GDK_KEY_z as u32);
    pub const braceleft: Key = Key(gdk_sys::GDK_KEY_braceleft as u32);
    pub const bar: Key = Key(gdk_sys::GDK_KEY_bar as u32);
    pub const braceright: Key = Key(gdk_sys::GDK_KEY_braceright as u32);
    pub const asciitilde: Key = Key(gdk_sys::GDK_KEY_asciitilde as u32);
    pub const nobreakspace: Key = Key(gdk_sys::GDK_KEY_nobreakspace as u32);
    pub const exclamdown: Key = Key(gdk_sys::GDK_KEY_exclamdown as u32);
    pub const cent: Key = Key(gdk_sys::GDK_KEY_cent as u32);
    pub const sterling: Key = Key(gdk_sys::GDK_KEY_sterling as u32);
    pub const currency: Key = Key(gdk_sys::GDK_KEY_currency as u32);
    pub const yen: Key = Key(gdk_sys::GDK_KEY_yen as u32);
    pub const brokenbar: Key = Key(gdk_sys::GDK_KEY_brokenbar as u32);
    pub const section: Key = Key(gdk_sys::GDK_KEY_section as u32);
    pub const diaeresis: Key = Key(gdk_sys::GDK_KEY_diaeresis as u32);
    pub const copyright: Key = Key(gdk_sys::GDK_KEY_copyright as u32);
    pub const ordfeminine: Key = Key(gdk_sys::GDK_KEY_ordfeminine as u32);
    pub const guillemotleft: Key = Key(gdk_sys::GDK_KEY_guillemotleft as u32);
    pub const notsign: Key = Key(gdk_sys::GDK_KEY_notsign as u32);
    pub const hyphen: Key = Key(gdk_sys::GDK_KEY_hyphen as u32);
    pub const registered: Key = Key(gdk_sys::GDK_KEY_registered as u32);
    pub const macron: Key = Key(gdk_sys::GDK_KEY_macron as u32);
    pub const degree: Key = Key(gdk_sys::GDK_KEY_degree as u32);
    pub const plusminus: Key = Key(gdk_sys::GDK_KEY_plusminus as u32);
    pub const twosuperior: Key = Key(gdk_sys::GDK_KEY_twosuperior as u32);
    pub const threesuperior: Key = Key(gdk_sys::GDK_KEY_threesuperior as u32);
    pub const acute: Key = Key(gdk_sys::GDK_KEY_acute as u32);
    pub const mu: Key = Key(gdk_sys::GDK_KEY_mu as u32);
    pub const paragraph: Key = Key(gdk_sys::GDK_KEY_paragraph as u32);
    pub const periodcentered: Key = Key(gdk_sys::GDK_KEY_periodcentered as u32);
    pub const cedilla: Key = Key(gdk_sys::GDK_KEY_cedilla as u32);
    pub const onesuperior: Key = Key(gdk_sys::GDK_KEY_onesuperior as u32);
    pub const masculine: Key = Key(gdk_sys::GDK_KEY_masculine as u32);
    pub const guillemotright: Key = Key(gdk_sys::GDK_KEY_guillemotright as u32);
    pub const onequarter: Key = Key(gdk_sys::GDK_KEY_onequarter as u32);
    pub const onehalf: Key = Key(gdk_sys::GDK_KEY_onehalf as u32);
    pub const threequarters: Key = Key(gdk_sys::GDK_KEY_threequarters as u32);
    pub const questiondown: Key = Key(gdk_sys::GDK_KEY_questiondown as u32);
    pub const Agrave: Key = Key(gdk_sys::GDK_KEY_Agrave as u32);
    pub const Aacute: Key = Key(gdk_sys::GDK_KEY_Aacute as u32);
    pub const Acircumflex: Key = Key(gdk_sys::GDK_KEY_Acircumflex as u32);
    pub const Atilde: Key = Key(gdk_sys::GDK_KEY_Atilde as u32);
    pub const Adiaeresis: Key = Key(gdk_sys::GDK_KEY_Adiaeresis as u32);
    pub const Aring: Key = Key(gdk_sys::GDK_KEY_Aring as u32);
    pub const AE: Key = Key(gdk_sys::GDK_KEY_AE as u32);
    pub const Ccedilla: Key = Key(gdk_sys::GDK_KEY_Ccedilla as u32);
    pub const Egrave: Key = Key(gdk_sys::GDK_KEY_Egrave as u32);
    pub const Eacute: Key = Key(gdk_sys::GDK_KEY_Eacute as u32);
    pub const Ecircumflex: Key = Key(gdk_sys::GDK_KEY_Ecircumflex as u32);
    pub const Ediaeresis: Key = Key(gdk_sys::GDK_KEY_Ediaeresis as u32);
    pub const Igrave: Key = Key(gdk_sys::GDK_KEY_Igrave as u32);
    pub const Iacute: Key = Key(gdk_sys::GDK_KEY_Iacute as u32);
    pub const Icircumflex: Key = Key(gdk_sys::GDK_KEY_Icircumflex as u32);
    pub const Idiaeresis: Key = Key(gdk_sys::GDK_KEY_Idiaeresis as u32);
    pub const ETH: Key = Key(gdk_sys::GDK_KEY_ETH as u32);
    pub const Eth: Key = Key(gdk_sys::GDK_KEY_Eth as u32);
    pub const Ntilde: Key = Key(gdk_sys::GDK_KEY_Ntilde as u32);
    pub const Ograve: Key = Key(gdk_sys::GDK_KEY_Ograve as u32);
    pub const Oacute: Key = Key(gdk_sys::GDK_KEY_Oacute as u32);
    pub const Ocircumflex: Key = Key(gdk_sys::GDK_KEY_Ocircumflex as u32);
    pub const Otilde: Key = Key(gdk_sys::GDK_KEY_Otilde as u32);
    pub const Odiaeresis: Key = Key(gdk_sys::GDK_KEY_Odiaeresis as u32);
    pub const multiply: Key = Key(gdk_sys::GDK_KEY_multiply as u32);
    pub const Oslash: Key = Key(gdk_sys::GDK_KEY_Oslash as u32);
    pub const Ooblique: Key = Key(gdk_sys::GDK_KEY_Ooblique as u32);
    pub const Ugrave: Key = Key(gdk_sys::GDK_KEY_Ugrave as u32);
    pub const Uacute: Key = Key(gdk_sys::GDK_KEY_Uacute as u32);
    pub const Ucircumflex: Key = Key(gdk_sys::GDK_KEY_Ucircumflex as u32);
    pub const Udiaeresis: Key = Key(gdk_sys::GDK_KEY_Udiaeresis as u32);
    pub const Yacute: Key = Key(gdk_sys::GDK_KEY_Yacute as u32);
    pub const THORN: Key = Key(gdk_sys::GDK_KEY_THORN as u32);
    pub const Thorn: Key = Key(gdk_sys::GDK_KEY_Thorn as u32);
    pub const ssharp: Key = Key(gdk_sys::GDK_KEY_ssharp as u32);
    pub const agrave: Key = Key(gdk_sys::GDK_KEY_agrave as u32);
    pub const aacute: Key = Key(gdk_sys::GDK_KEY_aacute as u32);
    pub const acircumflex: Key = Key(gdk_sys::GDK_KEY_acircumflex as u32);
    pub const atilde: Key = Key(gdk_sys::GDK_KEY_atilde as u32);
    pub const adiaeresis: Key = Key(gdk_sys::GDK_KEY_adiaeresis as u32);
    pub const aring: Key = Key(gdk_sys::GDK_KEY_aring as u32);
    pub const ae: Key = Key(gdk_sys::GDK_KEY_ae as u32);
    pub const ccedilla: Key = Key(gdk_sys::GDK_KEY_ccedilla as u32);
    pub const egrave: Key = Key(gdk_sys::GDK_KEY_egrave as u32);
    pub const eacute: Key = Key(gdk_sys::GDK_KEY_eacute as u32);
    pub const ecircumflex: Key = Key(gdk_sys::GDK_KEY_ecircumflex as u32);
    pub const ediaeresis: Key = Key(gdk_sys::GDK_KEY_ediaeresis as u32);
    pub const igrave: Key = Key(gdk_sys::GDK_KEY_igrave as u32);
    pub const iacute: Key = Key(gdk_sys::GDK_KEY_iacute as u32);
    pub const icircumflex: Key = Key(gdk_sys::GDK_KEY_icircumflex as u32);
    pub const idiaeresis: Key = Key(gdk_sys::GDK_KEY_idiaeresis as u32);
    pub const eth: Key = Key(gdk_sys::GDK_KEY_eth as u32);
    pub const ntilde: Key = Key(gdk_sys::GDK_KEY_ntilde as u32);
    pub const ograve: Key = Key(gdk_sys::GDK_KEY_ograve as u32);
    pub const oacute: Key = Key(gdk_sys::GDK_KEY_oacute as u32);
    pub const ocircumflex: Key = Key(gdk_sys::GDK_KEY_ocircumflex as u32);
    pub const otilde: Key = Key(gdk_sys::GDK_KEY_otilde as u32);
    pub const odiaeresis: Key = Key(gdk_sys::GDK_KEY_odiaeresis as u32);
    pub const division: Key = Key(gdk_sys::GDK_KEY_division as u32);
    pub const oslash: Key = Key(gdk_sys::GDK_KEY_oslash as u32);
    pub const ooblique: Key = Key(gdk_sys::GDK_KEY_ooblique as u32);
    pub const ugrave: Key = Key(gdk_sys::GDK_KEY_ugrave as u32);
    pub const uacute: Key = Key(gdk_sys::GDK_KEY_uacute as u32);
    pub const ucircumflex: Key = Key(gdk_sys::GDK_KEY_ucircumflex as u32);
    pub const udiaeresis: Key = Key(gdk_sys::GDK_KEY_udiaeresis as u32);
    pub const yacute: Key = Key(gdk_sys::GDK_KEY_yacute as u32);
    pub const thorn: Key = Key(gdk_sys::GDK_KEY_thorn as u32);
    pub const ydiaeresis: Key = Key(gdk_sys::GDK_KEY_ydiaeresis as u32);
    pub const Aogonek: Key = Key(gdk_sys::GDK_KEY_Aogonek as u32);
    pub const breve: Key = Key(gdk_sys::GDK_KEY_breve as u32);
    pub const Lstroke: Key = Key(gdk_sys::GDK_KEY_Lstroke as u32);
    pub const Lcaron: Key = Key(gdk_sys::GDK_KEY_Lcaron as u32);
    pub const Sacute: Key = Key(gdk_sys::GDK_KEY_Sacute as u32);
    pub const Scaron: Key = Key(gdk_sys::GDK_KEY_Scaron as u32);
    pub const Scedilla: Key = Key(gdk_sys::GDK_KEY_Scedilla as u32);
    pub const Tcaron: Key = Key(gdk_sys::GDK_KEY_Tcaron as u32);
    pub const Zacute: Key = Key(gdk_sys::GDK_KEY_Zacute as u32);
    pub const Zcaron: Key = Key(gdk_sys::GDK_KEY_Zcaron as u32);
    pub const Zabovedot: Key = Key(gdk_sys::GDK_KEY_Zabovedot as u32);
    pub const aogonek: Key = Key(gdk_sys::GDK_KEY_aogonek as u32);
    pub const ogonek: Key = Key(gdk_sys::GDK_KEY_ogonek as u32);
    pub const lstroke: Key = Key(gdk_sys::GDK_KEY_lstroke as u32);
    pub const lcaron: Key = Key(gdk_sys::GDK_KEY_lcaron as u32);
    pub const sacute: Key = Key(gdk_sys::GDK_KEY_sacute as u32);
    pub const caron: Key = Key(gdk_sys::GDK_KEY_caron as u32);
    pub const scaron: Key = Key(gdk_sys::GDK_KEY_scaron as u32);
    pub const scedilla: Key = Key(gdk_sys::GDK_KEY_scedilla as u32);
    pub const tcaron: Key = Key(gdk_sys::GDK_KEY_tcaron as u32);
    pub const zacute: Key = Key(gdk_sys::GDK_KEY_zacute as u32);
    pub const doubleacute: Key = Key(gdk_sys::GDK_KEY_doubleacute as u32);
    pub const zcaron: Key = Key(gdk_sys::GDK_KEY_zcaron as u32);
    pub const zabovedot: Key = Key(gdk_sys::GDK_KEY_zabovedot as u32);
    pub const Racute: Key = Key(gdk_sys::GDK_KEY_Racute as u32);
    pub const Abreve: Key = Key(gdk_sys::GDK_KEY_Abreve as u32);
    pub const Lacute: Key = Key(gdk_sys::GDK_KEY_Lacute as u32);
    pub const Cacute: Key = Key(gdk_sys::GDK_KEY_Cacute as u32);
    pub const Ccaron: Key = Key(gdk_sys::GDK_KEY_Ccaron as u32);
    pub const Eogonek: Key = Key(gdk_sys::GDK_KEY_Eogonek as u32);
    pub const Ecaron: Key = Key(gdk_sys::GDK_KEY_Ecaron as u32);
    pub const Dcaron: Key = Key(gdk_sys::GDK_KEY_Dcaron as u32);
    pub const Dstroke: Key = Key(gdk_sys::GDK_KEY_Dstroke as u32);
    pub const Nacute: Key = Key(gdk_sys::GDK_KEY_Nacute as u32);
    pub const Ncaron: Key = Key(gdk_sys::GDK_KEY_Ncaron as u32);
    pub const Odoubleacute: Key = Key(gdk_sys::GDK_KEY_Odoubleacute as u32);
    pub const Rcaron: Key = Key(gdk_sys::GDK_KEY_Rcaron as u32);
    pub const Uring: Key = Key(gdk_sys::GDK_KEY_Uring as u32);
    pub const Udoubleacute: Key = Key(gdk_sys::GDK_KEY_Udoubleacute as u32);
    pub const Tcedilla: Key = Key(gdk_sys::GDK_KEY_Tcedilla as u32);
    pub const racute: Key = Key(gdk_sys::GDK_KEY_racute as u32);
    pub const abreve: Key = Key(gdk_sys::GDK_KEY_abreve as u32);
    pub const lacute: Key = Key(gdk_sys::GDK_KEY_lacute as u32);
    pub const cacute: Key = Key(gdk_sys::GDK_KEY_cacute as u32);
    pub const ccaron: Key = Key(gdk_sys::GDK_KEY_ccaron as u32);
    pub const eogonek: Key = Key(gdk_sys::GDK_KEY_eogonek as u32);
    pub const ecaron: Key = Key(gdk_sys::GDK_KEY_ecaron as u32);
    pub const dcaron: Key = Key(gdk_sys::GDK_KEY_dcaron as u32);
    pub const dstroke: Key = Key(gdk_sys::GDK_KEY_dstroke as u32);
    pub const nacute: Key = Key(gdk_sys::GDK_KEY_nacute as u32);
    pub const ncaron: Key = Key(gdk_sys::GDK_KEY_ncaron as u32);
    pub const odoubleacute: Key = Key(gdk_sys::GDK_KEY_odoubleacute as u32);
    pub const rcaron: Key = Key(gdk_sys::GDK_KEY_rcaron as u32);
    pub const uring: Key = Key(gdk_sys::GDK_KEY_uring as u32);
    pub const udoubleacute: Key = Key(gdk_sys::GDK_KEY_udoubleacute as u32);
    pub const tcedilla: Key = Key(gdk_sys::GDK_KEY_tcedilla as u32);
    pub const abovedot: Key = Key(gdk_sys::GDK_KEY_abovedot as u32);
    pub const Hstroke: Key = Key(gdk_sys::GDK_KEY_Hstroke as u32);
    pub const Hcircumflex: Key = Key(gdk_sys::GDK_KEY_Hcircumflex as u32);
    pub const Iabovedot: Key = Key(gdk_sys::GDK_KEY_Iabovedot as u32);
    pub const Gbreve: Key = Key(gdk_sys::GDK_KEY_Gbreve as u32);
    pub const Jcircumflex: Key = Key(gdk_sys::GDK_KEY_Jcircumflex as u32);
    pub const hstroke: Key = Key(gdk_sys::GDK_KEY_hstroke as u32);
    pub const hcircumflex: Key = Key(gdk_sys::GDK_KEY_hcircumflex as u32);
    pub const idotless: Key = Key(gdk_sys::GDK_KEY_idotless as u32);
    pub const gbreve: Key = Key(gdk_sys::GDK_KEY_gbreve as u32);
    pub const jcircumflex: Key = Key(gdk_sys::GDK_KEY_jcircumflex as u32);
    pub const Cabovedot: Key = Key(gdk_sys::GDK_KEY_Cabovedot as u32);
    pub const Ccircumflex: Key = Key(gdk_sys::GDK_KEY_Ccircumflex as u32);
    pub const Gabovedot: Key = Key(gdk_sys::GDK_KEY_Gabovedot as u32);
    pub const Gcircumflex: Key = Key(gdk_sys::GDK_KEY_Gcircumflex as u32);
    pub const Ubreve: Key = Key(gdk_sys::GDK_KEY_Ubreve as u32);
    pub const Scircumflex: Key = Key(gdk_sys::GDK_KEY_Scircumflex as u32);
    pub const cabovedot: Key = Key(gdk_sys::GDK_KEY_cabovedot as u32);
    pub const ccircumflex: Key = Key(gdk_sys::GDK_KEY_ccircumflex as u32);
    pub const gabovedot: Key = Key(gdk_sys::GDK_KEY_gabovedot as u32);
    pub const gcircumflex: Key = Key(gdk_sys::GDK_KEY_gcircumflex as u32);
    pub const ubreve: Key = Key(gdk_sys::GDK_KEY_ubreve as u32);
    pub const scircumflex: Key = Key(gdk_sys::GDK_KEY_scircumflex as u32);
    pub const kra: Key = Key(gdk_sys::GDK_KEY_kra as u32);
    pub const kappa: Key = Key(gdk_sys::GDK_KEY_kappa as u32);
    pub const Rcedilla: Key = Key(gdk_sys::GDK_KEY_Rcedilla as u32);
    pub const Itilde: Key = Key(gdk_sys::GDK_KEY_Itilde as u32);
    pub const Lcedilla: Key = Key(gdk_sys::GDK_KEY_Lcedilla as u32);
    pub const Emacron: Key = Key(gdk_sys::GDK_KEY_Emacron as u32);
    pub const Gcedilla: Key = Key(gdk_sys::GDK_KEY_Gcedilla as u32);
    pub const Tslash: Key = Key(gdk_sys::GDK_KEY_Tslash as u32);
    pub const rcedilla: Key = Key(gdk_sys::GDK_KEY_rcedilla as u32);
    pub const itilde: Key = Key(gdk_sys::GDK_KEY_itilde as u32);
    pub const lcedilla: Key = Key(gdk_sys::GDK_KEY_lcedilla as u32);
    pub const emacron: Key = Key(gdk_sys::GDK_KEY_emacron as u32);
    pub const gcedilla: Key = Key(gdk_sys::GDK_KEY_gcedilla as u32);
    pub const tslash: Key = Key(gdk_sys::GDK_KEY_tslash as u32);
    pub const ENG: Key = Key(gdk_sys::GDK_KEY_ENG as u32);
    pub const eng: Key = Key(gdk_sys::GDK_KEY_eng as u32);
    pub const Amacron: Key = Key(gdk_sys::GDK_KEY_Amacron as u32);
    pub const Iogonek: Key = Key(gdk_sys::GDK_KEY_Iogonek as u32);
    pub const Eabovedot: Key = Key(gdk_sys::GDK_KEY_Eabovedot as u32);
    pub const Imacron: Key = Key(gdk_sys::GDK_KEY_Imacron as u32);
    pub const Ncedilla: Key = Key(gdk_sys::GDK_KEY_Ncedilla as u32);
    pub const Omacron: Key = Key(gdk_sys::GDK_KEY_Omacron as u32);
    pub const Kcedilla: Key = Key(gdk_sys::GDK_KEY_Kcedilla as u32);
    pub const Uogonek: Key = Key(gdk_sys::GDK_KEY_Uogonek as u32);
    pub const Utilde: Key = Key(gdk_sys::GDK_KEY_Utilde as u32);
    pub const Umacron: Key = Key(gdk_sys::GDK_KEY_Umacron as u32);
    pub const amacron: Key = Key(gdk_sys::GDK_KEY_amacron as u32);
    pub const iogonek: Key = Key(gdk_sys::GDK_KEY_iogonek as u32);
    pub const eabovedot: Key = Key(gdk_sys::GDK_KEY_eabovedot as u32);
    pub const imacron: Key = Key(gdk_sys::GDK_KEY_imacron as u32);
    pub const ncedilla: Key = Key(gdk_sys::GDK_KEY_ncedilla as u32);
    pub const omacron: Key = Key(gdk_sys::GDK_KEY_omacron as u32);
    pub const kcedilla: Key = Key(gdk_sys::GDK_KEY_kcedilla as u32);
    pub const uogonek: Key = Key(gdk_sys::GDK_KEY_uogonek as u32);
    pub const utilde: Key = Key(gdk_sys::GDK_KEY_utilde as u32);
    pub const umacron: Key = Key(gdk_sys::GDK_KEY_umacron as u32);
    pub const Wcircumflex: Key = Key(gdk_sys::GDK_KEY_Wcircumflex as u32);
    pub const wcircumflex: Key = Key(gdk_sys::GDK_KEY_wcircumflex as u32);
    pub const Ycircumflex: Key = Key(gdk_sys::GDK_KEY_Ycircumflex as u32);
    pub const ycircumflex: Key = Key(gdk_sys::GDK_KEY_ycircumflex as u32);
    pub const Babovedot: Key = Key(gdk_sys::GDK_KEY_Babovedot as u32);
    pub const babovedot: Key = Key(gdk_sys::GDK_KEY_babovedot as u32);
    pub const Dabovedot: Key = Key(gdk_sys::GDK_KEY_Dabovedot as u32);
    pub const dabovedot: Key = Key(gdk_sys::GDK_KEY_dabovedot as u32);
    pub const Fabovedot: Key = Key(gdk_sys::GDK_KEY_Fabovedot as u32);
    pub const fabovedot: Key = Key(gdk_sys::GDK_KEY_fabovedot as u32);
    pub const Mabovedot: Key = Key(gdk_sys::GDK_KEY_Mabovedot as u32);
    pub const mabovedot: Key = Key(gdk_sys::GDK_KEY_mabovedot as u32);
    pub const Pabovedot: Key = Key(gdk_sys::GDK_KEY_Pabovedot as u32);
    pub const pabovedot: Key = Key(gdk_sys::GDK_KEY_pabovedot as u32);
    pub const Sabovedot: Key = Key(gdk_sys::GDK_KEY_Sabovedot as u32);
    pub const sabovedot: Key = Key(gdk_sys::GDK_KEY_sabovedot as u32);
    pub const Tabovedot: Key = Key(gdk_sys::GDK_KEY_Tabovedot as u32);
    pub const tabovedot: Key = Key(gdk_sys::GDK_KEY_tabovedot as u32);
    pub const Wgrave: Key = Key(gdk_sys::GDK_KEY_Wgrave as u32);
    pub const wgrave: Key = Key(gdk_sys::GDK_KEY_wgrave as u32);
    pub const Wacute: Key = Key(gdk_sys::GDK_KEY_Wacute as u32);
    pub const wacute: Key = Key(gdk_sys::GDK_KEY_wacute as u32);
    pub const Wdiaeresis: Key = Key(gdk_sys::GDK_KEY_Wdiaeresis as u32);
    pub const wdiaeresis: Key = Key(gdk_sys::GDK_KEY_wdiaeresis as u32);
    pub const Ygrave: Key = Key(gdk_sys::GDK_KEY_Ygrave as u32);
    pub const ygrave: Key = Key(gdk_sys::GDK_KEY_ygrave as u32);
    pub const OE: Key = Key(gdk_sys::GDK_KEY_OE as u32);
    pub const oe: Key = Key(gdk_sys::GDK_KEY_oe as u32);
    pub const Ydiaeresis: Key = Key(gdk_sys::GDK_KEY_Ydiaeresis as u32);
    pub const overline: Key = Key(gdk_sys::GDK_KEY_overline as u32);
    pub const kana_fullstop: Key = Key(gdk_sys::GDK_KEY_kana_fullstop as u32);
    pub const kana_openingbracket: Key = Key(gdk_sys::GDK_KEY_kana_openingbracket as u32);
    pub const kana_closingbracket: Key = Key(gdk_sys::GDK_KEY_kana_closingbracket as u32);
    pub const kana_comma: Key = Key(gdk_sys::GDK_KEY_kana_comma as u32);
    pub const kana_conjunctive: Key = Key(gdk_sys::GDK_KEY_kana_conjunctive as u32);
    pub const kana_middledot: Key = Key(gdk_sys::GDK_KEY_kana_middledot as u32);
    pub const kana_WO: Key = Key(gdk_sys::GDK_KEY_kana_WO as u32);
    pub const kana_a: Key = Key(gdk_sys::GDK_KEY_kana_a as u32);
    pub const kana_i: Key = Key(gdk_sys::GDK_KEY_kana_i as u32);
    pub const kana_u: Key = Key(gdk_sys::GDK_KEY_kana_u as u32);
    pub const kana_e: Key = Key(gdk_sys::GDK_KEY_kana_e as u32);
    pub const kana_o: Key = Key(gdk_sys::GDK_KEY_kana_o as u32);
    pub const kana_ya: Key = Key(gdk_sys::GDK_KEY_kana_ya as u32);
    pub const kana_yu: Key = Key(gdk_sys::GDK_KEY_kana_yu as u32);
    pub const kana_yo: Key = Key(gdk_sys::GDK_KEY_kana_yo as u32);
    pub const kana_tsu: Key = Key(gdk_sys::GDK_KEY_kana_tsu as u32);
    pub const kana_tu: Key = Key(gdk_sys::GDK_KEY_kana_tu as u32);
    pub const prolongedsound: Key = Key(gdk_sys::GDK_KEY_prolongedsound as u32);
    pub const kana_A: Key = Key(gdk_sys::GDK_KEY_kana_A as u32);
    pub const kana_I: Key = Key(gdk_sys::GDK_KEY_kana_I as u32);
    pub const kana_U: Key = Key(gdk_sys::GDK_KEY_kana_U as u32);
    pub const kana_E: Key = Key(gdk_sys::GDK_KEY_kana_E as u32);
    pub const kana_O: Key = Key(gdk_sys::GDK_KEY_kana_O as u32);
    pub const kana_KA: Key = Key(gdk_sys::GDK_KEY_kana_KA as u32);
    pub const kana_KI: Key = Key(gdk_sys::GDK_KEY_kana_KI as u32);
    pub const kana_KU: Key = Key(gdk_sys::GDK_KEY_kana_KU as u32);
    pub const kana_KE: Key = Key(gdk_sys::GDK_KEY_kana_KE as u32);
    pub const kana_KO: Key = Key(gdk_sys::GDK_KEY_kana_KO as u32);
    pub const kana_SA: Key = Key(gdk_sys::GDK_KEY_kana_SA as u32);
    pub const kana_SHI: Key = Key(gdk_sys::GDK_KEY_kana_SHI as u32);
    pub const kana_SU: Key = Key(gdk_sys::GDK_KEY_kana_SU as u32);
    pub const kana_SE: Key = Key(gdk_sys::GDK_KEY_kana_SE as u32);
    pub const kana_SO: Key = Key(gdk_sys::GDK_KEY_kana_SO as u32);
    pub const kana_TA: Key = Key(gdk_sys::GDK_KEY_kana_TA as u32);
    pub const kana_CHI: Key = Key(gdk_sys::GDK_KEY_kana_CHI as u32);
    pub const kana_TI: Key = Key(gdk_sys::GDK_KEY_kana_TI as u32);
    pub const kana_TSU: Key = Key(gdk_sys::GDK_KEY_kana_TSU as u32);
    pub const kana_TU: Key = Key(gdk_sys::GDK_KEY_kana_TU as u32);
    pub const kana_TE: Key = Key(gdk_sys::GDK_KEY_kana_TE as u32);
    pub const kana_TO: Key = Key(gdk_sys::GDK_KEY_kana_TO as u32);
    pub const kana_NA: Key = Key(gdk_sys::GDK_KEY_kana_NA as u32);
    pub const kana_NI: Key = Key(gdk_sys::GDK_KEY_kana_NI as u32);
    pub const kana_NU: Key = Key(gdk_sys::GDK_KEY_kana_NU as u32);
    pub const kana_NE: Key = Key(gdk_sys::GDK_KEY_kana_NE as u32);
    pub const kana_NO: Key = Key(gdk_sys::GDK_KEY_kana_NO as u32);
    pub const kana_HA: Key = Key(gdk_sys::GDK_KEY_kana_HA as u32);
    pub const kana_HI: Key = Key(gdk_sys::GDK_KEY_kana_HI as u32);
    pub const kana_FU: Key = Key(gdk_sys::GDK_KEY_kana_FU as u32);
    pub const kana_HU: Key = Key(gdk_sys::GDK_KEY_kana_HU as u32);
    pub const kana_HE: Key = Key(gdk_sys::GDK_KEY_kana_HE as u32);
    pub const kana_HO: Key = Key(gdk_sys::GDK_KEY_kana_HO as u32);
    pub const kana_MA: Key = Key(gdk_sys::GDK_KEY_kana_MA as u32);
    pub const kana_MI: Key = Key(gdk_sys::GDK_KEY_kana_MI as u32);
    pub const kana_MU: Key = Key(gdk_sys::GDK_KEY_kana_MU as u32);
    pub const kana_ME: Key = Key(gdk_sys::GDK_KEY_kana_ME as u32);
    pub const kana_MO: Key = Key(gdk_sys::GDK_KEY_kana_MO as u32);
    pub const kana_YA: Key = Key(gdk_sys::GDK_KEY_kana_YA as u32);
    pub const kana_YU: Key = Key(gdk_sys::GDK_KEY_kana_YU as u32);
    pub const kana_YO: Key = Key(gdk_sys::GDK_KEY_kana_YO as u32);
    pub const kana_RA: Key = Key(gdk_sys::GDK_KEY_kana_RA as u32);
    pub const kana_RI: Key = Key(gdk_sys::GDK_KEY_kana_RI as u32);
    pub const kana_RU: Key = Key(gdk_sys::GDK_KEY_kana_RU as u32);
    pub const kana_RE: Key = Key(gdk_sys::GDK_KEY_kana_RE as u32);
    pub const kana_RO: Key = Key(gdk_sys::GDK_KEY_kana_RO as u32);
    pub const kana_WA: Key = Key(gdk_sys::GDK_KEY_kana_WA as u32);
    pub const kana_N: Key = Key(gdk_sys::GDK_KEY_kana_N as u32);
    pub const voicedsound: Key = Key(gdk_sys::GDK_KEY_voicedsound as u32);
    pub const semivoicedsound: Key = Key(gdk_sys::GDK_KEY_semivoicedsound as u32);
    pub const kana_switch: Key = Key(gdk_sys::GDK_KEY_kana_switch as u32);
    pub const Farsi_0: Key = Key(gdk_sys::GDK_KEY_Farsi_0 as u32);
    pub const Farsi_1: Key = Key(gdk_sys::GDK_KEY_Farsi_1 as u32);
    pub const Farsi_2: Key = Key(gdk_sys::GDK_KEY_Farsi_2 as u32);
    pub const Farsi_3: Key = Key(gdk_sys::GDK_KEY_Farsi_3 as u32);
    pub const Farsi_4: Key = Key(gdk_sys::GDK_KEY_Farsi_4 as u32);
    pub const Farsi_5: Key = Key(gdk_sys::GDK_KEY_Farsi_5 as u32);
    pub const Farsi_6: Key = Key(gdk_sys::GDK_KEY_Farsi_6 as u32);
    pub const Farsi_7: Key = Key(gdk_sys::GDK_KEY_Farsi_7 as u32);
    pub const Farsi_8: Key = Key(gdk_sys::GDK_KEY_Farsi_8 as u32);
    pub const Farsi_9: Key = Key(gdk_sys::GDK_KEY_Farsi_9 as u32);
    pub const Arabic_percent: Key = Key(gdk_sys::GDK_KEY_Arabic_percent as u32);
    pub const Arabic_superscript_alef: Key = Key(gdk_sys::GDK_KEY_Arabic_superscript_alef as u32);
    pub const Arabic_tteh: Key = Key(gdk_sys::GDK_KEY_Arabic_tteh as u32);
    pub const Arabic_peh: Key = Key(gdk_sys::GDK_KEY_Arabic_peh as u32);
    pub const Arabic_tcheh: Key = Key(gdk_sys::GDK_KEY_Arabic_tcheh as u32);
    pub const Arabic_ddal: Key = Key(gdk_sys::GDK_KEY_Arabic_ddal as u32);
    pub const Arabic_rreh: Key = Key(gdk_sys::GDK_KEY_Arabic_rreh as u32);
    pub const Arabic_comma: Key = Key(gdk_sys::GDK_KEY_Arabic_comma as u32);
    pub const Arabic_fullstop: Key = Key(gdk_sys::GDK_KEY_Arabic_fullstop as u32);
    pub const Arabic_0: Key = Key(gdk_sys::GDK_KEY_Arabic_0 as u32);
    pub const Arabic_1: Key = Key(gdk_sys::GDK_KEY_Arabic_1 as u32);
    pub const Arabic_2: Key = Key(gdk_sys::GDK_KEY_Arabic_2 as u32);
    pub const Arabic_3: Key = Key(gdk_sys::GDK_KEY_Arabic_3 as u32);
    pub const Arabic_4: Key = Key(gdk_sys::GDK_KEY_Arabic_4 as u32);
    pub const Arabic_5: Key = Key(gdk_sys::GDK_KEY_Arabic_5 as u32);
    pub const Arabic_6: Key = Key(gdk_sys::GDK_KEY_Arabic_6 as u32);
    pub const Arabic_7: Key = Key(gdk_sys::GDK_KEY_Arabic_7 as u32);
    pub const Arabic_8: Key = Key(gdk_sys::GDK_KEY_Arabic_8 as u32);
    pub const Arabic_9: Key = Key(gdk_sys::GDK_KEY_Arabic_9 as u32);
    pub const Arabic_semicolon: Key = Key(gdk_sys::GDK_KEY_Arabic_semicolon as u32);
    pub const Arabic_question_mark: Key = Key(gdk_sys::GDK_KEY_Arabic_question_mark as u32);
    pub const Arabic_hamza: Key = Key(gdk_sys::GDK_KEY_Arabic_hamza as u32);
    pub const Arabic_maddaonalef: Key = Key(gdk_sys::GDK_KEY_Arabic_maddaonalef as u32);
    pub const Arabic_hamzaonalef: Key = Key(gdk_sys::GDK_KEY_Arabic_hamzaonalef as u32);
    pub const Arabic_hamzaonwaw: Key = Key(gdk_sys::GDK_KEY_Arabic_hamzaonwaw as u32);
    pub const Arabic_hamzaunderalef: Key = Key(gdk_sys::GDK_KEY_Arabic_hamzaunderalef as u32);
    pub const Arabic_hamzaonyeh: Key = Key(gdk_sys::GDK_KEY_Arabic_hamzaonyeh as u32);
    pub const Arabic_alef: Key = Key(gdk_sys::GDK_KEY_Arabic_alef as u32);
    pub const Arabic_beh: Key = Key(gdk_sys::GDK_KEY_Arabic_beh as u32);
    pub const Arabic_tehmarbuta: Key = Key(gdk_sys::GDK_KEY_Arabic_tehmarbuta as u32);
    pub const Arabic_teh: Key = Key(gdk_sys::GDK_KEY_Arabic_teh as u32);
    pub const Arabic_theh: Key = Key(gdk_sys::GDK_KEY_Arabic_theh as u32);
    pub const Arabic_jeem: Key = Key(gdk_sys::GDK_KEY_Arabic_jeem as u32);
    pub const Arabic_hah: Key = Key(gdk_sys::GDK_KEY_Arabic_hah as u32);
    pub const Arabic_khah: Key = Key(gdk_sys::GDK_KEY_Arabic_khah as u32);
    pub const Arabic_dal: Key = Key(gdk_sys::GDK_KEY_Arabic_dal as u32);
    pub const Arabic_thal: Key = Key(gdk_sys::GDK_KEY_Arabic_thal as u32);
    pub const Arabic_ra: Key = Key(gdk_sys::GDK_KEY_Arabic_ra as u32);
    pub const Arabic_zain: Key = Key(gdk_sys::GDK_KEY_Arabic_zain as u32);
    pub const Arabic_seen: Key = Key(gdk_sys::GDK_KEY_Arabic_seen as u32);
    pub const Arabic_sheen: Key = Key(gdk_sys::GDK_KEY_Arabic_sheen as u32);
    pub const Arabic_sad: Key = Key(gdk_sys::GDK_KEY_Arabic_sad as u32);
    pub const Arabic_dad: Key = Key(gdk_sys::GDK_KEY_Arabic_dad as u32);
    pub const Arabic_tah: Key = Key(gdk_sys::GDK_KEY_Arabic_tah as u32);
    pub const Arabic_zah: Key = Key(gdk_sys::GDK_KEY_Arabic_zah as u32);
    pub const Arabic_ain: Key = Key(gdk_sys::GDK_KEY_Arabic_ain as u32);
    pub const Arabic_ghain: Key = Key(gdk_sys::GDK_KEY_Arabic_ghain as u32);
    pub const Arabic_tatweel: Key = Key(gdk_sys::GDK_KEY_Arabic_tatweel as u32);
    pub const Arabic_feh: Key = Key(gdk_sys::GDK_KEY_Arabic_feh as u32);
    pub const Arabic_qaf: Key = Key(gdk_sys::GDK_KEY_Arabic_qaf as u32);
    pub const Arabic_kaf: Key = Key(gdk_sys::GDK_KEY_Arabic_kaf as u32);
    pub const Arabic_lam: Key = Key(gdk_sys::GDK_KEY_Arabic_lam as u32);
    pub const Arabic_meem: Key = Key(gdk_sys::GDK_KEY_Arabic_meem as u32);
    pub const Arabic_noon: Key = Key(gdk_sys::GDK_KEY_Arabic_noon as u32);
    pub const Arabic_ha: Key = Key(gdk_sys::GDK_KEY_Arabic_ha as u32);
    pub const Arabic_heh: Key = Key(gdk_sys::GDK_KEY_Arabic_heh as u32);
    pub const Arabic_waw: Key = Key(gdk_sys::GDK_KEY_Arabic_waw as u32);
    pub const Arabic_alefmaksura: Key = Key(gdk_sys::GDK_KEY_Arabic_alefmaksura as u32);
    pub const Arabic_yeh: Key = Key(gdk_sys::GDK_KEY_Arabic_yeh as u32);
    pub const Arabic_fathatan: Key = Key(gdk_sys::GDK_KEY_Arabic_fathatan as u32);
    pub const Arabic_dammatan: Key = Key(gdk_sys::GDK_KEY_Arabic_dammatan as u32);
    pub const Arabic_kasratan: Key = Key(gdk_sys::GDK_KEY_Arabic_kasratan as u32);
    pub const Arabic_fatha: Key = Key(gdk_sys::GDK_KEY_Arabic_fatha as u32);
    pub const Arabic_damma: Key = Key(gdk_sys::GDK_KEY_Arabic_damma as u32);
    pub const Arabic_kasra: Key = Key(gdk_sys::GDK_KEY_Arabic_kasra as u32);
    pub const Arabic_shadda: Key = Key(gdk_sys::GDK_KEY_Arabic_shadda as u32);
    pub const Arabic_sukun: Key = Key(gdk_sys::GDK_KEY_Arabic_sukun as u32);
    pub const Arabic_madda_above: Key = Key(gdk_sys::GDK_KEY_Arabic_madda_above as u32);
    pub const Arabic_hamza_above: Key = Key(gdk_sys::GDK_KEY_Arabic_hamza_above as u32);
    pub const Arabic_hamza_below: Key = Key(gdk_sys::GDK_KEY_Arabic_hamza_below as u32);
    pub const Arabic_jeh: Key = Key(gdk_sys::GDK_KEY_Arabic_jeh as u32);
    pub const Arabic_veh: Key = Key(gdk_sys::GDK_KEY_Arabic_veh as u32);
    pub const Arabic_keheh: Key = Key(gdk_sys::GDK_KEY_Arabic_keheh as u32);
    pub const Arabic_gaf: Key = Key(gdk_sys::GDK_KEY_Arabic_gaf as u32);
    pub const Arabic_noon_ghunna: Key = Key(gdk_sys::GDK_KEY_Arabic_noon_ghunna as u32);
    pub const Arabic_heh_doachashmee: Key = Key(gdk_sys::GDK_KEY_Arabic_heh_doachashmee as u32);
    pub const Farsi_yeh: Key = Key(gdk_sys::GDK_KEY_Farsi_yeh as u32);
    pub const Arabic_farsi_yeh: Key = Key(gdk_sys::GDK_KEY_Arabic_farsi_yeh as u32);
    pub const Arabic_yeh_baree: Key = Key(gdk_sys::GDK_KEY_Arabic_yeh_baree as u32);
    pub const Arabic_heh_goal: Key = Key(gdk_sys::GDK_KEY_Arabic_heh_goal as u32);
    pub const Arabic_switch: Key = Key(gdk_sys::GDK_KEY_Arabic_switch as u32);
    pub const Cyrillic_GHE_bar: Key = Key(gdk_sys::GDK_KEY_Cyrillic_GHE_bar as u32);
    pub const Cyrillic_ghe_bar: Key = Key(gdk_sys::GDK_KEY_Cyrillic_ghe_bar as u32);
    pub const Cyrillic_ZHE_descender: Key = Key(gdk_sys::GDK_KEY_Cyrillic_ZHE_descender as u32);
    pub const Cyrillic_zhe_descender: Key = Key(gdk_sys::GDK_KEY_Cyrillic_zhe_descender as u32);
    pub const Cyrillic_KA_descender: Key = Key(gdk_sys::GDK_KEY_Cyrillic_KA_descender as u32);
    pub const Cyrillic_ka_descender: Key = Key(gdk_sys::GDK_KEY_Cyrillic_ka_descender as u32);
    pub const Cyrillic_KA_vertstroke: Key = Key(gdk_sys::GDK_KEY_Cyrillic_KA_vertstroke as u32);
    pub const Cyrillic_ka_vertstroke: Key = Key(gdk_sys::GDK_KEY_Cyrillic_ka_vertstroke as u32);
    pub const Cyrillic_EN_descender: Key = Key(gdk_sys::GDK_KEY_Cyrillic_EN_descender as u32);
    pub const Cyrillic_en_descender: Key = Key(gdk_sys::GDK_KEY_Cyrillic_en_descender as u32);
    pub const Cyrillic_U_straight: Key = Key(gdk_sys::GDK_KEY_Cyrillic_U_straight as u32);
    pub const Cyrillic_u_straight: Key = Key(gdk_sys::GDK_KEY_Cyrillic_u_straight as u32);
    pub const Cyrillic_U_straight_bar: Key = Key(gdk_sys::GDK_KEY_Cyrillic_U_straight_bar as u32);
    pub const Cyrillic_u_straight_bar: Key = Key(gdk_sys::GDK_KEY_Cyrillic_u_straight_bar as u32);
    pub const Cyrillic_HA_descender: Key = Key(gdk_sys::GDK_KEY_Cyrillic_HA_descender as u32);
    pub const Cyrillic_ha_descender: Key = Key(gdk_sys::GDK_KEY_Cyrillic_ha_descender as u32);
    pub const Cyrillic_CHE_descender: Key = Key(gdk_sys::GDK_KEY_Cyrillic_CHE_descender as u32);
    pub const Cyrillic_che_descender: Key = Key(gdk_sys::GDK_KEY_Cyrillic_che_descender as u32);
    pub const Cyrillic_CHE_vertstroke: Key = Key(gdk_sys::GDK_KEY_Cyrillic_CHE_vertstroke as u32);
    pub const Cyrillic_che_vertstroke: Key = Key(gdk_sys::GDK_KEY_Cyrillic_che_vertstroke as u32);
    pub const Cyrillic_SHHA: Key = Key(gdk_sys::GDK_KEY_Cyrillic_SHHA as u32);
    pub const Cyrillic_shha: Key = Key(gdk_sys::GDK_KEY_Cyrillic_shha as u32);
    pub const Cyrillic_SCHWA: Key = Key(gdk_sys::GDK_KEY_Cyrillic_SCHWA as u32);
    pub const Cyrillic_schwa: Key = Key(gdk_sys::GDK_KEY_Cyrillic_schwa as u32);
    pub const Cyrillic_I_macron: Key = Key(gdk_sys::GDK_KEY_Cyrillic_I_macron as u32);
    pub const Cyrillic_i_macron: Key = Key(gdk_sys::GDK_KEY_Cyrillic_i_macron as u32);
    pub const Cyrillic_O_bar: Key = Key(gdk_sys::GDK_KEY_Cyrillic_O_bar as u32);
    pub const Cyrillic_o_bar: Key = Key(gdk_sys::GDK_KEY_Cyrillic_o_bar as u32);
    pub const Cyrillic_U_macron: Key = Key(gdk_sys::GDK_KEY_Cyrillic_U_macron as u32);
    pub const Cyrillic_u_macron: Key = Key(gdk_sys::GDK_KEY_Cyrillic_u_macron as u32);
    pub const Serbian_dje: Key = Key(gdk_sys::GDK_KEY_Serbian_dje as u32);
    pub const Macedonia_gje: Key = Key(gdk_sys::GDK_KEY_Macedonia_gje as u32);
    pub const Cyrillic_io: Key = Key(gdk_sys::GDK_KEY_Cyrillic_io as u32);
    pub const Ukrainian_ie: Key = Key(gdk_sys::GDK_KEY_Ukrainian_ie as u32);
    pub const Ukranian_je: Key = Key(gdk_sys::GDK_KEY_Ukranian_je as u32);
    pub const Macedonia_dse: Key = Key(gdk_sys::GDK_KEY_Macedonia_dse as u32);
    pub const Ukrainian_i: Key = Key(gdk_sys::GDK_KEY_Ukrainian_i as u32);
    pub const Ukranian_i: Key = Key(gdk_sys::GDK_KEY_Ukranian_i as u32);
    pub const Ukrainian_yi: Key = Key(gdk_sys::GDK_KEY_Ukrainian_yi as u32);
    pub const Ukranian_yi: Key = Key(gdk_sys::GDK_KEY_Ukranian_yi as u32);
    pub const Cyrillic_je: Key = Key(gdk_sys::GDK_KEY_Cyrillic_je as u32);
    pub const Serbian_je: Key = Key(gdk_sys::GDK_KEY_Serbian_je as u32);
    pub const Cyrillic_lje: Key = Key(gdk_sys::GDK_KEY_Cyrillic_lje as u32);
    pub const Serbian_lje: Key = Key(gdk_sys::GDK_KEY_Serbian_lje as u32);
    pub const Cyrillic_nje: Key = Key(gdk_sys::GDK_KEY_Cyrillic_nje as u32);
    pub const Serbian_nje: Key = Key(gdk_sys::GDK_KEY_Serbian_nje as u32);
    pub const Serbian_tshe: Key = Key(gdk_sys::GDK_KEY_Serbian_tshe as u32);
    pub const Macedonia_kje: Key = Key(gdk_sys::GDK_KEY_Macedonia_kje as u32);
    pub const Ukrainian_ghe_with_upturn: Key =
        Key(gdk_sys::GDK_KEY_Ukrainian_ghe_with_upturn as u32);
    pub const Byelorussian_shortu: Key = Key(gdk_sys::GDK_KEY_Byelorussian_shortu as u32);
    pub const Cyrillic_dzhe: Key = Key(gdk_sys::GDK_KEY_Cyrillic_dzhe as u32);
    pub const Serbian_dze: Key = Key(gdk_sys::GDK_KEY_Serbian_dze as u32);
    pub const numerosign: Key = Key(gdk_sys::GDK_KEY_numerosign as u32);
    pub const Serbian_DJE: Key = Key(gdk_sys::GDK_KEY_Serbian_DJE as u32);
    pub const Macedonia_GJE: Key = Key(gdk_sys::GDK_KEY_Macedonia_GJE as u32);
    pub const Cyrillic_IO: Key = Key(gdk_sys::GDK_KEY_Cyrillic_IO as u32);
    pub const Ukrainian_IE: Key = Key(gdk_sys::GDK_KEY_Ukrainian_IE as u32);
    pub const Ukranian_JE: Key = Key(gdk_sys::GDK_KEY_Ukranian_JE as u32);
    pub const Macedonia_DSE: Key = Key(gdk_sys::GDK_KEY_Macedonia_DSE as u32);
    pub const Ukrainian_I: Key = Key(gdk_sys::GDK_KEY_Ukrainian_I as u32);
    pub const Ukranian_I: Key = Key(gdk_sys::GDK_KEY_Ukranian_I as u32);
    pub const Ukrainian_YI: Key = Key(gdk_sys::GDK_KEY_Ukrainian_YI as u32);
    pub const Ukranian_YI: Key = Key(gdk_sys::GDK_KEY_Ukranian_YI as u32);
    pub const Cyrillic_JE: Key = Key(gdk_sys::GDK_KEY_Cyrillic_JE as u32);
    pub const Serbian_JE: Key = Key(gdk_sys::GDK_KEY_Serbian_JE as u32);
    pub const Cyrillic_LJE: Key = Key(gdk_sys::GDK_KEY_Cyrillic_LJE as u32);
    pub const Serbian_LJE: Key = Key(gdk_sys::GDK_KEY_Serbian_LJE as u32);
    pub const Cyrillic_NJE: Key = Key(gdk_sys::GDK_KEY_Cyrillic_NJE as u32);
    pub const Serbian_NJE: Key = Key(gdk_sys::GDK_KEY_Serbian_NJE as u32);
    pub const Serbian_TSHE: Key = Key(gdk_sys::GDK_KEY_Serbian_TSHE as u32);
    pub const Macedonia_KJE: Key = Key(gdk_sys::GDK_KEY_Macedonia_KJE as u32);
    pub const Ukrainian_GHE_WITH_UPTURN: Key =
        Key(gdk_sys::GDK_KEY_Ukrainian_GHE_WITH_UPTURN as u32);
    pub const Byelorussian_SHORTU: Key = Key(gdk_sys::GDK_KEY_Byelorussian_SHORTU as u32);
    pub const Cyrillic_DZHE: Key = Key(gdk_sys::GDK_KEY_Cyrillic_DZHE as u32);
    pub const Serbian_DZE: Key = Key(gdk_sys::GDK_KEY_Serbian_DZE as u32);
    pub const Cyrillic_yu: Key = Key(gdk_sys::GDK_KEY_Cyrillic_yu as u32);
    pub const Cyrillic_a: Key = Key(gdk_sys::GDK_KEY_Cyrillic_a as u32);
    pub const Cyrillic_be: Key = Key(gdk_sys::GDK_KEY_Cyrillic_be as u32);
    pub const Cyrillic_tse: Key = Key(gdk_sys::GDK_KEY_Cyrillic_tse as u32);
    pub const Cyrillic_de: Key = Key(gdk_sys::GDK_KEY_Cyrillic_de as u32);
    pub const Cyrillic_ie: Key = Key(gdk_sys::GDK_KEY_Cyrillic_ie as u32);
    pub const Cyrillic_ef: Key = Key(gdk_sys::GDK_KEY_Cyrillic_ef as u32);
    pub const Cyrillic_ghe: Key = Key(gdk_sys::GDK_KEY_Cyrillic_ghe as u32);
    pub const Cyrillic_ha: Key = Key(gdk_sys::GDK_KEY_Cyrillic_ha as u32);
    pub const Cyrillic_i: Key = Key(gdk_sys::GDK_KEY_Cyrillic_i as u32);
    pub const Cyrillic_shorti: Key = Key(gdk_sys::GDK_KEY_Cyrillic_shorti as u32);
    pub const Cyrillic_ka: Key = Key(gdk_sys::GDK_KEY_Cyrillic_ka as u32);
    pub const Cyrillic_el: Key = Key(gdk_sys::GDK_KEY_Cyrillic_el as u32);
    pub const Cyrillic_em: Key = Key(gdk_sys::GDK_KEY_Cyrillic_em as u32);
    pub const Cyrillic_en: Key = Key(gdk_sys::GDK_KEY_Cyrillic_en as u32);
    pub const Cyrillic_o: Key = Key(gdk_sys::GDK_KEY_Cyrillic_o as u32);
    pub const Cyrillic_pe: Key = Key(gdk_sys::GDK_KEY_Cyrillic_pe as u32);
    pub const Cyrillic_ya: Key = Key(gdk_sys::GDK_KEY_Cyrillic_ya as u32);
    pub const Cyrillic_er: Key = Key(gdk_sys::GDK_KEY_Cyrillic_er as u32);
    pub const Cyrillic_es: Key = Key(gdk_sys::GDK_KEY_Cyrillic_es as u32);
    pub const Cyrillic_te: Key = Key(gdk_sys::GDK_KEY_Cyrillic_te as u32);
    pub const Cyrillic_u: Key = Key(gdk_sys::GDK_KEY_Cyrillic_u as u32);
    pub const Cyrillic_zhe: Key = Key(gdk_sys::GDK_KEY_Cyrillic_zhe as u32);
    pub const Cyrillic_ve: Key = Key(gdk_sys::GDK_KEY_Cyrillic_ve as u32);
    pub const Cyrillic_softsign: Key = Key(gdk_sys::GDK_KEY_Cyrillic_softsign as u32);
    pub const Cyrillic_yeru: Key = Key(gdk_sys::GDK_KEY_Cyrillic_yeru as u32);
    pub const Cyrillic_ze: Key = Key(gdk_sys::GDK_KEY_Cyrillic_ze as u32);
    pub const Cyrillic_sha: Key = Key(gdk_sys::GDK_KEY_Cyrillic_sha as u32);
    pub const Cyrillic_e: Key = Key(gdk_sys::GDK_KEY_Cyrillic_e as u32);
    pub const Cyrillic_shcha: Key = Key(gdk_sys::GDK_KEY_Cyrillic_shcha as u32);
    pub const Cyrillic_che: Key = Key(gdk_sys::GDK_KEY_Cyrillic_che as u32);
    pub const Cyrillic_hardsign: Key = Key(gdk_sys::GDK_KEY_Cyrillic_hardsign as u32);
    pub const Cyrillic_YU: Key = Key(gdk_sys::GDK_KEY_Cyrillic_YU as u32);
    pub const Cyrillic_A: Key = Key(gdk_sys::GDK_KEY_Cyrillic_A as u32);
    pub const Cyrillic_BE: Key = Key(gdk_sys::GDK_KEY_Cyrillic_BE as u32);
    pub const Cyrillic_TSE: Key = Key(gdk_sys::GDK_KEY_Cyrillic_TSE as u32);
    pub const Cyrillic_DE: Key = Key(gdk_sys::GDK_KEY_Cyrillic_DE as u32);
    pub const Cyrillic_IE: Key = Key(gdk_sys::GDK_KEY_Cyrillic_IE as u32);
    pub const Cyrillic_EF: Key = Key(gdk_sys::GDK_KEY_Cyrillic_EF as u32);
    pub const Cyrillic_GHE: Key = Key(gdk_sys::GDK_KEY_Cyrillic_GHE as u32);
    pub const Cyrillic_HA: Key = Key(gdk_sys::GDK_KEY_Cyrillic_HA as u32);
    pub const Cyrillic_I: Key = Key(gdk_sys::GDK_KEY_Cyrillic_I as u32);
    pub const Cyrillic_SHORTI: Key = Key(gdk_sys::GDK_KEY_Cyrillic_SHORTI as u32);
    pub const Cyrillic_KA: Key = Key(gdk_sys::GDK_KEY_Cyrillic_KA as u32);
    pub const Cyrillic_EL: Key = Key(gdk_sys::GDK_KEY_Cyrillic_EL as u32);
    pub const Cyrillic_EM: Key = Key(gdk_sys::GDK_KEY_Cyrillic_EM as u32);
    pub const Cyrillic_EN: Key = Key(gdk_sys::GDK_KEY_Cyrillic_EN as u32);
    pub const Cyrillic_O: Key = Key(gdk_sys::GDK_KEY_Cyrillic_O as u32);
    pub const Cyrillic_PE: Key = Key(gdk_sys::GDK_KEY_Cyrillic_PE as u32);
    pub const Cyrillic_YA: Key = Key(gdk_sys::GDK_KEY_Cyrillic_YA as u32);
    pub const Cyrillic_ER: Key = Key(gdk_sys::GDK_KEY_Cyrillic_ER as u32);
    pub const Cyrillic_ES: Key = Key(gdk_sys::GDK_KEY_Cyrillic_ES as u32);
    pub const Cyrillic_TE: Key = Key(gdk_sys::GDK_KEY_Cyrillic_TE as u32);
    pub const Cyrillic_U: Key = Key(gdk_sys::GDK_KEY_Cyrillic_U as u32);
    pub const Cyrillic_ZHE: Key = Key(gdk_sys::GDK_KEY_Cyrillic_ZHE as u32);
    pub const Cyrillic_VE: Key = Key(gdk_sys::GDK_KEY_Cyrillic_VE as u32);
    pub const Cyrillic_SOFTSIGN: Key = Key(gdk_sys::GDK_KEY_Cyrillic_SOFTSIGN as u32);
    pub const Cyrillic_YERU: Key = Key(gdk_sys::GDK_KEY_Cyrillic_YERU as u32);
    pub const Cyrillic_ZE: Key = Key(gdk_sys::GDK_KEY_Cyrillic_ZE as u32);
    pub const Cyrillic_SHA: Key = Key(gdk_sys::GDK_KEY_Cyrillic_SHA as u32);
    pub const Cyrillic_E: Key = Key(gdk_sys::GDK_KEY_Cyrillic_E as u32);
    pub const Cyrillic_SHCHA: Key = Key(gdk_sys::GDK_KEY_Cyrillic_SHCHA as u32);
    pub const Cyrillic_CHE: Key = Key(gdk_sys::GDK_KEY_Cyrillic_CHE as u32);
    pub const Cyrillic_HARDSIGN: Key = Key(gdk_sys::GDK_KEY_Cyrillic_HARDSIGN as u32);
    pub const Greek_ALPHAaccent: Key = Key(gdk_sys::GDK_KEY_Greek_ALPHAaccent as u32);
    pub const Greek_EPSILONaccent: Key = Key(gdk_sys::GDK_KEY_Greek_EPSILONaccent as u32);
    pub const Greek_ETAaccent: Key = Key(gdk_sys::GDK_KEY_Greek_ETAaccent as u32);
    pub const Greek_IOTAaccent: Key = Key(gdk_sys::GDK_KEY_Greek_IOTAaccent as u32);
    pub const Greek_IOTAdieresis: Key = Key(gdk_sys::GDK_KEY_Greek_IOTAdieresis as u32);
    pub const Greek_IOTAdiaeresis: Key = Key(gdk_sys::GDK_KEY_Greek_IOTAdiaeresis as u32);
    pub const Greek_OMICRONaccent: Key = Key(gdk_sys::GDK_KEY_Greek_OMICRONaccent as u32);
    pub const Greek_UPSILONaccent: Key = Key(gdk_sys::GDK_KEY_Greek_UPSILONaccent as u32);
    pub const Greek_UPSILONdieresis: Key = Key(gdk_sys::GDK_KEY_Greek_UPSILONdieresis as u32);
    pub const Greek_OMEGAaccent: Key = Key(gdk_sys::GDK_KEY_Greek_OMEGAaccent as u32);
    pub const Greek_accentdieresis: Key = Key(gdk_sys::GDK_KEY_Greek_accentdieresis as u32);
    pub const Greek_horizbar: Key = Key(gdk_sys::GDK_KEY_Greek_horizbar as u32);
    pub const Greek_alphaaccent: Key = Key(gdk_sys::GDK_KEY_Greek_alphaaccent as u32);
    pub const Greek_epsilonaccent: Key = Key(gdk_sys::GDK_KEY_Greek_epsilonaccent as u32);
    pub const Greek_etaaccent: Key = Key(gdk_sys::GDK_KEY_Greek_etaaccent as u32);
    pub const Greek_iotaaccent: Key = Key(gdk_sys::GDK_KEY_Greek_iotaaccent as u32);
    pub const Greek_iotadieresis: Key = Key(gdk_sys::GDK_KEY_Greek_iotadieresis as u32);
    pub const Greek_iotaaccentdieresis: Key = Key(gdk_sys::GDK_KEY_Greek_iotaaccentdieresis as u32);
    pub const Greek_omicronaccent: Key = Key(gdk_sys::GDK_KEY_Greek_omicronaccent as u32);
    pub const Greek_upsilonaccent: Key = Key(gdk_sys::GDK_KEY_Greek_upsilonaccent as u32);
    pub const Greek_upsilondieresis: Key = Key(gdk_sys::GDK_KEY_Greek_upsilondieresis as u32);
    pub const Greek_upsilonaccentdieresis: Key =
        Key(gdk_sys::GDK_KEY_Greek_upsilonaccentdieresis as u32);
    pub const Greek_omegaaccent: Key = Key(gdk_sys::GDK_KEY_Greek_omegaaccent as u32);
    pub const Greek_ALPHA: Key = Key(gdk_sys::GDK_KEY_Greek_ALPHA as u32);
    pub const Greek_BETA: Key = Key(gdk_sys::GDK_KEY_Greek_BETA as u32);
    pub const Greek_GAMMA: Key = Key(gdk_sys::GDK_KEY_Greek_GAMMA as u32);
    pub const Greek_DELTA: Key = Key(gdk_sys::GDK_KEY_Greek_DELTA as u32);
    pub const Greek_EPSILON: Key = Key(gdk_sys::GDK_KEY_Greek_EPSILON as u32);
    pub const Greek_ZETA: Key = Key(gdk_sys::GDK_KEY_Greek_ZETA as u32);
    pub const Greek_ETA: Key = Key(gdk_sys::GDK_KEY_Greek_ETA as u32);
    pub const Greek_THETA: Key = Key(gdk_sys::GDK_KEY_Greek_THETA as u32);
    pub const Greek_IOTA: Key = Key(gdk_sys::GDK_KEY_Greek_IOTA as u32);
    pub const Greek_KAPPA: Key = Key(gdk_sys::GDK_KEY_Greek_KAPPA as u32);
    pub const Greek_LAMDA: Key = Key(gdk_sys::GDK_KEY_Greek_LAMDA as u32);
    pub const Greek_LAMBDA: Key = Key(gdk_sys::GDK_KEY_Greek_LAMBDA as u32);
    pub const Greek_MU: Key = Key(gdk_sys::GDK_KEY_Greek_MU as u32);
    pub const Greek_NU: Key = Key(gdk_sys::GDK_KEY_Greek_NU as u32);
    pub const Greek_XI: Key = Key(gdk_sys::GDK_KEY_Greek_XI as u32);
    pub const Greek_OMICRON: Key = Key(gdk_sys::GDK_KEY_Greek_OMICRON as u32);
    pub const Greek_PI: Key = Key(gdk_sys::GDK_KEY_Greek_PI as u32);
    pub const Greek_RHO: Key = Key(gdk_sys::GDK_KEY_Greek_RHO as u32);
    pub const Greek_SIGMA: Key = Key(gdk_sys::GDK_KEY_Greek_SIGMA as u32);
    pub const Greek_TAU: Key = Key(gdk_sys::GDK_KEY_Greek_TAU as u32);
    pub const Greek_UPSILON: Key = Key(gdk_sys::GDK_KEY_Greek_UPSILON as u32);
    pub const Greek_PHI: Key = Key(gdk_sys::GDK_KEY_Greek_PHI as u32);
    pub const Greek_CHI: Key = Key(gdk_sys::GDK_KEY_Greek_CHI as u32);
    pub const Greek_PSI: Key = Key(gdk_sys::GDK_KEY_Greek_PSI as u32);
    pub const Greek_OMEGA: Key = Key(gdk_sys::GDK_KEY_Greek_OMEGA as u32);
    pub const Greek_alpha: Key = Key(gdk_sys::GDK_KEY_Greek_alpha as u32);
    pub const Greek_beta: Key = Key(gdk_sys::GDK_KEY_Greek_beta as u32);
    pub const Greek_gamma: Key = Key(gdk_sys::GDK_KEY_Greek_gamma as u32);
    pub const Greek_delta: Key = Key(gdk_sys::GDK_KEY_Greek_delta as u32);
    pub const Greek_epsilon: Key = Key(gdk_sys::GDK_KEY_Greek_epsilon as u32);
    pub const Greek_zeta: Key = Key(gdk_sys::GDK_KEY_Greek_zeta as u32);
    pub const Greek_eta: Key = Key(gdk_sys::GDK_KEY_Greek_eta as u32);
    pub const Greek_theta: Key = Key(gdk_sys::GDK_KEY_Greek_theta as u32);
    pub const Greek_iota: Key = Key(gdk_sys::GDK_KEY_Greek_iota as u32);
    pub const Greek_kappa: Key = Key(gdk_sys::GDK_KEY_Greek_kappa as u32);
    pub const Greek_lamda: Key = Key(gdk_sys::GDK_KEY_Greek_lamda as u32);
    pub const Greek_lambda: Key = Key(gdk_sys::GDK_KEY_Greek_lambda as u32);
    pub const Greek_mu: Key = Key(gdk_sys::GDK_KEY_Greek_mu as u32);
    pub const Greek_nu: Key = Key(gdk_sys::GDK_KEY_Greek_nu as u32);
    pub const Greek_xi: Key = Key(gdk_sys::GDK_KEY_Greek_xi as u32);
    pub const Greek_omicron: Key = Key(gdk_sys::GDK_KEY_Greek_omicron as u32);
    pub const Greek_pi: Key = Key(gdk_sys::GDK_KEY_Greek_pi as u32);
    pub const Greek_rho: Key = Key(gdk_sys::GDK_KEY_Greek_rho as u32);
    pub const Greek_sigma: Key = Key(gdk_sys::GDK_KEY_Greek_sigma as u32);
    pub const Greek_finalsmallsigma: Key = Key(gdk_sys::GDK_KEY_Greek_finalsmallsigma as u32);
    pub const Greek_tau: Key = Key(gdk_sys::GDK_KEY_Greek_tau as u32);
    pub const Greek_upsilon: Key = Key(gdk_sys::GDK_KEY_Greek_upsilon as u32);
    pub const Greek_phi: Key = Key(gdk_sys::GDK_KEY_Greek_phi as u32);
    pub const Greek_chi: Key = Key(gdk_sys::GDK_KEY_Greek_chi as u32);
    pub const Greek_psi: Key = Key(gdk_sys::GDK_KEY_Greek_psi as u32);
    pub const Greek_omega: Key = Key(gdk_sys::GDK_KEY_Greek_omega as u32);
    pub const Greek_switch: Key = Key(gdk_sys::GDK_KEY_Greek_switch as u32);
    pub const leftradical: Key = Key(gdk_sys::GDK_KEY_leftradical as u32);
    pub const topleftradical: Key = Key(gdk_sys::GDK_KEY_topleftradical as u32);
    pub const horizconnector: Key = Key(gdk_sys::GDK_KEY_horizconnector as u32);
    pub const topintegral: Key = Key(gdk_sys::GDK_KEY_topintegral as u32);
    pub const botintegral: Key = Key(gdk_sys::GDK_KEY_botintegral as u32);
    pub const vertconnector: Key = Key(gdk_sys::GDK_KEY_vertconnector as u32);
    pub const topleftsqbracket: Key = Key(gdk_sys::GDK_KEY_topleftsqbracket as u32);
    pub const botleftsqbracket: Key = Key(gdk_sys::GDK_KEY_botleftsqbracket as u32);
    pub const toprightsqbracket: Key = Key(gdk_sys::GDK_KEY_toprightsqbracket as u32);
    pub const botrightsqbracket: Key = Key(gdk_sys::GDK_KEY_botrightsqbracket as u32);
    pub const topleftparens: Key = Key(gdk_sys::GDK_KEY_topleftparens as u32);
    pub const botleftparens: Key = Key(gdk_sys::GDK_KEY_botleftparens as u32);
    pub const toprightparens: Key = Key(gdk_sys::GDK_KEY_toprightparens as u32);
    pub const botrightparens: Key = Key(gdk_sys::GDK_KEY_botrightparens as u32);
    pub const leftmiddlecurlybrace: Key = Key(gdk_sys::GDK_KEY_leftmiddlecurlybrace as u32);
    pub const rightmiddlecurlybrace: Key = Key(gdk_sys::GDK_KEY_rightmiddlecurlybrace as u32);
    pub const topleftsummation: Key = Key(gdk_sys::GDK_KEY_topleftsummation as u32);
    pub const botleftsummation: Key = Key(gdk_sys::GDK_KEY_botleftsummation as u32);
    pub const topvertsummationconnector: Key =
        Key(gdk_sys::GDK_KEY_topvertsummationconnector as u32);
    pub const botvertsummationconnector: Key =
        Key(gdk_sys::GDK_KEY_botvertsummationconnector as u32);
    pub const toprightsummation: Key = Key(gdk_sys::GDK_KEY_toprightsummation as u32);
    pub const botrightsummation: Key = Key(gdk_sys::GDK_KEY_botrightsummation as u32);
    pub const rightmiddlesummation: Key = Key(gdk_sys::GDK_KEY_rightmiddlesummation as u32);
    pub const lessthanequal: Key = Key(gdk_sys::GDK_KEY_lessthanequal as u32);
    pub const notequal: Key = Key(gdk_sys::GDK_KEY_notequal as u32);
    pub const greaterthanequal: Key = Key(gdk_sys::GDK_KEY_greaterthanequal as u32);
    pub const integral: Key = Key(gdk_sys::GDK_KEY_integral as u32);
    pub const therefore: Key = Key(gdk_sys::GDK_KEY_therefore as u32);
    pub const variation: Key = Key(gdk_sys::GDK_KEY_variation as u32);
    pub const infinity: Key = Key(gdk_sys::GDK_KEY_infinity as u32);
    pub const nabla: Key = Key(gdk_sys::GDK_KEY_nabla as u32);
    pub const approximate: Key = Key(gdk_sys::GDK_KEY_approximate as u32);
    pub const similarequal: Key = Key(gdk_sys::GDK_KEY_similarequal as u32);
    pub const ifonlyif: Key = Key(gdk_sys::GDK_KEY_ifonlyif as u32);
    pub const implies: Key = Key(gdk_sys::GDK_KEY_implies as u32);
    pub const identical: Key = Key(gdk_sys::GDK_KEY_identical as u32);
    pub const radical: Key = Key(gdk_sys::GDK_KEY_radical as u32);
    pub const includedin: Key = Key(gdk_sys::GDK_KEY_includedin as u32);
    pub const includes: Key = Key(gdk_sys::GDK_KEY_includes as u32);
    pub const intersection: Key = Key(gdk_sys::GDK_KEY_intersection as u32);
    pub const union: Key = Key(gdk_sys::GDK_KEY_union as u32);
    pub const logicaland: Key = Key(gdk_sys::GDK_KEY_logicaland as u32);
    pub const logicalor: Key = Key(gdk_sys::GDK_KEY_logicalor as u32);
    pub const partialderivative: Key = Key(gdk_sys::GDK_KEY_partialderivative as u32);
    pub const function: Key = Key(gdk_sys::GDK_KEY_function as u32);
    pub const leftarrow: Key = Key(gdk_sys::GDK_KEY_leftarrow as u32);
    pub const uparrow: Key = Key(gdk_sys::GDK_KEY_uparrow as u32);
    pub const rightarrow: Key = Key(gdk_sys::GDK_KEY_rightarrow as u32);
    pub const downarrow: Key = Key(gdk_sys::GDK_KEY_downarrow as u32);
    pub const blank: Key = Key(gdk_sys::GDK_KEY_blank as u32);
    pub const soliddiamond: Key = Key(gdk_sys::GDK_KEY_soliddiamond as u32);
    pub const checkerboard: Key = Key(gdk_sys::GDK_KEY_checkerboard as u32);
    pub const ht: Key = Key(gdk_sys::GDK_KEY_ht as u32);
    pub const ff: Key = Key(gdk_sys::GDK_KEY_ff as u32);
    pub const cr: Key = Key(gdk_sys::GDK_KEY_cr as u32);
    pub const lf: Key = Key(gdk_sys::GDK_KEY_lf as u32);
    pub const nl: Key = Key(gdk_sys::GDK_KEY_nl as u32);
    pub const vt: Key = Key(gdk_sys::GDK_KEY_vt as u32);
    pub const lowrightcorner: Key = Key(gdk_sys::GDK_KEY_lowrightcorner as u32);
    pub const uprightcorner: Key = Key(gdk_sys::GDK_KEY_uprightcorner as u32);
    pub const upleftcorner: Key = Key(gdk_sys::GDK_KEY_upleftcorner as u32);
    pub const lowleftcorner: Key = Key(gdk_sys::GDK_KEY_lowleftcorner as u32);
    pub const crossinglines: Key = Key(gdk_sys::GDK_KEY_crossinglines as u32);
    pub const horizlinescan1: Key = Key(gdk_sys::GDK_KEY_horizlinescan1 as u32);
    pub const horizlinescan3: Key = Key(gdk_sys::GDK_KEY_horizlinescan3 as u32);
    pub const horizlinescan5: Key = Key(gdk_sys::GDK_KEY_horizlinescan5 as u32);
    pub const horizlinescan7: Key = Key(gdk_sys::GDK_KEY_horizlinescan7 as u32);
    pub const horizlinescan9: Key = Key(gdk_sys::GDK_KEY_horizlinescan9 as u32);
    pub const leftt: Key = Key(gdk_sys::GDK_KEY_leftt as u32);
    pub const rightt: Key = Key(gdk_sys::GDK_KEY_rightt as u32);
    pub const bott: Key = Key(gdk_sys::GDK_KEY_bott as u32);
    pub const topt: Key = Key(gdk_sys::GDK_KEY_topt as u32);
    pub const vertbar: Key = Key(gdk_sys::GDK_KEY_vertbar as u32);
    pub const emspace: Key = Key(gdk_sys::GDK_KEY_emspace as u32);
    pub const enspace: Key = Key(gdk_sys::GDK_KEY_enspace as u32);
    pub const em3space: Key = Key(gdk_sys::GDK_KEY_em3space as u32);
    pub const em4space: Key = Key(gdk_sys::GDK_KEY_em4space as u32);
    pub const digitspace: Key = Key(gdk_sys::GDK_KEY_digitspace as u32);
    pub const punctspace: Key = Key(gdk_sys::GDK_KEY_punctspace as u32);
    pub const thinspace: Key = Key(gdk_sys::GDK_KEY_thinspace as u32);
    pub const hairspace: Key = Key(gdk_sys::GDK_KEY_hairspace as u32);
    pub const emdash: Key = Key(gdk_sys::GDK_KEY_emdash as u32);
    pub const endash: Key = Key(gdk_sys::GDK_KEY_endash as u32);
    pub const signifblank: Key = Key(gdk_sys::GDK_KEY_signifblank as u32);
    pub const ellipsis: Key = Key(gdk_sys::GDK_KEY_ellipsis as u32);
    pub const doubbaselinedot: Key = Key(gdk_sys::GDK_KEY_doubbaselinedot as u32);
    pub const onethird: Key = Key(gdk_sys::GDK_KEY_onethird as u32);
    pub const twothirds: Key = Key(gdk_sys::GDK_KEY_twothirds as u32);
    pub const onefifth: Key = Key(gdk_sys::GDK_KEY_onefifth as u32);
    pub const twofifths: Key = Key(gdk_sys::GDK_KEY_twofifths as u32);
    pub const threefifths: Key = Key(gdk_sys::GDK_KEY_threefifths as u32);
    pub const fourfifths: Key = Key(gdk_sys::GDK_KEY_fourfifths as u32);
    pub const onesixth: Key = Key(gdk_sys::GDK_KEY_onesixth as u32);
    pub const fivesixths: Key = Key(gdk_sys::GDK_KEY_fivesixths as u32);
    pub const careof: Key = Key(gdk_sys::GDK_KEY_careof as u32);
    pub const figdash: Key = Key(gdk_sys::GDK_KEY_figdash as u32);
    pub const leftanglebracket: Key = Key(gdk_sys::GDK_KEY_leftanglebracket as u32);
    pub const decimalpoint: Key = Key(gdk_sys::GDK_KEY_decimalpoint as u32);
    pub const rightanglebracket: Key = Key(gdk_sys::GDK_KEY_rightanglebracket as u32);
    pub const marker: Key = Key(gdk_sys::GDK_KEY_marker as u32);
    pub const oneeighth: Key = Key(gdk_sys::GDK_KEY_oneeighth as u32);
    pub const threeeighths: Key = Key(gdk_sys::GDK_KEY_threeeighths as u32);
    pub const fiveeighths: Key = Key(gdk_sys::GDK_KEY_fiveeighths as u32);
    pub const seveneighths: Key = Key(gdk_sys::GDK_KEY_seveneighths as u32);
    pub const trademark: Key = Key(gdk_sys::GDK_KEY_trademark as u32);
    pub const signaturemark: Key = Key(gdk_sys::GDK_KEY_signaturemark as u32);
    pub const trademarkincircle: Key = Key(gdk_sys::GDK_KEY_trademarkincircle as u32);
    pub const leftopentriangle: Key = Key(gdk_sys::GDK_KEY_leftopentriangle as u32);
    pub const rightopentriangle: Key = Key(gdk_sys::GDK_KEY_rightopentriangle as u32);
    pub const emopencircle: Key = Key(gdk_sys::GDK_KEY_emopencircle as u32);
    pub const emopenrectangle: Key = Key(gdk_sys::GDK_KEY_emopenrectangle as u32);
    pub const leftsinglequotemark: Key = Key(gdk_sys::GDK_KEY_leftsinglequotemark as u32);
    pub const rightsinglequotemark: Key = Key(gdk_sys::GDK_KEY_rightsinglequotemark as u32);
    pub const leftdoublequotemark: Key = Key(gdk_sys::GDK_KEY_leftdoublequotemark as u32);
    pub const rightdoublequotemark: Key = Key(gdk_sys::GDK_KEY_rightdoublequotemark as u32);
    pub const prescription: Key = Key(gdk_sys::GDK_KEY_prescription as u32);
    pub const permille: Key = Key(gdk_sys::GDK_KEY_permille as u32);
    pub const minutes: Key = Key(gdk_sys::GDK_KEY_minutes as u32);
    pub const seconds: Key = Key(gdk_sys::GDK_KEY_seconds as u32);
    pub const latincross: Key = Key(gdk_sys::GDK_KEY_latincross as u32);
    pub const hexagram: Key = Key(gdk_sys::GDK_KEY_hexagram as u32);
    pub const filledrectbullet: Key = Key(gdk_sys::GDK_KEY_filledrectbullet as u32);
    pub const filledlefttribullet: Key = Key(gdk_sys::GDK_KEY_filledlefttribullet as u32);
    pub const filledrighttribullet: Key = Key(gdk_sys::GDK_KEY_filledrighttribullet as u32);
    pub const emfilledcircle: Key = Key(gdk_sys::GDK_KEY_emfilledcircle as u32);
    pub const emfilledrect: Key = Key(gdk_sys::GDK_KEY_emfilledrect as u32);
    pub const enopencircbullet: Key = Key(gdk_sys::GDK_KEY_enopencircbullet as u32);
    pub const enopensquarebullet: Key = Key(gdk_sys::GDK_KEY_enopensquarebullet as u32);
    pub const openrectbullet: Key = Key(gdk_sys::GDK_KEY_openrectbullet as u32);
    pub const opentribulletup: Key = Key(gdk_sys::GDK_KEY_opentribulletup as u32);
    pub const opentribulletdown: Key = Key(gdk_sys::GDK_KEY_opentribulletdown as u32);
    pub const openstar: Key = Key(gdk_sys::GDK_KEY_openstar as u32);
    pub const enfilledcircbullet: Key = Key(gdk_sys::GDK_KEY_enfilledcircbullet as u32);
    pub const enfilledsqbullet: Key = Key(gdk_sys::GDK_KEY_enfilledsqbullet as u32);
    pub const filledtribulletup: Key = Key(gdk_sys::GDK_KEY_filledtribulletup as u32);
    pub const filledtribulletdown: Key = Key(gdk_sys::GDK_KEY_filledtribulletdown as u32);
    pub const leftpointer: Key = Key(gdk_sys::GDK_KEY_leftpointer as u32);
    pub const rightpointer: Key = Key(gdk_sys::GDK_KEY_rightpointer as u32);
    pub const club: Key = Key(gdk_sys::GDK_KEY_club as u32);
    pub const diamond: Key = Key(gdk_sys::GDK_KEY_diamond as u32);
    pub const heart: Key = Key(gdk_sys::GDK_KEY_heart as u32);
    pub const maltesecross: Key = Key(gdk_sys::GDK_KEY_maltesecross as u32);
    pub const dagger: Key = Key(gdk_sys::GDK_KEY_dagger as u32);
    pub const doubledagger: Key = Key(gdk_sys::GDK_KEY_doubledagger as u32);
    pub const checkmark: Key = Key(gdk_sys::GDK_KEY_checkmark as u32);
    pub const ballotcross: Key = Key(gdk_sys::GDK_KEY_ballotcross as u32);
    pub const musicalsharp: Key = Key(gdk_sys::GDK_KEY_musicalsharp as u32);
    pub const musicalflat: Key = Key(gdk_sys::GDK_KEY_musicalflat as u32);
    pub const malesymbol: Key = Key(gdk_sys::GDK_KEY_malesymbol as u32);
    pub const femalesymbol: Key = Key(gdk_sys::GDK_KEY_femalesymbol as u32);
    pub const telephone: Key = Key(gdk_sys::GDK_KEY_telephone as u32);
    pub const telephonerecorder: Key = Key(gdk_sys::GDK_KEY_telephonerecorder as u32);
    pub const phonographcopyright: Key = Key(gdk_sys::GDK_KEY_phonographcopyright as u32);
    pub const caret: Key = Key(gdk_sys::GDK_KEY_caret as u32);
    pub const singlelowquotemark: Key = Key(gdk_sys::GDK_KEY_singlelowquotemark as u32);
    pub const doublelowquotemark: Key = Key(gdk_sys::GDK_KEY_doublelowquotemark as u32);
    pub const cursor: Key = Key(gdk_sys::GDK_KEY_cursor as u32);
    pub const leftcaret: Key = Key(gdk_sys::GDK_KEY_leftcaret as u32);
    pub const rightcaret: Key = Key(gdk_sys::GDK_KEY_rightcaret as u32);
    pub const downcaret: Key = Key(gdk_sys::GDK_KEY_downcaret as u32);
    pub const upcaret: Key = Key(gdk_sys::GDK_KEY_upcaret as u32);
    pub const overbar: Key = Key(gdk_sys::GDK_KEY_overbar as u32);
    pub const downtack: Key = Key(gdk_sys::GDK_KEY_downtack as u32);
    pub const upshoe: Key = Key(gdk_sys::GDK_KEY_upshoe as u32);
    pub const downstile: Key = Key(gdk_sys::GDK_KEY_downstile as u32);
    pub const underbar: Key = Key(gdk_sys::GDK_KEY_underbar as u32);
    pub const jot: Key = Key(gdk_sys::GDK_KEY_jot as u32);
    pub const quad: Key = Key(gdk_sys::GDK_KEY_quad as u32);
    pub const uptack: Key = Key(gdk_sys::GDK_KEY_uptack as u32);
    pub const circle: Key = Key(gdk_sys::GDK_KEY_circle as u32);
    pub const upstile: Key = Key(gdk_sys::GDK_KEY_upstile as u32);
    pub const downshoe: Key = Key(gdk_sys::GDK_KEY_downshoe as u32);
    pub const rightshoe: Key = Key(gdk_sys::GDK_KEY_rightshoe as u32);
    pub const leftshoe: Key = Key(gdk_sys::GDK_KEY_leftshoe as u32);
    pub const lefttack: Key = Key(gdk_sys::GDK_KEY_lefttack as u32);
    pub const righttack: Key = Key(gdk_sys::GDK_KEY_righttack as u32);
    pub const hebrew_doublelowline: Key = Key(gdk_sys::GDK_KEY_hebrew_doublelowline as u32);
    pub const hebrew_aleph: Key = Key(gdk_sys::GDK_KEY_hebrew_aleph as u32);
    pub const hebrew_bet: Key = Key(gdk_sys::GDK_KEY_hebrew_bet as u32);
    pub const hebrew_beth: Key = Key(gdk_sys::GDK_KEY_hebrew_beth as u32);
    pub const hebrew_gimel: Key = Key(gdk_sys::GDK_KEY_hebrew_gimel as u32);
    pub const hebrew_gimmel: Key = Key(gdk_sys::GDK_KEY_hebrew_gimmel as u32);
    pub const hebrew_dalet: Key = Key(gdk_sys::GDK_KEY_hebrew_dalet as u32);
    pub const hebrew_daleth: Key = Key(gdk_sys::GDK_KEY_hebrew_daleth as u32);
    pub const hebrew_he: Key = Key(gdk_sys::GDK_KEY_hebrew_he as u32);
    pub const hebrew_waw: Key = Key(gdk_sys::GDK_KEY_hebrew_waw as u32);
    pub const hebrew_zain: Key = Key(gdk_sys::GDK_KEY_hebrew_zain as u32);
    pub const hebrew_zayin: Key = Key(gdk_sys::GDK_KEY_hebrew_zayin as u32);
    pub const hebrew_chet: Key = Key(gdk_sys::GDK_KEY_hebrew_chet as u32);
    pub const hebrew_het: Key = Key(gdk_sys::GDK_KEY_hebrew_het as u32);
    pub const hebrew_tet: Key = Key(gdk_sys::GDK_KEY_hebrew_tet as u32);
    pub const hebrew_teth: Key = Key(gdk_sys::GDK_KEY_hebrew_teth as u32);
    pub const hebrew_yod: Key = Key(gdk_sys::GDK_KEY_hebrew_yod as u32);
    pub const hebrew_finalkaph: Key = Key(gdk_sys::GDK_KEY_hebrew_finalkaph as u32);
    pub const hebrew_kaph: Key = Key(gdk_sys::GDK_KEY_hebrew_kaph as u32);
    pub const hebrew_lamed: Key = Key(gdk_sys::GDK_KEY_hebrew_lamed as u32);
    pub const hebrew_finalmem: Key = Key(gdk_sys::GDK_KEY_hebrew_finalmem as u32);
    pub const hebrew_mem: Key = Key(gdk_sys::GDK_KEY_hebrew_mem as u32);
    pub const hebrew_finalnun: Key = Key(gdk_sys::GDK_KEY_hebrew_finalnun as u32);
    pub const hebrew_nun: Key = Key(gdk_sys::GDK_KEY_hebrew_nun as u32);
    pub const hebrew_samech: Key = Key(gdk_sys::GDK_KEY_hebrew_samech as u32);
    pub const hebrew_samekh: Key = Key(gdk_sys::GDK_KEY_hebrew_samekh as u32);
    pub const hebrew_ayin: Key = Key(gdk_sys::GDK_KEY_hebrew_ayin as u32);
    pub const hebrew_finalpe: Key = Key(gdk_sys::GDK_KEY_hebrew_finalpe as u32);
    pub const hebrew_pe: Key = Key(gdk_sys::GDK_KEY_hebrew_pe as u32);
    pub const hebrew_finalzade: Key = Key(gdk_sys::GDK_KEY_hebrew_finalzade as u32);
    pub const hebrew_finalzadi: Key = Key(gdk_sys::GDK_KEY_hebrew_finalzadi as u32);
    pub const hebrew_zade: Key = Key(gdk_sys::GDK_KEY_hebrew_zade as u32);
    pub const hebrew_zadi: Key = Key(gdk_sys::GDK_KEY_hebrew_zadi as u32);
    pub const hebrew_qoph: Key = Key(gdk_sys::GDK_KEY_hebrew_qoph as u32);
    pub const hebrew_kuf: Key = Key(gdk_sys::GDK_KEY_hebrew_kuf as u32);
    pub const hebrew_resh: Key = Key(gdk_sys::GDK_KEY_hebrew_resh as u32);
    pub const hebrew_shin: Key = Key(gdk_sys::GDK_KEY_hebrew_shin as u32);
    pub const hebrew_taw: Key = Key(gdk_sys::GDK_KEY_hebrew_taw as u32);
    pub const hebrew_taf: Key = Key(gdk_sys::GDK_KEY_hebrew_taf as u32);
    pub const Hebrew_switch: Key = Key(gdk_sys::GDK_KEY_Hebrew_switch as u32);
    pub const Thai_kokai: Key = Key(gdk_sys::GDK_KEY_Thai_kokai as u32);
    pub const Thai_khokhai: Key = Key(gdk_sys::GDK_KEY_Thai_khokhai as u32);
    pub const Thai_khokhuat: Key = Key(gdk_sys::GDK_KEY_Thai_khokhuat as u32);
    pub const Thai_khokhwai: Key = Key(gdk_sys::GDK_KEY_Thai_khokhwai as u32);
    pub const Thai_khokhon: Key = Key(gdk_sys::GDK_KEY_Thai_khokhon as u32);
    pub const Thai_khorakhang: Key = Key(gdk_sys::GDK_KEY_Thai_khorakhang as u32);
    pub const Thai_ngongu: Key = Key(gdk_sys::GDK_KEY_Thai_ngongu as u32);
    pub const Thai_chochan: Key = Key(gdk_sys::GDK_KEY_Thai_chochan as u32);
    pub const Thai_choching: Key = Key(gdk_sys::GDK_KEY_Thai_choching as u32);
    pub const Thai_chochang: Key = Key(gdk_sys::GDK_KEY_Thai_chochang as u32);
    pub const Thai_soso: Key = Key(gdk_sys::GDK_KEY_Thai_soso as u32);
    pub const Thai_chochoe: Key = Key(gdk_sys::GDK_KEY_Thai_chochoe as u32);
    pub const Thai_yoying: Key = Key(gdk_sys::GDK_KEY_Thai_yoying as u32);
    pub const Thai_dochada: Key = Key(gdk_sys::GDK_KEY_Thai_dochada as u32);
    pub const Thai_topatak: Key = Key(gdk_sys::GDK_KEY_Thai_topatak as u32);
    pub const Thai_thothan: Key = Key(gdk_sys::GDK_KEY_Thai_thothan as u32);
    pub const Thai_thonangmontho: Key = Key(gdk_sys::GDK_KEY_Thai_thonangmontho as u32);
    pub const Thai_thophuthao: Key = Key(gdk_sys::GDK_KEY_Thai_thophuthao as u32);
    pub const Thai_nonen: Key = Key(gdk_sys::GDK_KEY_Thai_nonen as u32);
    pub const Thai_dodek: Key = Key(gdk_sys::GDK_KEY_Thai_dodek as u32);
    pub const Thai_totao: Key = Key(gdk_sys::GDK_KEY_Thai_totao as u32);
    pub const Thai_thothung: Key = Key(gdk_sys::GDK_KEY_Thai_thothung as u32);
    pub const Thai_thothahan: Key = Key(gdk_sys::GDK_KEY_Thai_thothahan as u32);
    pub const Thai_thothong: Key = Key(gdk_sys::GDK_KEY_Thai_thothong as u32);
    pub const Thai_nonu: Key = Key(gdk_sys::GDK_KEY_Thai_nonu as u32);
    pub const Thai_bobaimai: Key = Key(gdk_sys::GDK_KEY_Thai_bobaimai as u32);
    pub const Thai_popla: Key = Key(gdk_sys::GDK_KEY_Thai_popla as u32);
    pub const Thai_phophung: Key = Key(gdk_sys::GDK_KEY_Thai_phophung as u32);
    pub const Thai_fofa: Key = Key(gdk_sys::GDK_KEY_Thai_fofa as u32);
    pub const Thai_phophan: Key = Key(gdk_sys::GDK_KEY_Thai_phophan as u32);
    pub const Thai_fofan: Key = Key(gdk_sys::GDK_KEY_Thai_fofan as u32);
    pub const Thai_phosamphao: Key = Key(gdk_sys::GDK_KEY_Thai_phosamphao as u32);
    pub const Thai_moma: Key = Key(gdk_sys::GDK_KEY_Thai_moma as u32);
    pub const Thai_yoyak: Key = Key(gdk_sys::GDK_KEY_Thai_yoyak as u32);
    pub const Thai_rorua: Key = Key(gdk_sys::GDK_KEY_Thai_rorua as u32);
    pub const Thai_ru: Key = Key(gdk_sys::GDK_KEY_Thai_ru as u32);
    pub const Thai_loling: Key = Key(gdk_sys::GDK_KEY_Thai_loling as u32);
    pub const Thai_lu: Key = Key(gdk_sys::GDK_KEY_Thai_lu as u32);
    pub const Thai_wowaen: Key = Key(gdk_sys::GDK_KEY_Thai_wowaen as u32);
    pub const Thai_sosala: Key = Key(gdk_sys::GDK_KEY_Thai_sosala as u32);
    pub const Thai_sorusi: Key = Key(gdk_sys::GDK_KEY_Thai_sorusi as u32);
    pub const Thai_sosua: Key = Key(gdk_sys::GDK_KEY_Thai_sosua as u32);
    pub const Thai_hohip: Key = Key(gdk_sys::GDK_KEY_Thai_hohip as u32);
    pub const Thai_lochula: Key = Key(gdk_sys::GDK_KEY_Thai_lochula as u32);
    pub const Thai_oang: Key = Key(gdk_sys::GDK_KEY_Thai_oang as u32);
    pub const Thai_honokhuk: Key = Key(gdk_sys::GDK_KEY_Thai_honokhuk as u32);
    pub const Thai_paiyannoi: Key = Key(gdk_sys::GDK_KEY_Thai_paiyannoi as u32);
    pub const Thai_saraa: Key = Key(gdk_sys::GDK_KEY_Thai_saraa as u32);
    pub const Thai_maihanakat: Key = Key(gdk_sys::GDK_KEY_Thai_maihanakat as u32);
    pub const Thai_saraaa: Key = Key(gdk_sys::GDK_KEY_Thai_saraaa as u32);
    pub const Thai_saraam: Key = Key(gdk_sys::GDK_KEY_Thai_saraam as u32);
    pub const Thai_sarai: Key = Key(gdk_sys::GDK_KEY_Thai_sarai as u32);
    pub const Thai_saraii: Key = Key(gdk_sys::GDK_KEY_Thai_saraii as u32);
    pub const Thai_saraue: Key = Key(gdk_sys::GDK_KEY_Thai_saraue as u32);
    pub const Thai_sarauee: Key = Key(gdk_sys::GDK_KEY_Thai_sarauee as u32);
    pub const Thai_sarau: Key = Key(gdk_sys::GDK_KEY_Thai_sarau as u32);
    pub const Thai_sarauu: Key = Key(gdk_sys::GDK_KEY_Thai_sarauu as u32);
    pub const Thai_phinthu: Key = Key(gdk_sys::GDK_KEY_Thai_phinthu as u32);
    pub const Thai_maihanakat_maitho: Key = Key(gdk_sys::GDK_KEY_Thai_maihanakat_maitho as u32);
    pub const Thai_baht: Key = Key(gdk_sys::GDK_KEY_Thai_baht as u32);
    pub const Thai_sarae: Key = Key(gdk_sys::GDK_KEY_Thai_sarae as u32);
    pub const Thai_saraae: Key = Key(gdk_sys::GDK_KEY_Thai_saraae as u32);
    pub const Thai_sarao: Key = Key(gdk_sys::GDK_KEY_Thai_sarao as u32);
    pub const Thai_saraaimaimuan: Key = Key(gdk_sys::GDK_KEY_Thai_saraaimaimuan as u32);
    pub const Thai_saraaimaimalai: Key = Key(gdk_sys::GDK_KEY_Thai_saraaimaimalai as u32);
    pub const Thai_lakkhangyao: Key = Key(gdk_sys::GDK_KEY_Thai_lakkhangyao as u32);
    pub const Thai_maiyamok: Key = Key(gdk_sys::GDK_KEY_Thai_maiyamok as u32);
    pub const Thai_maitaikhu: Key = Key(gdk_sys::GDK_KEY_Thai_maitaikhu as u32);
    pub const Thai_maiek: Key = Key(gdk_sys::GDK_KEY_Thai_maiek as u32);
    pub const Thai_maitho: Key = Key(gdk_sys::GDK_KEY_Thai_maitho as u32);
    pub const Thai_maitri: Key = Key(gdk_sys::GDK_KEY_Thai_maitri as u32);
    pub const Thai_maichattawa: Key = Key(gdk_sys::GDK_KEY_Thai_maichattawa as u32);
    pub const Thai_thanthakhat: Key = Key(gdk_sys::GDK_KEY_Thai_thanthakhat as u32);
    pub const Thai_nikhahit: Key = Key(gdk_sys::GDK_KEY_Thai_nikhahit as u32);
    pub const Thai_leksun: Key = Key(gdk_sys::GDK_KEY_Thai_leksun as u32);
    pub const Thai_leknung: Key = Key(gdk_sys::GDK_KEY_Thai_leknung as u32);
    pub const Thai_leksong: Key = Key(gdk_sys::GDK_KEY_Thai_leksong as u32);
    pub const Thai_leksam: Key = Key(gdk_sys::GDK_KEY_Thai_leksam as u32);
    pub const Thai_leksi: Key = Key(gdk_sys::GDK_KEY_Thai_leksi as u32);
    pub const Thai_lekha: Key = Key(gdk_sys::GDK_KEY_Thai_lekha as u32);
    pub const Thai_lekhok: Key = Key(gdk_sys::GDK_KEY_Thai_lekhok as u32);
    pub const Thai_lekchet: Key = Key(gdk_sys::GDK_KEY_Thai_lekchet as u32);
    pub const Thai_lekpaet: Key = Key(gdk_sys::GDK_KEY_Thai_lekpaet as u32);
    pub const Thai_lekkao: Key = Key(gdk_sys::GDK_KEY_Thai_lekkao as u32);
    pub const Hangul: Key = Key(gdk_sys::GDK_KEY_Hangul as u32);
    pub const Hangul_Start: Key = Key(gdk_sys::GDK_KEY_Hangul_Start as u32);
    pub const Hangul_End: Key = Key(gdk_sys::GDK_KEY_Hangul_End as u32);
    pub const Hangul_Hanja: Key = Key(gdk_sys::GDK_KEY_Hangul_Hanja as u32);
    pub const Hangul_Jamo: Key = Key(gdk_sys::GDK_KEY_Hangul_Jamo as u32);
    pub const Hangul_Romaja: Key = Key(gdk_sys::GDK_KEY_Hangul_Romaja as u32);
    pub const Hangul_Codeinput: Key = Key(gdk_sys::GDK_KEY_Hangul_Codeinput as u32);
    pub const Hangul_Jeonja: Key = Key(gdk_sys::GDK_KEY_Hangul_Jeonja as u32);
    pub const Hangul_Banja: Key = Key(gdk_sys::GDK_KEY_Hangul_Banja as u32);
    pub const Hangul_PreHanja: Key = Key(gdk_sys::GDK_KEY_Hangul_PreHanja as u32);
    pub const Hangul_PostHanja: Key = Key(gdk_sys::GDK_KEY_Hangul_PostHanja as u32);
    pub const Hangul_SingleCandidate: Key = Key(gdk_sys::GDK_KEY_Hangul_SingleCandidate as u32);
    pub const Hangul_MultipleCandidate: Key = Key(gdk_sys::GDK_KEY_Hangul_MultipleCandidate as u32);
    pub const Hangul_PreviousCandidate: Key = Key(gdk_sys::GDK_KEY_Hangul_PreviousCandidate as u32);
    pub const Hangul_Special: Key = Key(gdk_sys::GDK_KEY_Hangul_Special as u32);
    pub const Hangul_switch: Key = Key(gdk_sys::GDK_KEY_Hangul_switch as u32);
    pub const Hangul_Kiyeog: Key = Key(gdk_sys::GDK_KEY_Hangul_Kiyeog as u32);
    pub const Hangul_SsangKiyeog: Key = Key(gdk_sys::GDK_KEY_Hangul_SsangKiyeog as u32);
    pub const Hangul_KiyeogSios: Key = Key(gdk_sys::GDK_KEY_Hangul_KiyeogSios as u32);
    pub const Hangul_Nieun: Key = Key(gdk_sys::GDK_KEY_Hangul_Nieun as u32);
    pub const Hangul_NieunJieuj: Key = Key(gdk_sys::GDK_KEY_Hangul_NieunJieuj as u32);
    pub const Hangul_NieunHieuh: Key = Key(gdk_sys::GDK_KEY_Hangul_NieunHieuh as u32);
    pub const Hangul_Dikeud: Key = Key(gdk_sys::GDK_KEY_Hangul_Dikeud as u32);
    pub const Hangul_SsangDikeud: Key = Key(gdk_sys::GDK_KEY_Hangul_SsangDikeud as u32);
    pub const Hangul_Rieul: Key = Key(gdk_sys::GDK_KEY_Hangul_Rieul as u32);
    pub const Hangul_RieulKiyeog: Key = Key(gdk_sys::GDK_KEY_Hangul_RieulKiyeog as u32);
    pub const Hangul_RieulMieum: Key = Key(gdk_sys::GDK_KEY_Hangul_RieulMieum as u32);
    pub const Hangul_RieulPieub: Key = Key(gdk_sys::GDK_KEY_Hangul_RieulPieub as u32);
    pub const Hangul_RieulSios: Key = Key(gdk_sys::GDK_KEY_Hangul_RieulSios as u32);
    pub const Hangul_RieulTieut: Key = Key(gdk_sys::GDK_KEY_Hangul_RieulTieut as u32);
    pub const Hangul_RieulPhieuf: Key = Key(gdk_sys::GDK_KEY_Hangul_RieulPhieuf as u32);
    pub const Hangul_RieulHieuh: Key = Key(gdk_sys::GDK_KEY_Hangul_RieulHieuh as u32);
    pub const Hangul_Mieum: Key = Key(gdk_sys::GDK_KEY_Hangul_Mieum as u32);
    pub const Hangul_Pieub: Key = Key(gdk_sys::GDK_KEY_Hangul_Pieub as u32);
    pub const Hangul_SsangPieub: Key = Key(gdk_sys::GDK_KEY_Hangul_SsangPieub as u32);
    pub const Hangul_PieubSios: Key = Key(gdk_sys::GDK_KEY_Hangul_PieubSios as u32);
    pub const Hangul_Sios: Key = Key(gdk_sys::GDK_KEY_Hangul_Sios as u32);
    pub const Hangul_SsangSios: Key = Key(gdk_sys::GDK_KEY_Hangul_SsangSios as u32);
    pub const Hangul_Ieung: Key = Key(gdk_sys::GDK_KEY_Hangul_Ieung as u32);
    pub const Hangul_Jieuj: Key = Key(gdk_sys::GDK_KEY_Hangul_Jieuj as u32);
    pub const Hangul_SsangJieuj: Key = Key(gdk_sys::GDK_KEY_Hangul_SsangJieuj as u32);
    pub const Hangul_Cieuc: Key = Key(gdk_sys::GDK_KEY_Hangul_Cieuc as u32);
    pub const Hangul_Khieuq: Key = Key(gdk_sys::GDK_KEY_Hangul_Khieuq as u32);
    pub const Hangul_Tieut: Key = Key(gdk_sys::GDK_KEY_Hangul_Tieut as u32);
    pub const Hangul_Phieuf: Key = Key(gdk_sys::GDK_KEY_Hangul_Phieuf as u32);
    pub const Hangul_Hieuh: Key = Key(gdk_sys::GDK_KEY_Hangul_Hieuh as u32);
    pub const Hangul_A: Key = Key(gdk_sys::GDK_KEY_Hangul_A as u32);
    pub const Hangul_AE: Key = Key(gdk_sys::GDK_KEY_Hangul_AE as u32);
    pub const Hangul_YA: Key = Key(gdk_sys::GDK_KEY_Hangul_YA as u32);
    pub const Hangul_YAE: Key = Key(gdk_sys::GDK_KEY_Hangul_YAE as u32);
    pub const Hangul_EO: Key = Key(gdk_sys::GDK_KEY_Hangul_EO as u32);
    pub const Hangul_E: Key = Key(gdk_sys::GDK_KEY_Hangul_E as u32);
    pub const Hangul_YEO: Key = Key(gdk_sys::GDK_KEY_Hangul_YEO as u32);
    pub const Hangul_YE: Key = Key(gdk_sys::GDK_KEY_Hangul_YE as u32);
    pub const Hangul_O: Key = Key(gdk_sys::GDK_KEY_Hangul_O as u32);
    pub const Hangul_WA: Key = Key(gdk_sys::GDK_KEY_Hangul_WA as u32);
    pub const Hangul_WAE: Key = Key(gdk_sys::GDK_KEY_Hangul_WAE as u32);
    pub const Hangul_OE: Key = Key(gdk_sys::GDK_KEY_Hangul_OE as u32);
    pub const Hangul_YO: Key = Key(gdk_sys::GDK_KEY_Hangul_YO as u32);
    pub const Hangul_U: Key = Key(gdk_sys::GDK_KEY_Hangul_U as u32);
    pub const Hangul_WEO: Key = Key(gdk_sys::GDK_KEY_Hangul_WEO as u32);
    pub const Hangul_WE: Key = Key(gdk_sys::GDK_KEY_Hangul_WE as u32);
    pub const Hangul_WI: Key = Key(gdk_sys::GDK_KEY_Hangul_WI as u32);
    pub const Hangul_YU: Key = Key(gdk_sys::GDK_KEY_Hangul_YU as u32);
    pub const Hangul_EU: Key = Key(gdk_sys::GDK_KEY_Hangul_EU as u32);
    pub const Hangul_YI: Key = Key(gdk_sys::GDK_KEY_Hangul_YI as u32);
    pub const Hangul_I: Key = Key(gdk_sys::GDK_KEY_Hangul_I as u32);
    pub const Hangul_J_Kiyeog: Key = Key(gdk_sys::GDK_KEY_Hangul_J_Kiyeog as u32);
    pub const Hangul_J_SsangKiyeog: Key = Key(gdk_sys::GDK_KEY_Hangul_J_SsangKiyeog as u32);
    pub const Hangul_J_KiyeogSios: Key = Key(gdk_sys::GDK_KEY_Hangul_J_KiyeogSios as u32);
    pub const Hangul_J_Nieun: Key = Key(gdk_sys::GDK_KEY_Hangul_J_Nieun as u32);
    pub const Hangul_J_NieunJieuj: Key = Key(gdk_sys::GDK_KEY_Hangul_J_NieunJieuj as u32);
    pub const Hangul_J_NieunHieuh: Key = Key(gdk_sys::GDK_KEY_Hangul_J_NieunHieuh as u32);
    pub const Hangul_J_Dikeud: Key = Key(gdk_sys::GDK_KEY_Hangul_J_Dikeud as u32);
    pub const Hangul_J_Rieul: Key = Key(gdk_sys::GDK_KEY_Hangul_J_Rieul as u32);
    pub const Hangul_J_RieulKiyeog: Key = Key(gdk_sys::GDK_KEY_Hangul_J_RieulKiyeog as u32);
    pub const Hangul_J_RieulMieum: Key = Key(gdk_sys::GDK_KEY_Hangul_J_RieulMieum as u32);
    pub const Hangul_J_RieulPieub: Key = Key(gdk_sys::GDK_KEY_Hangul_J_RieulPieub as u32);
    pub const Hangul_J_RieulSios: Key = Key(gdk_sys::GDK_KEY_Hangul_J_RieulSios as u32);
    pub const Hangul_J_RieulTieut: Key = Key(gdk_sys::GDK_KEY_Hangul_J_RieulTieut as u32);
    pub const Hangul_J_RieulPhieuf: Key = Key(gdk_sys::GDK_KEY_Hangul_J_RieulPhieuf as u32);
    pub const Hangul_J_RieulHieuh: Key = Key(gdk_sys::GDK_KEY_Hangul_J_RieulHieuh as u32);
    pub const Hangul_J_Mieum: Key = Key(gdk_sys::GDK_KEY_Hangul_J_Mieum as u32);
    pub const Hangul_J_Pieub: Key = Key(gdk_sys::GDK_KEY_Hangul_J_Pieub as u32);
    pub const Hangul_J_PieubSios: Key = Key(gdk_sys::GDK_KEY_Hangul_J_PieubSios as u32);
    pub const Hangul_J_Sios: Key = Key(gdk_sys::GDK_KEY_Hangul_J_Sios as u32);
    pub const Hangul_J_SsangSios: Key = Key(gdk_sys::GDK_KEY_Hangul_J_SsangSios as u32);
    pub const Hangul_J_Ieung: Key = Key(gdk_sys::GDK_KEY_Hangul_J_Ieung as u32);
    pub const Hangul_J_Jieuj: Key = Key(gdk_sys::GDK_KEY_Hangul_J_Jieuj as u32);
    pub const Hangul_J_Cieuc: Key = Key(gdk_sys::GDK_KEY_Hangul_J_Cieuc as u32);
    pub const Hangul_J_Khieuq: Key = Key(gdk_sys::GDK_KEY_Hangul_J_Khieuq as u32);
    pub const Hangul_J_Tieut: Key = Key(gdk_sys::GDK_KEY_Hangul_J_Tieut as u32);
    pub const Hangul_J_Phieuf: Key = Key(gdk_sys::GDK_KEY_Hangul_J_Phieuf as u32);
    pub const Hangul_J_Hieuh: Key = Key(gdk_sys::GDK_KEY_Hangul_J_Hieuh as u32);
    pub const Hangul_RieulYeorinHieuh: Key = Key(gdk_sys::GDK_KEY_Hangul_RieulYeorinHieuh as u32);
    pub const Hangul_SunkyeongeumMieum: Key = Key(gdk_sys::GDK_KEY_Hangul_SunkyeongeumMieum as u32);
    pub const Hangul_SunkyeongeumPieub: Key = Key(gdk_sys::GDK_KEY_Hangul_SunkyeongeumPieub as u32);
    pub const Hangul_PanSios: Key = Key(gdk_sys::GDK_KEY_Hangul_PanSios as u32);
    pub const Hangul_KkogjiDalrinIeung: Key = Key(gdk_sys::GDK_KEY_Hangul_KkogjiDalrinIeung as u32);
    pub const Hangul_SunkyeongeumPhieuf: Key =
        Key(gdk_sys::GDK_KEY_Hangul_SunkyeongeumPhieuf as u32);
    pub const Hangul_YeorinHieuh: Key = Key(gdk_sys::GDK_KEY_Hangul_YeorinHieuh as u32);
    pub const Hangul_AraeA: Key = Key(gdk_sys::GDK_KEY_Hangul_AraeA as u32);
    pub const Hangul_AraeAE: Key = Key(gdk_sys::GDK_KEY_Hangul_AraeAE as u32);
    pub const Hangul_J_PanSios: Key = Key(gdk_sys::GDK_KEY_Hangul_J_PanSios as u32);
    pub const Hangul_J_KkogjiDalrinIeung: Key =
        Key(gdk_sys::GDK_KEY_Hangul_J_KkogjiDalrinIeung as u32);
    pub const Hangul_J_YeorinHieuh: Key = Key(gdk_sys::GDK_KEY_Hangul_J_YeorinHieuh as u32);
    pub const Korean_Won: Key = Key(gdk_sys::GDK_KEY_Korean_Won as u32);
    pub const Armenian_ligature_ew: Key = Key(gdk_sys::GDK_KEY_Armenian_ligature_ew as u32);
    pub const Armenian_full_stop: Key = Key(gdk_sys::GDK_KEY_Armenian_full_stop as u32);
    pub const Armenian_verjaket: Key = Key(gdk_sys::GDK_KEY_Armenian_verjaket as u32);
    pub const Armenian_separation_mark: Key = Key(gdk_sys::GDK_KEY_Armenian_separation_mark as u32);
    pub const Armenian_but: Key = Key(gdk_sys::GDK_KEY_Armenian_but as u32);
    pub const Armenian_hyphen: Key = Key(gdk_sys::GDK_KEY_Armenian_hyphen as u32);
    pub const Armenian_yentamna: Key = Key(gdk_sys::GDK_KEY_Armenian_yentamna as u32);
    pub const Armenian_exclam: Key = Key(gdk_sys::GDK_KEY_Armenian_exclam as u32);
    pub const Armenian_amanak: Key = Key(gdk_sys::GDK_KEY_Armenian_amanak as u32);
    pub const Armenian_accent: Key = Key(gdk_sys::GDK_KEY_Armenian_accent as u32);
    pub const Armenian_shesht: Key = Key(gdk_sys::GDK_KEY_Armenian_shesht as u32);
    pub const Armenian_question: Key = Key(gdk_sys::GDK_KEY_Armenian_question as u32);
    pub const Armenian_paruyk: Key = Key(gdk_sys::GDK_KEY_Armenian_paruyk as u32);
    pub const Armenian_AYB: Key = Key(gdk_sys::GDK_KEY_Armenian_AYB as u32);
    pub const Armenian_ayb: Key = Key(gdk_sys::GDK_KEY_Armenian_ayb as u32);
    pub const Armenian_BEN: Key = Key(gdk_sys::GDK_KEY_Armenian_BEN as u32);
    pub const Armenian_ben: Key = Key(gdk_sys::GDK_KEY_Armenian_ben as u32);
    pub const Armenian_GIM: Key = Key(gdk_sys::GDK_KEY_Armenian_GIM as u32);
    pub const Armenian_gim: Key = Key(gdk_sys::GDK_KEY_Armenian_gim as u32);
    pub const Armenian_DA: Key = Key(gdk_sys::GDK_KEY_Armenian_DA as u32);
    pub const Armenian_da: Key = Key(gdk_sys::GDK_KEY_Armenian_da as u32);
    pub const Armenian_YECH: Key = Key(gdk_sys::GDK_KEY_Armenian_YECH as u32);
    pub const Armenian_yech: Key = Key(gdk_sys::GDK_KEY_Armenian_yech as u32);
    pub const Armenian_ZA: Key = Key(gdk_sys::GDK_KEY_Armenian_ZA as u32);
    pub const Armenian_za: Key = Key(gdk_sys::GDK_KEY_Armenian_za as u32);
    pub const Armenian_E: Key = Key(gdk_sys::GDK_KEY_Armenian_E as u32);
    pub const Armenian_e: Key = Key(gdk_sys::GDK_KEY_Armenian_e as u32);
    pub const Armenian_AT: Key = Key(gdk_sys::GDK_KEY_Armenian_AT as u32);
    pub const Armenian_at: Key = Key(gdk_sys::GDK_KEY_Armenian_at as u32);
    pub const Armenian_TO: Key = Key(gdk_sys::GDK_KEY_Armenian_TO as u32);
    pub const Armenian_to: Key = Key(gdk_sys::GDK_KEY_Armenian_to as u32);
    pub const Armenian_ZHE: Key = Key(gdk_sys::GDK_KEY_Armenian_ZHE as u32);
    pub const Armenian_zhe: Key = Key(gdk_sys::GDK_KEY_Armenian_zhe as u32);
    pub const Armenian_INI: Key = Key(gdk_sys::GDK_KEY_Armenian_INI as u32);
    pub const Armenian_ini: Key = Key(gdk_sys::GDK_KEY_Armenian_ini as u32);
    pub const Armenian_LYUN: Key = Key(gdk_sys::GDK_KEY_Armenian_LYUN as u32);
    pub const Armenian_lyun: Key = Key(gdk_sys::GDK_KEY_Armenian_lyun as u32);
    pub const Armenian_KHE: Key = Key(gdk_sys::GDK_KEY_Armenian_KHE as u32);
    pub const Armenian_khe: Key = Key(gdk_sys::GDK_KEY_Armenian_khe as u32);
    pub const Armenian_TSA: Key = Key(gdk_sys::GDK_KEY_Armenian_TSA as u32);
    pub const Armenian_tsa: Key = Key(gdk_sys::GDK_KEY_Armenian_tsa as u32);
    pub const Armenian_KEN: Key = Key(gdk_sys::GDK_KEY_Armenian_KEN as u32);
    pub const Armenian_ken: Key = Key(gdk_sys::GDK_KEY_Armenian_ken as u32);
    pub const Armenian_HO: Key = Key(gdk_sys::GDK_KEY_Armenian_HO as u32);
    pub const Armenian_ho: Key = Key(gdk_sys::GDK_KEY_Armenian_ho as u32);
    pub const Armenian_DZA: Key = Key(gdk_sys::GDK_KEY_Armenian_DZA as u32);
    pub const Armenian_dza: Key = Key(gdk_sys::GDK_KEY_Armenian_dza as u32);
    pub const Armenian_GHAT: Key = Key(gdk_sys::GDK_KEY_Armenian_GHAT as u32);
    pub const Armenian_ghat: Key = Key(gdk_sys::GDK_KEY_Armenian_ghat as u32);
    pub const Armenian_TCHE: Key = Key(gdk_sys::GDK_KEY_Armenian_TCHE as u32);
    pub const Armenian_tche: Key = Key(gdk_sys::GDK_KEY_Armenian_tche as u32);
    pub const Armenian_MEN: Key = Key(gdk_sys::GDK_KEY_Armenian_MEN as u32);
    pub const Armenian_men: Key = Key(gdk_sys::GDK_KEY_Armenian_men as u32);
    pub const Armenian_HI: Key = Key(gdk_sys::GDK_KEY_Armenian_HI as u32);
    pub const Armenian_hi: Key = Key(gdk_sys::GDK_KEY_Armenian_hi as u32);
    pub const Armenian_NU: Key = Key(gdk_sys::GDK_KEY_Armenian_NU as u32);
    pub const Armenian_nu: Key = Key(gdk_sys::GDK_KEY_Armenian_nu as u32);
    pub const Armenian_SHA: Key = Key(gdk_sys::GDK_KEY_Armenian_SHA as u32);
    pub const Armenian_sha: Key = Key(gdk_sys::GDK_KEY_Armenian_sha as u32);
    pub const Armenian_VO: Key = Key(gdk_sys::GDK_KEY_Armenian_VO as u32);
    pub const Armenian_vo: Key = Key(gdk_sys::GDK_KEY_Armenian_vo as u32);
    pub const Armenian_CHA: Key = Key(gdk_sys::GDK_KEY_Armenian_CHA as u32);
    pub const Armenian_cha: Key = Key(gdk_sys::GDK_KEY_Armenian_cha as u32);
    pub const Armenian_PE: Key = Key(gdk_sys::GDK_KEY_Armenian_PE as u32);
    pub const Armenian_pe: Key = Key(gdk_sys::GDK_KEY_Armenian_pe as u32);
    pub const Armenian_JE: Key = Key(gdk_sys::GDK_KEY_Armenian_JE as u32);
    pub const Armenian_je: Key = Key(gdk_sys::GDK_KEY_Armenian_je as u32);
    pub const Armenian_RA: Key = Key(gdk_sys::GDK_KEY_Armenian_RA as u32);
    pub const Armenian_ra: Key = Key(gdk_sys::GDK_KEY_Armenian_ra as u32);
    pub const Armenian_SE: Key = Key(gdk_sys::GDK_KEY_Armenian_SE as u32);
    pub const Armenian_se: Key = Key(gdk_sys::GDK_KEY_Armenian_se as u32);
    pub const Armenian_VEV: Key = Key(gdk_sys::GDK_KEY_Armenian_VEV as u32);
    pub const Armenian_vev: Key = Key(gdk_sys::GDK_KEY_Armenian_vev as u32);
    pub const Armenian_TYUN: Key = Key(gdk_sys::GDK_KEY_Armenian_TYUN as u32);
    pub const Armenian_tyun: Key = Key(gdk_sys::GDK_KEY_Armenian_tyun as u32);
    pub const Armenian_RE: Key = Key(gdk_sys::GDK_KEY_Armenian_RE as u32);
    pub const Armenian_re: Key = Key(gdk_sys::GDK_KEY_Armenian_re as u32);
    pub const Armenian_TSO: Key = Key(gdk_sys::GDK_KEY_Armenian_TSO as u32);
    pub const Armenian_tso: Key = Key(gdk_sys::GDK_KEY_Armenian_tso as u32);
    pub const Armenian_VYUN: Key = Key(gdk_sys::GDK_KEY_Armenian_VYUN as u32);
    pub const Armenian_vyun: Key = Key(gdk_sys::GDK_KEY_Armenian_vyun as u32);
    pub const Armenian_PYUR: Key = Key(gdk_sys::GDK_KEY_Armenian_PYUR as u32);
    pub const Armenian_pyur: Key = Key(gdk_sys::GDK_KEY_Armenian_pyur as u32);
    pub const Armenian_KE: Key = Key(gdk_sys::GDK_KEY_Armenian_KE as u32);
    pub const Armenian_ke: Key = Key(gdk_sys::GDK_KEY_Armenian_ke as u32);
    pub const Armenian_O: Key = Key(gdk_sys::GDK_KEY_Armenian_O as u32);
    pub const Armenian_o: Key = Key(gdk_sys::GDK_KEY_Armenian_o as u32);
    pub const Armenian_FE: Key = Key(gdk_sys::GDK_KEY_Armenian_FE as u32);
    pub const Armenian_fe: Key = Key(gdk_sys::GDK_KEY_Armenian_fe as u32);
    pub const Armenian_apostrophe: Key = Key(gdk_sys::GDK_KEY_Armenian_apostrophe as u32);
    pub const Georgian_an: Key = Key(gdk_sys::GDK_KEY_Georgian_an as u32);
    pub const Georgian_ban: Key = Key(gdk_sys::GDK_KEY_Georgian_ban as u32);
    pub const Georgian_gan: Key = Key(gdk_sys::GDK_KEY_Georgian_gan as u32);
    pub const Georgian_don: Key = Key(gdk_sys::GDK_KEY_Georgian_don as u32);
    pub const Georgian_en: Key = Key(gdk_sys::GDK_KEY_Georgian_en as u32);
    pub const Georgian_vin: Key = Key(gdk_sys::GDK_KEY_Georgian_vin as u32);
    pub const Georgian_zen: Key = Key(gdk_sys::GDK_KEY_Georgian_zen as u32);
    pub const Georgian_tan: Key = Key(gdk_sys::GDK_KEY_Georgian_tan as u32);
    pub const Georgian_in: Key = Key(gdk_sys::GDK_KEY_Georgian_in as u32);
    pub const Georgian_kan: Key = Key(gdk_sys::GDK_KEY_Georgian_kan as u32);
    pub const Georgian_las: Key = Key(gdk_sys::GDK_KEY_Georgian_las as u32);
    pub const Georgian_man: Key = Key(gdk_sys::GDK_KEY_Georgian_man as u32);
    pub const Georgian_nar: Key = Key(gdk_sys::GDK_KEY_Georgian_nar as u32);
    pub const Georgian_on: Key = Key(gdk_sys::GDK_KEY_Georgian_on as u32);
    pub const Georgian_par: Key = Key(gdk_sys::GDK_KEY_Georgian_par as u32);
    pub const Georgian_zhar: Key = Key(gdk_sys::GDK_KEY_Georgian_zhar as u32);
    pub const Georgian_rae: Key = Key(gdk_sys::GDK_KEY_Georgian_rae as u32);
    pub const Georgian_san: Key = Key(gdk_sys::GDK_KEY_Georgian_san as u32);
    pub const Georgian_tar: Key = Key(gdk_sys::GDK_KEY_Georgian_tar as u32);
    pub const Georgian_un: Key = Key(gdk_sys::GDK_KEY_Georgian_un as u32);
    pub const Georgian_phar: Key = Key(gdk_sys::GDK_KEY_Georgian_phar as u32);
    pub const Georgian_khar: Key = Key(gdk_sys::GDK_KEY_Georgian_khar as u32);
    pub const Georgian_ghan: Key = Key(gdk_sys::GDK_KEY_Georgian_ghan as u32);
    pub const Georgian_qar: Key = Key(gdk_sys::GDK_KEY_Georgian_qar as u32);
    pub const Georgian_shin: Key = Key(gdk_sys::GDK_KEY_Georgian_shin as u32);
    pub const Georgian_chin: Key = Key(gdk_sys::GDK_KEY_Georgian_chin as u32);
    pub const Georgian_can: Key = Key(gdk_sys::GDK_KEY_Georgian_can as u32);
    pub const Georgian_jil: Key = Key(gdk_sys::GDK_KEY_Georgian_jil as u32);
    pub const Georgian_cil: Key = Key(gdk_sys::GDK_KEY_Georgian_cil as u32);
    pub const Georgian_char: Key = Key(gdk_sys::GDK_KEY_Georgian_char as u32);
    pub const Georgian_xan: Key = Key(gdk_sys::GDK_KEY_Georgian_xan as u32);
    pub const Georgian_jhan: Key = Key(gdk_sys::GDK_KEY_Georgian_jhan as u32);
    pub const Georgian_hae: Key = Key(gdk_sys::GDK_KEY_Georgian_hae as u32);
    pub const Georgian_he: Key = Key(gdk_sys::GDK_KEY_Georgian_he as u32);
    pub const Georgian_hie: Key = Key(gdk_sys::GDK_KEY_Georgian_hie as u32);
    pub const Georgian_we: Key = Key(gdk_sys::GDK_KEY_Georgian_we as u32);
    pub const Georgian_har: Key = Key(gdk_sys::GDK_KEY_Georgian_har as u32);
    pub const Georgian_hoe: Key = Key(gdk_sys::GDK_KEY_Georgian_hoe as u32);
    pub const Georgian_fi: Key = Key(gdk_sys::GDK_KEY_Georgian_fi as u32);
    pub const Xabovedot: Key = Key(gdk_sys::GDK_KEY_Xabovedot as u32);
    pub const Ibreve: Key = Key(gdk_sys::GDK_KEY_Ibreve as u32);
    pub const Zstroke: Key = Key(gdk_sys::GDK_KEY_Zstroke as u32);
    pub const Gcaron: Key = Key(gdk_sys::GDK_KEY_Gcaron as u32);
    pub const Ocaron: Key = Key(gdk_sys::GDK_KEY_Ocaron as u32);
    pub const Obarred: Key = Key(gdk_sys::GDK_KEY_Obarred as u32);
    pub const xabovedot: Key = Key(gdk_sys::GDK_KEY_xabovedot as u32);
    pub const ibreve: Key = Key(gdk_sys::GDK_KEY_ibreve as u32);
    pub const zstroke: Key = Key(gdk_sys::GDK_KEY_zstroke as u32);
    pub const gcaron: Key = Key(gdk_sys::GDK_KEY_gcaron as u32);
    pub const ocaron: Key = Key(gdk_sys::GDK_KEY_ocaron as u32);
    pub const obarred: Key = Key(gdk_sys::GDK_KEY_obarred as u32);
    pub const SCHWA: Key = Key(gdk_sys::GDK_KEY_SCHWA as u32);
    pub const schwa: Key = Key(gdk_sys::GDK_KEY_schwa as u32);
    pub const EZH: Key = Key(gdk_sys::GDK_KEY_EZH as u32);
    pub const ezh: Key = Key(gdk_sys::GDK_KEY_ezh as u32);
    pub const Lbelowdot: Key = Key(gdk_sys::GDK_KEY_Lbelowdot as u32);
    pub const lbelowdot: Key = Key(gdk_sys::GDK_KEY_lbelowdot as u32);
    pub const Abelowdot: Key = Key(gdk_sys::GDK_KEY_Abelowdot as u32);
    pub const abelowdot: Key = Key(gdk_sys::GDK_KEY_abelowdot as u32);
    pub const Ahook: Key = Key(gdk_sys::GDK_KEY_Ahook as u32);
    pub const ahook: Key = Key(gdk_sys::GDK_KEY_ahook as u32);
    pub const Acircumflexacute: Key = Key(gdk_sys::GDK_KEY_Acircumflexacute as u32);
    pub const acircumflexacute: Key = Key(gdk_sys::GDK_KEY_acircumflexacute as u32);
    pub const Acircumflexgrave: Key = Key(gdk_sys::GDK_KEY_Acircumflexgrave as u32);
    pub const acircumflexgrave: Key = Key(gdk_sys::GDK_KEY_acircumflexgrave as u32);
    pub const Acircumflexhook: Key = Key(gdk_sys::GDK_KEY_Acircumflexhook as u32);
    pub const acircumflexhook: Key = Key(gdk_sys::GDK_KEY_acircumflexhook as u32);
    pub const Acircumflextilde: Key = Key(gdk_sys::GDK_KEY_Acircumflextilde as u32);
    pub const acircumflextilde: Key = Key(gdk_sys::GDK_KEY_acircumflextilde as u32);
    pub const Acircumflexbelowdot: Key = Key(gdk_sys::GDK_KEY_Acircumflexbelowdot as u32);
    pub const acircumflexbelowdot: Key = Key(gdk_sys::GDK_KEY_acircumflexbelowdot as u32);
    pub const Abreveacute: Key = Key(gdk_sys::GDK_KEY_Abreveacute as u32);
    pub const abreveacute: Key = Key(gdk_sys::GDK_KEY_abreveacute as u32);
    pub const Abrevegrave: Key = Key(gdk_sys::GDK_KEY_Abrevegrave as u32);
    pub const abrevegrave: Key = Key(gdk_sys::GDK_KEY_abrevegrave as u32);
    pub const Abrevehook: Key = Key(gdk_sys::GDK_KEY_Abrevehook as u32);
    pub const abrevehook: Key = Key(gdk_sys::GDK_KEY_abrevehook as u32);
    pub const Abrevetilde: Key = Key(gdk_sys::GDK_KEY_Abrevetilde as u32);
    pub const abrevetilde: Key = Key(gdk_sys::GDK_KEY_abrevetilde as u32);
    pub const Abrevebelowdot: Key = Key(gdk_sys::GDK_KEY_Abrevebelowdot as u32);
    pub const abrevebelowdot: Key = Key(gdk_sys::GDK_KEY_abrevebelowdot as u32);
    pub const Ebelowdot: Key = Key(gdk_sys::GDK_KEY_Ebelowdot as u32);
    pub const ebelowdot: Key = Key(gdk_sys::GDK_KEY_ebelowdot as u32);
    pub const Ehook: Key = Key(gdk_sys::GDK_KEY_Ehook as u32);
    pub const ehook: Key = Key(gdk_sys::GDK_KEY_ehook as u32);
    pub const Etilde: Key = Key(gdk_sys::GDK_KEY_Etilde as u32);
    pub const etilde: Key = Key(gdk_sys::GDK_KEY_etilde as u32);
    pub const Ecircumflexacute: Key = Key(gdk_sys::GDK_KEY_Ecircumflexacute as u32);
    pub const ecircumflexacute: Key = Key(gdk_sys::GDK_KEY_ecircumflexacute as u32);
    pub const Ecircumflexgrave: Key = Key(gdk_sys::GDK_KEY_Ecircumflexgrave as u32);
    pub const ecircumflexgrave: Key = Key(gdk_sys::GDK_KEY_ecircumflexgrave as u32);
    pub const Ecircumflexhook: Key = Key(gdk_sys::GDK_KEY_Ecircumflexhook as u32);
    pub const ecircumflexhook: Key = Key(gdk_sys::GDK_KEY_ecircumflexhook as u32);
    pub const Ecircumflextilde: Key = Key(gdk_sys::GDK_KEY_Ecircumflextilde as u32);
    pub const ecircumflextilde: Key = Key(gdk_sys::GDK_KEY_ecircumflextilde as u32);
    pub const Ecircumflexbelowdot: Key = Key(gdk_sys::GDK_KEY_Ecircumflexbelowdot as u32);
    pub const ecircumflexbelowdot: Key = Key(gdk_sys::GDK_KEY_ecircumflexbelowdot as u32);
    pub const Ihook: Key = Key(gdk_sys::GDK_KEY_Ihook as u32);
    pub const ihook: Key = Key(gdk_sys::GDK_KEY_ihook as u32);
    pub const Ibelowdot: Key = Key(gdk_sys::GDK_KEY_Ibelowdot as u32);
    pub const ibelowdot: Key = Key(gdk_sys::GDK_KEY_ibelowdot as u32);
    pub const Obelowdot: Key = Key(gdk_sys::GDK_KEY_Obelowdot as u32);
    pub const obelowdot: Key = Key(gdk_sys::GDK_KEY_obelowdot as u32);
    pub const Ohook: Key = Key(gdk_sys::GDK_KEY_Ohook as u32);
    pub const ohook: Key = Key(gdk_sys::GDK_KEY_ohook as u32);
    pub const Ocircumflexacute: Key = Key(gdk_sys::GDK_KEY_Ocircumflexacute as u32);
    pub const ocircumflexacute: Key = Key(gdk_sys::GDK_KEY_ocircumflexacute as u32);
    pub const Ocircumflexgrave: Key = Key(gdk_sys::GDK_KEY_Ocircumflexgrave as u32);
    pub const ocircumflexgrave: Key = Key(gdk_sys::GDK_KEY_ocircumflexgrave as u32);
    pub const Ocircumflexhook: Key = Key(gdk_sys::GDK_KEY_Ocircumflexhook as u32);
    pub const ocircumflexhook: Key = Key(gdk_sys::GDK_KEY_ocircumflexhook as u32);
    pub const Ocircumflextilde: Key = Key(gdk_sys::GDK_KEY_Ocircumflextilde as u32);
    pub const ocircumflextilde: Key = Key(gdk_sys::GDK_KEY_ocircumflextilde as u32);
    pub const Ocircumflexbelowdot: Key = Key(gdk_sys::GDK_KEY_Ocircumflexbelowdot as u32);
    pub const ocircumflexbelowdot: Key = Key(gdk_sys::GDK_KEY_ocircumflexbelowdot as u32);
    pub const Ohornacute: Key = Key(gdk_sys::GDK_KEY_Ohornacute as u32);
    pub const ohornacute: Key = Key(gdk_sys::GDK_KEY_ohornacute as u32);
    pub const Ohorngrave: Key = Key(gdk_sys::GDK_KEY_Ohorngrave as u32);
    pub const ohorngrave: Key = Key(gdk_sys::GDK_KEY_ohorngrave as u32);
    pub const Ohornhook: Key = Key(gdk_sys::GDK_KEY_Ohornhook as u32);
    pub const ohornhook: Key = Key(gdk_sys::GDK_KEY_ohornhook as u32);
    pub const Ohorntilde: Key = Key(gdk_sys::GDK_KEY_Ohorntilde as u32);
    pub const ohorntilde: Key = Key(gdk_sys::GDK_KEY_ohorntilde as u32);
    pub const Ohornbelowdot: Key = Key(gdk_sys::GDK_KEY_Ohornbelowdot as u32);
    pub const ohornbelowdot: Key = Key(gdk_sys::GDK_KEY_ohornbelowdot as u32);
    pub const Ubelowdot: Key = Key(gdk_sys::GDK_KEY_Ubelowdot as u32);
    pub const ubelowdot: Key = Key(gdk_sys::GDK_KEY_ubelowdot as u32);
    pub const Uhook: Key = Key(gdk_sys::GDK_KEY_Uhook as u32);
    pub const uhook: Key = Key(gdk_sys::GDK_KEY_uhook as u32);
    pub const Uhornacute: Key = Key(gdk_sys::GDK_KEY_Uhornacute as u32);
    pub const uhornacute: Key = Key(gdk_sys::GDK_KEY_uhornacute as u32);
    pub const Uhorngrave: Key = Key(gdk_sys::GDK_KEY_Uhorngrave as u32);
    pub const uhorngrave: Key = Key(gdk_sys::GDK_KEY_uhorngrave as u32);
    pub const Uhornhook: Key = Key(gdk_sys::GDK_KEY_Uhornhook as u32);
    pub const uhornhook: Key = Key(gdk_sys::GDK_KEY_uhornhook as u32);
    pub const Uhorntilde: Key = Key(gdk_sys::GDK_KEY_Uhorntilde as u32);
    pub const uhorntilde: Key = Key(gdk_sys::GDK_KEY_uhorntilde as u32);
    pub const Uhornbelowdot: Key = Key(gdk_sys::GDK_KEY_Uhornbelowdot as u32);
    pub const uhornbelowdot: Key = Key(gdk_sys::GDK_KEY_uhornbelowdot as u32);
    pub const Ybelowdot: Key = Key(gdk_sys::GDK_KEY_Ybelowdot as u32);
    pub const ybelowdot: Key = Key(gdk_sys::GDK_KEY_ybelowdot as u32);
    pub const Yhook: Key = Key(gdk_sys::GDK_KEY_Yhook as u32);
    pub const yhook: Key = Key(gdk_sys::GDK_KEY_yhook as u32);
    pub const Ytilde: Key = Key(gdk_sys::GDK_KEY_Ytilde as u32);
    pub const ytilde: Key = Key(gdk_sys::GDK_KEY_ytilde as u32);
    pub const Ohorn: Key = Key(gdk_sys::GDK_KEY_Ohorn as u32);
    pub const ohorn: Key = Key(gdk_sys::GDK_KEY_ohorn as u32);
    pub const Uhorn: Key = Key(gdk_sys::GDK_KEY_Uhorn as u32);
    pub const uhorn: Key = Key(gdk_sys::GDK_KEY_uhorn as u32);
    pub const EcuSign: Key = Key(gdk_sys::GDK_KEY_EcuSign as u32);
    pub const ColonSign: Key = Key(gdk_sys::GDK_KEY_ColonSign as u32);
    pub const CruzeiroSign: Key = Key(gdk_sys::GDK_KEY_CruzeiroSign as u32);
    pub const FFrancSign: Key = Key(gdk_sys::GDK_KEY_FFrancSign as u32);
    pub const LiraSign: Key = Key(gdk_sys::GDK_KEY_LiraSign as u32);
    pub const MillSign: Key = Key(gdk_sys::GDK_KEY_MillSign as u32);
    pub const NairaSign: Key = Key(gdk_sys::GDK_KEY_NairaSign as u32);
    pub const PesetaSign: Key = Key(gdk_sys::GDK_KEY_PesetaSign as u32);
    pub const RupeeSign: Key = Key(gdk_sys::GDK_KEY_RupeeSign as u32);
    pub const WonSign: Key = Key(gdk_sys::GDK_KEY_WonSign as u32);
    pub const NewSheqelSign: Key = Key(gdk_sys::GDK_KEY_NewSheqelSign as u32);
    pub const DongSign: Key = Key(gdk_sys::GDK_KEY_DongSign as u32);
    pub const EuroSign: Key = Key(gdk_sys::GDK_KEY_EuroSign as u32);
    pub const zerosuperior: Key = Key(gdk_sys::GDK_KEY_zerosuperior as u32);
    pub const foursuperior: Key = Key(gdk_sys::GDK_KEY_foursuperior as u32);
    pub const fivesuperior: Key = Key(gdk_sys::GDK_KEY_fivesuperior as u32);
    pub const sixsuperior: Key = Key(gdk_sys::GDK_KEY_sixsuperior as u32);
    pub const sevensuperior: Key = Key(gdk_sys::GDK_KEY_sevensuperior as u32);
    pub const eightsuperior: Key = Key(gdk_sys::GDK_KEY_eightsuperior as u32);
    pub const ninesuperior: Key = Key(gdk_sys::GDK_KEY_ninesuperior as u32);
    pub const zerosubscript: Key = Key(gdk_sys::GDK_KEY_zerosubscript as u32);
    pub const onesubscript: Key = Key(gdk_sys::GDK_KEY_onesubscript as u32);
    pub const twosubscript: Key = Key(gdk_sys::GDK_KEY_twosubscript as u32);
    pub const threesubscript: Key = Key(gdk_sys::GDK_KEY_threesubscript as u32);
    pub const foursubscript: Key = Key(gdk_sys::GDK_KEY_foursubscript as u32);
    pub const fivesubscript: Key = Key(gdk_sys::GDK_KEY_fivesubscript as u32);
    pub const sixsubscript: Key = Key(gdk_sys::GDK_KEY_sixsubscript as u32);
    pub const sevensubscript: Key = Key(gdk_sys::GDK_KEY_sevensubscript as u32);
    pub const eightsubscript: Key = Key(gdk_sys::GDK_KEY_eightsubscript as u32);
    pub const ninesubscript: Key = Key(gdk_sys::GDK_KEY_ninesubscript as u32);
    pub const partdifferential: Key = Key(gdk_sys::GDK_KEY_partdifferential as u32);
    pub const emptyset: Key = Key(gdk_sys::GDK_KEY_emptyset as u32);
    pub const elementof: Key = Key(gdk_sys::GDK_KEY_elementof as u32);
    pub const notelementof: Key = Key(gdk_sys::GDK_KEY_notelementof as u32);
    pub const containsas: Key = Key(gdk_sys::GDK_KEY_containsas as u32);
    pub const squareroot: Key = Key(gdk_sys::GDK_KEY_squareroot as u32);
    pub const cuberoot: Key = Key(gdk_sys::GDK_KEY_cuberoot as u32);
    pub const fourthroot: Key = Key(gdk_sys::GDK_KEY_fourthroot as u32);
    pub const dintegral: Key = Key(gdk_sys::GDK_KEY_dintegral as u32);
    pub const tintegral: Key = Key(gdk_sys::GDK_KEY_tintegral as u32);
    pub const because: Key = Key(gdk_sys::GDK_KEY_because as u32);
    pub const approxeq: Key = Key(gdk_sys::GDK_KEY_approxeq as u32);
    pub const notapproxeq: Key = Key(gdk_sys::GDK_KEY_notapproxeq as u32);
    pub const notidentical: Key = Key(gdk_sys::GDK_KEY_notidentical as u32);
    pub const stricteq: Key = Key(gdk_sys::GDK_KEY_stricteq as u32);
    pub const braille_dot_1: Key = Key(gdk_sys::GDK_KEY_braille_dot_1 as u32);
    pub const braille_dot_2: Key = Key(gdk_sys::GDK_KEY_braille_dot_2 as u32);
    pub const braille_dot_3: Key = Key(gdk_sys::GDK_KEY_braille_dot_3 as u32);
    pub const braille_dot_4: Key = Key(gdk_sys::GDK_KEY_braille_dot_4 as u32);
    pub const braille_dot_5: Key = Key(gdk_sys::GDK_KEY_braille_dot_5 as u32);
    pub const braille_dot_6: Key = Key(gdk_sys::GDK_KEY_braille_dot_6 as u32);
    pub const braille_dot_7: Key = Key(gdk_sys::GDK_KEY_braille_dot_7 as u32);
    pub const braille_dot_8: Key = Key(gdk_sys::GDK_KEY_braille_dot_8 as u32);
    pub const braille_dot_9: Key = Key(gdk_sys::GDK_KEY_braille_dot_9 as u32);
    pub const braille_dot_10: Key = Key(gdk_sys::GDK_KEY_braille_dot_10 as u32);
    pub const braille_blank: Key = Key(gdk_sys::GDK_KEY_braille_blank as u32);
    pub const braille_dots_1: Key = Key(gdk_sys::GDK_KEY_braille_dots_1 as u32);
    pub const braille_dots_2: Key = Key(gdk_sys::GDK_KEY_braille_dots_2 as u32);
    pub const braille_dots_12: Key = Key(gdk_sys::GDK_KEY_braille_dots_12 as u32);
    pub const braille_dots_3: Key = Key(gdk_sys::GDK_KEY_braille_dots_3 as u32);
    pub const braille_dots_13: Key = Key(gdk_sys::GDK_KEY_braille_dots_13 as u32);
    pub const braille_dots_23: Key = Key(gdk_sys::GDK_KEY_braille_dots_23 as u32);
    pub const braille_dots_123: Key = Key(gdk_sys::GDK_KEY_braille_dots_123 as u32);
    pub const braille_dots_4: Key = Key(gdk_sys::GDK_KEY_braille_dots_4 as u32);
    pub const braille_dots_14: Key = Key(gdk_sys::GDK_KEY_braille_dots_14 as u32);
    pub const braille_dots_24: Key = Key(gdk_sys::GDK_KEY_braille_dots_24 as u32);
    pub const braille_dots_124: Key = Key(gdk_sys::GDK_KEY_braille_dots_124 as u32);
    pub const braille_dots_34: Key = Key(gdk_sys::GDK_KEY_braille_dots_34 as u32);
    pub const braille_dots_134: Key = Key(gdk_sys::GDK_KEY_braille_dots_134 as u32);
    pub const braille_dots_234: Key = Key(gdk_sys::GDK_KEY_braille_dots_234 as u32);
    pub const braille_dots_1234: Key = Key(gdk_sys::GDK_KEY_braille_dots_1234 as u32);
    pub const braille_dots_5: Key = Key(gdk_sys::GDK_KEY_braille_dots_5 as u32);
    pub const braille_dots_15: Key = Key(gdk_sys::GDK_KEY_braille_dots_15 as u32);
    pub const braille_dots_25: Key = Key(gdk_sys::GDK_KEY_braille_dots_25 as u32);
    pub const braille_dots_125: Key = Key(gdk_sys::GDK_KEY_braille_dots_125 as u32);
    pub const braille_dots_35: Key = Key(gdk_sys::GDK_KEY_braille_dots_35 as u32);
    pub const braille_dots_135: Key = Key(gdk_sys::GDK_KEY_braille_dots_135 as u32);
    pub const braille_dots_235: Key = Key(gdk_sys::GDK_KEY_braille_dots_235 as u32);
    pub const braille_dots_1235: Key = Key(gdk_sys::GDK_KEY_braille_dots_1235 as u32);
    pub const braille_dots_45: Key = Key(gdk_sys::GDK_KEY_braille_dots_45 as u32);
    pub const braille_dots_145: Key = Key(gdk_sys::GDK_KEY_braille_dots_145 as u32);
    pub const braille_dots_245: Key = Key(gdk_sys::GDK_KEY_braille_dots_245 as u32);
    pub const braille_dots_1245: Key = Key(gdk_sys::GDK_KEY_braille_dots_1245 as u32);
    pub const braille_dots_345: Key = Key(gdk_sys::GDK_KEY_braille_dots_345 as u32);
    pub const braille_dots_1345: Key = Key(gdk_sys::GDK_KEY_braille_dots_1345 as u32);
    pub const braille_dots_2345: Key = Key(gdk_sys::GDK_KEY_braille_dots_2345 as u32);
    pub const braille_dots_12345: Key = Key(gdk_sys::GDK_KEY_braille_dots_12345 as u32);
    pub const braille_dots_6: Key = Key(gdk_sys::GDK_KEY_braille_dots_6 as u32);
    pub const braille_dots_16: Key = Key(gdk_sys::GDK_KEY_braille_dots_16 as u32);
    pub const braille_dots_26: Key = Key(gdk_sys::GDK_KEY_braille_dots_26 as u32);
    pub const braille_dots_126: Key = Key(gdk_sys::GDK_KEY_braille_dots_126 as u32);
    pub const braille_dots_36: Key = Key(gdk_sys::GDK_KEY_braille_dots_36 as u32);
    pub const braille_dots_136: Key = Key(gdk_sys::GDK_KEY_braille_dots_136 as u32);
    pub const braille_dots_236: Key = Key(gdk_sys::GDK_KEY_braille_dots_236 as u32);
    pub const braille_dots_1236: Key = Key(gdk_sys::GDK_KEY_braille_dots_1236 as u32);
    pub const braille_dots_46: Key = Key(gdk_sys::GDK_KEY_braille_dots_46 as u32);
    pub const braille_dots_146: Key = Key(gdk_sys::GDK_KEY_braille_dots_146 as u32);
    pub const braille_dots_246: Key = Key(gdk_sys::GDK_KEY_braille_dots_246 as u32);
    pub const braille_dots_1246: Key = Key(gdk_sys::GDK_KEY_braille_dots_1246 as u32);
    pub const braille_dots_346: Key = Key(gdk_sys::GDK_KEY_braille_dots_346 as u32);
    pub const braille_dots_1346: Key = Key(gdk_sys::GDK_KEY_braille_dots_1346 as u32);
    pub const braille_dots_2346: Key = Key(gdk_sys::GDK_KEY_braille_dots_2346 as u32);
    pub const braille_dots_12346: Key = Key(gdk_sys::GDK_KEY_braille_dots_12346 as u32);
    pub const braille_dots_56: Key = Key(gdk_sys::GDK_KEY_braille_dots_56 as u32);
    pub const braille_dots_156: Key = Key(gdk_sys::GDK_KEY_braille_dots_156 as u32);
    pub const braille_dots_256: Key = Key(gdk_sys::GDK_KEY_braille_dots_256 as u32);
    pub const braille_dots_1256: Key = Key(gdk_sys::GDK_KEY_braille_dots_1256 as u32);
    pub const braille_dots_356: Key = Key(gdk_sys::GDK_KEY_braille_dots_356 as u32);
    pub const braille_dots_1356: Key = Key(gdk_sys::GDK_KEY_braille_dots_1356 as u32);
    pub const braille_dots_2356: Key = Key(gdk_sys::GDK_KEY_braille_dots_2356 as u32);
    pub const braille_dots_12356: Key = Key(gdk_sys::GDK_KEY_braille_dots_12356 as u32);
    pub const braille_dots_456: Key = Key(gdk_sys::GDK_KEY_braille_dots_456 as u32);
    pub const braille_dots_1456: Key = Key(gdk_sys::GDK_KEY_braille_dots_1456 as u32);
    pub const braille_dots_2456: Key = Key(gdk_sys::GDK_KEY_braille_dots_2456 as u32);
    pub const braille_dots_12456: Key = Key(gdk_sys::GDK_KEY_braille_dots_12456 as u32);
    pub const braille_dots_3456: Key = Key(gdk_sys::GDK_KEY_braille_dots_3456 as u32);
    pub const braille_dots_13456: Key = Key(gdk_sys::GDK_KEY_braille_dots_13456 as u32);
    pub const braille_dots_23456: Key = Key(gdk_sys::GDK_KEY_braille_dots_23456 as u32);
    pub const braille_dots_123456: Key = Key(gdk_sys::GDK_KEY_braille_dots_123456 as u32);
    pub const braille_dots_7: Key = Key(gdk_sys::GDK_KEY_braille_dots_7 as u32);
    pub const braille_dots_17: Key = Key(gdk_sys::GDK_KEY_braille_dots_17 as u32);
    pub const braille_dots_27: Key = Key(gdk_sys::GDK_KEY_braille_dots_27 as u32);
    pub const braille_dots_127: Key = Key(gdk_sys::GDK_KEY_braille_dots_127 as u32);
    pub const braille_dots_37: Key = Key(gdk_sys::GDK_KEY_braille_dots_37 as u32);
    pub const braille_dots_137: Key = Key(gdk_sys::GDK_KEY_braille_dots_137 as u32);
    pub const braille_dots_237: Key = Key(gdk_sys::GDK_KEY_braille_dots_237 as u32);
    pub const braille_dots_1237: Key = Key(gdk_sys::GDK_KEY_braille_dots_1237 as u32);
    pub const braille_dots_47: Key = Key(gdk_sys::GDK_KEY_braille_dots_47 as u32);
    pub const braille_dots_147: Key = Key(gdk_sys::GDK_KEY_braille_dots_147 as u32);
    pub const braille_dots_247: Key = Key(gdk_sys::GDK_KEY_braille_dots_247 as u32);
    pub const braille_dots_1247: Key = Key(gdk_sys::GDK_KEY_braille_dots_1247 as u32);
    pub const braille_dots_347: Key = Key(gdk_sys::GDK_KEY_braille_dots_347 as u32);
    pub const braille_dots_1347: Key = Key(gdk_sys::GDK_KEY_braille_dots_1347 as u32);
    pub const braille_dots_2347: Key = Key(gdk_sys::GDK_KEY_braille_dots_2347 as u32);
    pub const braille_dots_12347: Key = Key(gdk_sys::GDK_KEY_braille_dots_12347 as u32);
    pub const braille_dots_57: Key = Key(gdk_sys::GDK_KEY_braille_dots_57 as u32);
    pub const braille_dots_157: Key = Key(gdk_sys::GDK_KEY_braille_dots_157 as u32);
    pub const braille_dots_257: Key = Key(gdk_sys::GDK_KEY_braille_dots_257 as u32);
    pub const braille_dots_1257: Key = Key(gdk_sys::GDK_KEY_braille_dots_1257 as u32);
    pub const braille_dots_357: Key = Key(gdk_sys::GDK_KEY_braille_dots_357 as u32);
    pub const braille_dots_1357: Key = Key(gdk_sys::GDK_KEY_braille_dots_1357 as u32);
    pub const braille_dots_2357: Key = Key(gdk_sys::GDK_KEY_braille_dots_2357 as u32);
    pub const braille_dots_12357: Key = Key(gdk_sys::GDK_KEY_braille_dots_12357 as u32);
    pub const braille_dots_457: Key = Key(gdk_sys::GDK_KEY_braille_dots_457 as u32);
    pub const braille_dots_1457: Key = Key(gdk_sys::GDK_KEY_braille_dots_1457 as u32);
    pub const braille_dots_2457: Key = Key(gdk_sys::GDK_KEY_braille_dots_2457 as u32);
    pub const braille_dots_12457: Key = Key(gdk_sys::GDK_KEY_braille_dots_12457 as u32);
    pub const braille_dots_3457: Key = Key(gdk_sys::GDK_KEY_braille_dots_3457 as u32);
    pub const braille_dots_13457: Key = Key(gdk_sys::GDK_KEY_braille_dots_13457 as u32);
    pub const braille_dots_23457: Key = Key(gdk_sys::GDK_KEY_braille_dots_23457 as u32);
    pub const braille_dots_123457: Key = Key(gdk_sys::GDK_KEY_braille_dots_123457 as u32);
    pub const braille_dots_67: Key = Key(gdk_sys::GDK_KEY_braille_dots_67 as u32);
    pub const braille_dots_167: Key = Key(gdk_sys::GDK_KEY_braille_dots_167 as u32);
    pub const braille_dots_267: Key = Key(gdk_sys::GDK_KEY_braille_dots_267 as u32);
    pub const braille_dots_1267: Key = Key(gdk_sys::GDK_KEY_braille_dots_1267 as u32);
    pub const braille_dots_367: Key = Key(gdk_sys::GDK_KEY_braille_dots_367 as u32);
    pub const braille_dots_1367: Key = Key(gdk_sys::GDK_KEY_braille_dots_1367 as u32);
    pub const braille_dots_2367: Key = Key(gdk_sys::GDK_KEY_braille_dots_2367 as u32);
    pub const braille_dots_12367: Key = Key(gdk_sys::GDK_KEY_braille_dots_12367 as u32);
    pub const braille_dots_467: Key = Key(gdk_sys::GDK_KEY_braille_dots_467 as u32);
    pub const braille_dots_1467: Key = Key(gdk_sys::GDK_KEY_braille_dots_1467 as u32);
    pub const braille_dots_2467: Key = Key(gdk_sys::GDK_KEY_braille_dots_2467 as u32);
    pub const braille_dots_12467: Key = Key(gdk_sys::GDK_KEY_braille_dots_12467 as u32);
    pub const braille_dots_3467: Key = Key(gdk_sys::GDK_KEY_braille_dots_3467 as u32);
    pub const braille_dots_13467: Key = Key(gdk_sys::GDK_KEY_braille_dots_13467 as u32);
    pub const braille_dots_23467: Key = Key(gdk_sys::GDK_KEY_braille_dots_23467 as u32);
    pub const braille_dots_123467: Key = Key(gdk_sys::GDK_KEY_braille_dots_123467 as u32);
    pub const braille_dots_567: Key = Key(gdk_sys::GDK_KEY_braille_dots_567 as u32);
    pub const braille_dots_1567: Key = Key(gdk_sys::GDK_KEY_braille_dots_1567 as u32);
    pub const braille_dots_2567: Key = Key(gdk_sys::GDK_KEY_braille_dots_2567 as u32);
    pub const braille_dots_12567: Key = Key(gdk_sys::GDK_KEY_braille_dots_12567 as u32);
    pub const braille_dots_3567: Key = Key(gdk_sys::GDK_KEY_braille_dots_3567 as u32);
    pub const braille_dots_13567: Key = Key(gdk_sys::GDK_KEY_braille_dots_13567 as u32);
    pub const braille_dots_23567: Key = Key(gdk_sys::GDK_KEY_braille_dots_23567 as u32);
    pub const braille_dots_123567: Key = Key(gdk_sys::GDK_KEY_braille_dots_123567 as u32);
    pub const braille_dots_4567: Key = Key(gdk_sys::GDK_KEY_braille_dots_4567 as u32);
    pub const braille_dots_14567: Key = Key(gdk_sys::GDK_KEY_braille_dots_14567 as u32);
    pub const braille_dots_24567: Key = Key(gdk_sys::GDK_KEY_braille_dots_24567 as u32);
    pub const braille_dots_124567: Key = Key(gdk_sys::GDK_KEY_braille_dots_124567 as u32);
    pub const braille_dots_34567: Key = Key(gdk_sys::GDK_KEY_braille_dots_34567 as u32);
    pub const braille_dots_134567: Key = Key(gdk_sys::GDK_KEY_braille_dots_134567 as u32);
    pub const braille_dots_234567: Key = Key(gdk_sys::GDK_KEY_braille_dots_234567 as u32);
    pub const braille_dots_1234567: Key = Key(gdk_sys::GDK_KEY_braille_dots_1234567 as u32);
    pub const braille_dots_8: Key = Key(gdk_sys::GDK_KEY_braille_dots_8 as u32);
    pub const braille_dots_18: Key = Key(gdk_sys::GDK_KEY_braille_dots_18 as u32);
    pub const braille_dots_28: Key = Key(gdk_sys::GDK_KEY_braille_dots_28 as u32);
    pub const braille_dots_128: Key = Key(gdk_sys::GDK_KEY_braille_dots_128 as u32);
    pub const braille_dots_38: Key = Key(gdk_sys::GDK_KEY_braille_dots_38 as u32);
    pub const braille_dots_138: Key = Key(gdk_sys::GDK_KEY_braille_dots_138 as u32);
    pub const braille_dots_238: Key = Key(gdk_sys::GDK_KEY_braille_dots_238 as u32);
    pub const braille_dots_1238: Key = Key(gdk_sys::GDK_KEY_braille_dots_1238 as u32);
    pub const braille_dots_48: Key = Key(gdk_sys::GDK_KEY_braille_dots_48 as u32);
    pub const braille_dots_148: Key = Key(gdk_sys::GDK_KEY_braille_dots_148 as u32);
    pub const braille_dots_248: Key = Key(gdk_sys::GDK_KEY_braille_dots_248 as u32);
    pub const braille_dots_1248: Key = Key(gdk_sys::GDK_KEY_braille_dots_1248 as u32);
    pub const braille_dots_348: Key = Key(gdk_sys::GDK_KEY_braille_dots_348 as u32);
    pub const braille_dots_1348: Key = Key(gdk_sys::GDK_KEY_braille_dots_1348 as u32);
    pub const braille_dots_2348: Key = Key(gdk_sys::GDK_KEY_braille_dots_2348 as u32);
    pub const braille_dots_12348: Key = Key(gdk_sys::GDK_KEY_braille_dots_12348 as u32);
    pub const braille_dots_58: Key = Key(gdk_sys::GDK_KEY_braille_dots_58 as u32);
    pub const braille_dots_158: Key = Key(gdk_sys::GDK_KEY_braille_dots_158 as u32);
    pub const braille_dots_258: Key = Key(gdk_sys::GDK_KEY_braille_dots_258 as u32);
    pub const braille_dots_1258: Key = Key(gdk_sys::GDK_KEY_braille_dots_1258 as u32);
    pub const braille_dots_358: Key = Key(gdk_sys::GDK_KEY_braille_dots_358 as u32);
    pub const braille_dots_1358: Key = Key(gdk_sys::GDK_KEY_braille_dots_1358 as u32);
    pub const braille_dots_2358: Key = Key(gdk_sys::GDK_KEY_braille_dots_2358 as u32);
    pub const braille_dots_12358: Key = Key(gdk_sys::GDK_KEY_braille_dots_12358 as u32);
    pub const braille_dots_458: Key = Key(gdk_sys::GDK_KEY_braille_dots_458 as u32);
    pub const braille_dots_1458: Key = Key(gdk_sys::GDK_KEY_braille_dots_1458 as u32);
    pub const braille_dots_2458: Key = Key(gdk_sys::GDK_KEY_braille_dots_2458 as u32);
    pub const braille_dots_12458: Key = Key(gdk_sys::GDK_KEY_braille_dots_12458 as u32);
    pub const braille_dots_3458: Key = Key(gdk_sys::GDK_KEY_braille_dots_3458 as u32);
    pub const braille_dots_13458: Key = Key(gdk_sys::GDK_KEY_braille_dots_13458 as u32);
    pub const braille_dots_23458: Key = Key(gdk_sys::GDK_KEY_braille_dots_23458 as u32);
    pub const braille_dots_123458: Key = Key(gdk_sys::GDK_KEY_braille_dots_123458 as u32);
    pub const braille_dots_68: Key = Key(gdk_sys::GDK_KEY_braille_dots_68 as u32);
    pub const braille_dots_168: Key = Key(gdk_sys::GDK_KEY_braille_dots_168 as u32);
    pub const braille_dots_268: Key = Key(gdk_sys::GDK_KEY_braille_dots_268 as u32);
    pub const braille_dots_1268: Key = Key(gdk_sys::GDK_KEY_braille_dots_1268 as u32);
    pub const braille_dots_368: Key = Key(gdk_sys::GDK_KEY_braille_dots_368 as u32);
    pub const braille_dots_1368: Key = Key(gdk_sys::GDK_KEY_braille_dots_1368 as u32);
    pub const braille_dots_2368: Key = Key(gdk_sys::GDK_KEY_braille_dots_2368 as u32);
    pub const braille_dots_12368: Key = Key(gdk_sys::GDK_KEY_braille_dots_12368 as u32);
    pub const braille_dots_468: Key = Key(gdk_sys::GDK_KEY_braille_dots_468 as u32);
    pub const braille_dots_1468: Key = Key(gdk_sys::GDK_KEY_braille_dots_1468 as u32);
    pub const braille_dots_2468: Key = Key(gdk_sys::GDK_KEY_braille_dots_2468 as u32);
    pub const braille_dots_12468: Key = Key(gdk_sys::GDK_KEY_braille_dots_12468 as u32);
    pub const braille_dots_3468: Key = Key(gdk_sys::GDK_KEY_braille_dots_3468 as u32);
    pub const braille_dots_13468: Key = Key(gdk_sys::GDK_KEY_braille_dots_13468 as u32);
    pub const braille_dots_23468: Key = Key(gdk_sys::GDK_KEY_braille_dots_23468 as u32);
    pub const braille_dots_123468: Key = Key(gdk_sys::GDK_KEY_braille_dots_123468 as u32);
    pub const braille_dots_568: Key = Key(gdk_sys::GDK_KEY_braille_dots_568 as u32);
    pub const braille_dots_1568: Key = Key(gdk_sys::GDK_KEY_braille_dots_1568 as u32);
    pub const braille_dots_2568: Key = Key(gdk_sys::GDK_KEY_braille_dots_2568 as u32);
    pub const braille_dots_12568: Key = Key(gdk_sys::GDK_KEY_braille_dots_12568 as u32);
    pub const braille_dots_3568: Key = Key(gdk_sys::GDK_KEY_braille_dots_3568 as u32);
    pub const braille_dots_13568: Key = Key(gdk_sys::GDK_KEY_braille_dots_13568 as u32);
    pub const braille_dots_23568: Key = Key(gdk_sys::GDK_KEY_braille_dots_23568 as u32);
    pub const braille_dots_123568: Key = Key(gdk_sys::GDK_KEY_braille_dots_123568 as u32);
    pub const braille_dots_4568: Key = Key(gdk_sys::GDK_KEY_braille_dots_4568 as u32);
    pub const braille_dots_14568: Key = Key(gdk_sys::GDK_KEY_braille_dots_14568 as u32);
    pub const braille_dots_24568: Key = Key(gdk_sys::GDK_KEY_braille_dots_24568 as u32);
    pub const braille_dots_124568: Key = Key(gdk_sys::GDK_KEY_braille_dots_124568 as u32);
    pub const braille_dots_34568: Key = Key(gdk_sys::GDK_KEY_braille_dots_34568 as u32);
    pub const braille_dots_134568: Key = Key(gdk_sys::GDK_KEY_braille_dots_134568 as u32);
    pub const braille_dots_234568: Key = Key(gdk_sys::GDK_KEY_braille_dots_234568 as u32);
    pub const braille_dots_1234568: Key = Key(gdk_sys::GDK_KEY_braille_dots_1234568 as u32);
    pub const braille_dots_78: Key = Key(gdk_sys::GDK_KEY_braille_dots_78 as u32);
    pub const braille_dots_178: Key = Key(gdk_sys::GDK_KEY_braille_dots_178 as u32);
    pub const braille_dots_278: Key = Key(gdk_sys::GDK_KEY_braille_dots_278 as u32);
    pub const braille_dots_1278: Key = Key(gdk_sys::GDK_KEY_braille_dots_1278 as u32);
    pub const braille_dots_378: Key = Key(gdk_sys::GDK_KEY_braille_dots_378 as u32);
    pub const braille_dots_1378: Key = Key(gdk_sys::GDK_KEY_braille_dots_1378 as u32);
    pub const braille_dots_2378: Key = Key(gdk_sys::GDK_KEY_braille_dots_2378 as u32);
    pub const braille_dots_12378: Key = Key(gdk_sys::GDK_KEY_braille_dots_12378 as u32);
    pub const braille_dots_478: Key = Key(gdk_sys::GDK_KEY_braille_dots_478 as u32);
    pub const braille_dots_1478: Key = Key(gdk_sys::GDK_KEY_braille_dots_1478 as u32);
    pub const braille_dots_2478: Key = Key(gdk_sys::GDK_KEY_braille_dots_2478 as u32);
    pub const braille_dots_12478: Key = Key(gdk_sys::GDK_KEY_braille_dots_12478 as u32);
    pub const braille_dots_3478: Key = Key(gdk_sys::GDK_KEY_braille_dots_3478 as u32);
    pub const braille_dots_13478: Key = Key(gdk_sys::GDK_KEY_braille_dots_13478 as u32);
    pub const braille_dots_23478: Key = Key(gdk_sys::GDK_KEY_braille_dots_23478 as u32);
    pub const braille_dots_123478: Key = Key(gdk_sys::GDK_KEY_braille_dots_123478 as u32);
    pub const braille_dots_578: Key = Key(gdk_sys::GDK_KEY_braille_dots_578 as u32);
    pub const braille_dots_1578: Key = Key(gdk_sys::GDK_KEY_braille_dots_1578 as u32);
    pub const braille_dots_2578: Key = Key(gdk_sys::GDK_KEY_braille_dots_2578 as u32);
    pub const braille_dots_12578: Key = Key(gdk_sys::GDK_KEY_braille_dots_12578 as u32);
    pub const braille_dots_3578: Key = Key(gdk_sys::GDK_KEY_braille_dots_3578 as u32);
    pub const braille_dots_13578: Key = Key(gdk_sys::GDK_KEY_braille_dots_13578 as u32);
    pub const braille_dots_23578: Key = Key(gdk_sys::GDK_KEY_braille_dots_23578 as u32);
    pub const braille_dots_123578: Key = Key(gdk_sys::GDK_KEY_braille_dots_123578 as u32);
    pub const braille_dots_4578: Key = Key(gdk_sys::GDK_KEY_braille_dots_4578 as u32);
    pub const braille_dots_14578: Key = Key(gdk_sys::GDK_KEY_braille_dots_14578 as u32);
    pub const braille_dots_24578: Key = Key(gdk_sys::GDK_KEY_braille_dots_24578 as u32);
    pub const braille_dots_124578: Key = Key(gdk_sys::GDK_KEY_braille_dots_124578 as u32);
    pub const braille_dots_34578: Key = Key(gdk_sys::GDK_KEY_braille_dots_34578 as u32);
    pub const braille_dots_134578: Key = Key(gdk_sys::GDK_KEY_braille_dots_134578 as u32);
    pub const braille_dots_234578: Key = Key(gdk_sys::GDK_KEY_braille_dots_234578 as u32);
    pub const braille_dots_1234578: Key = Key(gdk_sys::GDK_KEY_braille_dots_1234578 as u32);
    pub const braille_dots_678: Key = Key(gdk_sys::GDK_KEY_braille_dots_678 as u32);
    pub const braille_dots_1678: Key = Key(gdk_sys::GDK_KEY_braille_dots_1678 as u32);
    pub const braille_dots_2678: Key = Key(gdk_sys::GDK_KEY_braille_dots_2678 as u32);
    pub const braille_dots_12678: Key = Key(gdk_sys::GDK_KEY_braille_dots_12678 as u32);
    pub const braille_dots_3678: Key = Key(gdk_sys::GDK_KEY_braille_dots_3678 as u32);
    pub const braille_dots_13678: Key = Key(gdk_sys::GDK_KEY_braille_dots_13678 as u32);
    pub const braille_dots_23678: Key = Key(gdk_sys::GDK_KEY_braille_dots_23678 as u32);
    pub const braille_dots_123678: Key = Key(gdk_sys::GDK_KEY_braille_dots_123678 as u32);
    pub const braille_dots_4678: Key = Key(gdk_sys::GDK_KEY_braille_dots_4678 as u32);
    pub const braille_dots_14678: Key = Key(gdk_sys::GDK_KEY_braille_dots_14678 as u32);
    pub const braille_dots_24678: Key = Key(gdk_sys::GDK_KEY_braille_dots_24678 as u32);
    pub const braille_dots_124678: Key = Key(gdk_sys::GDK_KEY_braille_dots_124678 as u32);
    pub const braille_dots_34678: Key = Key(gdk_sys::GDK_KEY_braille_dots_34678 as u32);
    pub const braille_dots_134678: Key = Key(gdk_sys::GDK_KEY_braille_dots_134678 as u32);
    pub const braille_dots_234678: Key = Key(gdk_sys::GDK_KEY_braille_dots_234678 as u32);
    pub const braille_dots_1234678: Key = Key(gdk_sys::GDK_KEY_braille_dots_1234678 as u32);
    pub const braille_dots_5678: Key = Key(gdk_sys::GDK_KEY_braille_dots_5678 as u32);
    pub const braille_dots_15678: Key = Key(gdk_sys::GDK_KEY_braille_dots_15678 as u32);
    pub const braille_dots_25678: Key = Key(gdk_sys::GDK_KEY_braille_dots_25678 as u32);
    pub const braille_dots_125678: Key = Key(gdk_sys::GDK_KEY_braille_dots_125678 as u32);
    pub const braille_dots_35678: Key = Key(gdk_sys::GDK_KEY_braille_dots_35678 as u32);
    pub const braille_dots_135678: Key = Key(gdk_sys::GDK_KEY_braille_dots_135678 as u32);
    pub const braille_dots_235678: Key = Key(gdk_sys::GDK_KEY_braille_dots_235678 as u32);
    pub const braille_dots_1235678: Key = Key(gdk_sys::GDK_KEY_braille_dots_1235678 as u32);
    pub const braille_dots_45678: Key = Key(gdk_sys::GDK_KEY_braille_dots_45678 as u32);
    pub const braille_dots_145678: Key = Key(gdk_sys::GDK_KEY_braille_dots_145678 as u32);
    pub const braille_dots_245678: Key = Key(gdk_sys::GDK_KEY_braille_dots_245678 as u32);
    pub const braille_dots_1245678: Key = Key(gdk_sys::GDK_KEY_braille_dots_1245678 as u32);
    pub const braille_dots_345678: Key = Key(gdk_sys::GDK_KEY_braille_dots_345678 as u32);
    pub const braille_dots_1345678: Key = Key(gdk_sys::GDK_KEY_braille_dots_1345678 as u32);
    pub const braille_dots_2345678: Key = Key(gdk_sys::GDK_KEY_braille_dots_2345678 as u32);
    pub const braille_dots_12345678: Key = Key(gdk_sys::GDK_KEY_braille_dots_12345678 as u32);
    pub const Sinh_ng: Key = Key(gdk_sys::GDK_KEY_Sinh_ng as u32);
    pub const Sinh_h2: Key = Key(gdk_sys::GDK_KEY_Sinh_h2 as u32);
    pub const Sinh_a: Key = Key(gdk_sys::GDK_KEY_Sinh_a as u32);
    pub const Sinh_aa: Key = Key(gdk_sys::GDK_KEY_Sinh_aa as u32);
    pub const Sinh_ae: Key = Key(gdk_sys::GDK_KEY_Sinh_ae as u32);
    pub const Sinh_aee: Key = Key(gdk_sys::GDK_KEY_Sinh_aee as u32);
    pub const Sinh_i: Key = Key(gdk_sys::GDK_KEY_Sinh_i as u32);
    pub const Sinh_ii: Key = Key(gdk_sys::GDK_KEY_Sinh_ii as u32);
    pub const Sinh_u: Key = Key(gdk_sys::GDK_KEY_Sinh_u as u32);
    pub const Sinh_uu: Key = Key(gdk_sys::GDK_KEY_Sinh_uu as u32);
    pub const Sinh_ri: Key = Key(gdk_sys::GDK_KEY_Sinh_ri as u32);
    pub const Sinh_rii: Key = Key(gdk_sys::GDK_KEY_Sinh_rii as u32);
    pub const Sinh_lu: Key = Key(gdk_sys::GDK_KEY_Sinh_lu as u32);
    pub const Sinh_luu: Key = Key(gdk_sys::GDK_KEY_Sinh_luu as u32);
    pub const Sinh_e: Key = Key(gdk_sys::GDK_KEY_Sinh_e as u32);
    pub const Sinh_ee: Key = Key(gdk_sys::GDK_KEY_Sinh_ee as u32);
    pub const Sinh_ai: Key = Key(gdk_sys::GDK_KEY_Sinh_ai as u32);
    pub const Sinh_o: Key = Key(gdk_sys::GDK_KEY_Sinh_o as u32);
    pub const Sinh_oo: Key = Key(gdk_sys::GDK_KEY_Sinh_oo as u32);
    pub const Sinh_au: Key = Key(gdk_sys::GDK_KEY_Sinh_au as u32);
    pub const Sinh_ka: Key = Key(gdk_sys::GDK_KEY_Sinh_ka as u32);
    pub const Sinh_kha: Key = Key(gdk_sys::GDK_KEY_Sinh_kha as u32);
    pub const Sinh_ga: Key = Key(gdk_sys::GDK_KEY_Sinh_ga as u32);
    pub const Sinh_gha: Key = Key(gdk_sys::GDK_KEY_Sinh_gha as u32);
    pub const Sinh_ng2: Key = Key(gdk_sys::GDK_KEY_Sinh_ng2 as u32);
    pub const Sinh_nga: Key = Key(gdk_sys::GDK_KEY_Sinh_nga as u32);
    pub const Sinh_ca: Key = Key(gdk_sys::GDK_KEY_Sinh_ca as u32);
    pub const Sinh_cha: Key = Key(gdk_sys::GDK_KEY_Sinh_cha as u32);
    pub const Sinh_ja: Key = Key(gdk_sys::GDK_KEY_Sinh_ja as u32);
    pub const Sinh_jha: Key = Key(gdk_sys::GDK_KEY_Sinh_jha as u32);
    pub const Sinh_nya: Key = Key(gdk_sys::GDK_KEY_Sinh_nya as u32);
    pub const Sinh_jnya: Key = Key(gdk_sys::GDK_KEY_Sinh_jnya as u32);
    pub const Sinh_nja: Key = Key(gdk_sys::GDK_KEY_Sinh_nja as u32);
    pub const Sinh_tta: Key = Key(gdk_sys::GDK_KEY_Sinh_tta as u32);
    pub const Sinh_ttha: Key = Key(gdk_sys::GDK_KEY_Sinh_ttha as u32);
    pub const Sinh_dda: Key = Key(gdk_sys::GDK_KEY_Sinh_dda as u32);
    pub const Sinh_ddha: Key = Key(gdk_sys::GDK_KEY_Sinh_ddha as u32);
    pub const Sinh_nna: Key = Key(gdk_sys::GDK_KEY_Sinh_nna as u32);
    pub const Sinh_ndda: Key = Key(gdk_sys::GDK_KEY_Sinh_ndda as u32);
    pub const Sinh_tha: Key = Key(gdk_sys::GDK_KEY_Sinh_tha as u32);
    pub const Sinh_thha: Key = Key(gdk_sys::GDK_KEY_Sinh_thha as u32);
    pub const Sinh_dha: Key = Key(gdk_sys::GDK_KEY_Sinh_dha as u32);
    pub const Sinh_dhha: Key = Key(gdk_sys::GDK_KEY_Sinh_dhha as u32);
    pub const Sinh_na: Key = Key(gdk_sys::GDK_KEY_Sinh_na as u32);
    pub const Sinh_ndha: Key = Key(gdk_sys::GDK_KEY_Sinh_ndha as u32);
    pub const Sinh_pa: Key = Key(gdk_sys::GDK_KEY_Sinh_pa as u32);
    pub const Sinh_pha: Key = Key(gdk_sys::GDK_KEY_Sinh_pha as u32);
    pub const Sinh_ba: Key = Key(gdk_sys::GDK_KEY_Sinh_ba as u32);
    pub const Sinh_bha: Key = Key(gdk_sys::GDK_KEY_Sinh_bha as u32);
    pub const Sinh_ma: Key = Key(gdk_sys::GDK_KEY_Sinh_ma as u32);
    pub const Sinh_mba: Key = Key(gdk_sys::GDK_KEY_Sinh_mba as u32);
    pub const Sinh_ya: Key = Key(gdk_sys::GDK_KEY_Sinh_ya as u32);
    pub const Sinh_ra: Key = Key(gdk_sys::GDK_KEY_Sinh_ra as u32);
    pub const Sinh_la: Key = Key(gdk_sys::GDK_KEY_Sinh_la as u32);
    pub const Sinh_va: Key = Key(gdk_sys::GDK_KEY_Sinh_va as u32);
    pub const Sinh_sha: Key = Key(gdk_sys::GDK_KEY_Sinh_sha as u32);
    pub const Sinh_ssha: Key = Key(gdk_sys::GDK_KEY_Sinh_ssha as u32);
    pub const Sinh_sa: Key = Key(gdk_sys::GDK_KEY_Sinh_sa as u32);
    pub const Sinh_ha: Key = Key(gdk_sys::GDK_KEY_Sinh_ha as u32);
    pub const Sinh_lla: Key = Key(gdk_sys::GDK_KEY_Sinh_lla as u32);
    pub const Sinh_fa: Key = Key(gdk_sys::GDK_KEY_Sinh_fa as u32);
    pub const Sinh_al: Key = Key(gdk_sys::GDK_KEY_Sinh_al as u32);
    pub const Sinh_aa2: Key = Key(gdk_sys::GDK_KEY_Sinh_aa2 as u32);
    pub const Sinh_ae2: Key = Key(gdk_sys::GDK_KEY_Sinh_ae2 as u32);
    pub const Sinh_aee2: Key = Key(gdk_sys::GDK_KEY_Sinh_aee2 as u32);
    pub const Sinh_i2: Key = Key(gdk_sys::GDK_KEY_Sinh_i2 as u32);
    pub const Sinh_ii2: Key = Key(gdk_sys::GDK_KEY_Sinh_ii2 as u32);
    pub const Sinh_u2: Key = Key(gdk_sys::GDK_KEY_Sinh_u2 as u32);
    pub const Sinh_uu2: Key = Key(gdk_sys::GDK_KEY_Sinh_uu2 as u32);
    pub const Sinh_ru2: Key = Key(gdk_sys::GDK_KEY_Sinh_ru2 as u32);
    pub const Sinh_e2: Key = Key(gdk_sys::GDK_KEY_Sinh_e2 as u32);
    pub const Sinh_ee2: Key = Key(gdk_sys::GDK_KEY_Sinh_ee2 as u32);
    pub const Sinh_ai2: Key = Key(gdk_sys::GDK_KEY_Sinh_ai2 as u32);
    pub const Sinh_o2: Key = Key(gdk_sys::GDK_KEY_Sinh_o2 as u32);
    pub const Sinh_oo2: Key = Key(gdk_sys::GDK_KEY_Sinh_oo2 as u32);
    pub const Sinh_au2: Key = Key(gdk_sys::GDK_KEY_Sinh_au2 as u32);
    pub const Sinh_lu2: Key = Key(gdk_sys::GDK_KEY_Sinh_lu2 as u32);
    pub const Sinh_ruu2: Key = Key(gdk_sys::GDK_KEY_Sinh_ruu2 as u32);
    pub const Sinh_luu2: Key = Key(gdk_sys::GDK_KEY_Sinh_luu2 as u32);
    pub const Sinh_kunddaliya: Key = Key(gdk_sys::GDK_KEY_Sinh_kunddaliya as u32);
    pub const ModeLock: Key = Key(gdk_sys::GDK_KEY_ModeLock as u32);
    pub const MonBrightnessUp: Key = Key(gdk_sys::GDK_KEY_MonBrightnessUp as u32);
    pub const MonBrightnessDown: Key = Key(gdk_sys::GDK_KEY_MonBrightnessDown as u32);
    pub const KbdLightOnOff: Key = Key(gdk_sys::GDK_KEY_KbdLightOnOff as u32);
    pub const KbdBrightnessUp: Key = Key(gdk_sys::GDK_KEY_KbdBrightnessUp as u32);
    pub const KbdBrightnessDown: Key = Key(gdk_sys::GDK_KEY_KbdBrightnessDown as u32);
    pub const Standby: Key = Key(gdk_sys::GDK_KEY_Standby as u32);
    pub const AudioLowerVolume: Key = Key(gdk_sys::GDK_KEY_AudioLowerVolume as u32);
    pub const AudioMute: Key = Key(gdk_sys::GDK_KEY_AudioMute as u32);
    pub const AudioRaiseVolume: Key = Key(gdk_sys::GDK_KEY_AudioRaiseVolume as u32);
    pub const AudioPlay: Key = Key(gdk_sys::GDK_KEY_AudioPlay as u32);
    pub const AudioStop: Key = Key(gdk_sys::GDK_KEY_AudioStop as u32);
    pub const AudioPrev: Key = Key(gdk_sys::GDK_KEY_AudioPrev as u32);
    pub const AudioNext: Key = Key(gdk_sys::GDK_KEY_AudioNext as u32);
    pub const HomePage: Key = Key(gdk_sys::GDK_KEY_HomePage as u32);
    pub const Mail: Key = Key(gdk_sys::GDK_KEY_Mail as u32);
    pub const Start: Key = Key(gdk_sys::GDK_KEY_Start as u32);
    pub const Search: Key = Key(gdk_sys::GDK_KEY_Search as u32);
    pub const AudioRecord: Key = Key(gdk_sys::GDK_KEY_AudioRecord as u32);
    pub const Calculator: Key = Key(gdk_sys::GDK_KEY_Calculator as u32);
    pub const Memo: Key = Key(gdk_sys::GDK_KEY_Memo as u32);
    pub const ToDoList: Key = Key(gdk_sys::GDK_KEY_ToDoList as u32);
    pub const Calendar: Key = Key(gdk_sys::GDK_KEY_Calendar as u32);
    pub const PowerDown: Key = Key(gdk_sys::GDK_KEY_PowerDown as u32);
    pub const ContrastAdjust: Key = Key(gdk_sys::GDK_KEY_ContrastAdjust as u32);
    pub const RockerUp: Key = Key(gdk_sys::GDK_KEY_RockerUp as u32);
    pub const RockerDown: Key = Key(gdk_sys::GDK_KEY_RockerDown as u32);
    pub const RockerEnter: Key = Key(gdk_sys::GDK_KEY_RockerEnter as u32);
    pub const Back: Key = Key(gdk_sys::GDK_KEY_Back as u32);
    pub const Forward: Key = Key(gdk_sys::GDK_KEY_Forward as u32);
    pub const Stop: Key = Key(gdk_sys::GDK_KEY_Stop as u32);
    pub const Refresh: Key = Key(gdk_sys::GDK_KEY_Refresh as u32);
    pub const PowerOff: Key = Key(gdk_sys::GDK_KEY_PowerOff as u32);
    pub const WakeUp: Key = Key(gdk_sys::GDK_KEY_WakeUp as u32);
    pub const Eject: Key = Key(gdk_sys::GDK_KEY_Eject as u32);
    pub const ScreenSaver: Key = Key(gdk_sys::GDK_KEY_ScreenSaver as u32);
    pub const WWW: Key = Key(gdk_sys::GDK_KEY_WWW as u32);
    pub const Sleep: Key = Key(gdk_sys::GDK_KEY_Sleep as u32);
    pub const Favorites: Key = Key(gdk_sys::GDK_KEY_Favorites as u32);
    pub const AudioPause: Key = Key(gdk_sys::GDK_KEY_AudioPause as u32);
    pub const AudioMedia: Key = Key(gdk_sys::GDK_KEY_AudioMedia as u32);
    pub const MyComputer: Key = Key(gdk_sys::GDK_KEY_MyComputer as u32);
    pub const VendorHome: Key = Key(gdk_sys::GDK_KEY_VendorHome as u32);
    pub const LightBulb: Key = Key(gdk_sys::GDK_KEY_LightBulb as u32);
    pub const Shop: Key = Key(gdk_sys::GDK_KEY_Shop as u32);
    pub const History: Key = Key(gdk_sys::GDK_KEY_History as u32);
    pub const OpenURL: Key = Key(gdk_sys::GDK_KEY_OpenURL as u32);
    pub const AddFavorite: Key = Key(gdk_sys::GDK_KEY_AddFavorite as u32);
    pub const HotLinks: Key = Key(gdk_sys::GDK_KEY_HotLinks as u32);
    pub const BrightnessAdjust: Key = Key(gdk_sys::GDK_KEY_BrightnessAdjust as u32);
    pub const Finance: Key = Key(gdk_sys::GDK_KEY_Finance as u32);
    pub const Community: Key = Key(gdk_sys::GDK_KEY_Community as u32);
    pub const AudioRewind: Key = Key(gdk_sys::GDK_KEY_AudioRewind as u32);
    pub const BackForward: Key = Key(gdk_sys::GDK_KEY_BackForward as u32);
    pub const Launch0: Key = Key(gdk_sys::GDK_KEY_Launch0 as u32);
    pub const Launch1: Key = Key(gdk_sys::GDK_KEY_Launch1 as u32);
    pub const Launch2: Key = Key(gdk_sys::GDK_KEY_Launch2 as u32);
    pub const Launch3: Key = Key(gdk_sys::GDK_KEY_Launch3 as u32);
    pub const Launch4: Key = Key(gdk_sys::GDK_KEY_Launch4 as u32);
    pub const Launch5: Key = Key(gdk_sys::GDK_KEY_Launch5 as u32);
    pub const Launch6: Key = Key(gdk_sys::GDK_KEY_Launch6 as u32);
    pub const Launch7: Key = Key(gdk_sys::GDK_KEY_Launch7 as u32);
    pub const Launch8: Key = Key(gdk_sys::GDK_KEY_Launch8 as u32);
    pub const Launch9: Key = Key(gdk_sys::GDK_KEY_Launch9 as u32);
    pub const LaunchA: Key = Key(gdk_sys::GDK_KEY_LaunchA as u32);
    pub const LaunchB: Key = Key(gdk_sys::GDK_KEY_LaunchB as u32);
    pub const LaunchC: Key = Key(gdk_sys::GDK_KEY_LaunchC as u32);
    pub const LaunchD: Key = Key(gdk_sys::GDK_KEY_LaunchD as u32);
    pub const LaunchE: Key = Key(gdk_sys::GDK_KEY_LaunchE as u32);
    pub const LaunchF: Key = Key(gdk_sys::GDK_KEY_LaunchF as u32);
    pub const ApplicationLeft: Key = Key(gdk_sys::GDK_KEY_ApplicationLeft as u32);
    pub const ApplicationRight: Key = Key(gdk_sys::GDK_KEY_ApplicationRight as u32);
    pub const Book: Key = Key(gdk_sys::GDK_KEY_Book as u32);
    pub const CD: Key = Key(gdk_sys::GDK_KEY_CD as u32);
    pub const WindowClear: Key = Key(gdk_sys::GDK_KEY_WindowClear as u32);
    pub const Close: Key = Key(gdk_sys::GDK_KEY_Close as u32);
    pub const Copy: Key = Key(gdk_sys::GDK_KEY_Copy as u32);
    pub const Cut: Key = Key(gdk_sys::GDK_KEY_Cut as u32);
    pub const Display: Key = Key(gdk_sys::GDK_KEY_Display as u32);
    pub const DOS: Key = Key(gdk_sys::GDK_KEY_DOS as u32);
    pub const Documents: Key = Key(gdk_sys::GDK_KEY_Documents as u32);
    pub const Excel: Key = Key(gdk_sys::GDK_KEY_Excel as u32);
    pub const Explorer: Key = Key(gdk_sys::GDK_KEY_Explorer as u32);
    pub const Game: Key = Key(gdk_sys::GDK_KEY_Game as u32);
    pub const Go: Key = Key(gdk_sys::GDK_KEY_Go as u32);
    pub const iTouch: Key = Key(gdk_sys::GDK_KEY_iTouch as u32);
    pub const LogOff: Key = Key(gdk_sys::GDK_KEY_LogOff as u32);
    pub const Market: Key = Key(gdk_sys::GDK_KEY_Market as u32);
    pub const Meeting: Key = Key(gdk_sys::GDK_KEY_Meeting as u32);
    pub const MenuKB: Key = Key(gdk_sys::GDK_KEY_MenuKB as u32);
    pub const MenuPB: Key = Key(gdk_sys::GDK_KEY_MenuPB as u32);
    pub const MySites: Key = Key(gdk_sys::GDK_KEY_MySites as u32);
    pub const New: Key = Key(gdk_sys::GDK_KEY_New as u32);
    pub const News: Key = Key(gdk_sys::GDK_KEY_News as u32);
    pub const OfficeHome: Key = Key(gdk_sys::GDK_KEY_OfficeHome as u32);
    pub const Open: Key = Key(gdk_sys::GDK_KEY_Open as u32);
    pub const Option: Key = Key(gdk_sys::GDK_KEY_Option as u32);
    pub const Paste: Key = Key(gdk_sys::GDK_KEY_Paste as u32);
    pub const Phone: Key = Key(gdk_sys::GDK_KEY_Phone as u32);
    pub const Reply: Key = Key(gdk_sys::GDK_KEY_Reply as u32);
    pub const Reload: Key = Key(gdk_sys::GDK_KEY_Reload as u32);
    pub const RotateWindows: Key = Key(gdk_sys::GDK_KEY_RotateWindows as u32);
    pub const RotationPB: Key = Key(gdk_sys::GDK_KEY_RotationPB as u32);
    pub const RotationKB: Key = Key(gdk_sys::GDK_KEY_RotationKB as u32);
    pub const Save: Key = Key(gdk_sys::GDK_KEY_Save as u32);
    pub const ScrollUp: Key = Key(gdk_sys::GDK_KEY_ScrollUp as u32);
    pub const ScrollDown: Key = Key(gdk_sys::GDK_KEY_ScrollDown as u32);
    pub const ScrollClick: Key = Key(gdk_sys::GDK_KEY_ScrollClick as u32);
    pub const Send: Key = Key(gdk_sys::GDK_KEY_Send as u32);
    pub const Spell: Key = Key(gdk_sys::GDK_KEY_Spell as u32);
    pub const SplitScreen: Key = Key(gdk_sys::GDK_KEY_SplitScreen as u32);
    pub const Support: Key = Key(gdk_sys::GDK_KEY_Support as u32);
    pub const TaskPane: Key = Key(gdk_sys::GDK_KEY_TaskPane as u32);
    pub const Terminal: Key = Key(gdk_sys::GDK_KEY_Terminal as u32);
    pub const Tools: Key = Key(gdk_sys::GDK_KEY_Tools as u32);
    pub const Travel: Key = Key(gdk_sys::GDK_KEY_Travel as u32);
    pub const UserPB: Key = Key(gdk_sys::GDK_KEY_UserPB as u32);
    pub const User1KB: Key = Key(gdk_sys::GDK_KEY_User1KB as u32);
    pub const User2KB: Key = Key(gdk_sys::GDK_KEY_User2KB as u32);
    pub const Video: Key = Key(gdk_sys::GDK_KEY_Video as u32);
    pub const WheelButton: Key = Key(gdk_sys::GDK_KEY_WheelButton as u32);
    pub const Word: Key = Key(gdk_sys::GDK_KEY_Word as u32);
    pub const Xfer: Key = Key(gdk_sys::GDK_KEY_Xfer as u32);
    pub const ZoomIn: Key = Key(gdk_sys::GDK_KEY_ZoomIn as u32);
    pub const ZoomOut: Key = Key(gdk_sys::GDK_KEY_ZoomOut as u32);
    pub const Away: Key = Key(gdk_sys::GDK_KEY_Away as u32);
    pub const Messenger: Key = Key(gdk_sys::GDK_KEY_Messenger as u32);
    pub const WebCam: Key = Key(gdk_sys::GDK_KEY_WebCam as u32);
    pub const MailForward: Key = Key(gdk_sys::GDK_KEY_MailForward as u32);
    pub const Pictures: Key = Key(gdk_sys::GDK_KEY_Pictures as u32);
    pub const Music: Key = Key(gdk_sys::GDK_KEY_Music as u32);
    pub const Battery: Key = Key(gdk_sys::GDK_KEY_Battery as u32);
    pub const Bluetooth: Key = Key(gdk_sys::GDK_KEY_Bluetooth as u32);
    pub const WLAN: Key = Key(gdk_sys::GDK_KEY_WLAN as u32);
    pub const UWB: Key = Key(gdk_sys::GDK_KEY_UWB as u32);
    pub const AudioForward: Key = Key(gdk_sys::GDK_KEY_AudioForward as u32);
    pub const AudioRepeat: Key = Key(gdk_sys::GDK_KEY_AudioRepeat as u32);
    pub const AudioRandomPlay: Key = Key(gdk_sys::GDK_KEY_AudioRandomPlay as u32);
    pub const Subtitle: Key = Key(gdk_sys::GDK_KEY_Subtitle as u32);
    pub const AudioCycleTrack: Key = Key(gdk_sys::GDK_KEY_AudioCycleTrack as u32);
    pub const CycleAngle: Key = Key(gdk_sys::GDK_KEY_CycleAngle as u32);
    pub const FrameBack: Key = Key(gdk_sys::GDK_KEY_FrameBack as u32);
    pub const FrameForward: Key = Key(gdk_sys::GDK_KEY_FrameForward as u32);
    pub const Time: Key = Key(gdk_sys::GDK_KEY_Time as u32);
    pub const SelectButton: Key = Key(gdk_sys::GDK_KEY_SelectButton as u32);
    pub const View: Key = Key(gdk_sys::GDK_KEY_View as u32);
    pub const TopMenu: Key = Key(gdk_sys::GDK_KEY_TopMenu as u32);
    pub const Red: Key = Key(gdk_sys::GDK_KEY_Red as u32);
    pub const Green: Key = Key(gdk_sys::GDK_KEY_Green as u32);
    pub const Yellow: Key = Key(gdk_sys::GDK_KEY_Yellow as u32);
    pub const Blue: Key = Key(gdk_sys::GDK_KEY_Blue as u32);
    pub const Suspend: Key = Key(gdk_sys::GDK_KEY_Suspend as u32);
    pub const Hibernate: Key = Key(gdk_sys::GDK_KEY_Hibernate as u32);
    pub const TouchpadToggle: Key = Key(gdk_sys::GDK_KEY_TouchpadToggle as u32);
    pub const TouchpadOn: Key = Key(gdk_sys::GDK_KEY_TouchpadOn as u32);
    pub const TouchpadOff: Key = Key(gdk_sys::GDK_KEY_TouchpadOff as u32);
    pub const AudioMicMute: Key = Key(gdk_sys::GDK_KEY_AudioMicMute as u32);
    pub const Switch_VT_1: Key = Key(gdk_sys::GDK_KEY_Switch_VT_1 as u32);
    pub const Switch_VT_2: Key = Key(gdk_sys::GDK_KEY_Switch_VT_2 as u32);
    pub const Switch_VT_3: Key = Key(gdk_sys::GDK_KEY_Switch_VT_3 as u32);
    pub const Switch_VT_4: Key = Key(gdk_sys::GDK_KEY_Switch_VT_4 as u32);
    pub const Switch_VT_5: Key = Key(gdk_sys::GDK_KEY_Switch_VT_5 as u32);
    pub const Switch_VT_6: Key = Key(gdk_sys::GDK_KEY_Switch_VT_6 as u32);
    pub const Switch_VT_7: Key = Key(gdk_sys::GDK_KEY_Switch_VT_7 as u32);
    pub const Switch_VT_8: Key = Key(gdk_sys::GDK_KEY_Switch_VT_8 as u32);
    pub const Switch_VT_9: Key = Key(gdk_sys::GDK_KEY_Switch_VT_9 as u32);
    pub const Switch_VT_10: Key = Key(gdk_sys::GDK_KEY_Switch_VT_10 as u32);
    pub const Switch_VT_11: Key = Key(gdk_sys::GDK_KEY_Switch_VT_11 as u32);
    pub const Switch_VT_12: Key = Key(gdk_sys::GDK_KEY_Switch_VT_12 as u32);
    pub const Ungrab: Key = Key(gdk_sys::GDK_KEY_Ungrab as u32);
    pub const ClearGrab: Key = Key(gdk_sys::GDK_KEY_ClearGrab as u32);
    pub const Next_VMode: Key = Key(gdk_sys::GDK_KEY_Next_VMode as u32);
    pub const Prev_VMode: Key = Key(gdk_sys::GDK_KEY_Prev_VMode as u32);
    pub const LogWindowTree: Key = Key(gdk_sys::GDK_KEY_LogWindowTree as u32);
    pub const LogGrabInfo: Key = Key(gdk_sys::GDK_KEY_LogGrabInfo as u32);
}
