/*
 * peas-helpers.c
 * This file is part of libpeas
 *
 * Copyright (C) 2010 Steve Frécinaux
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Library General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>

#include <gobject/gvaluecollector.h>

#include "peas-helpers.h"

static void
add_all_interfaces (GType      iface_type,
                    GPtrArray *type_structs)
{
  GType *prereq;
  guint n_prereq;
  guint i;

  g_ptr_array_add (type_structs,
                   g_type_default_interface_ref (iface_type));

  prereq = g_type_interface_prerequisites (iface_type, &n_prereq);

  for (i = 0; i < n_prereq; ++i)
    {
      if (G_TYPE_IS_INTERFACE (prereq[i]))
        add_all_interfaces (prereq[i], type_structs);
    }

  g_free (prereq);
}

static GParamSpec *
find_param_spec_in_interfaces (GPtrArray   *type_structs,
                               const gchar *name)
{
  guint i;

  for (i = 0; i < type_structs->len; ++i)
    {
      GParamSpec *pspec;

      pspec = g_object_interface_find_property (g_ptr_array_index (type_structs, i),
                                                name);

      if (pspec != NULL)
        return pspec;
    }

  return NULL;
}

gboolean
_valist_to_parameter_list (GType         iface_type,
                           const gchar  *first_property_name,
                           va_list       args,
                           GParameter  **params,
                           guint        *n_params)
{
  GPtrArray *type_structs;
  const gchar *name;
  guint n_allocated_params;

  g_return_val_if_fail (G_TYPE_IS_INTERFACE (iface_type), FALSE);

  type_structs = g_ptr_array_new ();
  g_ptr_array_set_free_func (type_structs,
                             (GDestroyNotify) g_type_default_interface_unref);
  add_all_interfaces (iface_type, type_structs);

  *n_params = 0;
  n_allocated_params = 16;
  *params = g_new0 (GParameter, n_allocated_params);

  name = first_property_name;
  while (name)
    {
      gchar *error_msg = NULL;
      GParamSpec *pspec = find_param_spec_in_interfaces (type_structs, name);

      if (!pspec)
        {
          g_warning ("%s: type '%s' has no property named '%s'",
                     G_STRFUNC, g_type_name (iface_type), name);
          goto error;
        }

      if (*n_params >= n_allocated_params)
        {
          n_allocated_params += 16;
          *params = g_renew (GParameter, *params, n_allocated_params);
          memset (*params + sizeof (GParameter) * (n_allocated_params - 16),
                  0, sizeof (GParameter) * 16);
        }

      (*params)[*n_params].name = name;
      G_VALUE_COLLECT_INIT (&(*params)[*n_params].value, pspec->value_type,
                            args, 0, &error_msg);

      (*n_params)++;

      if (error_msg)
        {
          g_warning ("%s: %s", G_STRFUNC, error_msg);
          g_free (error_msg);
          goto error;
        }

      name = va_arg (args, gchar*);
    }

  g_ptr_array_unref (type_structs);

  return TRUE;

error:

  for (; *n_params > 0; --(*n_params))
    g_value_unset (&(*params)[*n_params].value);

  g_free (*params);
  g_ptr_array_unref (type_structs);

  return FALSE;
}
