// Licensed under the Apache License, Version 2.0
// <LICENSE-APACHE or http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your option.
// All files in the project carrying such notice may not be copied, modified, or distributed
// except according to those terms.
//! Definitions for native 802.11 miniport driver specifications.
use shared::basetsd::UINT8;
use shared::minwindef::{UCHAR, ULONG, USHORT};
use shared::ntddndis::{
    NDIS_OBJECT_HEADER, NDIS_PACKET_TYPE_ALL_MULTICAST, NDIS_PACKET_TYPE_BROADCAST,
    NDIS_PACKET_TYPE_DIRECTED, NDIS_PACKET_TYPE_MULTICAST, NDIS_PACKET_TYPE_PROMISCUOUS,
    NDIS_STATUS,
};
use shared::wlantypes::{
    DOT11_AUTH_ALGORITHM, DOT11_AUTH_CIPHER_PAIR, DOT11_BSS_TYPE, DOT11_CIPHER_ALGORITHM,
    DOT11_SSID, PDOT11_AUTH_CIPHER_PAIR,
};
use um::winnt::{BOOLEAN, HANDLE, LARGE_INTEGER, LONG, PVOID, ULONGLONG};
pub type DOT11_MAC_ADDRESS = [UCHAR; 6];
pub type PDOT11_MAC_ADDRESS = *mut DOT11_MAC_ADDRESS;
STRUCT!{struct DOT11_BSSID_LIST {
    Header: NDIS_OBJECT_HEADER,
    uNumOfEntries: ULONG,
    uTotalNumOfEntries: ULONG,
    BSSIDs: [DOT11_MAC_ADDRESS; 1],
}}
pub type PDOT11_BSSID_LIST = *mut DOT11_BSSID_LIST;
pub const DOT11_BSSID_LIST_REVISION_1: UCHAR = 1;
pub const DOT11_HESSID_LENGTH: usize = 6;
pub type DOT11_HESSID = [UCHAR; DOT11_HESSID_LENGTH];
pub type PDOT11_HESSID = *mut DOT11_HESSID;
ENUM!{enum DOT11_PHY_TYPE {
    dot11_phy_type_unknown = 0,
    dot11_phy_type_any = dot11_phy_type_unknown,
    dot11_phy_type_fhss = 1,
    dot11_phy_type_dsss = 2,
    dot11_phy_type_irbaseband = 3,
    dot11_phy_type_ofdm = 4,
    dot11_phy_type_hrdsss = 5,
    dot11_phy_type_erp = 6,
    dot11_phy_type_ht = 7,
    dot11_phy_type_vht = 8,
    dot11_phy_type_dmg = 9,
    dot11_phy_type_IHV_start = 0x80000000,
    dot11_phy_type_IHV_end = 0xffffffff,
}}
pub type PDOT11_PHY_TYPE = *mut DOT11_PHY_TYPE;
pub const DOT11_RATE_SET_MAX_LENGTH: usize = 126;
STRUCT!{struct DOT11_RATE_SET {
    uRateSetLength: ULONG,
    ucRateSet: [UCHAR; DOT11_RATE_SET_MAX_LENGTH],
}}
pub type PDOT11_RATE_SET = *mut DOT11_RATE_SET;
pub type DOT11_COUNTRY_OR_REGION_STRING = [UCHAR; 3];
pub type PDOT11_COUNTRY_OR_REGION_STRING = *mut DOT11_COUNTRY_OR_REGION_STRING;
pub type DOT11_DIALOG_TOKEN = UCHAR;
pub type DOT11_WFD_STATUS_CODE = UCHAR;
pub type DOT11_WFD_MINOR_REASON_CODE = UCHAR;
pub type DOT11_WFD_SERVICE_HASH = [UCHAR; 6];
pub const DOT11_WFD_SERVICE_NAME_MAX_LENGTH: usize = 255;
pub const DOT11_WFD_APS2_SERVICE_TYPE_MAX_LENGTH: usize = 21;
pub const DOT11_WFD_ASP2_INSTANCE_NAME_MAX_LENGTH: usize = 63;
pub const DOT11_WFD_SERVICE_INFORMATION_MAX_LENGTH: usize = 65535;
pub const DOT11_MAX_REQUESTED_SERVICE_INFORMATION_LENGTH: usize = 255;
pub const DOT11_WFD_SESSION_INFO_MAX_LENGTH: usize = 144;
STRUCT!{struct DOT11_WFD_SESSION_INFO {
    uSessionInfoLength: USHORT,
    ucSessionInfo: [UCHAR; DOT11_WFD_SESSION_INFO_MAX_LENGTH],
}}
pub type PDOT11_WFD_SESSION_INFO = *mut DOT11_WFD_SESSION_INFO;
pub const NDIS_PACKET_TYPE_MEDIA_SPECIFIC_MASK: u32 = 0x0fff0000;
pub const NDIS_PACKET_TYPE_802_11_DIRECTED_DATA: u32 = NDIS_PACKET_TYPE_DIRECTED;
pub const NDIS_PACKET_TYPE_802_11_BROADCAST_DATA: u32 = NDIS_PACKET_TYPE_BROADCAST;
pub const NDIS_PACKET_TYPE_802_11_MULTICAST_DATA: u32 = NDIS_PACKET_TYPE_MULTICAST;
pub const NDIS_PACKET_TYPE_802_11_ALL_MULTICAST_DATA: u32 = NDIS_PACKET_TYPE_ALL_MULTICAST;
pub const NDIS_PACKET_TYPE_802_11_PROMISCUOUS_DATA: u32 = NDIS_PACKET_TYPE_PROMISCUOUS;
pub const NDIS_PACKET_TYPE_802_11_RAW_DATA: u32 = 0x00010000;
pub const NDIS_PACKET_TYPE_802_11_DIRECTED_MGMT: u32 = 0x00020000;
pub const NDIS_PACKET_TYPE_802_11_BROADCAST_MGMT: u32 = 0x00040000;
pub const NDIS_PACKET_TYPE_802_11_MULTICAST_MGMT: u32 = 0x00080000;
pub const NDIS_PACKET_TYPE_802_11_ALL_MULTICAST_MGMT: u32 = 0x00100000;
pub const NDIS_PACKET_TYPE_802_11_PROMISCUOUS_MGMT: u32 = 0x00200000;
pub const NDIS_PACKET_TYPE_802_11_RAW_MGMT: u32 = 0x00400000;
pub const NDIS_PACKET_TYPE_802_11_DIRECTED_CTRL: u32 = 0x00800000;
pub const NDIS_PACKET_TYPE_802_11_BROADCAST_CTRL: u32 = 0x01000000;
pub const NDIS_PACKET_TYPE_802_11_PROMISCUOUS_CTRL: u32 = 0x02000000;
pub const NDIS_PACKET_TYPE_ALL_802_11_FILTERS: u32 = NDIS_PACKET_TYPE_DIRECTED
    | NDIS_PACKET_TYPE_MULTICAST | NDIS_PACKET_TYPE_ALL_MULTICAST | NDIS_PACKET_TYPE_BROADCAST
    | NDIS_PACKET_TYPE_PROMISCUOUS | NDIS_PACKET_TYPE_802_11_RAW_DATA
    | NDIS_PACKET_TYPE_802_11_DIRECTED_MGMT | NDIS_PACKET_TYPE_802_11_BROADCAST_MGMT
    | NDIS_PACKET_TYPE_802_11_MULTICAST_MGMT | NDIS_PACKET_TYPE_802_11_ALL_MULTICAST_MGMT
    | NDIS_PACKET_TYPE_802_11_PROMISCUOUS_MGMT | NDIS_PACKET_TYPE_802_11_RAW_MGMT
    | NDIS_PACKET_TYPE_802_11_DIRECTED_CTRL | NDIS_PACKET_TYPE_802_11_BROADCAST_CTRL
    | NDIS_PACKET_TYPE_802_11_PROMISCUOUS_CTRL;
pub const DOT11_MAX_PDU_SIZE: i32 = 2346;
pub const DOT11_MIN_PDU_SIZE: i32 = 256;
pub const DOT11_MAX_NUM_DEFAULT_KEY: i32 = 4;
pub const DOT11_MAX_NUM_DEFAULT_KEY_MFP: i32 = DOT11_MAX_NUM_DEFAULT_KEY + 2;
pub const OID_DOT11_NDIS_START: u32 = 0x0D010300;
pub const NWF_MANDATORY_OID: u32 = 0x01;
pub const NWF_OPTIONAL_OID: u32 = 0x02;
pub const NWF_OPERATIONAL_OID: u32 = 0x01;
pub const NWF_STATISTICS_OID: u32 = 0x02;
#[inline]
pub fn NWF_DEFINE_OID(Seq: u32, o: u32, m: u32) -> u32 {
    0x0E000000 | (o << 16) | (m << 8) | Seq
}
macro_rules! NWF_DEFINE_OID {
    ($Seq:expr, $o:expr, $m:expr) => { 0x0E000000 | $o << 16 | $m << 8 | $Seq };
}
pub const OID_DOT11_OFFLOAD_CAPABILITY: u32 = OID_DOT11_NDIS_START + 0;
pub const DOT11_HW_WEP_SUPPORTED_TX: u32 = 0x00000001;
pub const DOT11_HW_WEP_SUPPORTED_RX: u32 = 0x00000002;
pub const DOT11_HW_FRAGMENTATION_SUPPORTED: u32 = 0x00000004;
pub const DOT11_HW_DEFRAGMENTATION_SUPPORTED: u32 = 0x00000008;
pub const DOT11_HW_MSDU_AUTH_SUPPORTED_TX: u32 = 0x00000010;
pub const DOT11_HW_MSDU_AUTH_SUPPORTED_RX: u32 = 0x00000020;
pub const DOT11_CONF_ALGO_WEP_RC4: u32 = 0x00000001;
pub const DOT11_CONF_ALGO_TKIP: u32 = 0x00000002;
pub const DOT11_AUTH_ALGO_MICHAEL: u32 = 0x00000001;
STRUCT!{struct DOT11_OFFLOAD_CAPABILITY {
    uReserved: ULONG,
    uFlags: ULONG,
    uSupportedWEPAlgorithms: ULONG,
    uNumOfReplayWindows: ULONG,
    uMaxWEPKeyMappingLength: ULONG,
    uSupportedAuthAlgorithms: ULONG,
    uMaxAuthKeyMappingLength: ULONG,
}}
pub type PDOT11_OFFLOAD_CAPABILITY = *mut DOT11_OFFLOAD_CAPABILITY;
pub const OID_DOT11_CURRENT_OFFLOAD_CAPABILITY: u32 = OID_DOT11_NDIS_START + 1;
STRUCT!{struct DOT11_CURRENT_OFFLOAD_CAPABILITY {
    uReserved: ULONG,
    uFlags: ULONG,
}}
pub type PDOT11_CURRENT_OFFLOAD_CAPABILITY = *mut DOT11_CURRENT_OFFLOAD_CAPABILITY;
pub const OID_DOT11_WEP_OFFLOAD: u32 = OID_DOT11_NDIS_START + 2;
ENUM!{enum DOT11_OFFLOAD_TYPE {
    dot11_offload_type_wep = 1,
    dot11_offload_type_auth = 2,
}}
pub type PDOT11_OFFLOAD_TYPE = *mut DOT11_OFFLOAD_TYPE;
STRUCT!{struct DOT11_IV48_COUNTER {
    uIV32Counter: ULONG,
    usIV16Counter: USHORT,
}}
pub type PDOT11_IV48_COUNTER = *mut DOT11_IV48_COUNTER;
STRUCT!{struct DOT11_WEP_OFFLOAD {
    uReserved: ULONG,
    hOffloadContext: HANDLE,
    hOffload: HANDLE,
    dot11OffloadType: DOT11_OFFLOAD_TYPE,
    dwAlgorithm: ULONG,
    bRowIsOutbound: BOOLEAN,
    bUseDefault: BOOLEAN,
    uFlags: ULONG,
    ucMacAddress: [UCHAR; 6],
    uNumOfRWsOnPeer: ULONG,
    uNumOfRWsOnMe: ULONG,
    dot11IV48Counters: [DOT11_IV48_COUNTER; 16],
    usDot11RWBitMaps: [USHORT; 16],
    usKeyLength: USHORT,
    ucKey: [UCHAR; 1],
}}
pub type PDOT11_WEP_OFFLOAD = *mut DOT11_WEP_OFFLOAD;
pub const OID_DOT11_WEP_UPLOAD: u32 = OID_DOT11_NDIS_START + 3;
STRUCT!{struct DOT11_WEP_UPLOAD {
    uReserved: ULONG,
    dot11OffloadType: DOT11_OFFLOAD_TYPE,
    hOffload: HANDLE,
    uNumOfRWsUsed: ULONG,
    dot11IV48Counters: [DOT11_IV48_COUNTER; 16],
    usDot11RWBitMaps: [USHORT; 16],
}}
pub type PDOT11_WEP_UPLOAD = *mut DOT11_WEP_UPLOAD;
pub const OID_DOT11_DEFAULT_WEP_OFFLOAD: u32 = OID_DOT11_NDIS_START + 4;
ENUM!{enum DOT11_KEY_DIRECTION {
    dot11_key_direction_both = 1,
    dot11_key_direction_inbound = 2,
    dot11_key_direction_outbound = 3,
}}
pub type PDOT11_KEY_DIRECTION = *mut DOT11_KEY_DIRECTION;
STRUCT!{struct DOT11_DEFAULT_WEP_OFFLOAD {
    uReserved: ULONG,
    hOffloadContext: HANDLE,
    hOffload: HANDLE,
    dwIndex: ULONG,
    dot11OffloadType: DOT11_OFFLOAD_TYPE,
    dwAlgorithm: ULONG,
    uFlags: ULONG,
    dot11KeyDirection: DOT11_KEY_DIRECTION,
    ucMacAddress: [UCHAR; 6],
    uNumOfRWsOnMe: ULONG,
    dot11IV48Counters: [DOT11_IV48_COUNTER; 16],
    usDot11RWBitMaps: [USHORT; 16],
    usKeyLength: USHORT,
    ucKey: [UCHAR; 1],
}}
pub type PDOT11_DEFAULT_WEP_OFFLOAD = *mut DOT11_DEFAULT_WEP_OFFLOAD;
pub const OID_DOT11_DEFAULT_WEP_UPLOAD: u32 = OID_DOT11_NDIS_START + 5;
STRUCT!{struct DOT11_DEFAULT_WEP_UPLOAD {
    uReserved: ULONG,
    dot11OffloadType: DOT11_OFFLOAD_TYPE,
    hOffload: HANDLE,
    uNumOfRWsUsed: ULONG,
    dot11IV48Counters: [DOT11_IV48_COUNTER; 16],
    usDot11RWBitMaps: [USHORT; 16],
}}
pub type PDOT11_DEFAULT_WEP_UPLOAD = *mut DOT11_DEFAULT_WEP_UPLOAD;
pub const OID_DOT11_MPDU_MAX_LENGTH: u32 = OID_DOT11_NDIS_START + 6;
pub const OID_DOT11_OPERATION_MODE_CAPABILITY: u32 = OID_DOT11_NDIS_START + 7;
pub const DOT11_OPERATION_MODE_UNKNOWN: ULONG = 0x00000000;
pub const DOT11_OPERATION_MODE_STATION: ULONG = 0x00000001;
pub const DOT11_OPERATION_MODE_AP: ULONG = 0x00000002;
pub const DOT11_OPERATION_MODE_EXTENSIBLE_STATION: ULONG = 0x00000004;
pub const DOT11_OPERATION_MODE_EXTENSIBLE_AP: ULONG = 0x00000008;
pub const DOT11_OPERATION_MODE_WFD_DEVICE: ULONG = 0x00000010;
pub const DOT11_OPERATION_MODE_WFD_GROUP_OWNER: ULONG = 0x00000020;
pub const DOT11_OPERATION_MODE_WFD_CLIENT: ULONG = 0x00000040;
pub const DOT11_OPERATION_MODE_MANUFACTURING: ULONG = 0x40000000;
pub const DOT11_OPERATION_MODE_NETWORK_MONITOR: ULONG = 0x80000000;
STRUCT!{struct DOT11_OPERATION_MODE_CAPABILITY {
    uReserved: ULONG,
    uMajorVersion: ULONG,
    uMinorVersion: ULONG,
    uNumOfTXBuffers: ULONG,
    uNumOfRXBuffers: ULONG,
    uOpModeCapability: ULONG,
}}
pub type PDOT11_OPERATION_MODE_CAPABILITY = *mut DOT11_OPERATION_MODE_CAPABILITY;
pub const OID_DOT11_CURRENT_OPERATION_MODE: u32 = OID_DOT11_NDIS_START + 8;
STRUCT!{struct DOT11_CURRENT_OPERATION_MODE {
    uReserved: ULONG,
    uCurrentOpMode: ULONG,
}}
pub type PDOT11_CURRENT_OPERATION_MODE = *mut DOT11_CURRENT_OPERATION_MODE;
pub const OID_DOT11_CURRENT_PACKET_FILTER: u32 = OID_DOT11_NDIS_START + 9;
pub const DOT11_PACKET_TYPE_DIRECTED_CTRL: u32 = 0x00000001;
pub const DOT11_PACKET_TYPE_DIRECTED_MGMT: u32 = 0x00000002;
pub const DOT11_PACKET_TYPE_DIRECTED_DATA: u32 = 0x00000004;
pub const DOT11_PACKET_TYPE_MULTICAST_CTRL: u32 = 0x00000008;
pub const DOT11_PACKET_TYPE_MULTICAST_MGMT: u32 = 0x00000010;
pub const DOT11_PACKET_TYPE_MULTICAST_DATA: u32 = 0x00000020;
pub const DOT11_PACKET_TYPE_BROADCAST_CTRL: u32 = 0x00000040;
pub const DOT11_PACKET_TYPE_BROADCAST_MGMT: u32 = 0x00000080;
pub const DOT11_PACKET_TYPE_BROADCAST_DATA: u32 = 0x00000100;
pub const DOT11_PACKET_TYPE_PROMISCUOUS_CTRL: u32 = 0x00000200;
pub const DOT11_PACKET_TYPE_PROMISCUOUS_MGMT: u32 = 0x00000400;
pub const DOT11_PACKET_TYPE_PROMISCUOUS_DATA: u32 = 0x00000800;
pub const DOT11_PACKET_TYPE_ALL_MULTICAST_CTRL: u32 = 0x00001000;
pub const DOT11_PACKET_TYPE_ALL_MULTICAST_MGMT: u32 = 0x00002000;
pub const DOT11_PACKET_TYPE_ALL_MULTICAST_DATA: u32 = 0x00004000;
pub const DOT11_PACKET_TYPE_RESERVED: u32 = !(DOT11_PACKET_TYPE_DIRECTED_CTRL
    | DOT11_PACKET_TYPE_DIRECTED_MGMT | DOT11_PACKET_TYPE_DIRECTED_DATA
    | DOT11_PACKET_TYPE_MULTICAST_CTRL | DOT11_PACKET_TYPE_MULTICAST_MGMT
    | DOT11_PACKET_TYPE_MULTICAST_DATA | DOT11_PACKET_TYPE_BROADCAST_CTRL
    | DOT11_PACKET_TYPE_BROADCAST_MGMT | DOT11_PACKET_TYPE_BROADCAST_DATA
    | DOT11_PACKET_TYPE_PROMISCUOUS_CTRL | DOT11_PACKET_TYPE_PROMISCUOUS_MGMT
    | DOT11_PACKET_TYPE_PROMISCUOUS_DATA | DOT11_PACKET_TYPE_ALL_MULTICAST_CTRL
    | DOT11_PACKET_TYPE_ALL_MULTICAST_MGMT | DOT11_PACKET_TYPE_ALL_MULTICAST_DATA | 0);
pub const OID_DOT11_ATIM_WINDOW: u32 = OID_DOT11_NDIS_START + 10;
pub const OID_DOT11_SCAN_REQUEST: u32 = OID_DOT11_NDIS_START + 11;
ENUM!{enum DOT11_SCAN_TYPE {
    dot11_scan_type_active = 1,
    dot11_scan_type_passive = 2,
    dot11_scan_type_auto = 3,
    dot11_scan_type_forced = 0x80000000,
}}
pub type PDOT11_SCAN_TYPE = *mut DOT11_SCAN_TYPE;
STRUCT!{struct DOT11_SCAN_REQUEST {
    dot11BSSType: DOT11_BSS_TYPE,
    dot11BSSID: DOT11_MAC_ADDRESS,
    dot11SSID: DOT11_SSID,
    dot11ScanType: DOT11_SCAN_TYPE,
    bRestrictedScan: BOOLEAN,
    bUseRequestIE: BOOLEAN,
    uRequestIDsOffset: ULONG,
    uNumOfRequestIDs: ULONG,
    uPhyTypesOffset: ULONG,
    uNumOfPhyTypes: ULONG,
    uIEsOffset: ULONG,
    uIEsLength: ULONG,
    ucBuffer: [UCHAR; 1],
}}
pub type PDOT11_SCAN_REQUEST = *mut DOT11_SCAN_REQUEST;
ENUM!{enum CH_DESCRIPTION_TYPE {
    ch_description_type_logical = 1,
    ch_description_type_center_frequency = 2,
    ch_description_type_phy_specific = 3,
}}
pub type PCH_DESCRIPTION_TYPE = *mut CH_DESCRIPTION_TYPE;
STRUCT!{struct DOT11_PHY_TYPE_INFO {
    dot11PhyType: DOT11_PHY_TYPE,
    bUseParameters: BOOLEAN,
    uProbeDelay: ULONG,
    uMinChannelTime: ULONG,
    uMaxChannelTime: ULONG,
    ChDescriptionType: CH_DESCRIPTION_TYPE,
    uChannelListSize: ULONG,
    ucChannelListBuffer: [UCHAR; 1],
}}
pub type PDOT11_PHY_TYPE_INFO = *mut DOT11_PHY_TYPE_INFO;
STRUCT!{struct DOT11_SCAN_REQUEST_V2 {
    dot11BSSType: DOT11_BSS_TYPE,
    dot11BSSID: DOT11_MAC_ADDRESS,
    dot11ScanType: DOT11_SCAN_TYPE,
    bRestrictedScan: BOOLEAN,
    udot11SSIDsOffset: ULONG,
    uNumOfdot11SSIDs: ULONG,
    bUseRequestIE: BOOLEAN,
    uRequestIDsOffset: ULONG,
    uNumOfRequestIDs: ULONG,
    uPhyTypeInfosOffset: ULONG,
    uNumOfPhyTypeInfos: ULONG,
    uIEsOffset: ULONG,
    uIEsLength: ULONG,
    ucBuffer: [UCHAR; 1],
}}
pub type PDOT11_SCAN_REQUEST_V2 = *mut DOT11_SCAN_REQUEST_V2;
pub const OID_DOT11_CURRENT_PHY_TYPE: u32 = OID_DOT11_NDIS_START + 12;
STRUCT!{struct DOT11_PHY_TYPE_LIST {
    Header: NDIS_OBJECT_HEADER,
    uNumOfEntries: ULONG,
    uTotalNumOfEntries: ULONG,
    dot11PhyType: [DOT11_PHY_TYPE; 1],
}}
pub type PDOT11_PHY_TYPE_LIST = *mut DOT11_PHY_TYPE_LIST;
pub const DOT11_PHY_TYPE_LIST_REVISION_1: UCHAR = 1;
pub const OID_DOT11_JOIN_REQUEST: u32 = OID_DOT11_NDIS_START + 13;
pub const DOT11_CAPABILITY_INFO_ESS: USHORT = 0x0001;
pub const DOT11_CAPABILITY_INFO_IBSS: USHORT = 0x0002;
pub const DOT11_CAPABILITY_INFO_CF_POLLABLE: USHORT = 0x0004;
pub const DOT11_CAPABILITY_INFO_CF_POLL_REQ: USHORT = 0x0008;
pub const DOT11_CAPABILITY_INFO_PRIVACY: USHORT = 0x0010;
pub const DOT11_CAPABILITY_SHORT_PREAMBLE: USHORT = 0x0020;
pub const DOT11_CAPABILITY_PBCC: USHORT = 0x0040;
pub const DOT11_CAPABILITY_CHANNEL_AGILITY: USHORT = 0x0080;
pub const DOT11_CAPABILITY_SHORT_SLOT_TIME: USHORT = 0x0400;
pub const DOT11_CAPABILITY_DSSSOFDM: USHORT = 0x2000;
STRUCT!{struct DOT11_BSS_DESCRIPTION {
    uReserved: ULONG,
    dot11BSSID: DOT11_MAC_ADDRESS,
    dot11BSSType: DOT11_BSS_TYPE,
    usBeaconPeriod: USHORT,
    ullTimestamp: ULONGLONG,
    usCapabilityInformation: USHORT,
    uBufferLength: ULONG,
    ucBuffer: [UCHAR; 1],
}}
pub type PDOT11_BSS_DESCRIPTION = *mut DOT11_BSS_DESCRIPTION;
STRUCT!{struct DOT11_JOIN_REQUEST {
    uJoinFailureTimeout: ULONG,
    OperationalRateSet: DOT11_RATE_SET,
    uChCenterFrequency: ULONG,
    dot11BSSDescription: DOT11_BSS_DESCRIPTION,
}}
pub type PDOT11_JOIN_REQUEST = *mut DOT11_JOIN_REQUEST;
pub const OID_DOT11_START_REQUEST: u32 = OID_DOT11_NDIS_START + 14;
STRUCT!{struct DOT11_START_REQUEST {
    uStartFailureTimeout: ULONG,
    OperationalRateSet: DOT11_RATE_SET,
    uChCenterFrequency: ULONG,
    dot11BSSDescription: DOT11_BSS_DESCRIPTION,
}}
pub type PDOT11_START_REQUEST = *mut DOT11_START_REQUEST;
pub const OID_DOT11_UPDATE_IE: u32 = OID_DOT11_NDIS_START + 15;
ENUM!{enum DOT11_UPDATE_IE_OP {
    dot11_update_ie_op_create_replace = 1,
    dot11_update_ie_op_delete = 2,
}}
pub type PDOT11_UPDATE_IE_OP = *mut DOT11_UPDATE_IE_OP;
STRUCT!{struct DOT11_UPDATE_IE {
    dot11UpdateIEOp: DOT11_UPDATE_IE_OP,
    uBufferLength: ULONG,
    ucBuffer: [UCHAR; 1],
}}
pub type PDOT11_UPDATE_IE = *mut DOT11_UPDATE_IE;
pub const OID_DOT11_RESET_REQUEST: u32 = OID_DOT11_NDIS_START + 16;
ENUM!{enum DOT11_RESET_TYPE {
    dot11_reset_type_phy = 1,
    dot11_reset_type_mac = 2,
    dot11_reset_type_phy_and_mac = 3,
}}
pub type PDOT11_RESET_TYPE = *mut DOT11_RESET_TYPE;
STRUCT!{struct DOT11_RESET_REQUEST {
    dot11ResetType: DOT11_RESET_TYPE,
    dot11MacAddress: DOT11_MAC_ADDRESS,
    bSetDefaultMIB: BOOLEAN,
}}
pub type PDOT11_RESET_REQUEST = *mut DOT11_RESET_REQUEST;
pub const OID_DOT11_NIC_POWER_STATE: u32 = OID_DOT11_NDIS_START + 17;
pub const OID_DOT11_OPTIONAL_CAPABILITY: u32 = OID_DOT11_NDIS_START + 18;
STRUCT!{struct DOT11_OPTIONAL_CAPABILITY {
    uReserved: ULONG,
    bDot11PCF: BOOLEAN,
    bDot11PCFMPDUTransferToPC: BOOLEAN,
    bStrictlyOrderedServiceClass: BOOLEAN,
}}
pub type PDOT11_OPTIONAL_CAPABILITY = *mut DOT11_OPTIONAL_CAPABILITY;
pub const OID_DOT11_CURRENT_OPTIONAL_CAPABILITY: u32 = OID_DOT11_NDIS_START + 19;
STRUCT!{struct DOT11_CURRENT_OPTIONAL_CAPABILITY {
    uReserved: ULONG,
    bDot11CFPollable: BOOLEAN,
    bDot11PCF: BOOLEAN,
    bDot11PCFMPDUTransferToPC: BOOLEAN,
    bStrictlyOrderedServiceClass: BOOLEAN,
}}
pub type PDOT11_CURRENT_OPTIONAL_CAPABILITY = *mut DOT11_CURRENT_OPTIONAL_CAPABILITY;
pub const OID_DOT11_STATION_ID: u32 = OID_DOT11_NDIS_START + 20;
pub const OID_DOT11_MEDIUM_OCCUPANCY_LIMIT: u32 = OID_DOT11_NDIS_START + 21;
pub const OID_DOT11_CF_POLLABLE: u32 = OID_DOT11_NDIS_START + 22;
pub const OID_DOT11_CFP_PERIOD: u32 = OID_DOT11_NDIS_START + 23;
pub const OID_DOT11_CFP_MAX_DURATION: u32 = OID_DOT11_NDIS_START + 24;
pub const OID_DOT11_POWER_MGMT_MODE: u32 = OID_DOT11_NDIS_START + 25;
ENUM!{enum DOT11_POWER_MODE {
    dot11_power_mode_unknown = 0,
    dot11_power_mode_active = 1,
    dot11_power_mode_powersave = 2,
}}
pub type PDOT11_POWER_MODE = *mut DOT11_POWER_MODE;
pub const DOT11_POWER_SAVE_LEVEL_MAX_PSP: ULONG = 1;
pub const DOT11_POWER_SAVE_LEVEL_FAST_PSP: ULONG = 2;
STRUCT!{struct DOT11_POWER_MGMT_MODE {
    dot11PowerMode: DOT11_POWER_MODE,
    uPowerSaveLevel: ULONG,
    usListenInterval: USHORT,
    usAID: USHORT,
    bReceiveDTIMs: BOOLEAN,
}}
pub type PDOT11_POWER_MGMT_MODE = *mut DOT11_POWER_MGMT_MODE;
pub const OID_DOT11_OPERATIONAL_RATE_SET: u32 = OID_DOT11_NDIS_START + 26;
pub const OID_DOT11_BEACON_PERIOD: u32 = OID_DOT11_NDIS_START + 27;
pub const OID_DOT11_DTIM_PERIOD: u32 = OID_DOT11_NDIS_START + 28;
pub const OID_DOT11_WEP_ICV_ERROR_COUNT: u32 = OID_DOT11_NDIS_START + 29;
pub const OID_DOT11_MAC_ADDRESS: u32 = OID_DOT11_NDIS_START + 30;
pub const OID_DOT11_RTS_THRESHOLD: u32 = OID_DOT11_NDIS_START + 31;
pub const OID_DOT11_SHORT_RETRY_LIMIT: u32 = OID_DOT11_NDIS_START + 32;
pub const OID_DOT11_LONG_RETRY_LIMIT: u32 = OID_DOT11_NDIS_START + 33;
pub const OID_DOT11_FRAGMENTATION_THRESHOLD: u32 = OID_DOT11_NDIS_START + 34;
pub const OID_DOT11_MAX_TRANSMIT_MSDU_LIFETIME: u32 = OID_DOT11_NDIS_START + 35;
pub const OID_DOT11_MAX_RECEIVE_LIFETIME: u32 = OID_DOT11_NDIS_START + 36;
pub const OID_DOT11_COUNTERS_ENTRY: u32 = OID_DOT11_NDIS_START + 37;
STRUCT!{struct DOT11_COUNTERS_ENTRY {
    uTransmittedFragmentCount: ULONG,
    uMulticastTransmittedFrameCount: ULONG,
    uFailedCount: ULONG,
    uRetryCount: ULONG,
    uMultipleRetryCount: ULONG,
    uFrameDuplicateCount: ULONG,
    uRTSSuccessCount: ULONG,
    uRTSFailureCount: ULONG,
    uACKFailureCount: ULONG,
    uReceivedFragmentCount: ULONG,
    uMulticastReceivedFrameCount: ULONG,
    uFCSErrorCount: ULONG,
    uTransmittedFrameCount: ULONG,
}}
pub type PDOT11_COUNTERS_ENTRY = *mut DOT11_COUNTERS_ENTRY;
pub const OID_DOT11_SUPPORTED_PHY_TYPES: u32 = OID_DOT11_NDIS_START + 38;
STRUCT!{struct DOT11_SUPPORTED_PHY_TYPES {
    uNumOfEntries: ULONG,
    uTotalNumOfEntries: ULONG,
    dot11PHYType: [DOT11_PHY_TYPE; 1],
}}
pub type PDOT11_SUPPORTED_PHY_TYPES = *mut DOT11_SUPPORTED_PHY_TYPES;
pub const OID_DOT11_CURRENT_REG_DOMAIN: u32 = OID_DOT11_NDIS_START + 39;
pub const DOT11_REG_DOMAIN_OTHER: ULONG = 0x00000000;
pub const DOT11_REG_DOMAIN_FCC: ULONG = 0x00000010;
pub const DOT11_REG_DOMAIN_DOC: ULONG = 0x00000020;
pub const DOT11_REG_DOMAIN_ETSI: ULONG = 0x00000030;
pub const DOT11_REG_DOMAIN_SPAIN: ULONG = 0x00000031;
pub const DOT11_REG_DOMAIN_FRANCE: ULONG = 0x00000032;
pub const DOT11_REG_DOMAIN_MKK: ULONG = 0x00000040;
pub const OID_DOT11_TEMP_TYPE: u32 = OID_DOT11_NDIS_START + 40;
ENUM!{enum DOT11_TEMP_TYPE {
    dot11_temp_type_unknown = 0,
    dot11_temp_type_1 = 1,
    dot11_temp_type_2 = 2,
}}
pub type PDOT11_TEMP_TYPE = *mut DOT11_TEMP_TYPE;
pub const OID_DOT11_CURRENT_TX_ANTENNA: u32 = OID_DOT11_NDIS_START + 41;
pub const OID_DOT11_DIVERSITY_SUPPORT: u32 = OID_DOT11_NDIS_START + 42;
ENUM!{enum DOT11_DIVERSITY_SUPPORT {
    dot11_diversity_support_unknown = 0,
    dot11_diversity_support_fixedlist = 1,
    dot11_diversity_support_notsupported = 2,
    dot11_diversity_support_dynamic = 3,
}}
pub type PDOT11_DIVERSITY_SUPPORT = *mut DOT11_DIVERSITY_SUPPORT;
pub const OID_DOT11_CURRENT_RX_ANTENNA: u32 = OID_DOT11_NDIS_START + 43;
pub const OID_DOT11_SUPPORTED_POWER_LEVELS: u32 = OID_DOT11_NDIS_START + 44;
STRUCT!{struct DOT11_SUPPORTED_POWER_LEVELS {
    uNumOfSupportedPowerLevels: ULONG,
    uTxPowerLevelValues: [ULONG; 8],
}}
pub type PDOT11_SUPPORTED_POWER_LEVELS = *mut DOT11_SUPPORTED_POWER_LEVELS;
pub const OID_DOT11_CURRENT_TX_POWER_LEVEL: u32 = OID_DOT11_NDIS_START + 45;
pub const OID_DOT11_HOP_TIME: u32 = OID_DOT11_NDIS_START + 46;
pub const OID_DOT11_CURRENT_CHANNEL_NUMBER: u32 = OID_DOT11_NDIS_START + 47;
pub const OID_DOT11_MAX_DWELL_TIME: u32 = OID_DOT11_NDIS_START + 48;
pub const OID_DOT11_CURRENT_DWELL_TIME: u32 = OID_DOT11_NDIS_START + 49;
pub const OID_DOT11_CURRENT_SET: u32 = OID_DOT11_NDIS_START + 50;
pub const OID_DOT11_CURRENT_PATTERN: u32 = OID_DOT11_NDIS_START + 51;
pub const OID_DOT11_CURRENT_INDEX: u32 = OID_DOT11_NDIS_START + 52;
pub const OID_DOT11_CURRENT_CHANNEL: u32 = OID_DOT11_NDIS_START + 53;
pub const OID_DOT11_CCA_MODE_SUPPORTED: u32 = OID_DOT11_NDIS_START + 54;
pub const DOT11_CCA_MODE_ED_ONLY: ULONG = 0x00000001;
pub const DOT11_CCA_MODE_CS_ONLY: ULONG = 0x00000002;
pub const DOT11_CCA_MODE_ED_and_CS: ULONG = 0x00000004;
pub const DOT11_CCA_MODE_CS_WITH_TIMER: ULONG = 0x00000008;
pub const DOT11_CCA_MODE_HRCS_AND_ED: ULONG = 0x00000010;
pub const OID_DOT11_CURRENT_CCA_MODE: u32 = OID_DOT11_NDIS_START + 55;
pub const OID_DOT11_ED_THRESHOLD: u32 = OID_DOT11_NDIS_START + 56;
pub const OID_DOT11_CCA_WATCHDOG_TIMER_MAX: u32 = OID_DOT11_NDIS_START + 57;
pub const OID_DOT11_CCA_WATCHDOG_COUNT_MAX: u32 = OID_DOT11_NDIS_START + 58;
pub const OID_DOT11_CCA_WATCHDOG_TIMER_MIN: u32 = OID_DOT11_NDIS_START + 59;
pub const OID_DOT11_CCA_WATCHDOG_COUNT_MIN: u32 = OID_DOT11_NDIS_START + 60;
pub const OID_DOT11_REG_DOMAINS_SUPPORT_VALUE: u32 = OID_DOT11_NDIS_START + 61;
STRUCT!{struct DOT11_REG_DOMAIN_VALUE {
    uRegDomainsSupportIndex: ULONG,
    uRegDomainsSupportValue: ULONG,
}}
pub type PDOT11_REG_DOMAIN_VALUE = *mut DOT11_REG_DOMAIN_VALUE;
STRUCT!{struct DOT11_REG_DOMAINS_SUPPORT_VALUE {
    uNumOfEntries: ULONG,
    uTotalNumOfEntries: ULONG,
    dot11RegDomainValue: [DOT11_REG_DOMAIN_VALUE; 1],
}}
pub type PDOT11_REG_DOMAINS_SUPPORT_VALUE = *mut DOT11_REG_DOMAINS_SUPPORT_VALUE;
pub const OID_DOT11_SUPPORTED_TX_ANTENNA: u32 = OID_DOT11_NDIS_START + 62;
STRUCT!{struct DOT11_SUPPORTED_ANTENNA {
    uAntennaListIndex: ULONG,
    bSupportedAntenna: BOOLEAN,
}}
pub type PDOT11_SUPPORTED_ANTENNA = *mut DOT11_SUPPORTED_ANTENNA;
STRUCT!{struct DOT11_SUPPORTED_ANTENNA_LIST {
    uNumOfEntries: ULONG,
    uTotalNumOfEntries: ULONG,
    dot11SupportedAntenna: [DOT11_SUPPORTED_ANTENNA; 1],
}}
pub type PDOT11_SUPPORTED_ANTENNA_LIST = *mut DOT11_SUPPORTED_ANTENNA_LIST;
pub const OID_DOT11_SUPPORTED_RX_ANTENNA: u32 = OID_DOT11_NDIS_START + 63;
pub const OID_DOT11_DIVERSITY_SELECTION_RX: u32 = OID_DOT11_NDIS_START + 64;
STRUCT!{struct DOT11_DIVERSITY_SELECTION_RX {
    uAntennaListIndex: ULONG,
    bDiversitySelectionRX: BOOLEAN,
}}
pub type PDOT11_DIVERSITY_SELECTION_RX = *mut DOT11_DIVERSITY_SELECTION_RX;
STRUCT!{struct DOT11_DIVERSITY_SELECTION_RX_LIST {
    uNumOfEntries: ULONG,
    uTotalNumOfEntries: ULONG,
    dot11DiversitySelectionRx: [DOT11_DIVERSITY_SELECTION_RX; 1],
}}
pub type PDOT11_DIVERSITY_SELECTION_RX_LIST = *mut DOT11_DIVERSITY_SELECTION_RX_LIST;
pub const OID_DOT11_SUPPORTED_DATA_RATES_VALUE: u32 = OID_DOT11_NDIS_START + 65;
pub const MAX_NUM_SUPPORTED_RATES: usize = 8;
pub const MAX_NUM_SUPPORTED_RATES_V2: usize = 255;
STRUCT!{struct DOT11_SUPPORTED_DATA_RATES_VALUE {
    ucSupportedTxDataRatesValue: [UCHAR; MAX_NUM_SUPPORTED_RATES],
    ucSupportedRxDataRatesValue: [UCHAR; MAX_NUM_SUPPORTED_RATES],
}}
pub type PDOT11_SUPPORTED_DATA_RATES_VALUE = *mut DOT11_SUPPORTED_DATA_RATES_VALUE;
STRUCT!{struct DOT11_SUPPORTED_DATA_RATES_VALUE_V2 {
    ucSupportedTxDataRatesValue: [UCHAR; MAX_NUM_SUPPORTED_RATES_V2],
    ucSupportedRxDataRatesValue: [UCHAR; MAX_NUM_SUPPORTED_RATES_V2],
}}
pub type PDOT11_SUPPORTED_DATA_RATES_VALUE_V2 = *mut DOT11_SUPPORTED_DATA_RATES_VALUE_V2;
pub type DOT11_SUPPORTED_DATA_RATES_VALUE_V1 = DOT11_SUPPORTED_DATA_RATES_VALUE_V2;
pub type PDOT11_SUPPORTED_DATA_RATES_VALUE_V1 = *mut DOT11_SUPPORTED_DATA_RATES_VALUE_V2;
pub const OID_DOT11_CURRENT_FREQUENCY: u32 = OID_DOT11_NDIS_START + 66;
pub const OID_DOT11_TI_THRESHOLD: u32 = OID_DOT11_NDIS_START + 67;
pub const OID_DOT11_FREQUENCY_BANDS_SUPPORTED: u32 = OID_DOT11_NDIS_START + 68;
pub const DOT11_FREQUENCY_BANDS_LOWER: u32 = 0x00000001;
pub const DOT11_FREQUENCY_BANDS_MIDDLE: u32 = 0x00000002;
pub const DOT11_FREQUENCY_BANDS_UPPER: u32 = 0x00000004;
pub const OID_DOT11_SHORT_PREAMBLE_OPTION_IMPLEMENTED: u32 = OID_DOT11_NDIS_START + 69;
pub const OID_DOT11_PBCC_OPTION_IMPLEMENTED: u32 = OID_DOT11_NDIS_START + 70;
pub const OID_DOT11_CHANNEL_AGILITY_PRESENT: u32 = OID_DOT11_NDIS_START + 71;
pub const OID_DOT11_CHANNEL_AGILITY_ENABLED: u32 = OID_DOT11_NDIS_START + 72;
pub const OID_DOT11_HR_CCA_MODE_SUPPORTED: u32 = OID_DOT11_NDIS_START + 73;
pub const DOT11_HR_CCA_MODE_ED_ONLY: ULONG = 0x00000001;
pub const DOT11_HR_CCA_MODE_CS_ONLY: ULONG = 0x00000002;
pub const DOT11_HR_CCA_MODE_CS_AND_ED: ULONG = 0x00000004;
pub const DOT11_HR_CCA_MODE_CS_WITH_TIMER: ULONG = 0x00000008;
pub const DOT11_HR_CCA_MODE_HRCS_AND_ED: ULONG = 0x00000010;
pub const OID_DOT11_MULTI_DOMAIN_CAPABILITY_IMPLEMENTED: u32 = OID_DOT11_NDIS_START + 74;
pub const OID_DOT11_MULTI_DOMAIN_CAPABILITY_ENABLED: u32 = OID_DOT11_NDIS_START + 75;
pub const OID_DOT11_COUNTRY_STRING: u32 = OID_DOT11_NDIS_START + 76;
STRUCT!{struct DOT11_MULTI_DOMAIN_CAPABILITY_ENTRY {
    uMultiDomainCapabilityIndex: ULONG,
    uFirstChannelNumber: ULONG,
    uNumberOfChannels: ULONG,
    lMaximumTransmitPowerLevel: LONG,
}}
pub type PDOT11_MULTI_DOMAIN_CAPABILITY_ENTRY = *mut DOT11_MULTI_DOMAIN_CAPABILITY_ENTRY;
STRUCT!{struct DOT11_MD_CAPABILITY_ENTRY_LIST {
    uNumOfEntries: ULONG,
    uTotalNumOfEntries: ULONG,
    dot11MDCapabilityEntry: [DOT11_MULTI_DOMAIN_CAPABILITY_ENTRY; 1],
}}
pub type PDOT11_MD_CAPABILITY_ENTRY_LIST = *mut DOT11_MD_CAPABILITY_ENTRY_LIST;
pub const OID_DOT11_MULTI_DOMAIN_CAPABILITY: u32 = OID_DOT11_NDIS_START + 77;
pub const OID_DOT11_EHCC_PRIME_RADIX: u32 = OID_DOT11_NDIS_START + 78;
pub const OID_DOT11_EHCC_NUMBER_OF_CHANNELS_FAMILY_INDEX: u32 = OID_DOT11_NDIS_START + 79;
pub const OID_DOT11_EHCC_CAPABILITY_IMPLEMENTED: u32 = OID_DOT11_NDIS_START + 80;
pub const OID_DOT11_EHCC_CAPABILITY_ENABLED: u32 = OID_DOT11_NDIS_START + 81;
pub const OID_DOT11_HOP_ALGORITHM_ADOPTED: u32 = OID_DOT11_NDIS_START + 82;
ENUM!{enum DOT11_HOP_ALGO_ADOPTED {
    dot11_hop_algo_current = 0,
    dot11_hop_algo_hop_index = 1,
    dot11_hop_algo_hcc = 2,
}}
pub type PDOT11_HOP_ALGO_ADOPTED = *mut DOT11_HOP_ALGO_ADOPTED;
pub const OID_DOT11_RANDOM_TABLE_FLAG: u32 = OID_DOT11_NDIS_START + 83;
pub const OID_DOT11_NUMBER_OF_HOPPING_SETS: u32 = OID_DOT11_NDIS_START + 84;
pub const OID_DOT11_HOP_MODULUS: u32 = OID_DOT11_NDIS_START + 85;
pub const OID_DOT11_HOP_OFFSET: u32 = OID_DOT11_NDIS_START + 86;
pub const OID_DOT11_HOPPING_PATTERN: u32 = OID_DOT11_NDIS_START + 87;
STRUCT!{struct DOT11_HOPPING_PATTERN_ENTRY {
    uHoppingPatternIndex: ULONG,
    uRandomTableFieldNumber: ULONG,
}}
pub type PDOT11_HOPPING_PATTERN_ENTRY = *mut DOT11_HOPPING_PATTERN_ENTRY;
STRUCT!{struct DOT11_HOPPING_PATTERN_ENTRY_LIST {
    uNumOfEntries: ULONG,
    uTotalNumOfEntries: ULONG,
    dot11HoppingPatternEntry: [DOT11_HOPPING_PATTERN_ENTRY; 1],
}}
pub type PDOT11_HOPPING_PATTERN_ENTRY_LIST = *mut DOT11_HOPPING_PATTERN_ENTRY_LIST;
pub const OID_DOT11_RANDOM_TABLE_FIELD_NUMBER: u32 = OID_DOT11_NDIS_START + 88;
pub const OID_DOT11_WPA_TSC: u32 = OID_DOT11_NDIS_START + 89;
STRUCT!{struct DOT11_WPA_TSC {
    uReserved: ULONG,
    dot11OffloadType: DOT11_OFFLOAD_TYPE,
    hOffload: HANDLE,
    dot11IV48Counter: DOT11_IV48_COUNTER,
}}
pub type PDOT11_WPA_TSC = *mut DOT11_WPA_TSC;
pub const OID_DOT11_RSSI_RANGE: u32 = OID_DOT11_NDIS_START + 90;
STRUCT!{struct DOT11_RSSI_RANGE {
    dot11PhyType: DOT11_PHY_TYPE,
    uRSSIMin: ULONG,
    uRSSIMax: ULONG,
}}
pub type PDOT11_RSSI_RANGE = *mut DOT11_RSSI_RANGE;
pub const OID_DOT11_RF_USAGE: u32 = OID_DOT11_NDIS_START + 91;
pub const OID_DOT11_NIC_SPECIFIC_EXTENSION: u32 = OID_DOT11_NDIS_START + 92;
STRUCT!{struct DOT11_NIC_SPECIFIC_EXTENSION {
    uBufferLength: ULONG,
    uTotalBufferLength: ULONG,
    ucBuffer: [UCHAR; 1],
}}
pub type PDOT11_NIC_SPECIFIC_EXTENSION = *mut DOT11_NIC_SPECIFIC_EXTENSION;
pub const OID_DOT11_AP_JOIN_REQUEST: u32 = OID_DOT11_NDIS_START + 93;
STRUCT!{struct DOT11_AP_JOIN_REQUEST {
    uJoinFailureTimeout: ULONG,
    OperationalRateSet: DOT11_RATE_SET,
    uChCenterFrequency: ULONG,
    dot11BSSDescription: DOT11_BSS_DESCRIPTION,
}}
pub type PDOT11_AP_JOIN_REQUEST = *mut DOT11_AP_JOIN_REQUEST;
pub const OID_DOT11_ERP_PBCC_OPTION_IMPLEMENTED: u32 = OID_DOT11_NDIS_START + 94;
pub const OID_DOT11_ERP_PBCC_OPTION_ENABLED: u32 = OID_DOT11_NDIS_START + 95;
pub const OID_DOT11_DSSS_OFDM_OPTION_IMPLEMENTED: u32 = OID_DOT11_NDIS_START + 96;
pub const OID_DOT11_DSSS_OFDM_OPTION_ENABLED: u32 = OID_DOT11_NDIS_START + 97;
pub const OID_DOT11_SHORT_SLOT_TIME_OPTION_IMPLEMENTED: u32 = OID_DOT11_NDIS_START + 98;
pub const OID_DOT11_SHORT_SLOT_TIME_OPTION_ENABLED: u32 = OID_DOT11_NDIS_START + 99;
pub const OID_DOT11_MAX_MAC_ADDRESS_STATES: u32 = OID_DOT11_NDIS_START + 100;
pub const OID_DOT11_RECV_SENSITIVITY_LIST: u32 = OID_DOT11_NDIS_START + 101;
STRUCT!{struct DOT11_RECV_SENSITIVITY {
    ucDataRate: UCHAR,
    lRSSIMin: LONG,
    lRSSIMax: LONG,
}}
pub type PDOT11_RECV_SENSITIVITY = *mut DOT11_RECV_SENSITIVITY;
UNION!{union DOT11_RECV_SENSITIVITY_LIST_u {
    [u32; 1],
    dot11PhyType dot11PhyType_mut: DOT11_PHY_TYPE,
    uPhyId uPhyId_mut: ULONG,
}}
STRUCT!{struct DOT11_RECV_SENSITIVITY_LIST {
    u: DOT11_RECV_SENSITIVITY_LIST_u,
    uNumOfEntries: ULONG,
    uTotalNumOfEntries: ULONG,
    dot11RecvSensitivity: [DOT11_RECV_SENSITIVITY; 1],
}}
pub type PDOT11_RECV_SENSITIVITY_LIST = *mut DOT11_RECV_SENSITIVITY_LIST;
pub const OID_DOT11_WME_IMPLEMENTED: u32 = OID_DOT11_NDIS_START + 102;
pub const OID_DOT11_WME_ENABLED: u32 = OID_DOT11_NDIS_START + 103;
pub const OID_DOT11_WME_AC_PARAMETERS: u32 = OID_DOT11_NDIS_START + 104;
ENUM!{enum DOT11_AC_PARAM {
    dot11_AC_param_BE = 0,
    dot11_AC_param_BK = 1,
    dot11_AC_param_VI = 2,
    dot11_AC_param_VO = 3,
    dot11_AC_param_max = 4,
}}
pub type PDOT11_AC_PARAM = *mut DOT11_AC_PARAM;
STRUCT!{struct DOT11_WME_AC_PARAMETERS {
    ucAccessCategoryIndex: UCHAR,
    ucAIFSN: UCHAR,
    ucECWmin: UCHAR,
    ucECWmax: UCHAR,
    usTXOPLimit: USHORT,
}}
pub type PDOT11_WME_AC_PARAMETERS = *mut DOT11_WME_AC_PARAMETERS;
STRUCT!{struct DOT11_WME_AC_PARAMETERS_LIST {
    uNumOfEntries: ULONG,
    uTotalNumOfEntries: ULONG,
    dot11WMEACParameters: [DOT11_WME_AC_PARAMETERS; 1],
}}
pub type PDOT11_WME_AC_PARAMETERS_LIST = *mut DOT11_WME_AC_PARAMETERS_LIST;
pub const OID_DOT11_WME_UPDATE_IE: u32 = OID_DOT11_NDIS_START + 105;
STRUCT!{struct DOT11_WME_UPDATE_IE {
    uParamElemMinBeaconIntervals: ULONG,
    uWMEInfoElemOffset: ULONG,
    uWMEInfoElemLength: ULONG,
    uWMEParamElemOffset: ULONG,
    uWMEParamElemLength: ULONG,
    ucBuffer: [UCHAR; 1],
}}
pub type PDOT11_WME_UPDATE_IE = *mut DOT11_WME_UPDATE_IE;
pub const OID_DOT11_QOS_TX_QUEUES_SUPPORTED: u32 = OID_DOT11_NDIS_START + 106;
pub const OID_DOT11_QOS_TX_DURATION: u32 = OID_DOT11_NDIS_START + 107;
STRUCT!{struct DOT11_QOS_TX_DURATION {
    uNominalMSDUSize: ULONG,
    uMinPHYRate: ULONG,
    uDuration: ULONG,
}}
pub type PDOT11_QOS_TX_DURATION = *mut DOT11_QOS_TX_DURATION;
pub const OID_DOT11_QOS_TX_MEDIUM_TIME: u32 = OID_DOT11_NDIS_START + 108;
STRUCT!{struct DOT11_QOS_TX_MEDIUM_TIME {
    dot11PeerAddress: DOT11_MAC_ADDRESS,
    ucQoSPriority: UCHAR,
    uMediumTimeAdmited: ULONG,
}}
pub type PDOT11_QOS_TX_MEDIUM_TIME = *mut DOT11_QOS_TX_MEDIUM_TIME;
pub const OID_DOT11_SUPPORTED_OFDM_FREQUENCY_LIST: u32 = OID_DOT11_NDIS_START + 109;
STRUCT!{struct DOT11_SUPPORTED_OFDM_FREQUENCY {
    uCenterFrequency: ULONG,
}}
pub type PDOT11_SUPPORTED_OFDM_FREQUENCY = *mut DOT11_SUPPORTED_OFDM_FREQUENCY;
STRUCT!{struct DOT11_SUPPORTED_OFDM_FREQUENCY_LIST {
    uNumOfEntries: ULONG,
    uTotalNumOfEntries: ULONG,
    dot11SupportedOFDMFrequency: [DOT11_SUPPORTED_OFDM_FREQUENCY; 1],
}}
pub type PDOT11_SUPPORTED_OFDM_FREQUENCY_LIST = *mut DOT11_SUPPORTED_OFDM_FREQUENCY_LIST;
pub const OID_DOT11_SUPPORTED_DSSS_CHANNEL_LIST: u32 = OID_DOT11_NDIS_START + 110;
STRUCT!{struct DOT11_SUPPORTED_DSSS_CHANNEL {
    uChannel: ULONG,
}}
pub type PDOT11_SUPPORTED_DSSS_CHANNEL = *mut DOT11_SUPPORTED_DSSS_CHANNEL;
STRUCT!{struct DOT11_SUPPORTED_DSSS_CHANNEL_LIST {
    uNumOfEntries: ULONG,
    uTotalNumOfEntries: ULONG,
    dot11SupportedDSSSChannel: [DOT11_SUPPORTED_DSSS_CHANNEL; 1],
}}
pub type PDOT11_SUPPORTED_DSSS_CHANNEL_LIST = *mut DOT11_SUPPORTED_DSSS_CHANNEL_LIST;
STRUCT!{struct DOT11_BYTE_ARRAY {
    Header: NDIS_OBJECT_HEADER,
    uNumOfBytes: ULONG,
    uTotalNumOfBytes: ULONG,
    ucBuffer: [UCHAR; 1],
}}
pub type PDOT11_BYTE_ARRAY = *mut DOT11_BYTE_ARRAY;
pub const OID_DOT11_AUTO_CONFIG_ENABLED: u32 =
    NWF_DEFINE_OID!(120, NWF_OPERATIONAL_OID, NWF_MANDATORY_OID);
pub const DOT11_PHY_AUTO_CONFIG_ENABLED_FLAG: ULONG = 0x00000001;
pub const DOT11_MAC_AUTO_CONFIG_ENABLED_FLAG: ULONG = 0x00000002;
pub const OID_DOT11_ENUM_BSS_LIST: u32 =
    NWF_DEFINE_OID!(121, NWF_OPERATIONAL_OID, NWF_MANDATORY_OID);
pub const DOT11_BSS_ENTRY_BYTE_ARRAY_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_BSS_ENTRY_PHY_SPECIFIC_INFO_FHSS {
    uHopPattern: ULONG,
    uHopSet: ULONG,
    uDwellTime: ULONG,
}}
UNION!{union DOT11_BSS_ENTRY_PHY_SPECIFIC_INFO {
    [u32; 3],
    uChCenterFrequency uChCenterFrequency_mut: ULONG,
    FHSS FHSS_mut: DOT11_BSS_ENTRY_PHY_SPECIFIC_INFO_FHSS,
}}
pub type PDOT11_BSS_ENTRY_PHY_SPECIFIC_INFO = *mut DOT11_BSS_ENTRY_PHY_SPECIFIC_INFO;
STRUCT!{struct DOT11_BSS_ENTRY {
    uPhyId: ULONG,
    PhySpecificInfo: DOT11_BSS_ENTRY_PHY_SPECIFIC_INFO,
    dot11BSSID: DOT11_MAC_ADDRESS,
    dot11BSSType: DOT11_BSS_TYPE,
    lRSSI: LONG,
    uLinkQuality: ULONG,
    bInRegDomain: BOOLEAN,
    usBeaconPeriod: USHORT,
    ullTimestamp: ULONGLONG,
    ullHostTimestamp: ULONGLONG,
    usCapabilityInformation: USHORT,
    uBufferLength: ULONG,
    ucBuffer: [UCHAR; 1],
}}
pub type PDOT11_BSS_ENTRY = *mut DOT11_BSS_ENTRY;
pub const OID_DOT11_FLUSH_BSS_LIST: u32 =
    NWF_DEFINE_OID!(122, NWF_OPERATIONAL_OID, NWF_MANDATORY_OID);
pub const OID_DOT11_POWER_MGMT_REQUEST: u32 =
    NWF_DEFINE_OID!(123, NWF_OPERATIONAL_OID, NWF_MANDATORY_OID);
pub const DOT11_POWER_SAVING_NO_POWER_SAVING: ULONG = 0;
pub const DOT11_POWER_SAVING_FAST_PSP: ULONG = 8;
pub const DOT11_POWER_SAVING_MAX_PSP: ULONG = 16;
pub const DOT11_POWER_SAVING_MAXIMUM_LEVEL: ULONG = 24;
pub const OID_DOT11_DESIRED_SSID_LIST: u32 =
    NWF_DEFINE_OID!(124, NWF_OPERATIONAL_OID, NWF_MANDATORY_OID);
STRUCT!{struct DOT11_SSID_LIST {
    Header: NDIS_OBJECT_HEADER,
    uNumOfEntries: ULONG,
    uTotalNumOfEntries: ULONG,
    SSIDs: [DOT11_SSID; 1],
}}
pub type PDOT11_SSID_LIST = *mut DOT11_SSID_LIST;
pub const DOT11_SSID_LIST_REVISION_1: UCHAR = 1;
pub const OID_DOT11_EXCLUDED_MAC_ADDRESS_LIST: u32 =
    NWF_DEFINE_OID!(125, NWF_OPERATIONAL_OID, NWF_MANDATORY_OID);
STRUCT!{struct DOT11_MAC_ADDRESS_LIST {
    Header: NDIS_OBJECT_HEADER,
    uNumOfEntries: ULONG,
    uTotalNumOfEntries: ULONG,
    MacAddrs: [DOT11_MAC_ADDRESS; 1],
}}
pub type PDOT11_MAC_ADDRESS_LIST = *mut DOT11_MAC_ADDRESS_LIST;
pub const DOT11_MAC_ADDRESS_LIST_REVISION_1: UCHAR = 1;
pub const OID_DOT11_DESIRED_BSSID_LIST: u32 =
    NWF_DEFINE_OID!(126, NWF_OPERATIONAL_OID, NWF_MANDATORY_OID);
pub const OID_DOT11_DESIRED_BSS_TYPE: u32 =
    NWF_DEFINE_OID!(127, NWF_OPERATIONAL_OID, NWF_MANDATORY_OID);
pub const OID_DOT11_PMKID_LIST: u32 = NWF_DEFINE_OID!(128, NWF_OPERATIONAL_OID, NWF_OPTIONAL_OID);
pub type DOT11_PMKID_VALUE = [UCHAR; 16];
STRUCT!{struct DOT11_PMKID_ENTRY {
    BSSID: DOT11_MAC_ADDRESS,
    PMKID: DOT11_PMKID_VALUE,
    uFlags: ULONG,
}}
pub type PDOT11_PMKID_ENTRY = *mut DOT11_PMKID_ENTRY;
STRUCT!{struct DOT11_PMKID_LIST {
    Header: NDIS_OBJECT_HEADER,
    uNumOfEntries: ULONG,
    uTotalNumOfEntries: ULONG,
    PMKIDs: [DOT11_PMKID_ENTRY; 1],
}}
pub type PDOT11_PMKID_LIST = *mut DOT11_PMKID_LIST;
pub const DOT11_PMKID_LIST_REVISION_1: UCHAR = 1;
pub const OID_DOT11_CONNECT_REQUEST: u32 =
    NWF_DEFINE_OID!(129, NWF_OPERATIONAL_OID, NWF_MANDATORY_OID);
pub const OID_DOT11_EXCLUDE_UNENCRYPTED: u32 =
    NWF_DEFINE_OID!(130, NWF_OPERATIONAL_OID, NWF_MANDATORY_OID);
pub const OID_DOT11_STATISTICS: u32 =
    NWF_DEFINE_OID!(131, NWF_STATISTICS_OID, NWF_MANDATORY_OID);
pub const DOT11_STATISTICS_UNKNOWN: ULONGLONG = -1i64 as u64;
STRUCT!{struct DOT11_PHY_FRAME_STATISTICS {
    ullTransmittedFrameCount: ULONGLONG,
    ullMulticastTransmittedFrameCount: ULONGLONG,
    ullFailedCount: ULONGLONG,
    ullRetryCount: ULONGLONG,
    ullMultipleRetryCount: ULONGLONG,
    ullMaxTXLifetimeExceededCount: ULONGLONG,
    ullTransmittedFragmentCount: ULONGLONG,
    ullRTSSuccessCount: ULONGLONG,
    ullRTSFailureCount: ULONGLONG,
    ullACKFailureCount: ULONGLONG,
    ullReceivedFrameCount: ULONGLONG,
    ullMulticastReceivedFrameCount: ULONGLONG,
    ullPromiscuousReceivedFrameCount: ULONGLONG,
    ullMaxRXLifetimeExceededCount: ULONGLONG,
    ullFrameDuplicateCount: ULONGLONG,
    ullReceivedFragmentCount: ULONGLONG,
    ullPromiscuousReceivedFragmentCount: ULONGLONG,
    ullFCSErrorCount: ULONGLONG,
}}
pub type PDOT11_PHY_FRAME_STATISTICS = *mut DOT11_PHY_FRAME_STATISTICS;
STRUCT!{struct DOT11_MAC_FRAME_STATISTICS {
    ullTransmittedFrameCount: ULONGLONG,
    ullReceivedFrameCount: ULONGLONG,
    ullTransmittedFailureFrameCount: ULONGLONG,
    ullReceivedFailureFrameCount: ULONGLONG,
    ullWEPExcludedCount: ULONGLONG,
    ullTKIPLocalMICFailures: ULONGLONG,
    ullTKIPReplays: ULONGLONG,
    ullTKIPICVErrorCount: ULONGLONG,
    ullCCMPReplays: ULONGLONG,
    ullCCMPDecryptErrors: ULONGLONG,
    ullWEPUndecryptableCount: ULONGLONG,
    ullWEPICVErrorCount: ULONGLONG,
    ullDecryptSuccessCount: ULONGLONG,
    ullDecryptFailureCount: ULONGLONG,
}}
pub type PDOT11_MAC_FRAME_STATISTICS = *mut DOT11_MAC_FRAME_STATISTICS;
STRUCT!{struct DOT11_STATISTICS {
    Header: NDIS_OBJECT_HEADER,
    ullFourWayHandshakeFailures: ULONGLONG,
    ullTKIPCounterMeasuresInvoked: ULONGLONG,
    ullReserved: ULONGLONG,
    MacUcastCounters: DOT11_MAC_FRAME_STATISTICS,
    MacMcastCounters: DOT11_MAC_FRAME_STATISTICS,
    PhyCounters: [DOT11_PHY_FRAME_STATISTICS; 1],
}}
pub type PDOT11_STATISTICS = *mut DOT11_STATISTICS;
pub const DOT11_STATISTICS_REVISION_1: UCHAR = 1;
pub const OID_DOT11_PRIVACY_EXEMPTION_LIST: u32 =
    NWF_DEFINE_OID!(132, NWF_OPERATIONAL_OID, NWF_MANDATORY_OID);
STRUCT!{struct DOT11_PRIVACY_EXEMPTION {
    usEtherType: USHORT,
    usExemptionActionType: USHORT,
    usExemptionPacketType: USHORT,
}}
pub type PDOT11_PRIVACY_EXEMPTION = *mut DOT11_PRIVACY_EXEMPTION;
pub const DOT11_EXEMPT_NO_EXEMPTION: USHORT = 0;
pub const DOT11_EXEMPT_ALWAYS: USHORT = 1;
pub const DOT11_EXEMPT_ON_KEY_MAPPING_KEY_UNAVAILABLE: USHORT = 2;
pub const DOT11_EXEMPT_UNICAST: USHORT = 1;
pub const DOT11_EXEMPT_MULTICAST: USHORT = 2;
pub const DOT11_EXEMPT_BOTH: USHORT = 3;
STRUCT!{struct DOT11_PRIVACY_EXEMPTION_LIST {
    Header: NDIS_OBJECT_HEADER,
    uNumOfEntries: ULONG,
    uTotalNumOfEntries: ULONG,
    PrivacyExemptionEntries: [DOT11_PRIVACY_EXEMPTION; 1],
}}
pub type PDOT11_PRIVACY_EXEMPTION_LIST = *mut DOT11_PRIVACY_EXEMPTION_LIST;
pub const DOT11_PRIVACY_EXEMPTION_LIST_REVISION_1: UCHAR = 1;
pub const OID_DOT11_ENABLED_AUTHENTICATION_ALGORITHM: u32 =
    NWF_DEFINE_OID!(133, NWF_OPERATIONAL_OID, NWF_MANDATORY_OID);
STRUCT!{struct DOT11_AUTH_ALGORITHM_LIST {
    Header: NDIS_OBJECT_HEADER,
    uNumOfEntries: ULONG,
    uTotalNumOfEntries: ULONG,
    AlgorithmIds: [DOT11_AUTH_ALGORITHM; 1],
}}
pub type PDOT11_AUTH_ALGORITHM_LIST = *mut DOT11_AUTH_ALGORITHM_LIST;
pub const DOT11_AUTH_ALGORITHM_LIST_REVISION_1: UCHAR = 1;
pub const OID_DOT11_SUPPORTED_UNICAST_ALGORITHM_PAIR: u32 =
    NWF_DEFINE_OID!(134, NWF_OPERATIONAL_OID, NWF_MANDATORY_OID);
STRUCT!{struct DOT11_AUTH_CIPHER_PAIR_LIST {
    Header: NDIS_OBJECT_HEADER,
    uNumOfEntries: ULONG,
    uTotalNumOfEntries: ULONG,
    AuthCipherPairs: [DOT11_AUTH_CIPHER_PAIR; 1],
}}
pub type PDOT11_AUTH_CIPHER_PAIR_LIST = *mut DOT11_AUTH_CIPHER_PAIR_LIST;
pub const DOT11_AUTH_CIPHER_PAIR_LIST_REVISION_1: UCHAR = 1;
pub const OID_DOT11_ENABLED_UNICAST_CIPHER_ALGORITHM: u32 =
    NWF_DEFINE_OID!(135, NWF_OPERATIONAL_OID, NWF_MANDATORY_OID);
STRUCT!{struct DOT11_CIPHER_ALGORITHM_LIST {
    Header: NDIS_OBJECT_HEADER,
    uNumOfEntries: ULONG,
    uTotalNumOfEntries: ULONG,
    AlgorithmIds: [DOT11_CIPHER_ALGORITHM; 1],
}}
pub type PDOT11_CIPHER_ALGORITHM_LIST = *mut DOT11_CIPHER_ALGORITHM_LIST;
pub const DOT11_CIPHER_ALGORITHM_LIST_REVISION_1: UCHAR = 1;
pub const OID_DOT11_SUPPORTED_MULTICAST_ALGORITHM_PAIR: u32 =
    NWF_DEFINE_OID!(136, NWF_OPERATIONAL_OID, NWF_MANDATORY_OID);
pub const OID_DOT11_ENABLED_MULTICAST_CIPHER_ALGORITHM: u32 =
    NWF_DEFINE_OID!(137, NWF_OPERATIONAL_OID, NWF_MANDATORY_OID);
pub const OID_DOT11_CIPHER_DEFAULT_KEY_ID: u32 =
    NWF_DEFINE_OID!(138, NWF_OPERATIONAL_OID, NWF_MANDATORY_OID);
pub const OID_DOT11_CIPHER_DEFAULT_KEY: u32 =
    NWF_DEFINE_OID!(139, NWF_OPERATIONAL_OID, NWF_MANDATORY_OID);
STRUCT!{struct DOT11_CIPHER_DEFAULT_KEY_VALUE {
    Header: NDIS_OBJECT_HEADER,
    uKeyIndex: ULONG,
    AlgorithmId: DOT11_CIPHER_ALGORITHM,
    MacAddr: DOT11_MAC_ADDRESS,
    bDelete: BOOLEAN,
    bStatic: BOOLEAN,
    usKeyLength: USHORT,
    ucKey: [UCHAR; 1],
}}
pub type PDOT11_CIPHER_DEFAULT_KEY_VALUE = *mut DOT11_CIPHER_DEFAULT_KEY_VALUE;
pub const DOT11_CIPHER_DEFAULT_KEY_VALUE_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_KEY_ALGO_TKIP_MIC {
    ucIV48Counter: [UCHAR; 6],
    ulTKIPKeyLength: ULONG,
    ulMICKeyLength: ULONG,
    ucTKIPMICKeys: [UCHAR; 1],
}}
pub type PDOT11_KEY_ALGO_TKIP_MIC = *mut DOT11_KEY_ALGO_TKIP_MIC;
STRUCT!{struct DOT11_KEY_ALGO_CCMP {
    ucIV48Counter: [UCHAR; 6],
    ulCCMPKeyLength: ULONG,
    ucCCMPKey: [UCHAR; 1],
}}
pub type PDOT11_KEY_ALGO_CCMP = *mut DOT11_KEY_ALGO_CCMP;
STRUCT!{struct DOT11_KEY_ALGO_GCMP {
    ucIV48Counter: [UCHAR; 6],
    ulGCMPKeyLength: ULONG,
    ucGCMPKey: [UCHAR; 1],
}}
pub type PDOT11_KEY_ALGO_GCMP = *mut DOT11_KEY_ALGO_GCMP;
STRUCT!{struct DOT11_KEY_ALGO_BIP {
    ucIPN: [UCHAR; 6],
    ulBIPKeyLength: ULONG,
    ucBIPKey: [UCHAR; 1],
}}
pub type PDOT11_KEY_ALGO_BIP = *mut DOT11_KEY_ALGO_BIP;
pub const OID_DOT11_CIPHER_KEY_MAPPING_KEY: u32 =
    NWF_DEFINE_OID!(140, NWF_OPERATIONAL_OID, NWF_MANDATORY_OID);
ENUM!{enum DOT11_DIRECTION {
    DOT11_DIR_INBOUND = 1,
    DOT11_DIR_OUTBOUND = 2,
    DOT11_DIR_BOTH = 3,
}}
pub type PDOT11_DIRECTION = *mut DOT11_DIRECTION;
pub const DOT11_CIPHER_KEY_MAPPING_KEY_VALUE_BYTE_ARRAY_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_CIPHER_KEY_MAPPING_KEY_VALUE {
    PeerMacAddr: DOT11_MAC_ADDRESS,
    AlgorithmId: DOT11_CIPHER_ALGORITHM,
    Direction: DOT11_DIRECTION,
    bDelete: BOOLEAN,
    bStatic: BOOLEAN,
    usKeyLength: USHORT,
    ucKey: [UCHAR; 1],
}}
pub type PDOT11_CIPHER_KEY_MAPPING_KEY_VALUE = *mut DOT11_CIPHER_KEY_MAPPING_KEY_VALUE;
pub const OID_DOT11_ENUM_ASSOCIATION_INFO: u32 =
    NWF_DEFINE_OID!(141, NWF_OPERATIONAL_OID, NWF_MANDATORY_OID);
ENUM!{enum DOT11_ASSOCIATION_STATE {
    dot11_assoc_state_zero = 0,
    dot11_assoc_state_unauth_unassoc = 1,
    dot11_assoc_state_auth_unassoc = 2,
    dot11_assoc_state_auth_assoc = 3,
}}
pub type PDOT11_ASSOCIATION_STATE = *mut DOT11_ASSOCIATION_STATE;
STRUCT!{struct DOT11_ASSOCIATION_INFO_EX {
    PeerMacAddress: DOT11_MAC_ADDRESS,
    BSSID: DOT11_MAC_ADDRESS,
    usCapabilityInformation: USHORT,
    usListenInterval: USHORT,
    ucPeerSupportedRates: [UCHAR; 255],
    usAssociationID: USHORT,
    dot11AssociationState: DOT11_ASSOCIATION_STATE,
    dot11PowerMode: DOT11_POWER_MODE,
    liAssociationUpTime: LARGE_INTEGER,
    ullNumOfTxPacketSuccesses: ULONGLONG,
    ullNumOfTxPacketFailures: ULONGLONG,
    ullNumOfRxPacketSuccesses: ULONGLONG,
    ullNumOfRxPacketFailures: ULONGLONG,
}}
pub type PDOT11_ASSOCIATION_INFO_EX = *mut DOT11_ASSOCIATION_INFO_EX;
STRUCT!{struct DOT11_ASSOCIATION_INFO_LIST {
    Header: NDIS_OBJECT_HEADER,
    uNumOfEntries: ULONG,
    uTotalNumOfEntries: ULONG,
    dot11AssocInfo: [DOT11_ASSOCIATION_INFO_EX; 1],
}}
pub type PDOT11_ASSOCIATION_INFO_LIST = *mut DOT11_ASSOCIATION_INFO_LIST;
pub const DOT11_ASSOCIATION_INFO_LIST_REVISION_1: UCHAR = 1;
pub const OID_DOT11_DISCONNECT_REQUEST: u32 =
    NWF_DEFINE_OID!(142, NWF_OPERATIONAL_OID, NWF_MANDATORY_OID);
pub const OID_DOT11_UNICAST_USE_GROUP_ENABLED: u32 =
    NWF_DEFINE_OID!(143, NWF_OPERATIONAL_OID, NWF_MANDATORY_OID);
pub const OID_DOT11_HARDWARE_PHY_STATE: u32 =
    NWF_DEFINE_OID!(144, NWF_OPERATIONAL_OID, NWF_MANDATORY_OID);
pub const OID_DOT11_DESIRED_PHY_LIST: u32 =
    NWF_DEFINE_OID!(145, NWF_OPERATIONAL_OID, NWF_MANDATORY_OID);
STRUCT!{struct DOT11_PHY_ID_LIST {
    Header: NDIS_OBJECT_HEADER,
    uNumOfEntries: ULONG,
    uTotalNumOfEntries: ULONG,
    dot11PhyId: [ULONG; 1],
}}
pub type PDOT11_PHY_ID_LIST = *mut DOT11_PHY_ID_LIST;
pub const DOT11_PHY_ID_LIST_REVISION_1: UCHAR = 1;
pub const DOT11_PHY_ID_ANY: ULONG = 0xffffffff;
pub const OID_DOT11_CURRENT_PHY_ID: u32 =
    NWF_DEFINE_OID!(146, NWF_OPERATIONAL_OID, NWF_MANDATORY_OID);
pub const OID_DOT11_MEDIA_STREAMING_ENABLED: u32 =
    NWF_DEFINE_OID!(147, NWF_OPERATIONAL_OID, NWF_MANDATORY_OID);
pub const OID_DOT11_UNREACHABLE_DETECTION_THRESHOLD: u32 =
    NWF_DEFINE_OID!(148, NWF_OPERATIONAL_OID, NWF_MANDATORY_OID);
pub const OID_DOT11_ACTIVE_PHY_LIST: u32 =
    NWF_DEFINE_OID!(149, NWF_OPERATIONAL_OID, NWF_MANDATORY_OID);
pub const OID_DOT11_EXTSTA_CAPABILITY: u32 =
    NWF_DEFINE_OID!(150, NWF_OPERATIONAL_OID, NWF_MANDATORY_OID);
STRUCT!{struct DOT11_EXTSTA_CAPABILITY {
    Header: NDIS_OBJECT_HEADER,
    uScanSSIDListSize: ULONG,
    uDesiredBSSIDListSize: ULONG,
    uDesiredSSIDListSize: ULONG,
    uExcludedMacAddressListSize: ULONG,
    uPrivacyExemptionListSize: ULONG,
    uKeyMappingTableSize: ULONG,
    uDefaultKeyTableSize: ULONG,
    uWEPKeyValueMaxLength: ULONG,
    uPMKIDCacheSize: ULONG,
    uMaxNumPerSTADefaultKeyTables: ULONG,
}}
pub type PDOT11_EXTSTA_CAPABILITY = *mut DOT11_EXTSTA_CAPABILITY;
pub const DOT11_EXTSTA_CAPABILITY_REVISION_1: UCHAR = 1;
pub const OID_DOT11_DATA_RATE_MAPPING_TABLE: u32 =
    NWF_DEFINE_OID!(151, NWF_OPERATIONAL_OID, NWF_MANDATORY_OID);
STRUCT!{struct DOT11_DATA_RATE_MAPPING_ENTRY {
    ucDataRateIndex: UCHAR,
    ucDataRateFlag: UCHAR,
    usDataRateValue: USHORT,
}}
pub type PDOT11_DATA_RATE_MAPPING_ENTRY = *mut DOT11_DATA_RATE_MAPPING_ENTRY;
STRUCT!{struct DOT11_DATA_RATE_MAPPING_TABLE {
    Header: NDIS_OBJECT_HEADER,
    uDataRateMappingLength: ULONG,
    DataRateMappingEntries: [DOT11_DATA_RATE_MAPPING_ENTRY; DOT11_RATE_SET_MAX_LENGTH],
}}
pub type PDOT11_DATA_RATE_MAPPING_TABLE = *mut DOT11_DATA_RATE_MAPPING_TABLE;
pub const DOT11_DATA_RATE_MAPPING_TABLE_REVISION_1: UCHAR = 1;
pub const DOT11_DATA_RATE_NON_STANDARD: UCHAR = 0x01;
pub const DOT11_DATA_RATE_INDEX_MASK: UCHAR = 0x7f;
pub const OID_DOT11_SUPPORTED_COUNTRY_OR_REGION_STRING: u32 =
    NWF_DEFINE_OID!(152, NWF_OPERATIONAL_OID, NWF_MANDATORY_OID);
STRUCT!{struct DOT11_COUNTRY_OR_REGION_STRING_LIST {
    Header: NDIS_OBJECT_HEADER,
    uNumOfEntries: ULONG,
    uTotalNumOfEntries: ULONG,
    CountryOrRegionStrings: [DOT11_COUNTRY_OR_REGION_STRING; 1],
}}
pub type PDOT11_COUNTRY_OR_REGION_STRING_LIST = *mut DOT11_COUNTRY_OR_REGION_STRING_LIST;
pub const DOT11_COUNTRY_OR_REGION_STRING_LIST_REVISION_1: UCHAR = 1;
pub const OID_DOT11_DESIRED_COUNTRY_OR_REGION_STRING: u32 =
    NWF_DEFINE_OID!(153, NWF_OPERATIONAL_OID, NWF_MANDATORY_OID);
pub const OID_DOT11_PORT_STATE_NOTIFICATION: u32 =
    NWF_DEFINE_OID!(154, NWF_OPERATIONAL_OID, NWF_OPTIONAL_OID);
STRUCT!{struct DOT11_PORT_STATE_NOTIFICATION {
    Header: NDIS_OBJECT_HEADER,
    PeerMac: DOT11_MAC_ADDRESS,
    bOpen: BOOLEAN,
}}
pub type PDOT11_PORT_STATE_NOTIFICATION = *mut DOT11_PORT_STATE_NOTIFICATION;
pub const DOT11_PORT_STATE_NOTIFICATION_REVISION_1: UCHAR = 1;
pub const OID_DOT11_IBSS_PARAMS: u32 =
    NWF_DEFINE_OID!(155, NWF_OPERATIONAL_OID, NWF_MANDATORY_OID);
STRUCT!{struct DOT11_IBSS_PARAMS {
    Header: NDIS_OBJECT_HEADER,
    bJoinOnly: BOOLEAN,
    uIEsOffset: ULONG,
    uIEsLength: ULONG,
}}
pub type PDOT11_IBSS_PARAMS = *mut DOT11_IBSS_PARAMS;
pub const DOT11_IBSS_PARAMS_REVISION_1: UCHAR = 1;
pub const OID_DOT11_QOS_PARAMS: u32 = NWF_DEFINE_OID!(156, NWF_OPERATIONAL_OID, NWF_OPTIONAL_OID);
STRUCT!{struct DOT11_QOS_PARAMS {
    Header: NDIS_OBJECT_HEADER,
    ucEnabledQoSProtocolFlags: UCHAR,
}}
pub type PDOT11_QOS_PARAMS = *mut DOT11_QOS_PARAMS;
pub const DOT11_QOS_PARAMS_REVISION_1: UCHAR = 1;
pub const DOT11_QOS_PROTOCOL_FLAG_WMM: UCHAR = 0x01;
pub const DOT11_QOS_PROTOCOL_FLAG_11E: UCHAR = 0x02;
pub const OID_DOT11_SAFE_MODE_ENABLED: u32 =
    NWF_DEFINE_OID!(157, NWF_OPERATIONAL_OID, NWF_MANDATORY_OID);
pub const OID_DOT11_HIDDEN_NETWORK_ENABLED: u32 =
    NWF_DEFINE_OID!(158, NWF_OPERATIONAL_OID, NWF_MANDATORY_OID);
pub const OID_DOT11_ASSOCIATION_PARAMS: u32 =
    NWF_DEFINE_OID!(159, NWF_OPERATIONAL_OID, NWF_MANDATORY_OID);
STRUCT!{struct DOT11_ASSOCIATION_PARAMS {
    Header: NDIS_OBJECT_HEADER,
    BSSID: DOT11_MAC_ADDRESS,
    uAssocRequestIEsOffset: ULONG,
    uAssocRequestIEsLength: ULONG,
}}
pub type PDOT11_ASSOCIATION_PARAMS = *mut DOT11_ASSOCIATION_PARAMS;
pub const DOT11_ASSOCIATION_PARAMS_REVISION_1: UCHAR = 1;
pub const OID_DOT11_SAFE_MODE_HT_ENABLED: u32 =
    NWF_DEFINE_OID!(160, NWF_OPERATIONAL_OID, NWF_MANDATORY_OID);
pub const DOT11_MAX_NUM_OF_FRAGMENTS: USHORT = 16;
pub const DOT11_PRIORITY_CONTENTION: i32 = 0;
pub const DOT11_PRIORITY_CONTENTION_FREE: i32 = 1;
pub const DOT11_SERVICE_CLASS_REORDERABLE_MULTICAST: i32 = 0;
pub const DOT11_SERVICE_CLASS_STRICTLY_ORDERED: i32 = 1;
pub const DOT11_FLAGS_80211B_SHORT_PREAMBLE: u32 = 0x00000001;
pub const DOT11_FLAGS_80211B_PBCC: u32 = 0x00000002;
pub const DOT11_FLAGS_80211B_CHANNEL_AGILITY: u32 = 0x00000004;
pub const DOT11_FLAGS_PS_ON: u32 = 0x00000008;
pub const DOT11_FLAGS_80211G_DSSS_OFDM: u32 = 0x00000010;
pub const DOT11_FLAGS_80211G_USE_PROTECTION: u32 = 0x00000020;
pub const DOT11_FLAGS_80211G_NON_ERP_PRESENT: u32 = 0x00000040;
pub const DOT11_FLAGS_80211G_BARKER_PREAMBLE_MODE: u32 = 0x00000080;
pub const DOT11_WME_PACKET: u32 = 0x00000100;
STRUCT!{struct DOT11_FRAGMENT_DESCRIPTOR {
    uOffset: ULONG,
    uLength: ULONG,
}}
pub type PDOT11_FRAGMENT_DESCRIPTOR = *mut DOT11_FRAGMENT_DESCRIPTOR;
STRUCT!{struct DOT11_PER_MSDU_COUNTERS {
    uTransmittedFragmentCount: ULONG,
    uRetryCount: ULONG,
    uRTSSuccessCount: ULONG,
    uRTSFailureCount: ULONG,
    uACKFailureCount: ULONG,
}}
pub type PDOT11_PER_MSDU_COUNTERS = *mut DOT11_PER_MSDU_COUNTERS;
STRUCT!{struct DOT11_HRDSSS_PHY_ATTRIBUTES {
    bShortPreambleOptionImplemented: BOOLEAN,
    bPBCCOptionImplemented: BOOLEAN,
    bChannelAgilityPresent: BOOLEAN,
    uHRCCAModeSupported: ULONG,
}}
pub type PDOT11_HRDSSS_PHY_ATTRIBUTES = *mut DOT11_HRDSSS_PHY_ATTRIBUTES;
STRUCT!{struct DOT11_OFDM_PHY_ATTRIBUTES {
    uFrequencyBandsSupported: ULONG,
}}
pub type PDOT11_OFDM_PHY_ATTRIBUTES = *mut DOT11_OFDM_PHY_ATTRIBUTES;
STRUCT!{struct DOT11_ERP_PHY_ATTRIBUTES {
    HRDSSSAttributes: DOT11_HRDSSS_PHY_ATTRIBUTES,
    bERPPBCCOptionImplemented: BOOLEAN,
    bDSSSOFDMOptionImplemented: BOOLEAN,
    bShortSlotTimeOptionImplemented: BOOLEAN,
}}
pub type PDOT11_ERP_PHY_ATTRIBUTES = *mut DOT11_ERP_PHY_ATTRIBUTES;
pub const DOT11_PHY_ATTRIBUTES_REVISION_1: UCHAR = 1;
UNION!{union DOT11_PHY_ATTRIBUTES_u {
    [u32; 3],
    HRDSSSAttributes HRDSSSAttributes_mut: DOT11_HRDSSS_PHY_ATTRIBUTES,
    OFDMAttributes OFDMAttributes_mut: DOT11_OFDM_PHY_ATTRIBUTES,
    ERPAttributes ERPAttributes_mut: DOT11_ERP_PHY_ATTRIBUTES,
}}
STRUCT!{struct DOT11_PHY_ATTRIBUTES {
    Header: NDIS_OBJECT_HEADER,
    PhyType: DOT11_PHY_TYPE,
    bHardwarePhyState: BOOLEAN,
    bSoftwarePhyState: BOOLEAN,
    bCFPollable: BOOLEAN,
    uMPDUMaxLength: ULONG,
    TempType: DOT11_TEMP_TYPE,
    DiversitySupport: DOT11_DIVERSITY_SUPPORT,
    u: DOT11_PHY_ATTRIBUTES_u,
    uNumberSupportedPowerLevels: ULONG,
    TxPowerLevels: [ULONG; 8],
    uNumDataRateMappingEntries: ULONG,
    DataRateMappingEntries: [DOT11_DATA_RATE_MAPPING_ENTRY; DOT11_RATE_SET_MAX_LENGTH],
    SupportedDataRatesValue: DOT11_SUPPORTED_DATA_RATES_VALUE_V2,
}}
pub type PDOT11_PHY_ATTRIBUTES = *mut DOT11_PHY_ATTRIBUTES;
pub const DOT11_EXTSTA_ATTRIBUTES_SAFEMODE_OID_SUPPORTED: UINT8 = 0x1;
pub const DOT11_EXTSTA_ATTRIBUTES_SAFEMODE_CERTIFIED: UINT8 = 0x2;
pub const DOT11_EXTSTA_ATTRIBUTES_SAFEMODE_RESERVED: UINT8 = 0xC;
pub const DOT11_EXTSTA_ATTRIBUTES_REVISION_1: UCHAR = 1;
pub const DOT11_EXTSTA_ATTRIBUTES_REVISION_2: UCHAR = 2;
pub const DOT11_EXTSTA_ATTRIBUTES_REVISION_3: UCHAR = 3;
pub const DOT11_EXTSTA_ATTRIBUTES_REVISION_4: UCHAR = 4;
STRUCT!{struct DOT11_EXTSTA_ATTRIBUTES {
    Header: NDIS_OBJECT_HEADER,
    uScanSSIDListSize: ULONG,
    uDesiredBSSIDListSize: ULONG,
    uDesiredSSIDListSize: ULONG,
    uExcludedMacAddressListSize: ULONG,
    uPrivacyExemptionListSize: ULONG,
    uKeyMappingTableSize: ULONG,
    uDefaultKeyTableSize: ULONG,
    uWEPKeyValueMaxLength: ULONG,
    uPMKIDCacheSize: ULONG,
    uMaxNumPerSTADefaultKeyTables: ULONG,
    bStrictlyOrderedServiceClassImplemented: BOOLEAN,
    ucSupportedQoSProtocolFlags: UCHAR,
    bSafeModeImplemented: BOOLEAN,
    uNumSupportedCountryOrRegionStrings: ULONG,
    pSupportedCountryOrRegionStrings: PDOT11_COUNTRY_OR_REGION_STRING,
    uInfraNumSupportedUcastAlgoPairs: ULONG,
    pInfraSupportedUcastAlgoPairs: PDOT11_AUTH_CIPHER_PAIR,
    uInfraNumSupportedMcastAlgoPairs: ULONG,
    pInfraSupportedMcastAlgoPairs: PDOT11_AUTH_CIPHER_PAIR,
    uAdhocNumSupportedUcastAlgoPairs: ULONG,
    pAdhocSupportedUcastAlgoPairs: PDOT11_AUTH_CIPHER_PAIR,
    uAdhocNumSupportedMcastAlgoPairs: ULONG,
    pAdhocSupportedMcastAlgoPairs: PDOT11_AUTH_CIPHER_PAIR,
    bAutoPowerSaveMode: BOOLEAN,
    uMaxNetworkOffloadListSize: ULONG,
    bMFPCapable: BOOLEAN,
    uInfraNumSupportedMcastMgmtAlgoPairs: ULONG,
    pInfraSupportedMcastMgmtAlgoPairs: PDOT11_AUTH_CIPHER_PAIR,
    bNeighborReportSupported: BOOLEAN,
    bAPChannelReportSupported: BOOLEAN,
    bActionFramesSupported: BOOLEAN,
    bANQPQueryOffloadSupported: BOOLEAN,
    bHESSIDConnectionSupported: BOOLEAN,
}}
pub type PDOT11_EXTSTA_ATTRIBUTES = *mut DOT11_EXTSTA_ATTRIBUTES;
STRUCT!{struct DOT11_RECV_EXTENSION_INFO {
    uVersion: ULONG,
    pvReserved: PVOID,
    dot11PhyType: DOT11_PHY_TYPE,
    uChCenterFrequency: ULONG,
    lRSSI: LONG,
    lRSSIMin: LONG,
    lRSSIMax: LONG,
    uRSSI: ULONG,
    ucPriority: UCHAR,
    ucDataRate: UCHAR,
    ucPeerMacAddress: [UCHAR; 6],
    dwExtendedStatus: ULONG,
    hWEPOffloadContext: HANDLE,
    hAuthOffloadContext: HANDLE,
    usWEPAppliedMask: USHORT,
    usWPAMSDUPriority: USHORT,
    dot11LowestIV48Counter: DOT11_IV48_COUNTER,
    usDot11LeftRWBitMap: USHORT,
    dot11HighestIV48Counter: DOT11_IV48_COUNTER,
    usDot11RightRWBitMap: USHORT,
    usNumberOfMPDUsReceived: USHORT,
    usNumberOfFragments: USHORT,
    pNdisPackets: [PVOID; 1],
}}
pub type PDOT11_RECV_EXTENSION_INFO = *mut DOT11_RECV_EXTENSION_INFO;
STRUCT!{struct DOT11_RECV_EXTENSION_INFO_V2 {
    uVersion: ULONG,
    pvReserved: PVOID,
    dot11PhyType: DOT11_PHY_TYPE,
    uChCenterFrequency: ULONG,
    lRSSI: LONG,
    uRSSI: ULONG,
    ucPriority: UCHAR,
    ucDataRate: UCHAR,
    ucPeerMacAddress: [UCHAR; 6],
    dwExtendedStatus: ULONG,
    hWEPOffloadContext: HANDLE,
    hAuthOffloadContext: HANDLE,
    usWEPAppliedMask: USHORT,
    usWPAMSDUPriority: USHORT,
    dot11LowestIV48Counter: DOT11_IV48_COUNTER,
    usDot11LeftRWBitMap: USHORT,
    dot11HighestIV48Counter: DOT11_IV48_COUNTER,
    usDot11RightRWBitMap: USHORT,
    usNumberOfMPDUsReceived: USHORT,
    usNumberOfFragments: USHORT,
    pNdisPackets: [PVOID; 1],
}}
pub type PDOT11_RECV_EXTENSION_INFO_V2 = *mut DOT11_RECV_EXTENSION_INFO_V2;
pub const DOT11_STATUS_SUCCESS: NDIS_STATUS = 0x00000001;
pub const DOT11_STATUS_RETRY_LIMIT_EXCEEDED: NDIS_STATUS = 0x00000002;
pub const DOT11_STATUS_UNSUPPORTED_PRIORITY: NDIS_STATUS = 0x00000004;
pub const DOT11_STATUS_UNSUPPORTED_SERVICE_CLASS: NDIS_STATUS = 0x00000008;
pub const DOT11_STATUS_UNAVAILABLE_PRIORITY: NDIS_STATUS = 0x00000010;
pub const DOT11_STATUS_UNAVAILABLE_SERVICE_CLASS: NDIS_STATUS = 0x00000020;
pub const DOT11_STATUS_XMIT_MSDU_TIMER_EXPIRED: NDIS_STATUS = 0x00000040;
pub const DOT11_STATUS_UNAVAILABLE_BSS: NDIS_STATUS = 0x00000080;
pub const DOT11_STATUS_EXCESSIVE_DATA_LENGTH: NDIS_STATUS = 0x00000100;
pub const DOT11_STATUS_ENCRYPTION_FAILED: NDIS_STATUS = 0x00000200;
pub const DOT11_STATUS_WEP_KEY_UNAVAILABLE: NDIS_STATUS = 0x00000400;
pub const DOT11_STATUS_ICV_VERIFIED: NDIS_STATUS = 0x00000800;
pub const DOT11_STATUS_PACKET_REASSEMBLED: NDIS_STATUS = 0x00001000;
pub const DOT11_STATUS_PACKET_NOT_REASSEMBLED: NDIS_STATUS = 0x00002000;
pub const DOT11_STATUS_GENERATE_AUTH_FAILED: NDIS_STATUS = 0x00004000;
pub const DOT11_STATUS_AUTH_NOT_VERIFIED: NDIS_STATUS = 0x00008000;
pub const DOT11_STATUS_AUTH_VERIFIED: NDIS_STATUS = 0x00010000;
pub const DOT11_STATUS_AUTH_FAILED: NDIS_STATUS = 0x00020000;
pub const DOT11_STATUS_PS_LIFETIME_EXPIRED: NDIS_STATUS = 0x00040000;
STRUCT!{struct DOT11_STATUS_INDICATION {
    uStatusType: ULONG,
    ndisStatus: NDIS_STATUS,
}}
pub type PDOT11_STATUS_INDICATION = *mut DOT11_STATUS_INDICATION;
pub const DOT11_STATUS_RESET_CONFIRM: ULONG = 4;
pub const DOT11_STATUS_SCAN_CONFIRM: ULONG = 1;
pub const DOT11_STATUS_JOIN_CONFIRM: ULONG = 2;
pub const DOT11_STATUS_START_CONFIRM: ULONG = 3;
pub const DOT11_STATUS_AP_JOIN_CONFIRM: ULONG = 5;
pub const DOT11_STATUS_MPDU_MAX_LENGTH_CHANGED: ULONG = 6;
STRUCT!{struct DOT11_MPDU_MAX_LENGTH_INDICATION {
    Header: NDIS_OBJECT_HEADER,
    uPhyId: ULONG,
    uMPDUMaxLength: ULONG,
}}
pub type PDOT11_MPDU_MAX_LENGTH_INDICATION = *mut DOT11_MPDU_MAX_LENGTH_INDICATION;
pub const DOT11_MPDU_MAX_LENGTH_INDICATION_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_ASSOCIATION_START_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    MacAddr: DOT11_MAC_ADDRESS,
    SSID: DOT11_SSID,
    uIHVDataOffset: ULONG,
    uIHVDataSize: ULONG,
}}
pub type PDOT11_ASSOCIATION_START_PARAMETERS = *mut DOT11_ASSOCIATION_START_PARAMETERS;
pub const DOT11_ASSOCIATION_START_PARAMETERS_REVISION_1: UCHAR = 1;
pub const DOT11_ENCAP_RFC_1042: USHORT = 1;
pub const DOT11_ENCAP_802_1H: USHORT = 2;
STRUCT!{struct DOT11_ENCAP_ENTRY {
    usEtherType: USHORT,
    usEncapType: USHORT,
}}
pub type PDOT11_ENCAP_ENTRY = *mut DOT11_ENCAP_ENTRY;
ENUM!{enum DOT11_DS_INFO {
    DOT11_DS_CHANGED = 0,
    DOT11_DS_UNCHANGED = 1,
    DOT11_DS_UNKNOWN = 2,
}}
pub type PDOT11_DS_INFO = *mut DOT11_DS_INFO;
pub type DOT11_ASSOC_STATUS = ULONG;
pub const DOT11_ASSOC_STATUS_SUCCESS: DOT11_ASSOC_STATUS = 0;
pub const DOT11_ASSOC_STATUS_FAILURE: DOT11_ASSOC_STATUS = 0x00000001;
pub const DOT11_ASSOC_STATUS_UNREACHABLE: DOT11_ASSOC_STATUS = 0x00000002;
pub const DOT11_ASSOC_STATUS_RADIO_OFF: DOT11_ASSOC_STATUS = 0x00000003;
pub const DOT11_ASSOC_STATUS_PHY_DISABLED: DOT11_ASSOC_STATUS = 0x00000004;
pub const DOT11_ASSOC_STATUS_CANCELLED: DOT11_ASSOC_STATUS = 0x00000005;
pub const DOT11_ASSOC_STATUS_CANDIDATE_LIST_EXHAUSTED: DOT11_ASSOC_STATUS = 0x00000006;
pub const DOT11_ASSOC_STATUS_DISASSOCIATED_BY_OS: DOT11_ASSOC_STATUS = 0x00000007;
pub const DOT11_ASSOC_STATUS_DISASSOCIATED_BY_ROAMING: DOT11_ASSOC_STATUS = 0x00000008;
pub const DOT11_ASSOC_STATUS_DISASSOCIATED_BY_RESET: DOT11_ASSOC_STATUS = 0x00000009;
pub const DOT11_ASSOC_STATUS_SYSTEM_ERROR: DOT11_ASSOC_STATUS = 0x0000000a;
pub const DOT11_ASSOC_STATUS_ROAMING_BETTER_AP_FOUND: DOT11_ASSOC_STATUS = 0x0000000b;
pub const DOT11_ASSOC_STATUS_ROAMING_ASSOCIATION_LOST: DOT11_ASSOC_STATUS = 0x0000000c;
pub const DOT11_ASSOC_STATUS_ROAMING_ADHOC: DOT11_ASSOC_STATUS = 0x0000000d;
pub const DOT11_ASSOC_STATUS_PEER_DEAUTHENTICATED: DOT11_ASSOC_STATUS = 0x00010000;
pub const DOT11_ASSOC_STATUS_PEER_DEAUTHENTICATED_START: DOT11_ASSOC_STATUS =
    DOT11_ASSOC_STATUS_PEER_DEAUTHENTICATED;
pub const DOT11_ASSOC_STATUS_PEER_DEAUTHENTICATED_END: DOT11_ASSOC_STATUS = 0x0001ffff;
pub const DOT11_ASSOC_STATUS_PEER_DISASSOCIATED: DOT11_ASSOC_STATUS = 0x00020000;
pub const DOT11_ASSOC_STATUS_PEER_DISASSOCIATED_START: DOT11_ASSOC_STATUS =
    DOT11_ASSOC_STATUS_PEER_DISASSOCIATED;
pub const DOT11_ASSOC_STATUS_PEER_DISASSOCIATED_END: DOT11_ASSOC_STATUS = 0x0002ffff;
pub const DOT11_ASSOC_STATUS_ASSOCIATION_RESPONSE: DOT11_ASSOC_STATUS = 0x00030000;
pub const DOT11_ASSOC_STATUS_ASSOCIATION_RESPONSE_START: DOT11_ASSOC_STATUS =
    DOT11_ASSOC_STATUS_ASSOCIATION_RESPONSE;
pub const DOT11_ASSOC_STATUS_ASSOCIATION_RESPONSE_END: DOT11_ASSOC_STATUS = 0x0003ffff;
pub const DOT11_ASSOC_STATUS_REASON_CODE_MASK: DOT11_ASSOC_STATUS = 0xffff;
pub const DOT11_ASSOC_STATUS_IHV_START: DOT11_ASSOC_STATUS = 0x80000000;
pub const DOT11_ASSOC_STATUS_IHV_END: DOT11_ASSOC_STATUS = 0xffffffff;
STRUCT!{struct DOT11_ASSOCIATION_COMPLETION_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    MacAddr: DOT11_MAC_ADDRESS,
    uStatus: DOT11_ASSOC_STATUS,
    bReAssocReq: BOOLEAN,
    bReAssocResp: BOOLEAN,
    uAssocReqOffset: ULONG,
    uAssocReqSize: ULONG,
    uAssocRespOffset: ULONG,
    uAssocRespSize: ULONG,
    uBeaconOffset: ULONG,
    uBeaconSize: ULONG,
    uIHVDataOffset: ULONG,
    uIHVDataSize: ULONG,
    AuthAlgo: DOT11_AUTH_ALGORITHM,
    UnicastCipher: DOT11_CIPHER_ALGORITHM,
    MulticastCipher: DOT11_CIPHER_ALGORITHM,
    uActivePhyListOffset: ULONG,
    uActivePhyListSize: ULONG,
    bFourAddressSupported: BOOLEAN,
    bPortAuthorized: BOOLEAN,
    ucActiveQoSProtocol: UCHAR,
    DSInfo: DOT11_DS_INFO,
    uEncapTableOffset: ULONG,
    uEncapTableSize: ULONG,
    MulticastMgmtCipher: DOT11_CIPHER_ALGORITHM,
    uAssocComebackTime: ULONG,
}}
pub type PDOT11_ASSOCIATION_COMPLETION_PARAMETERS = *mut DOT11_ASSOCIATION_COMPLETION_PARAMETERS;
pub const DOT11_ASSOCIATION_COMPLETION_PARAMETERS_REVISION_1: UCHAR = 1;
pub const DOT11_ASSOCIATION_COMPLETION_PARAMETERS_REVISION_2: UCHAR = 2;
STRUCT!{struct DOT11_CONNECTION_START_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    BSSType: DOT11_BSS_TYPE,
    AdhocBSSID: DOT11_MAC_ADDRESS,
    AdhocSSID: DOT11_SSID,
}}
pub type PDOT11_CONNECTION_START_PARAMETERS = *mut DOT11_CONNECTION_START_PARAMETERS;
pub const DOT11_CONNECTION_START_PARAMETERS_REVISION_1: UCHAR = 1;
pub const DOT11_CONNECTION_STATUS_SUCCESS: DOT11_ASSOC_STATUS = DOT11_ASSOC_STATUS_SUCCESS;
pub const DOT11_CONNECTION_STATUS_FAILURE: DOT11_ASSOC_STATUS = DOT11_ASSOC_STATUS_FAILURE;
pub const DOT11_CONNECTION_STATUS_CANDIDATE_LIST_EXHAUSTED: DOT11_ASSOC_STATUS =
    DOT11_ASSOC_STATUS_CANDIDATE_LIST_EXHAUSTED;
pub const DOT11_CONNECTION_STATUS_PHY_POWER_DOWN: DOT11_ASSOC_STATUS =
    DOT11_ASSOC_STATUS_RADIO_OFF;
pub const DOT11_CONNECTION_STATUS_CANCELLED: DOT11_ASSOC_STATUS = DOT11_ASSOC_STATUS_CANCELLED;
pub const DOT11_CONNECTION_STATUS_IHV_START: DOT11_ASSOC_STATUS = DOT11_ASSOC_STATUS_IHV_START;
pub const DOT11_CONNECTION_STATUS_IHV_END: DOT11_ASSOC_STATUS = DOT11_ASSOC_STATUS_IHV_END;
STRUCT!{struct DOT11_CONNECTION_COMPLETION_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    uStatus: DOT11_ASSOC_STATUS,
}}
pub type PDOT11_CONNECTION_COMPLETION_PARAMETERS = *mut DOT11_CONNECTION_COMPLETION_PARAMETERS;
pub const DOT11_CONNECTION_COMPLETION_PARAMETERS_REVISION_1: UCHAR = 1;
pub const DOT11_ROAMING_REASON_BETTER_AP_FOUND: DOT11_ASSOC_STATUS =
    DOT11_ASSOC_STATUS_ROAMING_BETTER_AP_FOUND;
pub const DOT11_ROAMING_REASON_ASSOCIATION_LOST: DOT11_ASSOC_STATUS =
    DOT11_ASSOC_STATUS_ROAMING_ASSOCIATION_LOST;
pub const DOT11_ROAMING_REASON_ADHOC: DOT11_ASSOC_STATUS = DOT11_ASSOC_STATUS_ROAMING_ADHOC;
pub const DOT11_ROAMING_REASON_IHV_START: DOT11_ASSOC_STATUS = DOT11_ASSOC_STATUS_IHV_START;
pub const DOT11_ROAMING_REASON_IHV_END: DOT11_ASSOC_STATUS = DOT11_ASSOC_STATUS_IHV_END;
STRUCT!{struct DOT11_ROAMING_START_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    AdhocBSSID: DOT11_MAC_ADDRESS,
    AdhocSSID: DOT11_SSID,
    uRoamingReason: DOT11_ASSOC_STATUS,
}}
pub type PDOT11_ROAMING_START_PARAMETERS = *mut DOT11_ROAMING_START_PARAMETERS;
pub const DOT11_ROAMING_START_PARAMETERS_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_ROAMING_COMPLETION_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    uStatus: DOT11_ASSOC_STATUS,
}}
pub type PDOT11_ROAMING_COMPLETION_PARAMETERS = *mut DOT11_ROAMING_COMPLETION_PARAMETERS;
pub const DOT11_ROAMING_COMPLETION_PARAMETERS_REVISION_1: UCHAR = 1;
pub const DOT11_DISASSOC_REASON_OS: DOT11_ASSOC_STATUS = DOT11_ASSOC_STATUS_DISASSOCIATED_BY_OS;
pub const DOT11_DISASSOC_REASON_PEER_UNREACHABLE: DOT11_ASSOC_STATUS =
    DOT11_ASSOC_STATUS_UNREACHABLE;
pub const DOT11_DISASSOC_REASON_PEER_DEAUTHENTICATED: DOT11_ASSOC_STATUS =
    DOT11_ASSOC_STATUS_PEER_DEAUTHENTICATED;
pub const DOT11_DISASSOC_REASON_PEER_DISASSOCIATED: DOT11_ASSOC_STATUS =
    DOT11_ASSOC_STATUS_PEER_DISASSOCIATED;
pub const DOT11_DISASSOC_REASON_RADIO_OFF: DOT11_ASSOC_STATUS = DOT11_ASSOC_STATUS_RADIO_OFF;
pub const DOT11_DISASSOC_REASON_PHY_DISABLED: DOT11_ASSOC_STATUS = DOT11_ASSOC_STATUS_PHY_DISABLED;
pub const DOT11_DISASSOC_REASON_IHV_START: DOT11_ASSOC_STATUS = DOT11_ASSOC_STATUS_IHV_START;
pub const DOT11_DISASSOC_REASON_IHV_END: DOT11_ASSOC_STATUS = DOT11_ASSOC_STATUS_IHV_END;
STRUCT!{struct DOT11_DISASSOCIATION_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    MacAddr: DOT11_MAC_ADDRESS,
    uReason: DOT11_ASSOC_STATUS,
    uIHVDataOffset: ULONG,
    uIHVDataSize: ULONG,
}}
pub type PDOT11_DISASSOCIATION_PARAMETERS = *mut DOT11_DISASSOCIATION_PARAMETERS;
pub const DOT11_DISASSOCIATION_PARAMETERS_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_TKIPMIC_FAILURE_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    bDefaultKeyFailure: BOOLEAN,
    uKeyIndex: ULONG,
    PeerMac: DOT11_MAC_ADDRESS,
}}
pub type PDOT11_TKIPMIC_FAILURE_PARAMETERS = *mut DOT11_TKIPMIC_FAILURE_PARAMETERS;
pub const DOT11_TKIPMIC_FAILURE_PARAMETERS_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_PMKID_CANDIDATE_LIST_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    uCandidateListSize: ULONG,
    uCandidateListOffset: ULONG,
}}
pub type PDOT11_PMKID_CANDIDATE_LIST_PARAMETERS = *mut DOT11_PMKID_CANDIDATE_LIST_PARAMETERS;
pub const DOT11_PMKID_CANDIDATE_LIST_PARAMETERS_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_BSSID_CANDIDATE {
    BSSID: DOT11_MAC_ADDRESS,
    uFlags: ULONG,
}}
pub type PDOT11_BSSID_CANDIDATE = *mut DOT11_BSSID_CANDIDATE;
pub const DOT11_PMKID_CANDIDATE_PREAUTH_ENABLED: ULONG = 0x00000001;
STRUCT!{struct DOT11_PHY_STATE_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    uPhyId: ULONG,
    bHardwarePhyState: BOOLEAN,
    bSoftwarePhyState: BOOLEAN,
}}
pub type PDOT11_PHY_STATE_PARAMETERS = *mut DOT11_PHY_STATE_PARAMETERS;
pub const DOT11_PHY_STATE_PARAMETERS_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_LINK_QUALITY_ENTRY {
    PeerMacAddr: DOT11_MAC_ADDRESS,
    ucLinkQuality: UCHAR,
}}
pub type PDOT11_LINK_QUALITY_ENTRY = *mut DOT11_LINK_QUALITY_ENTRY;
STRUCT!{struct DOT11_LINK_QUALITY_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    uLinkQualityListSize: ULONG,
    uLinkQualityListOffset: ULONG,
}}
pub type PDOT11_LINK_QUALITY_PARAMETERS = *mut DOT11_LINK_QUALITY_PARAMETERS;
pub const DOT11_LINK_QUALITY_PARAMETERS_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_EXTSTA_SEND_CONTEXT {
    Header: NDIS_OBJECT_HEADER,
    usExemptionActionType: USHORT,
    uPhyId: ULONG,
    uDelayedSleepValue: ULONG,
    pvMediaSpecificInfo: PVOID,
    uSendFlags: ULONG,
}}
pub type PDOT11_EXTSTA_SEND_CONTEXT = *mut DOT11_EXTSTA_SEND_CONTEXT;
pub type DOT11_EXTAP_SEND_CONTEXT = DOT11_EXTSTA_SEND_CONTEXT;
pub type PDOT11_EXTAP_SEND_CONTEXT = *mut DOT11_EXTSTA_SEND_CONTEXT;
pub const DOT11_EXTSTA_SEND_CONTEXT_REVISION_1: UCHAR = 1;
pub const DOT11_RECV_FLAG_RAW_PACKET: ULONG = 0x00000001;
pub const DOT11_RECV_FLAG_RAW_PACKET_FCS_FAILURE: ULONG = 0x00000002;
pub const DOT11_RECV_FLAG_RAW_PACKET_TIMESTAMP: ULONG = 0x00000004;
pub const DOT11_EXTSTA_RECV_CONTEXT_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_EXTSTA_RECV_CONTEXT {
    Header: NDIS_OBJECT_HEADER,
    uReceiveFlags: ULONG,
    uPhyId: ULONG,
    uChCenterFrequency: ULONG,
    usNumberOfMPDUsReceived: USHORT,
    lRSSI: LONG,
    ucDataRate: UCHAR,
    uSizeMediaSpecificInfo: ULONG,
    pvMediaSpecificInfo: PVOID,
    ullTimestamp: ULONGLONG,
}}
pub type PDOT11_EXTSTA_RECV_CONTEXT = *mut DOT11_EXTSTA_RECV_CONTEXT;
pub type DOT11_EXTAP_RECV_CONTEXT = DOT11_EXTSTA_RECV_CONTEXT;
pub type PDOT11_EXTAP_RECV_CONTEXT = *mut DOT11_EXTSTA_RECV_CONTEXT;
pub const OID_DOT11_PRIVATE_OIDS_START: u32 = OID_DOT11_NDIS_START + 1024;
pub const OID_DOT11_CURRENT_ADDRESS: u32 = OID_DOT11_PRIVATE_OIDS_START + 2;
pub const OID_DOT11_PERMANENT_ADDRESS: u32 = OID_DOT11_PRIVATE_OIDS_START + 3;
pub const OID_DOT11_MULTICAST_LIST: u32 = OID_DOT11_PRIVATE_OIDS_START + 4;
pub const OID_DOT11_MAXIMUM_LIST_SIZE: u32 = OID_DOT11_PRIVATE_OIDS_START + 5;
macro_rules! DEFINE_NWF_GUID {
    ($name:ident, $ord:expr) => {
        DEFINE_GUID!{
            $name, 0x6cb9a43e + $ord, 0xc45f, 0x4039, 0x9f, 0xe6, 0xd0, 0x8c, 0xb0, 0x57, 0x18,
            0x4c
        }
    };
}
DEFINE_NWF_GUID!{GUID_NWF_OFFLOAD_CAPABILITY, 0}
DEFINE_NWF_GUID!{GUID_NWF_CURRENT_OFFLOAD_CAPABILITY, 1}
DEFINE_NWF_GUID!{GUID_NWF_WEP_OFFLOAD, 2}
DEFINE_NWF_GUID!{GUID_NWF_WEP_UPLOAD, 3}
DEFINE_NWF_GUID!{GUID_NWF_DEFAULT_WEP_OFFLOAD, 4}
DEFINE_NWF_GUID!{GUID_NWF_DEFAULT_WEP_UPLOAD, 5}
DEFINE_NWF_GUID!{GUID_NWF_MPDU_MAX_LENGTH, 6}
DEFINE_NWF_GUID!{GUID_NWF_OPERATION_MODE_CAPABILITY, 7}
DEFINE_NWF_GUID!{GUID_NWF_CURRENT_OPERATION_MODE, 8}
DEFINE_NWF_GUID!{GUID_NWF_CURRENT_PACKET_FILTER, 9}
DEFINE_NWF_GUID!{GUID_NWF_ATIM_WINDOW, 10}
DEFINE_NWF_GUID!{GUID_NWF_SCAN_REQUEST, 11}
DEFINE_NWF_GUID!{GUID_NWF_CURRENT_PHY_TYPE, 12}
DEFINE_NWF_GUID!{GUID_NWF_JOIN_REQUEST, 13}
DEFINE_NWF_GUID!{GUID_NWF_START_REQUEST, 14}
DEFINE_NWF_GUID!{GUID_NWF_UPDATE_IE, 15}
DEFINE_NWF_GUID!{GUID_NWF_RESET_REQUEST, 16}
DEFINE_NWF_GUID!{GUID_NWF_NIC_POWER_STATE, 17}
DEFINE_NWF_GUID!{GUID_NWF_OPTIONAL_CAPABILITY, 18}
DEFINE_NWF_GUID!{GUID_NWF_CURRENT_OPTIONAL_CAPABILITY, 19}
DEFINE_NWF_GUID!{GUID_NWF_STATION_ID, 20}
DEFINE_NWF_GUID!{GUID_NWF_MEDIUM_OCCUPANCY_LIMIT, 21}
DEFINE_NWF_GUID!{GUID_NWF_CF_POLLABLE, 22}
DEFINE_NWF_GUID!{GUID_NWF_CFP_PERIOD, 23}
DEFINE_NWF_GUID!{GUID_NWF_CFP_MAX_DURATION, 24}
DEFINE_NWF_GUID!{GUID_NWF_POWER_MGMT_MODE, 25}
DEFINE_NWF_GUID!{GUID_NWF_OPERATIONAL_RATE_SET, 26}
DEFINE_NWF_GUID!{GUID_NWF_BEACON_PERIOD, 27}
DEFINE_NWF_GUID!{GUID_NWF_DTIM_PERIOD, 28}
DEFINE_NWF_GUID!{GUID_NWF_WEP_ICV_ERROR_COUNT, 29}
DEFINE_NWF_GUID!{GUID_NWF_MAC_ADDRESS, 30}
DEFINE_NWF_GUID!{GUID_NWF_RTS_THRESHOLD, 31}
DEFINE_NWF_GUID!{GUID_NWF_SHORT_RETRY_LIMIT, 32}
DEFINE_NWF_GUID!{GUID_NWF_LONG_RETRY_LIMIT, 33}
DEFINE_NWF_GUID!{GUID_NWF_FRAGMENTATION_THRESHOLD, 34}
DEFINE_NWF_GUID!{GUID_NWF_MAX_TRANSMIT_MSDU_LIFETIME, 35}
DEFINE_NWF_GUID!{GUID_NWF_MAX_RECEIVE_LIFETIME, 36}
DEFINE_NWF_GUID!{GUID_NWF_COUNTERS_ENTRY, 37}
DEFINE_NWF_GUID!{GUID_NWF_SUPPORTED_PHY_TYPES, 38}
DEFINE_NWF_GUID!{GUID_NWF_CURRENT_REG_DOMAIN, 39}
DEFINE_NWF_GUID!{GUID_NWF_TEMP_TYPE, 40}
DEFINE_NWF_GUID!{GUID_NWF_CURRENT_TX_ANTENNA, 41}
DEFINE_NWF_GUID!{GUID_NWF_DIVERSITY_SUPPORT, 42}
DEFINE_NWF_GUID!{GUID_NWF_CURRENT_RX_ANTENNA, 43}
DEFINE_NWF_GUID!{GUID_NWF_SUPPORTED_POWER_LEVELS, 44}
DEFINE_NWF_GUID!{GUID_NWF_CURRENT_TX_POWER_LEVEL, 45}
DEFINE_NWF_GUID!{GUID_NWF_HOP_TIME, 46}
DEFINE_NWF_GUID!{GUID_NWF_CURRENT_CHANNEL_NUMBER, 47}
DEFINE_NWF_GUID!{GUID_NWF_MAX_DWELL_TIME, 48}
DEFINE_NWF_GUID!{GUID_NWF_CURRENT_DWELL_TIME, 49}
DEFINE_NWF_GUID!{GUID_NWF_CURRENT_SET, 50}
DEFINE_NWF_GUID!{GUID_NWF_CURRENT_PATTERN, 51}
DEFINE_NWF_GUID!{GUID_NWF_CURRENT_INDEX, 52}
DEFINE_NWF_GUID!{GUID_NWF_CURRENT_CHANNEL, 53}
DEFINE_NWF_GUID!{GUID_NWF_CCA_MODE_SUPPORTED, 54}
DEFINE_NWF_GUID!{GUID_NWF_CURRENT_CCA_MODE, 55}
DEFINE_NWF_GUID!{GUID_NWF_ED_THRESHOLD, 56}
DEFINE_NWF_GUID!{GUID_NWF_CCA_WATCHDOG_TIMER_MAX, 57}
DEFINE_NWF_GUID!{GUID_NWF_CCA_WATCHDOG_COUNT_MAX, 58}
DEFINE_NWF_GUID!{GUID_NWF_CCA_WATCHDOG_TIMER_MIN, 59}
DEFINE_NWF_GUID!{GUID_NWF_CCA_WATCHDOG_COUNT_MIN, 60}
DEFINE_NWF_GUID!{GUID_NWF_REG_DOMAINS_SUPPORT_VALUE, 61}
DEFINE_NWF_GUID!{GUID_NWF_SUPPORTED_TX_ANTENNA, 62}
DEFINE_NWF_GUID!{GUID_NWF_SUPPORTED_RX_ANTENNA, 63}
DEFINE_NWF_GUID!{GUID_NWF_DIVERSITY_SELECTION_RX, 64}
DEFINE_NWF_GUID!{GUID_NWF_SUPPORTED_DATA_RATES_VALUE, 65}
DEFINE_NWF_GUID!{GUID_NWF_CURRENT_FREQUENCY, 66}
DEFINE_NWF_GUID!{GUID_NWF_TI_THRESHOLD, 67}
DEFINE_NWF_GUID!{GUID_NWF_FREQUENCY_BANDS_SUPPORTED, 68}
DEFINE_NWF_GUID!{GUID_NWF_SHORT_PREAMBLE_OPTION_IMPLEMENTED, 69}
DEFINE_NWF_GUID!{GUID_NWF_PBCC_OPTION_IMPLEMENTED, 70}
DEFINE_NWF_GUID!{GUID_NWF_CHANNEL_AGILITY_PRESENT, 71}
DEFINE_NWF_GUID!{GUID_NWF_CHANNEL_AGILITY_ENABLED, 72}
DEFINE_NWF_GUID!{GUID_NWF_HR_CCA_MODE_SUPPORTED, 73}
DEFINE_NWF_GUID!{GUID_NWF_MULTI_DOMAIN_CAPABILITY_IMPLEMENTED, 74}
DEFINE_NWF_GUID!{GUID_NWF_MULTI_DOMAIN_CAPABILITY_ENABLED, 75}
DEFINE_NWF_GUID!{GUID_NWF_COUNTRY_STRING, 76}
DEFINE_NWF_GUID!{GUID_NWF_MULTI_DOMAIN_CAPABILITY, 77}
DEFINE_NWF_GUID!{GUID_NWF_EHCC_PRIME_RADIX, 78}
DEFINE_NWF_GUID!{GUID_NWF_EHCC_NUMBER_OF_CHANNELS_FAMILY_INDEX, 79}
DEFINE_NWF_GUID!{GUID_NWF_EHCC_CAPABILITY_IMPLEMENTED, 80}
DEFINE_NWF_GUID!{GUID_NWF_EHCC_CAPABILITY_ENABLED, 81}
DEFINE_NWF_GUID!{GUID_NWF_HOP_ALGORITHM_ADOPTED, 82}
DEFINE_NWF_GUID!{GUID_NWF_RANDOM_TABLE_FLAG, 83}
DEFINE_NWF_GUID!{GUID_NWF_NUMBER_OF_HOPPING_SETS, 84}
DEFINE_NWF_GUID!{GUID_NWF_HOP_MODULUS, 85}
DEFINE_NWF_GUID!{GUID_NWF_HOP_OFFSET, 86}
DEFINE_NWF_GUID!{GUID_NWF_HOPPING_PATTERN, 87}
DEFINE_NWF_GUID!{GUID_NWF_RANDOM_TABLE_FIELD_NUMBER, 88}
DEFINE_NWF_GUID!{GUID_NWF_WPA_TSC, 89}
DEFINE_NWF_GUID!{GUID_NWF_RSSI_RANGE, 90}
DEFINE_NWF_GUID!{GUID_NWF_RF_USAGE, 91}
DEFINE_NWF_GUID!{GUID_NWF_NIC_SPECIFIC_EXTENSION, 92}
DEFINE_NWF_GUID!{GUID_NWF_AP_JOIN_REQUEST, 93}
DEFINE_NWF_GUID!{GUID_NWF_ERP_PBCC_OPTION_IMPLEMENTED, 94}
DEFINE_NWF_GUID!{GUID_NWF_ERP_PBCC_OPTION_ENABLED, 95}
DEFINE_NWF_GUID!{GUID_NWF_DSSS_OFDM_OPTION_IMPLEMENTED, 96}
DEFINE_NWF_GUID!{GUID_NWF_DSSS_OFDM_OPTION_ENABLED, 97}
DEFINE_NWF_GUID!{GUID_NWF_SHORT_SLOT_TIME_OPTION_IMPLEMENTED, 98}
DEFINE_NWF_GUID!{GUID_NWF_SHORT_SLOT_TIME_OPTION_ENABLED, 99}
DEFINE_NWF_GUID!{GUID_NWF_MAX_MAC_ADDRESS_STATES, 100}
DEFINE_NWF_GUID!{GUID_NWF_RECV_SENSITIVITY_LIST, 101}
DEFINE_NWF_GUID!{GUID_NWF_WME_IMPLEMENTED, 102}
DEFINE_NWF_GUID!{GUID_NWF_WME_ENABLED, 103}
DEFINE_NWF_GUID!{GUID_NWF_WME_AC_PARAMETERS, 104}
DEFINE_NWF_GUID!{GUID_NWF_WME_UPDATE_IE, 105}
DEFINE_NWF_GUID!{GUID_NWF_QOS_TX_QUEUES_SUPPORTED, 106}
DEFINE_NWF_GUID!{GUID_NWF_QOS_TX_DURATION, 107}
DEFINE_NWF_GUID!{GUID_NWF_QOS_TX_MEDIUM_TIME, 108}
DEFINE_NWF_GUID!{GUID_NWF_SUPPORTED_OFDM_FREQUENCY_LIST, 109}
DEFINE_NWF_GUID!{GUID_NWF_SUPPORTED_DSSS_CHANNEL_LIST, 110}
DEFINE_NWF_GUID!{GUID_NWF_AUTO_CONFIG_ENABLED, 120}
DEFINE_NWF_GUID!{GUID_NWF_ENUM_BSS_LIST, 121}
DEFINE_NWF_GUID!{GUID_NWF_FLUSH_BSS_LIST, 122}
DEFINE_NWF_GUID!{GUID_NWF_POWER_MGMT_REQUEST, 123}
DEFINE_NWF_GUID!{GUID_NWF_DESIRED_SSID_LIST, 124}
DEFINE_NWF_GUID!{GUID_NWF_EXCLUDED_MAC_ADDRESS_LIST, 125}
DEFINE_NWF_GUID!{GUID_NWF_DESIRED_BSSID_LIST, 126}
DEFINE_NWF_GUID!{GUID_NWF_DESIRED_BSS_TYPE, 127}
DEFINE_NWF_GUID!{GUID_NWF_PMKID_LIST, 128}
DEFINE_NWF_GUID!{GUID_NWF_CONNECT_REQUEST, 129}
DEFINE_NWF_GUID!{GUID_NWF_EXCLUDE_UNENCRYPTED, 130}
DEFINE_NWF_GUID!{GUID_NWF_STATISTICS, 131}
DEFINE_NWF_GUID!{GUID_NWF_PRIVACY_EXEMPTION_LIST, 132}
DEFINE_NWF_GUID!{GUID_NWF_ENABLED_AUTHENTICATION_ALGORITHM, 133}
DEFINE_NWF_GUID!{GUID_NWF_SUPPORTED_UNICAST_ALGORITHM_PAIR, 134}
DEFINE_NWF_GUID!{GUID_NWF_ENABLED_UNICAST_CIPHER_ALGORITHM, 135}
DEFINE_NWF_GUID!{GUID_NWF_SUPPORTED_MULTICAST_ALGORITHM_PAIR, 136}
DEFINE_NWF_GUID!{GUID_NWF_ENABLED_MULTICAST_CIPHER_ALGORITHM, 137}
DEFINE_NWF_GUID!{GUID_NWF_CIPHER_DEFAULT_KEY_ID, 138}
DEFINE_NWF_GUID!{GUID_NWF_CIPHER_DEFAULT_KEY, 139}
DEFINE_NWF_GUID!{GUID_NWF_CIPHER_KEY_MAPPING_KEY, 140}
DEFINE_NWF_GUID!{GUID_NWF_ENUM_ASSOCIATION_INFO, 141}
DEFINE_NWF_GUID!{GUID_NWF_DISCONNECT_REQUEST, 142}
DEFINE_NWF_GUID!{GUID_NWF_UNICAST_USE_GROUP_ENABLED, 143}
DEFINE_NWF_GUID!{GUID_NWF_PHY_STATE, 144}
DEFINE_NWF_GUID!{GUID_NWF_DESIRED_PHY_LIST, 145}
DEFINE_NWF_GUID!{GUID_NWF_CURRENT_PHY_ID, 146}
DEFINE_NWF_GUID!{GUID_NWF_MEDIA_STREAMING_ENABLED, 147}
DEFINE_NWF_GUID!{GUID_NWF_UNREACHABLE_DETECTION_THRESHOLD, 148}
DEFINE_NWF_GUID!{GUID_NWF_ACTIVE_PHY_LIST, 149}
DEFINE_NWF_GUID!{GUID_NWF_EXTSTA_CAPABILITY, 150}
DEFINE_NWF_GUID!{GUID_NWF_DATA_RATE_MAPPING_TABLE, 151}
DEFINE_NWF_GUID!{GUID_NWF_SUPPORTED_COUNTRY_OR_REGION_STRING, 152}
DEFINE_NWF_GUID!{GUID_NWF_DESIRED_COUNTRY_OR_REGION_STRING, 153}
DEFINE_NWF_GUID!{GUID_NWF_PORT_STATE_NOTIFICATION, 154}
DEFINE_NWF_GUID!{GUID_NWF_IBSS_PARAMS, 155}
DEFINE_NWF_GUID!{GUID_NWF_QOS_PARAMS, 156}
DEFINE_NWF_GUID!{GUID_NWF_SAFE_MODE_ENABLED, 157}
DEFINE_NWF_GUID!{GUID_NWF_HIDDEN_NETWORK_ENABLED, 158}
DEFINE_NWF_GUID!{GUID_NWF_ASSOCIATION_PARAMS, 159}
DEFINE_NWF_GUID!{GUID_NWF_CURRENT_ADDRESS, 1024 + 2}
DEFINE_NWF_GUID!{GUID_NWF_PERMANENT_ADDRESS, 1024 + 3}
DEFINE_NWF_GUID!{GUID_NWF_MULTICAST_LIST, 1024 + 4}
DEFINE_NWF_GUID!{GUID_NWF_MAXIMUM_LIST_SIZE, 1024 + 5}
pub const DOT11_EXTAP_ATTRIBUTES_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_EXTAP_ATTRIBUTES {
    Header: NDIS_OBJECT_HEADER,
    uScanSSIDListSize: ULONG,
    uDesiredSSIDListSize: ULONG,
    uPrivacyExemptionListSize: ULONG,
    uAssociationTableSize: ULONG,
    uDefaultKeyTableSize: ULONG,
    uWEPKeyValueMaxLength: ULONG,
    bStrictlyOrderedServiceClassImplemented: BOOLEAN,
    uNumSupportedCountryOrRegionStrings: ULONG,
    pSupportedCountryOrRegionStrings: PDOT11_COUNTRY_OR_REGION_STRING,
    uInfraNumSupportedUcastAlgoPairs: ULONG,
    pInfraSupportedUcastAlgoPairs: PDOT11_AUTH_CIPHER_PAIR,
    uInfraNumSupportedMcastAlgoPairs: ULONG,
    pInfraSupportedMcastAlgoPairs: PDOT11_AUTH_CIPHER_PAIR,
}}
pub type PDOT11_EXTAP_ATTRIBUTES = *mut DOT11_EXTAP_ATTRIBUTES;
pub const DOT11_INCOMING_ASSOC_STARTED_PARAMETERS_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_INCOMING_ASSOC_STARTED_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    PeerMacAddr: DOT11_MAC_ADDRESS,
}}
pub type PDOT11_INCOMING_ASSOC_STARTED_PARAMETERS = *mut DOT11_INCOMING_ASSOC_STARTED_PARAMETERS;
pub const DOT11_INCOMING_ASSOC_REQUEST_RECEIVED_PARAMETERS_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_INCOMING_ASSOC_REQUEST_RECEIVED_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    PeerMacAddr: DOT11_MAC_ADDRESS,
    bReAssocReq: BOOLEAN,
    uAssocReqOffset: ULONG,
    uAssocReqSize: ULONG,
}}
pub type PDOT11_INCOMING_ASSOC_REQUEST_RECEIVED_PARAMETERS =
    *mut DOT11_INCOMING_ASSOC_REQUEST_RECEIVED_PARAMETERS;
pub const DOT11_ASSOC_ERROR_SOURCE_OS: UCHAR = 0x0;
pub const DOT11_ASSOC_ERROR_SOURCE_REMOTE: UCHAR = 0x01;
pub const DOT11_ASSOC_ERROR_SOURCE_OTHER: UCHAR = 0xFF;
pub const DOT11_INCOMING_ASSOC_COMPLETION_PARAMETERS_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_INCOMING_ASSOC_COMPLETION_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    PeerMacAddr: DOT11_MAC_ADDRESS,
    uStatus: ULONG,
    ucErrorSource: UCHAR,
    bReAssocReq: BOOLEAN,
    bReAssocResp: BOOLEAN,
    uAssocReqOffset: ULONG,
    uAssocReqSize: ULONG,
    uAssocRespOffset: ULONG,
    uAssocRespSize: ULONG,
    AuthAlgo: DOT11_AUTH_ALGORITHM,
    UnicastCipher: DOT11_CIPHER_ALGORITHM,
    MulticastCipher: DOT11_CIPHER_ALGORITHM,
    uActivePhyListOffset: ULONG,
    uActivePhyListSize: ULONG,
    uBeaconOffset: ULONG,
    uBeaconSize: ULONG,
}}
pub type PDOT11_INCOMING_ASSOC_COMPLETION_PARAMETERS =
    *mut DOT11_INCOMING_ASSOC_COMPLETION_PARAMETERS;
pub const DOT11_STOP_AP_PARAMETERS_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_STOP_AP_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    ulReason: ULONG,
}}
pub type PDOT11_STOP_AP_PARAMETERS = *mut DOT11_STOP_AP_PARAMETERS;
pub const DOT11_STOP_AP_REASON_FREQUENCY_NOT_AVAILABLE: ULONG = 0x1;
pub const DOT11_STOP_AP_REASON_CHANNEL_NOT_AVAILABLE: ULONG = 0x2;
pub const DOT11_STOP_AP_REASON_AP_ACTIVE: ULONG = 0x3;
pub const DOT11_STOP_AP_REASON_IHV_START: ULONG = 0xFF000000;
pub const DOT11_STOP_AP_REASON_IHV_END: ULONG = 0xFFFFFFFF;
pub const DOT11_PHY_FREQUENCY_ADOPTED_PARAMETERS_REVISION_1: UCHAR = 1;
UNION!{union DOT11_PHY_FREQUENCY_ADOPTED_PARAMETERS_u {
    [u32; 1],
    ulChannel ulChannel_mut: ULONG,
    ulFrequency ulFrequency_mut: ULONG,
}}
STRUCT!{struct DOT11_PHY_FREQUENCY_ADOPTED_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    ulPhyId: ULONG,
    u: DOT11_PHY_FREQUENCY_ADOPTED_PARAMETERS_u,
}}
pub type PDOT11_PHY_FREQUENCY_ADOPTED_PARAMETERS = *mut DOT11_PHY_FREQUENCY_ADOPTED_PARAMETERS;
pub const DOT11_CAN_SUSTAIN_AP_PARAMETERS_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_CAN_SUSTAIN_AP_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    ulReason: ULONG,
}}
pub type PDOT11_CAN_SUSTAIN_AP_PARAMETERS = *mut DOT11_CAN_SUSTAIN_AP_PARAMETERS;
pub const DOT11_CAN_SUSTAIN_AP_REASON_IHV_START: ULONG = 0xFF000000;
pub const DOT11_CAN_SUSTAIN_AP_REASON_IHV_END: ULONG = 0xFFFFFFFF;
pub const NWF_EXTAP_OID: u32 = 0x03;
pub const OID_DOT11_WPS_ENABLED: u32 = NWF_DEFINE_OID!(0x01, NWF_EXTAP_OID, NWF_MANDATORY_OID);
pub const OID_DOT11_START_AP_REQUEST: u32 =
    NWF_DEFINE_OID!(0x02, NWF_EXTAP_OID, NWF_MANDATORY_OID);
pub const OID_DOT11_AVAILABLE_CHANNEL_LIST: u32 =
    NWF_DEFINE_OID!(0x03, NWF_EXTAP_OID, NWF_MANDATORY_OID);
pub const DOT11_AVAILABLE_CHANNEL_LIST_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_AVAILABLE_CHANNEL_LIST {
    Header: NDIS_OBJECT_HEADER,
    uNumOfEntries: ULONG,
    uTotalNumOfEntries: ULONG,
    uChannelNumber: [ULONG; 1],
}}
pub type PDOT11_AVAILABLE_CHANNEL_LIST = *mut DOT11_AVAILABLE_CHANNEL_LIST;
pub const OID_DOT11_AVAILABLE_FREQUENCY_LIST: u32 =
    NWF_DEFINE_OID!(0x04, NWF_EXTAP_OID, NWF_MANDATORY_OID);
pub const DOT11_AVAILABLE_FREQUENCY_LIST_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_AVAILABLE_FREQUENCY_LIST {
    Header: NDIS_OBJECT_HEADER,
    uNumOfEntries: ULONG,
    uTotalNumOfEntries: ULONG,
    uFrequencyValue: [ULONG; 1],
}}
pub type PDOT11_AVAILABLE_FREQUENCY_LIST = *mut DOT11_AVAILABLE_FREQUENCY_LIST;
pub const OID_DOT11_DISASSOCIATE_PEER_REQUEST: u32 =
    NWF_DEFINE_OID!(0x05, NWF_EXTAP_OID, NWF_MANDATORY_OID);
pub const DOT11_DISASSOCIATE_PEER_REQUEST_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_DISASSOCIATE_PEER_REQUEST {
    Header: NDIS_OBJECT_HEADER,
    PeerMacAddr: DOT11_MAC_ADDRESS,
    usReason: USHORT,
}}
pub type PDOT11_DISASSOCIATE_PEER_REQUEST = *mut DOT11_DISASSOCIATE_PEER_REQUEST;
pub const OID_DOT11_INCOMING_ASSOCIATION_DECISION: u32 =
    NWF_DEFINE_OID!(0x06, NWF_EXTAP_OID, NWF_MANDATORY_OID);
pub const DOT11_INCOMING_ASSOC_DECISION_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_INCOMING_ASSOC_DECISION {
    Header: NDIS_OBJECT_HEADER,
    PeerMacAddr: DOT11_MAC_ADDRESS,
    bAccept: BOOLEAN,
    usReasonCode: USHORT,
    uAssocResponseIEsOffset: ULONG,
    uAssocResponseIEsLength: ULONG,
}}
pub type PDOT11_INCOMING_ASSOC_DECISION = *mut DOT11_INCOMING_ASSOC_DECISION;
pub const DOT11_INCOMING_ASSOC_DECISION_REVISION_2: UCHAR = 2;
STRUCT!{struct DOT11_INCOMING_ASSOC_DECISION_V2 {
    Header: NDIS_OBJECT_HEADER,
    PeerMacAddr: DOT11_MAC_ADDRESS,
    bAccept: BOOLEAN,
    usReasonCode: USHORT,
    uAssocResponseIEsOffset: ULONG,
    uAssocResponseIEsLength: ULONG,
    WFDStatus: DOT11_WFD_STATUS_CODE,
}}
pub type PDOT11_INCOMING_ASSOC_DECISION_V2 = *mut DOT11_INCOMING_ASSOC_DECISION_V2;
pub const OID_DOT11_ADDITIONAL_IE: u32 = NWF_DEFINE_OID!(0x07, NWF_EXTAP_OID, NWF_MANDATORY_OID);
pub const DOT11_ADDITIONAL_IE_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_ADDITIONAL_IE {
    Header: NDIS_OBJECT_HEADER,
    uBeaconIEsOffset: ULONG,
    uBeaconIEsLength: ULONG,
    uResponseIEsOffset: ULONG,
    uResponseIEsLength: ULONG,
}}
pub type PDOT11_ADDITIONAL_IE = *mut DOT11_ADDITIONAL_IE;
pub const DOT11_EXTAP_SEND_CONTEXT_REVISION_1: UCHAR = 1;
pub const DOT11_EXTAP_RECV_CONTEXT_REVISION_1: UCHAR = 1;
pub const OID_DOT11_ENUM_PEER_INFO: u32 = NWF_DEFINE_OID!(0x08, NWF_EXTAP_OID, NWF_MANDATORY_OID);
STRUCT!{struct DOT11_PEER_STATISTICS {
    ullDecryptSuccessCount: ULONGLONG,
    ullDecryptFailureCount: ULONGLONG,
    ullTxPacketSuccessCount: ULONGLONG,
    ullTxPacketFailureCount: ULONGLONG,
    ullRxPacketSuccessCount: ULONGLONG,
    ullRxPacketFailureCount: ULONGLONG,
}}
pub type PDOT11_PEER_STATISTICS = *mut DOT11_PEER_STATISTICS;
STRUCT!{struct DOT11_PEER_INFO {
    MacAddress: DOT11_MAC_ADDRESS,
    usCapabilityInformation: USHORT,
    AuthAlgo: DOT11_AUTH_ALGORITHM,
    UnicastCipherAlgo: DOT11_CIPHER_ALGORITHM,
    MulticastCipherAlgo: DOT11_CIPHER_ALGORITHM,
    bWpsEnabled: BOOLEAN,
    usListenInterval: USHORT,
    ucSupportedRates: [UCHAR; MAX_NUM_SUPPORTED_RATES_V2],
    usAssociationID: USHORT,
    AssociationState: DOT11_ASSOCIATION_STATE,
    PowerMode: DOT11_POWER_MODE,
    liAssociationUpTime: LARGE_INTEGER,
    Statistics: DOT11_PEER_STATISTICS,
}}
pub type PDOT11_PEER_INFO = *mut DOT11_PEER_INFO;
pub const DOT11_PEER_INFO_LIST_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_PEER_INFO_LIST {
    Header: NDIS_OBJECT_HEADER,
    uNumOfEntries: ULONG,
    uTotalNumOfEntries: ULONG,
    PeerInfo: [DOT11_PEER_INFO; 1],
}}
pub type PDOT11_PEER_INFO_LIST = *mut DOT11_PEER_INFO_LIST;
pub const DOT11_VWIFI_COMBINATION_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_VWIFI_COMBINATION {
    Header: NDIS_OBJECT_HEADER,
    uNumInfrastructure: ULONG,
    uNumAdhoc: ULONG,
    uNumSoftAP: ULONG,
}}
pub type PDOT11_VWIFI_COMBINATION = *mut DOT11_VWIFI_COMBINATION;
pub const DOT11_VWIFI_COMBINATION_REVISION_2: UCHAR = 2;
STRUCT!{struct DOT11_VWIFI_COMBINATION_V2 {
    Header: NDIS_OBJECT_HEADER,
    uNumInfrastructure: ULONG,
    uNumAdhoc: ULONG,
    uNumSoftAP: ULONG,
    uNumVirtualStation: ULONG,
}}
pub type PDOT11_VWIFI_COMBINATION_V2 = *mut DOT11_VWIFI_COMBINATION_V2;
pub const DOT11_VWIFI_COMBINATION_REVISION_3: UCHAR = 3;
STRUCT!{struct DOT11_VWIFI_COMBINATION_V3 {
    Header: NDIS_OBJECT_HEADER,
    uNumInfrastructure: ULONG,
    uNumAdhoc: ULONG,
    uNumSoftAP: ULONG,
    uNumVirtualStation: ULONG,
    uNumWFDGroup: ULONG,
}}
pub type PDOT11_VWIFI_COMBINATION_V3 = *mut DOT11_VWIFI_COMBINATION_V3;
pub const DOT11_VWIFI_ATTRIBUTES_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_VWIFI_ATTRIBUTES {
    Header: NDIS_OBJECT_HEADER,
    uTotalNumOfEntries: ULONG,
    Combinations: [DOT11_VWIFI_COMBINATION; 1],
}}
pub type PDOT11_VWIFI_ATTRIBUTES = *mut DOT11_VWIFI_ATTRIBUTES;
pub const NWF_VWIFI_OID: u32 = 0x04;
pub const OID_DOT11_CREATE_MAC: u32 = NWF_DEFINE_OID!(0x01, NWF_VWIFI_OID, NWF_MANDATORY_OID);
pub const DOT11_MAC_PARAMETERS_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_MAC_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    uOpmodeMask: ULONG,
}}
pub type PDOT11_MAC_PARAMETERS = *mut DOT11_MAC_PARAMETERS;
STRUCT!{struct DOT11_MAC_INFO {
    uReserved: ULONG,
    uNdisPortNumber: ULONG,
    MacAddr: DOT11_MAC_ADDRESS,
}}
pub type PDOT11_MAC_INFO = *mut DOT11_MAC_INFO;
pub const OID_DOT11_DELETE_MAC: u32 = NWF_DEFINE_OID!(0x02, NWF_VWIFI_OID, NWF_MANDATORY_OID);
pub const OID_DOT11_PREFERRED_MAC: u32 = NWF_DEFINE_OID!(0x03, NWF_VWIFI_OID, NWF_MANDATORY_OID);
pub const OID_DOT11_VIRTUAL_STATION_CAPABILITY: u32 =
    NWF_DEFINE_OID!(0x04, NWF_VWIFI_OID, NWF_OPTIONAL_OID);
pub const DOT11_WFD_ATTRIBUTES_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_WFD_ATTRIBUTES {
    Header: NDIS_OBJECT_HEADER,
    uNumConcurrentGORole: ULONG,
    uNumConcurrentClientRole: ULONG,
    WPSVersionsSupported: ULONG,
    bServiceDiscoverySupported: BOOLEAN,
    bClientDiscoverabilitySupported: BOOLEAN,
    bInfrastructureManagementSupported: BOOLEAN,
    uMaxSecondaryDeviceTypeListSize: ULONG,
    DeviceAddress: DOT11_MAC_ADDRESS,
    uInterfaceAddressListCount: ULONG,
    pInterfaceAddressList: PDOT11_MAC_ADDRESS,
    uNumSupportedCountryOrRegionStrings: ULONG,
    pSupportedCountryOrRegionStrings: PDOT11_COUNTRY_OR_REGION_STRING,
    uDiscoveryFilterListSize: ULONG,
    uGORoleClientTableSize: ULONG,
}}
pub type PDOT11_WFD_ATTRIBUTES = *mut DOT11_WFD_ATTRIBUTES;
pub type DOT11_WFD_GROUP_CAPABILITY = UCHAR;
pub const DOT11_WFD_STATUS_SUCCESS: DOT11_WFD_STATUS_CODE = 0;
pub const DOT11_WFD_STATUS_FAILED_INFORMATION_IS_UNAVAILABLE: DOT11_WFD_STATUS_CODE = 1;
pub const DOT11_WFD_STATUS_FAILED_INCOMPATIBLE_PARAMETERS: DOT11_WFD_STATUS_CODE = 2;
pub const DOT11_WFD_STATUS_FAILED_LIMIT_REACHED: DOT11_WFD_STATUS_CODE = 3;
pub const DOT11_WFD_STATUS_FAILED_INVALID_PARAMETERS: DOT11_WFD_STATUS_CODE = 4;
pub const DOT11_WFD_STATUS_FAILED_UNABLE_TO_ACCOMODATE_REQUEST: DOT11_WFD_STATUS_CODE = 5;
pub const DOT11_WFD_STATUS_FAILED_PREVIOUS_PROTOCOL_ERROR: DOT11_WFD_STATUS_CODE = 6;
pub const DOT11_WFD_STATUS_FAILED_NO_COMMON_CHANNELS: DOT11_WFD_STATUS_CODE = 7;
pub const DOT11_WFD_STATUS_FAILED_UNKNOWN_WFD_GROUP: DOT11_WFD_STATUS_CODE = 8;
pub const DOT11_WFD_STATUS_FAILED_MATCHING_MAX_INTENT: DOT11_WFD_STATUS_CODE = 9;
pub const DOT11_WFD_STATUS_FAILED_INCOMPATIBLE_PROVISIONING_METHOD: DOT11_WFD_STATUS_CODE = 10;
pub const DOT11_WFD_STATUS_FAILED_REJECTED_BY_USER: DOT11_WFD_STATUS_CODE = 11;
pub const DOT11_WFD_STATUS_SUCCESS_ACCEPTED_BY_USER: DOT11_WFD_STATUS_CODE = 12;
#[inline]
pub fn WFD_STATUS_SUCCEEDED(status: DOT11_WFD_STATUS_CODE) -> bool {
    status == DOT11_WFD_STATUS_SUCCESS || status == DOT11_WFD_STATUS_SUCCESS_ACCEPTED_BY_USER
}
#[inline]
pub fn WFD_STATUS_FAILED(status: DOT11_WFD_STATUS_CODE) -> bool {
    status != DOT11_WFD_STATUS_SUCCESS || status != DOT11_WFD_STATUS_SUCCESS_ACCEPTED_BY_USER
}
pub const DOT11_WFD_MINOR_REASON_SUCCESS: i32 = 0;
pub const DOT11_WFD_MINOR_REASON_DISASSOCIATED_FROM_WLAN_CROSS_CONNECTION_POLICY: i32 = 1;
pub const DOT11_WFD_MINOR_REASON_DISASSOCIATED_NOT_MANAGED_INFRASTRUCTURE_CAPABLE: i32 = 2;
pub const DOT11_WFD_MINOR_REASON_DISASSOCIATED_WFD_COEXISTENCE_POLICY: i32 = 3;
pub const DOT11_WFD_MINOR_REASON_DISASSOCIATED_INFRASTRUCTURE_MANAGED_POLICY: i32 = 4;
pub const DOT11_WPS_VERSION_1_0: u8 = 0x01;
pub const DOT11_WPS_VERSION_2_0: u8 = 0x02;
pub const DOT11_WFD_DEVICE_CAPABILITY_SERVICE_DISCOVERY: u8 = 0x01;
pub const DOT11_WFD_DEVICE_CAPABILITY_P2P_CLIENT_DISCOVERABILITY: u8 = 0x02;
pub const DOT11_WFD_DEVICE_CAPABILITY_CONCURRENT_OPERATION: u8 = 0x04;
pub const DOT11_WFD_DEVICE_CAPABILITY_P2P_INFRASTRUCTURE_MANAGED: u8 = 0x08;
pub const DOT11_WFD_DEVICE_CAPABILITY_P2P_DEVICE_LIMIT: u8 = 0x10;
pub const DOT11_WFD_DEVICE_CAPABILITY_P2P_INVITATION_PROCEDURE: u8 = 0x20;
pub const DOT11_WFD_DEVICE_CAPABILITY_RESERVED_6: u8 = 0x40;
pub const DOT11_WFD_DEVICE_CAPABILITY_RESERVED_7: u8 = 0x80;
pub const DOT11_WFD_GROUP_CAPABILITY_NONE: u8 = 0x00;
pub const DOT11_WFD_GROUP_CAPABILITY_GROUP_OWNER: u8 = 0x01;
pub const DOT11_WFD_GROUP_CAPABILITY_PERSISTENT_GROUP: u8 = 0x02;
pub const DOT11_WFD_GROUP_CAPABILITY_GROUP_LIMIT_REACHED: u8 = 0x04;
pub const DOT11_WFD_GROUP_CAPABILITY_INTRABSS_DISTRIBUTION_SUPPORTED: u8 = 0x08;
pub const DOT11_WFD_GROUP_CAPABILITY_CROSS_CONNECTION_SUPPORTED: u8 = 0x10;
pub const DOT11_WFD_GROUP_CAPABILITY_PERSISTENT_RECONNECT_SUPPORTED: u8 = 0x20;
pub const DOT11_WFD_GROUP_CAPABILITY_IN_GROUP_FORMATION: u8 = 0x40;
pub const DOT11_WFD_GROUP_CAPABILITY_RESERVED_7: u8 = 0x80;
pub const DOT11_WFD_GROUP_CAPABILITY_EAPOL_KEY_IP_ADDRESS_ALLOCATION_SUPPORTED: u8 = 0x80;
pub const DOT11_WPS_DEVICE_NAME_MAX_LENGTH: usize = 32;
pub const DOT11_WPS_MAX_PASSKEY_LENGTH: usize = 8;
pub const DOT11_WPS_MAX_MODEL_NAME_LENGTH: usize = 32;
pub const DOT11_WPS_MAX_MODEL_NUMBER_LENGTH: usize = 32;
STRUCT!{struct DOT11_WFD_DEVICE_TYPE {
    CategoryID: USHORT,
    SubCategoryID: USHORT,
    OUI: [UCHAR; 4],
}}
pub type PDOT11_WFD_DEVICE_TYPE = *mut DOT11_WFD_DEVICE_TYPE;
STRUCT!{struct DOT11_WPS_DEVICE_NAME {
    uDeviceNameLength: ULONG,
    ucDeviceName: [UCHAR; DOT11_WPS_DEVICE_NAME_MAX_LENGTH],
}}
pub type PDOT11_WPS_DEVICE_NAME = *mut DOT11_WPS_DEVICE_NAME;
STRUCT!{struct DOT11_WFD_CONFIGURATION_TIMEOUT {
    GOTimeout: UCHAR,
    ClientTimeout: UCHAR,
}}
pub type PDOT11_WFD_CONFIGURATION_TIMEOUT = *mut DOT11_WFD_CONFIGURATION_TIMEOUT;
STRUCT!{struct DOT11_WFD_GROUP_ID {
    DeviceAddress: DOT11_MAC_ADDRESS,
    SSID: DOT11_SSID,
}}
pub type PDOT11_WFD_GROUP_ID = *mut DOT11_WFD_GROUP_ID;
STRUCT!{#[repr(packed)] struct DOT11_WFD_GO_INTENT {
    Bitfields: UCHAR,
}}
BITFIELD!{DOT11_WFD_GO_INTENT Bitfields: UCHAR [
    TieBreaker set_TieBreaker[0..1],
    Intent set_Intent[1..8],
]}
pub type PDOT11_WFD_GO_INTENT = *mut DOT11_WFD_GO_INTENT;
STRUCT!{struct DOT11_WFD_CHANNEL {
    CountryRegionString: DOT11_COUNTRY_OR_REGION_STRING,
    OperatingClass: UCHAR,
    ChannelNumber: UCHAR,
}}
pub type PDOT11_WFD_CHANNEL = *mut DOT11_WFD_CHANNEL;
ENUM!{enum DOT11_WPS_CONFIG_METHOD {
    DOT11_WPS_CONFIG_METHOD_NULL = 0,
    DOT11_WPS_CONFIG_METHOD_DISPLAY = 0x0008,
    DOT11_WPS_CONFIG_METHOD_NFC_TAG = 0x0020,
    DOT11_WPS_CONFIG_METHOD_NFC_INTERFACE = 0x0040,
    DOT11_WPS_CONFIG_METHOD_PUSHBUTTON = 0x0080,
    DOT11_WPS_CONFIG_METHOD_KEYPAD = 0x0100,
    DOT11_WPS_CONFIG_METHOD_WFDS_DEFAULT = 0x1000,
}}
pub type PDOT11_WPS_CONFIG_METHOD = *mut DOT11_WPS_CONFIG_METHOD;
ENUM!{enum DOT11_WPS_DEVICE_PASSWORD_ID {
    DOT11_WPS_PASSWORD_ID_DEFAULT = 0x0000,
    DOT11_WPS_PASSWORD_ID_USER_SPECIFIED = 0x0001,
    DOT11_WPS_PASSWORD_ID_MACHINE_SPECIFIED = 0x0002,
    DOT11_WPS_PASSWORD_ID_REKEY = 0x0003,
    DOT11_WPS_PASSWORD_ID_PUSHBUTTON = 0x0004,
    DOT11_WPS_PASSWORD_ID_REGISTRAR_SPECIFIED = 0x0005,
    DOT11_WPS_PASSWORD_ID_NFC_CONNECTION_HANDOVER = 0x0007,
    DOT11_WPS_PASSWORD_ID_WFD_SERVICES = 0x0008,
    DOT11_WPS_PASSWORD_ID_OOB_RANGE_MIN = 0x0010,
    DOT11_WPS_PASSWORD_ID_OOB_RANGE_MAX = 0xFFFF,
}}
pub type PDOT11_WPS_DEVICE_PASSWORD_ID = *mut DOT11_WPS_DEVICE_PASSWORD_ID;
STRUCT!{struct WFDSVC_CONNECTION_CAPABILITY {
    bNew: BOOLEAN,
    bClient: BOOLEAN,
    bGO: BOOLEAN,
}}
pub type PWFDSVC_CONNECTION_CAPABILITY = *mut WFDSVC_CONNECTION_CAPABILITY;
pub const WFDSVC_CONNECTION_CAPABILITY_NEW: BOOLEAN = 0x01;
pub const WFDSVC_CONNECTION_CAPABILITY_CLIENT: BOOLEAN = 0x02;
pub const WFDSVC_CONNECTION_CAPABILITY_GO: BOOLEAN = 0x04;
STRUCT!{struct DOT11_WFD_SERVICE_HASH_LIST {
    ServiceHashCount: USHORT,
    ServiceHash: [DOT11_WFD_SERVICE_HASH; 1],
}}
pub type PDOT11_WFD_SERVICE_HASH_LIST = *mut DOT11_WFD_SERVICE_HASH_LIST;
STRUCT!{struct DOT11_WFD_ADVERTISEMENT_ID {
    AdvertisementID: ULONG,
    ServiceAddress: DOT11_MAC_ADDRESS,
}}
pub type PDOT11_WFD_ADVERTISEMENT_ID = *mut DOT11_WFD_ADVERTISEMENT_ID;
STRUCT!{struct DOT11_WFD_SESSION_ID {
    SessionID: ULONG,
    SessionAddress: DOT11_MAC_ADDRESS,
}}
pub type PDOT11_WFD_SESSION_ID = *mut DOT11_WFD_SESSION_ID;
STRUCT!{struct DOT11_WFD_ADVERTISED_SERVICE_DESCRIPTOR {
    AdvertisementID: ULONG,
    ConfigMethods: USHORT,
    ServiceNameLength: UCHAR,
    ServiceName: [UCHAR; DOT11_WFD_SERVICE_NAME_MAX_LENGTH],
}}
pub type PDOT11_WFD_ADVERTISED_SERVICE_DESCRIPTOR = *mut DOT11_WFD_ADVERTISED_SERVICE_DESCRIPTOR;
STRUCT!{struct DOT11_WFD_ADVERTISED_SERVICE_LIST {
    ServiceCount: USHORT,
    AdvertisedService: [DOT11_WFD_ADVERTISED_SERVICE_DESCRIPTOR; 1],
}}
pub type PDOT11_WFD_ADVERTISED_SERVICE_LIST = *mut DOT11_WFD_ADVERTISED_SERVICE_LIST;
pub const DOT11_WFD_DISCOVER_COMPLETE_PARAMETERS_REVISION_1: UCHAR = 1;
pub const DOT11_WFD_DISCOVER_COMPLETE_MAX_LIST_SIZE: ULONG = 128;
STRUCT!{struct DOT11_WFD_DISCOVER_COMPLETE_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    Status: NDIS_STATUS,
    uNumOfEntries: ULONG,
    uTotalNumOfEntries: ULONG,
    uListOffset: ULONG,
    uListLength: ULONG,
}}
pub type PDOT11_WFD_DISCOVER_COMPLETE_PARAMETERS = *mut DOT11_WFD_DISCOVER_COMPLETE_PARAMETERS;
pub const DOT11_GO_NEGOTIATION_REQUEST_SEND_COMPLETE_PARAMETERS_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_GO_NEGOTIATION_REQUEST_SEND_COMPLETE_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    PeerDeviceAddress: DOT11_MAC_ADDRESS,
    DialogToken: DOT11_DIALOG_TOKEN,
    Status: NDIS_STATUS,
    uIEsOffset: ULONG,
    uIEsLength: ULONG,
}}
pub type PDOT11_GO_NEGOTIATION_REQUEST_SEND_COMPLETE_PARAMETERS =
    *mut DOT11_GO_NEGOTIATION_REQUEST_SEND_COMPLETE_PARAMETERS;
pub const DOT11_RECEIVED_GO_NEGOTIATION_REQUEST_PARAMETERS_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_RECEIVED_GO_NEGOTIATION_REQUEST_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    PeerDeviceAddress: DOT11_MAC_ADDRESS,
    DialogToken: DOT11_DIALOG_TOKEN,
    RequestContext: PVOID,
    uIEsOffset: ULONG,
    uIEsLength: ULONG,
}}
pub type PDOT11_RECEIVED_GO_NEGOTIATION_REQUEST_PARAMETERS =
    *mut DOT11_RECEIVED_GO_NEGOTIATION_REQUEST_PARAMETERS;
pub const DOT11_GO_NEGOTIATION_RESPONSE_SEND_COMPLETE_PARAMETERS_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_GO_NEGOTIATION_RESPONSE_SEND_COMPLETE_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    PeerDeviceAddress: DOT11_MAC_ADDRESS,
    DialogToken: DOT11_DIALOG_TOKEN,
    Status: NDIS_STATUS,
    uIEsOffset: ULONG,
    uIEsLength: ULONG,
}}
pub type PDOT11_GO_NEGOTIATION_RESPONSE_SEND_COMPLETE_PARAMETERS =
    *mut DOT11_GO_NEGOTIATION_RESPONSE_SEND_COMPLETE_PARAMETERS;
pub const DOT11_RECEIVED_GO_NEGOTIATION_RESPONSE_PARAMETERS_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_RECEIVED_GO_NEGOTIATION_RESPONSE_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    PeerDeviceAddress: DOT11_MAC_ADDRESS,
    DialogToken: DOT11_DIALOG_TOKEN,
    ResponseContext: PVOID,
    uIEsOffset: ULONG,
    uIEsLength: ULONG,
}}
pub type PDOT11_RECEIVED_GO_NEGOTIATION_RESPONSE_PARAMETERS =
    *mut DOT11_RECEIVED_GO_NEGOTIATION_RESPONSE_PARAMETERS;
pub const DOT11_GO_NEGOTIATION_CONFIRMATION_SEND_COMPLETE_PARAMETERS_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_GO_NEGOTIATION_CONFIRMATION_SEND_COMPLETE_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    PeerDeviceAddress: DOT11_MAC_ADDRESS,
    DialogToken: DOT11_DIALOG_TOKEN,
    Status: NDIS_STATUS,
    uIEsOffset: ULONG,
    uIEsLength: ULONG,
}}
pub type PDOT11_GO_NEGOTIATION_CONFIRMATION_SEND_COMPLETE_PARAMETERS =
    *mut DOT11_GO_NEGOTIATION_CONFIRMATION_SEND_COMPLETE_PARAMETERS;
pub const DOT11_RECEIVED_GO_NEGOTIATION_CONFIRMATION_PARAMETERS_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_RECEIVED_GO_NEGOTIATION_CONFIRMATION_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    PeerDeviceAddress: DOT11_MAC_ADDRESS,
    DialogToken: DOT11_DIALOG_TOKEN,
    uIEsOffset: ULONG,
    uIEsLength: ULONG,
}}
pub type PDOT11_RECEIVED_GO_NEGOTIATION_CONFIRMATION_PARAMETERS =
    *mut DOT11_RECEIVED_GO_NEGOTIATION_CONFIRMATION_PARAMETERS;
pub const DOT11_INVITATION_REQUEST_SEND_COMPLETE_PARAMETERS_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_INVITATION_REQUEST_SEND_COMPLETE_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    PeerDeviceAddress: DOT11_MAC_ADDRESS,
    ReceiverAddress: DOT11_MAC_ADDRESS,
    DialogToken: DOT11_DIALOG_TOKEN,
    Status: NDIS_STATUS,
    uIEsOffset: ULONG,
    uIEsLength: ULONG,
}}
pub type PDOT11_INVITATION_REQUEST_SEND_COMPLETE_PARAMETERS =
    *mut DOT11_INVITATION_REQUEST_SEND_COMPLETE_PARAMETERS;
pub const DOT11_RECEIVED_INVITATION_REQUEST_PARAMETERS_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_RECEIVED_INVITATION_REQUEST_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    TransmitterDeviceAddress: DOT11_MAC_ADDRESS,
    BSSID: DOT11_MAC_ADDRESS,
    DialogToken: DOT11_DIALOG_TOKEN,
    RequestContext: PVOID,
    uIEsOffset: ULONG,
    uIEsLength: ULONG,
}}
pub type PDOT11_RECEIVED_INVITATION_REQUEST_PARAMETERS =
    *mut DOT11_RECEIVED_INVITATION_REQUEST_PARAMETERS;
pub const DOT11_INVITATION_RESPONSE_SEND_COMPLETE_PARAMETERS_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_INVITATION_RESPONSE_SEND_COMPLETE_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    ReceiverDeviceAddress: DOT11_MAC_ADDRESS,
    DialogToken: DOT11_DIALOG_TOKEN,
    Status: NDIS_STATUS,
    uIEsOffset: ULONG,
    uIEsLength: ULONG,
}}
pub type PDOT11_INVITATION_RESPONSE_SEND_COMPLETE_PARAMETERS =
    *mut DOT11_INVITATION_RESPONSE_SEND_COMPLETE_PARAMETERS;
pub const DOT11_RECEIVED_INVITATION_RESPONSE_PARAMETERS_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_RECEIVED_INVITATION_RESPONSE_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    TransmitterDeviceAddress: DOT11_MAC_ADDRESS,
    BSSID: DOT11_MAC_ADDRESS,
    DialogToken: DOT11_DIALOG_TOKEN,
    uIEsOffset: ULONG,
    uIEsLength: ULONG,
}}
pub type PDOT11_RECEIVED_INVITATION_RESPONSE_PARAMETERS =
    *mut DOT11_RECEIVED_INVITATION_RESPONSE_PARAMETERS;
pub const DOT11_PROVISION_DISCOVERY_REQUEST_SEND_COMPLETE_PARAMETERS_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_PROVISION_DISCOVERY_REQUEST_SEND_COMPLETE_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    PeerDeviceAddress: DOT11_MAC_ADDRESS,
    ReceiverAddress: DOT11_MAC_ADDRESS,
    DialogToken: DOT11_DIALOG_TOKEN,
    Status: NDIS_STATUS,
    uIEsOffset: ULONG,
    uIEsLength: ULONG,
}}
pub type PDOT11_PROVISION_DISCOVERY_REQUEST_SEND_COMPLETE_PARAMETERS =
    *mut DOT11_PROVISION_DISCOVERY_REQUEST_SEND_COMPLETE_PARAMETERS;
pub const DOT11_RECEIVED_PROVISION_DISCOVERY_REQUEST_PARAMETERS_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_RECEIVED_PROVISION_DISCOVERY_REQUEST_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    TransmitterDeviceAddress: DOT11_MAC_ADDRESS,
    BSSID: DOT11_MAC_ADDRESS,
    DialogToken: DOT11_DIALOG_TOKEN,
    RequestContext: PVOID,
    uIEsOffset: ULONG,
    uIEsLength: ULONG,
}}
pub type PDOT11_RECEIVED_PROVISION_DISCOVERY_REQUEST_PARAMETERS =
    *mut DOT11_RECEIVED_PROVISION_DISCOVERY_REQUEST_PARAMETERS;
pub const DOT11_PROVISION_DISCOVERY_RESPONSE_SEND_COMPLETE_PARAMETERS_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_PROVISION_DISCOVERY_RESPONSE_SEND_COMPLETE_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    ReceiverDeviceAddress: DOT11_MAC_ADDRESS,
    DialogToken: DOT11_DIALOG_TOKEN,
    Status: NDIS_STATUS,
    uIEsOffset: ULONG,
    uIEsLength: ULONG,
}}
pub type PDOT11_PROVISION_DISCOVERY_RESPONSE_SEND_COMPLETE_PARAMETERS =
    *mut DOT11_PROVISION_DISCOVERY_RESPONSE_SEND_COMPLETE_PARAMETERS;
pub const DOT11_RECEIVED_PROVISION_DISCOVERY_RESPONSE_PARAMETERS_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_RECEIVED_PROVISION_DISCOVERY_RESPONSE_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    TransmitterDeviceAddress: DOT11_MAC_ADDRESS,
    BSSID: DOT11_MAC_ADDRESS,
    DialogToken: DOT11_DIALOG_TOKEN,
    uIEsOffset: ULONG,
    uIEsLength: ULONG,
}}
pub type PDOT11_RECEIVED_PROVISION_DISCOVERY_RESPONSE_PARAMETERS =
    *mut DOT11_RECEIVED_PROVISION_DISCOVERY_RESPONSE_PARAMETERS;
pub const DOT11_ANQP_QUERY_COMPLETE_PARAMETERS_REVISION_1: UCHAR = 1;
ENUM!{enum DOT11_ANQP_QUERY_RESULT {
    dot11_ANQP_query_result_success = 0,
    dot11_ANQP_query_result_failure = 1,
    dot11_ANQP_query_result_timed_out = 2,
    dot11_ANQP_query_result_resources = 3,
    dot11_ANQP_query_result_advertisement_protocol_not_supported_on_remote = 4,
    dot11_ANQP_query_result_gas_protocol_failure = 5,
    dot11_ANQP_query_result_advertisement_server_not_responding = 6,
    dot11_ANQP_query_result_access_issues = 7,
}}
pub type PDOT11_ANQP_QUERY_RESULT = *mut DOT11_ANQP_QUERY_RESULT;
STRUCT!{struct DOT11_ANQP_QUERY_COMPLETE_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    Status: DOT11_ANQP_QUERY_RESULT,
    hContext: HANDLE,
    uResponseLength: ULONG,
}}
pub type PDOT11_ANQP_QUERY_COMPLETE_PARAMETERS = *mut DOT11_ANQP_QUERY_COMPLETE_PARAMETERS;
pub const NWF_WFD_DEVICE_OID: u32 = 0x05;
pub const NWF_WFD_ROLE_OID: u32 = 0x06;
pub const OID_DOT11_WFD_DEVICE_CAPABILITY: u32 =
    NWF_DEFINE_OID!(0x01, NWF_WFD_DEVICE_OID, NWF_MANDATORY_OID);
pub const DOT11_WFD_DEVICE_CAPABILITY_CONFIG_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_WFD_DEVICE_CAPABILITY_CONFIG {
    Header: NDIS_OBJECT_HEADER,
    bServiceDiscoveryEnabled: BOOLEAN,
    bClientDiscoverabilityEnabled: BOOLEAN,
    bConcurrentOperationSupported: BOOLEAN,
    bInfrastructureManagementEnabled: BOOLEAN,
    bDeviceLimitReached: BOOLEAN,
    bInvitationProcedureEnabled: BOOLEAN,
    WPSVersionsEnabled: ULONG,
}}
pub type PDOT11_WFD_DEVICE_CAPABILITY_CONFIG = *mut DOT11_WFD_DEVICE_CAPABILITY_CONFIG;
pub const OID_DOT11_WFD_GROUP_OWNER_CAPABILITY: u32 =
    NWF_DEFINE_OID!(0x02, NWF_WFD_DEVICE_OID, NWF_MANDATORY_OID);
pub const DOT11_WFD_GROUP_OWNER_CAPABILITY_CONFIG_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_WFD_GROUP_OWNER_CAPABILITY_CONFIG {
    Header: NDIS_OBJECT_HEADER,
    bPersistentGroupEnabled: BOOLEAN,
    bIntraBSSDistributionSupported: BOOLEAN,
    bCrossConnectionSupported: BOOLEAN,
    bPersistentReconnectSupported: BOOLEAN,
    bGroupFormationEnabled: BOOLEAN,
    uMaximumGroupLimit: ULONG,
}}
pub type PDOT11_WFD_GROUP_OWNER_CAPABILITY_CONFIG = *mut DOT11_WFD_GROUP_OWNER_CAPABILITY_CONFIG;
pub const DOT11_WFD_GROUP_OWNER_CAPABILITY_CONFIG_REVISION_2: UCHAR = 2;
STRUCT!{struct DOT11_WFD_GROUP_OWNER_CAPABILITY_CONFIG_V2 {
    Header: NDIS_OBJECT_HEADER,
    bPersistentGroupEnabled: BOOLEAN,
    bIntraBSSDistributionSupported: BOOLEAN,
    bCrossConnectionSupported: BOOLEAN,
    bPersistentReconnectSupported: BOOLEAN,
    bGroupFormationEnabled: BOOLEAN,
    uMaximumGroupLimit: ULONG,
    bEapolKeyIpAddressAllocationSupported: BOOLEAN,
}}
pub type PDOT11_WFD_GROUP_OWNER_CAPABILITY_CONFIG_V2 =
    *mut DOT11_WFD_GROUP_OWNER_CAPABILITY_CONFIG_V2;
pub const OID_DOT11_WFD_DEVICE_INFO: u32 =
    NWF_DEFINE_OID!(0x03, NWF_WFD_DEVICE_OID, NWF_MANDATORY_OID);
pub const DOT11_WFD_DEVICE_INFO_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_WFD_DEVICE_INFO {
    Header: NDIS_OBJECT_HEADER,
    DeviceAddress: DOT11_MAC_ADDRESS,
    ConfigMethods: USHORT,
    PrimaryDeviceType: DOT11_WFD_DEVICE_TYPE,
    DeviceName: DOT11_WPS_DEVICE_NAME,
}}
pub type PDOT11_WFD_DEVICE_INFO = *mut DOT11_WFD_DEVICE_INFO;
pub const OID_DOT11_WFD_SECONDARY_DEVICE_TYPE_LIST: u32 =
    NWF_DEFINE_OID!(0x04, NWF_WFD_DEVICE_OID, NWF_MANDATORY_OID);
pub const DOT11_WFD_SECONDARY_DEVICE_TYPE_LIST_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_WFD_SECONDARY_DEVICE_TYPE_LIST {
    Header: NDIS_OBJECT_HEADER,
    uNumOfEntries: ULONG,
    uTotalNumOfEntries: ULONG,
    SecondaryDeviceTypes: [DOT11_WFD_DEVICE_TYPE; 1],
}}
pub type PDOT11_WFD_SECONDARY_DEVICE_TYPE_LIST = *mut DOT11_WFD_SECONDARY_DEVICE_TYPE_LIST;
// pub const DOT11_SIZEOF_WFD_SECONDARY_DEVICE_TYPE_LIST_REVISION_1: usize =
//    FIELD_OFFSET(DOT11_WFD_SECONDARY_DEVICE_TYPE_LIST, SecondaryDeviceTypes);
pub const OID_DOT11_WFD_DISCOVER_REQUEST: u32 =
    NWF_DEFINE_OID!(0x05, NWF_WFD_DEVICE_OID, NWF_MANDATORY_OID);
ENUM!{enum DOT11_WFD_DISCOVER_TYPE {
    dot11_wfd_discover_type_scan_only = 1,
    dot11_wfd_discover_type_find_only = 2,
    dot11_wfd_discover_type_auto = 3,
    dot11_wfd_discover_type_scan_social_channels = 4,
    dot11_wfd_discover_type_forced = 0x80000000,
}}
pub type PDOT11_WFD_DISCOVER_TYPE = *mut DOT11_WFD_DISCOVER_TYPE;
ENUM!{enum DOT11_WFD_SCAN_TYPE {
    dot11_wfd_scan_type_active = 1,
    dot11_wfd_scan_type_passive = 2,
    dot11_wfd_scan_type_auto = 3,
}}
pub type PDOT11_WFD_SCAN_TYPE = *mut DOT11_WFD_SCAN_TYPE;
pub const DISCOVERY_FILTER_BITMASK_DEVICE: UCHAR = 0x1;
pub const DISCOVERY_FILTER_BITMASK_GO: UCHAR = 0x2;
pub const DISCOVERY_FILTER_BITMASK_ANY: UCHAR = 0xF;
STRUCT!{struct DOT11_WFD_DISCOVER_DEVICE_FILTER {
    DeviceID: DOT11_MAC_ADDRESS,
    ucBitmask: UCHAR,
    GroupSSID: DOT11_SSID,
}}
pub type PDOT11_WFD_DISCOVER_DEVICE_FILTER = *mut DOT11_WFD_DISCOVER_DEVICE_FILTER;
pub const DOT11_WFD_DISCOVER_REQUEST_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_WFD_DISCOVER_REQUEST {
    Header: NDIS_OBJECT_HEADER,
    DiscoverType: DOT11_WFD_DISCOVER_TYPE,
    ScanType: DOT11_WFD_SCAN_TYPE,
    uDiscoverTimeout: ULONG,
    uDeviceFilterListOffset: ULONG,
    uNumDeviceFilters: ULONG,
    uIEsOffset: ULONG,
    uIEsLength: ULONG,
    bForceScanLegacyNetworks: BOOLEAN,
}}
pub type PDOT11_WFD_DISCOVER_REQUEST = *mut DOT11_WFD_DISCOVER_REQUEST;
pub const OID_DOT11_WFD_ENUM_DEVICE_LIST: u32 =
    NWF_DEFINE_OID!(0x06, NWF_WFD_DEVICE_OID, NWF_MANDATORY_OID);
pub const DOT11_DEVICE_ENTRY_BYTE_ARRAY_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_WFD_DEVICE_ENTRY {
    uPhyId: ULONG,
    PhySpecificInfo: DOT11_BSS_ENTRY_PHY_SPECIFIC_INFO,
    dot11BSSID: DOT11_MAC_ADDRESS,
    dot11BSSType: DOT11_BSS_TYPE,
    TransmitterAddress: DOT11_MAC_ADDRESS,
    lRSSI: LONG,
    uLinkQuality: ULONG,
    usBeaconPeriod: USHORT,
    ullTimestamp: ULONGLONG,
    ullBeaconHostTimestamp: ULONGLONG,
    ullProbeResponseHostTimestamp: ULONGLONG,
    usCapabilityInformation: USHORT,
    uBeaconIEsOffset: ULONG,
    uBeaconIEsLength: ULONG,
    uProbeResponseIEsOffset: ULONG,
    uProbeResponseIEsLength: ULONG,
}}
pub type PDOT11_WFD_DEVICE_ENTRY = *mut DOT11_WFD_DEVICE_ENTRY;
pub const OID_DOT11_WFD_LISTEN_STATE_DISCOVERABILITY: u32 =
    NWF_DEFINE_OID!(0x07, NWF_WFD_DEVICE_OID, NWF_MANDATORY_OID);
pub const DOT11_WFD_DEVICE_NOT_DISCOVERABLE: i32 = 0;
pub const DOT11_WFD_DEVICE_AUTO_AVAILABILITY: i32 = 16;
pub const DOT11_WFD_DEVICE_HIGH_AVAILABILITY: i32 = 24;
pub const OID_DOT11_WFD_ADDITIONAL_IE: u32 =
    NWF_DEFINE_OID!(0x08, NWF_WFD_DEVICE_OID, NWF_MANDATORY_OID);
pub const DOT11_WFD_ADDITIONAL_IE_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_WFD_ADDITIONAL_IE {
    Header: NDIS_OBJECT_HEADER,
    uBeaconIEsOffset: ULONG,
    uBeaconIEsLength: ULONG,
    uProbeResponseIEsOffset: ULONG,
    uProbeResponseIEsLength: ULONG,
    uDefaultRequestIEsOffset: ULONG,
    uDefaultRequestIEsLength: ULONG,
}}
pub type PDOT11_WFD_ADDITIONAL_IE = *mut DOT11_WFD_ADDITIONAL_IE;
pub const OID_DOT11_WFD_FLUSH_DEVICE_LIST: u32 =
    NWF_DEFINE_OID!(0x09, NWF_WFD_DEVICE_OID, NWF_MANDATORY_OID);
pub const OID_DOT11_WFD_SEND_GO_NEGOTIATION_REQUEST: u32 =
    NWF_DEFINE_OID!(0x0A, NWF_WFD_DEVICE_OID, NWF_MANDATORY_OID);
pub const DOT11_SEND_GO_NEGOTIATION_REQUEST_PARAMETERS_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_SEND_GO_NEGOTIATION_REQUEST_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    PeerDeviceAddress: DOT11_MAC_ADDRESS,
    DialogToken: DOT11_DIALOG_TOKEN,
    uSendTimeout: ULONG,
    GroupOwnerIntent: DOT11_WFD_GO_INTENT,
    MinimumConfigTimeout: DOT11_WFD_CONFIGURATION_TIMEOUT,
    IntendedInterfaceAddress: DOT11_MAC_ADDRESS,
    GroupCapability: DOT11_WFD_GROUP_CAPABILITY,
    uIEsOffset: ULONG,
    uIEsLength: ULONG,
}}
pub type PDOT11_SEND_GO_NEGOTIATION_REQUEST_PARAMETERS =
    *mut DOT11_SEND_GO_NEGOTIATION_REQUEST_PARAMETERS;
pub const OID_DOT11_WFD_SEND_GO_NEGOTIATION_RESPONSE: u32 =
    NWF_DEFINE_OID!(0x0B, NWF_WFD_DEVICE_OID, NWF_MANDATORY_OID);
pub const DOT11_SEND_GO_NEGOTIATION_RESPONSE_PARAMETERS_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_SEND_GO_NEGOTIATION_RESPONSE_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    PeerDeviceAddress: DOT11_MAC_ADDRESS,
    DialogToken: DOT11_DIALOG_TOKEN,
    RequestContext: PVOID,
    uSendTimeout: ULONG,
    Status: DOT11_WFD_STATUS_CODE,
    GroupOwnerIntent: DOT11_WFD_GO_INTENT,
    MinimumConfigTimeout: DOT11_WFD_CONFIGURATION_TIMEOUT,
    IntendedInterfaceAddress: DOT11_MAC_ADDRESS,
    GroupCapability: DOT11_WFD_GROUP_CAPABILITY,
    GroupID: DOT11_WFD_GROUP_ID,
    bUseGroupID: BOOLEAN,
    uIEsOffset: ULONG,
    uIEsLength: ULONG,
}}
pub type PDOT11_SEND_GO_NEGOTIATION_RESPONSE_PARAMETERS =
    *mut DOT11_SEND_GO_NEGOTIATION_RESPONSE_PARAMETERS;
pub const OID_DOT11_WFD_SEND_GO_NEGOTIATION_CONFIRMATION: u32 =
    NWF_DEFINE_OID!(0x0C, NWF_WFD_DEVICE_OID, NWF_MANDATORY_OID);
pub const DOT11_SEND_GO_NEGOTIATION_CONFIRMATION_PARAMETERS_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_SEND_GO_NEGOTIATION_CONFIRMATION_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    PeerDeviceAddress: DOT11_MAC_ADDRESS,
    DialogToken: DOT11_DIALOG_TOKEN,
    ResponseContext: PVOID,
    uSendTimeout: ULONG,
    Status: DOT11_WFD_STATUS_CODE,
    GroupCapability: DOT11_WFD_GROUP_CAPABILITY,
    GroupID: DOT11_WFD_GROUP_ID,
    bUseGroupID: BOOLEAN,
    uIEsOffset: ULONG,
    uIEsLength: ULONG,
}}
pub type PDOT11_SEND_GO_NEGOTIATION_CONFIRMATION_PARAMETERS =
    *mut DOT11_SEND_GO_NEGOTIATION_CONFIRMATION_PARAMETERS;
pub const OID_DOT11_WFD_SEND_INVITATION_REQUEST: u32 =
    NWF_DEFINE_OID!(0x0D, NWF_WFD_DEVICE_OID, NWF_MANDATORY_OID);
STRUCT!{#[repr(packed)] struct DOT11_WFD_INVITATION_FLAGS {
    Bitfields: UCHAR,
}}
BITFIELD!{DOT11_WFD_INVITATION_FLAGS Bitfields: UCHAR [
    InvitationType set_InvitationType[0..1],
    Reserved set_Reserved[1..8],
]}
pub type PDOT11_WFD_INVITATION_FLAGS = *mut DOT11_WFD_INVITATION_FLAGS;
pub const DOT11_SEND_INVITATION_REQUEST_PARAMETERS_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_SEND_INVITATION_REQUEST_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    DialogToken: DOT11_DIALOG_TOKEN,
    PeerDeviceAddress: DOT11_MAC_ADDRESS,
    uSendTimeout: ULONG,
    MinimumConfigTimeout: DOT11_WFD_CONFIGURATION_TIMEOUT,
    InvitationFlags: DOT11_WFD_INVITATION_FLAGS,
    GroupBSSID: DOT11_MAC_ADDRESS,
    bUseGroupBSSID: BOOLEAN,
    OperatingChannel: DOT11_WFD_CHANNEL,
    bUseSpecifiedOperatingChannel: BOOLEAN,
    GroupID: DOT11_WFD_GROUP_ID,
    bLocalGO: BOOLEAN,
    uIEsOffset: ULONG,
    uIEsLength: ULONG,
}}
pub type PDOT11_SEND_INVITATION_REQUEST_PARAMETERS =
    *mut DOT11_SEND_INVITATION_REQUEST_PARAMETERS;
pub const OID_DOT11_WFD_SEND_INVITATION_RESPONSE: u32 =
    NWF_DEFINE_OID!(0x0E, NWF_WFD_DEVICE_OID, NWF_MANDATORY_OID);
pub const DOT11_SEND_INVITATION_RESPONSE_PARAMETERS_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_SEND_INVITATION_RESPONSE_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    ReceiverDeviceAddress: DOT11_MAC_ADDRESS,
    DialogToken: DOT11_DIALOG_TOKEN,
    RequestContext: PVOID,
    uSendTimeout: ULONG,
    Status: DOT11_WFD_STATUS_CODE,
    MinimumConfigTimeout: DOT11_WFD_CONFIGURATION_TIMEOUT,
    GroupBSSID: DOT11_MAC_ADDRESS,
    bUseGroupBSSID: BOOLEAN,
    OperatingChannel: DOT11_WFD_CHANNEL,
    bUseSpecifiedOperatingChannel: BOOLEAN,
    uIEsOffset: ULONG,
    uIEsLength: ULONG,
}}
pub type PDOT11_SEND_INVITATION_RESPONSE_PARAMETERS =
    *mut DOT11_SEND_INVITATION_RESPONSE_PARAMETERS;
pub const OID_DOT11_WFD_SEND_PROVISION_DISCOVERY_REQUEST: u32 =
    NWF_DEFINE_OID!(0x0F, NWF_WFD_DEVICE_OID, NWF_MANDATORY_OID);
pub const DOT11_SEND_PROVISION_DISCOVERY_REQUEST_PARAMETERS_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_SEND_PROVISION_DISCOVERY_REQUEST_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    DialogToken: DOT11_DIALOG_TOKEN,
    PeerDeviceAddress: DOT11_MAC_ADDRESS,
    uSendTimeout: ULONG,
    GroupCapability: DOT11_WFD_GROUP_CAPABILITY,
    GroupID: DOT11_WFD_GROUP_ID,
    bUseGroupID: BOOLEAN,
    uIEsOffset: ULONG,
    uIEsLength: ULONG,
}}
pub type PDOT11_SEND_PROVISION_DISCOVERY_REQUEST_PARAMETERS =
    *mut DOT11_SEND_PROVISION_DISCOVERY_REQUEST_PARAMETERS;
pub const OID_DOT11_WFD_SEND_PROVISION_DISCOVERY_RESPONSE: u32 =
    NWF_DEFINE_OID!(0x10, NWF_WFD_DEVICE_OID, NWF_MANDATORY_OID);
pub const DOT11_SEND_PROVISION_DISCOVERY_RESPONSE_PARAMETERS_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_SEND_PROVISION_DISCOVERY_RESPONSE_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    ReceiverDeviceAddress: DOT11_MAC_ADDRESS,
    DialogToken: DOT11_DIALOG_TOKEN,
    RequestContext: PVOID,
    uSendTimeout: ULONG,
    uIEsOffset: ULONG,
    uIEsLength: ULONG,
}}
pub type PDOT11_SEND_PROVISION_DISCOVERY_RESPONSE_PARAMETERS =
    *mut DOT11_SEND_PROVISION_DISCOVERY_RESPONSE_PARAMETERS;
pub const OID_DOT11_WFD_GET_DIALOG_TOKEN: u32 =
    NWF_DEFINE_OID!(0x11, NWF_WFD_DEVICE_OID, NWF_MANDATORY_OID);
pub const OID_DOT11_WFD_STOP_DISCOVERY: u32 =
    NWF_DEFINE_OID!(0x12, NWF_WFD_DEVICE_OID, NWF_MANDATORY_OID);
pub const OID_DOT11_WFD_ENABLE_HRDSSS_DEVICES: u32 =
    NWF_DEFINE_OID!(0x13, NWF_WFD_DEVICE_OID, NWF_OPTIONAL_OID);
pub const OID_DOT11_WFD_DEVICE_LISTEN_CHANNEL: u32 =
    NWF_DEFINE_OID!(0x14, NWF_WFD_DEVICE_OID, NWF_MANDATORY_OID);
pub const DOT11_WFD_DEVICE_LISTEN_CHANNEL_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_WFD_DEVICE_LISTEN_CHANNEL {
    Header: NDIS_OBJECT_HEADER,
    ChannelNumber: UCHAR,
}}
pub type PDOT11_WFD_DEVICE_LISTEN_CHANNEL = *mut DOT11_WFD_DEVICE_LISTEN_CHANNEL;
pub const OID_DOT11_WFD_DESIRED_GROUP_ID: u32 =
    NWF_DEFINE_OID!(0x01, NWF_WFD_ROLE_OID, NWF_MANDATORY_OID);
pub const OID_DOT11_WFD_START_GO_REQUEST: u32 =
    NWF_DEFINE_OID!(0x02, NWF_WFD_ROLE_OID, NWF_MANDATORY_OID);
pub const OID_DOT11_WFD_GROUP_START_PARAMETERS: u32 =
    NWF_DEFINE_OID!(0x03, NWF_WFD_ROLE_OID, NWF_MANDATORY_OID);
pub const DOT11_WFD_GROUP_START_PARAMETERS_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_WFD_GROUP_START_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    AdvertisedOperatingChannel: DOT11_WFD_CHANNEL,
}}
pub type PDOT11_WFD_GROUP_START_PARAMETERS = *mut DOT11_WFD_GROUP_START_PARAMETERS;
pub const OID_DOT11_WFD_CONNECT_TO_GROUP_REQUEST: u32 =
    NWF_DEFINE_OID!(0x04, NWF_WFD_ROLE_OID, NWF_MANDATORY_OID);
pub const OID_DOT11_WFD_DISCONNECT_FROM_GROUP_REQUEST: u32 =
    NWF_DEFINE_OID!(0x05, NWF_WFD_ROLE_OID, NWF_MANDATORY_OID);
pub const OID_DOT11_WFD_GROUP_JOIN_PARAMETERS: u32 =
    NWF_DEFINE_OID!(0x06, NWF_WFD_ROLE_OID, NWF_MANDATORY_OID);
pub const DOT11_WFD_GROUP_JOIN_PARAMETERS_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_WFD_GROUP_JOIN_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    GOOperatingChannel: DOT11_WFD_CHANNEL,
    GOConfigTime: ULONG,
    bInGroupFormation: BOOLEAN,
    bWaitForWPSReady: BOOLEAN,
}}
pub type PDOT11_WFD_GROUP_JOIN_PARAMETERS = *mut DOT11_WFD_GROUP_JOIN_PARAMETERS;
pub const NWF_POWER_SAVE_OID: u32 = 0x07;
pub const OID_DOT11_POWER_MGMT_MODE_AUTO_ENABLED: u32 =
    NWF_DEFINE_OID!(0x01, NWF_POWER_SAVE_OID, NWF_MANDATORY_OID);
pub const DOT11_POWER_MGMT_AUTO_MODE_ENABLED_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_POWER_MGMT_AUTO_MODE_ENABLED_INFO {
    Header: NDIS_OBJECT_HEADER,
    bEnabled: BOOLEAN,
}}
pub type PDOT11_POWER_MGMT_AUTO_MODE_ENABLED_INFO = *mut DOT11_POWER_MGMT_AUTO_MODE_ENABLED_INFO;
pub const OID_DOT11_POWER_MGMT_MODE_STATUS: u32 =
    NWF_DEFINE_OID!(0x02, NWF_POWER_SAVE_OID, NWF_MANDATORY_OID);
ENUM!{enum DOT11_POWER_MODE_REASON {
    dot11_power_mode_reason_no_change = 0,
    dot11_power_mode_reason_noncompliant_AP = 1,
    dot11_power_mode_reason_legacy_WFD_device = 2,
    dot11_power_mode_reason_compliant_AP = 3,
    dot11_power_mode_reason_compliant_WFD_device = 4,
    dot11_power_mode_reason_others = 5,
}}
pub const DOT11_POWER_MGMT_MODE_STATUS_INFO_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_POWER_MGMT_MODE_STATUS_INFO {
    Header: NDIS_OBJECT_HEADER,
    PowerSaveMode: DOT11_POWER_MODE,
    uPowerSaveLevel: ULONG,
    Reason: DOT11_POWER_MODE_REASON,
}}
pub type PDOT11_POWER_MGMT_MODE_STATUSINFO = *mut DOT11_POWER_MGMT_MODE_STATUS_INFO;
pub const OID_DOT11_OFFLOAD_NETWORK_LIST: u32 =
    NWF_DEFINE_OID!(0x03, NWF_POWER_SAVE_OID, NWF_MANDATORY_OID);
pub const DOT11_MAX_CHANNEL_HINTS: usize = 4;
pub const DOT11_INVALID_CHANNEL_NUMBER: ULONG = 0;
STRUCT!{struct DOT11_CHANNEL_HINT {
    Dot11PhyType: DOT11_PHY_TYPE,
    uChannelNumber: ULONG,
}}
pub type PDOT11_CHANNEL_HINT = *mut DOT11_CHANNEL_HINT;
STRUCT!{struct DOT11_OFFLOAD_NETWORK {
    Ssid: DOT11_SSID,
    UnicastCipher: DOT11_CIPHER_ALGORITHM,
    AuthAlgo: DOT11_AUTH_ALGORITHM,
    Dot11ChannelHints: [DOT11_CHANNEL_HINT; DOT11_MAX_CHANNEL_HINTS],
}}
pub type PDOT11_OFFLOAD_NETWORK = *mut DOT11_OFFLOAD_NETWORK;
pub const DOT11_NLO_FLAG_STOP_NLO_INDICATION: ULONG = 0x00000001;
pub const DOT11_NLO_FLAG_SCAN_ON_AOAC_PLATFORM: ULONG = 0x00000002;
pub const DOT11_NLO_FLAG_SCAN_AT_SYSTEM_RESUME: ULONG = 0x00000004;
pub const DOT11_OFFLOAD_NETWORK_LIST_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_OFFLOAD_NETWORK_LIST_INFO {
    Header: NDIS_OBJECT_HEADER,
    ulFlags: ULONG,
    FastScanPeriod: ULONG,
    FastScanIterations: ULONG,
    SlowScanPeriod: ULONG,
    uNumOfEntries: ULONG,
    offloadNetworkList: [DOT11_OFFLOAD_NETWORK; 1],
}}
pub type PDOT11_OFFLOAD_NETWORK_LIST_INFO = *mut DOT11_OFFLOAD_NETWORK_LIST_INFO;
pub const DOT11_OFFLOAD_NETWORK_STATUS_PARAMETERS_REVISION_1: UCHAR = 1;
STRUCT!{struct DOT11_OFFLOAD_NETWORK_STATUS_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    Status: NDIS_STATUS,
}}
pub type PDOT11_OFFLOAD_NETWORK_STATUS_PARAMETERS = *mut DOT11_OFFLOAD_NETWORK_STATUS_PARAMETERS;
pub const NWF_MANUFACTURING_OID: u32 = 0x08;
pub const OID_DOT11_MANUFACTURING_TEST: u32 =
    NWF_DEFINE_OID!(0x01, NWF_MANUFACTURING_OID, NWF_OPTIONAL_OID);
pub const DOT11_MANUFACTURING_TEST_REVISION_1: UCHAR = 1;
ENUM!{enum DOT11_MANUFACTURING_TEST_TYPE {
    dot11_manufacturing_test_unknown = 0,
    dot11_manufacturing_test_self_start = 1,
    dot11_manufacturing_test_self_query_result = 2,
    dot11_manufacturing_test_rx = 3,
    dot11_manufacturing_test_tx = 4,
    dot11_manufacturing_test_query_adc = 5,
    dot11_manufacturing_test_set_data = 6,
    dot11_manufacturing_test_query_data = 7,
    dot11_manufacturing_test_sleep = 8,
    dot11_manufacturing_test_awake = 9,
    dot11_manufacturing_test_IHV_start = 0x80000000,
    dot11_manufacturing_test_IHV_end = 0xffffffff,
}}
pub type PDOT11_MANUFACTURING_TEST_TYPE = *mut DOT11_MANUFACTURING_TEST_TYPE;
STRUCT!{struct DOT11_MANUFACTURING_TEST {
    dot11ManufacturingTestType: DOT11_MANUFACTURING_TEST_TYPE,
    uBufferLength: ULONG,
    ucBuffer: [UCHAR; 1],
}}
pub type PDOT11_MANUFACTURING_TEST = *mut DOT11_MANUFACTURING_TEST;
ENUM!{enum DOT11_MANUFACTURING_SELF_TEST_TYPE {
    DOT11_MANUFACTURING_SELF_TEST_TYPE_INTERFACE = 1,
    DOT11_MANUFACTURING_SELF_TEST_TYPE_RF_INTERFACE = 2,
    DOT11_MANUFACTURING_SELF_TEST_TYPE_BT_COEXISTENCE = 3,
}}
pub type PDOT11_MANUFACTURING_SELF_TEST_TYPE = *mut DOT11_MANUFACTURING_SELF_TEST_TYPE;
STRUCT!{struct DOT11_MANUFACTURING_SELF_TEST_SET_PARAMS {
    SelfTestType: DOT11_MANUFACTURING_SELF_TEST_TYPE,
    uTestID: ULONG,
    uPinBitMask: ULONG,
    pvContext: PVOID,
    uBufferLength: ULONG,
    ucBufferIn: [UCHAR; 1],
}}
pub type PDOT11_MANUFACTURING_SELF_TEST_SET_PARAMS =
    *mut DOT11_MANUFACTURING_SELF_TEST_SET_PARAMS;
STRUCT!{struct DOT11_MANUFACTURING_SELF_TEST_QUERY_RESULTS {
    SelfTestType: DOT11_MANUFACTURING_SELF_TEST_TYPE,
    uTestID: ULONG,
    bResult: BOOLEAN,
    uPinFailedBitMask: ULONG,
    pvContext: PVOID,
    uBytesWrittenOut: ULONG,
    ucBufferOut: [UCHAR; 1],
}}
pub type PDOT11_MANUFACTURING_SELF_TEST_QUERY_RESULTS =
    *mut DOT11_MANUFACTURING_SELF_TEST_QUERY_RESULTS;
ENUM!{enum DOT11_BAND {
    dot11_band_2p4g = 1,
    dot11_band_4p9g = 2,
    dot11_band_5g = 3,
}}
pub type PDOT11_BAND = *mut DOT11_BAND;
STRUCT!{struct DOT11_MANUFACTURING_FUNCTIONAL_TEST_RX {
    bEnabled: BOOLEAN,
    Dot11Band: DOT11_BAND,
    uChannel: ULONG,
    PowerLevel: LONG,
}}
pub type PDOT11_MANUFACTURING_FUNCTIONAL_TEST_RX = *mut DOT11_MANUFACTURING_FUNCTIONAL_TEST_RX;
STRUCT!{struct DOT11_MANUFACTURING_FUNCTIONAL_TEST_TX {
    bEnable: BOOLEAN,
    bOpenLoop: BOOLEAN,
    Dot11Band: DOT11_BAND,
    uChannel: ULONG,
    uSetPowerLevel: ULONG,
    ADCPowerLevel: LONG,
}}
pub type PDOT11_MANUFACTURING_FUNCTIONAL_TEST_TX = *mut DOT11_MANUFACTURING_FUNCTIONAL_TEST_TX;
STRUCT!{struct DOT11_MANUFACTURING_FUNCTIONAL_TEST_QUERY_ADC {
    Dot11Band: DOT11_BAND,
    uChannel: ULONG,
    ADCPowerLevel: LONG,
}}
pub type PDOT11_MANUFACTURING_FUNCTIONAL_TEST_QUERY_ADC =
    *mut DOT11_MANUFACTURING_FUNCTIONAL_TEST_QUERY_ADC;
STRUCT!{struct DOT11_MANUFACTURING_TEST_SET_DATA {
    uKey: ULONG,
    uOffset: ULONG,
    uBufferLength: ULONG,
    ucBufferIn: [UCHAR; 1],
}}
pub type PDOT11_MANUFACTURING_TEST_SET_DATA = *mut DOT11_MANUFACTURING_TEST_SET_DATA;
STRUCT!{struct DOT11_MANUFACTURING_TEST_QUERY_DATA {
    uKey: ULONG,
    uOffset: ULONG,
    uBufferLength: ULONG,
    uBytesRead: ULONG,
    ucBufferOut: [UCHAR; 1],
}}
pub type PDOT11_MANUFACTURING_TEST_QUERY_DATA = *mut DOT11_MANUFACTURING_TEST_QUERY_DATA;
STRUCT!{struct DOT11_MANUFACTURING_TEST_SLEEP {
    uSleepTime: ULONG,
    pvContext: PVOID,
}}
pub type PDOT11_MANUFACTURING_TEST_SLEEP = *mut DOT11_MANUFACTURING_TEST_SLEEP;
ENUM!{enum DOT11_MANUFACTURING_CALLBACK_TYPE {
    dot11_manufacturing_callback_unknown = 0,
    dot11_manufacturing_callback_self_test_complete = 1,
    dot11_manufacturing_callback_sleep_complete = 2,
    dot11_manufacturing_callback_IHV_start = 0x80000000,
    dot11_manufacturing_callback_IHV_end = 0xffffffff,
}}
pub type PDOT11_MANUFACTURING_CALLBACK_TYPE = *mut DOT11_MANUFACTURING_CALLBACK_TYPE;
STRUCT!{struct DOT11_MANUFACTURING_CALLBACK_PARAMETERS {
    Header: NDIS_OBJECT_HEADER,
    dot11ManufacturingCallbackType: DOT11_MANUFACTURING_CALLBACK_TYPE,
    uStatus: ULONG,
    pvContext: PVOID,
}}
pub type PDOT11_MANUFACTURING_CALLBACK_PARAMETERS = *mut DOT11_MANUFACTURING_CALLBACK_PARAMETERS;
pub const DOT11_MANUFACTURING_CALLBACK_REVISION_1: UCHAR = 1;
pub const OID_DOT11_SET_FT_REASSOCIATION_PARAMETERS: u32 =
    NWF_DEFINE_OID!(0x168, NWF_OPERATIONAL_OID, NWF_MANDATORY_OID);
