/* -*- Mode: vala; indent-tabs-mode: t; c-basic-offset: 2; tab-width: 8 -*- */
/*
 * Copyright (C) 2011 Alexander Larsson <alexl@redhat.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gtk;
using Folks;
using Gee;

public class Contacts.View : TreeView {
  private class ContactData {
    public Contact contact;
    public TreeIter iter;
    public bool visible;
    public bool is_first;
    public int sort_prio;
  }

  public enum Subset {
    MAIN,
    OTHER,
    ALL_SEPARATED,
    ALL
  }

  Store contacts_store;
  Subset show_subset;
  ListStore list_store;
  HashSet<Contact> hidden_contacts;
  string []? filter_values;
  int custom_visible_count;
  ContactData suggestions_header_data;
  ContactData padding_data;
  ContactData other_header_data;

  public View (Store store, TextDisplay text_display = TextDisplay.PRESENCE) {
    contacts_store = store;
    hidden_contacts = new HashSet<Contact>();
    show_subset = Subset.ALL;

    list_store = new ListStore (2, typeof (Contact), typeof (ContactData *));
    suggestions_header_data = new ContactData ();
    suggestions_header_data.sort_prio = int.MAX;
    padding_data = new ContactData ();
    padding_data.sort_prio = 1;

    other_header_data = new ContactData ();
    other_header_data.sort_prio = -1;

    list_store.set_sort_func (0, (model, iter_a, iter_b) => {
	ContactData *aa, bb;
	model.get (iter_a, 1, out aa);
	model.get (iter_b, 1, out bb);

	return compare_data (aa, bb);
      });
    list_store.set_sort_column_id (0, SortType.ASCENDING);

    contacts_store.added.connect (contact_added_cb);
    contacts_store.removed.connect (contact_removed_cb);
    contacts_store.changed.connect (contact_changed_cb);
    foreach (var c in store.get_contacts ())
      contact_added_cb (store, c);

      init_view (text_display);
  }

  private int compare_data (ContactData a_data, ContactData b_data) {
    int a_prio = get_sort_prio (a_data);
    int b_prio = get_sort_prio (b_data);

    if (a_prio > b_prio)
      return -1;
    if (a_prio < b_prio)
      return 1;

    var a = a_data.contact;
    var b = b_data.contact;

    if (is_set (a.display_name) && is_set (b.display_name))
      return a.display_name.collate (b.display_name);

    // Sort empty names last
    if (is_set (a.display_name))
      return -1;
    if (is_set (b.display_name))
      return 1;

    return 0;
  }

  private bool is_other (ContactData data) {
    if (show_subset == Subset.ALL_SEPARATED &&
	data.contact != null &&
	!data.contact.is_main)
      return true;
    return false;
  }

  /* The hardcoded prio if set, otherwise 0 for the
     main/combined group, or -2 for the separated other group */
  private int get_sort_prio (ContactData *data) {
    if (data->sort_prio != 0)
      return data->sort_prio;

    if (is_other (data))
      return -2;
    return 0;
  }

  public string get_header_text (TreeIter iter) {
    ContactData *data;
    list_store.get (iter, 1, out data);
    if (data == suggestions_header_data) {
      /* Translators: This is the header for the list of suggested contacts to
	 link to the current contact */
      return ngettext ("Suggestion", "Suggestions", custom_visible_count);
    }
    if (data == other_header_data) {
      /* Translators: This is the header for the list of suggested contacts to
	 link to the current contact */
      return _("Other Contacts");
    }
    return "";
  }

  public void set_show_subset (Subset subset) {
    show_subset = subset;

    bool new_visible = show_subset == Subset.ALL_SEPARATED;
    if (new_visible && !other_header_data.visible) {
      other_header_data.visible = true;
      list_store.append (out other_header_data.iter);
      list_store.set (other_header_data.iter, 1, other_header_data);
    }
    if (!new_visible && other_header_data.visible) {
      other_header_data.visible = false;
      list_store.remove (other_header_data.iter);
    }

    refilter ();
  }

  public void set_custom_sort_prio (Contact c, int prio) {
    /* We use negative prios internally */
    assert (prio >= 0);

    var data = lookup_data (c);

    if (data == null)
      return;

    // We insert a priority between 0 and 1 for the padding
    if (prio > 0)
      prio += 1;
    data.sort_prio = prio;
    contact_changed_cb (contacts_store, c);

    if (data.visible) {
      if (prio > 0) {
	if (custom_visible_count++ == 0)
	  add_custom_headers ();
      } else {
	if (custom_visible_count-- == 1)
	  remove_custom_headers ();
      }
    }
  }

  private bool apply_filter (Contact contact) {
    if (contact.is_hidden)
      return false;

    if (contact in hidden_contacts)
      return false;

    if ((show_subset == Subset.MAIN &&
	 !contact.is_main) ||
	(show_subset == Subset.OTHER &&
	 contact.is_main))
      return false;

    if (filter_values == null || filter_values.length == 0)
      return true;

    return contact.contains_strings (filter_values);
  }

  public bool is_first (TreeIter iter) {
    ContactData *data;
    list_store.get (iter, 1, out data);
    if (data != null)
      return data->is_first;
    return false;
  }

  private ContactData? get_previous (ContactData data) {
    ContactData *previous = null;
    TreeIter iter = data.iter;
    if (list_store.iter_previous (ref iter))
      list_store.get (iter, 1, out previous);
    return previous;
  }

  private ContactData? get_next (ContactData data) {
    ContactData *next = null;
    TreeIter iter = data.iter;
    if (list_store.iter_next (ref iter))
      list_store.get (iter, 1, out next);
    return next;
  }

  private void row_changed_no_resort (ContactData data) {
    var path = list_store.get_path (data.iter);
    list_store.row_changed (path, data.iter);
  }

  private void row_changed_resort (ContactData data) {
    list_store.set (data.iter, 0, data.contact);
  }

  private bool update_is_first (ContactData data, ContactData? previous) {
    bool old_is_first = data.is_first;

    bool is_custom = data.sort_prio != 0;
    bool previous_is_custom = previous != null && (previous.sort_prio != 0) ;

    if (is_custom) {
      data.is_first = false;
    } else if (previous != null && !previous_is_custom) {
      unichar previous_initial = previous.contact.initial_letter;
      unichar initial = data.contact.initial_letter;
      data.is_first = previous_initial != initial;
    } else {
      data.is_first = true;
    }

    if (old_is_first != data.is_first) {
      row_changed_no_resort (data);
      return true;
    }

    return false;
  }

  private void add_custom_headers () {
    suggestions_header_data.visible = true;
    list_store.append (out suggestions_header_data.iter);
    list_store.set (suggestions_header_data.iter, 1, suggestions_header_data);
    padding_data.visible = true;
    list_store.append (out padding_data.iter);
    list_store.set (padding_data.iter, 1, padding_data);
  }

  private void remove_custom_headers () {
    suggestions_header_data.visible = false;
    list_store.remove (suggestions_header_data.iter);
    padding_data.visible = false;
    list_store.remove (padding_data.iter);
  }

  private void add_to_model (ContactData data) {
    list_store.append (out data.iter);
    list_store.set (data.iter, 0, data.contact, 1, data);

    if (data.sort_prio > 0) {
      if (custom_visible_count++ == 0)
	add_custom_headers ();
    }

    if (update_is_first (data, get_previous (data)) && data.is_first) {
      /* The newly added row is first, the next one might not be anymore */
      var next = get_next (data);
      if (next != null)
	update_is_first (next, data);
    }
  }

  private void remove_from_model (ContactData data) {
    if (data.sort_prio > 0) {
      if (custom_visible_count-- == 1)
	remove_custom_headers ();
    }

    ContactData? next = null;
    if (data.is_first)
      next = get_next (data);

    list_store.remove (data.iter);
    data.is_first = false;

    if (next != null)
      update_is_first (next, get_previous (next));
  }

  private void update_visible (ContactData data) {
    bool was_visible = data.visible;
    data.visible = apply_filter (data.contact);

    if (was_visible && !data.visible)
      remove_from_model (data);

    if (!was_visible && data.visible)
      add_to_model (data);
  }

  private void refilter () {
    foreach (var c in contacts_store.get_contacts ()) {
      update_visible (lookup_data (c));
    }
  }

  public void hide_contact (Contact contact) {
    hidden_contacts.add (contact);
    refilter ();
  }

  public void set_filter_values (string []? values) {
    filter_values = values;
    refilter ();
  }

  private void contact_changed_cb (Store store, Contact c) {
    ContactData data = lookup_data (c);

    bool was_visible = data.visible;

    ContactData? next = null;
    if (data.visible)
      next = get_next (data);

    update_visible (data);

    if (was_visible && data.visible) {
      /* We just moved position in the list while visible */

      row_changed_resort (data);

      /* Update the is_first on the previous next row */
      if (next != null)
	update_is_first (next, get_previous (next));

      /* Update the is_first on the new next row */
      next = get_next (data);
      if (next != null)
	update_is_first (next, data);
    }
  }

  private ContactData lookup_data (Contact c) {
    return c.lookup<ContactData> (this);
  }

  private void contact_added_cb (Store store, Contact c) {
    ContactData data =  new ContactData();
    data.contact = c;
    data.visible = false;

    c.set_lookup (this, data);

    update_visible (data);
  }

  private void contact_removed_cb (Store store, Contact c) {
    var data = lookup_data (c);

    if (data.visible)
      remove_from_model (data);

    c.remove_lookup<ContactData> (this);
  }

  public bool lookup_iter (Contact c, out TreeIter iter) {
    var data = lookup_data (c);
    iter = data.iter;
    return data.visible;
  }



  private CellRendererShape shape;
  public enum TextDisplay {
    NONE,
    PRESENCE,
    STORES
  }
  private TextDisplay text_display;

  public signal void selection_changed (Contact? contact);

  private void init_view (TextDisplay text_display) {
    this.text_display = text_display;

    set_model (list_store);
    set_headers_visible (false);

    var row_padding = 12;

    var selection = get_selection ();
    selection.set_mode (SelectionMode.BROWSE);
    selection.set_select_function ( (selection, model, path, path_currently_selected) => {
	Contact contact;
	TreeIter iter;
	model.get_iter (out iter, path);
	model.get (iter, 0, out contact);
	return contact != null;
      });
    selection.changed.connect (contacts_selection_changed);

    var column = new TreeViewColumn ();
    column.set_spacing (8);

    var icon = new CellRendererPixbuf ();
    icon.set_padding (0, row_padding);
    icon.xalign = 1.0f;
    icon.yalign = 0.0f;
    icon.width = Contact.SMALL_AVATAR_SIZE + 12;
    column.pack_start (icon, false);
    column.set_cell_data_func (icon, (column, cell, model, iter) => {
	Contact contact;

	model.get (iter, 0, out contact);

	if (contact == null) {
	  cell.set ("pixbuf", null);
	  cell.visible = false;
	  return;
	}
	cell.visible = true;

	if (contact != null)
	  cell.set ("pixbuf", contact.small_avatar);
	else
	  cell.set ("pixbuf", null);
      });

    shape = new CellRendererShape ();
    shape.set_padding (0, row_padding);

    Pango.cairo_context_set_shape_renderer (get_pango_context (), shape.render_shape);

    column.pack_start (shape, true);
    column.set_cell_data_func (shape, (column, cell, model, iter) => {
	Contact contact;

	model.get (iter, 0, out contact);

	if (contact == null) {
	  cell.visible = false;
	  return;
	}
	cell.visible = true;

	var name = contact.display_name;
	switch (text_display) {
	default:
	case TextDisplay.NONE:
	  cell.set ("name", name,
		    "show_presence", false,
		    "message", "");
	  break;
	case TextDisplay.PRESENCE:
	  cell.set ("name", name,
		    "show_presence", true,
		    "presence", contact.presence_type,
		    "message", contact.presence_message,
		    "is_phone", contact.is_phone);
	  break;
	case TextDisplay.STORES:
	  string stores = contact.format_persona_stores ();
	  cell.set ("name", name,
		    "show_presence", false,
		    "message", stores);
	  break;
	}
      });

    var text = new CellRendererText ();
    text.set_alignment (0, 0);
    column.pack_start (text, true);
    text.set ("weight", Pango.Weight.BOLD);
    column.set_cell_data_func (text, (column, cell, model, iter) => {
	Contact contact;

	model.get (iter, 0, out contact);
	cell.visible = contact == null;
	if (cell.visible) {
	  string header = get_header_text (iter);
	  cell.set ("text", header);
	  if (header == "")
	    cell.height = 6; // PADDING
	  else
	    cell.height = -1;
	}
      });

    append_column (column);
  }

  private void contacts_selection_changed (TreeSelection selection) {
    TreeIter iter;
    TreeModel model;

    Contact? contact = null;
    if (selection.get_selected (out model, out iter)) {
      model.get (iter, 0, out contact);
    }

    selection_changed (contact);
  }

  public void select_contact (Contact contact) {
    TreeIter iter;
    if (lookup_iter (contact, out iter)) {
      get_selection ().select_iter (iter);
      scroll_to_cell (list_store.get_path (iter),
		      null, true, 0.0f, 0.0f);
    }
  }
}
