/*
 * Cogl
 *
 * An object oriented GL/GLES Abstraction/Utility Layer
 *
 * Copyright (C) 2010 Intel Corporation.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see <http://www.gnu.org/licenses/>.
 *
 * Authors:
 *  Robert Bragg <robert@linux.intel.com>
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <cogl.h>
#include <cogl-material-compat.h>
#include <cogl-pipeline.h>
#include <cogl-pipeline-private.h>
#include <cogl-types.h>
#include <cogl-matrix.h>
#include <cogl-context-private.h>

CoglMaterial *
cogl_material_new (void)
{
  return COGL_MATERIAL (cogl_pipeline_new ());
}

CoglMaterial *
cogl_material_copy (CoglMaterial *source)
{
  return COGL_MATERIAL (cogl_pipeline_copy (COGL_PIPELINE (source)));
}

CoglHandle
cogl_material_ref (CoglHandle handle)
{
  return cogl_object_ref (handle);
}

void
cogl_material_unref (CoglHandle handle)
{
  cogl_object_unref (handle);
}

gboolean
cogl_is_material (CoglHandle handle)
{
  return cogl_is_pipeline (handle);
}

void
cogl_material_set_color (CoglMaterial    *material,
                         const CoglColor *color)
{
  cogl_pipeline_set_color (COGL_PIPELINE (material), color);
}

void
cogl_material_set_color4ub (CoglMaterial *material,
			    guint8        red,
                            guint8        green,
                            guint8        blue,
                            guint8        alpha)
{
  cogl_pipeline_set_color4ub (COGL_PIPELINE (material),
                              red, green, blue, alpha);
}

void
cogl_material_set_color4f (CoglMaterial *material,
                           float         red,
                           float         green,
                           float         blue,
                           float         alpha)
{
  cogl_pipeline_set_color4f (COGL_PIPELINE (material),
                             red, green, blue, alpha);
}

void
cogl_material_get_color (CoglMaterial *material,
                         CoglColor    *color)
{
  cogl_pipeline_get_color (COGL_PIPELINE (material), color);
}

void
cogl_material_set_ambient (CoglMaterial    *material,
			   const CoglColor *ambient)
{
  cogl_pipeline_set_ambient (COGL_PIPELINE (material), ambient);
}

void
cogl_material_get_ambient (CoglMaterial *material,
                           CoglColor    *ambient)
{
  cogl_pipeline_get_ambient (COGL_PIPELINE (material), ambient);
}

void
cogl_material_set_diffuse (CoglMaterial    *material,
			   const CoglColor *diffuse)
{
  cogl_pipeline_set_diffuse (COGL_PIPELINE (material), diffuse);
}

void
cogl_material_get_diffuse (CoglMaterial *material,
                           CoglColor    *diffuse)
{
  cogl_pipeline_get_diffuse (COGL_PIPELINE (material), diffuse);
}

void
cogl_material_set_ambient_and_diffuse (CoglMaterial    *material,
				       const CoglColor *color)
{
  cogl_pipeline_set_ambient_and_diffuse (COGL_PIPELINE (material), color);

}

void
cogl_material_set_specular (CoglMaterial    *material,
			    const CoglColor *specular)
{
  cogl_pipeline_set_specular (COGL_PIPELINE (material), specular);
}

void
cogl_material_get_specular (CoglMaterial *material,
                            CoglColor    *specular)
{
  cogl_pipeline_get_specular (COGL_PIPELINE (material), specular);
}

void
cogl_material_set_shininess (CoglMaterial *material,
			     float         shininess)
{
  cogl_pipeline_set_shininess (COGL_PIPELINE (material), shininess);
}

float
cogl_material_get_shininess (CoglMaterial *material)
{
  return cogl_pipeline_get_shininess (COGL_PIPELINE (material));
}

void
cogl_material_set_emission (CoglMaterial    *material,
			    const CoglColor *emission)
{
  cogl_pipeline_set_emission (COGL_PIPELINE (material), emission);

}

void
cogl_material_get_emission (CoglMaterial *material,
                            CoglColor    *emission)
{
  cogl_pipeline_get_emission (COGL_PIPELINE (material), emission);

}

void
cogl_material_set_alpha_test_function (CoglMaterial         *material,
				       CoglMaterialAlphaFunc alpha_func,
				       float                 alpha_reference)
{
  cogl_pipeline_set_alpha_test_function (COGL_PIPELINE (material),
                                         alpha_func,
                                         alpha_reference);
}

gboolean
cogl_material_set_blend (CoglMaterial *material,
                         const char   *blend_string,
                         GError      **error)
{
  return cogl_pipeline_set_blend (COGL_PIPELINE (material),
                                  blend_string,
                                  error);
}

void
cogl_material_set_blend_constant (CoglMaterial *material,
                                  const CoglColor *constant_color)
{
  cogl_pipeline_set_blend_constant (COGL_PIPELINE (material), constant_color);
}

void
cogl_material_set_point_size (CoglMaterial *material,
                              float         point_size)
{
  cogl_pipeline_set_point_size (COGL_PIPELINE (material), point_size);
}

float
cogl_material_get_point_size (CoglMaterial *material)
{
  return cogl_pipeline_get_point_size (COGL_PIPELINE (material));
}

CoglHandle
cogl_material_get_user_program (CoglMaterial *material)
{
  return cogl_pipeline_get_user_program (COGL_PIPELINE (material));
}

void
cogl_material_set_user_program (CoglMaterial *material,
                                CoglHandle program)
{
  cogl_pipeline_set_user_program (COGL_PIPELINE (material), program);
}

void
cogl_material_set_layer (CoglMaterial *material,
			 int           layer_index,
			 CoglHandle    texture)
{
  cogl_pipeline_set_layer_texture (COGL_PIPELINE (material),
                                   layer_index, texture);
}

void
cogl_material_remove_layer (CoglMaterial *material,
			    int           layer_index)
{
  cogl_pipeline_remove_layer (COGL_PIPELINE (material), layer_index);
}

gboolean
cogl_material_set_layer_combine (CoglMaterial *material,
				 int           layer_index,
				 const char   *blend_string,
                                 GError      **error)
{
  return cogl_pipeline_set_layer_combine (COGL_PIPELINE (material),
                                          layer_index,
                                          blend_string,
                                          error);
}

void
cogl_material_set_layer_combine_constant (CoglMaterial    *material,
                                          int              layer_index,
                                          const CoglColor *constant)
{
  cogl_pipeline_set_layer_combine_constant (COGL_PIPELINE (material),
                                            layer_index,
                                            constant);
}

void
cogl_material_set_layer_matrix (CoglMaterial     *material,
				int               layer_index,
				const CoglMatrix *matrix)
{
  cogl_pipeline_set_layer_matrix (COGL_PIPELINE (material),
                                  layer_index, matrix);
}

const GList *
cogl_material_get_layers (CoglMaterial *material)
{
  return _cogl_pipeline_get_layers (COGL_PIPELINE (material));
}

int
cogl_material_get_n_layers (CoglMaterial *material)
{
  return cogl_pipeline_get_n_layers (COGL_PIPELINE (material));
}

CoglMaterialLayerType
cogl_material_layer_get_type (CoglMaterialLayer *layer)
{
  return COGL_MATERIAL_LAYER_TYPE_TEXTURE;
}

CoglHandle
cogl_material_layer_get_texture (CoglMaterialLayer *layer)
{
  return _cogl_pipeline_layer_get_texture (COGL_PIPELINE_LAYER (layer));
}

CoglMaterialFilter
cogl_material_layer_get_min_filter (CoglMaterialLayer *layer)
{
  return _cogl_pipeline_layer_get_min_filter (COGL_PIPELINE_LAYER (layer));
}

CoglMaterialFilter
cogl_material_layer_get_mag_filter (CoglMaterialLayer *layer)
{
  return _cogl_pipeline_layer_get_mag_filter (COGL_PIPELINE_LAYER (layer));
}

void
cogl_material_set_layer_filters (CoglMaterial      *material,
                                 int                layer_index,
                                 CoglMaterialFilter min_filter,
                                 CoglMaterialFilter mag_filter)
{
  cogl_pipeline_set_layer_filters (COGL_PIPELINE (material),
                                   layer_index,
                                   min_filter,
                                   mag_filter);
}

gboolean
cogl_material_set_layer_point_sprite_coords_enabled (CoglMaterial *material,
                                                     int           layer_index,
                                                     gboolean      enable,
                                                     GError      **error)
{
  CoglPipeline *pipeline = COGL_PIPELINE (material);
  return cogl_pipeline_set_layer_point_sprite_coords_enabled (pipeline,
                                                              layer_index,
                                                              enable,
                                                              error);
}

gboolean
cogl_material_get_layer_point_sprite_coords_enabled (CoglMaterial *material,
                                                     int           layer_index)
{
  CoglPipeline *pipeline = COGL_PIPELINE (material);
  return cogl_pipeline_get_layer_point_sprite_coords_enabled (pipeline,
                                                              layer_index);
}

CoglMaterialWrapMode
cogl_material_get_layer_wrap_mode_s (CoglMaterial *material,
                                     int           layer_index)
{
  return cogl_pipeline_get_layer_wrap_mode_s (COGL_PIPELINE (material),
                                              layer_index);
}

void
cogl_material_set_layer_wrap_mode_s (CoglMaterial        *material,
                                     int                  layer_index,
                                     CoglMaterialWrapMode mode)
{
  cogl_pipeline_set_layer_wrap_mode_s (COGL_PIPELINE (material), layer_index,
                                       mode);
}

CoglMaterialWrapMode
cogl_material_get_layer_wrap_mode_t (CoglMaterial *material,
                                     int           layer_index)
{
  return cogl_pipeline_get_layer_wrap_mode_t (COGL_PIPELINE (material),
                                              layer_index);
}

void
cogl_material_set_layer_wrap_mode_t (CoglMaterial        *material,
                                     int                  layer_index,
                                     CoglMaterialWrapMode mode)
{
  cogl_pipeline_set_layer_wrap_mode_t (COGL_PIPELINE (material), layer_index,
                                       mode);
}

CoglMaterialWrapMode
cogl_material_get_layer_wrap_mode_p (CoglMaterial *material,
                                     int           layer_index)
{
  return cogl_pipeline_get_layer_wrap_mode_p (COGL_PIPELINE (material),
                                              layer_index);
}

void
cogl_material_set_layer_wrap_mode_p (CoglMaterial        *material,
                                     int                  layer_index,
                                     CoglMaterialWrapMode mode)
{
  cogl_pipeline_set_layer_wrap_mode_p (COGL_PIPELINE (material), layer_index,
                                       mode);
}

void
cogl_material_set_layer_wrap_mode (CoglMaterial        *material,
                                   int                  layer_index,
                                   CoglMaterialWrapMode mode)
{
  cogl_pipeline_set_layer_wrap_mode (COGL_PIPELINE (material), layer_index,
                                     mode);
}

CoglMaterialWrapMode
cogl_material_layer_get_wrap_mode_s (CoglMaterialLayer *layer)
{
  return _cogl_pipeline_layer_get_wrap_mode_s (COGL_PIPELINE_LAYER (layer));
}

CoglMaterialWrapMode
cogl_material_layer_get_wrap_mode_t (CoglMaterialLayer *layer)
{
  return _cogl_pipeline_layer_get_wrap_mode_t (COGL_PIPELINE_LAYER (layer));
}

CoglMaterialWrapMode
cogl_material_layer_get_wrap_mode_p (CoglMaterialLayer *layer)
{
  return _cogl_pipeline_layer_get_wrap_mode_p (COGL_PIPELINE_LAYER (layer));
}

void
cogl_material_foreach_layer (CoglMaterial *material,
                             CoglMaterialLayerCallback callback,
                             void *user_data)
{
  cogl_pipeline_foreach_layer (COGL_PIPELINE (material),
                               (CoglPipelineLayerCallback)callback, user_data);
}

gboolean
cogl_material_set_depth_state (CoglMaterial *material,
                               const CoglDepthState *state,
                               GError **error)
{
  return cogl_pipeline_set_depth_state (COGL_PIPELINE (material),
                                        state, error);
}

void
cogl_material_get_depth_state (CoglMaterial *material,
                               CoglDepthState *state_out)
{
  cogl_pipeline_get_depth_state (COGL_PIPELINE (material), state_out);
}

