private class KeyValueRenderer: Gtk.CellRenderer
{
    private DConfKeyView view;
    private Gtk.CellRendererText text_renderer;
    private Gtk.CellRendererSpin spin_renderer;
    private Gtk.CellRendererToggle toggle_renderer;
    private Gtk.CellRendererCombo combo_renderer;

    private Key _key;
    public Key key
    {
        get { return _key; }
        set
        {
            _key = value;
            switch (key.type_string)
            {
            case "<enum>":
                combo_renderer.text = key.value.get_string();
                combo_renderer.model = new EnumModel(key.schema.schema.list.enums.lookup(key.schema.enum_name));
                mode = Gtk.CellRendererMode.EDITABLE;
                break;
            case "b":
                toggle_renderer.active = key.value.get_boolean();
                mode = Gtk.CellRendererMode.ACTIVATABLE;
                break;
            case "s":
                text_renderer.text = key.value.get_string();
                mode = Gtk.CellRendererMode.EDITABLE;
                break;
            case "y":
            case "n":
            case "q":
            case "i":
            case "u":
            case "x":
            case "t":
            case "d":
                spin_renderer.text = key.value.print(false);
                double min, max;
                var v = get_variant_as_double(key.value, out min, out max);
                if (key.schema.range != null)
                {
                    double t;
                    min = get_variant_as_double(key.schema.range.min, out t, out t);
                    max = get_variant_as_double(key.schema.range.max, out t, out t);
                }
                spin_renderer.adjustment = new Gtk.Adjustment(v, min, max, 1, 0, 0);
                spin_renderer.digits = 0;
                mode = Gtk.CellRendererMode.EDITABLE;
                break;
            default:
                text_renderer.text = key.value.print(false);            
                mode = Gtk.CellRendererMode.INERT;
                break;
            }
        }
    }

    private static double get_variant_as_double(Variant value, out double min, out double max)
    {
        switch (value.classify ())
        {
        case Variant.Class.BYTE:
            min = 0.0;
            max = 255.0;
            return (double)value.get_byte();
        case Variant.Class.INT16:
            min = int16.MIN;
            max = int16.MAX;
            return (double)value.get_int16();
        case Variant.Class.UINT16:
            min = uint16.MIN;
            max = uint16.MAX;
            return (double)value.get_uint16();
        case Variant.Class.INT32:
            min = int32.MIN;
            max = int32.MAX;
            return (double)value.get_int32();
        case Variant.Class.UINT32:
            min = uint32.MAX;
            max = uint32.MIN;
            return (double)value.get_uint32();
        case Variant.Class.INT64:
            min = int64.MIN;
            max = int64.MAX;
            return (double)value.get_int64();
        case Variant.Class.UINT64:
            min = uint64.MIN;
            max = uint64.MAX;
            return (double)value.get_uint64();
        case Variant.Class.DOUBLE:
            min = double.MIN;
            max = double.MAX;
            return value.get_double();
        default:
            min = 0.0;
            max = 0.0;
            return 0.0;
        }
    }

    public KeyValueRenderer(DConfKeyView view)
    {
        this.view = view;

        text_renderer = new Gtk.CellRendererText();
        text_renderer.editable = true;
        text_renderer.edited.connect(text_edited_cb);

        spin_renderer = new Gtk.CellRendererSpin();
        spin_renderer.editable = true;
        spin_renderer.edited.connect(spin_edited_cb);

        toggle_renderer = new Gtk.CellRendererToggle();
        toggle_renderer.xalign = 0f;
        toggle_renderer.activatable = true;
        toggle_renderer.toggled.connect(toggle_cb);

        combo_renderer = new Gtk.CellRendererCombo();
        combo_renderer.has_entry = false;
        combo_renderer.text_column = 0;
        combo_renderer.editable = true;
        combo_renderer.edited.connect(text_edited_cb);
    }

    private Gtk.CellRenderer renderer
    {
        set {}
        get
        {
            switch (key.type_string)
            {
            case "<enum>":
                return combo_renderer;
            case "b":
                return toggle_renderer;
            case "y":
            case "n":
            case "q":
            case "i":
            case "u":
            case "x":
            case "t":
            case "d":
                return spin_renderer;
            default:
            case "s":
                return text_renderer;
            }
        }
    }

    public override void get_size(Gtk.Widget     widget,
                                  Gdk.Rectangle? cell_area,
                                  out int        x_offset,
                                  out int        y_offset,
                                  out int        width,
                                  out int        height)
    {
        renderer.get_size(widget, cell_area, out x_offset, out y_offset, out width, out height);
    }

    public override void get_preferred_width(Gtk.Widget widget,
                                             out int    minimum_size,
                                             out int    natural_size)
    {
        renderer.get_preferred_width(widget, out minimum_size, out natural_size);
    }

    public override void get_preferred_height_for_width(Gtk.Widget widget,
                                                        int        width,
                                                        out int    minimum_height,
                                                        out int    natural_height)
    {
        renderer.get_preferred_height_for_width(widget, width, out minimum_height, out natural_height);
    }

    public override void get_preferred_height(Gtk.Widget widget,
                                              out int    minimum_size,
                                              out int    natural_size)
    {
        renderer.get_preferred_height(widget, out minimum_size, out natural_size);
    }

    public override void get_preferred_width_for_height(Gtk.Widget widget,
                                                        int        height,
                                                        out int    minimum_width,
                                                        out int    natural_width)
    {
        renderer.get_preferred_width_for_height(widget, height, out minimum_width, out natural_width);
    }

    public override void render(Cairo.Context context,
                                Gtk.Widget    widget,
                                Gdk.Rectangle background_area,
                                Gdk.Rectangle cell_area,
                                Gtk.CellRendererState flags)
    {
        renderer.render(context, widget, background_area, cell_area, flags);
    }

    public override bool activate(Gdk.Event event,
                                  Gtk.Widget widget,
                                  string path,
                                  Gdk.Rectangle background_area,
                                  Gdk.Rectangle cell_area,
                                  Gtk.CellRendererState flags)
    {
        return renderer.activate(event, widget, path, background_area, cell_area, flags);
    }

    public override unowned Gtk.CellEditable start_editing(Gdk.Event event,
                                                           Gtk.Widget widget,
                                                           string path,
                                                           Gdk.Rectangle background_area,
                                                           Gdk.Rectangle cell_area,
                                                           Gtk.CellRendererState flags)
    {
        return renderer.start_editing(event, widget, path, background_area, cell_area, flags);
    }
    
    private Key get_key_from_path(string path)
    {
        Gtk.TreeIter iter;
        view.model.get_iter_from_string(out iter, path);

        Key key;
        view.model.get(iter, 0, out key, -1);
        
        return key;
    }

    private void toggle_cb(Gtk.CellRendererToggle renderer, string path)
    {
        var key = get_key_from_path(path);
        key.value = new Variant.boolean(!key.value.get_boolean());
    }

    private void text_edited_cb(Gtk.CellRendererText renderer, string path, string text)
    {
        var key = get_key_from_path(path);
        key.value = new Variant.string(text);
    }

    private void spin_edited_cb(Gtk.CellRendererText renderer, string path, string text)
    {
        Key key = get_key_from_path(path);
        if (key.type_string == "y")
            key.value = new Variant.byte((uchar)text.to_int());
        else if (key.type_string == "n")
            key.value = new Variant.int16((int16)text.to_int());
        else if (key.type_string == "q")
            key.value = new Variant.uint16((uint16)text.to_int());
        else if (key.type_string == "i")
            key.value = new Variant.int32(text.to_int());
        else if (key.type_string == "u")
            key.value = new Variant.uint32(text.to_int());
        else if (key.type_string == "x")
            key.value = new Variant.int64(text.to_int());
        else if (key.type_string == "t")
            key.value = new Variant.uint64(text.to_int());
        else if (key.type_string == "d")
            key.value = new Variant.double(text.to_double());
    }
}

public class DConfDirView : Gtk.TreeView
{
    public DConfDirView()
    {
        set_headers_visible(false);
        insert_column_with_attributes(-1, "Key", new Gtk.CellRendererText(), "text", 1, null);
    }
}

public class DConfKeyView : Gtk.TreeView
{
    public DConfKeyView()
    {
        var column = new Gtk.TreeViewColumn.with_attributes("Name", new Gtk.CellRendererText(), "text", 1, "weight", 4, null);
        /*column.set_sort_column_id(1);*/
        append_column(column);
        insert_column_with_attributes(-1, "Value", new KeyValueRenderer(this), "key", 0, null);
    }
}
