/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */

package org.gnu.pango;

import org.gnu.glib.Boxed;
import org.gnu.glib.Handle;
/**
 * The FontDescription represents the description of an ideal font.
 * This is used both to list what fonts are available on the system
 * and also for specifying the characteristics of a font to load.
 */
public class FontDescription extends Boxed {

	/**
	 * Create a new FontDescription using a handle from native methods. This
	 * should only be used internally by java-gnome.
	 */
	public FontDescription( Handle handle ){
		this.handle = handle;
	}
	
	/**
	 * Create a new FontDescription.
	 */
	public FontDescription() {
		handle = FontDescription.pango_font_description_new();
	}
	
	/**
	 * Create a new FontDescription from a string in the form
	 * "[FAMILY-LIST][STYLE-OPTIONS][SIZE]" where FAMILY-LIST is a comma
	 * separated list of families optionally terminated by a comma,
	 * STYLE-OPTIONS is a whitespace separated list of words where each
	 * WORD describes one of style, varient, weight, or stretch and
	 * SIZE is a decimal number (size in points).  Any one of the options
	 * may be absent.
	 * 
	 * @param str The string representation for the font description.
	 */
	public FontDescription(String str) {
		handle = FontDescription.pango_font_description_from_string(str);
	}
	
	/**
	 * Sets the family name field of a font description. The family name
	 * represents a family of related font styles, and will resolve to a
	 * particular PangoFontFamily. In some uses of PangoFontDescription, it is
	 * also possible to use a comma separated list of family names for this
	 * field.
	 */
	public void setFamily(String family){
		pango_font_description_set_family(handle, family);
	}

	/**
	 * Gets the family name field of a font description.
	 */
	public String getFamily(){
		return pango_font_description_get_family(handle);
	}

	/**
	 * Sets the style field of a PangoFontDescription. The PangoStyle
	 * enumeration describes whether the font is slanted and the manner in which
	 * it is slanted; it can be either Style.NORMAL, Style.ITALIC,
	 * or Style.OBLIQUE. Most fonts will either have a italic style or an
	 * oblique style, but not both, and font matching in Pango will match italic
	 * specifications with oblique fonts and vice-versa if an exact match is not
	 * found.
	 */
	public void setStyle(Style style){
		pango_font_description_set_style(handle, style.getValue());
	}

	/**
	 * Gets the style field of a PangoFontDescription.
	 */
	public Style getStyle(){
		return Style.intern( pango_font_description_get_style(handle) );
	}

	/**
	 * Sets the variant field of a font description. The Variant can either be
	 * Variant.NORMAL or Variant.SMALL_CAPS.
	 */
	public void setVariant(Variant var){
		pango_font_description_set_variant(handle, var.getValue());
	}

	/**
	 * Gets the variant field of the description
	 */
	public Variant getVariant(){
		return Variant.intern( pango_font_description_get_variant(handle) );
	}

	/**
	 * Sets the weight field of a font description. The weight field specifies
	 * how bold or light the font should be.
	 */
	public void setWeight(Weight weight){
		pango_font_description_set_weight(handle, weight.getValue());
	}

	/**
	 * Gets the weight field of a font description.
	 */
	public Weight getWeight(){
		return Weight.intern( pango_font_description_get_weight(handle) );
	}

	/**
	 * Sets the stretch field of a font description. The stretch field specifies
	 * how narrow or wide the font should be.
	 */
	public void setStretch(Stretch stretch){
		pango_font_description_set_stretch(handle, stretch.getValue());
	}

	/**
	 * Gets the stretch field of a font description.
	 */
	public Stretch getStretch(){
		return Stretch.intern( pango_font_description_get_stretch(handle) );
	}

	/**
	 * Sets the size field of a font description.
	 */
	public void setSize(int size){
		pango_font_description_set_size(handle, size);
	}

	/**
	 * Gets the size field of a font description.
	 */
	public int getSize(){
		return pango_font_description_get_size(handle);
	}

	/**
	 * Determines which fields in a font description have been set.
	 */
	public FontMask getSetFields(){
		return FontMask.intern( pango_font_description_get_set_fields(handle) );
	}

	/**
	 * Merges the fields that are set in desc_to_merge into the fields in desc.
	 * If replace_existing is FALSE, only fields in desc that are not already
	 * set are affected. If TRUE, then fields that are already set will be
	 * replaced as well.
	 */
	public void replace(FontDescription descToMerge, boolean replaceExisting){
		pango_font_description_merge(handle, descToMerge.getHandle(), replaceExisting);
	}

	/**
	 * Creates a string representation of a font description.
	 */
	public String toString(){
		return pango_font_description_to_string(handle);
	}



	/****************************************
	 * BEGINNING OF GENERATED CODE
	 ****************************************/
	native static final protected int pango_font_description_get_type();
	native static final protected Handle pango_font_description_new();
	native static final protected Handle pango_font_description_copy(Handle desc);
	native static final protected int pango_font_description_hash(Handle desc);
	native static final protected boolean pango_font_description_equal(Handle desc1, Handle desc2);
	native static final protected void pango_font_description_free(Handle desc);
	native static final protected void pango_font_description_set_family(Handle desc, String family);
	native static final protected String pango_font_description_get_family(Handle desc);
	native static final protected void pango_font_description_set_style(Handle desc, int style);
	native static final protected int pango_font_description_get_style(Handle desc);
	native static final protected void pango_font_description_set_variant(Handle desc, int variant);
	native static final protected int pango_font_description_get_variant(Handle desc);
	native static final protected void pango_font_description_set_weight(Handle desc, int weight);
	native static final protected int pango_font_description_get_weight(Handle desc);
	native static final protected void pango_font_description_set_stretch(Handle desc, int stretch);
	native static final protected int pango_font_description_get_stretch(Handle desc);
	native static final protected void pango_font_description_set_size(Handle desc, int size);
	native static final protected int pango_font_description_get_size(Handle desc);
	native static final protected int pango_font_description_get_set_fields(Handle desc);
	native static final protected void pango_font_description_unset_fields(Handle desc, int toUnset);
	native static final protected void pango_font_description_merge(Handle desc, Handle descToMerge, boolean replaceExisting);
	native static final protected boolean pango_font_description_better_match(Handle desc, int oldMatch, int newMatch);
	native static final protected Handle pango_font_description_from_string(String str);
	native static final protected String pango_font_description_to_string(Handle desc);
	native static final protected String pango_font_description_to_filename(Handle desc);
	/****************************************
	 * END OF GENERATED CODE
	 ****************************************/
}
