/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */

package org.gnu.gtk;

import java.util.Vector;

import org.gnu.glib.EventMap;
import org.gnu.gtk.event.GtkEventType;
import org.gnu.gtk.event.ToolButtonEvent;
import org.gnu.gtk.event.ToolButtonListener;

/**
 */
public class ToolButton extends ToolItem {
	
	public ToolButton(Widget iconWidget, String label) {
		handle = gtk_tool_button_new(iconWidget.getHandle(), label);
		initializeEventHandlers();
	}
	
	public ToolButton(String stockId) {
		handle = gtk_tool_button_new_from_stock(stockId);
		initializeEventHandlers();
	}
	
	protected ToolButton() {};
	
	public void setLabel(String label) {
		gtk_tool_button_set_label(handle, label);
	}
	
	public String getLabel() {
		return gtk_tool_button_get_label(handle);
	}
	
	public void setUseUnderline(boolean useUnderline) {
		gtk_tool_button_set_use_underline(handle, useUnderline);
	}
	
	public boolean getUseUnderline() {
		return gtk_tool_button_get_use_underline(handle);
	}
	
	public void setStockId(String stockId) {
		gtk_tool_button_set_stock_id(handle, stockId);
	}
	
	public String getStockId() {
		return gtk_tool_button_get_stock_id(handle);
	}
	
	public void setIconWidget(Widget iconWidget) {
		gtk_tool_button_set_icon_widget(handle, iconWidget.getHandle());
	}
	
	public Widget getIconWidget() {
		return new Widget(gtk_tool_button_get_icon_widget(handle));
	}
	
	public void setLabelWidget(Label labelWidget) {
		gtk_tool_button_set_label_widget(handle, labelWidget.getHandle());
	}
	
	public Label getLabelWidget() {
		return new Label(gtk_tool_button_get_label_widget(handle));
	}

	/***************************************
	 * EVENT HANDLING
	 ****************************************/
	/**
	 * Listeners for handling events
	 */
	private Vector tbListeners = null;

	/**
	 * Register an object to handle dialog events.
	 * @see ToolButtonListener
	 */
	public void addListener(ToolButtonListener listener) {
		// Don't add the listener a second time if it is in the Vector.
		int i = findListener(tbListeners, listener);
		if (i == -1) {
			if (null == tbListeners) {
				tbListeners = new Vector();
			}
			tbListeners.addElement(listener);
		}
	}
	
	/**
	 * Removes a listener
	 * @see #addListener(ToolButtonListener)
	 */
	public void removeListener(ToolButtonListener listener) {
		int i = findListener(tbListeners, listener);
		if (i > -1) {
			tbListeners.remove(i);
		}
		if (0 == tbListeners.size()) {
			tbListeners = null;
		}
	}

	protected void fireToolButtonEvent(ToolButtonEvent event) {
		if (null == tbListeners) {
			return;
		}
		int size = tbListeners.size();
		int i = 0;
		while (i < size) {
			ToolButtonListener tbl = (ToolButtonListener)tbListeners.elementAt(i);
			tbl.toolButtonEvent(event);
			i++;
		}
	}

	private void handleClicked() {
		ToolButtonEvent evt = new ToolButtonEvent(this, ToolButtonEvent.Type.CLICKED);
		fireToolButtonEvent(evt);
	}

	protected void initializeEventHandlers() {
		super.initializeEventHandlers();
		evtMap.initialize(this);
	}

	public Class getEventListenerClass(String signal) {
		return evtMap.getListenerClass(signal);
	}

	public GtkEventType getEventType(String signal) {
		return evtMap.getEventType(signal);
	}

	private static EventMap evtMap = new EventMap();
	static {
		addEvents(evtMap);
	}

	/**
	 * Implementation method to build an EventMap for this widget class.
	 * Not useful (or supported) for application use.
	 */
	private static void addEvents(EventMap anEvtMap) {
		anEvtMap.addEvent("clicked", "handleClicked", ToolButtonEvent.Type.CLICKED, ToolButtonListener.class);
	}

	/****************************************
	 * BEGINNING OF JNI CODE
	 ****************************************/
	native static final protected int gtk_tool_button_get_type();
	native static final protected int gtk_tool_button_new(int icon, String label);
	native static final protected int gtk_tool_button_new_from_stock(String stockId);
	native static final protected void gtk_tool_button_set_label(int button, String label);
	native static final protected String gtk_tool_button_get_label(int button);
	native static final protected void gtk_tool_button_set_use_underline(int button, boolean useUnderline);
	native static final protected boolean gtk_tool_button_get_use_underline(int button);
	native static final protected void gtk_tool_button_set_stock_id(int button, String stockId);
	native static final protected String gtk_tool_button_get_stock_id(int button);
	native static final protected void gtk_tool_button_set_icon_widget(int button, int widget);
	native static final protected int gtk_tool_button_get_icon_widget(int button);
	native static final protected void gtk_tool_button_set_label_widget(int button, int widget);
	native static final protected int gtk_tool_button_get_label_widget(int button);
	/****************************************
	 * END OF JNI CODE
	 ****************************************/
}
