/*	$NetBSD$	*/
/*	Id: isiboot.c,v 1.2 1999/12/26 14:33:33 nisimura Exp 	*/

/*-
 * Copyright (c) 2000, 2011 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Tohru Nishimura.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/param.h>
#include <sys/types.h>
#include <sys/endian.h>
#include <sys/ioctl.h>
#include <sys/socket.h>

#include <net/bpf.h>
#include <net/if.h>
#include <net/if_dl.h>
#include <net/if_ether.h>

#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <ifaddrs.h>
#include <paths.h>
#include <poll.h>
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#define	TRACE(l, x) if ((l) <= dbg) printf x

/*
 * TRFS (Integrated Solutions Inc. Transparent Remote File System)
 *
 * Following data format depends on m68k order, and aligned harmful
 * to RISC processors.
 */
struct frame {
	uint8_t dst[ETHER_ADDR_LEN];
	uint8_t src[ETHER_ADDR_LEN];
	uint16_t type;
	uint16_t pad_0;
	uint16_t seqno;
	uint8_t opcode;
	uint8_t pad_1;
	uint8_t pos[4];
	uint8_t siz[4];
	uint8_t data[1];
} __packed;
#define	FRAMETYPE	0x80df
#define	FRAMELEN	1468

struct station {
	int 	fd;
	char	name[MAXHOSTNAMELEN];
	char	ifname[IFNAMSIZ];
	uint8_t addr[ETHER_ADDR_LEN];
} station;

struct session {
	struct session *next;
	int state;
	FILE *file;
	uint8_t addr[ETHER_ADDR_LEN];
} *activelist, *freelist;
#define	NEWPOOL 10

#define	WAITING	0	/* implicit state after receiving the first frame */
#define	OPENING	1	/* waiting for OPEN after CONNECT is received */
#define	TRANSFER 2	/* data transferring state after OPEN is well done */
static const char __unused *state[] = { "WAITING", "OPENING", "TRANSFER" };

#define	CONNECT	0
#define	OPEN	1
#define	READ	2
#define	CLOSE	4
static const char __unused *op[] =
    { "CONNECT", "OPEN", "READ", "WRITE", "CLOSE", "FIND" };

void createbpfport(char *, uint8_t **, size_t *, struct station *);
struct session *search(uint8_t *);
void closedown(struct session *);
void makepool(void);
char *etheraddr(uint8_t *);
int pickif(char *, uint8_t *);

#define	NEQ(x, y)	strcmp(x, y)
#define	FRAME(buf)	((buf) + ((struct bpf_hdr *)(buf))->bh_hdrlen)

const char usage[] =
    "usage: %s [-i interface] [-s directory] [-d tracelevel]";

int
main(int argc, char *argv[])
{
	int cc, dbg;
	size_t iolen;
	uint32_t pos, siz;
	size_t nread;
	char *ifname, *bootwd, *p;
	uint8_t *iobuf;
	struct session *cp;
	struct frame *fp;
	struct pollfd pollfd;
	static const uint8_t om[ETHER_ADDR_LEN] =
	    { 0x00, 0x00, 0x0A, 0, 0, 0 }; /* XXX vendor ID OMRON */

	ifname = bootwd = NULL;
	dbg = 0;
	while ((cc = getopt(argc, argv, "i:s:d:")) != -1) {
		switch (cc) {
		case 'i':
			ifname = optarg;
			break;
		case 's':
			bootwd = optarg;
			break;
		case 'd':
			dbg = atoi(optarg);
			break;
		default:
			errx(EXIT_FAILURE, usage, argv[0]);
		}
	}
	argv += optind;
	argc -= optind;

	if (geteuid() != 0)
		warnx("WARNING: run by non root priviledge");

	memset(station.name, 0, sizeof(station.name));
	gethostname(station.name, sizeof(station.name) - 1);
	if ((p = strchr(station.name, '.')) != NULL)
		*p = '\0';

	createbpfport(ifname, &iobuf, &iolen, &station);

	TRACE(1, ("Using interface: %s\n", station.ifname));

	if (bootwd != NULL && chroot(bootwd) < 0)
		err(EXIT_FAILURE, "%s", argv[0]);

	pollfd.fd = station.fd;
	pollfd.events = POLLIN;
	for (;;) {
		poll(&pollfd, 1, INFTIM);
		read(pollfd.fd, iobuf, iolen);	/* returns 1468 */
		fp = (struct frame *)FRAME(iobuf);
		/* XXX should check ethers(5) and hosts(5) like ndbootd(8) */
		if (memcmp(fp->dst, om, 3) == 0)
			continue;	
		cp = search(fp->src);
		TRACE(2, ("[%s] ", etheraddr(fp->src)));
		switch (cp->state) {
		case WAITING:
			if (fp->opcode != CONNECT)
				continue;
			else if (NEQ((char *)fp->data, station.name)) {
				TRACE(3, ("'%s' not for me\n", fp->data));
				continue;
			}
			cp->state = OPENING;
			TRACE(2, ("new connection\n"));
			break;
		case OPENING:
			if (fp->opcode != OPEN)
				goto aborting;	/* out of phase */
			cp->file = fopen((char *)fp->data, "r");
			if (cp->file == NULL) {
				TRACE(1, ("failed to open '%s'\n", fp->data));
				goto closedown;	/* no such file */
			}
			cp->state = TRANSFER;
			TRACE(2, ("opened '%s'\n", fp->data));
			break;
		case TRANSFER:
			if (fp->opcode == CLOSE) {
				TRACE(2, ("connection closed\n"));
				goto closedown;	/* close request */
			}
			if (fp->opcode != READ)
				goto aborting;	/* out of phase */
			siz = be32dec(fp->siz);
			pos = be32dec(fp->pos);
			nread = siz;
			if (fseek(cp->file, pos, 0L) < 0 ||
			    fread(fp->data, 1, nread, cp->file) < nread) {
				memset(fp->siz, 0, 4); /* corrupted file */
			}
			TRACE(3, ("%u@%u\n", siz, pos));
			break;
 aborting:
			TRACE(1, ("out of phase\n"));
 closedown:
			closedown(cp);
			fp->opcode = CLOSE;
			break;
		}
		memcpy(fp->dst, fp->src, ETHER_ADDR_LEN);
		memcpy(fp->src, station.addr, ETHER_ADDR_LEN);
		write(pollfd.fd, fp, FRAMELEN);
	}
	/* NOTREACHED */
}

struct session *
search(uint8_t client[])
{
	struct session *cp;

	for (cp = activelist; cp; cp = cp->next) {
		if (memcmp(client, cp->addr, ETHER_ADDR_LEN) == 0)
			return cp;
	}
	if (freelist == NULL)
		makepool();
	cp = freelist;
	freelist = cp->next;	
	cp->next = activelist;
	activelist = cp;

	cp->state = WAITING;
	cp->file = NULL;
	memcpy(cp->addr, client, ETHER_ADDR_LEN);
	return cp;
}

void
closedown(struct session *cp)
{
	struct session *cpp;

	cpp = activelist;
	if (cpp == cp)
		activelist = cp->next;
	else {
		do {
			if (cpp->next == cp)
				break;
		} while (NULL != (cpp = cpp->next)); /* should never happen */
		cpp->next = cp->next;
	}
	cp->next = freelist;
	freelist = cp;

	if (cp->file != NULL)
		fclose(cp->file);
	cp->file = NULL;
	memset(cp->addr, 0, ETHER_ADDR_LEN);
}

void
makepool(void)
{
	struct session *cp;
	int n;

	freelist = calloc(NEWPOOL, sizeof(struct session));
	if (freelist == NULL)
		err(EXIT_FAILURE, "Can't allocate pool");
	cp = freelist;
	for (n = 0; n < NEWPOOL - 1; n++) {
		cp->next = cp + 1;
		cp++;
	}
}

char *
etheraddr(uint8_t e[])
{
	static char address[18];

	snprintf(address, sizeof(address), "%02x:%02x:%02x:%02x:%02x:%02x",
	    e[0], e[1], e[2], e[3], e[4], e[5]);
	return address;
}

static struct bpf_insn bpf_insn[] = {
	{ BPF_LD|BPF_H|BPF_ABS,  0, 0, offsetof(struct frame, type) },
	{ BPF_JMP|BPF_JEQ|BPF_K, 0, 1, FRAMETYPE },
	{ BPF_RET|BPF_K,         0, 0, FRAMELEN },
	{ BPF_RET|BPF_K,         0, 0, 0x0 }
};
static struct bpf_program bpf_pgm = {
	sizeof(bpf_insn) / sizeof(bpf_insn[0]),
	bpf_insn
};

void
createbpfport(char *ifname, uint8_t **iobufp, size_t *iolenp,
    struct station *st)
{
	struct ifreq ifr;
	int fd;
	u_int type;
	size_t buflen;
	uint8_t dladdr[ETHER_ADDR_LEN], *buf;
#ifdef BIOCIMMEDIATE
	u_int flag;
#endif
#ifndef _PATH_BPF
	char devbpf[PATH_MAX];
	int n;
#endif

#ifdef _PATH_BPF
	fd = open(_PATH_BPF, O_RDWR, 0);
#else
	n = 0;
	do {
		snprintf(devbpf, sizeof(devbpf), "/dev/bpf%d", n++);
		fd = open(devbpf, O_RDWR, 0);
	} while (fd == -1 && errno == EBUSY);
#endif
	if (fd == -1)
		err(EXIT_FAILURE, "No bpf device available");
	memset(&ifr, 0, sizeof(ifr));
	if (ifname != NULL)
		strlcpy(ifr.ifr_name, ifname, sizeof(ifr.ifr_name));
	if (pickif(ifr.ifr_name, dladdr) < 0)
		errx(EXIT_FAILURE,
		    "No network interface available: %s\n", ifr.ifr_name);

	ioctl(fd, BIOCSETIF, &ifr);
	ioctl(fd, BIOCGDLT, &type);	/* XXX - should check whether EN10MB */
#ifdef BIOCIMMEDIATE
	flag = 1;
	ioctl(fd, BIOCIMMEDIATE, &flag);
#endif
	ioctl(fd, BIOCGBLEN, &buflen);
	ioctl(fd, BIOCSETF, &bpf_pgm);

	buf = malloc(buflen);
	if (buf == NULL)
		err(EXIT_FAILURE, "Can't allocate buffer");
	*iobufp = buf;
	*iolenp = buflen;
	st->fd = fd;
	strlcpy(st->ifname, ifr.ifr_name, sizeof(st->ifname));
	memcpy(st->addr, dladdr, ETHER_ADDR_LEN);
}

int
pickif(char *xname, uint8_t dladdr[ETHER_ADDR_LEN])
{
#define	MATCH(x, v) ((v) == ((v) & (x)))
	int s, error;
	struct ifaddrs *ifaddrs, *ifa;
	struct ifreq ifr;

	error = -1;
	s = socket(AF_INET, SOCK_DGRAM, 0);
	if (s == -1)
		return error;
	if (getifaddrs(&ifaddrs) == -1)
		goto out;

	for (ifa = ifaddrs; ifa != NULL; ifa = ifa->ifa_next) {
		if (ifa->ifa_addr->sa_family == AF_LINK) {
			struct sockaddr_dl *sdl = (void *)&ifa->ifa_addr;

#ifndef CLLADDR
#define CLLADDR(s) ((const char *)((s)->sdl_data + (s)->sdl_nlen))
#endif
			memset(&ifr, 0, sizeof(ifr));
			strlcpy(ifr.ifr_name, ifa->ifa_name,
			    sizeof(ifr.ifr_name));
			ioctl(s, SIOCGIFFLAGS, &ifr);
			if (MATCH(ifr.ifr_flags, IFF_UP|IFF_BROADCAST)) {
				if (xname[0] == '\0') {
					strlcpy(xname, ifa->ifa_name,
					    IFNAMSIZ);
					memcpy(dladdr, CLLADDR(sdl),
					    ETHER_ADDR_LEN);
					error = 0;
					break;
				} else if (strcmp(xname, ifa->ifa_name) == 0) {
					memcpy(dladdr, CLLADDR(sdl),
					    ETHER_ADDR_LEN);
					error = 0;
					break;
				}
			}
		}
	}
	freeifaddrs(ifaddrs);
 out:
	close(s);
	return error;
#undef MATCH
}
