/* $NetBSD$ */

/*
 * Copyright (c) 2003 Dennis I. Chernoivanov
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <sys/socket.h>

#include "paneld.h"

#define DEFAULT_RESIST	2000000	/* 2 sec */
#define DEFAULT_REPEAT
#define DEFAULT_TIMEOUT

static int mode_daemon;

/*
 * Normalize ip address: Strip leading zeroes for each of the
 * address bytes
 */
void
util_ipstrip(int af, char *buf)
{
	char *ptr = buf;

	int bytelen = (af == AF_INET6) ? 4 : 3;
	char sep = (af == AF_INET6) ? ':' : '.';

	for (; (ptr != NULL) && (*ptr != 0); ) {
		char *prev = ptr;
		while (*ptr == '0')
			ptr++;

		if (prev != ptr) {
			if ((ptr - prev) == bytelen)
				prev++;
			strcpy(prev, ptr);
			ptr = prev;
		}
		if ( (ptr = strchr(ptr, sep)) != NULL)
			ptr++;
	}
}

/*
 * Normalize integers: Strip leading zeoroes
 */
void
util_intstrip(char *buf)
{
	char *ptr;
	for (ptr = buf; *ptr == '0'; ptr++)
		;

	if (*ptr == 0)
		strcpy(buf, "0");
	else
		strcpy(buf, ptr);
}

/*
 * Initialize logging
 */
void util_trace_init(int is_daemon)
{
	static char name[] = "paneld";

	mode_daemon = is_daemon;
	openlog(name, is_daemon ? LOG_PID : LOG_CONS, LOG_DAEMON);
}

/*
 * Close logging
 */
void util_trace_destroy(void)
{
	closelog();
}

/*
 * Put a message to a log or console
 */
void util_trace(int priority, const char *fmt, ...)
{
	va_list va;

	va_start(va, fmt);
	if (mode_daemon) {
		vsyslog(priority, fmt, va);
	} else {
		vfprintf(stderr, fmt, va);
		fprintf(stderr, "\n");
	}
	va_end(va);

	if (priority == LOG_ERR)
		exit(1);
}
