/* $NetBSD$ */

/*
 * Copyright (c) 2003 Dennis I. Chernoivanov
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef _PANELD_H_
#define _PANELD_H_

#include <syslog.h>

#include "cf-lex.h"

#define E_OK		0x0
#define E_IO		0x1
#define E_BADV		0x2

#define E_NEXT		0x3
#define E_PREV		0x4
#define E_SELECT	0x5
#define E_INC		0x6
#define E_DEC		0x7
#define E_EXIT		0x8
#define E_QUIT		0x9

struct menu;
struct lcd_devcap;

/* device operations */
struct lcd_dev {
	int	(* open)(const char *node);
	int	(* close)(void);

#define DEV_CAP_AUTOREPEAT	0x1
	void	(* devcap)(struct lcd_devcap*);
	int	(* getsize)(int*, int*);

	int	(* clear)(void);
	int	(* putchr)(int);
	int	(* putstr)(const char *buf, int len);

	int	(* setpos)(int x, int y);
	int	(* cursor)(int on, int blink);

	int	(* getchr)(int *c);
};

/* editor operations */
struct lcd_edit {
#define ALIGN_ANY	0x0
#define ALIGN_CENTER	0x1
	int	(* setbuf)(char *buf, int align_hint);

	int	(* putchr)(int c);
	int	(* getchr)(int *c);
	int	(* getpos)(int *x);
	int	(* setpos)(int x);
	int	(* left)(void);
	int	(* right)(void);
};

struct lcd_window {
	int		(* open)(struct lcd_dev *dev);
	int		(* set_title)(const char *s);
	int		(* set_list)(struct menu *m);
	int		(* set_focus)(struct menu *m);
	int		(* clear)(void);

	int		(* set_text)(char *lines[]);
	struct lcd_edit*(* edit)(void);
};

struct menu_ops {
	int	(* next)(struct menu *);
	int	(* prev)(struct menu *);
	int	(* inc)(struct menu *);
	int	(* dec)(struct menu *);

	int	(* select)(struct menu *);

	char*	(* allocbuf)(int *sz); /* XXX */
};

struct lcd_io {
	int len;

	char *buf;

	char *read;
	char *write;

	char **rargs;
	char **wargs;
};

struct lcd_xlate {
	char x_inc;
	char x_dec;
	char x_next;
	char x_prev;
	char x_enter;

	char *id;
	struct lcd_xlate *next;
};

struct lcd_devcap {
	long capabilities;
	long repeat;
	long keyrate;
	long resist;
};

struct lcd_globals {
	char x_enter;

	char *banner_cmd;
	char **banner_args;

	char *dev_name;
	char *dev_node;

	char *banner_buf;
	int buf_size;

	struct lcd_devcap devcap;
};

struct menu {
	char *id;
	char *nm;

	struct lcd_io io;
	struct menu_ops *ops;

	struct lcd_xlate *xlate;

	struct menu *next;
	struct menu *prev;

	struct menu *child;
};

/* globals */
extern struct menu *root;
extern struct lcd_dev *dev;
extern struct lcd_xlate *xlate;
extern struct lcd_window *window;
extern struct lcd_globals *globals;

/* well-defined menu operations */
struct menu_ops*	get_none_ops(void);
struct menu_ops*	get_exit_ops(void);
struct menu_ops*	get_quit_ops(void);
struct menu_ops*	get_select_ops(void);
struct menu_ops*	get_confirm_ops(void);
struct menu_ops*	get_sequence_ops(void);
struct menu_ops*	get_inet_ops(void);
struct menu_ops*	get_inet6_ops(void);
struct menu_ops*	get_text_ops(void);
struct menu_ops*	get_int_ops(void);
struct menu_ops*	get_hex_ops(void);

/* noop menu operations */
int	none_next(struct menu *m);
int	none_prev(struct menu *m);
int	none_inc(struct menu *m);
int	none_dec(struct menu *m);
int	none_select(struct menu *m);
char*	none_allocbuf(int *sz);

/* external command execution */
int	parse_args(char *buf, char ***r_args);
int	readconf(char *args[], char *buf, int *size);
int	exec_read(struct menu *m);
int	exec_write(struct menu *m);

/* menu logic */
int	run_menu(void);
int	enter_menu(struct menu *m);
void	cf_parse(const char *cfile);
int	scankey(struct lcd_devcap *cap, int do_resist, int *key);

/* lcd initialization */
struct lcd_window*	get_window(void);
struct lcd_dev*		get_console(void);
struct lcd_dev*		get_hd44780(void);

struct lcd_window* window_create(void);
void window_destroy(struct lcd_window *w);

struct lcd_dev*	device_attach(const char *type, const char *node);
void		device_detach(struct lcd_dev *dev);

/* int/hex normalization */
void	util_intstrip(char *buf);
void	util_ipstrip(int af, char *buf);

/* logging */
void	util_trace_init(int is_daemon);
void	util_trace_destroy(void);
void	util_trace(int priority, const char *fmt, ...);

#endif /* _PANELD_H_ */
