/* $NetBSD$ */

/*
 * Copyright (c) 2003 Dennis I. Chernoivanov
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <unistd.h>
#include <sys/time.h>

#include "paneld.h"

#define SEC_MICRO	1000000

static struct timeval tv_last;	/* last scan time */

/*
 * Time difference between two timeval structures, in
 * microseconds
 */
static long
timediff(struct timeval *before, struct timeval *after)
{
	long sec = after->tv_sec - before->tv_sec;
	long usec = after->tv_usec - before->tv_usec;

	if (usec < 0) {
		sec--;
		usec += SEC_MICRO;
	}

	return (sec * SEC_MICRO + usec);
}

/*
 * Read a character from device and analyze it:
 *  - Generate auto-repeat if needed
 *  - Ignore input if its generated too fast for the user
 */
int
scankey(struct lcd_devcap *cap, int do_resist, int *key)
{
	int c;

	struct timezone tz;
	struct timeval tv_after;
	struct timeval tv_before;

	long waittime = do_resist ? cap->resist : cap->repeat;

	if ((cap->capabilities & DEV_CAP_AUTOREPEAT) == 0)
		return dev->getchr(key);

	if (gettimeofday(&tv_after, &tz) == -1)
		return E_IO;

	for (;;) {
		tv_before = tv_after;
		if (dev->getchr(&c) != E_OK)
			return E_IO;
		if (gettimeofday(&tv_after, &tz) == -1)
			return E_IO;
		if (timediff(&tv_before, &tv_after) > cap->keyrate) {
			/* button was not pressed all the time, reset */
			if (do_resist) {
				waittime = cap->resist;
				continue;
			}
			tv_last = tv_after;
			break;
		}

		waittime -= timediff(&tv_before, &tv_after);
		if (waittime <= 0) {
			tv_last = tv_after;
			break;
		}
	}

	*key = c;
	return (c == -1) ? E_IO : E_OK;
}
