/*	$NetBSD: fstest_ffs.c,v 1.6 2012/08/05 02:03:05 riastradh Exp $	*/

/*-
 * Copyright (c) 2010 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Nicolas Joly.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/mount.h>
#include <sys/stat.h>

#include <atf-c.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>

#include <ufs/ufs/ufsmount.h>

#include <rump/rump.h>
#include <rump/rump_syscalls.h>

#include "h_fsmacros.h"

struct ffstestargs {
        struct ufs_args ta_uargs;
        char ta_devpath[MAXPATHLEN];
        char ta_imgpath[MAXPATHLEN];
};

int
ffs_fstest_newfs(const atf_tc_t *tc, void **buf, const char *image, off_t size,
	void *fspriv)
{
	char cmd[1024];
	int res;
	static unsigned int num = 0;
	struct ffstestargs *args;
	struct sigaction act, oact;

	size /= 512;
	snprintf(cmd, 1024, "newfs -F -s %"PRId64" %s >/dev/null", size, image);
	memset(&act, 0, sizeof(act));
	act.sa_handler = SIG_DFL;
	sigaction(SIGCHLD, &act, &oact);
	res = system(cmd);
	sigaction(SIGCHLD, &oact, NULL);
	if (res != 0)
		return res;

	res = rump_init();
	if (res != 0)
		return res;

	args = calloc(1, sizeof(*args));
	if (args == NULL)
		return -1;

	snprintf(args->ta_devpath, MAXPATHLEN, "/dev/device%d.ffs", num);
	snprintf(args->ta_imgpath, MAXPATHLEN, "%s", image);
	args->ta_uargs.fspec = args->ta_devpath;

	res = rump_pub_etfs_register(args->ta_devpath, image, RUMP_ETFS_BLK);
	if (res != 0) {
		free(args);
		return res;
	}

	*buf = args;
	num++;

	return 0;
}
__strong_alias(ffslog_fstest_newfs,ffs_fstest_newfs);

int
ffs_fstest_delfs(const atf_tc_t *tc, void *buf)
{
	int res;
	struct ffstestargs *args = buf;

	res = rump_pub_etfs_remove(args->ta_devpath);
	if (res != 0) {
		errno = res;
		return -1;
	}

	res = unlink(args->ta_imgpath);
	if (res != 0)
		return res;

	free(args);

	return 0;
}
__strong_alias(ffslog_fstest_delfs,ffs_fstest_delfs);

int
ffs_fstest_mount(const atf_tc_t *tc, void *buf, const char *path, int flags)
{
	int res;
	struct ffstestargs *args = buf;

	res = rump_sys_mkdir(path, 0777);
	if (res == -1)
		return res;

	res = rump_sys_mount(MOUNT_FFS, path, flags, &args->ta_uargs,
	    sizeof(args->ta_uargs));
	return res;
}

int
ffslog_fstest_mount(const atf_tc_t *tc, void *buf, const char *path, int flags)
{

	return ffs_fstest_mount(tc, buf, path, flags | MNT_LOG);
}

int
ffs_fstest_unmount(const atf_tc_t *tc, const char *path, int flags)
{
	int res;

	res = rump_sys_unmount(path, flags);
	if (res == -1)
		return res;

	res = rump_sys_rmdir(path);
	return res;
}
__strong_alias(ffslog_fstest_unmount,ffs_fstest_unmount);
