/*
 This file is part of GNU Taler
 (C) 2022-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import {
  urlPattern,
  useCurrentLocation,
  useNavigationContext
} from "@gnu-taler/web-util/browser";
import { Fragment, VNode, h } from "preact";

import { assertUnreachable } from "@gnu-taler/taler-util";
import { useErrorBoundary } from "preact/hooks";
import { CheckChallengeIsUpToDate } from "./components/CheckChallengeIsUpToDate.js";
import { SessionId } from "./hooks/session.js";
import { AnswerChallenge } from "./pages/AnswerChallenge.js";
import { AskChallenge } from "./pages/AskChallenge.js";
import { CallengeCompleted } from "./pages/CallengeCompleted.js";
import { Frame } from "./pages/Frame.js";
import { Setup } from "./pages/Setup.js";

export function Routing(): VNode {
  // check session and defined if this is
  // public routing or private
  return (
    <Frame>
      <PublicRounting />
    </Frame>
  );
}

const publicPages = {
  noinfo: urlPattern(/\/noinfo/, () => `#/noinfo`),
  authorize: urlPattern(/\/authorize/, () => `#/authorize`),
  ask: urlPattern(/\/ask/, () => `#/ask`),
  answer: urlPattern(/\/answer/, () => `#/answer`),
  completed: urlPattern(/\/completed/, () => `#/completed`),
  setup: urlPattern<{ client: string }>(
    /\/setup\/(?<client>[0-9]+)/,
    ({ client }) => `#/setup/${client}`,
  ),
};

function safeGetParam(
  ps: Record<string, string[]>,
  n: string,
): string | undefined {
  if (!ps[n] || ps[n].length == 0) return undefined;
  return ps[n][0];
}

export function safeToURL(s: string | undefined): URL | undefined {
  if (s === undefined) return undefined;
  try {
    return new URL(s);
  } catch (e) {
    return undefined;
  }
}

function PublicRounting(): VNode {
  const loc = useCurrentLocation(publicPages);
  const { navigateTo } = useNavigationContext();
  useErrorBoundary((e) => {
    console.log("error", e);
  });

  const location: typeof loc =
    loc.name === undefined
      ? {
          ...loc,
          name: "authorize",
        }
      : loc;

  switch (location.name) {
    case "noinfo": {
      return <div>no info</div>;
    }
    case "setup": {
      const secret = safeGetParam(location.params, "secret");
      const redirectURL = safeToURL(
        safeGetParam(location.params, "redirect_uri"),
      );

      return (
        <Setup
          clientId={location.values.client}
          secret={secret}
          redirectURL={redirectURL}
          onCreated={() => {
            navigateTo(publicPages.ask.url({}));
          }}
        />
      );
    }
    case "authorize": {
      const clientId = safeGetParam(location.params, "client_id");
      const redirectURL = safeToURL(
        safeGetParam(location.params, "redirect_uri"),
      );
      const state = safeGetParam(location.params, "state");
      const nonce = safeGetParam(location.params, "nonce");

      const sessionId: SessionId | undefined =
        !clientId || !redirectURL || !state || !nonce
          ? undefined
          : {
              clientId,
              nonce: nonce,
              redirectURL: redirectURL.href,
              state,
            };

      if (!sessionId) {
        return (
          <div>
            one of the params is missing{" "}
            {JSON.stringify(
              { clientId, redirectURL, state, nonce },
              undefined,
              2,
            )}
          </div>
        );
      }
      return (
        <CheckChallengeIsUpToDate
          session={sessionId}
          onCompleted={() => {
            navigateTo(
              publicPages.completed.url({
                nonce: location.values.nonce,
              }),
            );
          }}
          onChangeLeft={() => {
            navigateTo(
              publicPages.ask.url({
                nonce: location.values.nonce,
              }),
            );
          }}
          onNoMoreChanges={() => {
            navigateTo(
              publicPages.ask.url({
                nonce: location.values.nonce,
              }),
            );
          }}
        >
          No nonce has been found
        </CheckChallengeIsUpToDate>
      );
    }
    case "ask": {
      const clientId = safeGetParam(location.params, "client_id");
      const redirectURL = safeToURL(
        safeGetParam(location.params, "redirect_uri"),
      );
      const state = safeGetParam(location.params, "state");
      const nonce = safeGetParam(location.params, "nonce");

      const sessionId: SessionId | undefined =
        !clientId || !redirectURL || !state || !nonce
          ? undefined
          : {
              clientId,
              nonce: nonce,
              redirectURL: redirectURL.href,
              state,
            };

      if (!sessionId) {
        return (
          <div>
            one of the params is missing{" "}
            {JSON.stringify(sessionId, undefined, 2)}
          </div>
        );
      }
      return (
        <AskChallenge
          session={sessionId}
          focus
          routeSolveChallenge={publicPages.answer}
          onSendSuccesful={() => {
            navigateTo(
              publicPages.answer.url({
                nonce: location.values.nonce,
              }),
            );
          }}
          // onCompleted={() => {
          //   navigateTo(
          //     publicPages.completed.url({
          //       nonce: location.values.nonce,
          //     }),
          //   );
          // }}
        />
      );
    }
    case "answer": {
      const clientId = safeGetParam(location.params, "client_id");
      const redirectURL = safeToURL(
        safeGetParam(location.params, "redirect_uri"),
      );
      const state = safeGetParam(location.params, "state");
      const nonce = safeGetParam(location.params, "nonce");

      const sessionId: SessionId | undefined =
        !clientId || !redirectURL || !state || !nonce
          ? undefined
          : {
              clientId,
              nonce: nonce,
              redirectURL: redirectURL.href,
              state,
            };

      if (!sessionId) {
        return (
          <div>
            one of the params is missing{" "}
            {JSON.stringify(
              { clientId, redirectURL, state, nonce },
              undefined,
              2,
            )}
          </div>
        );
      }
      return (
        <AnswerChallenge
          focus
          session={sessionId}
          routeAsk={publicPages.ask}
          onComplete={() => {
            navigateTo(
              publicPages.completed.url({
                nonce: location.values.nonce,
              }),
            );
          }}
          // onCompleted={() => {
          //   navigateTo(
          //     publicPages.completed.url({
          //       nonce: location.values.nonce,
          //     }),
          //   );
          // }}
        />
      );
    }
    case "completed": {
      return <CallengeCompleted />;
    }
    default:
      assertUnreachable(location);
  }
}
