/*
 This file is part of GNU Taler
 (C) 2022-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
import { useState } from "preact/hooks";

// FIX default import https://github.com/microsoft/TypeScript/issues/49189
import {
  OfficerAccount,
  OperationOk,
  TalerExchangeApi,
  TalerExchangeResultByMethod,
  TalerHttpError,
} from "@gnu-taler/taler-util";
import _useSWR, { SWRHook } from "swr";
import { useOfficer } from "./officer.js";
import { useExchangeApiContext } from "@gnu-taler/web-util/browser";
const useSWR = _useSWR as unknown as SWRHook;

export const PAGINATED_LIST_SIZE = 10;
// when doing paginated request, ask for one more
// and use it to know if there are more to request
export const PAGINATED_LIST_REQUEST = PAGINATED_LIST_SIZE + 1;

/**
 * FIXME: mutate result when balance change (transaction )
 * @param account
 * @param args
 * @returns
 */
export function useCases(state: TalerExchangeApi.AmlState) {
  const officer = useOfficer();
  const session = officer.state === "ready" ? officer.account : undefined;
  const {
    lib: { exchange: api },
  } = useExchangeApiContext();

  const [offset, setOffset] = useState<string>();

  async function fetcher([officer, state, offset]: [
    OfficerAccount,
    TalerExchangeApi.AmlState,
    string | undefined,
  ]) {
    return await api.getDecisionsByState(officer, state, {
      order: "asc",
      offset,
      limit: PAGINATED_LIST_REQUEST,
    });
  }

  const { data, error } = useSWR<
    TalerExchangeResultByMethod<"getDecisionsByState">,
    TalerHttpError
  >(
    !session ? undefined : [session, state, offset, "getDecisionsByState"],
    fetcher,
  );

  if (error) return error;
  if (data === undefined) return undefined;
  if (data.type !== "ok") return data;

  return buildPaginatedResult(data.body.records, offset, setOffset, (d) =>
    String(d.rowid),
  );
}

type PaginatedResult<T> = OperationOk<T> & {
  isLastPage: boolean;
  isFirstPage: boolean;
  loadNext(): void;
  loadFirst(): void;
};

//TODO: consider sending this to web-util
export function buildPaginatedResult<R, OffId>(
  data: R[],
  offset: OffId | undefined,
  setOffset: (o: OffId | undefined) => void,
  getId: (r: R) => OffId,
): PaginatedResult<R[]> {
  const isLastPage = data.length < PAGINATED_LIST_REQUEST;
  const isFirstPage = offset === undefined;

  const result = structuredClone(data);
  if (result.length == PAGINATED_LIST_REQUEST) {
    result.pop();
  }
  return {
    type: "ok",
    body: result,
    isLastPage,
    isFirstPage,
    loadNext: () => {
      if (!result.length) return;
      const id = getId(result[result.length - 1]);
      setOffset(id);
    },
    loadFirst: () => {
      setOffset(undefined);
    },
  };
}
