/*
   This file is part of TALER
   Copyright (C) 2022--2025 Taler Systems SA

   TALER is free software; you can redistribute it and/or modify it under the
   terms of the GNU General Public License as published by the Free Software
   Foundation; either version 3, or (at your option) any later version.

   TALER is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
   A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along with
   TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
/**
 * @file backenddb/pg_lookup_instances.c
 * @brief Implementation of the lookup_instances function for Postgres
 * @author Christian Grothoff
 */
#include "platform.h"
#include <taler/taler_error_codes.h>
#include <taler/taler_pq_lib.h>
#include "pg_lookup_instances.h"
#include "pg_helper.h"


/**
 * Context for lookup_instances().
 */
struct LookupInstancesContext
{
  /**
   * Function to call with the results.
   */
  TALER_MERCHANTDB_InstanceCallback cb;

  /**
   * Closure for @e cb.
   */
  void *cb_cls;

  /**
   * Database context.
   */
  struct PostgresClosure *pg;

  /**
   * Set to the return value on errors.
   */
  enum GNUNET_DB_QueryStatus qs;

};


/**
 * Function to be called with the results of a SELECT statement
 * that has returned @a num_results results about instances.
 *
 * @param cls of type `struct FindInstancesContext *`
 * @param result the postgres result
 * @param num_results the number of results in @a result
 */
static void
lookup_instances_cb (void *cls,
                     PGresult *result,
                     unsigned int num_results)
{
  struct LookupInstancesContext *lic = cls;

  for (unsigned int i = 0; i < num_results; i++)
  {
    struct TALER_MERCHANTDB_InstanceSettings is;
    struct TALER_MERCHANTDB_InstanceAuthSettings ias;
    uint64_t instance_serial;
    struct TALER_MerchantPublicKeyP merchant_pub;
    struct TALER_MerchantPrivateKeyP merchant_priv;
    bool no_auth;
    bool no_salt;
    bool no_priv;
    char *dwtri;
    struct GNUNET_PQ_ResultSpec rs[] = {
      GNUNET_PQ_result_spec_uint64 ("merchant_serial",
                                    &instance_serial),
      GNUNET_PQ_result_spec_auto_from_type ("merchant_pub",
                                            &merchant_pub),
      GNUNET_PQ_result_spec_allow_null (
        GNUNET_PQ_result_spec_auto_from_type ("auth_hash",
                                              &ias.auth_hash),
        &no_auth),
      GNUNET_PQ_result_spec_allow_null (
        GNUNET_PQ_result_spec_auto_from_type ("auth_salt",
                                              &ias.auth_salt),
        &no_salt),
      GNUNET_PQ_result_spec_allow_null (
        GNUNET_PQ_result_spec_auto_from_type ("merchant_priv",
                                              &merchant_priv),
        &no_priv),
      GNUNET_PQ_result_spec_string ("merchant_id",
                                    &is.id),
      GNUNET_PQ_result_spec_string ("merchant_name",
                                    &is.name),
      TALER_PQ_result_spec_json ("address",
                                 &is.address),
      TALER_PQ_result_spec_json ("jurisdiction",
                                 &is.jurisdiction),
      GNUNET_PQ_result_spec_bool ("use_stefan",
                                  &is.use_stefan),
      GNUNET_PQ_result_spec_bool ("phone_validated",
                                  &is.phone_validated),
      GNUNET_PQ_result_spec_bool ("email_validated",
                                  &is.email_validated),
      GNUNET_PQ_result_spec_relative_time (
        "default_wire_transfer_delay",
        &is.default_wire_transfer_delay),
      GNUNET_PQ_result_spec_relative_time ("default_pay_delay",
                                           &is.default_pay_delay),
      GNUNET_PQ_result_spec_relative_time ("default_refund_delay",
                                           &is.default_refund_delay),
      GNUNET_PQ_result_spec_allow_null (
        GNUNET_PQ_result_spec_string ("website",
                                      &is.website),
        NULL),
      GNUNET_PQ_result_spec_allow_null (
        GNUNET_PQ_result_spec_string ("email",
                                      &is.email),
        NULL),
      GNUNET_PQ_result_spec_allow_null (
        GNUNET_PQ_result_spec_string ("phone_number",
                                      &is.phone),
        NULL),
      GNUNET_PQ_result_spec_allow_null (
        GNUNET_PQ_result_spec_string ("logo",
                                      &is.logo),
        NULL),
      GNUNET_PQ_result_spec_string (
        "default_wire_transfer_rounding_interval",
        &dwtri),
      GNUNET_PQ_result_spec_end
    };

    memset (&ias.auth_salt,
            0,
            sizeof (ias.auth_salt));
    memset (&ias.auth_hash,
            0,
            sizeof (ias.auth_hash));
    if (GNUNET_OK !=
        GNUNET_PQ_extract_result (result,
                                  rs,
                                  i))
    {
      GNUNET_break (0);
      lic->qs = GNUNET_DB_STATUS_HARD_ERROR;
      return;
    }
    if (GNUNET_OK !=
        GNUNET_TIME_string_to_round_interval (
          dwtri,
          &is.default_wire_transfer_rounding_interval))
    {
      GNUNET_break (0);
      lic->qs = GNUNET_DB_STATUS_HARD_ERROR;
      return;
    }
    lic->cb (lic->cb_cls,
             &merchant_pub,
             (no_priv) ? NULL : &merchant_priv,
             &is,
             &ias);
    GNUNET_PQ_cleanup_result (rs);
  }
}


enum GNUNET_DB_QueryStatus
TMH_PG_lookup_instances (void *cls,
                         bool active_only,
                         TALER_MERCHANTDB_InstanceCallback cb,
                         void *cb_cls)
{
  struct PostgresClosure *pg = cls;
  struct LookupInstancesContext lic = {
    .cb = cb,
    .cb_cls = cb_cls,
    .pg = pg
  };
  struct GNUNET_PQ_QueryParam params[] = {
    GNUNET_PQ_query_param_end
  };
  enum GNUNET_DB_QueryStatus qs;

  check_connection (pg);
  PREPARE (pg,
           "lookup_instances",
           "SELECT"
           " mi.merchant_serial"
           ",mi.merchant_pub"
           ",mi.auth_hash"
           ",mi.auth_salt"
           ",mi.merchant_id"
           ",mi.merchant_name"
           ",mi.address::TEXT"
           ",mi.jurisdiction::TEXT"
           ",mi.use_stefan"
           ",mi.default_wire_transfer_delay"
           ",mi.default_pay_delay"
           ",mi.default_refund_delay"
           ",mi.website"
           ",mi.email"
           ",mi.phone_number"
           ",mi.phone_validated"
           ",mi.email_validated"
           ",mi.logo"
           ",mi.default_wire_transfer_rounding_interval::TEXT"
           ",mk.merchant_priv"
           " FROM merchant_instances mi"
           " LEFT JOIN merchant_keys mk"
           "   USING (merchant_serial)");
  PREPARE (pg,
           "lookup_active_instances",
           "SELECT "
           " mi.merchant_serial"
           ",mi.merchant_pub"
           ",mi.auth_hash"
           ",mi.auth_salt"
           ",mi.merchant_id"
           ",mi.merchant_name"
           ",mi.address::TEXT"
           ",mi.jurisdiction::TEXT"
           ",mi.use_stefan"
           ",mi.default_wire_transfer_delay"
           ",mi.default_pay_delay"
           ",mi.default_refund_delay"
           ",mi.website"
           ",mi.email"
           ",mi.phone_number"
           ",mi.phone_validated"
           ",mi.email_validated"
           ",mi.logo"
           ",mi.default_wire_transfer_rounding_interval::TEXT"
           ",mk.merchant_priv"
           " FROM merchant_instances mi"
           " JOIN merchant_keys mk"
           "   USING (merchant_serial)");
  qs = GNUNET_PQ_eval_prepared_multi_select (
    pg->conn,
    active_only
    ? "lookup_active_instances"
    : "lookup_instances",
    params,
    &lookup_instances_cb,
    &lic);
  if (0 > lic.qs)
    return lic.qs;
  return qs;
}


enum GNUNET_DB_QueryStatus
TMH_PG_lookup_instance (void *cls,
                        const char *id,
                        bool active_only,
                        TALER_MERCHANTDB_InstanceCallback cb,
                        void *cb_cls)
{
  struct PostgresClosure *pg = cls;
  struct LookupInstancesContext lic = {
    .cb = cb,
    .cb_cls = cb_cls,
    .pg = pg
  };
  struct GNUNET_PQ_QueryParam params[] = {
    GNUNET_PQ_query_param_string (id),
    GNUNET_PQ_query_param_end
  };
  enum GNUNET_DB_QueryStatus qs;

  check_connection (pg);
  PREPARE (pg,
           "lookup_instance",
           "SELECT"
           " mi.merchant_serial"
           ",mi.merchant_pub"
           ",mi.auth_hash"
           ",mi.auth_salt"
           ",mi.merchant_id"
           ",mi.merchant_name"
           ",mi.address::TEXT"
           ",mi.jurisdiction::TEXT"
           ",mi.use_stefan"
           ",mi.default_wire_transfer_delay"
           ",mi.default_pay_delay"
           ",mi.default_refund_delay"
           ",mi.website"
           ",mi.email"
           ",mi.phone_number"
           ",mi.phone_validated"
           ",mi.email_validated"
           ",mi.logo"
           ",mi.default_wire_transfer_rounding_interval::TEXT"
           ",mk.merchant_priv"
           " FROM merchant_instances mi"
           " LEFT JOIN merchant_keys mk"
           "   USING (merchant_serial)"
           " WHERE merchant_id=$1");
  PREPARE (pg,
           "lookup_active_instance",
           "SELECT"
           " mi.merchant_serial"
           ",mi.merchant_pub"
           ",mi.auth_hash"
           ",mi.auth_salt"
           ",mi.merchant_id"
           ",mi.merchant_name"
           ",mi.address::TEXT"
           ",mi.jurisdiction::TEXT"
           ",mi.use_stefan"
           ",mi.default_wire_transfer_delay"
           ",mi.default_pay_delay"
           ",mi.default_refund_delay"
           ",mi.website"
           ",mi.email"
           ",mi.phone_number"
           ",mi.phone_validated"
           ",mi.email_validated"
           ",mi.logo"
           ",mi.default_wire_transfer_rounding_interval::TEXT"
           ",mk.merchant_priv"
           " FROM merchant_instances mi"
           " JOIN merchant_keys mk"
           "   USING (merchant_serial)"
           " WHERE merchant_id=$1");
  qs = GNUNET_PQ_eval_prepared_multi_select (
    pg->conn,
    active_only
    ? "lookup_active_instance"
    : "lookup_instance",
    params,
    &lookup_instances_cb,
    &lic);
  if (0 > lic.qs)
    return lic.qs;
  return qs;
}
