/* mathfuncs.c: -*- C -*-  Arithmetic functions for Meta-HTML. */

/* Author: Brian J. Fox (bfox@ai.mit.edu) Tue Jul 18 17:50:42 1995.

   This file is part of <Meta-HTML>(tm), a system for the rapid deployment
   of Internet and Intranet applications via the use of the Meta-HTML
   language.

   Copyright (c) 1995, 1996, Brian J. Fox (bfox@ai.mit.edu).
   Copyright (c) 1996, Universal Access Inc. (http://www.ua.com).

   Meta-HTML is free software; you can redistribute it and/or modify
   it under the terms of the UAI Free Software License as published
   by Universal Access Inc.; either version 1, or (at your option) any
   later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   UAI Free Software License for more details.

   You should have received a copy of the UAI Free Software License
   along with this program; if you have not, you may obtain one by
   writing to:

   Universal Access Inc.
   129 El Paseo Court
   Santa Barbara, CA
   93101  */

#include "language.h"

#if defined (MHTML_ARITHMETIC)
static void pf_gt (PFunArgs);
static void pf_lt (PFunArgs);
static void pf_eq (PFunArgs);
static void pf_add (PFunArgs);
static void pf_sub (PFunArgs);
static void pf_mul (PFunArgs);
static void pf_div (PFunArgs);
static void pf_mod (PFunArgs);

static PFunDesc func_table[] =
{
  { "GT",		0, 0, pf_gt },
  { "LT",		0, 0, pf_lt },
  { "EQ",		0, 0, pf_eq },
  { "ADD",		0, 0, pf_add },
  { "SUB",		0, 0, pf_sub },
  { "MUL",		0, 0, pf_mul },
  { "DIV",		0, 0, pf_div },
  { "MOD",		0, 0, pf_mod },

  { (char *)NULL,	0, 0, (PFunHandler *)NULL }
};

PACKAGE_INITIALIZER (initialize_arithmetic_functions)

/* Arithmetic operations.  This is pretty ugly. */
/* <gt  12 10> --> "true"
   <lt  10 12> --> "true"
   <eq  10 10> --> "true"
   <add 10 10> --> "20"
   <sub 10 10> --> "0"
   <mul 10 10> --> "100"
   <div 12 10> --> "1"
   <mod 12 10> --> "2" */
#define pf_GT	1
#define pf_LT	2
#define pf_EQ	3
#define pf_ADD	4
#define pf_SUB	5
#define pf_MUL	6
#define pf_DIV	7
#define pf_MOD	8

typedef struct { int op; char *name; } OP_ALIST;
static OP_ALIST op_alist[] = {
  { pf_GT, "GT" },
  { pf_LT, "LT" },
  { pf_EQ, "EQ" },
  { pf_ADD, "ADD" },
  { pf_SUB, "SUB" },
  { pf_MUL, "MUL" },
  { pf_DIV, "DIV" },
  { pf_MOD, "MOD" },

  { 0, (char *)NULL }
};

static char *
operator_name (int op)
{
  register int i;

  for (i = 0; op_alist[i].name != (char *)NULL; i++)
    if (op == op_alist[i].op)
      return (op_alist[i].name);

  return ("*invalid-op*");
}

#if !defined (macintosh)
/* Fucking SunOS doesn't declare this, result is assumed to be INT. */
extern double strtod (const char *, char **);
#endif

static char *
arithmetic_operate (int op, char *arg1, char *arg2)
{
  double val1 = arg1 ? strtod (arg1, (char **)NULL) : 0.0;
  double val2 = arg2 ? strtod (arg2, (char **)NULL) : 0.0;
  static char resbuf[60];
  static int orig_mhtml_decimal_notify = 0;
  char *result = resbuf;

  result[0] = '\0';

  switch (op)
    {
    case pf_GT:
      if (val1 > val2) result = "true";
      break;

    case pf_LT:
      if (val1 < val2) result = "true";
      break;

    case pf_EQ:
      if (!arg1) arg1 = "";
      if (!arg2) arg2 = "";

      if (number_p (arg1) && number_p (arg2))
	{
	  if (val1 == val2)
	    result = "true";
	}
      else
	{
	  if (strcasecmp (arg1, arg2) == 0)
	    result = "true";
	}
      break;

    default:
      {
	double arith_result = 0.0;
	int dot_present = ((arg1 ? (strchr (arg1, '.') != (char *)NULL) : 0) ||
			   (arg2 ? (strchr (arg2, '.') != (char *)NULL) : 0));
    
	switch (op)
	  {
	  case pf_ADD:
	    arith_result = val1 + val2;
	    break;

	  case pf_SUB:
	    arith_result = val1 - val2;
	    break;

	  case pf_MUL:
	    arith_result = val1 * val2;
	    break;

	  case pf_DIV:
	    arith_result = val2 ? val1 / val2 : 0.0;
	    break;

	  case pf_MOD:
	    arith_result = val2 ? (double)((int)val1 % (int)val2) : 0.0;
	    break;
	  }

	if (mhtml_decimal_notify != orig_mhtml_decimal_notify)
	  {
	    orig_mhtml_decimal_notify = mhtml_decimal_notify;

	    if (mhtml_decimal_notify)
	      {
		char *temp = pagefunc_get_variable ("mhtml::decimal-places");

		if (temp)
		  mhtml_decimal_places = atoi (temp);
		else
		  mhtml_decimal_places = 0;
	      }
	    else
	      mhtml_decimal_places = 2;
	  }

	if (mhtml_decimal_notify)
	  sprintf (result, "%.*f", mhtml_decimal_places, arith_result);
	else if (!dot_present /* || (arith_result == (int)arith_result) */)
	  sprintf (result, "%ld", (long int)arith_result);
	else
	  sprintf (result, "%.*f", mhtml_decimal_places, arith_result);
      }
    }

  return (result);
}

#define relational_op(op) ((op == pf_GT) || (op == pf_LT) || (op == pf_EQ))

static void
arithmetic_process (int op, PFunArgs)
{
  register int i, tsize = 0;
  int arg_index = 0;
  char *arg;
  char *value_1 = (char *)NULL;
  char *value_2 = (char *)NULL;
  int value_size = 0;
  char *result = (char *)NULL;

  if (!value_1) value_1 = (char *)xmalloc (value_size = 50);
  if (!value_2) value_2 = (char *)xmalloc (value_size = 50);

  value_1[0] = '\0';
  value_2[0] = '\0';

  /* Our arithmetic processing commands allow any number of arguments.
     The first two are handled in the standard binary fashion, and
     then sucessive arguments are operated on with the running result.
     Each argument may be a numeric value, an expression, or a symbol
     which should resolve to a numeric value. */
  while ((arg = get_positional_arg (vars, arg_index)) != (char *)NULL)
    {
      char *value_pointer = arg;
      char *temp = (char *)NULL;

      arg_index++;

      /* Is this argument already a number? */
      if (!number_p (arg))
	{
	  /* No, so make it one. */
	  temp = mhtml_evaluate_string (arg);
	  value_pointer = temp;

	  /* Did it change as a result of evaluation? */
	  if (strcmp (arg, temp) == 0)
	    {
	      /* No, so try looking it up as a variable name.
		 Only do this when the operation is not pf_EQ. */
	      /* if (op != pf_EQ) */
		{
		  for (i = 0; whitespace (arg[i]); i++);
		  value_pointer = pagefunc_get_variable (arg + i);
		}
	    }
	}

      /* Now, value_pointer contains the numeric argument.
	 Copy that value to either value_1 or _value_2 depending on
	 whether we already have a value_1. */
      if (value_pointer == (char *)NULL) value_pointer = "0";

      /* Make sure there is enough room. */
      tsize = strlen (value_pointer);
      if (tsize + 2 > value_size)
	{
	  value_1 = (char *)xrealloc (value_1, 1 + (value_size += tsize));
	  value_2 = (char *)xrealloc (value_2, value_size);
	}

      if (arg_index < 2)
	strcpy (value_1, value_pointer);
      else
	{
	  strcpy (value_2, value_pointer);

	  /* We have two values.  Call the function and get the result. */
	  result = arithmetic_operate (op, value_1, value_2);

	  /* Now, if the operator was relational, value_1 gets value_2,
	     iff result has a value. Otherwise, value_1 gets result. */
	  if (relational_op (op))
	    {
	      if (empty_string_p (result))
		{
		  xfree (temp);
		  break;
		}
	      else
		strcpy (value_1, value_2);
	    }
	  else
	    {
	      tsize = strlen (result);
	      if (tsize + 2 > value_size)
		{
		  value_1 = (char *)xrealloc
		    (value_1, 1 + (value_size += tsize));
		  value_2 = (char *)xrealloc (value_2, value_size + 1);
		}
	      strcpy (value_1, result);
	    }
	}

      /* Free possibly allocated variables. */
      xfree (temp);
    }

  /* If there weren't two arguments, then it might be right to complain. */
  if (arg_index < 2)
    if (arg_index < 1)
      page_debug ("<%s ?> seen with no args", operator_name (op));
    else
      page_debug ("<%s %s ?> seen with one arg", operator_name (op), value_1);

  xfree (value_1);
  xfree (value_2);

  if (!empty_string_p (result))
    {
      bprintf_insert (page, start, "%s", result);
      *newstart += strlen (result);
    }
}

DOC_SECTION (RELATIONAL-OPERATORS)
DEFUN (pf_gt, arg1 arg2 &rest more-args...,
"Returns \"true\" if the numeric value of <var arg1> is greater than
the numeric value of <var arg2> (which is greater than <var
more-args>.  Just as with the arithmetic functions (<secref
Arithmetic-Operators>), the arguments may be the names of
variables containing numeric values, and not just the values
themselves.  In that case, <code>gt</code> performs an implicit call
to <example code><get-var ...></example> in order to get the value to
operate on.

Examples:
<example>
<gt 4 3>            --> true
<set-var x=3.4 y=3.5>
<gt x y>            -->
<gt y x>            --> true
<gt <get-var y> x>  --> true

<defun between? item high low>
  <gt high item low>
</defun>

<between? 7 8 6>    --> true
</example>")
{
  arithmetic_process (pf_GT, PassPFunArgs);
}

DEFUN (pf_lt, arg1 arg2 &rest more-args,
"Returns \"true\" if the numeric value of <var arg1> is less than
the numeric value of <var arg2> (which is less than <var more-args>.
Just as with the arithmetic functions (<secref Arithmetic-Operators>),
<var arg1> and <var arg2> may be the names of variables containing
numeric values, and not just the values themselves.  In that case,
<code>lt</code> performs an implicit call to <example code><get-var ...></example> in order to get the value to operate on.

Examples:
<example>
<lt 3 4>            --> true
<lt 4 3>            --> 
<set-var x=3.4 y=3.5>
<lt x y>            --> true
<lt y x>            --> 
<lt x <get-var y>>  --> true
<lt 4 5 6>          --> true
</example>")
{
  arithmetic_process (pf_LT, PassPFunArgs);
}

DEFUN (pf_eq, arg1 arg2,
"Returns \"true\" if the numeric value of <var arg1> is exactly equal to
the numeric value of <var arg2>.  Just as with the arithmetic
functions (<secref Arithmetic-Operators>), the arguments
may be the names of variables containing numeric values, and not just
the values themselves.  In that case, <code>eq</code> performs an
implicit call to <example code><get-var ...></example> in order to get
the value to operate on.

Examples:
<example>
<eq 3 4>              --> 
<eq 3 3>              --> true
<eq 3.0 3>            --> true
<set-var x=3.01>
<eq <get-var x> 3.01> --> true
</example>")
{
  arithmetic_process (pf_EQ, PassPFunArgs);
}

DOC_SECTION (ARITHMETIC-OPERATORS)

DEFUN (pf_add, arg1 arg2 &rest more-args,
"Returns the summation of all of the arguments passed.

Examples:
<example>
<add 1 2>        --> 3
<add -1 2 2 1>   --> 4
</example>")
{
  arithmetic_process (pf_ADD, PassPFunArgs);
}

DEFUN (pf_sub, arg1 arg2 &rest more-args,
"Returns the difference of all of the arguments passed.

Examples:
<example>
<sub 2 1>        --> 1
<sub 6 2 1>      --> 3
<sub -1 -2>      --> 1
</example>")
{
  if ((body == (PAGE *)NULL) || (empty_string_p (body->buffer)))
    {
      bprintf_insert (page, start, "<SUB>");
      *newstart += 4;
    }
  else
    {
      arithmetic_process (pf_SUB, PassPFunArgs);
    }
}

DEFUN (pf_mul, arg1 arg2 &rest more-args,
"Returns the product of all of the arguments passed.

Examples:
<example>
<mul 2 2>        --> 4
<mul 2.3 8 3>    --> 55.20
<mul -4 10>      --> -40
</example>")
{
  arithmetic_process (pf_MUL, PassPFunArgs);
}

DEFUN (pf_div, arg1 arg2 &rest more-args,
"Returns the quotient of all of the arguments passed.

Examples:
<example>
<div 100 5>      --> 20
<div 5 100>      --> 0
<div 5.0 100.0>  --> 0.05
<div 100 5 2>    --> 10
</example>")
{
  /* If it only contains assigned attributes, it is meant for the browser,
     not us. */
  if (get_positional_arg (vars, 0) == (char *)NULL)
    {
      bprintf_insert (page, start, "<DIV %s>", mhtml_funargs (vars));
      *newstart += 4;
    }
  else
    {
      arithmetic_process (pf_DIV, PassPFunArgs);
    }
}

DEFUN (pf_mod, arg1 arg2 &rest more-args,
"Returns the remainder of all of the arguments passed.

Examples:
<example>
<mod 100 10>    --> 0
<mod 101 10>    --> 1
<mod 89 9 3>    --> 2
</example>")
{
  arithmetic_process (pf_MOD, PassPFunArgs);
}
#else /* ! MHTML_ARITHMETIC */

void initialize_arithmetic_functions (Package *package) {}

#endif /* MHTML_ARITHMETIC */

