/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif // HAVE_CONFIG_H

#include "p2p.h"
#include "p2p-internal.h"

int
_lw6p2p_tentacle_init (_lw6p2p_tentacle_t * tentacle,
		       _lw6p2p_backends_t * backends,
		       lw6srv_listener_t * listener, const char *local_url,
		       const char *remote_url, const char *real_remote_ip,
		       const char *password, u_int64_t local_id,
		       u_int64_t remote_id, int network_reliability,
		       lw6cnx_recv_callback_t recv_callback_func,
		       void *recv_callback_data)
{
  int ret = 1;
  int i = 0;
  char *repr = NULL;
  lw6sys_url_t *parsed_url = NULL;

  tentacle->backends = backends;
  tentacle->local_url = lw6sys_str_copy (local_url);
  tentacle->remote_url = lw6sys_str_copy (remote_url);
  parsed_url = lw6sys_url_parse (remote_url);
  if (parsed_url)
    {
      /*
       * In theory this could be a long blocking call,
       * in practise, when we're at this stage OOB messages
       * probably have already initialized the DNS cache
       * so response will be fast.
       */
      tentacle->remote_ip = lw6net_dns_gethostbyname (parsed_url->host);
      tentacle->remote_port = parsed_url->port;
      if (real_remote_ip)
	{
	  if (tentacle->remote_ip)
	    {
	      if (lw6sys_str_is_same (tentacle->remote_ip, real_remote_ip))
		{
		  lw6sys_log (LW6SYS_LOG_DEBUG,
			      _x_
			      ("OK, public URL \"%s\" for \"%s:%d\" is fine"),
			      remote_url, tentacle->remote_ip,
			      tentacle->remote_port);
		  tentacle->dns_ok = 1;
		}
	      else
		{
		  lw6sys_log (LW6SYS_LOG_WARNING,
			      _x_
			      ("URL \"%s\" appears to be associated to \"%s:%d\" but DNS reports it to be associated to \"%s:%d\", using \"%s:%d\""),
			      remote_url, real_remote_ip,
			      tentacle->remote_port, tentacle->remote_ip,
			      tentacle->remote_port, real_remote_ip,
			      tentacle->remote_port);
		  LW6SYS_FREE (tentacle->remote_ip);
		  tentacle->remote_ip = lw6sys_str_copy (real_remote_ip);
		}
	    }
	  else
	    {
	      lw6sys_log (LW6SYS_LOG_INFO,
			  _x_
			  ("unable to get IP from DNS for \"%s\", using \"%s:%d\" instead"),
			  remote_url, real_remote_ip, tentacle->remote_port);
	      tentacle->remote_ip = lw6sys_str_copy (real_remote_ip);
	    }
	}
      else
	{
	  if (tentacle->remote_ip)
	    {
	      lw6sys_log (LW6SYS_LOG_DEBUG,
			  _x_
			  ("no \"real\" IP passed, using default \"%s:%d\""),
			  tentacle->remote_ip, tentacle->remote_port);
	      tentacle->dns_ok = 1;
	    }
	  else
	    {
	      lw6sys_log (LW6SYS_LOG_WARNING,
			  _x_ ("unable to find host \"%s\""),
			  parsed_url->host);
	    }
	}
      lw6sys_url_free (parsed_url);
    }
  if (password && strlen (password) > 0)
    {
      tentacle->password = lw6sys_str_copy (password);
    }
  else
    {
      tentacle->password = lw6sys_str_copy ("");
    }
  tentacle->local_id_int = local_id;
  tentacle->local_id_str = lw6sys_id_ltoa (local_id);
  tentacle->remote_id_int = remote_id;
  tentacle->remote_id_str = lw6sys_id_ltoa (remote_id);

  if (tentacle->local_url && tentacle->remote_url && tentacle->remote_ip
      && tentacle->password && tentacle->local_id_str
      && tentacle->remote_id_str)
    {
      tentacle->nb_cli_connections = backends->nb_cli_backends;
      if (tentacle->nb_cli_connections > 0)
	{
	  tentacle->cli_connections =
	    (lw6cnx_connection_t **)
	    LW6SYS_CALLOC (tentacle->nb_cli_connections *
			   sizeof (lw6cnx_connection_t *));
	  if (tentacle->cli_connections)
	    {
	      for (i = 0; i < tentacle->nb_cli_connections; ++i)
		{
		  tentacle->cli_connections[i] =
		    lw6cli_open (tentacle->backends->cli_backends[i],
				 local_url, remote_url, tentacle->remote_ip,
				 tentacle->remote_port,
				 tentacle->password,
				 tentacle->local_id_int,
				 tentacle->remote_id_int, tentacle->dns_ok,
				 network_reliability, recv_callback_func,
				 recv_callback_data);
		  if (tentacle->cli_connections[i])
		    {
		      repr =
			lw6cli_repr (tentacle->backends->cli_backends[i],
				     tentacle->cli_connections[i]);
		      if (repr)
			{
			  lw6sys_log (LW6SYS_LOG_DEBUG,
				      _x_ ("connection \"%s\" opened"), repr);
			  LW6SYS_FREE (repr);
			}
		    }
		  else
		    {
		      lw6sys_log (LW6SYS_LOG_WARNING,
				  _x_
				  ("unable to create connection %d to connect on \"%s\""),
				  i, tentacle->remote_url);
		      ret = 0;
		    }
		}
	    }
	  else
	    {
	      ret = 0;
	    }
	}
      tentacle->nb_srv_connections = backends->nb_srv_backends;
      if (tentacle->nb_srv_connections > 0)
	{
	  tentacle->srv_connections =
	    (lw6cnx_connection_t **)
	    LW6SYS_CALLOC (tentacle->nb_srv_connections *
			   sizeof (lw6cnx_connection_t *));
	  if (tentacle->srv_connections)
	    {
	      for (i = 0; i < tentacle->nb_srv_connections; ++i)
		{
		  tentacle->srv_connections[i] =
		    lw6srv_open (tentacle->backends->srv_backends[i],
				 listener,
				 local_url, remote_url, tentacle->remote_ip,
				 tentacle->remote_port,
				 tentacle->password,
				 tentacle->local_id_int,
				 tentacle->remote_id_int, tentacle->dns_ok,
				 network_reliability, recv_callback_func,
				 recv_callback_data);
		  if (tentacle->srv_connections[i])
		    {
		      repr =
			lw6srv_repr (tentacle->backends->srv_backends[i],
				     tentacle->srv_connections[i]);
		      if (repr)
			{
			  lw6sys_log (LW6SYS_LOG_DEBUG,
				      _x_ ("connection \"%s\" opened"), repr);
			  LW6SYS_FREE (repr);
			}
		    }
		  else
		    {
		      lw6sys_log (LW6SYS_LOG_WARNING,
				  _x_
				  ("unable to create connection %d to connect on \"%s\""),
				  i, tentacle->remote_url);
		      ret = 0;
		    }
		}
	    }
	  else
	    {
	      ret = 0;
	    }
	}
    }
  else
    {
      ret = 0;
    }

  return ret;
}

void
_lw6p2p_tentacle_clear (_lw6p2p_tentacle_t * tentacle)
{
  int i = 0;

  if (tentacle->unsent_queue)
    {
      lw6sys_list_free (tentacle->unsent_queue);
    }
  if (tentacle->nb_srv_connections > 0 && tentacle->srv_connections)
    {
      for (i = 0; i < tentacle->nb_srv_connections; ++i)
	{
	  if (tentacle->srv_connections[i])
	    {
	      lw6srv_close (tentacle->backends->srv_backends[i],
			    tentacle->srv_connections[i]);
	    }
	}
      LW6SYS_FREE (tentacle->srv_connections);
    }
  if (tentacle->nb_cli_connections > 0 && tentacle->cli_connections)
    {
      for (i = 0; i < tentacle->nb_cli_connections; ++i)
	{
	  if (tentacle->cli_connections[i])
	    {
	      lw6cli_close (tentacle->backends->cli_backends[i],
			    tentacle->cli_connections[i]);
	    }
	}
      LW6SYS_FREE (tentacle->cli_connections);
    }
  if (tentacle->local_url)
    {
      LW6SYS_FREE (tentacle->local_url);
    }
  if (tentacle->remote_url)
    {
      LW6SYS_FREE (tentacle->remote_url);
    }
  if (tentacle->remote_ip)
    {
      LW6SYS_FREE (tentacle->remote_ip);
    }
  if (tentacle->password)
    {
      LW6SYS_FREE (tentacle->password);
    }
  if (tentacle->local_id_str)
    {
      LW6SYS_FREE (tentacle->local_id_str);
    }
  if (tentacle->remote_id_str)
    {
      LW6SYS_FREE (tentacle->remote_id_str);
    }
  memset (tentacle, 0, sizeof (_lw6p2p_tentacle_t));
}

int
_lw6p2p_tentacle_enabled (_lw6p2p_tentacle_t * tentacle)
{
  int ret = 0;

  ret = (tentacle->remote_id_int != 0);

  return ret;
}

void
_lw6p2p_tentacle_poll (_lw6p2p_tentacle_t * tentacle,
		       lw6nod_info_t * node_info,
		       lw6cnx_ticket_table_t * ticket_table,
		       const _lw6p2p_consts_t * consts, int serial)
{
  int i;
  char *msg;
  int64_t now;
  lw6cnx_connection_t *cnx = NULL;
  u_int32_t ticket_sig = 0;
  _lw6p2p_queue_item_t *queue_item = NULL;
  lw6sys_list_t *unsent_queue = NULL;
  int queue_i = 0;

  /*
   * IMPORTANT -> this function is called in unlock mode, it can
   * be in conflict with code from the recv callback. Normally this
   * should not be a problem.
   */
  now = lw6sys_get_timestamp ();
  if (!tentacle->hello_sent)
    {
      msg = lw6msg_cmd_generate_hello (node_info);
      if (msg)
	{
	  for (i = 0; i < tentacle->nb_cli_connections; ++i)
	    {
	      cnx = tentacle->cli_connections[i];
	      ticket_sig =
		lw6msg_ticket_calc_sig (lw6cnx_ticket_table_get_send
					(ticket_table, cnx->remote_id_str),
					cnx->local_id_int, cnx->remote_id_int,
					msg);
	      if (lw6cli_send
		  (tentacle->backends->cli_backends[i], cnx, now, ticket_sig,
		   ticket_sig, cnx->local_id_int, cnx->remote_id_int, msg))
		{
		  tentacle->hello_sent = 1;
		}
	    }
	  LW6SYS_FREE (msg);
	}
    }
  for (i = 0; i < tentacle->nb_cli_connections; ++i)
    {
      cnx = tentacle->cli_connections[i];
      if (lw6cnx_connection_should_send_foo (cnx, now))
	{
	  lw6cnx_connection_init_foo_bar_key (cnx, now, consts->foo_delay);
	  lw6sys_log (LW6SYS_LOG_DEBUG,
		      _x_ ("preparing foo with foo_bar_key=%08x"),
		      cnx->foo_bar_key);
	  msg = lw6msg_cmd_generate_foo (node_info, cnx->foo_bar_key, serial);
	  if (msg)
	    {
	      ticket_sig =
		lw6msg_ticket_calc_sig (lw6cnx_ticket_table_get_send
					(ticket_table, cnx->remote_id_str),
					cnx->local_id_int, cnx->remote_id_int,
					msg);
	      lw6cli_send (tentacle->backends->cli_backends[i], cnx, now,
			   ticket_sig, ticket_sig, cnx->local_id_int,
			   cnx->remote_id_int, msg);
	      LW6SYS_FREE (msg);
	    }
	  if (!lw6cnx_ticket_table_was_recv_exchanged
	      (ticket_table, cnx->remote_id_str))
	    {
	      msg =
		lw6msg_cmd_generate_ticket (node_info,
					    lw6cnx_ticket_table_get_recv
					    (ticket_table,
					     cnx->remote_id_str));
	      if (msg)
		{
		  ticket_sig =
		    lw6msg_ticket_calc_sig (lw6cnx_ticket_table_get_send
					    (ticket_table,
					     cnx->remote_id_str),
					    cnx->local_id_int,
					    cnx->remote_id_int, msg);
		  lw6cli_send (tentacle->backends->cli_backends[i], cnx, now,
			       ticket_sig, ticket_sig, cnx->local_id_int,
			       cnx->remote_id_int, msg);
		  LW6SYS_FREE (msg);
		}
	    }
	}
      lw6cli_poll (tentacle->backends->cli_backends[i], cnx);
    }
  for (i = 0; i < tentacle->nb_srv_connections; ++i)
    {
      cnx = tentacle->srv_connections[i];
      if (lw6cnx_connection_should_send_foo (cnx, now))
	{
	  lw6cnx_connection_init_foo_bar_key (cnx, now, consts->foo_delay);
	  lw6sys_log (LW6SYS_LOG_DEBUG,
		      _x_ ("preparing foo with foo_bar_key=%08x"),
		      cnx->foo_bar_key);
	  msg = lw6msg_cmd_generate_foo (node_info, cnx->foo_bar_key, serial);
	  if (msg)
	    {
	      ticket_sig =
		lw6msg_ticket_calc_sig (lw6cnx_ticket_table_get_send
					(ticket_table, cnx->remote_id_str),
					cnx->local_id_int, cnx->remote_id_int,
					msg);
	      lw6srv_send (tentacle->backends->srv_backends[i], cnx, now,
			   ticket_sig, ticket_sig, cnx->local_id_int,
			   cnx->remote_id_int, msg);
	      LW6SYS_FREE (msg);
	    }
	  if (!lw6cnx_ticket_table_was_recv_exchanged
	      (ticket_table, cnx->remote_id_str))
	    {
	      msg =
		lw6msg_cmd_generate_ticket (node_info,
					    lw6cnx_ticket_table_get_recv
					    (ticket_table,
					     cnx->remote_id_str));
	      if (msg)
		{
		  ticket_sig =
		    lw6msg_ticket_calc_sig (lw6cnx_ticket_table_get_send
					    (ticket_table,
					     cnx->remote_id_str),
					    cnx->local_id_int,
					    cnx->remote_id_int, msg);
		  lw6srv_send (tentacle->backends->srv_backends[i], cnx, now,
			       ticket_sig, ticket_sig, cnx->local_id_int,
			       cnx->remote_id_int, msg);
		  LW6SYS_FREE (msg);
		}
	    }
	}
      lw6srv_poll (tentacle->backends->srv_backends[i], cnx);
    }

  if (tentacle->unsent_queue)
    {
      lw6sys_log (LW6SYS_LOG_INFO, _x_ ("flushing unsent_queue"));
      /*
       * Empty the queue associated to the tentacle, push data
       * in our own queue, and flush it with send.
       */
      unsent_queue = tentacle->unsent_queue;
      tentacle->unsent_queue = NULL;

      while ((queue_item =
	      (_lw6p2p_queue_item_t *) lw6sys_lifo_pop (&unsent_queue)) !=
	     NULL)
	{
	  /*
	   * Some of the data we use here could be in some cases
	   * deduced from context but we prefer to use the
	   * data stored in the queue_item, in case we want
	   * to resend forwareded messages, you never know
	   */
	  /*
	   * Send them in reliable mode, if it's in the queue, it means
	   * we're probably sending lots of data, we'd like TCP and/or
	   * other reliable protocol to help us to the tiring job of
	   * checking everything...
	   */
	  _lw6p2p_tentacle_send_best (tentacle,
				      now,
				      ticket_table,
				      queue_item->logical_ticket_sig,
				      queue_item->logical_from_id,
				      queue_item->logical_to_id,
				      queue_item->msg, 1);
	  _lw6p2p_queue_item_free (queue_item);
	  ++queue_i;
	}
      lw6sys_log (LW6SYS_LOG_INFO,
		  _x_ ("there were %d messages in unsent_queue"), queue_i);
    }
}

int
_lw6p2p_tentacle_send_best (_lw6p2p_tentacle_t * tentacle,
			    int64_t now,
			    lw6cnx_ticket_table_t * ticket_table,
			    u_int32_t logical_ticket_sig,
			    u_int64_t logical_from_id,
			    u_int64_t logical_to_id, const char *msg,
			    int reliable)
{
  int ret = 0;
  lw6cnx_connection_t *cnx = NULL;
  u_int32_t physical_ticket_sig = 0;
  int i = 0;
  int ping_msec = 0;
  _lw6p2p_queue_item_t *queue_item = NULL;

  physical_ticket_sig =
    lw6msg_ticket_calc_sig (lw6cnx_ticket_table_get_send
			    (ticket_table, tentacle->remote_id_str),
			    tentacle->local_id_int,
			    tentacle->remote_id_int, msg);

  cnx =
    _lw6p2p_tentacle_find_connection_with_lowest_ping (tentacle, reliable);
  if (cnx)
    {
      /*
       * We store the cnx ping into a local variable since it might
       * be altered if the send fails, and we want the original
       * ping to be used/shown in logs.
       */
      ping_msec = cnx->ping_msec;
      if (ping_msec > 0)
	{
	  lw6sys_log (LW6SYS_LOG_DEBUG,
		      _x_ ("send of \"%s\" on fastest connection"), msg);
	  /*
	   * Now we have the cnx, we must figure out its type (cnx/srv)
	   * and index to fire the right code on it.
	   */
	  for (i = 0; i < tentacle->nb_cli_connections; ++i)
	    {
	      if (cnx == tentacle->cli_connections[i])
		{
		  lw6sys_log (LW6SYS_LOG_DEBUG,
			      _x_
			      ("found fastest connection to \"%s\", it's a client connection, backend name=\"%s\""),
			      tentacle->remote_url,
			      tentacle->backends->cli_backends[i]->name);
		  ret =
		    lw6cli_send (tentacle->backends->cli_backends[i], cnx,
				 now, physical_ticket_sig, logical_ticket_sig,
				 logical_from_id, logical_to_id, msg) || ret;
		}
	    }
	  for (i = 0; i < tentacle->nb_srv_connections; ++i)
	    {
	      if (cnx == tentacle->srv_connections[i])
		{
		  lw6sys_log (LW6SYS_LOG_DEBUG,
			      _x_
			      ("found fastest connection to \"%s\", it's a server connection, backend name=\"%s\""),
			      tentacle->remote_url,
			      tentacle->backends->srv_backends[i]->name);
		  ret =
		    lw6srv_send (tentacle->backends->srv_backends[i], cnx,
				 now, physical_ticket_sig, logical_ticket_sig,
				 logical_from_id, logical_to_id, msg) || ret;
		}
	    }
	}
      else
	{
	  /*
	   * OK, we did not really find a "best connection" since it pretends
	   * to have a zero ping, and ping zero is just a plain imaginary stuff,
	   * and actually, we always force real pings to be at least one.
	   * Well, anyway, just means we did not really find it, so meanwhile,
	   * until we get a real "best" one, we go redundant.
	   */
	  lw6sys_log (LW6SYS_LOG_INFO,
		      _x_
		      ("couldn't really find a \"best\" connection for now, fallback on redundant mode"));
	  ret =
	    _lw6p2p_tentacle_send_redundant (tentacle, now, ticket_table,
					     logical_ticket_sig,
					     logical_from_id, logical_to_id,
					     msg);
	}
    }

  if (ret)
    {
      if (cnx->properties.backend_id)
	{
	  lw6sys_log (LW6SYS_LOG_DEBUG,
		      _x_
		      ("successfully send on connexion with backend_id=\"%s\""),
		      cnx->properties.backend_id);
	}
    }
  else
    {
      if (cnx && cnx->properties.backend_id)
	{
	  lw6sys_log (LW6SYS_LOG_INFO,
		      _x_
		      ("failed send on \"best\" connexion with backend_id=\"%s\" current ping=%d msec"),
		      cnx->properties.backend_id, ping_msec);
	}

      if (!tentacle->unsent_queue)
	{
	  tentacle->unsent_queue =
	    lw6sys_list_new ((lw6sys_free_func_t) _lw6p2p_queue_item_free);
	}
      if (tentacle->unsent_queue)
	{
	  queue_item =
	    _lw6p2p_queue_item_new (logical_ticket_sig, logical_from_id,
				    logical_to_id, msg);
	  if (queue_item)
	    {
	      lw6sys_log (LW6SYS_LOG_DEBUG,
			  _x_
			  ("message \"%s\" not sent, pushing it to unsent_queue"),
			  queue_item->msg);
	      /*
	       * Use a lifo, a fifo would be cleaner but is more expensive
	       * on big lists, this is a fallback action anyway, packets
	       * will probably be in the wrong order anyway and we are
	       * very likely sending a big hudge message splitted into
	       * many atoms so order is irrelevant.
	       */
	      lw6sys_lifo_push (&(tentacle->unsent_queue), queue_item);
	    }
	}
    }

  return ret;
}

int
_lw6p2p_tentacle_send_redundant (_lw6p2p_tentacle_t * tentacle,
				 int64_t now,
				 lw6cnx_ticket_table_t * ticket_table,
				 u_int32_t logical_ticket_sig,
				 u_int64_t logical_from_id,
				 u_int64_t logical_to_id, const char *msg)
{
  int ret = 0;
  lw6cnx_connection_t *cnx = NULL;
  int i = 0;
  u_int32_t physical_ticket_sig = 0;

  lw6sys_log (LW6SYS_LOG_DEBUG, _x_ ("redundant send of \"%s\""), msg);
  physical_ticket_sig =
    lw6msg_ticket_calc_sig (lw6cnx_ticket_table_get_send
			    (ticket_table, tentacle->remote_id_str),
			    tentacle->local_id_int, tentacle->remote_id_int,
			    msg);

  for (i = 0; i < tentacle->nb_cli_connections; ++i)
    {
      cnx = tentacle->cli_connections[i];
      ret = lw6cli_send (tentacle->backends->cli_backends[i], cnx, now,
			 physical_ticket_sig, logical_ticket_sig,
			 logical_from_id, logical_to_id, msg) || ret;
    }

  for (i = 0; i < tentacle->nb_srv_connections; ++i)
    {
      cnx = tentacle->srv_connections[i];
      ret = lw6srv_send (tentacle->backends->srv_backends[i], cnx, now,
			 physical_ticket_sig, logical_ticket_sig,
			 logical_from_id, logical_to_id, msg) || ret;
    }

  return ret;
}

lw6cnx_connection_t *
_lw6p2p_tentacle_find_connection_with_foo_bar_key (_lw6p2p_tentacle_t *
						   tentacle,
						   u_int32_t foo_bar_key)
{
  lw6cnx_connection_t *ret = NULL;
  lw6cnx_connection_t *cnx = NULL;
  int i = 0;

  for (i = 0; i < tentacle->nb_cli_connections; ++i)
    {
      cnx = tentacle->cli_connections[i];
      if (cnx->foo_bar_key == foo_bar_key)
	{
	  ret = cnx;
	}
    }

  for (i = 0; i < tentacle->nb_srv_connections; ++i)
    {
      cnx = tentacle->srv_connections[i];
      if (cnx->foo_bar_key == foo_bar_key)
	{
	  ret = cnx;
	}
    }

  return ret;
}

lw6cnx_connection_t *
_lw6p2p_tentacle_find_connection_with_lowest_ping (_lw6p2p_tentacle_t *
						   tentacle, int reliable)
{
  lw6cnx_connection_t *ret = NULL;
  lw6cnx_connection_t *cnx = NULL;
  int i = 0;
  int best_ping_msec = LW6CNX_WORST_PING_MSEC;

  for (i = 0; i < tentacle->nb_cli_connections; ++i)
    {
      cnx = tentacle->cli_connections[i];
      if (!ret)
	{
	  if (cnx->properties.reliable)
	    {
	      best_ping_msec = cnx->ping_msec;
	      ret = cnx;
	    }
	}
      if (cnx->ping_msec > 0 && cnx->ping_msec < best_ping_msec
	  && (cnx->properties.reliable || !reliable))
	{
	  best_ping_msec = cnx->ping_msec;
	  ret = cnx;
	}
    }

  for (i = 0; i < tentacle->nb_srv_connections; ++i)
    {
      cnx = tentacle->srv_connections[i];
      if (!ret)
	{
	  if (cnx->properties.reliable)
	    {
	      best_ping_msec = cnx->ping_msec;
	      ret = cnx;
	    }
	}
      if (cnx->ping_msec > 0 && cnx->ping_msec < best_ping_msec
	  && (cnx->properties.reliable || !reliable))
	{
	  best_ping_msec = cnx->ping_msec;
	  ret = cnx;
	}
    }

  /*
   * If we couldn't find a reliable connexion, try to fallback
   * on a non-reliable one...
   */
  if (reliable && !ret)
    {
      ret = _lw6p2p_tentacle_find_connection_with_lowest_ping (tentacle, 0);
    }

  if (ret)
    {
      lw6sys_log (LW6SYS_LOG_DEBUG,
		  _x_ ("best ping is %d with backend \"%s\" reliable=%d"),
		  ret->ping_msec, ret->properties.backend_id,
		  ret->properties.reliable);
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_DEBUG,
		  _x_
		  ("unable to find any suitable connection, no reliable connection at all and no connection marked as ping better than %d"),
		  LW6CNX_WORST_PING_MSEC);
    }

  return ret;
}
