/*  gnutrition - a nutrition and diet analysis program.
 *  Copyright (C) 2000 Edgar Denny (e.denny@ic.ac.uk)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>
#include <glade/glade.h>
#include <gnome-xml/parser.h>
#include <gnome-xml/tree.h>
#include "text_util.h"
#include "food.h"
#include "recipe.h"
#include "nutrgoal_io.h"

gboolean
gnutr_write_nutr_goal( Nutrition_t *nutr)
{
	xmlDocPtr doc;
	xmlNodePtr cur;
	gint i;
	gchar *home = g_strdup( g_get_home_dir());
	gchar *dir = g_strconcat( home, "/.gnutrition", NULL);
	gchar *fn = g_strconcat( dir, "/nutr_goal.xml", NULL);

	/* make sure directory exists, if not, create it. */
	gnutr_dir_test( dir);

	/* create a new xml document with version 1.0 */
	doc = xmlNewDoc("1.0");

	/* create a new root node */
	doc->root = xmlNewDocNode( doc, NULL, "Gnutrition", NULL);

 	cur = xmlNewChild( doc->root, NULL, "NutrientGoal", NULL);

	for( i=0; i<NO_NUTR; i++)
	{
		xmlSetProp( cur, g_strconcat( "val", itoa(i), NULL),
				ftoa( nutr->goal[i]));
	}

	if( xmlSaveFile( fn, doc) == -1)
	{
		xmlFreeDoc( doc);
		g_free( fn);
		g_free( dir);
		g_free( home);
		return FALSE;
	}

	xmlFreeDoc( doc);
	g_free( fn);
	g_free( dir);
	g_free( home);
	return TRUE;
}

gboolean 
gnutr_read_nutr_goal( Nutrition_t **nutr) 
{
	xmlDocPtr doc;
	xmlNodePtr cur;
	gint i;
	gchar *home = g_strdup( g_get_home_dir());
	gchar *fn = g_strconcat( home, "/.gnutrition/nutr_goal.xml", NULL);

	/* read personal nutrient goal, else read the default values. */
	if ( gnutr_file_test( fn))
	{
		doc = xmlParseFile( fn);
	}
	else
	{
		doc = xmlParseFile( GNUTRITION_GLADEDIR "/xml/nutr_goal.xml");
	}

	/* make a series of checks. */
	if ( doc == NULL) return FALSE;

	cur = doc->root;
	if ( cur == NULL)
	{
		g_warning( "empty document.");
		xmlFreeDoc( doc);
		g_free( home);
		g_free( fn);
		return FALSE;
	}

	if ( strcmp( cur->name, "Gnutrition") != 0 )
	{
		g_warning( "document of the wrong type.");
		xmlFreeDoc( doc);
		g_free( home);
		g_free( fn);
		return FALSE;
	}

	cur = cur->childs;
	if ( strcmp( cur->name, "NutrientGoal") != 0 )
	{
		g_warning( "document of the wrong type.");
		xmlFreeDoc( doc);
		g_free( home);
		g_free( fn);
		return FALSE;
	}

	/* read in data. */
	for ( i=0; i<NO_NUTR; i++)
	{
		(*nutr)->goal[i] = atof( xmlGetProp( cur, 
						   g_strconcat( "val", itoa(i), NULL)));
	}

	xmlFreeDoc( doc);
	g_free( home);
	g_free( fn);
	return TRUE;
}
