/**
 * Copyright (C) 1999, 2000, 2001, 2002  Free Software Foundation, Inc.
 *
 * This file is part of GNU gengetopt
 *
 * GNU gengetopt is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * GNU gengetopt is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 * Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with gengetopt; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <string>

#include <fstream>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

extern "C"
{
#include "argsdef.h"
#include "global_opts.h"
};

#include "ggo_options.h"

#include "gm.h"
#include "my_sstream.h"

#include "groups.h"
#include "string_opt_arg.h"
#include "string_opt_arg_no_free.h"
#include "int_opt_arg.h"
#include "long_opt_arg.h"
#include "long_long_opt_arg.h"
#include "float_opt_arg.h"
#include "flag_opt_arg.h"
#include "required_option.h"
#include "generic_option.h"
#include "group_option.h"
#include "group_counter.h"
#include "generic_option_group.h"
#include "handle_help.h"
#include "handle_version.h"
#include "no_short_option.h"
#include "print_help_string.h"
#include "multiple_opt_list.h"
#include "multiple_opt_struct.h"
#include "multiple_option.h"
#include "multiple_option_no_short.h"
#include "multiple_option_new_list_node.h"
#include "multiple_fill_array.h"
#include "multiple_fill_array_default.h"
#include "free_string.h"
#include "free_multiple_string.h"
#include "free_inputs_string.h"
#include "reset_group.h"
#include "exit_failure.h"
#include "update_given.h"
#include "gm_utils.h"
#include "fileutils.h"

#ifndef HAVE_STRDUP
extern "C" char *strdup (const char *s) ;
#endif

#define MAX_STARTING_COLUMN 32

using std::endl;

extern char * gengetopt_package;
extern char * gengetopt_version;
extern char * gengetopt_purpose;
extern char * gengetopt_input_filename;

extern groups_collection_t gengetopt_groups;

static void _generate_option_arg(ostream &stream,
                                 unsigned int indent,
                                 struct gengetopt_option * opt);

static void
generate_help_desc_print(ostream &stream,
                         unsigned int desc_column,
                         const char *descript, const char *defval);

CmdlineParserCreator::CmdlineParserCreator (char *function_name,
                                            char *struct_name,
                                            char *unamed_options_,
                                            char *filename_,
                                            char *header_ext, char *c_ext,
                                            bool long_help_,
                                            bool no_handle_help_,
                                            bool no_handle_version_,
                                            bool no_handle_error_,
                                            bool conf_parser_,
                                            bool gen_version,
                                            const string &comment_) :
  filename (filename_),
  args_info_name (struct_name),
  comment (comment_),
  unamed_options (unamed_options_),
  long_help (long_help_), no_handle_help (no_handle_help_),
  no_handle_version (no_handle_version_), no_handle_error (no_handle_error_),
  conf_parser (conf_parser_), gen_gengetopt_version (gen_version),
  tab_indentation (0)
{
  parser_function_name = canonize_names (function_name);
  c_filename = create_filename (filename, c_ext);
  header_filename = create_filename (filename, header_ext);

  // header_gen_class
  const string stripped_header_file_name = strip_path (filename);
  set_header_file_name (stripped_header_file_name);
  if (gen_gengetopt_version)
    header_gen_class::set_generator_version
      ("version " VERSION);
  const string my_ifndefname =
    to_upper (strip_path (stripped_header_file_name));
  set_ifndefname (canonize_names (my_ifndefname.c_str ()));
  header_gen_class::set_parser_name (parser_function_name);
  const string my_package_var_name =
    to_upper (parser_function_name) + "_PACKAGE";
  const string my_version_var_name =
    to_upper (parser_function_name) + "_VERSION";
  header_gen_class::set_package_var_name (my_package_var_name);
  c_source_gen_class::set_package_var_name (my_package_var_name);
  header_gen_class::set_version_var_name (my_version_var_name);
  c_source_gen_class::set_version_var_name (my_version_var_name);
  header_gen_class::set_args_info (args_info_name);
  c_source_gen_class::set_args_info (args_info_name);
  const string uppersand = "\"";

  if (gengetopt_package)
    set_package_var_val
      (uppersand + gengetopt_package + uppersand);
  else
    set_package_var_val ("PACKAGE");

  if (gengetopt_version)
    set_version_var_val
      (uppersand + gengetopt_version + uppersand);
  else
    set_version_var_val ("VERSION");

  header_gen_class::set_generate_config_parser (conf_parser);

  // c_source_gen_class
  set_command_line (comment);
  if (gen_gengetopt_version)
    c_source_gen_class::set_generator_version
      ("version " VERSION);
  c_source_gen_class::set_parser_name (parser_function_name);
  set_source_name (filename);

  if (! no_handle_error)
    {
      ostringstream exit_failure_str;
      exit_failure_gen_class exit_gen;
      exit_gen.set_parser_name (c_source_gen_class::parser_name);

      exit_gen.generate_exit_failure (exit_failure_str);

      set_final_exit (exit_failure_str.str ());
    }
  else
    set_final_exit ("return result;");

  set_gen_exit (no_handle_error ? "return" : "exit");

  set_conf_parser (conf_parser);

  struct gengetopt_option * opt;
  gen_strdup = (unamed_options != 0 || conf_parser);

  if (! gen_strdup)
    {
      foropt
        if (opt->type == ARG_STRING) {
          gen_strdup = true;
          break;
        }
    }

  set_do_generate_strdup(gen_strdup);
  set_multiple_token_functions(has_multiple_options());
  set_multiple_token_vars(has_multiple_options());
  set_handle_unamed(unamed_options);
}

void
CmdlineParserCreator::generateBreak(ostream &stream, unsigned int indent)
{
  string indent_str (indent, ' ');

  stream << endl;
  stream << indent_str;
  stream << "break;";
}

void
CmdlineParserCreator::do_update_arg (struct gengetopt_option *opt,
                                     ostream &stream,
                                     unsigned int indent,
                                     const string argstr = "optarg" )
{
  string indent_str (indent, ' ');
  const int offset = 4;

  // FIXME: do not create all these generators for a single option
  string_opt_arg_gen_class str_gen;
  str_gen.set_optarg (argstr);
  string_opt_arg_no_free_gen_class str_gen_no_free;
  str_gen_no_free.set_optarg (argstr);
  int_opt_arg_gen_class int_gen;
  int_gen.set_optarg (argstr);
  long_opt_arg_gen_class long_gen;
  long_gen.set_optarg (argstr);
  long_long_opt_arg_gen_class long_long_gen;
  long_long_gen.set_optarg (argstr);
  float_opt_arg_gen_class float_gen;
  float_gen.set_optarg (argstr);
  flag_opt_arg_gen_class flag_gen;

  if (opt->multiple)
    {
      int_gen.set_structure (string (opt->var_arg) + "_new");
      str_gen_no_free.set_structure (string (opt->var_arg) + "_new");
      long_gen.set_structure (string (opt->var_arg) + "_new");
      long_long_gen.set_structure (string (opt->var_arg) + "_new");
      float_gen.set_structure (string (opt->var_arg) + "_new");
    }
  else
    {
      int_gen.set_structure (ARGS_STRUCT);
      str_gen.set_structure (ARGS_STRUCT);
      long_gen.set_structure (ARGS_STRUCT);
      long_long_gen.set_structure (ARGS_STRUCT);
      float_gen.set_structure (ARGS_STRUCT);
    }

  if (opt->arg_is_optional)
    {
      if (opt->type == ARG_STRING && opt->multiple)
        {
          string_opt_arg_no_free_gen_class str_gen_null;
          str_gen_null.set_optarg ("NULL");
          str_gen_null.set_structure (string (opt->var_arg) + "_new");
          str_gen_null.set_opt_var (opt->var_arg);
          str_gen_null.generate_string_opt_arg_no_free (stream, indent);
          stream << endl;
        }

      stream << "if (" << argstr << ")" << endl;
      stream << indent_str << "  {" << endl;
      stream << indent_str << "    ";
      indent += offset;
    }

  switch (opt->type)
    {
    case ARG_NO:
      stream << "break;";
      break;
    case ARG_FLAG:
      flag_gen.set_opt_var (opt->var_arg);
      flag_gen.generate_flag_opt_arg (stream, indent);
      break;
    case ARG_STRING:
      if (opt->multiple)
        {
          str_gen_no_free.set_opt_var (opt->var_arg);
          str_gen_no_free.generate_string_opt_arg_no_free (stream, indent);
        }
      else
        {
          str_gen.set_opt_var (opt->var_arg);
          str_gen.generate_string_opt_arg (stream, indent);
        }
      break;
    case ARG_INT:
      int_gen.set_cast ("");
      int_gen.set_opt_var (opt->var_arg);
      int_gen.generate_int_opt_arg (stream, indent);
      break;
    case ARG_SHORT:
      int_gen.set_cast ("(short)");
      int_gen.set_opt_var (opt->var_arg);
      int_gen.generate_int_opt_arg (stream, indent);
      break;
    case ARG_LONG:
      long_gen.set_cast ("");
      long_gen.set_opt_var (opt->var_arg);
      long_gen.generate_long_opt_arg (stream, indent);
      break;
    case ARG_FLOAT:
      float_gen.set_cast ("(float)");
      float_gen.set_opt_var (opt->var_arg);
      float_gen.generate_float_opt_arg (stream, indent);
      break;
    case ARG_DOUBLE:
      float_gen.set_cast ("");
      float_gen.set_opt_var (opt->var_arg);
      float_gen.generate_float_opt_arg (stream, indent);
      break;
    case ARG_LONGDOUBLE:
      float_gen.set_cast ("(long double)");
      float_gen.set_opt_var (opt->var_arg);
      float_gen.generate_float_opt_arg (stream, indent);
      break;
    case ARG_LONGLONG:
      long_long_gen.set_opt_var (opt->var_arg);
      long_long_gen.generate_long_long_opt_arg (stream, indent);
      break;
    default:
      fprintf (stderr, "gengetopt: bug found in %s:%d\n", __FILE__,
               __LINE__);
      abort ();
    }

  if (opt->arg_is_optional)
    {
      stream << endl;
      stream << indent_str << "  }";
      indent -= offset;
    }
}

int
CmdlineParserCreator::generate ()
{
  int head_result;

  head_result = generate_header_file ();
  if (head_result)
    return head_result;

  return generate_source ();
}

int
CmdlineParserCreator::generate_header_file ()
{
  if (gengetopt_options == 0)
    {
      fprintf (stderr, "gengetopt: none option given\n");
      return 1;
    }

  /* ****************************************************** */
  /* HEADER FILE******************************************* */
  /* ****************************************************** */

  ofstream *output_file = open_fstream (header_filename);
  generate_header (*output_file);
  output_file->close ();
  delete output_file;

  return 0;
}

void
CmdlineParserCreator::generate_option_arg(ostream &stream,
                                          unsigned int indent)
{
  struct gengetopt_option * opt;
  string indent_str (indent, ' ');
  bool first = true;

  foropt
    if (opt->type != ARG_NO) {
      switch (opt->type) {
      case ARG_FLAG:
      case ARG_STRING:
      case ARG_INT:
      case ARG_SHORT:
      case ARG_LONG:
      case ARG_FLOAT:
      case ARG_DOUBLE:
      case ARG_LONGDOUBLE:
      case ARG_LONGLONG:
        if (! first)
          stream << indent_str;
        else
          first = false;

        _generate_option_arg (stream, indent, opt);

        break;
      default:
        fprintf (stderr, "gengetopt: bug found in %s:%d!!\n",
                 __FILE__, __LINE__);
        abort ();
      }
    }
}

void
_generate_option_arg(ostream &stream,
                     unsigned int indent,
                     struct gengetopt_option *opt)
{
  ostringstream opt_stream;

  if (opt->multiple)
    opt_stream << "* ";
  else
    opt_stream << " ";

  if (opt->type == ARG_FLAG)
    opt_stream << opt->var_arg << "_flag";
  else
    opt_stream << opt->var_arg << "_arg";

  opt_stream << ";\t/* " << opt->desc;

  if (opt->default_given)
    {
      opt_stream << " (default='" << opt->default_string << "')";
    }

  if (opt->type == ARG_FLAG)
    {
      if (opt->flagstat)
        opt_stream << " (default=on)";
      else
        opt_stream << " (default=off)";
    }

  opt_stream << ".  */\n";
  const string opt_string = opt_stream.str ();

  if (opt->type == ARG_LONGLONG)
    {
      string indent_str (indent, ' ');

      stream << "#ifdef HAVE_LONG_LONG\n";
      stream << indent_str;
      stream << arg_types[opt->type] << opt_string;
      stream << indent_str;
      stream << "#else\n";
      stream << indent_str;
      stream << arg_types[ARG_LONG] << opt_string;
      stream << indent_str;
      stream << "#endif\n";
    }
  else
    stream << arg_types[opt->type] << opt_string;
}

void
CmdlineParserCreator::generate_option_given(ostream &stream,
                                            unsigned int indent)
{
  struct gengetopt_option * opt;
  string indent_str (indent, ' ');
  bool first = true;

  foropt
    {
      switch (opt->type) {
      case ARG_NO:
      case ARG_FLAG:
      case ARG_STRING:
      case ARG_INT:
      case ARG_SHORT:
      case ARG_LONG:
      case ARG_FLOAT:
      case ARG_DOUBLE:
      case ARG_LONGDOUBLE:
      case ARG_LONGLONG: break;
      default:
        fprintf (stderr, "gengetopt: bug found in %s:%d!!\n",
                 __FILE__, __LINE__);
        abort ();
      }
      if (! first)
        stream << indent_str;
      else
        first = false;

      if (opt->multiple)
        stream << "unsigned ";

      stream << "int " << opt->var_arg <<
        "_given ;\t/* Whether " << opt->long_opt << " was given.  */\n";
    }

  if (unamed_options)
    {
      stream << endl;
      stream << indent_str;
      stream << "char **inputs ; /* unamed options */\n" ;
      stream << indent_str;
      stream << "unsigned inputs_num ; /* unamed options number */" ;
    }
}

void
CmdlineParserCreator::generate_print_purpose(ostream &stream,
                                             unsigned int indent)
{
  if (gengetopt_purpose != NULL)
    {
      string indent_str (indent, ' ');

      string wrapped_purpose;
      wrap_cstr(wrapped_purpose, 2, 0, gengetopt_purpose);

      stream << indent_str;
      stream << "\"Purpose:\\n\"";
      stream << endl;
      stream << "\"  ";
      stream << wrapped_purpose;
      stream << "\\n\""; // close the last line
      stream << endl;
      stream << indent_str;
      stream << "\"\\n\"\n"; // generate the last \n
    }
}

string
CmdlineParserCreator::generate_usage_string()
{
  struct gengetopt_option * opt;
  ostringstream usage;
  const char   *type_str;

  usage << "Usage: %s ";

  if ( long_help )
    {
      foropt
        if (opt->required) /* required options */
          switch (opt->type) {
          case ARG_INT:
          case ARG_SHORT:
          case ARG_LONG:
          case ARG_FLOAT:
          case ARG_DOUBLE:
          case ARG_LONGDOUBLE:
          case ARG_LONGLONG:
          case ARG_STRING:
            if (opt->type_str)
              type_str = opt->type_str;
            else
              type_str = arg_names[opt->type];

            if (opt->short_opt)
              {
                usage << "-" << opt->short_opt
                      << type_str << "|";
              }
            usage << "--" << opt->long_opt << "=" <<
              type_str << " ";
            break;
          default:
            fprintf (stderr, "gengetopt: bug found in %s:%d!!\n",
                     __FILE__, __LINE__);
            abort ();
          }
      foropt
        if (!opt->required)
          switch (opt->type) {
          case ARG_NO:
          case ARG_FLAG:
            usage << "[";
            if (opt->short_opt)
              {
                usage << "-" << opt->short_opt << "|";
              }
            usage << "--" << opt->long_opt << "] ";
            break;
          case ARG_INT:
          case ARG_SHORT:
          case ARG_LONG:
          case ARG_FLOAT:
          case ARG_DOUBLE:
          case ARG_LONGDOUBLE:
          case ARG_LONGLONG:
          case ARG_STRING:
            if (opt->type_str)
              type_str = opt->type_str;
            else
              type_str = arg_names[opt->type];

            if (opt->short_opt)
              {
                usage << "-" << opt->short_opt
                      << type_str << "|";
              }
            usage << "--" << opt->long_opt << "=" <<
              type_str << " ";
            break;
          default: fprintf (stderr, "gengetopt: bug found in %s:%d!!\n",
                            __FILE__, __LINE__);
            abort ();
          }
    }
  else
    { /* if not long help we generate it as GNU standards */
      usage << "[OPTIONS]...";
    }

  if ( unamed_options )
    usage << " [" << unamed_options << "]...";

  usage << "\\n";

  return usage.str ();
}

static void
generate_help_desc_print(ostream &stream,
                         unsigned int desc_column,
                         const char *descript, const char *defval)
{
  string desc;
  string desc_with_default = descript;

  if (defval)
    {
      desc_with_default += "  (default=";
      desc_with_default += defval;
      desc_with_default += ")";
    }

  wrap_cstr ( desc, desc_column, 2, desc_with_default );

  stream << desc;
}


void
CmdlineParserCreator::generate_help_option_print(ostream &stream,
                                                 unsigned int indent)
{
  long desc_col;
  struct gengetopt_option * opt;
  string indent_str (indent, ' ');

  int                type_len;
  const char   *type_str;

  /* calculate columns */
  desc_col = 0;
  foropt {
    int width = 2 + 4 + 2;  // ws + "-a, " + ws

    width += strlen (opt->long_opt) + 2;  // "--"

    if ((opt->type != ARG_FLAG) &&
        (opt->type != ARG_NO))
      {
        if (opt->type_str)
          type_str = opt->type_str;
        else
          type_str = arg_names[opt->type];
        type_len = strlen(type_str);

        width += type_len + 1;        // "="

        if (opt->arg_is_optional)
          width += 2; // "[" and "]"
      }

    if (width > desc_col)
      desc_col = width;
  }

  if (desc_col > MAX_STARTING_COLUMN)
    desc_col = MAX_STARTING_COLUMN;

  /* print justified options */
  char *prev_group = 0;
  char *curr_section = 0;
  bool first_option = true;

  stream << indent_str;
  stream << "printf(\"\\n\");\n";

  print_help_string_gen_class print_gen;

  foropt
    {
      if (opt->group_value &&
          (! prev_group || strcmp (opt->group_value, prev_group) != 0))
        {
          string group_string = "\\n Group: ";
          string wrapped_desc;

          if (opt->group_desc && strlen (opt->group_desc))
            {
              wrapped_desc = "\\n  ";
              wrap_cstr (wrapped_desc, 2, 0, opt->group_desc);
            }

          stream << indent_str;
          group_string += opt->group_value + wrapped_desc;
          print_gen.set_helpstring (group_string);
          print_gen.generate_print_help_string (stream, indent);
          stream << endl;

          prev_group = opt->group_value;
        }

      if (opt->section &&
          (!curr_section || strcmp (curr_section, opt->section)))
        {
          curr_section = opt->section;

          ostringstream sec_string;

          if (! first_option)
            sec_string << "\\n";

          sec_string << opt->section << ":" ;
          print_gen.set_helpstring (sec_string.str());
          stream << indent_str;
          print_gen.generate_print_help_string (stream, indent);
          stream << endl;

          if (opt->section_desc)
            {
              string wrapped_desc ( 2, ' ');
              wrap_cstr ( wrapped_desc, 2, 0, opt->section_desc );

              stream << indent_str;
              print_gen.set_helpstring ( wrapped_desc );
              print_gen.generate_print_help_string (stream, indent);
              stream << endl;
            }
        }

      first_option = false;
      const char * def_val = NULL;
      string def_str = "`";

      stream << indent_str;
      stream << "printf(\"%s\\n\",\"";

      ostringstream option_stream;

      if (opt->type == ARG_FLAG || opt->type == ARG_NO)
        {
          def_val = NULL;

          if (opt->short_opt)
            option_stream << "  -" << opt->short_opt << ", ";
          else
            option_stream << "      ";

          option_stream << "--" << opt->long_opt;

          if (opt->type == ARG_FLAG)
            def_val = opt->flagstat ? "on" : "off";
        }
      else
        {
          def_val = NULL;

          if (opt->type_str)
            type_str = opt->type_str;
          else
            type_str = arg_names[opt->type];

          type_len = strlen(type_str);

          if (opt->short_opt)
            {
              option_stream << "  -" << opt->short_opt << ", ";
            }
          else
            {
              option_stream << "      ";
            }

          bool arg_optional = opt->arg_is_optional;
          option_stream << "--" << opt->long_opt
                 << (arg_optional ? "[" : "")
                 << "=" << type_str
                 << (arg_optional ? "]" : "");

          if (opt->default_string)
            {
              def_str += opt->default_string;
              def_str += "'";
              def_val = def_str.c_str();
            }
        }

      const string &option_string = option_stream.str();
      stream << option_string;
      const char *opt_desc = opt->desc;

      if (option_string.size() >= MAX_STARTING_COLUMN)
        {
          string indent (MAX_STARTING_COLUMN, ' ');
          stream << "\\n" << indent;
        }
      else
        {
          string indent (desc_col - option_string.size(), ' ');
          stream << indent;
        }

      generate_help_desc_print(stream, desc_col, opt_desc, def_val);
      stream << "\");\n";
    }
}

void
CmdlineParserCreator::generate_given_init(ostream &stream,
                                          unsigned int indent)
{
  struct gengetopt_option * opt;
  string indent_str (indent, ' ');

  /* now we initialize "given" fields */
  foropt
    {
      stream << indent_str;
      stream << ARGS_STRUCT << "->" << opt->var_arg << "_given = 0 ;";
      stream << endl;
    }

  groups_collection_t::const_iterator end = gengetopt_groups.end();
  for ( groups_collection_t::const_iterator idx = gengetopt_groups.begin();
        idx != end; ++idx)
    {
      stream << indent_str;
      stream << ARGS_STRUCT << "->" << canonize_name (idx->first) << "_group_counter = 0 ;";
      stream << endl;
    }
}

void
CmdlineParserCreator::generate_reset_groups(ostream &stream, unsigned int indent)
{
  struct gengetopt_option * opt;
  string indent_str (indent, ' ');
  ostringstream body;
  reset_group_gen_class reset_group;

  reset_group.set_args_info (c_source_gen_class::args_info);

  groups_collection_t::const_iterator end = gengetopt_groups.end();
  for ( groups_collection_t::const_iterator idx = gengetopt_groups.begin();
        idx != end; ++idx)
    {
      body.str ("");
      bool found_option = false;

      foropt
      {
        if (opt->group_value && strcmp(opt->group_value, idx->first.c_str()) == 0)
          {
            /* now we reset "given" fields */
            body << indent_str;
            body << ARGS_STRUCT << "->" << opt->var_arg << "_given = 0 ;";
            body << endl;

            free_option (opt, body, indent);
            found_option = true;
          }
      }

      if (found_option)
        {
          reset_group.set_name (canonize_name (idx->first));
          reset_group.set_body (body.str ());
          reset_group.generate_reset_group (stream);
        }
    }
}

void
CmdlineParserCreator::free_option(struct gengetopt_option *opt,
                                  ostream &stream, unsigned int indent)
{
  if (! opt->type)
    return;

  if (opt->type == ARG_STRING)
    {
      if (opt->multiple)
        {
          free_multiple_string_gen_class free_multiple_string;
          free_multiple_string.set_comment ("previous argument");
          free_multiple_string.set_structure (ARGS_STRUCT);

          free_multiple_string.set_opt_var (opt->var_arg);
          free_multiple_string.generate_free_multiple_string
            (stream, indent);
        }
      else
        {
          free_string_gen_class free_string;
          free_string.set_comment ("previous argument");
          free_string.set_structure (ARGS_STRUCT);

          free_string.set_opt_var (opt->var_arg);
          free_string.generate_free_string (stream, indent);
        }
    }
  else if (opt->multiple)
    {
      free_string_gen_class free_string;
      free_string.set_comment ("previous argument");
      free_string.set_structure (ARGS_STRUCT);

      // it's not really a string, but an array anyway
      free_string.set_opt_var (opt->var_arg);
      free_string.generate_free_string (stream, indent);
    }
}


void
CmdlineParserCreator::generate_struct_def(ostream &stream, unsigned int indent)
{
  struct gengetopt_option * opt;
  multiple_opt_struct_gen_class multiple_opt_struct;

  /* define structs for multiple options */
  foropt
    {
      if (opt->multiple && opt->type)
        {
          multiple_opt_struct.set_type (arg_types[opt->type]);
          multiple_opt_struct.set_arg_name (opt->var_arg);
          multiple_opt_struct.generate_multiple_opt_struct (stream, 0);
        }
    }
}

void
CmdlineParserCreator::generate_list_def(ostream &stream, unsigned int indent)
{
  struct gengetopt_option * opt;
  string indent_str (indent, ' ');
  multiple_opt_list_gen_class multiple_opt_list;
  bool generated_counter = false;

  /* define linked-list structs for multiple options */
  foropt
    {
      if (opt->multiple)
        {
          if (! generated_counter)
            {
              stream << indent_str;
              stream << "int i;        /* Counter */" << endl;
              generated_counter = true;
              stream << endl;
            }
          if (opt->type)
            {
              stream << indent_str;
              multiple_opt_list.set_arg_name (opt->var_arg);
              multiple_opt_list.generate_multiple_opt_list (stream, indent);
              stream << endl;
            }
        }
    }
}

void
CmdlineParserCreator::generate_multiple_fill_array(ostream &stream, unsigned int indent)
{
  struct gengetopt_option * opt;
  string indent_str (indent, ' ');
  multiple_fill_array_gen_class filler;

  /* copy linked list into the array */
  foropt
    {
      if (opt->multiple && opt->type)
        {
          stream << indent_str;
          filler.set_type (arg_types[opt->type]);
          filler.set_option_var_name (opt->var_arg);
          filler.generate_multiple_fill_array (stream, indent);

          if (opt->default_given)
            {
              multiple_fill_array_default_gen_class def_filler;
              string def_value = opt->default_string;

              def_filler.set_type (arg_types[opt->type]);
              def_filler.set_option_var_name (opt->var_arg);
              if (opt->type == ARG_STRING)
                def_value = "gengetopt_strdup(\"" +
                  def_value + "\")";
              def_filler.set_default_value (def_value);
              def_filler.generate_multiple_fill_array_default (stream, indent);
            }

          stream << endl;
        }
    }
}

void
CmdlineParserCreator::generate_update_multiple_given(ostream &stream, unsigned int indent)
{
  if (! has_multiple_options())
    return;

  string indent_str (indent, ' ');

  stream << endl;
  stream << indent_str;

  update_given_gen_class update_given_gen;
  struct gengetopt_option * opt;

  foropt
    {
      if (opt->multiple)
        {
          update_given_gen.set_option_var_name (opt->var_arg);
          update_given_gen.generate_update_given (stream, indent);
        }
    }
}

void
CmdlineParserCreator::generate_clear_arg(ostream &stream, unsigned int indent)
{
  struct gengetopt_option * opt;
  string indent_str (indent, ' ');

  /* now we initialize "given" fields */
  foropt
    {
      if (opt->multiple && opt->type)
        {
          // since multiple options are stored in arrays, we
          // always initialize it to NULL

          stream << indent_str;

          stream << ARGS_STRUCT << "->" << opt->var_arg
                 << "_arg = NULL;" << endl;
        }
      else if (opt->type == ARG_STRING)
        {
          stream << indent_str;

          if (opt->default_given)
            stream << ARGS_STRUCT << "->" << opt->var_arg
                   << "_arg = gengetopt_strdup (\"" << opt->default_string
                   << "\");";
          else
            stream << ARGS_STRUCT << "->" << opt->var_arg
                   << "_arg = NULL;";

          stream << endl;
        }
      else if (opt->type == ARG_FLAG)
        {
          stream << indent_str;

          stream << ARGS_STRUCT << "->" << opt->var_arg << "_flag";
          stream << " = " << opt->flagstat << ";";

          stream << endl;
        }
      else if (opt->type != ARG_NO && opt->default_given)
        {
          stream << indent_str;

          stream << ARGS_STRUCT << "->" << opt->var_arg << "_arg";
          stream << " = " << opt->default_string << " ;";

          stream << endl;
        }
    }
}

void
CmdlineParserCreator::generate_init_unamed(ostream &stream,
                                           unsigned int indent)
{
  if ( unamed_options )
    {
      string indent_str (indent, ' ');

      stream << endl;
      stream << indent_str;
      stream << ARGS_STRUCT << "->inputs = NULL;\n";
      stream << indent_str;
      stream << ARGS_STRUCT << "->inputs_num = 0;\n";
    }
}

void
CmdlineParserCreator::generate_long_option_struct(ostream &stream,
                                                  unsigned int indent)
{
  string indent_str (indent, ' ');
  struct gengetopt_option * opt;

  foropt
    {
      if (opt->short_opt == 'h' || opt->short_opt == 'V')
        continue;

      stream << indent_str;

      stream << "{ \"" << opt->long_opt << "\",\t"
             << (opt->type == ARG_NO || opt->type == ARG_FLAG ? 0 :
                 (opt->arg_is_optional ? 2 : 1))
             << ", NULL, ";

      if (opt->short_opt)
        stream << "\'" << opt->short_opt << "\'";
      else
        stream << "0";

      stream << " }," << endl;
    }
}

string
CmdlineParserCreator::generate_getopt_string()
{
  struct gengetopt_option * opt;
  ostringstream built_getopt_string;

  foropt
    if (opt->short_opt)
      {
        built_getopt_string << opt->short_opt <<
          (opt->type == ARG_NO || opt->type == ARG_FLAG ? "" : ":");
        built_getopt_string <<
          (opt->arg_is_optional ? ":" : "");
      }

  return built_getopt_string.str ();
}

void
CmdlineParserCreator::generate_handle_help(ostream &stream,
                                           unsigned int indent)
{
 if (no_handle_help)
   {
     generic_option_gen_class help_gen;
     help_gen.set_long_option (HELP_LONG_OPT);
     help_gen.set_short_option (HELP_SHORT_OPT_STR);
     help_gen.set_option_comment (HELP_OPT_DESCR);
     help_gen.set_option_var_name (HELP_LONG_OPT);
     help_gen.set_package_var_name (c_source_gen_class::package_var_name);

     help_gen.generate_generic_option (stream, indent);

     string indent_str (indent, ' ');
     stream << "return 0;";
   }
 else
   {
     handle_help_gen_class help_gen;
     help_gen.set_parser_name (parser_function_name);
     help_gen.generate_handle_help (stream, indent);
   }
}

void
CmdlineParserCreator::generate_handle_version(ostream &stream,
                                              unsigned int indent)
{
 if (no_handle_version)
   {
     generic_option_gen_class version_gen;
     version_gen.set_long_option (VERSION_LONG_OPT);
     version_gen.set_short_option (VERSION_SHORT_OPT_STR);
     version_gen.set_option_comment (VERSION_OPT_DESCR);
     version_gen.set_option_var_name (VERSION_LONG_OPT);
     version_gen.set_package_var_name (c_source_gen_class::package_var_name);

     version_gen.generate_generic_option (stream, indent);

     string indent_str (indent, ' ');
     stream << "return 0;";
   }
 else
   {
     handle_version_gen_class version_gen;
     version_gen.set_parser_name (parser_function_name);
     version_gen.generate_handle_version (stream, indent);
   }
}

void
CmdlineParserCreator::generate_handle_no_short_option(ostream &stream,
                                                      unsigned int indent)
{
  const string tmpvar("multi_token"); // reuse the `c' variable (returning getopt_long value)
  const string optarg("optarg");

  struct gengetopt_option * opt;
  no_short_option_gen_class opt_gen;
  string indent_str (indent, ' ');
  opt_gen.set_package_var_name (c_source_gen_class::package_var_name);
  multiple_option_no_short_gen_class multip_opt_gen;
  multiple_option_new_list_node_gen_class multi_gen_node;
  bool first = true;

  foropt
    if (! opt->short_opt)
      {
        stream << indent_str;
        ostringstream str_stream;

        if (opt->multiple)
          {
            ostringstream list_stream;

            multip_opt_gen.set_option_comment (opt->desc);
            multip_opt_gen.set_long_option (opt->long_opt);
            multip_opt_gen.set_option_var_name (opt->var_arg);

            if (opt->type)
              { // arguments possibly exists
                do_update_arg (opt, str_stream, 0, tmpvar);

                multi_gen_node.set_optarg(optarg);
                multi_gen_node.set_option_var_name (opt->var_arg);
                multi_gen_node.set_update_arg(str_stream.str());
                multi_gen_node.generate_multiple_option_new_list_node (list_stream);
              }

            multip_opt_gen.set_update_arg ("break;");

            multip_opt_gen.set_new_node_list (list_stream.str ());

            generic_option_group_gen_class group_gen;
            if (opt->group_value)
              {
                str_stream.str ("");
                str_stream << endl;
                str_stream << "  ";
                group_gen.set_group_var_name
                  (canonize_name (opt->group_value));
                group_gen.generate_generic_option_group (str_stream, 2);
                multip_opt_gen.set_update_group_count (str_stream.str ());
              }
            else
              multip_opt_gen.set_update_group_count ("");

            multip_opt_gen.generate_multiple_option_no_short
              (stream, indent);
          }
        else
          {
            opt_gen.set_option_comment (opt->desc);
            opt_gen.set_long_option (opt->long_opt);
            opt_gen.set_option_var_name (opt->var_arg);

            do_update_arg (opt, str_stream, 0);
            opt_gen.set_update_arg (str_stream.str ());

            generic_option_group_gen_class group_gen;
            if (opt->group_value)
              {
                str_stream.str ("");
                str_stream << endl;
                str_stream << "  ";
                group_gen.set_group_var_name
                  (canonize_name (opt->group_value));
                group_gen.generate_generic_option_group (str_stream, 2);
                opt_gen.set_update_group_count (str_stream.str ());
              }
            else
              opt_gen.set_update_group_count ("");

            opt_gen.generate_no_short_option (stream, indent);
          }

        stream << endl;

        if (first)
          {
            first = false;
            opt_gen.set_gen_else ("else ");
            multip_opt_gen.set_gen_else ("else ");
          }
      }

  if (! first) // something has been generated
    {
      generateBreak(stream, indent);
    }

  stream << endl;
}

void
CmdlineParserCreator::generate_handle_option(ostream &stream,
                                             unsigned int indent)
{
  const string tmpvar("multi_token"); // reuse the `c' variable (returning getopt_long value)
  const string optarg("optarg");

  struct gengetopt_option * opt;
  generic_option_gen_class option_gen;
  multiple_option_gen_class multi_gen;
  multiple_option_new_list_node_gen_class multi_gen_node;
  string indent_str (indent, ' ');

  option_gen.set_package_var_name (c_source_gen_class::package_var_name);

  foropt
    {
      if (opt->short_opt)
        {
          if (opt->short_opt == 'h' || opt->short_opt == 'V')
            continue;

          stream << indent_str;

          string short_opt (1, opt->short_opt);
          if (opt->multiple)
            {
              multi_gen.set_short_option (short_opt);
              multi_gen.set_option_comment (opt->desc);
              multi_gen.set_option_var_name (opt->var_arg);

              multi_gen.generate_multiple_option (stream, indent);

              if (opt->type)
                {
                  ostringstream update_arg;

                  do_update_arg (opt, update_arg, 0, tmpvar);

                  multi_gen_node.set_optarg(optarg);
                  multi_gen_node.set_option_var_name (opt->var_arg);
                  multi_gen_node.set_update_arg(update_arg.str());
                  multi_gen_node.generate_multiple_option_new_list_node (stream, indent+2);
                }
              else
                do_update_arg (opt, stream, indent + 2);
            }
          else
            {
              option_gen.set_short_option (short_opt);
              option_gen.set_option_comment (opt->desc);
              option_gen.set_long_option (opt->long_opt);
              option_gen.set_option_var_name (opt->var_arg);

              option_gen.generate_generic_option (stream, indent);

              // NOTE: I don't know why the following code was shared with the branch for
              // multiple options because you can't specify multiple groupoption
              generic_option_group_gen_class group_gen;
              if (opt->group_value)
                {
                  group_gen.set_group_var_name
                    (canonize_name (opt->group_value));
                  group_gen.generate_generic_option_group (stream, indent+2);
                }

              do_update_arg (opt, stream, indent + 2);
            }

          if (opt->type != ARG_NO)
            generateBreak(stream, indent + 2);

          stream << endl;
          stream << endl;
        }
    }
}

#define GROUP_REQUIRED_COMPARISON "!="
#define GROUP_NOT_REQUIRED_COMPARISON ">"
#define GROUP_REQUIRED_MESSAGE "One"
#define GROUP_NOT_REQUIRED_MESSAGE "At most one"

void
CmdlineParserCreator::generate_handle_group(ostream &stream,
                                            unsigned int indent)
{
  group_option_gen_class opt_gen;
  string indent_str (indent, ' ');
  opt_gen.set_package_var_name (c_source_gen_class::package_var_name);

  groups_collection_t::const_iterator end = gengetopt_groups.end();
  for ( groups_collection_t::const_iterator idx = gengetopt_groups.begin();
        idx != end; ++idx)
    {
      stream << indent_str;
      opt_gen.set_group_name (idx->first);
      opt_gen.set_group_var_name (canonize_name (idx->first));
      if (idx->second.required)
        {
          opt_gen.set_Comparison_rule(GROUP_REQUIRED_COMPARISON);
          opt_gen.set_number_required(GROUP_REQUIRED_MESSAGE);
        }
      else
        {
          opt_gen.set_Comparison_rule(GROUP_NOT_REQUIRED_COMPARISON);
          opt_gen.set_number_required(GROUP_NOT_REQUIRED_MESSAGE);
        }

      opt_gen.generate_group_option (stream, indent);
      stream << endl;
    }
}

void
CmdlineParserCreator::generate_handle_required(ostream &stream,
                                               unsigned int indent)
{
  struct gengetopt_option * opt;
  required_option_gen_class opt_gen;
  opt_gen.set_package_var_name (c_source_gen_class::package_var_name);
  string indent_str (indent, ' ');

  /* write test for required options */
  foropt
    if ( opt->required )
      {
        stream << indent_str;

        ostringstream req_opt;
        req_opt << "'--" << opt->long_opt << "'";
        if (opt->short_opt)
          req_opt << " ('-" << opt->short_opt << "')";

        opt_gen.set_option_var_name (opt->var_arg);
        opt_gen.set_option_descr (req_opt.str ());

        opt_gen.generate_required_option (stream, indent);

        stream << endl;
      }
}

void
CmdlineParserCreator::generate_group_counters(ostream &stream,
                                              unsigned int indent)
{
  group_counter_gen_class counter_gen;
  string indent_str (indent, ' ');

  groups_collection_t::const_iterator end = gengetopt_groups.end();
  for ( groups_collection_t::const_iterator idx = gengetopt_groups.begin();
        idx != end; ++idx)
    {
      stream << indent_str;
      counter_gen.set_group_name (canonize_name (idx->first));
      counter_gen.generate_group_counter (stream, indent);
      stream << endl;
    }
}

int
CmdlineParserCreator::generate_source ()
{
  /* ****************************************************** */
  /* ********************************************** C FILE  */
  /* ****************************************************** */

  set_usage_string (generate_usage_string ());
  set_getopt_string (generate_getopt_string ());

  ofstream *output_file = open_fstream (c_filename);
  generate_c_source (*output_file);
  output_file->close ();
  delete output_file;

  return 0;
}

void
CmdlineParserCreator::generate_free(ostream &stream,
                                    unsigned int indent)
{
  string indent_str (indent, ' ');
  struct gengetopt_option * opt;

  stream << endl;
  stream << indent_str;

  if (unamed_options || has_multiple_options_string ())
    {
      stream << "unsigned int i;";
      stream << endl;
      stream << indent_str;
    }

  foropt
    {
      free_option (opt, stream, indent);
    }

  if (unamed_options)
    {
      stream << endl;
      stream << indent_str;

      free_inputs_string_gen_class free_inputs_string;
      free_inputs_string.set_structure (ARGS_STRUCT);
      free_inputs_string.generate_free_inputs_string (stream, indent);
    }
}


