/**
 * Application configuration values

 * Copyright (C) 2008, 2009  Sylvain Beucler

 * This file is part of GNU FreeDink

 * GNU FreeDink is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 3 of the
 * License, or (at your option) any later version.

 * GNU FreeDink is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 */

#include "Config.hpp"
#include "IOUtils.hpp"
#include <wx/config.h>
#include <wx/dir.h>
#include <wx/filename.h>
#include <wx/utils.h>
#include <wx/intl.h>
#include <wx/log.h>

#if defined _WIN32 || defined __WIN32__ || defined __CYGWIN__
#define WIN32_LEAN_AND_MEAN
#include <windows.h> /* registry */
#include <fstream>
#endif

/**
 * Singleton getter
 */
static Config* theConfig = NULL;

Config* Config::GetConfig()
{
  if (theConfig == NULL)
    theConfig = new Config();
  return theConfig;
}

Config::Config():
  // Main frame
  mTrueColorValue(false),
  mWindowedValue(false),
  mSoundValue(true),
  mJoystickValue(true),
  mDebugValue(false),
  mV107Value(false),
  
  mShowIntroductionText(true),
  mWarnOnEdit(true),

  // Options
  mShowDeveloperButtons(false),
  mCloseDfarcOnPlay(false),
  mWriteIniValue(true),
  mOverrideDinkrefDir(false),
  mSpecifiedDinkrefDir(wxEmptyString),
  mDModDir(wxEmptyString)
{
  /* Default value for DModDir */
  mDModDir = ::wxGetHomeDir() + wxFileName::GetPathSeparator() + wxT("dmods");
  mDinkExe = wxT("dink") wxT(EXEEXT);
  mEditorExe = wxT("dinkedit") wxT(EXEEXT);
  mSelectedDmod = GetDinkrefDir() + wxFileName::GetPathSeparator() + wxT("dink");

  mwxConfig = new wxConfig(_T("dfarc3"));
  mwxConfig->Read(_T("True Color"), &mTrueColorValue);
  mwxConfig->Read(_T("Windowed"), &mWindowedValue);
  mwxConfig->Read(_T("Sound"), &mSoundValue);
  mwxConfig->Read(_T("Joystick"), &mJoystickValue);
  mwxConfig->Read(_T("Debug"), &mDebugValue);
  mwxConfig->Read(_T("Windowed"), &mWindowedValue);
  mwxConfig->Read(_T("v1.07"), &mV107Value);
  mwxConfig->Read(_T("Last Selected D-Mod"), &mSelectedDmod);
  mwxConfig->Read(_T("Show Introduction"), &mShowIntroductionText);
  mwxConfig->Read(_T("Warn on Edit"), &mWarnOnEdit);
  
  // Options dialog
  mwxConfig->Read(_T("Close DFArc on Play"), &mCloseDfarcOnPlay);
  mwxConfig->Read(_T("Show Developer Buttons"), &mShowDeveloperButtons);
  mwxConfig->Read(_T("Write Ini"), &mWriteIniValue);
  mwxConfig->Read(_T("Override Dink Directory"), &mOverrideDinkrefDir);
  mwxConfig->Read(_T("Specified Dink Directory"), &mSpecifiedDinkrefDir);
  mwxConfig->Read(_T("Additional D-Mods Directory"), &mDModDir);
  mwxConfig->Read(_T("Dink Executable"), &mDinkExe);
  mwxConfig->Read(_T("Editor Executable"), &mEditorExe);
}

Config::~Config()
{
  delete mwxConfig; // save changes on disk
}

void Config::Update()
{
  // Update config
  mwxConfig->Write(_T("Windowed"), mWindowedValue);
  mwxConfig->Write(_T("True Color"), mTrueColorValue);
  mwxConfig->Write(_T("Sound"), mSoundValue);
  mwxConfig->Write(_T("Joystick"), mJoystickValue);
  mwxConfig->Write(_T("Debug"), mDebugValue);
  mwxConfig->Write(_T("v1.07"), mV107Value);
  mwxConfig->Write(_T("Show Introduction"), mShowIntroductionText);
  mwxConfig->Write(_T("Warn on Edit"), mShowIntroductionText);
  
  // Options dialog
  mwxConfig->Write(_T("Write Ini"), mWriteIniValue);
  mwxConfig->Write(_T("Show Developer Buttons"), mShowDeveloperButtons);
  mwxConfig->Write(_T("Close DFArc on Play"), mCloseDfarcOnPlay);
  mwxConfig->Write(_T("Override Dink Directory"), mOverrideDinkrefDir);
  mwxConfig->Write(_T("Specified Dink Directory"), mSpecifiedDinkrefDir);
  mwxConfig->Write(_T("Additional D-Mods Directory"), mDModDir);
  mwxConfig->Write(_T("Dink Executable"), mDinkExe);
  mwxConfig->Write(_T("Editor Executable"), mEditorExe);
  
  mwxConfig->Write(_T("Last Selected D-Mod"), mSelectedDmod);
  return;
}



/*-------------------*/
/*- GetDinkrefDir() -*/
/*-------------------*/

static wxString FindDefaultDinkrefDir()
{
#if defined _WIN32 || defined __WIN32__ || defined __CYGWIN__
  const wxString fallback_woe_dir = _T("C:/Program Files/FreeDink");

  /* Just check in the same directory */
  char exe_dir[MAX_PATH];
  if (GetModuleFileName(NULL, exe_dir, sizeof(exe_dir)) < 0)
    return fallback_woe_dir;
  wxString str = exe_dir;
  return IOUtils::GetParentDir(str);

  /* The rest is too complex and error-prone when there's more than
     one Dink version installed, let's just skip */
if (false) {
  //
  // First see if it's in the registry.
  //
  unsigned long int iszPathSize = 512;
  unsigned long int iRegType = REG_SZ;
  unsigned long int liError;
  unsigned char *szPath = new unsigned char[512];
  HKEY hDS;
  
  memset(szPath, 0, 512);
  
  // Open the registry path.
  liError = RegOpenKeyEx(HKEY_LOCAL_MACHINE, "Software\\RTSOFT\\Dink Smallwood",
			 0, KEY_READ, &hDS);
  
  // Grab the path.
  while(RegQueryValueEx(hDS, "Path", 0, &iRegType, szPath,
			&iszPathSize) == ERROR_MORE_DATA)
    {
      // If the path is too small, make the buffer bigger.
      iszPathSize += 512;
      delete[] szPath;
      szPath = new unsigned char[iszPathSize];
      memset(szPath, 0, iszPathSize);
    }
  
  // Close the registry.
  RegCloseKey(hDS);
  
  if (liError == ERROR_SUCCESS && szPath[0] != '\0')
    {        
      wxString str = szPath;
      delete[] szPath;
      return str;
    }

  //
  // We don't have the registry key.  See if we can use the deprecated dinksmallwood.ini file.
  //
  std::fstream f_In;
  wxString strCurDir;
  wxString strSysRoot;
  wxString strBuf;
  char ch;

  if (getenv("SystemRoot")) // get %SystemRoot% and set our magic filename.
    strSysRoot = getenv("SystemRoot");
  else if (getenv("windir")) // check for Win9x's %windir%
    strSysRoot = getenv("windir");
  else
    strSysRoot = "C:/Windows";

  if (!::wxDirExists(strSysRoot))
    return fallback_woe_dir;


  //
  // Open dinksmallwood.ini
  //
  strSysRoot += _T("/dinksmallwood.ini");
  if (!::wxFileExists(strSysRoot))
    {
      ::wxLogVerbose(_("Error: dinksmallwood.ini not found. "
		       "Please run the main game and try running this program again."));
      return fallback_woe_dir;
    }
  
  f_In.open(strSysRoot.fn_str(), std::ios::in);
  if (!f_In.good())
    {
      ::wxLogVerbose(_("Error opening dinksmallwood.ini"), strSysRoot.c_str());
      return fallback_woe_dir;
    }
  
  // Parse
  while (!f_In.eof())
    {
      // Read one line
      while((ch = f_In.get()) != '\n')
	if (ch != '\r' && ch != '\t')
	  strBuf.Append(ch, 1);
      
      // Parse String.
      if (strBuf.GetChar(0) == '[')
	{
	  // We can't use this line.  Start over.
	  strBuf.Clear();
	  continue;
	}
      
      // Search for any redeemable qualities.  We'll look for a /.
      for (unsigned int i = 0; i < strBuf.Length(); ++i)
	{
	  if(strBuf.GetChar(i) == '/' || strBuf.GetChar(i) == '\\' || strBuf.GetChar(i) == ':')
	    {
	      f_In.close();
	      
	      strBuf;
	      return strBuf;
	    }
	}
      
      // We can skip any other parts of the file.
    }
  
  
  // We shouldn't get here, but we'll try and save the program.
  f_In.close();
  return strBuf;
}

# else /* !WIN32 */

  // Return standard location
  wxString testdir;

  // pkgdatadir
  testdir = wxT(DEFAULT_DATA_DIR) wxT("/dink/dink/graphics");
  IOUtils::ciconvert(testdir);
  if (wxDir::Exists(testdir))
    return wxT(DEFAULT_DATA_DIR) wxT("/dink");

  // /usr/local/share/games
  testdir = wxT("/usr/local/share/games/dink/dink/graphics");
  IOUtils::ciconvert(testdir);
  if (wxDir::Exists(testdir))
    return wxT("/usr/local/share/games/dink");

  // /usr/local/share
  testdir = wxT("/usr/local/share/dink/dink/graphics");
  IOUtils::ciconvert(testdir);
  if (wxDir::Exists(testdir))
    return wxT("/usr/local/share/dink");

  // /usr/share/games
  testdir = wxT("/usr/local/share/dink/dink/graphics");
  IOUtils::ciconvert(testdir);
  if (wxDir::Exists(testdir))
    return wxT("/usr/local/share/dink");

  // /usr/share
  return wxT("/usr/share/dink");

#endif
}

wxString Config::GetDinkrefDir()
{
  if (mOverrideDinkrefDir == true)
    return mSpecifiedDinkrefDir;
  return FindDefaultDinkrefDir();
}
