/* -*- tab-width: 4 -*-
 *
 * Electric(tm) VLSI Design System
 *
 * File: sim.c
 * Simulation tool: main module
 * Written by: Steven M. Rubin, Static Free Software
 *
 * Copyright (c) 2000 Static Free Software.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Static Free Software
 * 4119 Alpine Road
 * Portola Valley, California 94028
 * info@staticfreesoft.com
 */

#include "config.h"
#if SIMTOOL

#include "global.h"
#include "egraphics.h"
#include "edialogs.h"
#include "sim.h"
#include "simals.h"
#include "simirsim.h"
#include "tecgen.h"
#include "tecschem.h"
#include "usr.h"

/* prototypes for local routines */
static void    sim_checktostopsimulation(NODEPROTO *np);
static void    sim_writephases(float[], INTBIG[], INTBIG, void*);
static void    sim_optionsdlog(void);
static void    sim_spicedlog(void);
static void    sim_verilogdlog(void);
static void    sim_verdiasetfacetinfo(LIBRARY *curlib, void *dia);
static BOOLEAN sim_topofinstances(char **c);
static char   *sim_nextinstance(void);
static BOOLEAN sim_topoffacets(char **c);
static char   *sim_nextfacets(void);

/************************** ALS COMMANDS **************************/

/* ALS's build-actel-models command */
static COMCOMP qbuildp = {NOKEYWORD, topoffile, nextfile, NOPARAMS,
	0, " \t", M_("Actel table file to convert to library"), 0};

/* ALS's clock command */
static INTBIG sim_nextcustomclock(char*, COMCOMP*[], char);
static COMCOMP clockcdurp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, sim_nextcustomclock,
	0, " \t", M_("clock: duration of this phase"), 0};
static COMCOMP clockclevp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	0, " \t", M_("clock: level for this phase"), 0};
static INTBIG sim_nextcustomclock(char *i, COMCOMP *j[], char c)
{ j[0] = &clockclevp; j[1] = &clockcdurp; return(2); }
static COMCOMP clockccyp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, sim_nextcustomclock,
	0, " \t", M_("clock: number of cycles (0 for infinite)"), 0};
static COMCOMP clockcstrp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	0, " \t", M_("clock: strength"), 0};
static COMCOMP clockclp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	0, " \t", M_("clock: random distribution"), 0};
static COMCOMP clockfp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	0, " \t", M_("50/50 duty cycle frequency"), 0};
static COMCOMP clockpp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	0, " \t", M_("50/50 duty cycle period"), 0};
static KEYWORD clockopt[] =
{
	{"frequency",     1,{&clockfp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"period",        1,{&clockpp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"custom",        3,{&clockclp,&clockcstrp,&clockccyp,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP clockop = {clockopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	0, " \t", M_("clock options"), 0};
static COMCOMP clocknp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	0, " \t", M_("node name"), 0};

/* ALS's go command */
static COMCOMP gop = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	0, " \t", M_("maximum simulation time (in seconds)"), 0};

/* ALS's print command */
static COMCOMP printsp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	0, " \t", M_("node to print"), 0};
static KEYWORD printopt[] =
{
	{"display",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"instances",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"netlist",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"size",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"state",          1,{&printsp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"vector",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"xref",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP printp = {printopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	0, " \t", M_("printing options"), 0};

/* ALS's seed command */
static KEYWORD seedopt[] =
{
	{"reset",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"no-reset",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP seedp = {seedopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	0, " \t", M_("random number seed setting"), 0};

/* ALS's set command */
static COMCOMP settp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	0, " \t", M_("time of node set (in seconds)"), 0};
static COMCOMP setsp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	0, " \t", M_("strength of node set"), 0};
static KEYWORD setlopt[] =
{
	{"H",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"L",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"X",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP setlp = {setlopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	0, " \t", M_("state level"), 0};
static COMCOMP setnp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	0, " \t", M_("node name to set"), 0};

/* ALS's trace command */
static KEYWORD traceopt[] =
{
	{"on",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"off",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP tracep = {traceopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	0, " \t", M_("trace state"), 0};

/* ALS's vector command */
static COMCOMP vectordtp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	0, " \t", M_("time at which to delete"), 0};
static KEYWORD vectordopt[] =
{
	{"time",       1,{&vectordtp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"all",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP vectordop = {vectordopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	0, " \t", M_("vector deletion option"), 0};
static COMCOMP vectordp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	0, " \t", M_("node name to delete"), 0};
static COMCOMP vectorlp = {NOKEYWORD, topoffile, nextfile, NOPARAMS,
	0, " \t", M_("file name with vectors"), 0};
static COMCOMP vectorsp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	0, " \t", M_("file name to save vectors"), 0};
static KEYWORD vectoropt[] =
{
	{"delete",      2,{&vectordp,&vectordop,NOKEY,NOKEY,NOKEY}},
	{"load",        1,{&vectorlp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"save",        1,{&vectorsp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"new",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP vectorp = {vectoropt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	0, " \t", M_("vector options"), 0};
static KEYWORD annotateopt[] =
{
	{"minimum",    0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"typical",    0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"maximum",    0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP annotatep = {annotateopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	0, " \t", M_("annotate options"), 0};
static COMCOMP orderrp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	0, " \t", M_("ordered list of trace names"), 0};
static KEYWORD orderopt[] =
{
	{"save",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"restore",    1,{&orderrp,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP orderp = {orderopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	0, " \t", M_("order options"), 0};

/* ALS's command table */
static KEYWORD sim_alsopt[] =
{
	{"annotate",           1,{&annotatep,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"build-actel-models", 1,{&qbuildp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"clock",              2,{&clocknp,&clockop,NOKEY,NOKEY,NOKEY}},
	{"erase",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"go",                 1,{&gop,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"help",               0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"order",              1,{&orderp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"print",              1,{&printp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"seed",               1,{&seedp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"set",                4,{&setnp,&setlp,&setsp,&settp,NOKEY}},
	{"trace",              1,{&tracep,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"vector",             1,{&vectorp,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP sim_alsp = {sim_alsopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	0, " \t", M_("Simulation command"), 0};

/************************** SIMULATION WINDOW COMMANDS **************************/

static COMCOMP cursorgp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	0, " \t", M_("time at the extension cursor"), 0};
static COMCOMP cursorwp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	0, " \t", M_("time at the main cursor"), 0};
static KEYWORD cursoropt[] =
{
	{"extension",    1,{&cursorgp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"main",         1,{&cursorwp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"center",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP cursorp = {cursoropt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	0, " \t", M_("cursor color"), 0};

static COMCOMP moveap = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	0, " \t", M_("distance to move (in seconds)"), 0};
static COMCOMP movesp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	0, " \t", M_("distance to move (in signals)"), 0};
static KEYWORD moveopt[] =
{
	{"left",         1,{&moveap,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"right",        1,{&moveap,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"up",           1,{&movesp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"down",         1,{&movesp,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP movep = {moveopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	0, " \t", M_("window move option"), 0};

static COMCOMP zoomup = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	0, " \t", M_("upper time of zoom"), 0};
static COMCOMP zoomlp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	0, " \t", M_("lower time of zoom"), 0};
static COMCOMP zoomap = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	0, " \t", M_("amount to zoom"), 0};
static KEYWORD zoomopt[] =
{
	{"window",        2,{&zoomlp,&zoomup,NOKEY,NOKEY,NOKEY}},
	{"in",            1,{&zoomap,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"out",           1,{&zoomap,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"cursor",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"all-displayed", 0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP zoomp = {zoomopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	0, " \t", M_("window zoom options"), 0};
static COMCOMP tracessp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	0, " \t", M_("number of traces to show on the display"), 0};
static KEYWORD tracesopt[] =
{
	{"more",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"less",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"set",       1,{&tracessp,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP tracesp = {tracesopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	0, " \t", M_("window traces options"), 0};
static KEYWORD dispcolorsopt[] =
{
	{"white",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"black",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"red",            0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"blue",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"green",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"cyan",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"magenta",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"yellow",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"gray",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"orange",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"purple",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"brown",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"light-gray",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"dark-gray",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"light-red",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"dark-red",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"light-green",    0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"dark-green",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"light-blue",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"dark-blue",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP dispcolorsp = {dispcolorsopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	0, " \t", M_("color for this strength/level"), 0};
static KEYWORD dispcoloropt[] =
{
	{"off",       1,{&dispcolorsp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"node",      1,{&dispcolorsp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"gate",      1,{&dispcolorsp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"power",     1,{&dispcolorsp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"low",       1,{&dispcolorsp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"high",      1,{&dispcolorsp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"undefined", 1,{&dispcolorsp,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP dispcolorp = {dispcoloropt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	0, " \t", M_("window strength color options"), 0};
static COMCOMP windowlevelsp = {NOKEYWORD, sim_topofinstances, sim_nextinstance,
	NOPARAMS, 0, " \t", M_("instance to set"), 0};
static KEYWORD levelopt[] =
{
	{"up",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"set",        1,{&windowlevelsp,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP levelp = {levelopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	0, " \t", M_("level change option"), 0};
static COMCOMP vectorfp = {NOKEYWORD, topoffile, nextfile, NOPARAMS,
	0, " \t", M_("Vector file"), 0};
static KEYWORD sim_windowopt[] =
{
	{"cursor",              1,{&cursorp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"move",                1,{&movep,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"traces",              1,{&tracesp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"zoom",                1,{&zoomp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"color",               1,{&dispcolorp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"2-state-display",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"12-state-display",    0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"advance-time",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"freeze-time",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"display-waveform",    0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"ignore-waveform",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"level",               1,{&levelp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"vector-clear",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"vector-load",         1,{&vectorfp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"vector-save",         1,{&vectorfp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"vector-writespice",   0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP sim_windowp = {sim_windowopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	0, " \t", M_("Simulation window command"), 0};

/************************** SPICE COMMANDS **************************/

static KEYWORD sim_spicelevelopt[] =
{
	{"1",    0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"2",    0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"3",    0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP sim_spicelevelp = {sim_spicelevelopt, NOTOPLIST, NONEXTLIST,
	NOPARAMS, INPUTOPT, " \t", M_("Spice simulation level"), M_("show current")};
static KEYWORD sim_spiceformopt[] =
{
	{"2",                  0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"3",                  0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"hspice",             0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"pspice",             0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP spiceformp = {sim_spiceformopt, NOTOPLIST, NONEXTLIST,
	NOPARAMS, INPUTOPT, " \t", M_("Spice format"), 0};
static KEYWORD sim_spicenotopt[] =
{
	{"resistance",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"capacitances",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"plot",               0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"rawfile",            0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"global-pwr-gnd",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"use-nodenames",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP sim_spicenotp = {sim_spicenotopt, NOTOPLIST, NONEXTLIST,
	NOPARAMS, INPUTOPT, " \t", M_("Spice simulation NOT option"), 0};
static COMCOMP sim_spicereadp = {NOKEYWORD, topoffile, nextfile, NOPARAMS,
	INPUTOPT, " \t", M_("File containing spice output"), 0};
static COMCOMP sim_spicesavep = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	INPUTOPT, " \t", M_("File in which to save spice output"), 0};
static KEYWORD sim_spiceopt[] =
{
	{"level",              1,{&sim_spicelevelp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"save-output",        1,{&sim_spicesavep,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"format",             1,{&spiceformp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"parse-output",       1,{&sim_spicereadp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"not",                1,{&sim_spicenotp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"resistance",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"capacitances",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"plot",               0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"rawfile",            0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"global-pwr-gnd",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"use-nodenames",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP sim_spicep = {sim_spiceopt, NOTOPLIST, NONEXTLIST,
	NOPARAMS, INPUTOPT, " \t", M_("Spice simulation option"), 0};

/************************** VERILOG COMMANDS **************************/

static KEYWORD sim_vernotopt[] =
{
	{"use-assign",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"default-trireg",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP sim_vernotp = {sim_vernotopt, NOTOPLIST, NONEXTLIST,
	NOPARAMS, INPUTOPT, " \t", M_("Verilog simulation NOT option"), 0};
static KEYWORD sim_veropt[] =
{
	{"not",                     1,{&sim_vernotp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"use-assign",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"default-trireg",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP sim_verp = {sim_veropt, NOTOPLIST, NONEXTLIST,
	NOPARAMS, INPUTOPT, " \t", M_("Verilog simulation option"), 0};

/************************** FASTHENRY COMMANDS **************************/

static COMCOMP simfhaaddp = {NOKEYWORD, NOTOPLIST, NONEXTLIST,
	NOPARAMS, INPUTOPT, " \t", M_("FastHenry arc group name"), 0};
static COMCOMP simfhathickp = {NOKEYWORD, NOTOPLIST, NONEXTLIST,
	NOPARAMS, INPUTOPT, " \t", M_("FastHenry arc thickness"), 0};
static COMCOMP simfhawidsubp = {NOKEYWORD, NOTOPLIST, NONEXTLIST,
	NOPARAMS, INPUTOPT, " \t", M_("FastHenry arc width subdivisions"), 0};
static COMCOMP simfhaheisubp = {NOKEYWORD, NOTOPLIST, NONEXTLIST,
	NOPARAMS, INPUTOPT, " \t", M_("FastHenry arc height subdivisions"), 0};
static COMCOMP simfhazheadp = {NOKEYWORD, NOTOPLIST, NONEXTLIST,
	NOPARAMS, INPUTOPT, " \t", M_("FastHenry arc head Z"), 0};
static COMCOMP simfhaztailp = {NOKEYWORD, NOTOPLIST, NONEXTLIST,
	NOPARAMS, INPUTOPT, " \t", M_("FastHenry arc tail Z"), 0};
static KEYWORD sim_fhaopt[] =
{
	{"add",                 1,{&simfhaaddp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"remove",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"thickness",           1,{&simfhathickp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"width-subdivisions",  1,{&simfhawidsubp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"height-subdivisions", 1,{&simfhaheisubp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"z-head",              1,{&simfhazheadp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"z-tail",              1,{&simfhaztailp,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP sim_fhap = {sim_fhaopt, NOTOPLIST, NONEXTLIST,
	NOPARAMS, INPUTOPT, " \t", M_("FastHenry arc option"), 0};
static COMCOMP simfhpolemp = {NOKEYWORD, NOTOPLIST, NONEXTLIST,
	NOPARAMS, INPUTOPT, " \t", M_("Number of poles"), 0};
static KEYWORD simfhpoleopt[] =
{
	{"single",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"multiple",         1,{&simfhpolemp,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP simfhpolep = {simfhpoleopt, NOTOPLIST, NONEXTLIST,
	NOPARAMS, INPUTOPT, " \t", M_("FastHenry multipole option"), 0};
static COMCOMP simfhfreqsp = {NOKEYWORD, NOTOPLIST, NONEXTLIST,
	NOPARAMS, INPUTOPT, " \t", M_("Starting frequency"), 0};
static COMCOMP simfhfreqep = {NOKEYWORD, NOTOPLIST, NONEXTLIST,
	NOPARAMS, INPUTOPT, " \t", M_("Ending frequency"), 0};
static COMCOMP simfhfreqrp = {NOKEYWORD, NOTOPLIST, NONEXTLIST,
	NOPARAMS, INPUTOPT, " \t", M_("Runs per decade"), 0};
static KEYWORD simfhfreqopt[] =
{
	{"single",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"multiple",         3,{&simfhfreqsp,&simfhfreqep,&simfhfreqrp,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP simfhfreqp = {simfhfreqopt, NOTOPLIST, NONEXTLIST,
	NOPARAMS, INPUTOPT, " \t", M_("FastHenry frequency option"), 0};
static COMCOMP simfhthickp = {NOKEYWORD, NOTOPLIST, NONEXTLIST,
	NOPARAMS, INPUTOPT, " \t", M_("FastHenry default thickness"), 0};
static COMCOMP simfhwidthp = {NOKEYWORD, NOTOPLIST, NONEXTLIST,
	NOPARAMS, INPUTOPT, " \t", M_("FastHenry default width subdivisions"), 0};
static COMCOMP simfhheightp = {NOKEYWORD, NOTOPLIST, NONEXTLIST,
	NOPARAMS, INPUTOPT, " \t", M_("FastHenry default height subdivisions"), 0};
static COMCOMP simfhseglenp = {NOKEYWORD, NOTOPLIST, NONEXTLIST,
	NOPARAMS, INPUTOPT, " \t", M_("FastHenry maximum segment length"), 0};
static KEYWORD simfhexeopt[] =
{
	{"none",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"run",            0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"multiple-run",   0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP simfhexep = {simfhexeopt, NOTOPLIST, NONEXTLIST,
	NOPARAMS, INPUTOPT, " \t", M_("FastHenry execution option"), 0};
static KEYWORD simfhpsopt[] =
{
	{"on",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"off",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP simfhpsp = {simfhpsopt, NOTOPLIST, NONEXTLIST,
	NOPARAMS, INPUTOPT, " \t", M_("FastHenry PostScript option"), 0};
static KEYWORD simfhspiceopt[] =
{
	{"on",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"off",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP simfhspicep = {simfhspiceopt, NOTOPLIST, NONEXTLIST,
	NOPARAMS, INPUTOPT, " \t", M_("FastHenry Spice option"), 0};
static KEYWORD sim_fhopt[] =
{
	{"arc",                 1,{&sim_fhap,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"pole",                1,{&simfhpolep,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"frequency",           1,{&simfhfreqp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"thickness",           1,{&simfhthickp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"width-subdivisions",  1,{&simfhwidthp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"height-subdivisions", 1,{&simfhheightp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"max-segment-length",  1,{&simfhseglenp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"execute",             1,{&simfhexep,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"postscript",          1,{&simfhpsp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"spice",               1,{&simfhspicep,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP sim_fhp = {sim_fhopt, NOTOPLIST, NONEXTLIST,
	NOPARAMS, INPUTOPT, " \t", M_("FastHenry simulation option"), 0};

/************************** SIMULATOR COMMANDS **************************/

static KEYWORD simulatorsimulateopt[] =
{
	{"esim",             0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"rsim",             0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"rnl",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"cosmos",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"fasthenry",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"spice",            0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"cspice",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"mossim",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"texsim",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"als",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"verilog",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"abel-pal",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"silos",            0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"internal",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP simsimulatep = {simulatorsimulateopt, NOTOPLIST, NONEXTLIST,
	NOPARAMS, INPUTOPT, " \t", M_("Netlist format to generate"), 0};
static COMCOMP simulatornetp = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,
	INPUTOPT, " \t", M_("net or component to point out"), 0};
static KEYWORD simulatoropt[] =
{
	{"spice",                    1,{&sim_spicep,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"pointout",                 1,{&simulatornetp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"als",                      1,{&sim_alsp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"verilog",                  1,{&sim_verp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"fasthenry",                1,{&sim_fhp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"no-execute",               0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"execute-only",             0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"execute-quietly",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"execute-and-parse",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"execute-quietly-and-parse",0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"resume",                   0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"simulate",                 1,{&simsimulatep,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"window",                   1,{&sim_windowp,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP sim_simulatorp = {simulatoropt,NOTOPLIST,NONEXTLIST,NOPARAMS,
		0, " \t", M_("Simulator action"), M_("show current simulator")};

static struct
{
	char *simname;
	INTBIG min;
	INTBIG format;
} sim_simtable[] =
{
	{"abel-pal",  1, ABEL},
	{"cosmos",	  1, COSMOS},
	{"esim",      1, ESIM},
	{"fasthenry", 1, FASTHENRY},
	{"rnl",       2, RNL},
	{"rsim",      2, RSIM},
	{"irsim",     1, IRSIM},
	{"mossim",    1, MOSSIM},
	{"als",       1, ALS},
	{"texsim",    1, TEXSIM},
	{"silos",     2, SILOS},
	{"spice",     2, SPICE},
	{"cdl",       2, CDL},
	{"verilog",   2, VERILOG},
	{NULL, 0, 0}  /* 0 */
};

       TOOL      *sim_tool;					/* the Simulator tool object */
       INTBIG     sim_formatkey;			/* key for "SIM_format" (ESIM, etc) */
       INTBIG     sim_netfilekey;			/* key for "SIM_netfile" */
       INTBIG     sim_dontrunkey;			/* key for "SIM_dontrun" */
       INTBIG     sim_weaknodekey;			/* key for "SIM_weak_node" */
       INTBIG     sim_spice_partskey;		/* key for "SIM_spice_parts" */
       char      *sim_spice_parts = 0;		/* cached value for "SIM_spice_parts" */
       NODEPROTO *sim_simnt;				/* facet being simulated */
       int        sim_tosim[2];				/* pipe to the simulator process */
       int        sim_fromsim[2];			/* pipe from the simulator process */
       INTBIG     sim_process;				/* process number of simulator */
       INTBIG     sim_pseudonet;			/* for internal networks */
static BOOLEAN    sim_circuitchanged;		/* true if circuit being simulated was changed */
static BOOLEAN    sim_undoredochange;		/* true if change comes from undo/redo */
static ARCINST   *sim_modifyarc;			/* arc whose userbits are being changed */
static INTBIG     sim_modifyarcbits;		/* former value of userbits on changed arc */
static LIBRARY   *sim_curlib;				/* for listing facets alphabetically in dialogs */
static NODEPROTO *sim_oldfacetprotos;		/* for listing facets alphabetically in dialogs */
       INTBIG     sim_filetypeesim;			/* ESIM netlist file descriptor */
       INTBIG     sim_filetypefasthenry;	/* FastHenry netlist file descriptor */
       INTBIG     sim_filetypemossim;		/* MOSSIM netlist file descriptor */
       INTBIG     sim_filetypepal;			/* PAL netlist file descriptor */
       INTBIG     sim_filetypeals;			/* ALS netlist file descriptor */
       INTBIG     sim_filetypealsvec;		/* ALS vectors file descriptor */
       INTBIG     sim_filetypeirsimcmd;		/* IRSIM command (vectors) file descriptor */
       INTBIG     sim_filetypenetlisp;		/* Netlisp netlist file descriptor */
       INTBIG     sim_filetypequisc;		/* QUISC netlist file descriptor */
       INTBIG     sim_filetypersim;			/* RSIM netlist file descriptor */
       INTBIG     sim_filetypeirsim;		/* IRSIM netlist file descriptor */
       INTBIG     sim_filetypeirsimparam;	/* IRSIM parameter file descriptor */
       INTBIG     sim_filetypesilos;		/* Silos netlist file descriptor */
       INTBIG     sim_filetypespice;		/* SPICE input file descriptor */
       INTBIG     sim_filetypespicecmd;		/* SPICE command file descriptor */
       INTBIG     sim_filetypespiceout;		/* SPICE output file descriptor */
       INTBIG     sim_filetypehspiceout;	/* HSPICE output file descriptor */
       INTBIG     sim_filetyperawspiceout;	/* SPICE raw output disk file descriptor */
       INTBIG     sim_filetypecdl;			/* CDL output file descriptor */
       INTBIG     sim_filetypectemp;		/* CDL template disk file descriptor */
       INTBIG     sim_filetypetegas;		/* Tegas netlist file descriptor */
       INTBIG     sim_filetypetegastab;		/* Tegas table file descriptor */
       INTBIG     sim_filetypeverilog;		/* Verilog file descriptor */

void sim_init(INTBIG *argc, char *argv[], TOOL *thistool)
{
	/* nothing on pass 2 or 3 of initialization */
	if (thistool == NOTOOL || thistool == 0) return;

	/* pass 1 initialization */
	sim_tool = thistool;
	sim_dontrunkey = makekey("SIM_dontrun");
	sim_weaknodekey = makekey("SIM_weak_node");
	sim_netfilekey = makekey("SIM_netfile");

	/* create disk file descriptors */
	sim_filetypeesim        = setupfiletype("sim", "*.sim",       MACFSTAG('TEXT'), FALSE, "esim", _("ESIM netlist"));
	sim_filetypefasthenry   = setupfiletype("inp", "*.inp",       MACFSTAG('TEXT'), FALSE, "fasthenry", _("FastHenry netlist"));
	sim_filetypehspiceout   = setupfiletype("tr0", "*.tr0",       MACFSTAG('TEXT'), TRUE,  "hspiceout", _("HSPICE output"));
	sim_filetypemossim      = setupfiletype("ntk", "*.ntk",       MACFSTAG('TEXT'), FALSE, "mossim", _("MOSSIM netlist"));
	sim_filetypepal         = setupfiletype("pal", "*.pal",       MACFSTAG('TEXT'), FALSE, "pal", _("Abel PAL"));
	sim_filetypeals         = setupfiletype("net", "*.net",       MACFSTAG('TEXT'), FALSE, "als", _("ALS netlist"));
	sim_filetypealsvec      = setupfiletype("vec", "*.vec",       MACFSTAG('TEXT'), FALSE, "alsvec", _("ALS vector"));
	sim_filetypeirsimcmd    = setupfiletype("cmd", "*.cmd",       MACFSTAG('TEXT'), FALSE, "irsimcmd", _("IRSIM vector"));
	sim_filetypenetlisp     = setupfiletype("net", "*.net",       MACFSTAG('TEXT'), FALSE, "netlisp", _("NetLisp netlist"));
	sim_filetypequisc       = setupfiletype("sci", "*.sci",       MACFSTAG('TEXT'), FALSE, "quisc", _("QUISC netlist"));
	sim_filetypersim        = setupfiletype("sim", "*.sim",       MACFSTAG('TEXT'), FALSE, "rsim", _("RSIM netlist"));
	sim_filetypeirsim       = setupfiletype("sim", "*.sim",       MACFSTAG('TEXT'), FALSE, "irsim", _("IRSIM netlist"));
	sim_filetypeirsimparam  = setupfiletype("prm", "*.prm",       MACFSTAG('TEXT'), FALSE, "irsimparam", _("IRSIM parameters"));
	sim_filetypesilos       = setupfiletype("sil", "*.sil",       MACFSTAG('TEXT'), FALSE, "silos", _("SILOS netlist"));
	sim_filetypespice       = setupfiletype("spi", "*.spi;*.spo", MACFSTAG('TEXT'), FALSE, "spice", _("SPICE netlist"));
	sim_filetypespicecmd    = setupfiletype("cmd.sp","*.cmd.sp",  MACFSTAG('TEXT'), FALSE, "spicecmd", _("SPICE command"));
	sim_filetypespiceout    = setupfiletype("spo", "*.spo",       MACFSTAG('TEXT'), FALSE, "spiceout", _("SPICE output"));
	sim_filetyperawspiceout = setupfiletype("raw", "*.raw",       MACFSTAG('TEXT'), FALSE, "spicerawout", _("SPICE raw output"));
	sim_filetypecdl         = setupfiletype("cdl", "*.cdl",       MACFSTAG('TEXT'), FALSE, "cspice", _("CDL netlist"));
	sim_filetypectemp       = setupfiletype("cdltemplate", "*.cdltemplate", MACFSTAG('TEXT'), FALSE, "cdltemplate", _("CDL template"));
	sim_filetypetegas       = setupfiletype("tdl", "*.tdl",       MACFSTAG('TEXT'), FALSE, "tegas", _("TEGAS netlist"));
	sim_filetypetegastab    = setupfiletype("",    "*.*",         MACFSTAG('TEXT'), FALSE, "tegastab", _("TEGAS table"));
	sim_filetypeverilog     = setupfiletype("v",   "*.v;*.ver",   MACFSTAG('TEXT'), FALSE, "verilog", _("Verilog"));

	/* initialize default simulator format */
	changesquiet(TRUE);
	sim_formatkey = makekey("SIM_format");
	(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_formatkey, ALS, VINTEGER|VDONTSAVE);

	/* do not have the simulator automatically run */
	(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_dontrunkey, SIMRUNNO, VINTEGER|VDONTSAVE);

	/* ESIM/RSIM/RNL initialization */
	(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_netfilekey, (INTBIG)"", VSTRING|VDONTSAVE);

	/* IRSIM initializatio */
	sim_irsim_statekey = makekey("SIM_irsim_state");
	(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_irsim_statekey, DEFIRSIMSTATE,
		VINTEGER|VDONTSAVE);

	/* SPICE initialization */
	sim_spice_partskey = makekey("SIM_spice_parts");
	sim_spice_listingfilekey = makekey("SIM_listingfile");
	sim_spice_runargskey = makekey("SIM_spice_runarguments");
	sim_spice_levelkey = makekey("SIM_spice_level");
	sim_spice_statekey = makekey("SIM_spice_state");
	sim_spice_nameuniqueid = makekey("SIM_spice_nameuniqueid");
	(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_spice_listingfilekey, (INTBIG)"", VSTRING|VDONTSAVE);
	(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_spice_statekey, SPICERESISTANCE|SPICENODENAMES|SPICE3,
		VINTEGER|VDONTSAVE);
	(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_spice_levelkey, 1, VINTEGER|VDONTSAVE);
	(void)allocstring(&sim_spice_parts, "spiceparts", sim_tool->cluster);
	(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_spice_partskey, (INTBIG)sim_spice_parts,
		VSTRING|VDONTSAVE);
	DiaDeclareHook("spice", &sim_spicep, sim_spicedlog);
	changesquiet(FALSE);

	/* VERILOG initialization */
	sim_verilog_statekey = makekey("SIM_verilog_state");
	DiaDeclareHook("verilog", &sim_verp, sim_verilogdlog);

	/* FastHenry initialization */
	sim_fasthenrystatekey = makekey("SIM_fasthenry_state");
	sim_fasthenryfreqstartkey = makekey("SIM_fasthenry_freqstart");
	sim_fasthenryfreqendkey = makekey("SIM_fasthenry_freqend");
	sim_fasthenryrunsperdecadekey = makekey("SIM_fasthenry_runsperdecade");
	sim_fasthenrynumpoleskey = makekey("SIM_fasthenry_numpoles");
	sim_fasthenryseglimitkey = makekey("SIM_fasthenry_seglimit");
	sim_fasthenrythicknesskey = makekey("SIM_fasthenry_thickness");
	sim_fasthenrywidthsubdivkey = makekey("SIM_fasthenry_width_subdivs");
	sim_fasthenryheightsubdivkey = makekey("SIM_fasthenry_height_subdivs");
	sim_fasthenryzheadkey = makekey("SIM_fasthenry_z_head");
	sim_fasthenryztailkey = makekey("SIM_fasthenry_z_tail");
	sim_fasthenrygroupnamekey = makekey("SIM_fasthenry_group_name");
	sim_fasthenryinit();

	/* initialize the simulation window system */
	sim_window_init();

	/* miscellaneous initialization */
	sim_process = -1;
	sim_simnt = NONODEPROTO;
	sim_circuitchanged = FALSE;
	sim_undoredochange = FALSE;
	DiaDeclareHook("simopt", &sim_simulatorp, sim_optionsdlog);
}

void sim_done(void)
{
	REGISTER VARIABLE *var;

	if (sim_process >= 0)
	{
		ekill(sim_process);
		ewait(sim_process);

		var = getvalkey((INTBIG)sim_tool, VTOOL, VSTRING, sim_netfilekey);
		if (var != NOVARIABLE)
			ttyputmsg(_("Simulation net list saved in '%s'"), (char *)var->addr);
	}

#ifdef DEBUGMEMORY
	/* free all memory */
	if (sim_spice_parts != 0) efree((char *)sim_spice_parts);
	simals_term();
	sim_window_term();
	sim_freespicememory();
	sim_freespicerun_memory();
	sim_freeirsimmemory();
	sim_freeverilogmemory();
#  if SIMTOOLIRSIM != 0
	irsim_freememory();
#  endif
#endif
}

void sim_set(INTBIG count, char *par[])
{
	REGISTER INTBIG i, j, l, newformat, amount, curtop, internal, tr;
	REGISTER BOOLEAN (*charhandlerschem)(WINDOWPART*, INTSML, INTBIG),
		(*charhandlerwave)(WINDOWPART*, INTSML, INTBIG);
	REGISTER BOOLEAN (*startsimulation)(NODEPROTO*);
	REGISTER INTBIG spicestate, verilogstate, signals, options, value,
		active, vis;
	float fvalue;
	REGISTER char *pp;
	REGISTER VARIABLE *var;
	REGISTER ARCINST *ai;
	NODEPROTO *np, *simnp;
	float factor;
	double time, size, maintime, extensiontime, maxtime, mintime;

	if (count == 0)
	{
		ttyputusage("telltool simulation OPTIONS");
		return;
	}
	l = strlen(pp = par[0]);

	if (namesamen(pp, "simulate", l) == 0 && l >= 2)
	{
		if (count < 2)
		{
			ttyputusage("telltool simulation simulate FORMAT");
			return;
		}

		/* get the simulator name */
		internal = 0;
		l = strlen(pp = par[1]);
		if (namesame(pp, "internal") == 0)
		{
			internal = 1;
			switch (sim_window_state&SIMENGINE)
			{
				case SIMENGINEALS:
					l = strlen(pp = "als");
					break;
				case SIMENGINEIRSIM:
					l = strlen(pp = "irsim");
#if SIMTOOLIRSIM == 0
					ttyputmsg(_("IRSIM is not built-in: just writing the deck"));
					internal = 0;
#endif
					break;
			}
		}
		for(i=0; sim_simtable[i].simname != 0; i++)
			if (namesamen(pp, sim_simtable[i].simname, l) == 0 && l >= sim_simtable[i].min) break;
		if (sim_simtable[i].simname == 0)
		{
			ttyputbadusage("telltool simulation simulate");
			return;
		}
		newformat = sim_simtable[i].format;

		/* see if there is already a simulation running */
		if (sim_process != -1)
		{
			var = getvalkey((INTBIG)sim_tool, VTOOL, VINTEGER, sim_formatkey);
			if (var != NOVARIABLE && newformat == var->addr)
			{
				ttyputmsg(_("This simulator is already running"));
				return;
			}
			ttyputerr(_("Cannot switch simulators while one is running"));
			return;
		}

		/* make sure there is a facet to simulate */
		np = getcurfacet();
		if (np == NONODEPROTO)
		{
			ttyputerr(_("No current facet to simulate"));
			return;
		}

		/* remember the currently running format */
		(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_formatkey, newformat, VINTEGER|VDONTSAVE);

		/* write the appropriate simulation file */
		switch (newformat)
		{
			case ABEL:
				sim_writepalnetlist(np);
				break;
			case FASTHENRY:
				sim_writefasthenrynetlist(np);
				break;
			case COSMOS:
			case ESIM:
			case RNL:
			case RSIM:
				sim_writesim(np, newformat);
				break;
			case MOSSIM:
				sim_writemossim(np);
				break;
			case SILOS:
				sim_writesilnetlist(np);
				break;
			case SPICE:
				sim_writespice(np, FALSE);
				break;
			case CDL:
				sim_writespice(np, TRUE);
				break;
			case TEXSIM:
				sim_writetexnetlist(np);
				break;
			case VERILOG:
				sim_writevernetlist(np);
				break;
			case IRSIM:
#if SIMTOOLIRSIM != 0
				if (internal != 0)
				{
					charhandlerschem = irsim_charhandlerschem;
					charhandlerwave = irsim_charhandlerwave;
					startsimulation = irsim_startsimulation;
				} else
#endif
				{
					sim_writeirsim(np);
				}
				break;
			case ALS:
				charhandlerschem = simals_charhandlerschem;
				charhandlerwave = simals_charhandlerwave;
				startsimulation = simals_startsimulation;
				internal = 1;
				break;
		}

		if (internal != 0)
		{
			active = sim_window_isactive(&simnp);
			if (active != 0 && simnp != np)
			{
				if ((active&SIMWINDOWWAVEFORM) != 0)
				{
					ttyputerr(_("Close all simulation windows before simulating another facet"));
					return;
				}
				sim_window_stopsimulation();
				active = 0;
			}

			if (active != 0)
			{
				/* display schematic window if it is not up */
				if ((active&SIMWINDOWSCHEMATIC) == 0)
				{
					(void)sim_window_create(0, simnp, 0, charhandlerschem, newformat);
					return;
				}

				/* display waveform window if it is not up */
				if ((active&SIMWINDOWWAVEFORM) == 0 && (sim_window_state&SHOWWAVEFORM) != 0)
				{
					(void)sim_window_create(0, simnp, charhandlerwave, 0, newformat);
					return;
				}
				ttyputmsg(_("Simulation is already running"));
				return;
			}

			/* do not simulate icons */
			if (np->cellview == el_iconview)
			{
				ttyputerr(_("Cannot simulate an icon"));
				return;
			}

			/* start simulation of the facet */
			if ((*startsimulation)(np)) return;

			/* turn on this tool so that it can track changes to original circuit */
			toolturnon(sim_tool);
		}
		return;
	}

	if (namesamen(pp, "resume", l) == 0 && l >= 1)
	{
		var = getvalkey((INTBIG)sim_tool, VTOOL, VINTEGER, sim_formatkey);
		if (var == NOVARIABLE)
		{
			ttyputerr(_("No simulation is running"));
			return;
		}
		switch (var->addr)
		{
			case RNL:
			case RSIM:
			case ESIM:
				sim_resumesim(FALSE);
				break;
			default:
				ttyputerr(_("Cannot resume this simulator"));
				break;
		}
		return;
	}

	if (namesamen(pp, "no-execute", l) == 0 && l >= 1)
	{
		(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_dontrunkey, SIMRUNNO, VINTEGER);
		ttyputmsg(M_("Simulator will not be invoked (deck generation only)"));
		return;
	}
	if (namesamen(pp, "execute-only", l) == 0 && l >= 9)
	{
		(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_dontrunkey, SIMRUNYES, VINTEGER);
		ttyputmsg(M_("Simulator will be invoked"));
		return;
	}
	if (namesamen(pp, "execute-and-parse", l) == 0 && l >= 9)
	{
		(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_dontrunkey, SIMRUNYESPARSE, VINTEGER);
		ttyputmsg(M_("Simulator will be invoked, output parsed"));
		return;
	}
	if (namesamen(pp, "execute-quietly-and-parse", l) == 0 && l >= 16)
	{
		(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_dontrunkey, SIMRUNYESQPARSE, VINTEGER);
		ttyputmsg(M_("Simulator will be invoked quietly, output parsed"));
		return;
	}
	if (namesamen(pp, "execute-quietly", l) == 0 && l >= 9)
	{
		(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_dontrunkey, SIMRUNYESQ, VINTEGER);
		ttyputmsg(M_("Simulator will be invoked quietly, output parsed"));
		return;
	}

	if (namesamen(pp, "pointout", l) == 0 && l >= 1)
	{
		if (count <= 1)
		{
			ttyputusage("telltool simulation pointout ELEMENT");
			return;
		}
		var = getvalkey((INTBIG)sim_tool, VTOOL, VINTEGER, sim_formatkey);
		if (var != NOVARIABLE) switch (var->addr)
		{
			case RNL:
			case RSIM:
			case ESIM:
			case COSMOS: 
				sim_simpointout(par[1], var->addr);
				return;
		}
		ttyputerr(M_("Current simulator cannot pointout netlist objects"));
		return;
	}

	if (namesamen(pp, "window", l) == 0 && l >= 1)
	{
		if (count <= 1)
		{
			ttyputusage("telltool simulation window COMMAND");
			return;
		}
		l = strlen(pp = par[1]);

		if (namesamen(pp, "vector-clear", l) == 0)
		{
			switch (sim_window_state&SIMENGINE)
			{
				case SIMENGINEALS:
					par[0] = "vector";
					par[1] = "new";
					simals_com_comp(2, par);
					break;
#if SIMTOOLIRSIM != 0
				case SIMENGINEIRSIM:
					irsim_clearallvectors();
					break;
#endif
			}
			return;
		}
		if (namesamen(pp, "vector-load", l) == 0)
		{
			switch (sim_window_state&SIMENGINE)
			{
				case SIMENGINEALS:
					par[0] = "vector";
					par[1] = "load";
					simals_com_comp(count, par);
					break;
#if SIMTOOLIRSIM != 0
				case SIMENGINEIRSIM:
					irsim_loadvectorfile((count>2) ? par[2] : 0);
					break;
#endif
			}
			return;
		}
		if (namesamen(pp, "vector-save", l) == 0)
		{
			switch (sim_window_state&SIMENGINE)
			{
				case SIMENGINEALS:
					par[0] = "vector";
					par[1] = "save";
					simals_com_comp(count, par);
					break;
#if SIMTOOLIRSIM != 0
				case SIMENGINEIRSIM:
					irsim_savevectorfile((count>2) ? par[2] : 0);
					break;
#endif
			}
			return;
		}
		if (namesamen(pp, "vector-writespice", l) == 0)
		{
			sim_window_writespicecmd();
			return;
		}
		if (namesamen(pp, "level", l) == 0 && l >= 1)
		{
			if (count <= 2)
			{
				ttyputusage("telltool simulation window level COMMAND");
				return;
			}
			l = strlen(pp = par[2]);
			if (namesamen(pp, "up", l) == 0 && l >= 1)
			{
				switch (sim_window_state&SIMENGINE)
				{
					case SIMENGINEALS:
						simals_level_up_command();
						break;
#if SIMTOOLIRSIM != 0
					case SIMENGINEIRSIM:
						irsim_level_up();
						break;
#endif
				}
				return;
			}
			if (namesamen(pp, "set", l) == 0 && l >= 1)
			{
				if (count <= 3)
				{
					count = ttygetparam(M_("Instance to simulate: "), &windowlevelsp, 3, &par[3]) + 3;
					if (count <= 3) return;
				}
				switch (sim_window_state&SIMENGINE)
				{
					case SIMENGINEALS:
						simals_level_set_command(par[3]);
						break;
#if SIMTOOLIRSIM != 0
					case SIMENGINEIRSIM:
						irsim_level_set(par[3]);
						break;
#endif
				}
				return;
			}
			ttyputbadusage("telltool simulation window level");
			return;
		}

		if (namesamen(pp, "2-state-display", l) == 0 && l >= 1)
		{
			sim_window_setstate(sim_window_state & ~FULLSTATE);
			ttyputverbose(M_("Simulation will show only 2 states in schematics window"));
			return;
		}
		if (namesamen(pp, "12-state-display", l) == 0 && l >= 1)
		{
			sim_window_setstate(sim_window_state | FULLSTATE);
			ttyputverbose(M_("Simulation will show all 12 states in schematics window"));
			return;
		}
		if (namesamen(pp, "advance-time", l) == 0 && l >= 1)
		{
			sim_window_setstate(sim_window_state | ADVANCETIME);
			ttyputverbose(M_("State changes will advance time to end of activity"));
			return;
		}
		if (namesamen(pp, "freeze-time", l) == 0 && l >= 1)
		{
			sim_window_setstate(sim_window_state & ~ADVANCETIME);
			ttyputverbose(M_("State changes will not advance time"));
			return;
		}
		if (namesamen(pp, "display-waveform", l) == 0 && l >= 1)
		{
			sim_window_setstate(sim_window_state | SHOWWAVEFORM);
			ttyputverbose(M_("Simulation will show waveform window"));
			return;
		}
		if (namesamen(pp, "ignore-waveform", l) == 0 && l >= 1)
		{
			sim_window_setstate(sim_window_state & ~SHOWWAVEFORM);
			ttyputverbose(M_("Simulation will not show waveform window"));
			return;
		}
		if (namesamen(pp, "color", l) == 0 && l >= 2)
		{
			if (count <= 2)
			{
				sim_window_displaycolor(99, 0);
				return;
			}
			l = strlen(pp = par[2]);
			if (count < 4)
			{
				ttyputusage("telltool simulation window color SIGNAL COLORNAME");
				return;
			}
			i = getecolor(par[3]);
			if (i < 0)
			{
				ttyputbadusage("telltool simulation window color");
				return;
			}
			if (namesamen(pp, "off", l) == 0)
			{
				sim_window_displaycolor(OFF_STRENGTH, i);
				return;
			}
			if (namesamen(pp, "node", l) == 0)
			{
				sim_window_displaycolor(NODE_STRENGTH, i);
				return;
			}
			if (namesamen(pp, "gate", l) == 0)
			{
				sim_window_displaycolor(GATE_STRENGTH, i);
				return;
			}
			if (namesamen(pp, "power", l) == 0)
			{
				sim_window_displaycolor(VDD_STRENGTH, i);
				return;
			}
			if (namesamen(pp, "low", l) == 0)
			{
				sim_window_displaycolor(LOGIC_LOW, i);
				return;
			}
			if (namesamen(pp, "high", l) == 0)
			{
				sim_window_displaycolor(LOGIC_HIGH, i);
				return;
			}
			if (namesamen(pp, "undefined", l) == 0)
			{
				sim_window_displaycolor(LOGIC_X, i);
				return;
			}
			ttyputbadusage("telltool simulation window color");
			return;
		}
		if (namesamen(pp, "cursor", l) == 0 && l >= 2)
		{
			if (sim_window_isactive(&np) == 0)
			{
				ttyputerr(M_("No simulator active"));
				return;
			}
			if (count < 3)
			{
				ttyputusage("telltool simulation window cursor COMMAND");
				return;
			}
			if (namesamen(par[2], "center", strlen(par[2])) == 0)
			{
				/* center the cursors visibly */
				sim_window_getaveragetimerange(&mintime, &maxtime);
				size = maxtime - mintime;
				time = (mintime + maxtime) / 2.0;
				sim_window_setmaincursor(time - size/4.0);
				sim_window_setextensioncursor(time + size/4.0);
				return;
			}
			if (count < 4)
			{
				ttyputusage("telltool simulation window cursor WHICH TIME");
				return;
			}
			time = atof(par[3]);
			if (time < 0.0)
			{
				ttyputerr(M_("Warning: time cannot be negative, set to 0 sec."));
				time = 0.0;
			}
			if (namesamen(par[2], "main", strlen(par[2])) == 0)
			{
				sim_window_setmaincursor(time);
				return;
			}
			if (namesamen(par[2], "extension", strlen(par[2])) == 0)
			{
				sim_window_setextensioncursor(time);
				return;
			}
			ttyputbadusage("telltool simulation window cursor");
			return;
		}
		if (namesamen(pp, "move", l) == 0 && l >= 1)
		{
			if ((sim_window_isactive(&np)&SIMWINDOWWAVEFORM) == 0)
			{
				ttyputerr(M_("No simulator waveform window is active"));
				return;
			}
			if (count < 3)
			{
				ttyputusage("telltool simulation window move (left | right | up | down) [AMOUNT]");
				return;
			}

			l = strlen(pp = par[2]);
			if (namesamen(pp, "left", l) == 0 || namesamen(pp, "right", l) == 0)
			{
				/* determine amount to move time */
				sim_window_getaveragetimerange(&mintime, &maxtime);
				size = maxtime - mintime;
				if (count < 4) time = size / 2.0f; else
				{
					time = atof(par[3]);
				}
				if (time <= 0.0)
				{
					ttyputerr(M_("Window movement time must be greater than 0 seconds"));
					return;
				}

				sim_window_inittraceloop();
				for(j=0; ; j++)
				{
					tr = sim_window_nexttraceloop();
					if (tr == 0) break;
					sim_window_gettimerange(tr, &mintime, &maxtime);
					if (namesamen(pp, "left", l) == 0)
					{
						mintime += time;
					} else
					{
						mintime -= time;
						if (mintime < 0.0) mintime = 0.0;
					}
					maxtime = mintime + size;
					sim_window_settimerange(tr, mintime, maxtime);
				}
				if (sim_window_format == ALS) (void)simals_initialize_simulator(TRUE); else
					sim_window_redraw();
				return;
			}
			if (namesamen(pp, "up", l) == 0 || namesamen(pp, "down", l) == 0)
			{
				curtop = sim_window_gettopvisframe();
				vis = sim_window_getnumvisframes();
				signals = sim_window_getnumframes();
				if (count < 4) amount = vis-1; else
				{
					amount = myatoi(par[3]);
					if (amount <= 0)
					{
						ttyputerr(M_("Window movement time must be greater than 0 signals"));
						return;
					}
				}
				if (namesamen(pp, "up", l) == 0)
				{
					curtop += amount;
					if (curtop + vis > signals) curtop = signals - vis;
				} else
				{
					curtop -= amount;
					if (curtop < 0) curtop = 0;
				}
				sim_window_settopvisframe(curtop);
				sim_window_redraw();
				return;
			}

			ttyputbadusage("telltool simulation window move");
			return;
		}
		if (namesamen(pp, "traces", l) == 0 && l >= 1)
		{
			if ((sim_window_isactive(&np)&SIMWINDOWWAVEFORM) == 0)
			{
				ttyputerr(M_("No simulator waveform window is active"));
				return;
			}
			if (count < 3)
			{
				ttyputusage("telltool simulation window traces (more | less | set A)");
				return;
			}
			l = strlen(pp = par[2]);

			if (namesamen(pp, "more", l) == 0)
			{
				sim_window_setnumframes(sim_window_getnumframes()+1);
				sim_window_redraw();
				return;
			}
			if (namesamen(pp, "less", l) == 0)
			{
				i = sim_window_getnumframes();
				if (i <= 1)
				{
					ttyputerr(M_("Must be at least 1 signal on the display"));
					return;
				}
				sim_window_setnumframes(i+1);
				sim_window_redraw();
				return;
			}
			if (namesamen(pp, "set", l) == 0)
			{
				if (count < 4)
				{
					ttyputusage("telltool simulation window traces set AMOUNT");
					return;
				}
				i = myatoi(par[3]);
				if (i < 1)
				{
					ttyputerr(M_("Must be at least 1 signal on the display"));
					return;
				}
				sim_window_setnumframes(i);
				sim_window_redraw();
				return;
			}
			ttyputbadusage("telltool simulation window traces");
			return;
		}
		if (namesamen(pp, "zoom", l) == 0 && l >= 1)
		{
			if ((sim_window_isactive(&np)&SIMWINDOWWAVEFORM) == 0)
			{
				ttyputerr(M_("No simulator waveform window is active"));
				return;
			}
			if (count < 3)
			{
				ttyputusage("telltool simulation window zoom (in | out | window | cursor)");
				return;
			}
			l = strlen(pp = par[2]);

			if (namesamen(pp, "cursor", l) == 0)
			{
				maintime = sim_window_getmaincursor();
				extensiontime = sim_window_getextensioncursor();
				if (maintime == extensiontime) return;
				if (maintime > extensiontime)
				{
					size = (maintime-extensiontime) / 20.0;
					maxtime = maintime + size;
					mintime = extensiontime - size;
				} else
				{
					size = (extensiontime-maintime) / 20.0;
					maxtime = extensiontime + size;
					mintime = maintime - size;
				}
				sim_window_settimerange(0, mintime, maxtime);
				if (sim_window_format == ALS) (void)simals_initialize_simulator(TRUE); else
					sim_window_redraw();
				return;
			}

			if (namesamen(pp, "in", l) == 0)
			{
				if (count < 4) factor = 2.0; else
				{
					factor = (float)myatoi(par[3]);
					if (factor <= 1)
					{
						ttyputerr(M_("Zoom factor must be integer greater than 1"));
						return;
					}
				}

				sim_window_getaveragetimerange(&mintime, &maxtime);
				size = (maxtime - mintime) / factor;
				sim_window_inittraceloop();
				for(j=0; ; j++)
				{
					tr = sim_window_nexttraceloop();
					if (tr == 0) break;
					sim_window_gettimerange(tr, &mintime, &maxtime);
					mintime = sim_window_getmaincursor() - (size / 2.0);
					if (mintime < 0.0) mintime = 0.0;
					maxtime = mintime + size;
					sim_window_settimerange(tr, mintime, maxtime);
				}
				sim_window_redraw();
				return;
			}

			if (namesamen(pp, "out", l) == 0)
			{
				if (count < 4) factor = 2.0; else
				{
					factor = (float)myatoi(par[3]);
					if (factor <= 1)
					{
						ttyputerr(M_("Zoom factor must be integer greater than 1"));
						return;
					}
				}

				sim_window_getaveragetimerange(&mintime, &maxtime);
				size = (maxtime - mintime) * factor;
				sim_window_inittraceloop();
				for(j=0; ; j++)
				{
					tr = sim_window_nexttraceloop();
					if (tr == 0) break;
					sim_window_gettimerange(tr, &mintime, &maxtime);
					mintime = sim_window_getmaincursor() - (size / 2.0);
					if (mintime < 0.0) mintime = 0.0;
					maxtime = mintime + size;
					sim_window_settimerange(tr, mintime, maxtime);
				}
				if (sim_window_format == ALS) (void)simals_initialize_simulator(TRUE); else
					sim_window_redraw();
				return;
			}
			if (namesamen(pp, "all-displayed", l) == 0)
			{
				sim_window_gettimeextents(&mintime, &maxtime);
				sim_window_settimerange(0, mintime, maxtime);
				if (sim_window_format == ALS) (void)simals_initialize_simulator(TRUE); else
					sim_window_redraw();
				return;
			}
			if (namesamen(pp, "window", l) == 0)
			{
				if (count < 4)
				{
					ttyputusage("telltool simulation window zoom window MIN [MAX]");
					return;
				}

				mintime = atof(par[3]);
				maxtime = 0.0;
				if (count > 4) maxtime = atof(par[4]);
				if (mintime == maxtime)
				{
					ttyputerr(M_("ERROR: Window size must be greater than 0"));
					return;
				}
				if (mintime > maxtime)
				{
					time = maxtime;
					maxtime = mintime;
					mintime = time;
				}
				sim_window_settimerange(0, mintime, maxtime);
				if (sim_window_format == ALS) (void)simals_initialize_simulator(TRUE); else
					sim_window_redraw();
				return;
			}

			ttyputbadusage("telltool simulation window zoom");
			return;
		}
		ttyputbadusage("telltool simulation window");
		return;
	}

	if (namesamen(pp, "als", l) == 0 && l >= 1)
	{
		simals_com_comp(count-1, &par[1]);
		return;
	}

	if (namesamen(pp, "spice", l) == 0 && l >= 2)
	{
		l = strlen(pp = par[1]);
		if (namesamen(pp, "level", l) == 0)
		{
			if (count >= 3)
			{
				i = atoi(par[2]);
				if (i <= 0 || i > MAXSPICELEVEL) i = 1;
				(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_spice_levelkey, i, VINTEGER);
			}
			var = getvalkey((INTBIG)sim_tool, VTOOL, VINTEGER, sim_spice_levelkey);
			if (var == NOVARIABLE) i = 1; else i = var->addr;
			ttyputverbose(M_("Simulation level set to %ld"), i);
			return;
		}
		if (namesamen(pp, "save-output", l) == 0)
		{
			if (count < 3)
			{
				ttyputerr(M_("Should supply a file name to write"));
				par[2] = "electric_def.spo";
			}
			(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_spice_listingfilekey, (INTBIG)par[2], VSTRING|VDONTSAVE);
			ttyputverbose(M_("Simulation output will go to file %s"), par[2]);
			return;
		}
		if (namesamen(pp, "format", l) == 0)
		{
			var = getvalkey((INTBIG)sim_tool, VTOOL, VINTEGER, sim_spice_statekey);
			if (var != NOVARIABLE) spicestate = var->addr; else spicestate = 0;
			if (count <= 2)
			{
				switch (spicestate&SPICETYPE)
				{
					case SPICE2:      ttyputmsg(M_("SPICE decks for SPICE 2"));   break;
					case SPICE3:      ttyputmsg(M_("SPICE decks for SPICE 3"));   break;
					case SPICEHSPICE: ttyputmsg(M_("SPICE decks for HSPICE"));    break;
					case SPICEPSPICE: ttyputmsg(M_("SPICE decks for PSPICE"));    break;
				}
				return;
			}

			l = strlen(pp = par[2]);
			if (namesamen(pp, "2", l) == 0)
			{
				(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_spice_statekey,
					(spicestate & ~SPICETYPE) | SPICE2, VINTEGER);
				return;
			}
			if (namesamen(pp, "3", l) == 0)
			{
				(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_spice_statekey,
					(spicestate & ~SPICETYPE) | SPICE3, VINTEGER);
				return;
			}
			if (namesamen(pp, "hspice", l) == 0)
			{
				(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_spice_statekey,
					(spicestate & ~SPICETYPE) | SPICEHSPICE, VINTEGER);
				return;
			}
			if (namesamen(pp, "pspice", l) == 0)
			{
				(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_spice_statekey,
					(spicestate & ~SPICETYPE) | SPICEPSPICE, VINTEGER);
				return;
			}
		}
		if (namesamen(pp, "parse-output", l) == 0)
		{
			if (count < 3)
			{
				ttyputerr(M_("Must supply a SPICE output file to read"));
				return;
			}
			sim_spice_execute("", par[2], NONODEPROTO);
			return;
		}
		if (namesamen(pp, "not", l) == 0)
		{
			if (count <= 2)
			{
				ttyputusage("telltool simulation spice not OPTION");
				return;
			}
			l = strlen(pp = par[2]);
			if (namesamen(pp, "resistance", l) == 0 || namesamen(pp, "capacitances", l) == 0)
			{
				var = getvalkey((INTBIG)sim_tool, VTOOL, VINTEGER, sim_spice_statekey);
				if (var != NOVARIABLE)
					(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_spice_statekey,
						var->addr & ~SPICERESISTANCE, VINTEGER);
				ttyputverbose(M_("SPICE decks will not print parasitics"));
				return;
			}
			if (namesamen(pp, "plot", l) == 0)
			{
				var = getvalkey((INTBIG)sim_tool, VTOOL, VINTEGER, sim_spice_statekey);
				if (var != NOVARIABLE)
					(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_spice_statekey,
						var->addr & ~SPICEPLOT, VINTEGER);
				ttyputverbose(M_("SPICE decks will not do plots"));
				return;
			}
			if (namesamen(pp, "rawfile", l) == 0)
			{
				var = getvalkey((INTBIG)sim_tool, VTOOL, VINTEGER, sim_spice_statekey);
				if (var != NOVARIABLE)
					(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_spice_statekey,
						var->addr & ~SPICERAWFILE, VINTEGER);
				ttyputverbose(M_("Reads SPICE standard output (not Rawfile)"));
				return;
			}
			if (namesamen(pp, "global-pwr-gnd", l) == 0)
			{
				var = getvalkey((INTBIG)sim_tool, VTOOL, VINTEGER, sim_spice_statekey);
				if (var != NOVARIABLE)
					(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_spice_statekey,
						var->addr & ~SPICEGLOBALPG, VINTEGER);
				ttyputverbose(M_("Does not write global Power and Ground"));
				return;
			}
			if (namesamen(pp, "use-nodenames", l) == 0)
			{
				var = getvalkey((INTBIG)sim_tool, VTOOL, VINTEGER, sim_spice_statekey);
				if (var != NOVARIABLE)
					(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_spice_statekey,
						var->addr & ~SPICENODENAMES, VINTEGER);
				ttyputverbose(M_("SPICE decks will contain node numbers"));
				return;
			}
			ttyputbadusage("telltool simulation spice not");
			return;
		}
		if (namesamen(pp, "resistance", l) == 0 || namesamen(pp, "capacitances", l) == 0)
		{
			var = getvalkey((INTBIG)sim_tool, VTOOL, VINTEGER, sim_spice_statekey);
			if (var != NOVARIABLE)
				(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_spice_statekey,
					var->addr | SPICERESISTANCE, VINTEGER);
			ttyputverbose(M_("SPICE decks will print parasitics"));
			return;
		}
		if (namesamen(pp, "plot", l) == 0)
		{
			var = getvalkey((INTBIG)sim_tool, VTOOL, VINTEGER, sim_spice_statekey);
			if (var != NOVARIABLE)
				(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_spice_statekey,
					var->addr | SPICEPLOT, VINTEGER);
			ttyputverbose(M_("SPICE decks will do plots"));
			return;
		}
		if (namesamen(pp, "rawfile", l) == 0)
		{
			var = getvalkey((INTBIG)sim_tool, VTOOL, VINTEGER, sim_spice_statekey);
			if (var != NOVARIABLE)
				(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_spice_statekey,
					var->addr | SPICERAWFILE, VINTEGER);
			ttyputverbose(M_("Reads SPICE Rawfile output"));
			return;
		}
		if (namesamen(pp, "global-pwr-gnd", l) == 0)
		{
			var = getvalkey((INTBIG)sim_tool, VTOOL, VINTEGER, sim_spice_statekey);
			if (var != NOVARIABLE)
				(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_spice_statekey,
					var->addr | SPICEGLOBALPG, VINTEGER);
			ttyputverbose(M_("Writes global Power and Ground"));
			return;
		}
		if (namesamen(pp, "use-nodenames", l) == 0)
		{
			var = getvalkey((INTBIG)sim_tool, VTOOL, VINTEGER, sim_spice_statekey);
			if (var != NOVARIABLE)
				(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_spice_statekey,
					var->addr | SPICENODENAMES, VINTEGER);
			ttyputverbose(M_("SPICE decks will contain node names"));
			return;
		}
		ttyputbadusage("telltool simulation spice");
		return;
	}

	if (namesamen(pp, "verilog", l) == 0 && l >= 1)
	{
		l = strlen(pp = par[1]);
		if (namesamen(pp, "use-assign", l) == 0)
		{
			var = getvalkey((INTBIG)sim_tool, VTOOL, VINTEGER, sim_verilog_statekey);
			if (var != NOVARIABLE) verilogstate = var->addr; else
				verilogstate = 0;
			(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_verilog_statekey,
				verilogstate | VERILOGUSEASSIGN, VINTEGER);
			ttyputverbose(M_("Verilog decks will use the 'assign' construct"));
			return;
		}
		if (namesamen(pp, "default-trireg", l) == 0)
		{
			var = getvalkey((INTBIG)sim_tool, VTOOL, VINTEGER, sim_verilog_statekey);
			if (var != NOVARIABLE) verilogstate = var->addr; else
				verilogstate = 0;
			(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_verilog_statekey,
				verilogstate | VERILOGUSETRIREG, VINTEGER);
			ttyputverbose(M_("Verilog wires will default to 'trireg'"));
			return;
		}

		if (namesamen(pp, "not", l) == 0)
		{
			if (count <= 2)
			{
				ttyputusage("telltool simulation verilog not OPTION");
				return;
			}
			l = strlen(pp = par[2]);
			if (namesamen(pp, "use-assign", l) == 0)
			{
				var = getvalkey((INTBIG)sim_tool, VTOOL, VINTEGER, sim_verilog_statekey);
				if (var != NOVARIABLE) verilogstate = var->addr; else
					verilogstate = 0;
				(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_verilog_statekey,
					verilogstate & ~VERILOGUSEASSIGN, VINTEGER);
				ttyputverbose(M_("Verilog decks will not use the 'assign' construct"));
				return;
			}
			if (namesamen(pp, "default-trireg", l) == 0)
			{
				var = getvalkey((INTBIG)sim_tool, VTOOL, VINTEGER, sim_verilog_statekey);
				if (var != NOVARIABLE) verilogstate = var->addr; else
					verilogstate = 0;
				(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_verilog_statekey,
					verilogstate & ~VERILOGUSETRIREG, VINTEGER);
				ttyputverbose(M_("Verilog wires will not default to 'trireg'"));
				return;
			}
			ttyputbadusage("telltool simulation verilog not");
			return;
		}
		ttyputbadusage("telltool simulation verilog");
		return;
	}

	if (namesamen(pp, "fasthenry", l) == 0 && l >= 1)
	{
		l = strlen(pp = par[1]);
		if (namesamen(pp, "pole", l) == 0 && l >= 3)
		{
			if (count <= 2)
			{
				ttyputusage("telltool simulation fasthenry pole (single | multiple NUMPOLES)");
				return;
			}
			l = strlen(pp = par[2]);
			if (namesamen(pp, "single", l) == 0)
			{
				var = getvalkey((INTBIG)sim_tool, VTOOL, VINTEGER, sim_fasthenrystatekey);
				if (var != NOVARIABLE) options = var->addr; else options = 0;
				(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_fasthenrystatekey, options & ~FHMAKEMULTIPOLECKT, VINTEGER);
				return;
			}
			if (namesamen(pp, "multiple", l) == 0)
			{
				if (count <= 3)
				{
					ttyputusage("telltool simulation fasthenry pole multiple NUMPOLES");
					return;
				}
				var = getvalkey((INTBIG)sim_tool, VTOOL, VINTEGER, sim_fasthenrystatekey);
				if (var != NOVARIABLE) options = var->addr; else options = 0;
				(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_fasthenrystatekey, options | FHMAKEMULTIPOLECKT, VINTEGER);
				value = atoi(par[3]);
				(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_fasthenrynumpoleskey, value, VINTEGER);
				return;
			}
			ttyputbadusage("telltool simulation fasthenry pole");
			return;
		}
		if (namesamen(pp, "frequency", l) == 0)
		{
			if (count <= 2)
			{
				ttyputusage("telltool simulation fasthenry frequency (single | multiple START END RUNSPERDECADE)");
				return;
			}
			l = strlen(pp = par[2]);
			if (namesamen(pp, "single", l) == 0)
			{
				var = getvalkey((INTBIG)sim_tool, VTOOL, VINTEGER, sim_fasthenrystatekey);
				if (var != NOVARIABLE) options = var->addr; else options = 0;
				options |= FHUSESINGLEFREQ;
				(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_fasthenrystatekey, options, VINTEGER);
				return;
			}
			if (namesamen(pp, "multiple", l) == 0)
			{
				if (count <= 5)
				{
					ttyputusage("telltool simulation fasthenry frequency multiple START END RUNSPERDECADE");
					return;
				}
				var = getvalkey((INTBIG)sim_tool, VTOOL, VINTEGER, sim_fasthenrystatekey);
				if (var != NOVARIABLE) options = var->addr; else options = 0;
				options &= ~FHUSESINGLEFREQ;
				(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_fasthenrystatekey, options, VINTEGER);
				fvalue = (float)atof(par[3]);
				(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_fasthenryfreqstartkey, castint(fvalue), VFLOAT);
				fvalue = (float)atof(par[4]);
				(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_fasthenryfreqendkey, castint(fvalue), VFLOAT);
				value = atoi(par[5]);
				(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_fasthenryrunsperdecadekey, value, VINTEGER);
				return;
			}
			ttyputbadusage("telltool simulation fasthenry frequency");
			return;
		}
		if (namesamen(pp, "thickness", l) == 0)
		{
			if (count <= 2)
			{
				ttyputusage("telltool simulation fasthenry thickness DEFTHICKNESS");
				return;
			}
			value = atola(par[2], 0);
			(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_fasthenrythicknesskey, value, VINTEGER);
			return;
		}
		if (namesamen(pp, "width-subdivisions", l) == 0)
		{
			if (count <= 2)
			{
				ttyputusage("telltool simulation fasthenry width-subdivisions DEFWIDTHSUBS");
				return;
			}
			value = atoi(par[2]);
			(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_fasthenrywidthsubdivkey, value, VINTEGER);
			return;
		}
		if (namesamen(pp, "height-subdivisions", l) == 0)
		{
			if (count <= 2)
			{
				ttyputusage("telltool simulation fasthenry height-subdivisions DEFHEIGHTSUBS");
				return;
			}
			value = atoi(par[2]);
			(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_fasthenryheightsubdivkey, value, VINTEGER);
			return;
		}
		if (namesamen(pp, "max-segment-length", l) == 0)
		{
			if (count <= 2)
			{
				ttyputusage("telltool simulation fasthenry max-segment-length MAXSEGLENGTH");
				return;
			}
			value = atola(par[2], 0);
			(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_fasthenryseglimitkey, value, VINTEGER);
			return;
		}
		if (namesamen(pp, "execute", l) == 0)
		{
			if (count <= 2)
			{
				ttyputusage("telltool simulation fasthenry execute (none | run | multiple-run)");
				return;
			}
			l = strlen(pp = par[2]);
			if (namesamen(pp, "none", l) == 0)
			{
				var = getvalkey((INTBIG)sim_tool, VTOOL, VINTEGER, sim_fasthenrystatekey);
				if (var != NOVARIABLE) options = var->addr; else options = 0;
				(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_fasthenrystatekey,
					(options & ~FHEXECUTETYPE) | FHEXECUTENONE, VINTEGER);
				return;
			}
			if (namesamen(pp, "run", l) == 0)
			{
				var = getvalkey((INTBIG)sim_tool, VTOOL, VINTEGER, sim_fasthenrystatekey);
				if (var != NOVARIABLE) options = var->addr; else options = 0;
				(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_fasthenrystatekey,
					(options & ~FHEXECUTETYPE) | FHEXECUTERUNFH, VINTEGER);
				return;
			}
			if (namesamen(pp, "multiple-run", l) == 0)
			{
				var = getvalkey((INTBIG)sim_tool, VTOOL, VINTEGER, sim_fasthenrystatekey);
				if (var != NOVARIABLE) options = var->addr; else options = 0;
				(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_fasthenrystatekey,
					(options & ~FHEXECUTETYPE) | FHEXECUTERUNFHMUL, VINTEGER);
				return;
			}
			ttyputbadusage("telltool simulation fasthenry execute");
			return;
		}
		if (namesamen(pp, "postscript", l) == 0 && l >= 3)
		{
			if (count <= 2)
			{
				ttyputusage("telltool simulation fasthenry postscript (on | off)");
				return;
			}
			l = strlen(pp = par[2]);
			if (namesamen(pp, "on", l) == 0 && l >= 2)
			{
				var = getvalkey((INTBIG)sim_tool, VTOOL, VINTEGER, sim_fasthenrystatekey);
				if (var != NOVARIABLE) options = var->addr; else options = 0;
				(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_fasthenrystatekey,
					options | FHMAKEPOSTSCRIPTVIEW, VINTEGER);
				return;
			}
			if (namesamen(pp, "off", l) == 0 && l >= 2)
			{
				var = getvalkey((INTBIG)sim_tool, VTOOL, VINTEGER, sim_fasthenrystatekey);
				if (var != NOVARIABLE) options = var->addr; else options = 0;
				(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_fasthenrystatekey,
					options & ~FHMAKEPOSTSCRIPTVIEW, VINTEGER);
				return;
			}
			ttyputbadusage("telltool simulation fasthenry postscript");
			return;
		}
		if (namesamen(pp, "spice", l) == 0)
		{
			if (count <= 2)
			{
				ttyputusage("telltool simulation fasthenry spice (on | off)");
				return;
			}
			l = strlen(pp = par[2]);
			if (namesamen(pp, "on", l) == 0 && l >= 2)
			{
				var = getvalkey((INTBIG)sim_tool, VTOOL, VINTEGER, sim_fasthenrystatekey);
				if (var != NOVARIABLE) options = var->addr; else options = 0;
				(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_fasthenrystatekey,
					options | FHMAKESPICESUBCKT, VINTEGER);
				return;
			}
			if (namesamen(pp, "off", l) == 0 && l >= 2)
			{
				var = getvalkey((INTBIG)sim_tool, VTOOL, VINTEGER, sim_fasthenrystatekey);
				if (var != NOVARIABLE) options = var->addr; else options = 0;
				(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_fasthenrystatekey,
					options & ~FHMAKESPICESUBCKT, VINTEGER);
				return;
			}
			ttyputbadusage("telltool simulation fasthenry spice");
			return;
		}
		if (namesamen(pp, "arc", l) == 0)
		{
			ai = (ARCINST *)asktool(us_tool, "get-arc");
			if (ai == NOARCINST)
			{
				ttyputerr(M_("Select an arc first"));
				return;
			}
			if (count <= 2)
			{
				ttyputusage("telltool simulation fasthenry arc OPTIONS");
				return;
			}
			l = strlen(pp = par[2]);
			if (namesamen(pp, "add", l) == 0)
			{
				if (count <= 3)
				{
					ttyputusage("telltool simulation fasthenry arc add GROUPNAME");
					return;
				}
				startobjectchange((INTBIG)ai, VARCINST);
				setvalkey((INTBIG)ai, VARCINST, sim_fasthenrygroupnamekey, (INTBIG)par[3], VSTRING|VDISPLAY);
				endobjectchange((INTBIG)ai, VARCINST);
				return;
			}
			if (namesamen(pp, "remove", l) == 0)
			{
				if (getvalkey((INTBIG)ai, VARCINST, VSTRING, sim_fasthenrygroupnamekey) != NOVARIABLE)
				{
					startobjectchange((INTBIG)ai, VARCINST);
					(void)delvalkey((INTBIG)ai, VARCINST, sim_fasthenrygroupnamekey);
					endobjectchange((INTBIG)ai, VARCINST);
				}
				return;
			}
			if (namesamen(pp, "thickness", l) == 0)
			{
				if (count <= 3)
				{
					if (getvalkey((INTBIG)ai, VARCINST, VINTEGER, sim_fasthenrythicknesskey) != NOVARIABLE)
					{
						startobjectchange((INTBIG)ai, VARCINST);
						(void)delvalkey((INTBIG)ai, VARCINST, sim_fasthenrythicknesskey);
						endobjectchange((INTBIG)ai, VARCINST);
					}
				} else
				{
					startobjectchange((INTBIG)ai, VARCINST);
					setvalkey((INTBIG)ai, VARCINST, sim_fasthenrythicknesskey, atola(par[3], 0), VINTEGER);
					endobjectchange((INTBIG)ai, VARCINST);
				}
				return;
			}
			if (namesamen(pp, "width-subdivisions", l) == 0)
			{
				if (count <= 3)
				{
					if (getvalkey((INTBIG)ai, VARCINST, VINTEGER, sim_fasthenrywidthsubdivkey) != NOVARIABLE)
					{
						startobjectchange((INTBIG)ai, VARCINST);
						(void)delvalkey((INTBIG)ai, VARCINST, sim_fasthenrywidthsubdivkey);
						endobjectchange((INTBIG)ai, VARCINST);
					}
				} else
				{
					startobjectchange((INTBIG)ai, VARCINST);
					setvalkey((INTBIG)ai, VARCINST, sim_fasthenrywidthsubdivkey, atoi(par[3]), VINTEGER);
					endobjectchange((INTBIG)ai, VARCINST);
				}
				return;
			}
			if (namesamen(pp, "height-subdivisions", l) == 0)
			{
				if (count <= 3)
				{
					if (getvalkey((INTBIG)ai, VARCINST, VINTEGER, sim_fasthenryheightsubdivkey) != NOVARIABLE)
					{
						startobjectchange((INTBIG)ai, VARCINST);
						(void)delvalkey((INTBIG)ai, VARCINST, sim_fasthenryheightsubdivkey);
						endobjectchange((INTBIG)ai, VARCINST);
					}
				} else
				{
					startobjectchange((INTBIG)ai, VARCINST);
					setvalkey((INTBIG)ai, VARCINST, sim_fasthenryheightsubdivkey, atoi(par[3]), VINTEGER);
					endobjectchange((INTBIG)ai, VARCINST);
				}
				return;
			}
			if (namesamen(pp, "z-head", l) == 0 && l >= 3)
			{
				if (count <= 3)
				{
					if (getvalkey((INTBIG)ai, VARCINST, VINTEGER, sim_fasthenryzheadkey) != NOVARIABLE)
					{
						startobjectchange((INTBIG)ai, VARCINST);
						(void)delvalkey((INTBIG)ai, VARCINST, sim_fasthenryzheadkey);
						endobjectchange((INTBIG)ai, VARCINST);
					}
				} else
				{
					startobjectchange((INTBIG)ai, VARCINST);
					setvalkey((INTBIG)ai, VARCINST, sim_fasthenryzheadkey, atola(par[3], 0), VINTEGER);
					endobjectchange((INTBIG)ai, VARCINST);
				}
				return;
			}
			if (namesamen(pp, "z-tail", l) == 0 && l >= 3)
			{
				if (count <= 3)
				{
					if (getvalkey((INTBIG)ai, VARCINST, VINTEGER, sim_fasthenryztailkey) != NOVARIABLE)
					{
						startobjectchange((INTBIG)ai, VARCINST);
						(void)delvalkey((INTBIG)ai, VARCINST, sim_fasthenryztailkey);
						endobjectchange((INTBIG)ai, VARCINST);
					}
				} else
				{
					startobjectchange((INTBIG)ai, VARCINST);
					setvalkey((INTBIG)ai, VARCINST, sim_fasthenryztailkey, atola(par[3], 0), VINTEGER);
					endobjectchange((INTBIG)ai, VARCINST);
				}
				return;
			}
			ttyputbadusage("telltool simulation fasthenry arc");
			return;
		}
		ttyputbadusage("telltool simulation fasthenry");
		return;
	}

	/* unknown command */
	ttyputbadusage("telltool simulation");
}

void sim_slice(void)
{
	NODEPROTO *np;

	if (sim_circuitchanged && !sim_undoredochange)
	{
		if (sim_window_isactive(&np) != 0)
		{
			sim_window_stopsimulation();
			toolturnoff(sim_tool, FALSE);
			ttyputmsg(_("Circuit changed: simulation stopped"));
			setactivity(_("Simulation Halted"));
		}
	}
	sim_circuitchanged = FALSE;
	sim_undoredochange = FALSE;
}

void sim_startbatch(TOOL *source, BOOLEAN undoredo)
{
	sim_undoredochange = undoredo;
}

void sim_modifynodeinst(NODEINST *ni, INTBIG olx, INTBIG oly, INTBIG ohx, INTBIG ohy,
	INTBIG orot, INTBIG otran)
{
	sim_checktostopsimulation(ni->parent);
}

void sim_modifyarcinst(ARCINST *ai, INTBIG oxA, INTBIG oyA, INTBIG oxB, INTBIG oyB,
	INTBIG owid, INTBIG olen)
{
	sim_checktostopsimulation(ai->parent);
}

void sim_modifyportproto(PORTPROTO *pp, NODEINST *oni, PORTPROTO *opp)
{
	sim_checktostopsimulation(pp->parent);
}

void sim_newobject(INTBIG addr, INTBIG type)
{
	switch (type&VTYPE)
	{
		case VNODEINST:  sim_checktostopsimulation(((NODEINST *)addr)->parent);   break;
		case VARCINST:   sim_checktostopsimulation(((ARCINST *)addr)->parent);    break;
		case VPORTPROTO: sim_checktostopsimulation(((PORTPROTO *)addr)->parent);  break;
	}
}

void sim_killobject(INTBIG addr, INTBIG type)
{
	switch (type&VTYPE)
	{
		case VNODEINST:  sim_checktostopsimulation(((NODEINST *)addr)->parent);   break;
		case VARCINST:   sim_checktostopsimulation(((ARCINST *)addr)->parent);    break;
		case VPORTPROTO: sim_checktostopsimulation(((PORTPROTO *)addr)->parent);  break;
		case VNODEPROTO: sim_checktostopsimulation((NODEPROTO *)addr);            break;
	}
}

void sim_newvariable(INTBIG addr, INTBIG type, INTBIG key, INTBIG newtype)
{
	REGISTER char *name;
	REGISTER ARCINST *ai;
	REGISTER NODEINST *ni;

	if ((newtype&VCREF) != 0)
	{
		/* detect change to "ai->userbits" to see if arc negation changed */
		if ((type&VTYPE) != VARCINST) return;
		name = changedvariablename(type, key, newtype);
		if (namesame(name, "userbits") != 0) return;

		ai = (ARCINST *)addr;
		if (ai != sim_modifyarc) return;
		if (((INTBIG)(ai->userbits&ISNEGATED)) != (sim_modifyarcbits&ISNEGATED))
			sim_checktostopsimulation(ai->parent);
	} else
	{
		/* detect change to arc's name */
		if ((type&VTYPE) == VARCINST && key == el_arc_name_key)
		{
			ai = (ARCINST *)addr;
			sim_checktostopsimulation(ai->parent);
		}

		/* detect change to node's name */
		if ((type&VTYPE) == VNODEINST && key == el_node_name_key)
		{
			ni = (NODEINST *)addr;
			sim_checktostopsimulation(ni->parent);
		}
	}
}

void sim_killvariable(INTBIG addr, INTBIG type, INTBIG key, INTBIG oldaddr, INTBIG oldtype,
	UINTBIG *olddescript)
{
	REGISTER char *name;

	if ((oldtype&VCREF) != 0)
	{
		/* detect change to "ai->userbits" and save former value */
		if ((type&VTYPE) != VARCINST) return;
		name = changedvariablename(type, key, oldtype);
		if (namesame(name, "userbits") != 0) return;

		sim_modifyarc = (ARCINST *)addr;
		sim_modifyarcbits = sim_modifyarc->userbits;
	}
}

void sim_modifyvariable(INTBIG addr, INTBIG type, INTBIG key, INTBIG vartype,
	INTBIG aindex, INTBIG oldvalue)
{
	if (type != VNODEPROTO) return;
	if (key != el_facet_message_key) return;
	sim_checktostopsimulation((NODEPROTO *)addr);
}

void sim_insertvariable(INTBIG addr, INTBIG type, INTBIG key, INTBIG vartype, INTBIG index)
{
	if (type != VNODEPROTO) return;
	if (key != el_facet_message_key) return;
	sim_checktostopsimulation((NODEPROTO *)addr);
}

void sim_deletevariable(INTBIG addr, INTBIG type, INTBIG key, INTBIG vartype, INTBIG index,
	INTBIG oldvalue)
{
	if (type != VNODEPROTO) return;
	if (key != el_facet_message_key) return;
	sim_checktostopsimulation((NODEPROTO *)addr);
}

void sim_readlibrary(LIBRARY *lib)
{
	REGISTER VARIABLE *var;

	/* grab new window state information */
	var = getvalkey((INTBIG)sim_tool, VTOOL, VINTEGER, sim_window_statekey);
	if (var != NOVARIABLE)
		sim_window_state = var->addr;
}

/****************************** SUPPORT ******************************/

void sim_window_setstate(INTBIG newstate)
{
	(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_window_statekey, newstate, VINTEGER);
	sim_window_state = newstate;
}

void sim_checktostopsimulation(NODEPROTO *np)
{
	NODEPROTO *onp;

	if (np->primindex != 0) return;
	if (sim_window_isactive(&onp) == 0) return;
	if (onp != np)
	{
		/* facets not the same, but see if the modified one is VHDL or netlist */
		if (onp->cell != np->cell) return;
		if (np->cellview != el_vhdlview && np->cellview != el_netlistalsview)
			return;
	}
	sim_circuitchanged = TRUE;
}

BOOLEAN sim_topofinstances(char **c)
{
	if ((sim_window_state&SIMENGINE) == SIMENGINEALS)
		return(simals_topofinstances(c));
#if SIMTOOLIRSIM != 0
	if ((sim_window_state&SIMENGINE) == SIMENGINEIRSIM)
		return(irsim_topofinstances(c));
#endif
	return(FALSE);
}

char *sim_nextinstance(void)
{
	if ((sim_window_state&SIMENGINE) == SIMENGINEALS)
		return(simals_nextinstance());
#if SIMTOOLIRSIM != 0
	if ((sim_window_state&SIMENGINE) == SIMENGINEIRSIM)
		return(irsim_nextinstance());
#endif
	return(0);
}

/****************************** DIALOG ******************************/

/* Simulation Options */
static DIALOGITEM sim_optionsdialogitems[] =
{
 /*  1 */ {0, {80,440,104,504}, BUTTON, N_("OK")},
 /*  2 */ {0, {80,348,104,412}, BUTTON, N_("Cancel")},
 /*  3 */ {0, {8,316,24,513}, CHECK, N_("Resimulate each change")},
 /*  4 */ {0, {32,316,48,513}, CHECK, N_("Auto advance time")},
 /*  5 */ {0, {56,316,72,513}, CHECK, N_("Multistate display")},
 /*  6 */ {0, {56,8,72,205}, CHECK, N_("Show waveform window")},
 /*  7 */ {0, {144,296,160,421}, MESSAGE, N_("Maximum events:")},
 /*  8 */ {0, {144,424,160,505}, EDITTEXT, ""},
 /*  9 */ {0, {32,8,48,165}, MESSAGE, N_("Base for bus values:")},
 /* 10 */ {0, {32,168,48,221}, POPUP, ""},
 /* 11 */ {0, {80,20,96,205}, MESSAGE, N_("Place waveform window")},
 /* 12 */ {0, {80,212,96,320}, POPUP, ""},
 /* 13 */ {0, {8,8,24,133}, MESSAGE, N_("Simulation engine:")},
 /* 14 */ {0, {8,136,24,280}, POPUP, ""},
 /* 15 */ {0, {120,8,136,93}, MESSAGE, N_("IRSIM:")},
 /* 16 */ {0, {120,288,136,372}, MESSAGE, N_("ALS:")},
 /* 17 */ {0, {160,32,176,105}, RADIO, N_("Quick")},
 /* 18 */ {0, {160,108,176,181}, RADIO, N_("Local")},
 /* 19 */ {0, {160,184,176,257}, RADIO, N_("Full")},
 /* 20 */ {0, {112,8,113,513}, DIVIDELINE, ""},
 /* 21 */ {0, {184,20,200,133}, MESSAGE, N_("Parameter file:")},
 /* 22 */ {0, {204,32,236,277}, EDITTEXT, ""},
 /* 23 */ {0, {140,20,156,157}, MESSAGE, N_("Parasitics:")},
 /* 24 */ {0, {184,176,200,233}, BUTTON, N_("Set")},
 /* 25 */ {0, {112,284,260,285}, DIVIDELINE, ""},
 /* 26 */ {0, {244,20,260,217}, CHECK, N_("Show commands")}
};
static DIALOG sim_optionsdialog = {{50,75,326,597}, N_("Simulation Options"), 0, 26, sim_optionsdialogitems, 0, 0};

/* special items for the "Simulation Options" dialog: */
#define DSMO_RESIMULATE     3		/* Resimulate each change (check) */
#define DSMO_AUTOADVANCE    4		/* Auto advance time (check) */
#define DSMO_MULTISTATE     5		/* Multistate display (check) */
#define DSMO_WAVEFORMWIND   6		/* Show waveform window (check) */
#define DSMO_MAXEVENTS      8		/* Maximum events (edit text) */
#define DSMO_BUSBASE       10		/* Base for busses (popup) */
#define DSMO_WAVEFORMLOC   12		/* Place waveform window (popup) */
#define DSMO_SIMENGINE     14		/* Simulation engine to use (popup) */
#define DSMO_IRSIMPARAQUI  17		/* IRSIM: Quick parasitics (radio) */
#define DSMO_IRSIMPARALOC  18		/* IRSIM: Local parasitics (radio) */
#define DSMO_IRSIMPARAFULL 19		/* IRSIM: Full parasitics (radio) */
#define DSMO_IRSIMPARMFILE 22		/* IRSIM: Parameter file (edit text) */
#define DSMO_IRSIMSETPFILE 24		/* IRSIM: Set Parameter file (button) */
#define DSMO_IRSIMSHOWCMDS 26		/* IRSIM: Show command (check) */

void sim_optionsdlog(void)
{
	INTBIG itemHit, newtracesize, i;
	REGISTER VARIABLE *var;
	char line[20], *newlang[5], *iniparamfile, *pt;
	REGISTER INTBIG noupdate, newnoupdate, newwindowstate, irsimstate,
		newirsimstate, oldplease;
	REGISTER TRAKPTR newtrakroot;
	extern COMCOMP us_colorreadp;
	static char *busbase[4] = {"2", "8", "10", "16"};
	static char *placement[3] = {N_("Cascade"), N_("Tile Horizontally"), N_("Tile Vertically")};
	static char *engine[2] = {N_("ALS"), N_("IRSIM")};
	REGISTER void *infstr, *dia;

	dia = DiaInitDialog(&sim_optionsdialog);
	if (dia == 0) return;
	for(i=0; i<3; i++) newlang[i] = _(placement[i]);
	DiaSetPopup(dia, DSMO_WAVEFORMLOC, 3, newlang);
#if SIMTOOLIRSIM == 0
	engine[1] = N_("IRSIM (not installed)");
#endif
	for(i=0; i<2; i++) newlang[i] = _(engine[i]);
	DiaSetPopup(dia, DSMO_SIMENGINE, 2, newlang);
	var = getval((INTBIG)sim_tool, VTOOL, VINTEGER, "SIM_als_no_update");
	if (var == NOVARIABLE) noupdate = 0; else noupdate = var->addr;
	var = getvalkey((INTBIG)sim_tool, VTOOL, VINTEGER, sim_irsim_statekey);
	if (var == NOVARIABLE) irsimstate = DEFIRSIMSTATE; else irsimstate = var->addr;
	if (noupdate == 0) DiaSetControl(dia, DSMO_RESIMULATE, 1);
	if ((sim_window_state&ADVANCETIME) != 0) DiaSetControl(dia, DSMO_AUTOADVANCE, 1);
	if ((sim_window_state&FULLSTATE) != 0) DiaSetControl(dia, DSMO_MULTISTATE, 1);
	if ((sim_window_state&SHOWWAVEFORM) != 0) DiaSetControl(dia, DSMO_WAVEFORMWIND, 1);
	if ((irsimstate&IRSIMSHOWCOMMANDS) != 0) DiaSetControl(dia, DSMO_IRSIMSHOWCMDS, 1);
	DiaSetPopup(dia, DSMO_BUSBASE, 4, busbase);
	switch (sim_window_state&SIMENGINE)
	{
		case SIMENGINEALS:   DiaSetPopupEntry(dia, DSMO_SIMENGINE, 0);   break;
		case SIMENGINEIRSIM: DiaSetPopupEntry(dia, DSMO_SIMENGINE, 1);   break;
	}
	switch (irsimstate&IRSIMPARASITICS)
	{
		case IRSIMPARAQUICK: DiaSetControl(dia, DSMO_IRSIMPARAQUI, 1);   break;
		case IRSIMPARALOCAL: DiaSetControl(dia, DSMO_IRSIMPARALOC, 1);   break;
		case IRSIMPARAFULL:  DiaSetControl(dia, DSMO_IRSIMPARAFULL, 1);  break;
	}
	switch (sim_window_state&BUSBASEBITS)
	{
		case BUSBASE2:  DiaSetPopupEntry(dia, DSMO_BUSBASE, 0);   break;
		case BUSBASE8:  DiaSetPopupEntry(dia, DSMO_BUSBASE, 1);   break;
		case BUSBASE10: DiaSetPopupEntry(dia, DSMO_BUSBASE, 2);   break;
		case BUSBASE16: DiaSetPopupEntry(dia, DSMO_BUSBASE, 3);   break;
	}
	switch (sim_window_state&WAVEPLACE)
	{
		case WAVEPLACECAS:   DiaSetPopupEntry(dia, DSMO_WAVEFORMLOC, 0);   break;
		case WAVEPLACETHOR:  DiaSetPopupEntry(dia, DSMO_WAVEFORMLOC, 1);   break;
		case WAVEPLACETVER:  DiaSetPopupEntry(dia, DSMO_WAVEFORMLOC, 2);   break;
	}
	if (simals_trace_size == 0)
	{
		var = getval((INTBIG)sim_tool, VTOOL, VINTEGER, "SIM_als_num_events");
		if (var == NOVARIABLE) simals_trace_size = DEFAULT_TRACE_SIZE; else
			simals_trace_size = var->addr;
	}
	sprintf(line, "%ld", simals_trace_size);
	DiaSetText(dia, DSMO_MAXEVENTS, line);
	infstr = initinfstr();
	var = getval((INTBIG)sim_tool, VTOOL, VSTRING, "SIM_irsim_parameter_file");
	if (var == NOVARIABLE)
		formatinfstr(infstr, "%s%s", el_libdir, DEFIRSIMPARAMFILE); else
			addstringtoinfstr(infstr, (char *)var->addr);
	(void)allocstring(&iniparamfile, returninfstr(infstr), el_tempcluster);
	DiaSetText(dia, DSMO_IRSIMPARMFILE, iniparamfile);

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit(dia);
		if (itemHit == OK || itemHit == CANCEL) break;
		if (itemHit == DSMO_RESIMULATE || itemHit == DSMO_AUTOADVANCE ||
			itemHit == DSMO_MULTISTATE || itemHit == DSMO_WAVEFORMWIND ||
			itemHit == DSMO_IRSIMSHOWCMDS)
		{
			DiaSetControl(dia, itemHit, 1 - DiaGetControl(dia, itemHit));
			continue;
		}
		if (itemHit == DSMO_IRSIMPARAQUI || itemHit == DSMO_IRSIMPARALOC ||
			itemHit == DSMO_IRSIMPARAFULL)
		{
			DiaSetControl(dia, DSMO_IRSIMPARAQUI, 0);
			DiaSetControl(dia, DSMO_IRSIMPARALOC, 0);
			DiaSetControl(dia, DSMO_IRSIMPARAFULL, 0);
			DiaSetControl(dia, itemHit, 1);
			continue;
		}
		if (itemHit == DSMO_IRSIMSETPFILE)
		{
			oldplease = el_pleasestop;
			infstr = initinfstr();
			addstringtoinfstr(infstr, "irsimparam/");
			addstringtoinfstr(infstr, _("IRSIM Parameter File"));
			i = ttygetparam(returninfstr(infstr), &us_colorreadp, 1, newlang);
			el_pleasestop = oldplease;
			if (i != 0 && newlang[0][0] != 0) DiaSetText(dia, DSMO_IRSIMPARMFILE, newlang[0]);
			continue;
		}
	}

	if (itemHit != CANCEL)
	{
		if (DiaGetControl(dia, DSMO_RESIMULATE) != 0) newnoupdate = 0; else
			newnoupdate = 1;
		newwindowstate = 0;
		if (DiaGetControl(dia, DSMO_AUTOADVANCE) != 0) newwindowstate |= ADVANCETIME;
		if (DiaGetControl(dia, DSMO_MULTISTATE) != 0) newwindowstate |= FULLSTATE;
		if (DiaGetControl(dia, DSMO_WAVEFORMWIND) != 0) newwindowstate |= SHOWWAVEFORM;
		switch (DiaGetPopupEntry(dia, DSMO_BUSBASE))
		{
			case 0: newwindowstate |= BUSBASE2;    break;
			case 1: newwindowstate |= BUSBASE8;    break;
			case 2: newwindowstate |= BUSBASE10;   break;
			case 3: newwindowstate |= BUSBASE16;   break;
		}
		switch (DiaGetPopupEntry(dia, DSMO_WAVEFORMLOC))
		{
			case 0: newwindowstate |= WAVEPLACECAS;    break;
			case 1: newwindowstate |= WAVEPLACETHOR;   break;
			case 2: newwindowstate |= WAVEPLACETVER;   break;
		}
		switch(DiaGetPopupEntry(dia, DSMO_SIMENGINE))
		{
			case 0: newwindowstate |= SIMENGINEALS;    break;
			case 1: newwindowstate |= SIMENGINEIRSIM;  break;
		}
		if (newnoupdate != noupdate)
			(void)setval((INTBIG)sim_tool, VTOOL, "SIM_als_no_update",
				newnoupdate, VINTEGER);
		if (newwindowstate != sim_window_state)
			sim_window_setstate(newwindowstate);

		newirsimstate = 0;
		if (DiaGetControl(dia, DSMO_IRSIMPARALOC) != 0) newirsimstate |= IRSIMPARALOCAL; else
		if (DiaGetControl(dia, DSMO_IRSIMPARAFULL) != 0) newirsimstate |= IRSIMPARAFULL;
		if (DiaGetControl(dia, DSMO_IRSIMSHOWCMDS) != 0) newirsimstate |= IRSIMSHOWCOMMANDS;
		if (newirsimstate != irsimstate)
			(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_irsim_statekey,
				newirsimstate, VINTEGER);
		pt = DiaGetText(dia, DSMO_IRSIMPARMFILE);
		if (strcmp(pt, iniparamfile) != 0)
			(void)setval((INTBIG)sim_tool, VTOOL, "SIM_irsim_parameter_file",
				(INTBIG)pt, VSTRING);
		newtracesize = atoi(DiaGetText(dia, DSMO_MAXEVENTS));
		if (newtracesize != simals_trace_size)
		{
			newtrakroot = (TRAKPTR)simals_alloc_mem((INTBIG)(newtracesize * sizeof(TRAK)));
			if (newtrakroot != 0)
			{
				if (simals_trakroot != 0) efree((char *)simals_trakroot);
				simals_trakroot = newtrakroot;
				simals_trace_size = newtracesize;
				(void)setval((INTBIG)sim_tool, VTOOL, "SIM_als_num_events",
					newtracesize, VINTEGER);
			}
		}
	}
	efree((char *)iniparamfile);
	DiaDoneDialog(dia);
}

/* ALS Clock */
static DIALOGITEM sim_alsclockdialogitems[] =
{
 /*  1 */ {0, {8,320,32,384}, BUTTON, N_("OK")},
 /*  2 */ {0, {40,320,64,384}, BUTTON, N_("Cancel")},
 /*  3 */ {0, {16,8,32,101}, RADIO, N_("Frequency:")},
 /*  4 */ {0, {40,8,56,101}, RADIO, N_("Period:")},
 /*  5 */ {0, {80,8,96,101}, RADIO, N_("Custom:")},
 /*  6 */ {0, {28,208,44,283}, EDITTEXT, ""},
 /*  7 */ {0, {28,112,44,199}, MESSAGE, N_("Freq/Period:")},
 /*  8 */ {0, {112,236,128,384}, RADIO, N_("Normal (gate) strength")},
 /*  9 */ {0, {236,232,252,369}, RADIO, N_("Undefined Phase")},
 /* 10 */ {0, {112,8,128,151}, MESSAGE, N_("Random Distribution:")},
 /* 11 */ {0, {112,160,128,223}, EDITTEXT, ""},
 /* 12 */ {0, {136,236,152,384}, RADIO, N_("Strong (VDD) strength")},
 /* 13 */ {0, {88,236,104,384}, RADIO, N_("Weak (node) strength")},
 /* 14 */ {0, {212,232,228,328}, RADIO, N_("High Phase")},
 /* 15 */ {0, {292,272,308,363}, BUTTON, N_("Delete Phase")},
 /* 16 */ {0, {164,8,279,221}, SCROLL|INACTIVE, ""},
 /* 17 */ {0, {164,224,180,288}, MESSAGE, N_("Duration:")},
 /* 18 */ {0, {264,272,280,363}, BUTTON, N_("Add Phase")},
 /* 19 */ {0, {292,8,308,197}, MESSAGE, N_("Phase Cycles (0 for infinite):")},
 /* 20 */ {0, {292,200,308,247}, EDITTEXT, ""},
 /* 21 */ {0, {164,296,180,369}, EDITTEXT, ""},
 /* 22 */ {0, {148,8,164,99}, MESSAGE, N_("Phase List:")},
 /* 23 */ {0, {188,232,204,327}, RADIO, N_("Low Phase")},
 /* 24 */ {0, {72,8,73,384}, DIVIDELINE, ""}
};
static DIALOG sim_alsclockdialog = {{50,75,367,468}, N_("Clock Specification"), 0, 24, sim_alsclockdialogitems, 0, 0};

/* special items for the "als clock" dialog: */
#define DALC_FREQUENCY       3		/* Frequency (radio) */
#define DALC_PERIOD          4		/* Period (radio) */
#define DALC_CUSTOM          5		/* Custom (radio) */
#define DALC_FREQPERIOD      6		/* Frequency/Period (edit text) */
#define DALC_FREQPERIOD_L    7		/* Frequency/Period label (stat text) */
#define DALC_STRENGTHNORM    8		/* Normal strength (radio) */
#define DALC_PHASEUNDEF      9		/* Undefined phase (radio) */
#define DALC_DISTRIBUTION_L 10		/* Random distribution label (stat text) */
#define DALC_DISTRIBUTION   11		/* Random distribution (edit text) */
#define DALC_STRENGTHSTRONG 12		/* Strong strength (radio) */
#define DALC_STRENGTHWEAK   13		/* Weak strength (radio) */
#define DALC_PHASEHIGH      14		/* High phase (radio) */
#define DALC_DELPHASE       15		/* Delete a phase (button) */
#define DALC_PHASELIST      16		/* List of phases (scroll) */
#define DALC_DURATION_L     17		/* Duration label (stat text) */
#define DALC_ADDPHASE       18		/* Add a phase (button) */
#define DALC_CYCLES_L       19		/* Number of cycles label (stat text) */
#define DALC_CYCLES         20		/* Number of cycles (edit text) */
#define DALC_DURATION       21		/* Duration (edit text) */
#define DALC_PHASELIST_L    22		/* Phases list label (stat text) */
#define DALC_PHASELOW       23		/* Low phase (radio) */

#define MAXPHASES 20
INTBIG sim_alsclockdlog(char *paramstart[])
{
	INTBIG itemHit, i, j;
	INTBIG retval;
	char *line;
	float durval[MAXPHASES];
	INTBIG levval[MAXPHASES], phases;
	static char *retdur[MAXPHASES];
	static INTBIG first = 0;
	REGISTER void *dia;

	/* display the ALS clock dialog box */
	dia = DiaInitDialog(&sim_alsclockdialog);
	if (dia == 0) return(0);
	DiaInitTextDialog(dia, DALC_PHASELIST, DiaNullDlogList, DiaNullDlogItem, DiaNullDlogDone, -1, 0);

	/* setup complex clock specification */
	DiaSetText(dia, DALC_DISTRIBUTION, "0.0");	/* random */
	DiaSetControl(dia, DALC_STRENGTHNORM, 1);	/* gate strength */
	DiaSetControl(dia, DALC_PHASEHIGH, 1);		/* level */
	DiaSetText(dia, DALC_CYCLES, "0");			/* infinite cycles */

	/* now disable complex clock specification */
	DiaSetControl(dia, DALC_FREQUENCY, 1);		/* frequency */
	DiaDimItem(dia, DALC_DISTRIBUTION_L);		/* random */
	DiaDimItem(dia, DALC_DISTRIBUTION);
	DiaNoEditControl(dia, DALC_DISTRIBUTION);
	DiaDimItem(dia, DALC_STRENGTHWEAK);			/* strength */
	DiaDimItem(dia, DALC_STRENGTHNORM);
	DiaDimItem(dia, DALC_STRENGTHSTRONG);
	DiaDimItem(dia, DALC_PHASELOW);				/* level */
	DiaDimItem(dia, DALC_PHASEHIGH);
	DiaDimItem(dia, DALC_PHASEUNDEF);
	DiaDimItem(dia, DALC_DURATION_L);			/* duration */
	DiaDimItem(dia, DALC_DURATION);
	DiaNoEditControl(dia, DALC_DURATION);
	DiaDimItem(dia, DALC_ADDPHASE);				/* add/delete */
	DiaDimItem(dia, DALC_DELPHASE);
	DiaDimItem(dia, DALC_PHASELIST_L);			/* list header */
	DiaDimItem(dia, DALC_CYCLES_L);				/* cycles */
	DiaDimItem(dia, DALC_CYCLES);
	DiaNoEditControl(dia, DALC_CYCLES);
	DiaUnDimItem(dia, DALC_FREQPERIOD_L);		/* freq/per */
	DiaUnDimItem(dia, DALC_FREQPERIOD);
	DiaEditControl(dia, DALC_FREQPERIOD);

	/* loop until done */
	phases = 0;
	for(;;)
	{
		itemHit = DiaNextHit(dia);
		if (itemHit == OK || itemHit == CANCEL) break;
		if (itemHit == DALC_FREQUENCY || itemHit == DALC_PERIOD || itemHit == DALC_CUSTOM)
		{
			DiaSetControl(dia, DALC_FREQUENCY, 0);
			DiaSetControl(dia, DALC_PERIOD, 0);
			DiaSetControl(dia, DALC_CUSTOM, 0);
			DiaSetControl(dia, itemHit, 1);
			if (itemHit == DALC_CUSTOM)
			{
				DiaUnDimItem(dia, DALC_DISTRIBUTION_L);		/* random */
				DiaUnDimItem(dia, DALC_DISTRIBUTION);
				DiaEditControl(dia, DALC_DISTRIBUTION);
				DiaUnDimItem(dia, DALC_STRENGTHWEAK);		/* strength */
				DiaUnDimItem(dia, DALC_STRENGTHNORM);
				DiaUnDimItem(dia, DALC_STRENGTHSTRONG);
				DiaUnDimItem(dia, DALC_PHASELOW);			/* level */
				DiaUnDimItem(dia, DALC_PHASEHIGH);
				DiaUnDimItem(dia, DALC_PHASEUNDEF);
				DiaUnDimItem(dia, DALC_DURATION_L);			/* duration */
				DiaUnDimItem(dia, DALC_DURATION);
				DiaEditControl(dia, DALC_DURATION);
				DiaUnDimItem(dia, DALC_ADDPHASE);			/* add/delete */
				DiaUnDimItem(dia, DALC_DELPHASE);
				DiaUnDimItem(dia, DALC_PHASELIST_L);			/* list header */
				DiaUnDimItem(dia, DALC_CYCLES_L);			/* cycles */
				DiaUnDimItem(dia, DALC_CYCLES);
				DiaEditControl(dia, DALC_CYCLES);
				DiaDimItem(dia, DALC_FREQPERIOD_L);			/* freq/per */
				DiaDimItem(dia, DALC_FREQPERIOD);
				DiaNoEditControl(dia, DALC_FREQPERIOD);
			} else
			{
				DiaDimItem(dia, DALC_DISTRIBUTION_L);		/* random */
				DiaDimItem(dia, DALC_DISTRIBUTION);
				DiaNoEditControl(dia, DALC_DISTRIBUTION);
				DiaDimItem(dia, DALC_STRENGTHWEAK);			/* strength */
				DiaDimItem(dia, DALC_STRENGTHNORM);
				DiaDimItem(dia, DALC_STRENGTHSTRONG);
				DiaDimItem(dia, DALC_PHASELOW);				/* level */
				DiaDimItem(dia, DALC_PHASEHIGH);
				DiaDimItem(dia, DALC_PHASEUNDEF);
				DiaDimItem(dia, DALC_DURATION_L);			/* duration */
				DiaDimItem(dia, DALC_DURATION);
				DiaNoEditControl(dia, DALC_DURATION);
				DiaDimItem(dia, DALC_ADDPHASE);				/* add/delete */
				DiaDimItem(dia, DALC_DELPHASE);
				DiaDimItem(dia, DALC_PHASELIST_L);			/* list header */
				DiaDimItem(dia, DALC_CYCLES_L);				/* cycles */
				DiaDimItem(dia, DALC_CYCLES);
				DiaNoEditControl(dia, DALC_CYCLES);
				DiaUnDimItem(dia, DALC_FREQPERIOD_L);		/* freq/per */
				DiaUnDimItem(dia, DALC_FREQPERIOD);
				DiaEditControl(dia, DALC_FREQPERIOD);
			}
			continue;
		}
		if (itemHit == DALC_STRENGTHWEAK || itemHit == DALC_STRENGTHNORM ||
			itemHit == DALC_STRENGTHSTRONG)
		{
			DiaSetControl(dia, DALC_STRENGTHWEAK, 0);
			DiaSetControl(dia, DALC_STRENGTHNORM, 0);
			DiaSetControl(dia, DALC_STRENGTHSTRONG, 0);
			DiaSetControl(dia, itemHit, 1);
			continue;
		}
		if (itemHit == DALC_PHASELOW || itemHit == DALC_PHASEHIGH ||
			itemHit == DALC_PHASEUNDEF)
		{
			DiaSetControl(dia, DALC_PHASELOW, 0);
			DiaSetControl(dia, DALC_PHASEHIGH, 0);
			DiaSetControl(dia, DALC_PHASEUNDEF, 0);
			DiaSetControl(dia, itemHit, 1);
			continue;
		}
		if (itemHit == DALC_ADDPHASE)
		{
			/* add phase */
			if (phases >= MAXPHASES) continue;
			line = DiaGetText(dia, DALC_DURATION);
			durval[phases] = (float)atof(line);
			if (DiaGetControl(dia, DALC_PHASELOW) != 0) levval[phases] = 0; else
				if (DiaGetControl(dia, DALC_PHASEHIGH) != 0) levval[phases] = 1; else
					levval[phases] = 2;
			phases++;
			sim_writephases(durval, levval, phases, dia);
			continue;
		}
		if (itemHit == DALC_DELPHASE)
		{
			/* delete phase */
			if (phases <= 0) continue;
			j = DiaGetCurLine(dia, DALC_PHASELIST);
			for(i=j; i<phases; i++)
			{
				durval[i] = durval[i+1];
				levval[i] = levval[i+1];
			}
			phases--;
			sim_writephases(durval, levval, phases, dia);
			continue;
		}
	}
	retval = 0;
	if (itemHit == OK && DiaGetControl(dia, DALC_FREQUENCY) != 0)
	{
		paramstart[0] = "frequency";
		paramstart[1] = us_putintoinfstr(DiaGetText(dia, DALC_FREQPERIOD));
		retval = 2;
	} else if (itemHit == OK && DiaGetControl(dia, DALC_PERIOD) != 0)
	{
		paramstart[0] = "period";
		paramstart[1] = us_putintoinfstr(DiaGetText(dia, DALC_FREQPERIOD));
		retval = 2;
	} else if (itemHit == OK && DiaGetControl(dia, DALC_CUSTOM) != 0)
	{
		paramstart[0] = "custom";
		paramstart[1] = us_putintoinfstr(DiaGetText(dia, DALC_DISTRIBUTION));
		if (DiaGetControl(dia, DALC_STRENGTHWEAK) != 0) paramstart[2] = "1"; else
			if (DiaGetControl(dia, DALC_STRENGTHNORM) != 0) paramstart[2] = "2"; else
				paramstart[2] = "3";
		paramstart[3] = us_putintoinfstr(DiaGetText(dia, DALC_CYCLES));
		retval = 4;
		if (first == 0) for(i=0; i<MAXPHASES; i++) retdur[i] = 0;
		first++;
		for(i=0; i<phases; i++)
		{
			if (retdur[i] != 0) efree(retdur[i]);
			retdur[i] = (char *)emalloc(50, sim_tool->cluster);
			if (retdur[i] == 0) break;
			(void)sprintf(retdur[i], "%e", durval[i]);
			paramstart[retval++] = (char *)(levval[i]==0 ? "low" : (levval[i]==1 ? "high" : "undefined"));
			paramstart[retval++] = retdur[i];
		}
	}
	DiaDoneDialog(dia);
	return(retval);
}

void sim_writephases(float durval[], INTBIG levval[], INTBIG phases, void *dia)
{
	INTBIG i;
	char lne[50];

	DiaLoadTextDialog(dia, DALC_PHASELIST, DiaNullDlogList, DiaNullDlogItem, DiaNullDlogDone, -1);
	for(i=0; i<phases; i++)
	{
		(void)sprintf(lne, _("%s for %e"),
			(levval[i]==0 ? _("low") : (levval[i]==1 ? _("high") : _("undefined"))), durval[i]);
		DiaStuffLine(dia, DALC_PHASELIST, lne);
	}
	if (phases <= 0) DiaSelectLine(dia, DALC_PHASELIST, -1); else DiaSelectLine(dia, DALC_PHASELIST, 0);
}

/* SPICE Options */
static DIALOGITEM sim_spiceoptdialogitems[] =
{
 /*  1 */ {0, {48,420,72,480}, BUTTON, N_("OK")},
 /*  2 */ {0, {8,420,32,480}, BUTTON, N_("Cancel")},
 /*  3 */ {0, {104,8,120,224}, POPUP, ""},
 /*  4 */ {0, {8,120,24,211}, POPUP, ""},
 /*  5 */ {0, {588,216,604,375}, RADIO, N_("Use Model from File:")},
 /*  6 */ {0, {564,216,580,426}, RADIO, N_("Derive Model from Circuitry")},
 /*  7 */ {0, {32,120,48,164}, POPUP, "1"},
 /*  8 */ {0, {588,384,604,451}, BUTTON, N_("Browse")},
 /*  9 */ {0, {612,224,644,480}, EDITTEXT, ""},
 /* 10 */ {0, {32,240,48,372}, CHECK, N_("Use Node Names")},
 /* 11 */ {0, {8,240,24,359}, CHECK, N_("Use Parasitics")},
 /* 12 */ {0, {8,8,24,117}, MESSAGE, N_("SPICE Engine:")},
 /* 13 */ {0, {32,8,48,117}, MESSAGE, N_("SPICE Level:")},
 /* 14 */ {0, {352,8,368,192}, RADIO, N_("Use Built-in Header Cards")},
 /* 15 */ {0, {400,8,416,204}, RADIO, N_("Use Header Cards from File:")},
 /* 16 */ {0, {400,208,432,472}, EDITTEXT, ""},
 /* 17 */ {0, {448,8,464,132}, RADIO, N_("No Trailer Cards")},
 /* 18 */ {0, {496,8,512,204}, RADIO, N_("Include Trailer from File:")},
 /* 19 */ {0, {496,208,528,472}, EDITTEXT, ""},
 /* 20 */ {0, {208,8,224,281}, MESSAGE, ""},
 /* 21 */ {0, {416,36,432,204}, BUTTON, N_("Browse Header Card File")},
 /* 22 */ {0, {512,36,528,204}, BUTTON, N_("Browse Trailer Card File")},
 /* 23 */ {0, {564,8,644,206}, SCROLL, ""},
 /* 24 */ {0, {544,8,560,80}, MESSAGE, N_("For Facet:")},
 /* 25 */ {0, {352,196,368,480}, BUTTON, N_("Edit Built-in Headers for Technology/Level")},
 /* 26 */ {0, {200,8,201,480}, DIVIDELINE, ""},
 /* 27 */ {0, {536,8,537,480}, DIVIDELINE, ""},
 /* 28 */ {0, {232,72,312,266}, SCROLL, ""},
 /* 29 */ {0, {232,8,248,68}, MESSAGE, N_("Layer:")},
 /* 30 */ {0, {232,272,248,364}, MESSAGE, N_("Resistance:")},
 /* 31 */ {0, {256,272,272,364}, MESSAGE, N_("Capacitance:")},
 /* 32 */ {0, {280,272,296,364}, MESSAGE, N_("Edge Cap:")},
 /* 33 */ {0, {232,372,248,472}, EDITTEXT, ""},
 /* 34 */ {0, {256,372,272,472}, EDITTEXT, ""},
 /* 35 */ {0, {280,372,296,472}, EDITTEXT, ""},
 /* 36 */ {0, {320,8,336,124}, MESSAGE, N_("Min. Resistance:")},
 /* 37 */ {0, {320,128,336,216}, EDITTEXT, ""},
 /* 38 */ {0, {320,240,336,380}, MESSAGE, N_("Min. Capacitance:")},
 /* 39 */ {0, {320,384,336,472}, EDITTEXT, ""},
 /* 40 */ {0, {56,240,72,408}, CHECK, N_("Read RAWFILE Output")},
 /* 41 */ {0, {80,240,96,408}, CHECK, N_("Force Global VDD/GND")},
 /* 42 */ {0, {176,8,192,176}, MESSAGE, N_("SPICE primitive set:")},
 /* 43 */ {0, {176,180,192,408}, POPUP, ""},
 /* 44 */ {0, {128,60,144,232}, MESSAGE, ""},
 /* 45 */ {0, {128,8,144,56}, MESSAGE, N_("Run:")},
 /* 46 */ {0, {152,60,168,404}, EDITTEXT, ""},
 /* 47 */ {0, {152,8,168,56}, MESSAGE, N_("With:")},
 /* 48 */ {0, {104,240,120,408}, CHECK, N_("Use Facet Parameters")},
 /* 49 */ {0, {376,8,392,288}, RADIO, N_("Use Header Cards with extension:")},
 /* 50 */ {0, {376,296,392,392}, EDITTEXT, ""},
 /* 51 */ {0, {472,8,488,288}, RADIO, N_("Use Trailer Cards with extension:")},
 /* 52 */ {0, {472,296,488,392}, EDITTEXT, ""},
 /* 53 */ {0, {344,8,345,480}, DIVIDELINE, ""},
 /* 54 */ {0, {128,240,144,420}, CHECK, N_("Write Trans Sizes in Lambda")}
};
static DIALOG sim_spiceoptdialog = {{50,75,703,564}, N_("SPICE Options"), 0, 54, sim_spiceoptdialogitems, 0, 0};

/* special items for the "SPICE options" dialog: */
#define DSPO_AFTERWRITING        3		/* Action after writing deck (popup) */
#define DSPO_SPICETYPE           4		/* Spice type (2, 3, H, P) (popup) */
#define DSPO_MODELSFROMFILE      5		/* Use models from file (radio) */
#define DSPO_MODELSFROMCIRC      6		/* Derive models from circuitry (radio) */
#define DSPO_SPICELEVEL          7		/* Spice level (popup) */
#define DSPO_SETMODELFILE        8		/* Set model file (button) */
#define DSPO_MODELFILE           9		/* Model file (edit text) */
#define DSPO_NODENAMES          10		/* Use node names (check) */
#define DSPO_PARASITICS         11		/* Use parasitics (check) */
#define DSPO_USEOWNHEADERS      14		/* Use built-in header cards (radio) */
#define DSPO_USEFILEHEADERS     15		/* Use header cards from file (radio) */
#define DSPO_HEADERFILE         16		/* Header card file (edit text) */
#define DSPO_NOTRAILERS         17		/* No trailer cards (radio) */
#define DSPO_FILETRAILERS       18		/* Trailer cards from file (radio) */
#define DSPO_TRAILERFILE        19		/* Trailer card file (edit text) */
#define DSPO_TECHNAME           20		/* Technology name (stat text) */
#define DSPO_SETHEADERFILE      21		/* Set header card file (button) */
#define DSPO_SETTRAILERFILE     22		/* Set trailer card file (button) */
#define DSPO_FACETLIST          23		/* Facet list (scroll) */
#define DSPO_EDITHEADERS        25		/* Edit built-in header cards (button) */
#define DSPO_LAYERLIST          28		/* Layer list (scroll) */
#define DSPO_LAYERRESISTANCE    33		/* Layer resistance (edit text) */
#define DSPO_LAYERCAPACITANCE   34		/* Layer capacitance (edit text) */
#define DSPO_LAYERECAPACITANCE  35		/* Layer edge capacitance (edit text) */
#define DSPO_MINRESISTANCE      37		/* Minimum resistance (edit text) */
#define DSPO_MINCAPACITANCE     39		/* Minimum capacitance (edit text) */
#define DSPO_RAWFILE            40		/* Read RAWFILE output (check) */
#define DSPO_GLOBALPG           41		/* Use global Power and Ground (check) */
#define DSPO_PRIMS              43		/* Name of SPICE primitive set (popup) */
#define DSPO_RUNCOMMAND         44		/* SPICE command to execute (stat text) */
#define DSPO_RUNCOMMAND_L       45		/* SPICE command to execute label (stat text) */
#define DSPO_RUNARGS            46		/* SPICE command arguments (edit text) */
#define DSPO_RUNARGS_L          47		/* SPICE command arguments label (stat text) */
#define DSPO_USEFACETPARAMS     48		/* Use Facet Parameters (check) */
#define DSPO_USEEXTHEADERS      49		/* Use header cards with extension (button) */
#define DSPO_EXTHEADEREXT       50		/* Extension to use for header cards (edit text) */
#define DSPO_EXTTRAILERFILE     51		/* Use trailer cards with extension (button) */
#define DSPO_EXTTRAILEREXT      52		/* Extension to use for trailer cards (edit text) */
#define DSPO_WRITELAMBDA		54		/* Write transistors sizes in lambdas */

void sim_spicedlog(void)
{
	REGISTER INTBIG itemHit, i, j, k, l, spice_state, newspice_state, oldplease,
		level, execute, spicepartcount;
	INTBIG dummy;
	BOOLEAN reschanged, capchanged, ecapchanged, minreschanged, mincapchanged, canexecute;
	char *subparams[3], *pt, header[200], *dummyfile[1], floatconv[30], *newlang[5],
		**filelist, **spicepartlist, *runargs;
	static char *formatnames[4] = {N_("Spice 2"), N_("Spice 3"), N_("HSpice"), N_("PSpice")};
	static char *levelnames[3] = {"1", "2", "3"};
	static char *executenames[5] = {N_("Don't Run SPICE"), N_("Run SPICE"), N_("Run SPICE Quietly"),
		N_("Run SPICE, Read Output"), N_("Run SPICE Quietly, Read Output")};
	static char qual[50];
	float *res, *cap, *ecap, minres, mincap, v;
	REGISTER VARIABLE *var, *varres, *varcap, *varecap;
	REGISTER NODEPROTO *np;
	REGISTER WINDOWPART *w;
	REGISTER TECHNOLOGY *curtech;
	REGISTER EDITOR *ed;
	extern COMCOMP us_colorreadp;
	REGISTER void *infstr, *dia;

	/* Display the SPICE options dialog box */
	dia = DiaInitDialog(&sim_spiceoptdialog);
	if (dia == 0) return;
	for(i=0; i<5; i++) newlang[i] = _(executenames[i]);
	DiaSetPopup(dia, DSPO_AFTERWRITING, 5, newlang);
	for(i=0; i<4; i++) newlang[i] = _(formatnames[i]);
	DiaSetPopup(dia, DSPO_SPICETYPE, 4, newlang);
	for(i=0; i<3; i++) newlang[i] = _(levelnames[i]);
	DiaSetPopup(dia, DSPO_SPICELEVEL, 3, newlang);

	/* see which technology to describe */
	curtech = defschematictechnology(el_curtech);

	/* state of the execute flag */
	canexecute = graphicshas(CANRUNPROCESS);
	if (canexecute)
	{
		DiaUnDimItem(dia, DSPO_AFTERWRITING);
		DiaUnDimItem(dia, DSPO_RUNCOMMAND);
		DiaUnDimItem(dia, DSPO_RUNCOMMAND_L);
		DiaUnDimItem(dia, DSPO_RUNARGS);
		DiaUnDimItem(dia, DSPO_RUNARGS_L);
		var = getvalkey((INTBIG)sim_tool, VTOOL, VINTEGER, sim_dontrunkey);
		if (var == NOVARIABLE) execute = 3; else
		{
			switch (var->addr)
			{
				case SIMRUNNO:        execute = 0;   break;
				case SIMRUNYES:       execute = 1;   break;
				case SIMRUNYESQ:      execute = 2;   break;
				case SIMRUNYESPARSE:  execute = 3;   break;
				case SIMRUNYESQPARSE: execute = 4;   break;
			}
		}
		DiaSetPopupEntry(dia, DSPO_AFTERWRITING, execute);
		pt = egetenv("ELECTRIC_SPICELOC");
		if (pt == NULL) pt = SPICELOC;
		DiaSetText(dia, DSPO_RUNCOMMAND, pt);
		var = getvalkey((INTBIG)sim_tool, VTOOL, VSTRING, sim_spice_runargskey);
		if (var == NOVARIABLE) runargs = "%.spi"; else runargs = (char *)var->addr;
		DiaSetText(dia, DSPO_RUNARGS, runargs);
	} else
	{
		DiaDimItem(dia, DSPO_AFTERWRITING);
		DiaDimItem(dia, DSPO_RUNCOMMAND);
		DiaDimItem(dia, DSPO_RUNCOMMAND_L);
		DiaDimItem(dia, DSPO_RUNARGS);
		DiaDimItem(dia, DSPO_RUNARGS_L);
	}

	/* get the spice level */
	var = getvalkey((INTBIG)sim_tool, VTOOL, VINTEGER, sim_spice_levelkey);
	if (var == NOVARIABLE) level = 1; else level = var->addr;
	DiaSetPopupEntry(dia, DSPO_SPICELEVEL, level-1);

	/* get miscellaneous state */
	var = getvalkey((INTBIG)sim_tool, VTOOL, VINTEGER, sim_spice_statekey);
	if (var != NOVARIABLE) spice_state = var->addr; else spice_state = 0;
	if ((spice_state&SPICENODENAMES) != 0) DiaSetControl(dia, DSPO_NODENAMES, 1);
	if ((spice_state&SPICERESISTANCE) != 0) DiaSetControl(dia, DSPO_PARASITICS, 1);
	if ((spice_state&SPICERAWFILE) != 0) DiaSetControl(dia, DSPO_RAWFILE, 1);
	if ((spice_state&SPICEGLOBALPG) != 0) DiaSetControl(dia, DSPO_GLOBALPG, 1);
	if ((spice_state&SPICEFACETPARAM) != 0) DiaSetControl(dia, DSPO_USEFACETPARAMS, 1);
	if ((spice_state&SPICEUSELAMBDAS) != 0) DiaSetControl(dia, DSPO_WRITELAMBDA, 1);
	switch (spice_state & SPICETYPE)
	{
		case SPICE2:
			DiaSetPopupEntry(dia, DSPO_SPICETYPE, 0);
			DiaSetControl(dia, DSPO_NODENAMES, 0);
			DiaDimItem(dia, DSPO_NODENAMES);
			DiaSetControl(dia, DSPO_GLOBALPG, 0);
			DiaDimItem(dia, DSPO_GLOBALPG);
			break;
		case SPICE3:
			DiaSetPopupEntry(dia, DSPO_SPICETYPE, 1);
			DiaUnDimItem(dia, DSPO_NODENAMES);
			DiaSetControl(dia, DSPO_GLOBALPG, 0);
			DiaDimItem(dia, DSPO_GLOBALPG);
			break;
		case SPICEHSPICE:
			DiaSetPopupEntry(dia, DSPO_SPICETYPE, 2);
			DiaUnDimItem(dia, DSPO_NODENAMES);
			DiaUnDimItem(dia, DSPO_GLOBALPG);
			break;
		case SPICEPSPICE:
			DiaSetPopupEntry(dia, DSPO_SPICETYPE, 3);
			DiaUnDimItem(dia, DSPO_NODENAMES);
			DiaUnDimItem(dia, DSPO_GLOBALPG);
			break;
	}

	/* get model header cards */
	DiaDimItem(dia, DSPO_HEADERFILE);
	DiaDimItem(dia, DSPO_EXTHEADEREXT);
	infstr = initinfstr();
	formatinfstr(infstr, _("For technology: %s"), curtech->techname);
	DiaSetText(dia, DSPO_TECHNAME, returninfstr(infstr));
	var = getval((INTBIG)curtech, VTECHNOLOGY, VSTRING, "SIM_spice_model_file");
	if (var == NOVARIABLE) DiaSetControl(dia, DSPO_USEOWNHEADERS, 1); else
	{
		pt = (char *)var->addr;
		if (strncmp(pt, ":::::", 5) == 0)
		{
			DiaUnDimItem(dia, DSPO_EXTHEADEREXT);
			DiaSetText(dia, DSPO_EXTHEADEREXT, &pt[5]);
			DiaSetControl(dia, DSPO_USEEXTHEADERS, 1);
		} else
		{
			DiaUnDimItem(dia, DSPO_HEADERFILE);
			DiaSetText(dia, DSPO_HEADERFILE, pt);
			DiaSetControl(dia, DSPO_USEFILEHEADERS, 1);
		}
	}

	/* get trailer cards */
	DiaDimItem(dia, DSPO_TRAILERFILE);
	DiaDimItem(dia, DSPO_EXTTRAILEREXT);
	var = getval((INTBIG)curtech, VTECHNOLOGY, VSTRING, "SIM_spice_trailer_file");
	if (var == NOVARIABLE) DiaSetControl(dia, DSPO_NOTRAILERS, 1); else
	{
		pt = (char *)var->addr;
		if (strncmp(pt, ":::::", 5) == 0)
		{
			DiaUnDimItem(dia, DSPO_EXTTRAILEREXT);
			DiaSetText(dia, DSPO_EXTTRAILEREXT, &pt[5]);
			DiaSetControl(dia, DSPO_EXTTRAILERFILE, 1);
		} else
		{
			DiaUnDimItem(dia, DSPO_TRAILERFILE);
			DiaSetText(dia, DSPO_TRAILERFILE, pt);
			DiaSetControl(dia, DSPO_FILETRAILERS, 1);
		}
	}

	/* list of facets for model description files */
	DiaInitTextDialog(dia, DSPO_FACETLIST, topoffacets, nextfacets, DiaNullDlogDone,
		0, SCSELMOUSE|SCSELKEY|SCREPORT);
	for(np = el_curlib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
	{
		np->temp1 = 0;
		var = getval((INTBIG)np, VNODEPROTO, VSTRING, "SIM_spice_behave_file");
		if (var != NOVARIABLE)
			(void)allocstring((char **)&np->temp1, (char *)var->addr, el_tempcluster);
	}
	pt = DiaGetScrollLine(dia, DSPO_FACETLIST, DiaGetCurLine(dia, DSPO_FACETLIST));
	DiaSetControl(dia, DSPO_MODELSFROMCIRC, 1);
	DiaSetControl(dia, DSPO_MODELSFROMFILE, 0);
	DiaDimItem(dia, DSPO_MODELFILE);
	np = getnodeproto(pt);
	if (np != NONODEPROTO)
	{
		var = getval((INTBIG)np, VNODEPROTO, VSTRING, "SIM_spice_behave_file");
		if (var != NOVARIABLE)
		{
			DiaSetControl(dia, DSPO_MODELSFROMCIRC, 0);
			DiaSetControl(dia, DSPO_MODELSFROMFILE, 1);
			DiaUnDimItem(dia, DSPO_MODELFILE);
			DiaSetText(dia, DSPO_MODELFILE, (char *)var->addr);
		}
	}

	/* setup the SPICE primitives */
	j = filesindirectory(el_libdir, &filelist);
	spicepartcount = 0;
	for(i=0; i<j; i++)
	{
		pt = filelist[i];
		l = strlen(pt);
		if (namesamen(pt, "spiceparts", 10) != 0) continue;
		if (namesame(&pt[l-4], ".mac") != 0) continue;
		spicepartcount++;
	}
	if (spicepartcount > 0)
	{
		k = 0;
		spicepartlist = (char **)emalloc(spicepartcount * (sizeof (char *)), el_tempcluster);
		if (spicepartlist == 0) return;
		spicepartcount = 0;
		for(i=0; i<j; i++)
		{
			pt = filelist[i];
			l = strlen(pt);
			if (namesamen(pt, "spiceparts", 10) != 0) continue;
			if (namesame(&pt[l-4], ".mac") != 0) continue;
			(void)allocstring(&spicepartlist[spicepartcount], pt, el_tempcluster);
			spicepartlist[spicepartcount][l-4] = 0;
			if (namesame(spicepartlist[spicepartcount], sim_spice_parts) == 0)
				k = spicepartcount;
			spicepartcount++;
		}
		DiaSetPopup(dia, DSPO_PRIMS, spicepartcount, spicepartlist);
		DiaSetPopupEntry(dia, DSPO_PRIMS, k);
	}

	/* list of layers in this technology, with resistance, capacitance, etc. */
	DiaInitTextDialog(dia, DSPO_LAYERLIST, DiaNullDlogList, DiaNullDlogItem, DiaNullDlogDone,
		-1, SCSELMOUSE|SCREPORT);
	varres = getval((INTBIG)curtech, VTECHNOLOGY, VFLOAT|VISARRAY, "SIM_spice_resistance");
	varcap = getval((INTBIG)curtech, VTECHNOLOGY, VFLOAT|VISARRAY, "SIM_spice_capacitance");
	varecap = getval((INTBIG)curtech, VTECHNOLOGY, VFLOAT|VISARRAY, "SIM_spice_edge_capacitance");
	res = (float *)emalloc(curtech->layercount * (sizeof (float)), el_tempcluster);
	cap = (float *)emalloc(curtech->layercount * (sizeof (float)), el_tempcluster);
	ecap = (float *)emalloc(curtech->layercount * (sizeof (float)), el_tempcluster);
	for(i=0; i<curtech->layercount; i++)
	{
		DiaStuffLine(dia, DSPO_LAYERLIST, layername(curtech, i));
		res[i] = cap[i] = ecap[i] = 0.0;
		if (varres != NOVARIABLE && i < getlength(varres))
			res[i] = ((float *)varres->addr)[i];
		if (varcap != NOVARIABLE && i < getlength(varcap))
			cap[i] = ((float *)varcap->addr)[i];
		if (varecap != NOVARIABLE && i < getlength(varecap))
			ecap[i] = ((float *)varecap->addr)[i];
	}
	DiaSelectLine(dia, DSPO_LAYERLIST, 0);
	sprintf(floatconv, "%g", res[0]);    DiaSetText(dia, DSPO_LAYERRESISTANCE, floatconv);
	sprintf(floatconv, "%g", cap[0]);    DiaSetText(dia, DSPO_LAYERCAPACITANCE, floatconv);
	sprintf(floatconv, "%g", ecap[0]);   DiaSetText(dia, DSPO_LAYERECAPACITANCE, floatconv);
	reschanged = capchanged = ecapchanged = minreschanged = mincapchanged = FALSE;

	/* get minimum resistance and capacitances */
	var = getval((INTBIG)curtech, VTECHNOLOGY, VFLOAT, "SIM_spice_min_resistance");
	if (var == NOVARIABLE) minres = 0.0; else
		minres = castfloat(var->addr);
	sprintf(floatconv, "%g", minres);    DiaSetText(dia, DSPO_MINRESISTANCE, floatconv);
	var = getval((INTBIG)curtech, VTECHNOLOGY, VFLOAT, "SIM_spice_min_capacitance");
	if (var == NOVARIABLE) mincap = 0.0; else
		mincap = castfloat(var->addr);
	sprintf(floatconv, "%g", mincap);    DiaSetText(dia, DSPO_MINCAPACITANCE, floatconv);

	for(;;)
	{
		itemHit = DiaNextHit(dia);
		if (itemHit == OK || itemHit == CANCEL || itemHit == DSPO_EDITHEADERS) break;

		if (itemHit == DSPO_SPICETYPE)
		{
			/* file format button */
			i = DiaGetPopupEntry(dia, DSPO_SPICETYPE);
			if (i != 0) DiaUnDimItem(dia, DSPO_NODENAMES); else
			{
				DiaSetControl(dia, DSPO_NODENAMES, 0);
				DiaDimItem(dia, DSPO_NODENAMES);
			}
			if (i != 0 && i != 1) DiaUnDimItem(dia, DSPO_GLOBALPG); else
			{
				DiaSetControl(dia, DSPO_GLOBALPG, 0);
				DiaDimItem(dia, DSPO_GLOBALPG);
			}
			continue;
		}

		if (itemHit == DSPO_NODENAMES || itemHit == DSPO_PARASITICS ||
			itemHit == DSPO_RAWFILE || itemHit == DSPO_GLOBALPG ||
			itemHit == DSPO_USEFACETPARAMS || itemHit == DSPO_WRITELAMBDA)
		{
			/* check boxes */
			DiaSetControl(dia, itemHit, 1-DiaGetControl(dia, itemHit));
			continue;
		}

		if (itemHit == DSPO_USEOWNHEADERS || itemHit == DSPO_USEFILEHEADERS ||
			itemHit == DSPO_USEEXTHEADERS)
		{
			/* header model cards button */
			DiaSetControl(dia, DSPO_USEOWNHEADERS, 0);
			DiaSetControl(dia, DSPO_USEFILEHEADERS, 0);
			DiaSetControl(dia, DSPO_USEEXTHEADERS, 0);
			DiaSetControl(dia, itemHit, 1);
			DiaDimItem(dia, DSPO_HEADERFILE);
			DiaDimItem(dia, DSPO_EXTHEADEREXT);
			if (itemHit == DSPO_USEFILEHEADERS)
			{
				DiaUnDimItem(dia, DSPO_HEADERFILE);
			} else if (itemHit == DSPO_USEEXTHEADERS)
			{
				DiaUnDimItem(dia, DSPO_EXTHEADEREXT);
			}
			continue;
		}

		if (itemHit == DSPO_SETHEADERFILE)
		{
			/* select model cards */
			oldplease = el_pleasestop;
			infstr = initinfstr();
			addstringtoinfstr(infstr, "spice/");
			addstringtoinfstr(infstr, _("SPICE Model File"));
			i = ttygetparam(returninfstr(infstr), &us_colorreadp, 1, subparams);
			el_pleasestop = oldplease;
			if (i != 0 && subparams[0][0] != 0)
			{
				DiaDimItem(dia, DSPO_EXTHEADEREXT);
				DiaUnDimItem(dia, DSPO_HEADERFILE);
				DiaSetText(dia, DSPO_HEADERFILE, subparams[0]);
				DiaSetControl(dia, DSPO_USEOWNHEADERS, 0);
				DiaSetControl(dia, DSPO_USEFILEHEADERS, 1);
				DiaSetControl(dia, DSPO_USEEXTHEADERS, 0);
			}
			continue;
		}

		if (itemHit == DSPO_NOTRAILERS || itemHit == DSPO_FILETRAILERS ||
			itemHit == DSPO_EXTTRAILERFILE)
		{
			/* trailer cards button */
			DiaSetControl(dia, DSPO_NOTRAILERS, 0);
			DiaSetControl(dia, DSPO_FILETRAILERS, 0);
			DiaSetControl(dia, DSPO_EXTTRAILERFILE, 0);
			DiaSetControl(dia, itemHit, 1);
			DiaDimItem(dia, DSPO_TRAILERFILE);
			DiaDimItem(dia, DSPO_EXTTRAILEREXT);
			if (itemHit == DSPO_FILETRAILERS)
			{
				DiaUnDimItem(dia, DSPO_TRAILERFILE);
			} else if (itemHit == DSPO_EXTTRAILERFILE)
			{
				DiaUnDimItem(dia, DSPO_EXTTRAILEREXT);
			}
			continue;
		}

		if (itemHit == DSPO_SETTRAILERFILE)
		{
			/* select trailer cards */
			oldplease = el_pleasestop;
			infstr = initinfstr();
			addstringtoinfstr(infstr, "spice/");
			addstringtoinfstr(infstr, _("SPICE Trailer File"));
			i = ttygetparam(returninfstr(infstr), &us_colorreadp, 1, subparams);
			el_pleasestop = oldplease;
			if (i != 0 && subparams[0][0] != 0)
			{
				DiaUnDimItem(dia, DSPO_TRAILERFILE);
				DiaSetText(dia, DSPO_TRAILERFILE, subparams[0]);
				DiaDimItem(dia, DSPO_EXTTRAILEREXT);
				DiaSetControl(dia, DSPO_NOTRAILERS, 0);
				DiaSetControl(dia, DSPO_FILETRAILERS, 1);
				DiaSetControl(dia, DSPO_EXTTRAILERFILE, 0);
			}
			continue;
		}

		if (itemHit == DSPO_LAYERLIST)
		{
			/* selection of layers */
			i = DiaGetCurLine(dia, DSPO_LAYERLIST);
			if (i < 0 || i >= curtech->layercount) continue;
			sprintf(floatconv, "%g", res[i]);   DiaSetText(dia, DSPO_LAYERRESISTANCE, floatconv);
			sprintf(floatconv, "%g", cap[i]);   DiaSetText(dia, DSPO_LAYERCAPACITANCE, floatconv);
			sprintf(floatconv, "%g", ecap[i]);   DiaSetText(dia, DSPO_LAYERECAPACITANCE, floatconv);
			continue;
		}

		if (itemHit == DSPO_LAYERRESISTANCE)
		{
			i = DiaGetCurLine(dia, DSPO_LAYERLIST);
			if (i < 0 || i >= curtech->layercount) continue;
			v = (float)atof(DiaGetText(dia, DSPO_LAYERRESISTANCE));
			if (!floatsequal(res[i], v)) reschanged = TRUE;
			res[i] = v;
			continue;
		}
		if (itemHit == DSPO_LAYERCAPACITANCE)
		{
			i = DiaGetCurLine(dia, DSPO_LAYERLIST);
			if (i < 0 || i >= curtech->layercount) continue;
			v = (float)atof(DiaGetText(dia, DSPO_LAYERCAPACITANCE));
			if (!floatsequal(cap[i], v)) capchanged = TRUE;
			cap[i] = v;
			continue;
		}
		if (itemHit == DSPO_LAYERECAPACITANCE)
		{
			i = DiaGetCurLine(dia, DSPO_LAYERLIST);
			if (i < 0 || i >= curtech->layercount) continue;
			v = (float)atof(DiaGetText(dia, DSPO_LAYERECAPACITANCE));
			if (!floatsequal(ecap[i], v)) ecapchanged = TRUE;
			ecap[i] = v;
			continue;
		}
		if (itemHit == DSPO_MINRESISTANCE)
		{
			v = (float)atof(DiaGetText(dia, DSPO_MINRESISTANCE));
			if (!floatsequal(v, minres)) minreschanged = TRUE;
			continue;
		}
		if (itemHit == DSPO_MINCAPACITANCE)
		{
			v = (float)atof(DiaGetText(dia, DSPO_MINCAPACITANCE));
			if (!floatsequal(v, mincap)) mincapchanged = TRUE;
			continue;
		}

		if (itemHit == DSPO_FACETLIST)
		{
			/* selection of facets */
			DiaSetControl(dia, DSPO_MODELSFROMCIRC, 1);
			DiaSetControl(dia, DSPO_MODELSFROMFILE, 0);
			DiaSetText(dia, DSPO_MODELFILE, "");
			pt = DiaGetScrollLine(dia, DSPO_FACETLIST, DiaGetCurLine(dia, DSPO_FACETLIST));
			np = getnodeproto(pt);
			if (np != NONODEPROTO && np->temp1 != 0)
			{
				DiaSetControl(dia, DSPO_MODELSFROMCIRC, 0);
				DiaSetControl(dia, DSPO_MODELSFROMFILE, 1);
				DiaSetText(dia, DSPO_MODELFILE, (char *)np->temp1);
			}
			continue;
		}

		if (itemHit == DSPO_MODELFILE)
		{
			/* changing a facet's model file */
			pt = DiaGetScrollLine(dia, DSPO_FACETLIST, DiaGetCurLine(dia, DSPO_FACETLIST));
			np = getnodeproto(pt);
			if (np != NONODEPROTO)
			{
				if (np->temp1 != 0) efree((char *)np->temp1);
				(void)allocstring((char **)&np->temp1, DiaGetText(dia, DSPO_MODELFILE), el_tempcluster);
			}
			continue;
		}

		if (itemHit == DSPO_MODELSFROMCIRC || itemHit == DSPO_MODELSFROMFILE)
		{
			/* model for facet */
			pt = DiaGetScrollLine(dia, DSPO_FACETLIST, DiaGetCurLine(dia, DSPO_FACETLIST));
			np = getnodeproto(pt);
			DiaSetControl(dia, DSPO_MODELSFROMCIRC, 0);
			DiaSetControl(dia, DSPO_MODELSFROMFILE, 0);
			DiaSetControl(dia, itemHit, 1);
			if (itemHit == DSPO_MODELSFROMFILE)
			{
				DiaUnDimItem(dia, DSPO_MODELFILE);
				if (np != NONODEPROTO)
				{
					infstr = initinfstr();
					formatinfstr(infstr, "%s.spi", np->cell->cellname);
					pt = returninfstr(infstr);
					if (np->temp1 != 0) efree((char *)np->temp1);
					(void)allocstring((char **)&np->temp1, pt, el_tempcluster);
					DiaSetText(dia, DSPO_MODELFILE, pt);
				}
			} else
			{
				DiaSetText(dia, DSPO_MODELFILE, "");
				DiaDimItem(dia, DSPO_MODELFILE);
				if (np != NONODEPROTO)
				{
					if (np->temp1 != 0) efree((char *)np->temp1);
					np->temp1 = 0;
				}
			}
			continue;
		}

		if (itemHit == DSPO_SETMODELFILE)
		{
			/* set model for facet */
			pt = DiaGetScrollLine(dia, DSPO_FACETLIST, DiaGetCurLine(dia, DSPO_FACETLIST));
			np = getnodeproto(pt);
			if (np == NONODEPROTO) continue;
			oldplease = el_pleasestop;
			infstr = initinfstr();
			addstringtoinfstr(infstr, "spice/");
			addstringtoinfstr(infstr, _("Facet's Model File"));
			i = ttygetparam(returninfstr(infstr), &us_colorreadp, 1, subparams);
			el_pleasestop = oldplease;
			if (i != 0 && subparams[0][0] != 0)
			{
				DiaUnDimItem(dia, DSPO_MODELFILE);
				DiaSetText(dia, DSPO_MODELFILE, subparams[0]);
				if (np->temp1 != 0) efree((char *)np->temp1);
				(void)allocstring((char **)&np->temp1, subparams[0], el_tempcluster);
				DiaSetControl(dia, DSPO_MODELSFROMCIRC, 0);
				DiaSetControl(dia, DSPO_MODELSFROMFILE, 1);
			}
			continue;
		}
	}

	if (itemHit != CANCEL)  /* now update the values */
	{
		if (canexecute)
		{
			i = DiaGetPopupEntry(dia, DSPO_AFTERWRITING);
			if (i != execute)
			{
				switch (i)
				{
					case 0: execute = SIMRUNNO;          break;
					case 1: execute = SIMRUNYES;         break;
					case 2: execute = SIMRUNYESQ;        break;
					case 3: execute = SIMRUNYESPARSE;    break;
					case 4: execute = SIMRUNYESQPARSE;   break;
				}
				(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_dontrunkey, execute, VINTEGER);
			}
			pt = DiaGetText(dia, DSPO_RUNARGS);
			if (strcmp(runargs, pt) != 0)
				(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_spice_runargskey,
					(INTBIG)pt, VSTRING);
		}

		i = DiaGetPopupEntry(dia, DSPO_SPICELEVEL) + 1;
		if (i != level)
			(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_spice_levelkey, i, VINTEGER);

		newspice_state = 0;
		if (DiaGetControl(dia, DSPO_NODENAMES) != 0) newspice_state |= SPICENODENAMES;
		if (DiaGetControl(dia, DSPO_PARASITICS) != 0) newspice_state |= SPICERESISTANCE;
		if (DiaGetControl(dia, DSPO_RAWFILE) != 0) newspice_state |= SPICERAWFILE;
		if (DiaGetControl(dia, DSPO_GLOBALPG) != 0) newspice_state |= SPICEGLOBALPG;
		if (DiaGetControl(dia, DSPO_USEFACETPARAMS) != 0) newspice_state |= SPICEFACETPARAM;
		if (DiaGetControl(dia, DSPO_WRITELAMBDA) != 0) newspice_state |= SPICEUSELAMBDAS;
		switch (DiaGetPopupEntry(dia, DSPO_SPICETYPE))
		{
			case 0: newspice_state |= SPICE2;        break;
			case 1: newspice_state |= SPICE3;        break;
			case 2: newspice_state |= SPICEHSPICE;   break;
			case 3: newspice_state |= SPICEPSPICE;   break;
		}
		if (newspice_state != spice_state)
			(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_spice_statekey, newspice_state, VINTEGER);

		i = DiaGetPopupEntry(dia, DSPO_PRIMS);
		if (namesame(sim_spice_parts, spicepartlist[i]) != 0)
		{
			(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_spice_partskey,
				(INTBIG)spicepartlist[i], VSTRING);
		}

		var = getval((INTBIG)curtech, VTECHNOLOGY, VSTRING, "SIM_spice_model_file");
		if (DiaGetControl(dia, DSPO_USEFILEHEADERS) != 0)
		{
			/* set model cards */
			pt = DiaGetText(dia, DSPO_HEADERFILE);
			if (var == NOVARIABLE || strcmp(pt, (char *)var->addr) != 0)
			{
				(void)setval((INTBIG)curtech, VTECHNOLOGY, "SIM_spice_model_file",
					(INTBIG)pt, VSTRING);
			}
		} else if (DiaGetControl(dia, DSPO_USEEXTHEADERS) != 0)
		{
			/* set model extension */
			infstr = initinfstr();
			formatinfstr(infstr, ":::::%s", DiaGetText(dia, DSPO_EXTHEADEREXT));
			pt = returninfstr(infstr);
			if (var == NOVARIABLE || strcmp(pt, (char *)var->addr) != 0)
			{
				(void)setval((INTBIG)curtech, VTECHNOLOGY, "SIM_spice_model_file",
					(INTBIG)pt, VSTRING);
			}
		} else
		{
			/* remove model cards */
			if (var != NOVARIABLE)
				(void)delval((INTBIG)curtech, VTECHNOLOGY, "SIM_spice_model_file");
		}

		var = getval((INTBIG)curtech, VTECHNOLOGY, VSTRING, "SIM_spice_trailer_file");
		if (DiaGetControl(dia, DSPO_FILETRAILERS) != 0)
		{
			/* set trailer cards */
			pt = DiaGetText(dia, DSPO_TRAILERFILE);
			if (var == NOVARIABLE || strcmp(pt, (char *)var->addr) != 0)
			{
				(void)setval((INTBIG)curtech, VTECHNOLOGY, "SIM_spice_trailer_file",
					(INTBIG)pt, VSTRING);
			}
		} else if (DiaGetControl(dia, DSPO_EXTTRAILERFILE) != 0)
		{
			/* set trailer extension */
			infstr = initinfstr();
			formatinfstr(infstr, ":::::%s", DiaGetText(dia, DSPO_EXTTRAILEREXT));
			pt = returninfstr(infstr);
			if (var == NOVARIABLE || strcmp(pt, (char *)var->addr) != 0)
			{
				(void)setval((INTBIG)curtech, VTECHNOLOGY, "SIM_spice_trailer_file",
					(INTBIG)pt, VSTRING);
			}
		} else
		{
			/* remove trailer cards */
			if (var != NOVARIABLE)
				(void)delval((INTBIG)curtech, VTECHNOLOGY, "SIM_spice_trailer_file");
		}
		for(np = el_curlib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		{
			var = getval((INTBIG)np, VNODEPROTO, VSTRING, "SIM_spice_behave_file");
			if (var != NOVARIABLE)
			{
				if (np->temp1 == 0)
					(void)delval((INTBIG)np, VNODEPROTO, "SIM_spice_behave_file"); else
				{
					if (strcmp((char *)np->temp1, (char *)var->addr) != 0)
						(void)setval((INTBIG)np, VNODEPROTO, "SIM_spice_behave_file", (INTBIG)np->temp1, VSTRING);
				}
			} else if (np->temp1 != 0)
				(void)setval((INTBIG)np, VNODEPROTO, "SIM_spice_behave_file", (INTBIG)np->temp1, VSTRING);
			if (np->temp1 != 0) efree((char *)np->temp1);
		}

		/* update layer resistance, capacitance, etc. */
		if (reschanged)
			setval((INTBIG)curtech, VTECHNOLOGY, "SIM_spice_resistance",
				(INTBIG)res, VFLOAT|VISARRAY|(curtech->layercount << VLENGTHSH));
		if (capchanged)
			setval((INTBIG)curtech, VTECHNOLOGY, "SIM_spice_capacitance",
				(INTBIG)cap, VFLOAT|VISARRAY|(curtech->layercount << VLENGTHSH));
		if (ecapchanged)
			setval((INTBIG)curtech, VTECHNOLOGY, "SIM_spice_edge_capacitance",
				(INTBIG)ecap, VFLOAT|VISARRAY|(curtech->layercount << VLENGTHSH));

		/* update minimum resistance and capacitance */
		if (minreschanged)
		{
			v = (float)atof(DiaGetText(dia, DSPO_MINRESISTANCE));
			setval((INTBIG)curtech, VTECHNOLOGY, "SIM_spice_min_resistance",
				castint(v), VFLOAT);
		}
		if (mincapchanged)
		{
			v = (float)atof(DiaGetText(dia, DSPO_MINCAPACITANCE));
			setval((INTBIG)curtech, VTECHNOLOGY, "SIM_spice_min_capacitance",
				castint(v), VFLOAT);
		}
	}
	DiaDoneDialog(dia);
	efree((char *)res);
	efree((char *)cap);
	efree((char *)ecap);
	if (spicepartcount > 0)
	{
		for(i=0; i<spicepartcount; i++) efree((char *)spicepartlist[i]);
		efree((char *)spicepartlist);
	}
	if (itemHit == DSPO_EDITHEADERS)
	{
		/* now edit the model cards */
		sprintf(qual, "SIM_spice_header_level%ld", level);
		sprintf(header, _("Level %ld SPICE Model cards for %s"), level, curtech->techname);

		var = getval((INTBIG)curtech, VTECHNOLOGY, -1, qual);
		if (var == NOVARIABLE)
		{
			dummyfile[0] = "";
			var = setval((INTBIG)curtech, VTECHNOLOGY, qual, (INTBIG)dummyfile,
				VSTRING|VISARRAY|(1<<VLENGTHSH));
			if (var == NOVARIABLE)
			{
				ttyputerr(_("Cannot create %s on the technology"), qual);
				return;
			}
		} else
			var->type &= ~VDONTSAVE;

		/* get a new window, put an editor in it */
		w = us_wantnewwindow(0);
		if (w == NOWINDOWPART) return;
		if (us_makeeditor(w, header, &dummy, &dummy) == NOWINDOWPART) return;
		ed = w->editor;
		ed->editobjqual = qual;
		ed->editobjaddr = (char *)curtech;
		ed->editobjtype = VTECHNOLOGY;
		ed->editobjvar = var;
		us_suspendgraphics(w);

		l = getlength(var);
		for(j=0; j<l; j++)
		{
			infstr = initinfstr();
			addstringtoinfstr(infstr, describevariable(var, j, -1));
			us_addline(w, j, returninfstr(infstr));
		}
		us_resumegraphics(w);
		w->changehandler = us_varchanges;
	}
}

/* Verilog Options */
static DIALOGITEM sim_verilogoptdialogitems[] =
{
 /*  1 */ {0, {144,396,168,454}, BUTTON, N_("OK")},
 /*  2 */ {0, {144,268,168,326}, BUTTON, N_("Cancel")},
 /*  3 */ {0, {80,216,96,375}, RADIO, N_("Use Model from File:")},
 /*  4 */ {0, {56,216,72,426}, RADIO, N_("Derive Model from Circuitry")},
 /*  5 */ {0, {80,392,96,463}, BUTTON, N_("Browse")},
 /*  6 */ {0, {104,224,136,480}, EDITTEXT, ""},
 /*  7 */ {0, {8,256,24,428}, CHECK, N_("Use ASSIGN Construct")},
 /*  8 */ {0, {28,8,168,206}, SCROLL, ""},
 /*  9 */ {0, {8,8,24,72}, MESSAGE, N_("Library:")},
 /* 10 */ {0, {8,76,24,206}, POPUP, ""},
 /* 11 */ {0, {32,256,48,428}, CHECK, N_("Default wire is Trireg")}
};
static DIALOG sim_verilogoptdialog = {{50,75,227,564}, N_("Verilog Options"), 0, 11, sim_verilogoptdialogitems, 0, 0};

/* special items for the "Verilog options" dialog: */
#define DVEO_USEMODELFILE   3		/* Use model from file (radio) */
#define DVEO_USEMODELCIRC   4		/* Use model from circuitry (radio) */
#define DVEO_SETMODELFILE   5		/* Set model file (button) */
#define DVEO_MODELFILE      6		/* Model file (edit text) */
#define DVEO_USEASSIGN      7		/* Use 'assign' construct (check) */
#define DVEO_FACETLIST      8		/* Facet list (scroll) */
#define DVEO_LIBRARYPOPUP  10		/* Library list (popup) */
#define DVEO_USETRIREG     11		/* Default wire is Trireg (check) */

/*
 * special case for the "Verilog options" dialog
 */
void sim_verilogdlog(void)
{
	INTBIG itemHit, i, verilog_state, libcount, oldplease;
	char *subparams[3], *pt, **liblist;
	REGISTER VARIABLE *var;
	REGISTER NODEPROTO *np;
	REGISTER LIBRARY *lib;
	extern COMCOMP us_colorreadp;
	REGISTER void *infstr, *dia;

	/* Display the Verilog options dialog box */
	dia = DiaInitDialog(&sim_verilogoptdialog);
	if (dia == 0) return;

	/* load up list of libraries */
	for(lib = el_curlib, libcount = 0; lib != NOLIBRARY; lib = lib->nextlibrary)
		libcount++;
	liblist = (char **)emalloc(libcount * (sizeof (char *)), el_tempcluster);
	if (liblist == 0) return;
	for(lib = el_curlib, libcount = 0; lib != NOLIBRARY; lib = lib->nextlibrary)
	{
		if ((lib->userbits&HIDDENLIBRARY) != 0) continue;
		liblist[libcount] = lib->libname;
		libcount++;
	}
	esort(liblist, libcount, sizeof (char *), sort_stringascending);
	DiaSetPopup(dia, DVEO_LIBRARYPOPUP, libcount, liblist);
	for(i=0; i<libcount; i++)
		if (strcmp(el_curlib->libname, liblist[i]) == 0) break;
	if (i < libcount) DiaSetPopupEntry(dia, DVEO_LIBRARYPOPUP, i);

	/* get miscellaneous state */
	var = getvalkey((INTBIG)sim_tool, VTOOL, VINTEGER, sim_verilog_statekey);
	if (var != NOVARIABLE) verilog_state = var->addr; else verilog_state = 0;
	if ((verilog_state&VERILOGUSEASSIGN) != 0) DiaSetControl(dia, DVEO_USEASSIGN, 1);
	if ((verilog_state&VERILOGUSETRIREG) != 0) DiaSetControl(dia, DVEO_USETRIREG, 1);

	/* cache behavior file links */
	for(lib = el_curlib; lib != NOLIBRARY; lib = lib->nextlibrary)
		for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
	{
		np->temp1 = 0;
		var = getval((INTBIG)np, VNODEPROTO, VSTRING, "SIM_verilog_behave_file");
		if (var != NOVARIABLE)
			(void)allocstring((char **)&np->temp1, (char *)var->addr, el_tempcluster);
	}

	/* make list of facets in this library */
	sim_curlib = el_curlib;
	DiaInitTextDialog(dia, DVEO_FACETLIST, sim_topoffacets, sim_nextfacets, DiaNullDlogDone,
		0, SCSELMOUSE|SCSELKEY|SCREPORT);
	sim_verdiasetfacetinfo(sim_curlib, dia);

	for(;;)
	{
		itemHit = DiaNextHit(dia);
		if (itemHit == OK || itemHit == CANCEL) break;

		if (itemHit == DVEO_USEASSIGN || itemHit == DVEO_USETRIREG)
		{
			DiaSetControl(dia, itemHit, 1-DiaGetControl(dia, itemHit));
			continue;
		}

		if (itemHit == DVEO_LIBRARYPOPUP)
		{
			i = DiaGetPopupEntry(dia, DVEO_LIBRARYPOPUP);
			if (i < 0 || i >= libcount) continue;
			sim_curlib = getlibrary(liblist[i]);
			if (sim_curlib == NOLIBRARY) continue;
			DiaLoadTextDialog(dia, DVEO_FACETLIST, sim_topoffacets, sim_nextfacets, DiaNullDlogDone, 0);
			sim_verdiasetfacetinfo(sim_curlib, dia);
			continue;
		}

		if (itemHit == DVEO_FACETLIST)
		{
			/* selection of facets */
			sim_verdiasetfacetinfo(sim_curlib, dia);
			continue;
		}

		if (itemHit == DVEO_USEMODELFILE || itemHit == DVEO_USEMODELCIRC)
		{
			/* model for facet */
			DiaSetControl(dia, DVEO_USEMODELFILE, 0);
			DiaSetControl(dia, DVEO_USEMODELCIRC, 0);
			DiaSetControl(dia, itemHit, 1);
			pt = DiaGetScrollLine(dia, DVEO_FACETLIST, DiaGetCurLine(dia, DVEO_FACETLIST));
			lib = el_curlib;   el_curlib = sim_curlib;
			np = getnodeproto(pt);
			el_curlib = lib;
			if (itemHit == DVEO_USEMODELFILE)
			{
				DiaUnDimItem(dia, DVEO_MODELFILE);
				if (np != NONODEPROTO)
				{
					infstr = initinfstr();
					formatinfstr(infstr, "%s.v", np->cell->cellname);
					pt = returninfstr(infstr);
					DiaSetText(dia, DVEO_MODELFILE, pt);
					if (np->temp1 != 0) efree((char *)np->temp1);
					(void)allocstring((char **)&np->temp1, pt, el_tempcluster);
				}
			} else
			{
				DiaSetText(dia, DVEO_MODELFILE, "");
				DiaDimItem(dia, DVEO_MODELFILE);
				if (np != NONODEPROTO)
				{
					if (np->temp1 != 0) efree((char *)np->temp1);
					np->temp1 = 0;
				}
			}
			continue;
		}

		if (itemHit == DVEO_SETMODELFILE)
		{
			/* set model for facet */
			pt = DiaGetScrollLine(dia, DVEO_FACETLIST, DiaGetCurLine(dia, DVEO_FACETLIST));
			lib = el_curlib;   el_curlib = sim_curlib;
			np = getnodeproto(pt);
			el_curlib = lib;
			if (np == NONODEPROTO) continue;
			oldplease = el_pleasestop;
			infstr = initinfstr();
			addstringtoinfstr(infstr, "verilog/");
			addstringtoinfstr(infstr, _("Facet's Model File"));
			i = ttygetparam(returninfstr(infstr), &us_colorreadp, 1, subparams);
			el_pleasestop = oldplease;
			if (i != 0 && subparams[0][0] != 0)
			{
				DiaUnDimItem(dia, DVEO_MODELFILE);
				DiaSetText(dia, DVEO_MODELFILE, subparams[0]);
				if (np->temp1 != 0) efree((char *)np->temp1);
				(void)allocstring((char **)&np->temp1, subparams[0], el_tempcluster);
				DiaSetControl(dia, DVEO_USEMODELCIRC, 0);
				DiaSetControl(dia, DVEO_USEMODELFILE, 1);
			}
			continue;
		}
		if (itemHit == DVEO_MODELFILE)
		{
			pt = DiaGetScrollLine(dia, DVEO_FACETLIST, DiaGetCurLine(dia, DVEO_FACETLIST));
			lib = el_curlib;   el_curlib = sim_curlib;
			np = getnodeproto(pt);
			el_curlib = lib;
			if (np == NONODEPROTO) continue;
			if (np->temp1 != 0) efree((char *)np->temp1);
			(void)allocstring((char **)&np->temp1, DiaGetText(dia, DVEO_MODELFILE), el_tempcluster);
			continue;
		}
	}

	if (itemHit != CANCEL)  /* now update the values */
	{
		i = 0;
		if (DiaGetControl(dia, DVEO_USEASSIGN) != 0) i |= VERILOGUSEASSIGN;
		if (DiaGetControl(dia, DVEO_USETRIREG) != 0) i |= VERILOGUSETRIREG;
		if (i != verilog_state)
			(void)setvalkey((INTBIG)sim_tool, VTOOL, sim_verilog_statekey, i, VINTEGER);

		for(lib = el_curlib; lib != NOLIBRARY; lib = lib->nextlibrary)
			for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		{
			var = getval((INTBIG)np, VNODEPROTO, VSTRING, "SIM_verilog_behave_file");
			if (var != NOVARIABLE)
			{
				if (np->temp1 == 0)
					(void)delval((INTBIG)np, VNODEPROTO, "SIM_verilog_behave_file"); else
				{
					if (strcmp((char *)np->temp1, (char *)var->addr) != 0)
						(void)setval((INTBIG)np, VNODEPROTO, "SIM_verilog_behave_file",
							(INTBIG)np->temp1, VSTRING);
				}
			} else if (np->temp1 != 0)
				(void)setval((INTBIG)np, VNODEPROTO, "SIM_verilog_behave_file",
					(INTBIG)np->temp1, VSTRING);
		}
	}
	for(lib = el_curlib; lib != NOLIBRARY; lib = lib->nextlibrary)
		for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
			if (np->temp1 != 0) efree((char *)np->temp1);
	efree((char *)liblist);
	DiaDoneDialog(dia);
}

/*
 * Helper routine to set information about the currently selected facet in the
 * "Verilog Options" dialog.
 */
void sim_verdiasetfacetinfo(LIBRARY *curlib, void *dia)
{
	REGISTER NODEPROTO *np;
	REGISTER LIBRARY *savelib;
	REGISTER char *pt;

	DiaSetControl(dia, DVEO_USEMODELCIRC, 1);
	DiaSetControl(dia, DVEO_USEMODELFILE, 0);
	DiaSetText(dia, DVEO_MODELFILE, "");
	DiaDimItem(dia, DVEO_MODELFILE);
	pt = DiaGetScrollLine(dia, DVEO_FACETLIST, DiaGetCurLine(dia, DVEO_FACETLIST));
	savelib = el_curlib;   el_curlib = curlib;
	np = getnodeproto(pt);
	el_curlib = savelib;
	if (np != NONODEPROTO && np->temp1 != 0)
	{
		DiaSetControl(dia, DVEO_USEMODELCIRC, 0);
		DiaSetControl(dia, DVEO_USEMODELFILE, 1);
		DiaUnDimItem(dia, DVEO_MODELFILE);
		DiaSetText(dia, DVEO_MODELFILE, (char *)np->temp1);
	}
}

/*
 * Helper routines for listing all facets in library "sim_curlib".
 */
BOOLEAN sim_topoffacets(char **c)
{
	sim_oldfacetprotos = sim_curlib->firstnodeproto;
	return(TRUE);
}
char *sim_nextfacets(void)
{
	REGISTER NODEPROTO *thisnp;
	REGISTER char *ret;

	ret = 0;
	if (sim_oldfacetprotos != NONODEPROTO)
	{
		thisnp = sim_oldfacetprotos;
		sim_oldfacetprotos = sim_oldfacetprotos->nextnodeproto;
		ret = nldescribenodeproto(thisnp);
	}
	return(ret);
}

#endif  /* SIMTOOL */
