package XML::Grammar::Screenplay::API::Concat;
$XML::Grammar::Screenplay::API::Concat::VERSION = '0.32.0';
use strict;
use warnings;
use autodie;
use 5.014;

use MooX 'late';

use Carp        ();
use XML::LibXML ();

my $SCREENPLAY_XML_NS =
"http://web-cpan.berlios.de/modules/XML-Grammar-Screenplay/screenplay-xml-0.2/";

sub _calc_xpc
{
    my $xpc = XML::LibXML::XPathContext->new();
    $xpc->registerNs( "sp", $SCREENPLAY_XML_NS );

    return $xpc;
}

sub concat
{
    my ( $self, $args ) = @_;
    my $inputs = $args->{inputs};

    my $parser  = XML::LibXML->new();
    my $new_xml = $parser->parse_string(
qq#<document xmlns="$SCREENPLAY_XML_NS"><head></head><body id="index"></body></document>#
    );
    my $root        = $new_xml->documentElement();
    my $xpc         = _calc_xpc();
    my ($root_body) = $xpc->findnodes( './sp:body', $root );

    my $id_differentiator_counters = +{};
    my $chapter_idx                = 0;
    foreach my $src (@$inputs)
    {
        my $this_chapter_idx = ( ++$chapter_idx );
        my $src_type         = $src->{type};
        if ( $src_type ne 'file' )
        {
            Carp::confess(qq#Unknown input type "$src_type"#);
        }
        my $src_fn = $src->{filename};
        my $input  = $parser->parse_file($src_fn);
        my $doc    = $input->documentElement();
        my @el     = $xpc->findnodes( "//sp:document/sp:body/sp:scene", $doc, );
        my $dest_xml;

        if ( not @el )
        {
            Carp::confess(q#no scenes found in "$src_fn"#);
        }
        elsif ( 1 == @el )
        {
            $dest_xml = $el[0];

            # @el = $xpc->findnodes("//sp:document/sp:body/sp:scene/sp:scene");
        }
        else
        {
            $dest_xml = $parser->parse_string(
qq#<scene xmlns="$SCREENPLAY_XML_NS" id="chapter_$this_chapter_idx" title="Chapter $this_chapter_idx"></scene>#
            );
            foreach my $el (@el)
            {
                $dest_xml->documentElement()
                    ->appendWellBalancedChunk( $el->toString() );
            }
        }

        foreach my $el ($dest_xml)
        {
            my @idels = $xpc->findnodes( "//sp:scene[\@id]", $el, );
            foreach my $id_el (@idels)
            {
                my $old_id = $id_el->getAttribute('id');
                if ( exists $id_differentiator_counters->{$old_id} )
                {
                    my $new_idx = $id_differentiator_counters->{$old_id}++;
                    my $new_id  = sprintf( "%s_%d", $old_id, $new_idx );
                    $id_el->setAttribute( 'id', $new_id );
                }
                else
                {
                    $id_differentiator_counters->{$old_id} = 1;
                }
            }
        }

        $root_body->appendWellBalancedChunk(

            # $dest_xml->documentElement()->toString() );
            $dest_xml->toString()
        );
    }

    my $output_str = $new_xml->toString();
    my $output_dom = $parser->parse_string( $output_str, );
    return +{ 'dom' => $output_dom, 'string' => $output_str, };
}

1;

=pod

=encoding UTF-8

=head1 NAME

XML::Grammar::Screenplay::API::Concat - concatenate several screenplay-xml files.

=head1 VERSION

version 0.32.0

=head1 SYNOPSIS

=head1 DESCRIPTION

=head1 METHODS

=head2 my $rec = $obj->concat({ inputs => [@inputs] })

=head1 THANKS

L<https://stackoverflow.com/questions/1074233/how-can-i-combine-two-xml-files-with-perl> and L<https://github.com/robinsmidsrod/xml-merge> .

=for :stopwords cpan testmatrix url bugtracker rt cpants kwalitee diff irc mailto metadata placeholders metacpan

=head1 SUPPORT

=head2 Websites

The following websites have more information about this module, and may be of help to you. As always,
in addition to those websites please use your favorite search engine to discover more resources.

=over 4

=item *

MetaCPAN

A modern, open-source CPAN search engine, useful to view POD in HTML format.

L<https://metacpan.org/release/XML-Grammar-Fiction>

=item *

RT: CPAN's Bug Tracker

The RT ( Request Tracker ) website is the default bug/issue tracking system for CPAN.

L<https://rt.cpan.org/Public/Dist/Display.html?Name=XML-Grammar-Fiction>

=item *

CPANTS

The CPANTS is a website that analyzes the Kwalitee ( code metrics ) of a distribution.

L<http://cpants.cpanauthors.org/dist/XML-Grammar-Fiction>

=item *

CPAN Testers

The CPAN Testers is a network of smoke testers who run automated tests on uploaded CPAN distributions.

L<http://www.cpantesters.org/distro/X/XML-Grammar-Fiction>

=item *

CPAN Testers Matrix

The CPAN Testers Matrix is a website that provides a visual overview of the test results for a distribution on various Perls/platforms.

L<http://matrix.cpantesters.org/?dist=XML-Grammar-Fiction>

=item *

CPAN Testers Dependencies

The CPAN Testers Dependencies is a website that shows a chart of the test results of all dependencies for a distribution.

L<http://deps.cpantesters.org/?module=XML::Grammar::Fiction>

=back

=head2 Bugs / Feature Requests

Please report any bugs or feature requests by email to C<bug-xml-grammar-fiction at rt.cpan.org>, or through
the web interface at L<https://rt.cpan.org/Public/Bug/Report.html?Queue=XML-Grammar-Fiction>. You will be automatically notified of any
progress on the request by the system.

=head2 Source Code

The code is open to the world, and available for you to hack on. Please feel free to browse it and play
with it, or whatever. If you want to contribute patches, please send me a diff or prod me to pull
from your repository :)

L<https://github.com/shlomif/fiction-xml>

  git clone git://github.com/shlomif/fiction-xml.git

=head1 AUTHOR

Shlomi Fish <shlomif@cpan.org>

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website
L<https://github.com/shlomif/fiction-xml/issues>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2007 by Shlomi Fish.

This is free software, licensed under:

  The MIT (X11) License

=cut

__END__


__END__

