package Apache::Session::Browseable::Store::Cassandra;

use strict;

use Apache::Session::Store::DBI;
use Apache::Session::Browseable::Store::DBI;
use Apache::Session::Browseable::Store::Cassandra;

our @ISA     = qw(Apache::Session::Browseable::Store::DBI);
our $VERSION = '1.3.12';

our $DataSource = undef;
our $UserName   = undef;
our $Password   = undef;

sub connection {
    my $self    = shift;
    my $session = shift;

    return if ( defined $self->{dbh} );

    if ( exists $session->{args}->{Handle} ) {
        $self->{dbh} = $session->{args}->{Handle};
        return;
    }

    my $datasource = $session->{args}->{DataSource}
      || $DataSource;
    my $username = $session->{args}->{UserName}
      || $UserName;
    my $password = $session->{args}->{Password}
      || $Password;

    $self->{dbh} =
      DBI->connect( $datasource, $username, $password, { RaiseError => 1 } )
      || die $DBI::errstr;

    #If we open the connection, we close the connection
    $self->{disconnect} = 1;
}

sub materialize {
    my $self    = shift;
    my $session = shift;

    $self->connection($session);

    local $self->{dbh}->{RaiseError} = 1;
    local $self->{dbh}->{LongReadLen} =
      $session->{args}->{LongReadLen} || 8 * 2**10;

    if ( !defined $self->{materialize_sth} ) {
        $self->{materialize_sth} = $self->{dbh}->prepare_cached(
            qq{
                SELECT a_session FROM sessions WHERE id = ? FOR UPDATE}
        );
    }

    $self->{materialize_sth}->bind_param( 1, $session->{data}->{_session_id} );
    $self->{materialize_sth}->execute;

    my $results = $self->{materialize_sth}->fetchrow_arrayref;

    if ( !( defined $results ) ) {
        die "Object does not exist in the data store";
    }

    $self->{materialize_sth}->finish;

    $session->{serialized} = $results->[0];
}

sub DESTROY {
    my $self = shift;

    if ( $self->{disconnect} ) {
        $self->{dbh}->disconnect;
    }
}

1;

=pod

=head1 NAME

Apache::Session::Browseable::Store::Cassandra - Store persistent data in a Cassandra database

=head1 SYNOPSIS

 use Apache::Session::Browseable::Store::Cassandra;

 my $store = new Apache::Session::Browseable::Store::Cassandra;

 $store->insert($ref);
 $store->update($ref);
 $store->materialize($ref);
 $store->remove($ref);

=head1 DESCRIPTION

Apache::Session::Browseable::Store::Cassandra fulfills the storage interface of
Apache::Session. Session data is stored in a Cassandra database.

=head1 SCHEMA

To use this module, you will need at least these columns in a table
called 'sessions':

 id char(32)     # or however long your session IDs are.
 a_session lvarchar

To create this schema, you can execute this command using the sqlplus program:

 CREATE TABLE sessions (
    id char(32) not null primary key,
    a_session lvarchar
 );

If you use some other command, ensure that there is a unique index on the
table's id column.

=head1 CONFIGURATION

The module must know what datasource, username, and password to use when
connecting to the database.  These values can be set using the options hash
(see Apache::Session documentation).  The options are DataSource, UserName,
and Password.

Example:

 tie %hash, 'Apache::Session::Cassandra', $id, {
     DataSource => 'dbi:Cassandra:database',
     UserName   => 'database_user',
     Password   => 'K00l'
 };

Instead, you may pass in an already-opened DBI handle to your database.

 tie %hash, 'Apache::Session::Cassandra', $id, {
     Handle => $dbh
 };

The last option is LongReadLen, which specifies the maximum size of the session
object.  If not supplied, the default maximum size is 8 KB.

=head1 AUTHOR

This module was written by Mike Langen <mike.langen@tamedia.ch>, based
on the original for Oracle.

=head1 SEE ALSO

L<Apache::Session>, L<Apache::Session::Store::DBI>
1;

