#!perl -w

# before running this script make sure you have 'tclsh' in your path, 
# and this 'tcl' distribution is required one.
# FreeBSD users may want to modify name of tcl interpreter with --tclsh=tclsh86
# as long as 'tclsh' does not work in their case

use strict;
use Getopt::Long;
use ExtUtils::MakeMaker;
use Config;

my $usestubs = $^O eq 'MSWin32'? 0 : 1; # we prefer usestubs, but on windows
# default is to not use them, because stubs lib that come with AS TCL is impossible
# to link with GCC which comes with strawberry perl; please create a ticket for
# this; VKON 27-06-2018

my $arch = "$^O";
my $stub = "tclstub8.4";

# These need updating as more platforms are added to tcl-core/ area
if ($^O eq "MSWin32") {
    $stub = "tclstub84";
    $arch = "win32-x86" if ($Config{archname} =~ /-x86-/);
    $arch = "win32-x64" if ($Config{archname} =~ /-x64-/);
} elsif ($^O eq "darwin") {
    $arch = "darwin-universal";
} elsif ($^O eq "solaris") {
    $arch = "$^O-x86" if ($Config{archname} =~ /86/);
    $arch = "$^O-sparc" if ($Config{archname} =~ /sun4/);
} elsif ($^O eq "aix") {
    # $arch = "$^O";
} elsif ($^O eq "hpux") {
    $arch = "$^O-ia64" if ($Config{archname} =~ /ia64/i);
    $arch = "$^O-parisc" if ($Config{archname} =~ /pa-risc/i);
} elsif ($^O eq "linux" or $^O eq "cygwin") {
    $arch = "$^O-i686" if ($Config{archname} =~ /i\d86/);
    $arch = "$^O-ia64" if ($Config{archname} =~ /ia64/i);
    $arch = "$^O-x86_64" if ($Config{archname} =~ /x86_64/);
}

sub _die ($) {
    # CPAN smokers report FAIL if Makefile.PL dies, it should exit with status 0
    my $err = shift;
    warn $err;
    exit 0;
}

GetOptions(
   "tclsh=s",                  # Use this tclsh executable as a base to find 
       \(my $tclsh='tclsh'),   # the lib info needed
   "tclconfig=s",              # Use the specified Tcl config file instead 
       \my $tclconfig,         # of basing the values on the tclsh exe found
   "buildspec",                # Used with --tclconfig, use the build 
       \my $buildspec,         # (instead of install) values for determining lib info
   "usestubs!", \$usestubs,    # we want to use the Tcl stubs mechanism by default
   "library=s", \my $libpath,  # Use this specific Tcl library
   "include=s", \my $incpath,  # Use this specific include path
   "define=s", \(my $defs=''), # Use this specific set of defines
   "help!",    \my $help,      # --help
) || usage();
usage() if $help;
sub usage {
    _die <<EOT;
Usage: perl Makefile.PL [--tclsh <path>] [--tclconfig <path>]
                        [--buildspec] [--nousestubs] [<makemaker opts>...]

or for expert compilation:

       perl --library=-l/path/to/tcl(stub).a --include=-I/path/to/tcl/include
            --define="-DLIB_RUNTIME_DIR=... -DTCL_LIB_FILE=..."
EOT
}

my $buildtype = "";
my @extraargs;
if ($usestubs) {
    $defs .= " -DUSE_TCL_STUBS";
    $buildtype = "stub";
}

# If using stubs, we will set the LIB_RUNTIME_DIR and TCL_LIB_FILE
# to point to the install location as the default dll to load.

if (defined($libpath) && defined($incpath)) {
    # do nothing - set on command line
} else {
    # otherwise we *require* working tclsh; to avoid bogus FAIL reports
    # even on $^O eq 'darwin'
    # When user has its own tclConfig.sh with --tclconfig=..., then we
    # overwrite values in %tclcfg
    # but still we use workable tclsh in order to avoid bogus FAIL;

    open(TCLSH, "$tclsh tclcfg.tcl |") or _die "error starting $tclsh: $!\n";
    my $tclcfg = join '', <TCLSH>;
    close(TCLSH);
    print $tclcfg;
    my %tclcfg = $tclcfg =~ /^([^=]+)=(.*?)\n/gm;

    if ($^O eq 'darwin' && !defined($tclconfig)) {
        $tclconfig = $tclcfg{'tclConfig.sh'};
    }
    if ($^O eq 'MSWin32' && !defined($tclconfig) && !$usestubs) {
        $tclconfig = $tclcfg{'tclConfig.sh'};
    }

    # This is to allow propagation of this value to sub-Makefile.PLs
    $ENV{'TCLSH_PROG'} = $tclsh; # <-- candidate for deletion; WARN!

    if (!defined($tclconfig) && defined($arch) && $usestubs) {
	$incpath = "-Itcl-core/include";
	$libpath = "-Ltcl-core/$arch -l$stub";
	if ($^O eq 'darwin') {
	    # OS X also requires the Carbon framework by default
	    $libpath .= " -framework Carbon";
	}
    } elsif ($tclconfig) {
	_die "Tcl config file '$tclconfig' not found\n" unless (-f $tclconfig);

	my $variant    = ($usestubs ? "_STUB" : "");
	$variant       = "_BUILD$variant" if $buildspec;
	my $libspecvar = "TCL${variant}_LIB_SPEC";

	# Retrieve all info based on tclConfig.sh
	process_tclconfig($tclconfig, \%tclcfg);

	$libpath = $tclcfg{$libspecvar};
	$incpath = $tclcfg{TCL_INCLUDE_SPEC};
	if ($usestubs) {
	    if ($^O eq 'MSWin32') {
		$defs .= " -DLIB_RUNTIME_DIR=\\\"$tclcfg{'TCL_EXEC_PREFIX'}\\\"";
		$defs .= " -DTCL_LIB_FILE=\\\"$tclcfg{'TCL_DLL_FILE'}\\\""; 
	    } elsif ($^O eq 'darwin' && $tclcfg{'TCL_STUB_LIB_PATH'} =~ /\.framework/ ) {
		(my $fmk = $tclcfg{'TCL_STUB_LIB_PATH'}) =~ s/(?<=\.framework).*//;
		$defs .= " -DLIB_RUNTIME_DIR=\\\"$fmk\\\"";
		$defs .= " -DTCL_LIB_FILE=\\\"$tclcfg{'TCL_LIB_FILE'}\\\"";
		@extraargs = (dynamic_lib => {OTHERLDFLAGS => "-framework Carbon"});
	    } else {
		$defs .= " -DLIB_RUNTIME_DIR=\\\"$tclcfg{'TCL_EXEC_PREFIX'}/lib\\\"";
		$defs .= " -DTCL_LIB_FILE=\\\"$tclcfg{'TCL_LIB_FILE'}\\\""; 
	    }
	}
    } else {
	# no --tclconfig=... -> get values from ./tclcfg.tcl

	my $tclver = $tclcfg{tcl_version};

	if ($tclcfg{tcl_library} =~ /^(.*)[\\\/]lib[\\\/]/) {
	    $libpath = "-L$1/lib";
	    $incpath = "-I$1/include";
	    $defs .= " -DLIB_RUNTIME_DIR=\\\"$1/lib\\\"" if $usestubs;
	}

	if ($^O eq 'MSWin32') {
	    $tclver=~s/\.//;
	    $defs .= " -DTCL_LIB_FILE=\\\"tcl$tclver.$Config{so}\\\"" if $usestubs; 
	}
	elsif ($^O eq 'freebsd' or $^O eq 'netbsd') {
	    $tclver=~s/\.//;
	    $tclsh=~/([\d.]+)$/ and $incpath .= " -I/usr/local/include/tcl$1";
	    $defs .= " -DTCL_LIB_FILE=\\\"libtcl$tclver.$Config{so}\\\"" if $usestubs; 
	}
	elsif ($^O eq 'hpux') {
	    $defs .= " -DTCL_LIB_FILE=\\\"libtcl$tclver.$Config{so}\\\"" if $usestubs;
	}
	else {
	    $defs .= " -DTCL_LIB_FILE=\\\"libtcl$tclver.$Config{so}\\\"" if $usestubs; 
	}
	$libpath .= " -ltcl$buildtype$tclver";
    }
    # version must be 8.4+
    _die "Tcl requires Tcl v8.4 or greater, found '$tclcfg{tcl_version}'\n"
	if (defined $tclcfg{tcl_version} && $tclcfg{tcl_version} <8.4);
}

print "LIBS   = $libpath\n";
print "INC    = $incpath\n";
print "DEFINE = $defs\n";
print "tclConfig.sh = $tclconfig\n";

if ($^O eq 'darwin') {
    # darwin has a broken ranlib that requires you to run it anytime
    # you copy an archive file, so ensure ours it up-to-date
    system("ranlib tcl-core/$arch/libtclstub8.4.a");
    system("git update-index --assume-unchanged tcl-core/$arch/libtclstub8.4.a")
	if -d ".git";
    if ($libpath =~ /-framework/) {
	# Frameworks require slightly different compile options
	@extraargs = (dynamic_lib => {OTHERLDFLAGS => $libpath});
	$libpath = "";
    }
}

#print <<"#EOS";
WriteMakefile(
    NAME => "Tcl",
    VERSION_FROM => 'Tcl.pm',
    LICENSE => 'perl',
    MIN_PERL_VERSION => '5.006',
    ABSTRACT_FROM => 'Tcl.pm',
    META_MERGE => {
        resources => {
            repository => 'http://github.com/gisle/tcl.pm',
            MailingList => 'mailto:tcltk@perl.org',
        }
    },
    LIBS => ["$libpath"],
    INC => "$incpath",
    DEFINE => $defs,
    @extraargs,
);
#EOS

sub process_tclconfig {
    # Process a tclConfig.sh file for build info
    my $tclconfig = shift;
    my $hashref   = shift;

    open my $fh, $tclconfig or _die "error opening file '$tclconfig': $!\n";
    print "Using config data in $tclconfig\n";
    %$hashref = (join '', <$fh>) =~ /^(\w+)=['"]?(.*?)["']?$/gm;

    for my $k (keys %$hashref) {
	# Handle sh subs like ${TCL_DBGX}
	$hashref->{$k} =~ s/\$\{(\w+)\}/(exists $hashref->{$1} ? $hashref->{$1} : $&)/eg;
	# Handle any cygdrive-style paths
	$hashref->{$k} =~ s{/cygdrive/(\w)/}{$1:/}ig;
    }
    $hashref->{tcl_version} = $hashref->{TCL_VERSION};
}

sub MY::libscan {
    my($self, $path) =@_;
    return '' if $path =~ /\.pl$/i;
    return $path;
}

BEGIN {
    # compatibility with older versions of MakeMaker
    my $developer = -f ".git";
    my %mm_req = (
        LICENCE => 6.31,
        META_MERGE => 6.45,
        META_ADD => 6.45,
        MIN_PERL_VERSION => 6.48,
    );
    undef(*WriteMakefile);
    *WriteMakefile = sub {
        my %arg = @_;
        for (keys %mm_req) {
            unless (eval { ExtUtils::MakeMaker->VERSION($mm_req{$_}) }) {
                warn "$_ $@" if $developer;
                delete $arg{$_};
            }
        }
        ExtUtils::MakeMaker::WriteMakefile(%arg);
    };
}

