package Crypt::Passphrase::Bcrypt::Compat;
$Crypt::Passphrase::Bcrypt::Compat::VERSION = '0.004';
use strict;
use warnings;

use parent 'Crypt::Passphrase::Validator';

use Crypt::Eksblowfish::Bcrypt qw/bcrypt/;

sub new {
	my ($class) = @_;
	return bless {}, $class;
}

sub accepts_hash {
	my ($self, $hash) = @_;
	return $hash =~ /^\$2a?\$/;
}

sub verify_password {
	my ($class, $password, $hash) = @_;
	$class->secure_compare(bcrypt($password, $hash), $hash);
}

1;

#ABSTRACT: A bcrypt validator for Crypt::Passphrase

__END__

=pod

=encoding UTF-8

=head1 NAME

Crypt::Passphrase::Bcrypt::Compat - A bcrypt validator for Crypt::Passphrase

=head1 VERSION

version 0.004

=head1 SYNOPSIS

 my $passphrase = Crypt::Passphrase->new(
   encoder => {
     module => 'Bcrypt',
     cost   => 12,
   },
   validator => [ 'Bcrypt::Compat' ],
 );

=head1 DESCRIPTION

This module exists to support checking C<$2$> type bcrypt hashes (as opposed to more modern variants like C<$2b$> or C<$2a$>) as sometimes generated by L<Crypt::Eksblowfish::Bcrypt|Crypt::Eksblowfish::Bcrypt>. This ancient variant is not supported by by modern bcrypt implementations, hence the need for this compatability implementation. It should not be necessary for any other purpose.

=head1 AUTHOR

Leon Timmermans <fawaka@gmail.com>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2024 by Leon Timmermans.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
