package Sub::Meta;
use 5.010;
use strict;
use warnings;

our $VERSION = "0.10";

use Carp ();
use Scalar::Util ();
use Sub::Identify ();
use Sub::Util ();
use attributes ();

use Sub::Meta::Parameters;
use Sub::Meta::Returns;

BEGIN {
    # for Pure Perl
    $ENV{PERL_SUB_IDENTIFY_PP} = $ENV{PERL_SUB_META_PP}; ## no critic (RequireLocalizedPunctuationVars)
}

use overload
    fallback => 1,
    eq => \&is_same_interface
    ;

sub parameters_class { return 'Sub::Meta::Parameters' }
sub returns_class    { return 'Sub::Meta::Returns' }

sub _croak { require Carp; goto &Carp::croak }

sub new {
    my ($class, @args) = @_;
    my %args = @args == 1 ? %{$args[0]} : @args;

    my $self = bless \%args => $class;

    $self->set_sub(delete $args{sub})             if exists $args{sub}; # build subinfo
    $self->set_subname(delete $args{subname})     if exists $args{subname};
    $self->set_stashname(delete $args{stashname}) if exists $args{stashname};
    $self->set_fullname(delete $args{fullname})   if exists $args{fullname};

    if (my $is_method = $self->_normalize_args_is_method(\%args)) {
        $self->set_is_method($is_method);
    }

    if (my $parameters = $self->_normalize_args_parameters(\%args)) {
        $self->set_parameters($parameters);
    }

    if (exists $args{returns}) {
        $self->set_returns($args{returns})
    }

    # cleaning
    delete $args{args};
    delete $args{slurpy};
    delete $args{invocant};
    delete $args{nshift};

    return $self;
}

sub _normalize_args_is_method {
    my ($self, $args) = @_;

    if (exists $args->{parameters}) {
        my $is_method = $args->{is_method}
                     || $args->{parameters}{nshift}
                     || $args->{parameters}{invocant};

        my $exists_is_method = exists $args->{is_method}
                            || exists $args->{parameters}{nshift}
                            || exists $args->{parameters}{invocant};

        return $is_method if $exists_is_method
    }
    elsif(exists $args->{args}) {
        my $is_method = $args->{is_method}
                     || $args->{nshift}
                     || $args->{invocant};

        my $exists_is_method = exists $args->{is_method}
                            || exists $args->{nshift}
                            || exists $args->{invocant};

        return $is_method if $exists_is_method;
    }
    return;
}

sub _normalize_args_parameters {
    my ($self, $args) = @_;

    if (exists $args->{parameters}) {
        return $args->{parameters};
    }
    elsif(exists $args->{args}) {
        my $nshift = exists $args->{nshift}    ? $args->{nshift}
                   : $self->is_method          ? 1
                   : exists $self->{is_method} ? 0
                   : undef;

        my $parameters = { args => $args->{args} };
        $parameters->{slurpy}   = $args->{slurpy}   if exists $args->{slurpy};
        $parameters->{invocant} = $args->{invocant} if exists $args->{invocant};
        $parameters->{nshift}   = $nshift           if defined $nshift;
        return $parameters;
    }
    return;
}

sub sub() :method { my $self = shift; return $self->{sub} } ## no critic (ProhibitBuiltinHomonyms)
sub subname()     { my $self = shift; return $self->subinfo->[1] }
sub stashname()   { my $self = shift; return $self->subinfo->[0] }
sub fullname()    { my $self = shift; return @{$self->subinfo} ? sprintf('%s::%s', $self->stashname || '', $self->subname || '') : undef }

sub subinfo()     {
    my $self = shift;
    return $self->{subinfo} if $self->{subinfo};
    $self->{subinfo} = $self->_build_subinfo;
    return $self->{subinfo};
}

sub file()        { my $self = shift; return $self->{file}        ||= $self->_build_file }
sub line()        { my $self = shift; return $self->{line}        ||= $self->_build_line }
sub is_constant() { my $self = shift; return $self->{is_constant} ||= $self->_build_is_constant }
sub prototype() :method { my $self = shift; return $self->{prototype}   ||= $self->_build_prototype } ## no critic (ProhibitBuiltinHomonyms)
sub attribute()   { my $self = shift; return $self->{attribute}   ||= $self->_build_attribute }
sub is_method()   { my $self = shift; return !!$self->{is_method} }
sub parameters()  { my $self = shift; return $self->{parameters} }
sub returns()     { my $self = shift; return $self->{returns} }
sub args()        { my $self = shift; return $self->parameters->args }
sub all_args()    { my $self = shift; return $self->parameters->all_args }
sub slurpy()      { my $self = shift; return $self->parameters->slurpy }
sub nshift()      { my $self = shift; return $self->parameters->nshift }
sub invocant()    { my $self = shift; return $self->parameters->invocant }
sub invocants()   { my $self = shift; return $self->parameters->invocants }

sub set_sub {
    my ($self, $v) = @_;
    $self->{sub} = $v;

    # rebuild subinfo
    delete $self->{subinfo};
    $self->subinfo;
    return $self;
}

sub set_subname   { my ($self, $v) = @_; $self->{subinfo}[1]  = $v; return $self }
sub set_stashname { my ($self, $v) = @_; $self->{subinfo}[0]  = $v; return $self }
sub set_fullname  {
    my ($self, $v) = @_;
    $self->{subinfo} = $v =~ m!^(.+)::([^:]+)$! ? [$1, $2] : [];
    return $self;
}
sub set_subinfo {
    my ($self, @args) = @_;
    $self->{subinfo} = @args > 1 ? [ $args[0], $args[1] ] : $args[0];
    return $self;
}

sub set_file        { my ($self, $v) = @_; $self->{file}        = $v; return $self }
sub set_line        { my ($self, $v) = @_; $self->{line}        = $v; return $self }
sub set_is_constant { my ($self, $v) = @_; $self->{is_constant} = $v; return $self }
sub set_prototype   { my ($self, $v) = @_; $self->{prototype}   = $v; return $self }
sub set_attribute   { my ($self, $v) = @_; $self->{attribute}   = $v; return $self }
sub set_is_method   { my ($self, $v) = @_; $self->{is_method}   = $v; return $self }

sub set_parameters {
    my ($self, @args) = @_;
    my $v = $args[0];
    if (Scalar::Util::blessed($v)) {
        if ($v->isa('Sub::Meta::Parameters')) {
            $self->{parameters} = $v
        }
        else {
            _croak('object must be Sub::Meta::Parameters');
        }
    }
    else {
        $self->{parameters} = $self->parameters_class->new(@args);
    }
    return $self
}

sub set_args {
    my ($self, @args) = @_;
    if ($self->parameters) {
        $self->parameters->set_args(@args);
    }
    else {
        $self->set_parameters($self->parameters_class->new(args => @args));
    }
    return $self;
}

sub set_slurpy {
    my ($self, @args) = @_;
    $self->parameters->set_slurpy(@args);
    return $self;
}

sub set_nshift {
    my ($self, $v) = @_;
    if ($self->is_method && $v == 0) {
        _croak 'nshift of method cannot be zero';
    }
    $self->parameters->set_nshift($v);
    return $self;
}

sub set_invocant {
    my ($self, $v) = @_;
    $self->parameters->set_invocant($v);
    return $self;
}

sub set_returns {
    my ($self, @args) = @_;
    my $v = $args[0];
    if (Scalar::Util::blessed($v) && $v->isa('Sub::Meta::Returns')) {
        $self->{returns} = $v
    }
    else {
        $self->{returns} = $self->returns_class->new(@args);
    }
    return $self
}

sub _build_subinfo     { my $self = shift; return $self->sub ? [ Sub::Identify::get_code_info($self->sub) ] : [] }
sub _build_file        { my $self = shift; return $self->sub ? (Sub::Identify::get_code_location($self->sub))[0] : '' }
sub _build_line        { my $self = shift; return $self->sub ? (Sub::Identify::get_code_location($self->sub))[1] : undef }
sub _build_is_constant { my $self = shift; return $self->sub ? Sub::Identify::is_sub_constant($self->sub) : undef }
sub _build_prototype   { my $self = shift; return $self->sub ? Sub::Util::prototype($self->sub) : '' }
sub _build_attribute   { my $self = shift; return $self->sub ? [ attributes::get($self->sub) ] : undef }

sub apply_subname {
    my ($self, $subname) = @_;
    _croak 'apply_subname requires subroutine reference' unless $self->sub;
    $self->set_subname($subname);
    Sub::Util::set_subname($self->fullname, $self->sub);
    return $self;
}

sub apply_prototype {
    my ($self, $prototype) = @_;
    _croak 'apply_prototype requires subroutine reference' unless $self->sub;
    Sub::Util::set_prototype($prototype, $self->sub);
    $self->set_prototype($prototype);
    return $self;
}

sub apply_attribute {
    my ($self, @attribute) = @_;
    _croak 'apply_attribute requires subroutine reference' unless $self->sub;
    {
        no warnings qw(misc); ## no critic (ProhibitNoWarnings)
        attributes->import($self->stashname, $self->sub, @attribute);
    }
    $self->set_attribute($self->_build_attribute);
    return $self;
}

sub apply_meta {
    my ($self, $other) = @_;

    $self->apply_subname($other->subname);
    $self->apply_prototype($other->prototype);
    $self->apply_attribute(@{$other->attribute});

    return $self;
}

sub is_same_interface {
    my ($self, $other) = @_;

    return unless Scalar::Util::blessed($other) && $other->isa('Sub::Meta');

    return unless defined $self->subname ? defined $other->subname && $self->subname eq $other->subname
                                         : !defined $other->subname;

    return unless $self->is_method ? $other->is_method
                                   : !$other->is_method;

    return unless $self->parameters ? $self->parameters->is_same_interface($other->parameters)
                                    : !$other->parameters;

    return unless $self->returns ? $self->returns->is_same_interface($other->returns)
                                 : !$other->returns;

    return !!1;
}

sub is_same_interface_inlined {
    my ($self, $v) = @_;

    my @src;

    push @src => sprintf("Scalar::Util::blessed(%s) && %s->isa('Sub::Meta')", $v, $v);

    push @src => defined $self->subname ? sprintf("defined %s->subname && '%s' eq %s->subname", $v, "@{[$self->subname]}", $v)
                                        : sprintf('!defined %s->subname', $v);

    push @src => $self->is_method ? sprintf('%s->is_method', $v)
                                  : sprintf('!%s->is_method', $v);

    push @src => $self->parameters ? $self->parameters->is_same_interface_inlined(sprintf('%s->parameters', $v))
                                   : sprintf('!%s->parameters', $v);

    push @src => $self->returns ? $self->returns->is_same_interface_inlined(sprintf('%s->returns', $v))
                                : sprintf('!%s->returns', $v);

    return join "\n && ", @src;
}

sub display {
    my $self = shift;

    my $keyword = $self->is_method ? 'method' : 'sub';
    my $subname = $self->subname // '';

    my $s = $keyword;
    $s .= ' ' . $subname if $subname;
    $s .= '('. $self->parameters->display .')' if $self->parameters;
    $s .= ' => ' . $self->returns->display if $self->returns;
    return $s;
}

1;
__END__

=encoding utf-8

=head1 NAME

Sub::Meta - handle subroutine meta information

=head1 SYNOPSIS

    use Sub::Meta;

    sub hello($) :method { }
    my $meta = Sub::Meta->new(sub => \&hello);
    $meta->subname; # => hello

    $meta->sub;        # \&hello
    $meta->subname;    # hello
    $meta->fullname    # main::hello
    $meta->stashname   # main
    $meta->file        # path/to/file.pl
    $meta->line        # 5
    $meta->is_constant # !!0
    $meta->prototype   # $
    $meta->attribute   # ['method']
    $meta->is_method   # undef
    $meta->parameters  # undef
    $meta->returns     # undef
    $meta->display     # 'sub hello'

    # setter
    $meta->set_subname('world');
    $meta->subname; # world
    $meta->fullname; # main::world

    # apply to sub
    $meta->apply_prototype('$@');
    $meta->prototype; # $@
    Sub::Util::prototype($meta->sub); # $@

And you can hold meta information of parameter type and return type. See also L<Sub::Meta::Parameters> and L<Sub::Meta::Returns>.

    $meta->set_parameters(args => ['Str']));
    $meta->parameters->args; # [ Sub::Meta::Param->new({ type => 'Str' }) ]
    
    $meta->set_args(['Str']);
    $meta->args; # [ Sub::Meta::Param->new({ type => 'Str' }) ]

    $meta->set_returns('Str');
    $meta->returns->scalar; # 'Str'
    $meta->returns->list;   # 'Str'

And you can compare meta informations:

    my $other = Sub::Meta->new(subname => 'hello');
    $meta->is_same_interface($other); # 1
    $meta eq $other; # 1

=head1 DESCRIPTION

C<Sub::Meta> provides methods to handle subroutine meta information. In addition to information that can be obtained from subroutines using module L<B> etc., subroutines can have meta information such as arguments and return values.

=head1 METHODS

=head2 new

Constructor of C<Sub::Meta>.

    use Sub::Meta;
    use Types::Standard -types;

    # sub Greeting::hello(Str) -> Str
    Sub::Meta->new(
        fullname    => 'Greeting::hello',
        is_constant => 0,
        prototype   => '$',
        attribute   => ['method'],
        is_method   => 1,
        parameters  => { args => [{ type => Str }]},
        returns     => Str,
    );

Others are as follows:

    # sub add(Int, Int) -> Int
    Sub::Meta->new(
        subname => 'add',
        args    => [Int, Int],
        returns => Int,
    );

    # method hello(Str) -> Str 
    Sub::Meta->new(
        subname   => 'hello',
        args      => [{ message => Str }],
        is_method => 1,
        returns   => Str,
    );

    # sub twice(@numbers) -> ArrayRef[Int]
    Sub::Meta->new(
        subname   => 'twice',
        args      => [],
        slurpy    => { name => '@numbers' },
        returns   => ArrayRef[Int],
    );

    # Named parameters:
    # sub foo(Str :a) -> Str
    Sub::Meta->new(
        subname   => 'foo',
        args      => { a => Str },
        returns   => Str,
    );

    # is equivalent to
    Sub::Meta->new(
        subname   => 'foo',
        args      => [{ name => 'a', isa => Str, named => 1 }],
        returns   => Str,
    );

Another way to create a Sub::Meta is to use L<Sub::Meta::Creator>:

    use Sub::Meta::Creator;
    use Sub::Meta::Finder::FunctionParameters;

    my $creator = Sub::Meta::Creator->new(
        finders => [ \&Sub::Meta::Finder::FunctionParameters::find_materials ],
    );

    use Function::Parameters;
    use Types::Standard -types;

    method hello(Str $msg) { }
    my $meta = $creator->create(\&hello);
    # =>
    # Sub::Meta
    #   args [
    #       [0] Sub::Meta::Param->new(name => '$msg', type => Str)
    #   ],
    #   invocant   Sub::Meta::Param->(name => '$self', invocant => 1),
    #   nshift     1,
    #   slurpy     !!0

=head2 ACCESSORS

=head3 sub

A subroutine reference.

=head3 set_sub

Setter for subroutine reference.

    sub hello { ... }
    $meta->set_sub(\&hello);
    $meta->sub # => \&hello

=head3 subname

A subroutine name, e.g. C<hello>

=head3 set_subname($subname)

Setter for subroutine name.

    $meta->subname; # hello
    $meta->set_subname('world');
    $meta->subname; # world
    Sub::Util::subname($meta->sub); # hello (NOT apply to sub)

=head3 apply_subname($subname)

Sets subroutine name and apply to the subroutine reference.

    $meta->subname; # hello
    $meta->apply_subname('world');
    $meta->subname; # world
    Sub::Util::subname($meta->sub); # world

=head3 fullname

A subroutine full name, e.g. C<main::hello>

=head3 set_fullname($fullname)

Setter for subroutine full name.

=head3 stashname

A subroutine stash name, e.g. C<main>

=head3 set_stashname($stashname)

Setter for subroutine stash name.

=head3 subinfo

A subroutine information, e.g. C<['main', 'hello']>

=head3 set_subinfo([$stashname, $subname])

Setter for subroutine information.

=head3 file

A filename where subroutine is defined, e.g. C<path/to/main.pl>.

=head3 set_file($filepath)

Setter for C<file>.

=head3 line

A line where the definition of subroutine started, e.g. C<5>

=head3 set_line($line)

Setter for C<line>.

=head3 is_constant

A boolean value indicating whether the subroutine is a constant or not.

=head3 set_is_constant($bool)

Setter for C<is_constant>.

=head3 prototype

A prototype of subroutine reference, e.g. C<$@>

=head3 set_prototype($prototype)

Setter for C<prototype>.

=head3 apply_prototype($prototype)

Sets subroutine prototype and apply to the subroutine reference.

=head3 attribute

A attribute of subroutine reference, e.g. C<undef>, C<['method']>

=head3 set_attribute($attribute)

Setter for C<attribute>.

=head3 apply_attribute(@attribute)

Sets subroutine attributes and apply to the subroutine reference.

=head3 apply_meta($other_meta)

Apply subroutine subname, prototype and attributes of C<$other_meta>.

=head3 is_method

A boolean value indicating whether the subroutine is a method or not.

=head3 set_is_method($bool)

Setter for C<is_method>.

=head3 parameters

Parameters object of L<Sub::Meta::Parameters>.

=head3 set_parameters($parameters)

Sets the parameters object of L<Sub::Meta::Parameters>.

    my $meta = Sub::Meta->new;
    $meta->set_parameters(args => ['Str']);
    $meta->parameters; # => Sub::Meta::Parameters->new(args => ['Str']);

    # or
    $meta->set_parameters(Sub::Meta::Parameters->new(args => ['Str']));

    # alias
    $meta->set_args(['Str']);

=head3 args

The alias of C<parameters.args>.

=head3 set_args($args)

The alias of C<parameters.set_args>.

=head3 all_args

The alias of C<parameters.all_args>.

=head3 nshift

The alias of C<parameters.nshift>.

=head3 set_nshift($nshift)

The alias of C<parameters.set_nshift>.

=head3 invocant

The alias of C<parameters.invocant>.

=head3 invocants

The alias of C<parameters.invocants>.

=head3 set_invocant($invocant)

The alias of C<parameters.set_invocant>.

=head3 slurpy

The alias of C<parameters.slurpy>.

=head3 set_slurpy($slurpy)

The alias of C<parameters.set_slurpy>.

=head3 returns

Returns object of L<Sub::Meta::Returns>.

=head3 set_returns($returns)

Sets the returns object of L<Sub::Meta::Returns> or any object.

    my $meta = Sub::Meta->new;
    $meta->set_returns({ type => 'Type'});
    $meta->returns; # => Sub::Meta::Returns->new({type => 'Type'});

    # or
    $meta->set_returns(Sub::Meta::Returns->new(type => 'Foo'));
    $meta->set_returns(MyReturns->new)

=head2 METHODS

=head3 is_same_interface($other_meta)

A boolean value indicating whether the subroutine's interface is same or not.
Specifically, check whether C<subname>, C<is_method>, C<parameters> and C<returns> are equal.

=head3 is_same_interface_inlined($other_meta_inlined)

Returns inlined C<is_same_interface> string:

    use Sub::Meta;
    my $meta = Sub::Meta->new(subname => 'hello');
    my $inline = $meta->is_same_interface_inlined('$_[0]');
    # $inline looks like this:
    #    Scalar::Util::blessed($_[0]) && $_[0]->isa('Sub::Meta')
    #    && defined $_[0]->subname && 'hello' eq $_[0]->subname
    #    && !$_[0]->is_method
    #    && !$_[0]->parameters
    #    && !$_[0]->returns
    my $check = eval "sub { $inline }";
    $check->(Sub::Meta->new(subname => 'hello')); # => OK
    $check->(Sub::Meta->new(subname => 'world')); # => NG

=head3 display

Returns the display of Sub::Meta:

    use Sub::Meta;
    use Types::Standard qw(Str);
    my $meta = Sub::Meta->new(
        subname => 'hello',
        is_method => 1,
        args => [Str],
        returns => Str,
    );
    $meta->display;  # 'method hello(Str) => Str'

=head2 OTHERS

=head3 parameters_class

Returns class name of parameters. default: Sub::Meta::Parameters
Please override for customization.

=head3 returns_class

Returns class name of returns. default: Sub::Meta::Returns
Please override for customization.

=head1 NOTE

=head2 setter

You can set meta information of subroutine. C<set_xxx> sets C<xxx> and does not affect subroutine reference. On the other hands, C<apply_xxx> sets C<xxx> and apply C<xxx> to subroutine reference.

Setter methods of C<Sub::Meta> returns meta object. So you can chain setting:

    $meta->set_subname('foo')
         ->set_stashname('Some')

=head2 Pure-Perl version

By default C<Sub::Meta> tries to load an XS implementation for speed.
If that fails, or if the environment variable C<PERL_SUB_META_PP> is defined to a true value, it will fall back to a pure perl implementation.

=head1 SEE ALSO

L<Sub::Identify>, L<Sub::Util>, L<Sub::Info>, L<Function::Parameters::Info>, L<Function::Return::Info>

=head1 LICENSE

Copyright (C) kfly8.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHOR

kfly8 E<lt>kfly@cpan.orgE<gt>

=cut

