/**
 * Enhanced Seccomp x32 Syscall Table
 *
 * Copyright (c) 2013 Red Hat <pmoore@redhat.com>
 * Author: Paul Moore <paul@paul-moore.com>
 */

/*
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of version 2.1 of the GNU Lesser General Public License as
 * published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
 * for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, see <http://www.gnu.org/licenses>.
 */

#include <string.h>

#include <seccomp.h>

#include "arch.h"
#include "arch-x32.h"

/* NOTE: based on Linux 4.5-rc4 */
const struct arch_syscall_def x32_syscall_table[] = { \
	{ "_llseek", __PNR__llseek },
	{ "_newselect", __PNR__newselect },
	{ "_sysctl", __PNR__sysctl },
	{ "accept", (X32_SYSCALL_BIT + 43) },
	{ "accept4", (X32_SYSCALL_BIT + 288) },
	{ "access", (X32_SYSCALL_BIT + 21) },
	{ "acct", (X32_SYSCALL_BIT + 163) },
	{ "add_key", (X32_SYSCALL_BIT + 248) },
	{ "adjtimex", (X32_SYSCALL_BIT + 159) },
	{ "afs_syscall", (X32_SYSCALL_BIT + 183) },
	{ "alarm", (X32_SYSCALL_BIT + 37) },
	{ "arm_fadvise64_64", __PNR_arm_fadvise64_64 },
	{ "arm_sync_file_range", __PNR_arm_sync_file_range },
	{ "arch_prctl", (X32_SYSCALL_BIT + 158) },
	{ "bdflush", __PNR_bdflush },
	{ "bind", (X32_SYSCALL_BIT + 49) },
	{ "bpf", (X32_SYSCALL_BIT + 321) },
	{ "break", __PNR_break },
	{ "breakpoint", __PNR_breakpoint },
	{ "brk", (X32_SYSCALL_BIT + 12) },
	{ "cachectl", __PNR_cachectl },
	{ "cacheflush", __PNR_cacheflush },
	{ "capget", (X32_SYSCALL_BIT + 125) },
	{ "capset", (X32_SYSCALL_BIT + 126) },
	{ "chdir", (X32_SYSCALL_BIT + 80) },
	{ "chmod", (X32_SYSCALL_BIT + 90) },
	{ "chown", (X32_SYSCALL_BIT + 92) },
	{ "chown32", __PNR_chown32 },
	{ "chroot", (X32_SYSCALL_BIT + 161) },
	{ "clock_adjtime", (X32_SYSCALL_BIT + 305) },
	{ "clock_getres", (X32_SYSCALL_BIT + 229) },
	{ "clock_gettime", (X32_SYSCALL_BIT + 228) },
	{ "clock_nanosleep", (X32_SYSCALL_BIT + 230) },
	{ "clock_settime", (X32_SYSCALL_BIT + 227) },
	{ "clone", (X32_SYSCALL_BIT + 56) },
	{ "close", (X32_SYSCALL_BIT + 3) },
	{ "connect", (X32_SYSCALL_BIT + 42) },
	{ "copy_file_range", (X32_SYSCALL_BIT + 326) },
	{ "creat", (X32_SYSCALL_BIT + 85) },
	{ "create_module", __PNR_create_module },
	{ "delete_module", (X32_SYSCALL_BIT + 176) },
	{ "dup", (X32_SYSCALL_BIT + 32) },
	{ "dup2", (X32_SYSCALL_BIT + 33) },
	{ "dup3", (X32_SYSCALL_BIT + 292) },
	{ "epoll_create", (X32_SYSCALL_BIT + 213) },
	{ "epoll_create1", (X32_SYSCALL_BIT + 291) },
	{ "epoll_ctl", (X32_SYSCALL_BIT + 233) },
	{ "epoll_ctl_old", __PNR_epoll_ctl_old },
	{ "epoll_pwait", (X32_SYSCALL_BIT + 281) },
	{ "epoll_wait", (X32_SYSCALL_BIT + 232) },
	{ "epoll_wait_old", __PNR_epoll_wait_old },
	{ "eventfd", (X32_SYSCALL_BIT + 284) },
	{ "eventfd2", (X32_SYSCALL_BIT + 290) },
	{ "execve", (X32_SYSCALL_BIT + 520) },
	{ "execveat", (X32_SYSCALL_BIT + 545) },
	{ "exit", (X32_SYSCALL_BIT + 60) },
	{ "exit_group", (X32_SYSCALL_BIT + 231) },
	{ "faccessat", (X32_SYSCALL_BIT + 269) },
	{ "fadvise64", (X32_SYSCALL_BIT + 221) },
	{ "fadvise64_64", __PNR_fadvise64_64 },
	{ "fallocate", (X32_SYSCALL_BIT + 285) },
	{ "fanotify_init", (X32_SYSCALL_BIT + 300) },
	{ "fanotify_mark", (X32_SYSCALL_BIT + 301) },
	{ "fchdir", (X32_SYSCALL_BIT + 81) },
	{ "fchmod", (X32_SYSCALL_BIT + 91) },
	{ "fchmodat", (X32_SYSCALL_BIT + 268) },
	{ "fchown", (X32_SYSCALL_BIT + 93) },
	{ "fchown32", __PNR_fchown32 },
	{ "fchownat", (X32_SYSCALL_BIT + 260) },
	{ "fcntl", (X32_SYSCALL_BIT + 72) },
	{ "fcntl64", __PNR_fcntl64 },
	{ "fdatasync", (X32_SYSCALL_BIT + 75) },
	{ "fgetxattr", (X32_SYSCALL_BIT + 193) },
	{ "finit_module", (X32_SYSCALL_BIT + 313) },
	{ "flistxattr", (X32_SYSCALL_BIT + 196) },
	{ "flock", (X32_SYSCALL_BIT + 73) },
	{ "fork", (X32_SYSCALL_BIT + 57) },
	{ "fremovexattr", (X32_SYSCALL_BIT + 199) },
	{ "fsetxattr", (X32_SYSCALL_BIT + 190) },
	{ "fstat", (X32_SYSCALL_BIT + 5) },
	{ "fstat64", __PNR_fstat64 },
	{ "fstatat64", __PNR_fstatat64 },
	{ "fstatfs", (X32_SYSCALL_BIT + 138) },
	{ "fstatfs64", __PNR_fstatfs64 },
	{ "fsync", (X32_SYSCALL_BIT + 74) },
	{ "ftime", __PNR_ftime },
	{ "ftruncate", (X32_SYSCALL_BIT + 77) },
	{ "ftruncate64", __PNR_ftruncate64 },
	{ "futex", (X32_SYSCALL_BIT + 202) },
	{ "futimesat", (X32_SYSCALL_BIT + 261) },
	{ "get_kernel_syms", __PNR_get_kernel_syms },
	{ "get_mempolicy", (X32_SYSCALL_BIT + 239) },
	{ "get_robust_list", (X32_SYSCALL_BIT + 531) },
	{ "get_thread_area", __PNR_get_thread_area },
	{ "getcpu", (X32_SYSCALL_BIT + 309) },
	{ "getcwd", (X32_SYSCALL_BIT + 79) },
	{ "getdents", (X32_SYSCALL_BIT + 78) },
	{ "getdents64", (X32_SYSCALL_BIT + 217) },
	{ "getegid", (X32_SYSCALL_BIT + 108) },
	{ "getegid32", __PNR_getegid32 },
	{ "geteuid", (X32_SYSCALL_BIT + 107) },
	{ "geteuid32", __PNR_geteuid32 },
	{ "getgid", (X32_SYSCALL_BIT + 104) },
	{ "getgid32", __PNR_getgid32 },
	{ "getgroups", (X32_SYSCALL_BIT + 115) },
	{ "getgroups32", __PNR_getgroups32 },
	{ "getitimer", (X32_SYSCALL_BIT + 36) },
	{ "getpeername", (X32_SYSCALL_BIT + 52) },
	{ "getpgid", (X32_SYSCALL_BIT + 121) },
	{ "getpgrp", (X32_SYSCALL_BIT + 111) },
	{ "getpid", (X32_SYSCALL_BIT + 39) },
	{ "getpmsg", (X32_SYSCALL_BIT + 181) },
	{ "getppid", (X32_SYSCALL_BIT + 110) },
	{ "getpriority", (X32_SYSCALL_BIT + 140) },
	{ "getrandom", (X32_SYSCALL_BIT + 318) },
	{ "getresgid", (X32_SYSCALL_BIT + 120) },
	{ "getresgid32", __PNR_getresgid32 },
	{ "getresuid", (X32_SYSCALL_BIT + 118) },
	{ "getresuid32", __PNR_getresuid32 },
	{ "getrlimit", (X32_SYSCALL_BIT + 97) },
	{ "getrusage", (X32_SYSCALL_BIT + 98) },
	{ "getsid", (X32_SYSCALL_BIT + 124) },
	{ "getsockname", (X32_SYSCALL_BIT + 51) },
	{ "getsockopt", (X32_SYSCALL_BIT + 542) },
	{ "gettid", (X32_SYSCALL_BIT + 186) },
	{ "gettimeofday", (X32_SYSCALL_BIT + 96) },
	{ "getuid", (X32_SYSCALL_BIT + 102) },
	{ "getuid32", __PNR_getuid32 },
	{ "getxattr", (X32_SYSCALL_BIT + 191) },
	{ "gtty", __PNR_gtty },
	{ "idle", __PNR_idle },
	{ "init_module", (X32_SYSCALL_BIT + 175) },
	{ "inotify_add_watch", (X32_SYSCALL_BIT + 254) },
	{ "inotify_init", (X32_SYSCALL_BIT + 253) },
	{ "inotify_init1", (X32_SYSCALL_BIT + 294) },
	{ "inotify_rm_watch", (X32_SYSCALL_BIT + 255) },
	{ "io_cancel", (X32_SYSCALL_BIT + 210) },
	{ "io_destroy", (X32_SYSCALL_BIT + 207) },
	{ "io_getevents", (X32_SYSCALL_BIT + 208) },
	{ "io_setup", (X32_SYSCALL_BIT + 543) },
	{ "io_submit", (X32_SYSCALL_BIT + 544) },
	{ "ioctl", (X32_SYSCALL_BIT + 514) },
	{ "ioperm", (X32_SYSCALL_BIT + 173) },
	{ "iopl", (X32_SYSCALL_BIT + 172) },
	{ "ioprio_get", (X32_SYSCALL_BIT + 252) },
	{ "ioprio_set", (X32_SYSCALL_BIT + 251) },
	{ "ipc", __PNR_ipc },
	{ "kcmp", (X32_SYSCALL_BIT + 312) },
	{ "kexec_file_load", (X32_SYSCALL_BIT + 320) },
	{ "kexec_load", (X32_SYSCALL_BIT + 528) },
	{ "keyctl", (X32_SYSCALL_BIT + 250) },
	{ "kill", (X32_SYSCALL_BIT + 62) },
	{ "lchown", (X32_SYSCALL_BIT + 94) },
	{ "lchown32", __PNR_lchown32 },
	{ "lgetxattr", (X32_SYSCALL_BIT + 192) },
	{ "link", (X32_SYSCALL_BIT + 86) },
	{ "linkat", (X32_SYSCALL_BIT + 265) },
	{ "listen", (X32_SYSCALL_BIT + 50) },
	{ "listxattr", (X32_SYSCALL_BIT + 194) },
	{ "llistxattr", (X32_SYSCALL_BIT + 195) },
	{ "lock", __PNR_lock },
	{ "lookup_dcookie", (X32_SYSCALL_BIT + 212) },
	{ "lremovexattr", (X32_SYSCALL_BIT + 198) },
	{ "lseek", (X32_SYSCALL_BIT + 8) },
	{ "lsetxattr", (X32_SYSCALL_BIT + 189) },
	{ "lstat", (X32_SYSCALL_BIT + 6) },
	{ "lstat64", __PNR_lstat64 },
	{ "madvise", (X32_SYSCALL_BIT + 28) },
	{ "mbind", (X32_SYSCALL_BIT + 237) },
	{ "membarrier",  (X32_SYSCALL_BIT + 324) },
	{ "memfd_create", (X32_SYSCALL_BIT + 319) },
	{ "migrate_pages", (X32_SYSCALL_BIT + 256) },
	{ "mincore", (X32_SYSCALL_BIT + 27) },
	{ "mkdir", (X32_SYSCALL_BIT + 83) },
	{ "mkdirat", (X32_SYSCALL_BIT + 258) },
	{ "mknod", (X32_SYSCALL_BIT + 133) },
	{ "mknodat", (X32_SYSCALL_BIT + 259) },
	{ "mlock", (X32_SYSCALL_BIT + 149) },
	{ "mlock2", (X32_SYSCALL_BIT + 325) },
	{ "mlockall", (X32_SYSCALL_BIT + 151) },
	{ "mmap", (X32_SYSCALL_BIT + 9) },
	{ "mmap2", __PNR_mmap2 },
	{ "modify_ldt", (X32_SYSCALL_BIT + 154) },
	{ "mount", (X32_SYSCALL_BIT + 165) },
	{ "move_pages", (X32_SYSCALL_BIT + 533) },
	{ "mprotect", (X32_SYSCALL_BIT + 10) },
	{ "mpx", __PNR_mpx },
	{ "mq_getsetattr", (X32_SYSCALL_BIT + 245) },
	{ "mq_notify", (X32_SYSCALL_BIT + 527) },
	{ "mq_open", (X32_SYSCALL_BIT + 240) },
	{ "mq_timedreceive", (X32_SYSCALL_BIT + 243) },
	{ "mq_timedsend", (X32_SYSCALL_BIT + 242) },
	{ "mq_unlink", (X32_SYSCALL_BIT + 241) },
	{ "mremap", (X32_SYSCALL_BIT + 25) },
	{ "msgctl", (X32_SYSCALL_BIT + 71) },
	{ "msgget", (X32_SYSCALL_BIT + 68) },
	{ "msgrcv", (X32_SYSCALL_BIT + 70) },
	{ "msgsnd", (X32_SYSCALL_BIT + 69) },
	{ "msync", (X32_SYSCALL_BIT + 26) },
	{ "multiplexer", __PNR_multiplexer },
	{ "munlock", (X32_SYSCALL_BIT + 150) },
	{ "munlockall", (X32_SYSCALL_BIT + 152) },
	{ "munmap", (X32_SYSCALL_BIT + 11) },
	{ "name_to_handle_at", (X32_SYSCALL_BIT + 303) },
	{ "nanosleep", (X32_SYSCALL_BIT + 35) },
	{ "newfstatat", (X32_SYSCALL_BIT + 262) },
	{ "nfsservctl", __PNR_nfsservctl },
	{ "nice", __PNR_nice },
	{ "oldfstat", __PNR_oldfstat },
	{ "oldlstat", __PNR_oldlstat },
	{ "oldolduname", __PNR_oldolduname },
	{ "oldstat", __PNR_oldstat },
	{ "olduname", __PNR_olduname },
	{ "oldwait4", __PNR_oldwait4 },
	{ "open", (X32_SYSCALL_BIT + 2) },
	{ "open_by_handle_at", (X32_SYSCALL_BIT + 304) },
	{ "openat", (X32_SYSCALL_BIT + 257) },
	{ "pause", (X32_SYSCALL_BIT + 34) },
	{ "pciconfig_iobase", __PNR_pciconfig_iobase },
	{ "pciconfig_read", __PNR_pciconfig_read },
	{ "pciconfig_write", __PNR_pciconfig_write },
	{ "perf_event_open", (X32_SYSCALL_BIT + 298) },
	{ "personality", (X32_SYSCALL_BIT + 135) },
	{ "pipe", (X32_SYSCALL_BIT + 22) },
	{ "pipe2", (X32_SYSCALL_BIT + 293) },
	{ "pivot_root", (X32_SYSCALL_BIT + 155) },
	{ "poll", (X32_SYSCALL_BIT + 7) },
	{ "ppoll", (X32_SYSCALL_BIT + 271) },
	{ "prctl", (X32_SYSCALL_BIT + 157) },
	{ "pread64", (X32_SYSCALL_BIT + 17) },
	{ "preadv", (X32_SYSCALL_BIT + 534) },
	{ "prlimit64", (X32_SYSCALL_BIT + 302) },
	{ "process_vm_readv", (X32_SYSCALL_BIT + 539) },
	{ "process_vm_writev", (X32_SYSCALL_BIT + 540) },
	{ "prof", __PNR_prof },
	{ "profil", __PNR_profil },
	{ "pselect6", (X32_SYSCALL_BIT + 270) },
	{ "ptrace", (X32_SYSCALL_BIT + 521) },
	{ "putpmsg", (X32_SYSCALL_BIT + 182) },
	{ "pwrite64", (X32_SYSCALL_BIT + 18) },
	{ "pwritev", (X32_SYSCALL_BIT + 535) },
	{ "query_module", __PNR_query_module },
	{ "quotactl", (X32_SYSCALL_BIT + 179) },
	{ "read", (X32_SYSCALL_BIT + 0) },
	{ "readahead", (X32_SYSCALL_BIT + 187) },
	{ "readdir", __PNR_readdir },
	{ "readlink", (X32_SYSCALL_BIT + 89) },
	{ "readlinkat", (X32_SYSCALL_BIT + 267) },
	{ "readv", (X32_SYSCALL_BIT + 515) },
	{ "reboot", (X32_SYSCALL_BIT + 169) },
	{ "recv", __PNR_recv },
	{ "recvfrom", (X32_SYSCALL_BIT + 517) },
	{ "recvmmsg", (X32_SYSCALL_BIT + 537) },
	{ "recvmsg", (X32_SYSCALL_BIT + 519) },
	{ "remap_file_pages", (X32_SYSCALL_BIT + 216) },
	{ "removexattr", (X32_SYSCALL_BIT + 197) },
	{ "rename", (X32_SYSCALL_BIT + 82) },
	{ "renameat", (X32_SYSCALL_BIT + 264) },
	{ "renameat2", (X32_SYSCALL_BIT + 316) },
	{ "request_key", (X32_SYSCALL_BIT + 249) },
	{ "restart_syscall", (X32_SYSCALL_BIT + 219) },
	{ "rmdir", (X32_SYSCALL_BIT + 84) },
	{ "rt_sigaction", (X32_SYSCALL_BIT + 512) },
	{ "rt_sigpending", (X32_SYSCALL_BIT + 522) },
	{ "rt_sigprocmask", (X32_SYSCALL_BIT + 14) },
	{ "rt_sigqueueinfo", (X32_SYSCALL_BIT + 524) },
	{ "rt_sigreturn", (X32_SYSCALL_BIT + 513) },
	{ "rt_sigsuspend", (X32_SYSCALL_BIT + 130) },
	{ "rt_sigtimedwait", (X32_SYSCALL_BIT + 523) },
	{ "rt_tgsigqueueinfo", (X32_SYSCALL_BIT + 536) },
	{ "rtas", __PNR_rtas },
	{ "s390_pci_mmio_read", __PNR_s390_pci_mmio_read },
	{ "s390_pci_mmio_write", __PNR_s390_pci_mmio_write },
	{ "s390_runtime_instr", __PNR_s390_runtime_instr },
	{ "sched_get_priority_max", (X32_SYSCALL_BIT + 146) },
	{ "sched_get_priority_min", (X32_SYSCALL_BIT + 147) },
	{ "sched_getaffinity", (X32_SYSCALL_BIT + 204) },
	{ "sched_getattr", (X32_SYSCALL_BIT + 315) },
	{ "sched_getparam", (X32_SYSCALL_BIT + 143) },
	{ "sched_getscheduler", (X32_SYSCALL_BIT + 145) },
	{ "sched_rr_get_interval", (X32_SYSCALL_BIT + 148) },
	{ "sched_setaffinity", (X32_SYSCALL_BIT + 203) },
	{ "sched_setattr", (X32_SYSCALL_BIT + 314) },
	{ "sched_setparam", (X32_SYSCALL_BIT + 142) },
	{ "sched_setscheduler", (X32_SYSCALL_BIT + 144) },
	{ "sched_yield", (X32_SYSCALL_BIT + 24) },
	{ "seccomp", (X32_SYSCALL_BIT + 317) },
	{ "security", (X32_SYSCALL_BIT + 185) },
	{ "select", (X32_SYSCALL_BIT + 23) },
	{ "semctl", (X32_SYSCALL_BIT + 66) },
	{ "semget", (X32_SYSCALL_BIT + 64) },
	{ "semop", (X32_SYSCALL_BIT + 65) },
	{ "semtimedop", (X32_SYSCALL_BIT + 220) },
	{ "send", __PNR_send },
	{ "sendfile", (X32_SYSCALL_BIT + 40) },
	{ "sendfile64", __PNR_sendfile64 },
	{ "sendmmsg", (X32_SYSCALL_BIT + 538) },
	{ "sendmsg", (X32_SYSCALL_BIT + 518) },
	{ "sendto", (X32_SYSCALL_BIT + 44) },
	{ "set_mempolicy", (X32_SYSCALL_BIT + 238) },
	{ "set_robust_list", (X32_SYSCALL_BIT + 530) },
	{ "set_thread_area", __PNR_set_thread_area },
	{ "set_tid_address", (X32_SYSCALL_BIT + 218) },
	{ "set_tls", __PNR_set_tls },
	{ "setdomainname", (X32_SYSCALL_BIT + 171) },
	{ "setfsgid", (X32_SYSCALL_BIT + 123) },
	{ "setfsgid32", __PNR_setfsgid32 },
	{ "setfsuid", (X32_SYSCALL_BIT + 122) },
	{ "setfsuid32", __PNR_setfsuid32 },
	{ "setgid", (X32_SYSCALL_BIT + 106) },
	{ "setgid32", __PNR_setgid32 },
	{ "setgroups", (X32_SYSCALL_BIT + 116) },
	{ "setgroups32", __PNR_setgroups32 },
	{ "sethostname", (X32_SYSCALL_BIT + 170) },
	{ "setitimer", (X32_SYSCALL_BIT + 38) },
	{ "setns", (X32_SYSCALL_BIT + 308) },
	{ "setpgid", (X32_SYSCALL_BIT + 109) },
	{ "setpriority", (X32_SYSCALL_BIT + 141) },
	{ "setregid", (X32_SYSCALL_BIT + 114) },
	{ "setregid32", __PNR_setregid32 },
	{ "setresgid", (X32_SYSCALL_BIT + 119) },
	{ "setresgid32", __PNR_setresgid32 },
	{ "setresuid", (X32_SYSCALL_BIT + 117) },
	{ "setresuid32", __PNR_setresuid32 },
	{ "setreuid", (X32_SYSCALL_BIT + 113) },
	{ "setreuid32", __PNR_setreuid32 },
	{ "setrlimit", (X32_SYSCALL_BIT + 160) },
	{ "setsid", (X32_SYSCALL_BIT + 112) },
	{ "setsockopt", (X32_SYSCALL_BIT + 541) },
	{ "settimeofday", (X32_SYSCALL_BIT + 164) },
	{ "setuid", (X32_SYSCALL_BIT + 105) },
	{ "setuid32", __PNR_setuid32 },
	{ "setxattr", (X32_SYSCALL_BIT + 188) },
	{ "sgetmask", __PNR_sgetmask },
	{ "shmat", (X32_SYSCALL_BIT + 30) },
	{ "shmctl", (X32_SYSCALL_BIT + 31) },
	{ "shmdt", (X32_SYSCALL_BIT + 67) },
	{ "shmget", (X32_SYSCALL_BIT + 29) },
	{ "shutdown", (X32_SYSCALL_BIT + 48) },
	{ "sigaction", __PNR_sigaction },
	{ "sigaltstack", (X32_SYSCALL_BIT + 525) },
	{ "signal", __PNR_signal },
	{ "signalfd", (X32_SYSCALL_BIT + 282) },
	{ "signalfd4", (X32_SYSCALL_BIT + 289) },
	{ "sigpending", __PNR_sigpending },
	{ "sigprocmask", __PNR_sigprocmask },
	{ "sigreturn", __PNR_sigreturn },
	{ "sigsuspend", __PNR_sigsuspend },
	{ "socket", (X32_SYSCALL_BIT + 41) },
	{ "socketcall", __PNR_socketcall },
	{ "socketpair", (X32_SYSCALL_BIT + 53) },
	{ "splice", (X32_SYSCALL_BIT + 275) },
	{ "spu_create", __PNR_spu_create },
	{ "spu_run", __PNR_spu_run },
	{ "ssetmask", __PNR_ssetmask },
	{ "stat", (X32_SYSCALL_BIT + 4) },
	{ "stat64", __PNR_stat64 },
	{ "statfs", (X32_SYSCALL_BIT + 137) },
	{ "statfs64", __PNR_statfs64 },
	{ "stime", __PNR_stime },
	{ "stty", __PNR_stty },
	{ "subpage_prot", __PNR_subpage_prot },
	{ "swapcontext", __PNR_swapcontext },
	{ "swapoff", (X32_SYSCALL_BIT + 168) },
	{ "swapon", (X32_SYSCALL_BIT + 167) },
	{ "switch_endian", __PNR_switch_endian },
	{ "symlink", (X32_SYSCALL_BIT + 88) },
	{ "symlinkat", (X32_SYSCALL_BIT + 266) },
	{ "sync", (X32_SYSCALL_BIT + 162) },
	{ "sync_file_range", (X32_SYSCALL_BIT + 277) },
	{ "sync_file_range2", __PNR_sync_file_range2 },
	{ "syncfs", (X32_SYSCALL_BIT + 306) },
	{ "syscall", __PNR_syscall },
	{ "sys_debug_setcontext", __PNR_sys_debug_setcontext },
	{ "sysfs", (X32_SYSCALL_BIT + 139) },
	{ "sysinfo", (X32_SYSCALL_BIT + 99) },
	{ "syslog", (X32_SYSCALL_BIT + 103) },
	{ "sysmips", __PNR_sysmips },
	{ "tee", (X32_SYSCALL_BIT + 276) },
	{ "tgkill", (X32_SYSCALL_BIT + 234) },
	{ "time", (X32_SYSCALL_BIT + 201) },
	{ "timer_create", (X32_SYSCALL_BIT + 526) },
	{ "timer_delete", (X32_SYSCALL_BIT + 226) },
	{ "timer_getoverrun", (X32_SYSCALL_BIT + 225) },
	{ "timer_gettime", (X32_SYSCALL_BIT + 224) },
	{ "timer_settime", (X32_SYSCALL_BIT + 223) },
	{ "timerfd", __PNR_timerfd },
	{ "timerfd_create", (X32_SYSCALL_BIT + 283) },
	{ "timerfd_gettime", (X32_SYSCALL_BIT + 287) },
	{ "timerfd_settime", (X32_SYSCALL_BIT + 286) },
	{ "times", (X32_SYSCALL_BIT + 100) },
	{ "tkill", (X32_SYSCALL_BIT + 200) },
	{ "truncate", (X32_SYSCALL_BIT + 76) },
	{ "truncate64", __PNR_truncate64 },
	{ "tuxcall", (X32_SYSCALL_BIT + 184) },
	{ "ugetrlimit", __PNR_ugetrlimit },
	{ "ulimit", __PNR_ulimit },
	{ "umask", (X32_SYSCALL_BIT + 95) },
	{ "umount", __PNR_umount },
	{ "umount2", (X32_SYSCALL_BIT + 166) },
	{ "uname", (X32_SYSCALL_BIT + 63) },
	{ "unlink", (X32_SYSCALL_BIT + 87) },
	{ "unlinkat", (X32_SYSCALL_BIT + 263) },
	{ "unshare", (X32_SYSCALL_BIT + 272) },
	{ "uselib", __PNR_uselib },
	{ "userfaultfd", (X32_SYSCALL_BIT + 323) },
	{ "usr26", __PNR_usr26 },
	{ "usr32", __PNR_usr32 },
	{ "ustat", (X32_SYSCALL_BIT + 136) },
	{ "utime", (X32_SYSCALL_BIT + 132) },
	{ "utimensat", (X32_SYSCALL_BIT + 280) },
	{ "utimes", (X32_SYSCALL_BIT + 235) },
	{ "vfork", (X32_SYSCALL_BIT + 58) },
	{ "vhangup", (X32_SYSCALL_BIT + 153) },
	{ "vm86", __PNR_vm86 },
	{ "vm86old", __PNR_vm86old },
	{ "vmsplice", (X32_SYSCALL_BIT + 532) },
	{ "vserver", __PNR_vserver },
	{ "wait4", (X32_SYSCALL_BIT + 61) },
	{ "waitid", (X32_SYSCALL_BIT + 529) },
	{ "waitpid", __PNR_waitpid },
	{ "write", (X32_SYSCALL_BIT + 1) },
	{ "writev", (X32_SYSCALL_BIT + 516) },
	{ NULL, __NR_SCMP_ERROR},
};

/**
 * Resolve a syscall name to a number
 * @param name the syscall name
 *
 * Resolve the given syscall name to the syscall number using the syscall table.
 * Returns the syscall number on success, including negative pseudo syscall
 * numbers; returns __NR_SCMP_ERROR on failure.
 *
 */
int x32_syscall_resolve_name(const char *name)
{
	unsigned int iter;
	const struct arch_syscall_def *table = x32_syscall_table;

	/* XXX - plenty of room for future improvement here */
	for (iter = 0; table[iter].name != NULL; iter++) {
		if (strcmp(name, table[iter].name) == 0)
			return table[iter].num;
	}

	return __NR_SCMP_ERROR;
}

/**
 * Resolve a syscall number to a name
 * @param num the syscall number
 *
 * Resolve the given syscall number to the syscall name using the syscall table.
 * Returns a pointer to the syscall name string on success, including pseudo
 * syscall names; returns NULL on failure.
 *
 */
const char *x32_syscall_resolve_num(int num)
{
	unsigned int iter;
	const struct arch_syscall_def *table = x32_syscall_table;

	/* XXX - plenty of room for future improvement here */
	for (iter = 0; table[iter].num != __NR_SCMP_ERROR; iter++) {
		if (num == table[iter].num)
			return table[iter].name;
	}

	return NULL;
}

/**
 * Iterate through the syscall table and return the syscall name
 * @param spot the offset into the syscall table
 *
 * Return the syscall name at position @spot or NULL on failure.  This function
 * should only ever be used internally by libseccomp.
 *
 */
const char *x32_syscall_iterate_name(unsigned int spot)
{
	/* XXX - no safety checks here */
	return x32_syscall_table[spot].name;
}
