#! /usr/bin/env false

use v6.c;

use Terminal::ANSIColor;
use Pod::To::Pager::BorderedBlock;

unit module Pod::To::Pager::Handlers;

my @footnotes;

multi sub handle-pod (
	Pod::FormattingCode $pod,
	Int :$indent-size = 8,
	Int :$max-line-length = 80,
	Str :$prefix is copy = "",
	Str :$suffix is copy = "",
	--> Str
) {
	$suffix = 0 < $suffix.chars
		?? RESET ~ $prefix
		!! RESET
		;

	given $pod.type {
		when "B" { $prefix = BOLD; }
		when "C" { $prefix = BOLD; }
		when "E" {
			return handle-pod(
				$pod.contents,
				:$max-line-length,
				:$prefix,
				:$suffix,
				indent-size => 0,
			);
		}
		when "I" { $prefix = ITALIC; }
		when "K" { $prefix = color('yellow'); }
		when "L" {
			@footnotes.append(color('blue underline') ~ $pod.meta[0] ~ color('reset'));

			return handle-pod(
				$pod.contents[0] ~ "[" ~ @footnotes.elems ~ "]",
				:$max-line-length,
				indent-size => 0,
				prefix => "",
				suffix => "",
			);
		}
		when "N" {
			@footnotes.append($pod.contents[0]);

			return "[" ~ @footnotes.elems ~ "]";
		}
		when "T" { $prefix = color('magenta') }
		when "U" { $prefix = UNDERLINE; }
		when "Z" { return ""; }
	}

	handle-pod(
		$pod.contents,
		:$max-line-length,
		:$prefix,
		indent-size => 0,
		suffix => RESET,
	);
}

multi sub handle-pod (
	Pod::Block::Code $pod,
	Int :$indent-size = 8,
	Int :$max-line-length = 80,
	Str :$prefix = "",
	Str :$suffix = "",
	--> Str
) {
	my Pod::To::Pager::BorderedBlock $block .= new;

	my $code-block = $block.render(
		handle-pod(
			$pod.contents,
			indent-size => 0,
			max-line-length => 0,
			prefix => BOLD,
			suffix => color('reset'),
		),
		header => $pod.config<name> // "",
	);

	$code-block.split("\n").map({ $_.indent($indent-size + 4) }).join("\n") ~ "\n\n";
}

#| Handle definition lists.
multi sub handle-pod(
	Pod::Block::Named $pod where { $pod.name eq "defn" },
	Int :$indent-size = 8,
	Int :$max-line-length = 80,
	Str :$prefix = "",
	Str :$suffix = "",
	--> Str
) {
	my Str $first-line = $pod.contents[0].contents[0];
	my ($word, @leftovers) = $first-line.split(/\s+/);

	$pod.contents[0].contents[0] = @leftovers.join(" ");

	my Str $text = "{color('cyan')}{$word}{color('reset')}\n".indent($indent-size);

	$text ~= handle-pod(
		$pod.contents,
		indent-size => $indent-size + 2,
		:$max-line-length,
		:$prefix,
		:$suffix
	);

	$text;
}

multi sub handle-pod (
	Pod::Block::Named $pod where { $pod.name eq "input" },
	Int :$indent-size = 8,
	Int :$max-line-length = 80,
	Str :$prefix = "",
	Str :$suffix = "",
	--> Str
) {
	my Pod::To::Pager::BorderedBlock $block .= new;

	my $code-block = $block.render(
		handle-pod(
			$pod.contents,
			indent-size => 0,
			max-line-length => 0,
			prefix => color('yellow'),
			suffix => color('reset'),
		),
		header => $pod.config<name> // "",
	);

	$code-block.split("\n").map({ $_.indent($indent-size + 4) }).join("\n") ~ "\n\n";
}

multi sub handle-pod (
	Pod::Block::Named $pod where { $pod.name eq "output" },
	Int :$indent-size = 8,
	Int :$max-line-length = 80,
	Str :$prefix = "",
	Str :$suffix = "",
	--> Str
) {
	my Pod::To::Pager::BorderedBlock $block .= new;

	my $code-block = $block.render(
		handle-pod(
			$pod.contents,
			indent-size => 0,
			max-line-length => 0,
			prefix => color('magenta'),
			suffix => color('reset'),
		),
		header => $pod.config<name> // "",
	);

	$code-block.split("\n").map({ $_.indent($indent-size + 4) }).join("\n") ~ "\n\n";
}

#| Handle the main pod block
multi sub handle-pod(
	Pod::Block::Named $pod where { $pod.name eq "pod" },
	Int :$indent-size = 8,
	Int :$max-line-length = 80,
	Str :$prefix = "",
	Str :$suffix = "",
	--> Str
) {
	my Str $document;

	for $pod.contents {
		$document ~= handle-pod(
			$_,
			:$indent-size,
			:$max-line-length,
			:$prefix,
			:$suffix
		);
	}

	if (@footnotes) {
		my $index-length = @footnotes.elems.Str.chars;

		$document ~= handle-pod(
			Pod::Block::Named.new(
				name => "Footnotes and references",
			)
		);

		$document ~= "\n";

		loop (my $i = 0; $i < @footnotes.elems; $i++) {
			my @lines;
			my $line = [sprintf("%{$index-length}d: ", $i + 1)];
			my @words = @footnotes[$i].split(/\s+/);

			for @words -> $word {
				if ($max-line-length < ($indent-size + $line.chars + $word.chars)) {
					@lines.append(chomp($line).indent($indent-size));
					$line = $word;
					next;
				}

				$line ~= " $word";
			}

			@lines.append(chomp($line).indent($indent-size));

			$document ~= @lines.join("\n") ~ "\n";
		}
	}

	$document;
}

#| Handle all other non-specific named pod blocks.
multi sub handle-pod (
	Pod::Block::Named $pod,
	Int :$indent-size = 8,
	Int :$max-line-length = 80,
	Str :$prefix = "",
	Str :$suffix = "",
	--> Str
) {
	my Str $text;

	$text ~= "{color('bold blue')}{$pod.name}{color('reset')}\n";
	$text ~= handle-pod(
		$pod.contents,
		:$indent-size,
		:$max-line-length,
		:$prefix,
		:$suffix
	) // "";

	$text;
}

multi sub handle-pod(
	Pod::Block::Para $pod,
	Int :$indent-size = 8,
	Int :$max-line-length = 80,
	Str :$prefix = "",
	Str :$suffix = "",
	--> Str
) {
	my Str $text = "";
	
	for handle-pod(
		$pod.contents,
		indent-size => 0,
		max-line-length => $max-line-length - $indent-size,
		:$prefix,
		:$suffix
	).lines {
		$text ~= $_.indent($indent-size) ~ "\n";
	}

	$text ~ "\n";
}

multi sub handle-pod(
	Pod::Item $pod,
	Int :$indent-size = 8,
	Int :$max-line-length = 80,
	Str :$prefix = "",
	Str :$suffix = "",
	--> Str
) {
	my @contents = $pod.contents;
	my Str $bullet = "• ";
	my Int $width = ($pod.level - 1) × 2;
	my Str $text = $bullet.indent($indent-size + $width);

	$text ~= handle-pod(
		$pod.contents,
		indent-size => ($indent-size + $width + $bullet.chars),
		max-line-length => ($max-line-length - $indent-size),
		:$prefix,
		:$suffix,
	).trim-leading;

	$text;
}

multi sub handle-pod(
	Pod::Heading $pod,
	Int :$indent-size = 8,
	Int :$max-line-length = 80,
	Str :$prefix = "",
	Str :$suffix = "",
	--> Str
) {
	handle-pod(
		$pod.contents,
		:$max-line-length,
		indent-size => ($pod.level - 1) × 2,
		prefix => color('bold blue'),
		suffix => RESET,
	);
}

multi sub handle-pod(
	Str $text,
	Int :$indent-size = 8,
	Int :$max-line-length = 80,
	Str :$prefix = "",
	Str :$suffix = "",
	--> Str
) {
	if ($max-line-length < 1) {
		for $text.lines {
			if ($_ eq "") {
				return "\n";
			}

			return "{$prefix}{$_.indent($indent-size)}{$suffix}";
		}
	}

	my @words = "{$prefix}{$text}{$suffix}".split(/\s+/);
	my @lines;
	my $line = @words.shift;

	for @words -> $word {
		if ($max-line-length < ($indent-size + $line.chars + $word.chars)) {
			@lines.append(chomp($line).indent($indent-size));
			$line = $word;
			next;
		}

		$line ~= " $word";
	}

	@lines.append(chomp($line).indent($indent-size));

	return @lines.join("\n");
}

multi sub handle-pod (
	@nodes,
	Int :$indent-size = 8,
	Int :$max-line-length = 80,
	Str :$prefix = "",
	Str :$suffix = "",
	--> Str
) is export {
	my Str $text;

	for @nodes {
		$text ~= handle-pod(
			$_,
			:$indent-size,
			:$max-line-length,
			:$prefix,
			:$suffix,
		);
	}

	$text;
}

# vim: ft=perl6 noet
