package App::Manoc::Utils;
#ABSTRACT: Miscellaneous support functions
use strict;
use warnings;

our $VERSION = '2.99.4'; ##TRIAL VERSION

BEGIN {
    use Exporter 'import';
    our @EXPORT_OK = qw/
        clean_string
        check_mac_addr
        normalize_mac_addr
        /;
}

use Regexp::Common qw/net/;

use Carp;
use Archive::Tar;

########################################################################
#                                                                      #
#                    S t r i n g    F u n c t i o n s                  #
#                                                                      #
########################################################################


sub check_mac_addr {
    return defined( normalize_mac_addr(shift) );
}


sub normalize_mac_addr {
    my $addr = shift;

    return unless defined($addr);

    $addr =~
        /^[0-9a-fA-F]{2}:[0-9a-fA-F]{2}:[0-9a-fA-F]{2}:[0-9a-fA-F]{2}:[0-9a-fA-F]{2}:[0-9a-fA-F]{2}$/
        and
        return lc
        ($addr);

    $addr =~
        /^([0-9a-fA-F]{2})([0-9a-fA-F]{2})[-:]?([0-9a-fA-F]{2})([0-9a-fA-F]{2})[-:]?([0-9a-fA-F]{2})([0-9a-fA-F]{2})$/
        and
        return lc
        ("$1:$2:$3:$4:$5:$6");

    $addr =~
        /^([0-9a-fA-F]{2})-([0-9a-fA-F]{2})-([0-9a-fA-F]{2})-([0-9a-fA-F]{2})-([0-9a-fA-F]{2})-([0-9a-fA-F]{2})$/
        and
        return lc
        ("$1:$2:$3:$4:$5:$6");
}


sub clean_string {
    my $s = shift;
    return '' unless defined $s;
    $s =~ s/^\s+//o;
    $s =~ s/\s+$//o;
    return lc($s);
}

########################################################################
#                                                                      #
#                     S e t    F u n c t i o n s                       #
#                                                                      #
########################################################################


sub decode_bitset {
    my $bits  = shift;
    my $names = shift;

    my @result;

    my @bitlist = reverse split( //, $bits );
    my ( $n, $b );

    while ( @$names && @bitlist ) {
        $n = shift @$names;
        $b = shift @bitlist;

        $b or next;
        push @result, $n;
    }

    return @result;
}

########################################################################
#                                                                      #
#                   T a r   F u n c t i o n s                          #
#                                                                      #
########################################################################


sub tar {
    my ( $tarname, $basedir, $filelist_ref ) = @_;
    my $command = "tar";

    #check the existence of tar command
    #running tar --version
    `$command --version 2>&1`;
    if ( $? == 0 ) {
        #use system tar
        #remove leading path from filelist to avoid creating tar with
        #file that have complete path e.g. /tmp/device.yaml
        my @sanitized;
        foreach my $file (@$filelist_ref) {
            $file =~ s/^$basedir\/?//o;
            push @sanitized, $file;
        }
        `$command -zcf $tarname -C $basedir/ @sanitized 2>&1`;
        return $?;
    }
    else {
        #use Archive::Tar
        my $tar      = Archive::Tar->new;
        my @obj_list = $tar->add_files(@$filelist_ref);

        #remove prefix
        foreach my $o (@obj_list) {
            $o->prefix('');
        }
        return $tar->write( $tarname, 1 );
    }
}

1;

__END__

=pod

=head1 NAME

App::Manoc::Utils - Miscellaneous support functions

=head1 VERSION

version 2.99.4

=head1 FUNCTIONS

=head2 check_mac_addr($addr)

Return 1 if C<$addr> is a valid MAC address.

=head2 normalize_mac_addr($addr)

Return  C<$addr>  in 'aa:bb:cc:dd:ee' format if is a valid mac address.
Otherwise return undef.

=head2 clean_string($s)

Return C<$s> trimmed and lowercase.

=head2 decode_bitset($bits, \@names)

Given a string representation of a bitlist and a list of names return
the names corresponding to 1 bits.

 decode_bitset('0110', ['one', 'two', 'three', 'four' ])

gives C<('two', 'three')>.

=head2 tar($filename, $basedir, \@files)

Create a tar file called c<$filename> with files in C<@files> storing
their paths relatively to C<$basedir>.

Use the C<tar> command if present, otherwise Archive::Tar.

=head1 AUTHORS

=over 4

=item *

Gabriele Mambrini <gmambro@cpan.org>

=item *

Enrico Liguori

=back

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2017 by Gabriele Mambrini.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
